#include "period_job_worker.h"

#include <infra/pod_agent/libs/multi_unistat/multi_unistat.h>

#include <util/string/builder.h>
#include <util/string/cast.h>

namespace NInfra::NPodAgent {

namespace {

struct TThreadData {
    TThreadData(
        TPeriodJobPtr job
        , TLogFramePtr logFrame
    )
        : Job_(job)
        , LogFrame_(logFrame)

        , LastStartTime_(TInstant::Zero())
        , LastDuration_(TDuration::Zero())
        , IsRunning_(false)

        , NeedQuit_(true)
    {
    }

    TPeriodJobPtr Job_;
    TLogFramePtr LogFrame_;

    TMutex RunMutex_;
    TInstant LastStartTime_;
    TDuration LastDuration_;
    TAtomic IsRunning_;

    TAtomic NeedQuit_;
    TCondVar QuitSignal_;
    TMutex QuitMutex_;
};

} // namespace

void TPeriodJobWorker::Start() {
    TGuard<TMutex> gMutex(Mutex_);
    Y_ENSURE(!RunnerThread_.Running(), "TPeriodJobWorker already started");
    AtomicSet(NeedQuit_, false);
    RunnerThread_.Start();
}

void TPeriodJobWorker::Shutdown() {
    TGuard<TMutex> gMutex(Mutex_);
    AtomicSet(NeedQuit_, true);
    QuitSignal_.Signal();
}

void TPeriodJobWorker::Wait() {
    RunnerThread_.Join();
}

void TPeriodJobWorker::AddJob(TPeriodJobPtr job) {
    TGuard<TMutex> gMutex(Mutex_);
    Y_ENSURE(!RunnerThread_.Running(), "TPeriodJobWorker already started, stop it for new job");
    Jobs_.push_back(job);
}

void* TPeriodJobWorker::RunMainLoop(void* me) {
    TPeriodJobWorker* periodJobWorker = (TPeriodJobWorker*)me;

    periodJobWorker->InitSignals();

    TVector<THolder<TThreadData>> jobThreadsData;
    jobThreadsData.reserve(periodJobWorker->Jobs_.size());
    for (auto job : periodJobWorker->Jobs_) {
        jobThreadsData.emplace_back(MakeHolder<TThreadData>(job, job->GetPeriodWorkerEventsLogFrame()));
    }

    TVector<THolder<TThread>> jobThreads;
    jobThreads.reserve(jobThreadsData.size());
    for (auto& jobThreadData : jobThreadsData) {
        jobThreads.emplace_back(MakeHolder<TThread>(TPeriodJobWorker::RunJobLoop, (void*)jobThreadData.Get()));
    }

    for (size_t i = 0; i < jobThreadsData.size(); ++i) {
        // No guard because thead is not running
        AtomicSet(jobThreadsData[i]->NeedQuit_, false);
        jobThreads[i]->Start();
    }

    while (!AtomicGet(periodJobWorker->NeedQuit_)) {
        TInstant timeNow = TInstant::Now();
        for (auto& threadData : jobThreadsData) {
            TGuard<TMutex> guard(threadData->RunMutex_);
            if (AtomicGet(threadData->IsRunning_)) {
                periodJobWorker->SetJobDurationSignal(threadData->Job_->GetName(), timeNow - threadData->LastStartTime_);
            } else {
                periodJobWorker->SetJobDurationSignal(threadData->Job_->GetName(), threadData->LastDuration_);
            }
        }

        TGuard<TMutex> gMutex(periodJobWorker->Mutex_);
        if (!AtomicGet(periodJobWorker->NeedQuit_)) {
            periodJobWorker->QuitSignal_.WaitT(periodJobWorker->Mutex_, periodJobWorker->MainLoopPeriod_);
        }
    }

    for (auto& jobThreadData : jobThreadsData) {
        TGuard<TMutex> gMutex(jobThreadData->QuitMutex_);
        AtomicSet(jobThreadData->NeedQuit_, true);
        jobThreadData->QuitSignal_.Signal();
    }

    for (auto& jobThread : jobThreads) {
        jobThread->Join();
    }

    return nullptr;
}

void* TPeriodJobWorker::RunJobLoop(void* data) {
    TThreadData* threadData = (TThreadData*)data;

    while (!AtomicGet(threadData->NeedQuit_)) {
        TInstant timeBeforeRun = TInstant::Now();

        threadData->LogFrame_->LogEvent(
            ELogPriority::TLOG_INFO
            , NLogEvent::TPeriodJobWorkerJobStart(threadData->Job_->GetName())
        );

        {
            TGuard<TMutex> guard(threadData->RunMutex_);
            threadData->LastStartTime_ = timeBeforeRun;
            AtomicSet(threadData->IsRunning_, true);
        }

        try {
            threadData->Job_->Run();
        } catch (const yexception& e) {
            threadData->LogFrame_->LogEvent(
                ELogPriority::TLOG_ERR
                , NLogEvent::TPeriodJobWorkerException(CurrentExceptionMessage(), threadData->Job_->GetName())
            );
        }

        TInstant timeAfterRun = TInstant::Now();

        TDuration runDuration = timeAfterRun - timeBeforeRun;
        TDuration timeUntilNextRun = threadData->Job_->GetPeriod() - runDuration;

        {
            TGuard<TMutex> guard(threadData->RunMutex_);
            threadData->LastDuration_ = runDuration;
            AtomicSet(threadData->IsRunning_, false);
        }

        threadData->LogFrame_->LogEvent(
            ELogPriority::TLOG_INFO
            , NLogEvent::TPeriodJobWorkerJobEnd(threadData->Job_->GetName(), ToString(timeUntilNextRun))
        );

        {
            TGuard<TMutex> gMutex(threadData->QuitMutex_);
            if (!AtomicGet(threadData->NeedQuit_)) {
                threadData->QuitSignal_.WaitT(threadData->QuitMutex_, timeUntilNextRun);
            }
        }
    }

    threadData->LogFrame_->LogEvent(
        ELogPriority::TLOG_INFO
        , NLogEvent::TPeriodJobWorkerJobShutdownStart(threadData->Job_->GetName())
    );

    threadData->Job_->Shutdown();

    threadData->LogFrame_->LogEvent(
        ELogPriority::TLOG_INFO
        , NLogEvent::TPeriodJobWorkerJobShutdownEnd(threadData->Job_->GetName())
    );

    return nullptr;
}

void TPeriodJobWorker::InitSignals() {
    for (const auto& job : Jobs_) {
        const TString curName = SIGNAL_PREFIX + job->GetName() + SIGNAL_JOB_TIME_SUFFIX;

        TMultiUnistat::Instance().DrillFloatHole(
            TMultiUnistat::ESignalNamespace::INFRA
            , curName
            , "ahhh"
            , NUnistat::TPriority(TMultiUnistat::ESignalPriority::INFRA_INFO)
            , NUnistat::TStartValue(0)
            , EAggregationType::LastValue
        );
    }
}

void TPeriodJobWorker::PushSignal(const TString& name, double value) {
    if (!TMultiUnistat::Instance().PushSignalUnsafe(TMultiUnistat::ESignalNamespace::INFRA, name, value)) {
        LogFrame_->LogEvent(
            ELogPriority::TLOG_ERR
            , NLogEvent::TPeriodJobWorkerSignalError(
                name
                , TStringBuilder()
                    << "PushSingalUnsafe(" + name + ", " << value << ") returned false."
            )
        );
    }
}

void TPeriodJobWorker::SetJobDurationSignal(const TString& jobName, const TDuration& duration) {
    PushSignal(SIGNAL_PREFIX + jobName + SIGNAL_JOB_TIME_SUFFIX, duration.MicroSeconds() / 1000.);
}

} // namespace NInfra::NPodAgent
