#include "period_job_worker.h"

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NPeriodJobWorkerTest  {

static TLogger logger({});

static constexpr TDuration MAIN_LOOP_PERIOD = TDuration::Seconds(5);

class TTestPeriodJob;
using TTestPeriodJobPtr = TIntrusivePtr<TTestPeriodJob>;

class TTestPeriodJob : public IPeriodJob {
public:
    TTestPeriodJob(
        const TString& name
        , const TDuration& period
    )
        : IPeriodJob(name, period, logger.SpawnFrame())
        , Calls_(0)
    {}

    void Run() override {
        AtomicAdd(Calls_, 1);
    }

    ui32 GetCalls() const {
        return AtomicGet(Calls_);
    }

private:
    TAtomic Calls_;
};

Y_UNIT_TEST_SUITE(PeriodJobWorkerSuite) {

Y_UNIT_TEST(TestNoJob) {
    TPeriodJobWorker periodJobWorker(MAIN_LOOP_PERIOD, logger.SpawnFrame());

    periodJobWorker.Start();
    Sleep(TDuration::MilliSeconds(100));
    periodJobWorker.Stop();
}

Y_UNIT_TEST(TestDoubleStartException) {
    TPeriodJobWorker periodJobWorker(MAIN_LOOP_PERIOD, logger.SpawnFrame());

    periodJobWorker.Start();
    UNIT_ASSERT_EXCEPTION(periodJobWorker.Start(), yexception);
    periodJobWorker.Stop();
}

Y_UNIT_TEST(TestAddException) {
    TPeriodJobWorker periodJobWorker(MAIN_LOOP_PERIOD, logger.SpawnFrame());

    periodJobWorker.Start();
    UNIT_ASSERT_EXCEPTION(periodJobWorker.AddJob(new TTestPeriodJob("TestJob", TDuration::Seconds(1))), yexception);
    periodJobWorker.Stop();
}

Y_UNIT_TEST(TestOneJob) {
    TPeriodJobWorker periodJobWorker(MAIN_LOOP_PERIOD, logger.SpawnFrame());

    TTestPeriodJobPtr job = new TTestPeriodJob("TestJob", TDuration::MilliSeconds(100));
    periodJobWorker.AddJob(job);

    periodJobWorker.Start();
    Sleep(TDuration::MilliSeconds(350));
    periodJobWorker.Stop();

    UNIT_ASSERT(job->GetCalls() > 1);
}

Y_UNIT_TEST(TestOneJobLongPeriod) {
    TPeriodJobWorker periodJobWorker(MAIN_LOOP_PERIOD, logger.SpawnFrame());

    TTestPeriodJobPtr job = new TTestPeriodJob("TestJob", TDuration::Minutes(1));
    periodJobWorker.AddJob(job);

    periodJobWorker.Start();
    Sleep(TDuration::MilliSeconds(350));
    periodJobWorker.Stop();

    UNIT_ASSERT(job->GetCalls() == 1);
}

Y_UNIT_TEST(TestMultiplyJobs) {
    TPeriodJobWorker periodJobWorker(MAIN_LOOP_PERIOD, logger.SpawnFrame());

    constexpr ui32 specialJobId = 5;
    constexpr ui32 jobCnt = 10;
    TVector<TTestPeriodJobPtr> jobs;
    for (ui32 i = 0; i < jobCnt; ++i) {
        TDuration period = (i == specialJobId)
            ? TDuration::Minutes(1)
            : TDuration::MilliSeconds(100)
        ;
        jobs.push_back(new TTestPeriodJob("TestJob_" + ToString(i), period));
        periodJobWorker.AddJob(jobs.back());
    }

    periodJobWorker.Start();
    Sleep(TDuration::MilliSeconds(600));
    periodJobWorker.Stop();

    for (ui32 i = 0; i < jobCnt; ++i) {
        if (i == specialJobId) {
            UNIT_ASSERT(jobs[i]->GetCalls() == 1);
        } else {
            UNIT_ASSERT(jobs[i]->GetCalls() > 1);
        }
    }
}

Y_UNIT_TEST(TestExceptionInJob) {
    class TTestPeriodJobWithException : public IPeriodJob {
    public:
        TTestPeriodJobWithException(
            const TString& name
            , const TDuration& period
        )
            : IPeriodJob(name, period, logger.SpawnFrame())
            , Calls_(0)
        {}

        void Run() override {
            AtomicAdd(Calls_, 1);
            ythrow yexception() << "fail";
        }

        ui32 GetCalls() const {
            return AtomicGet(Calls_);
        }

    private:
        TAtomic Calls_;
    };

    TPeriodJobWorker periodJobWorker(MAIN_LOOP_PERIOD, logger.SpawnFrame());

    constexpr ui32 specialJobId = 5;
    constexpr ui32 jobCnt = 10;
    TVector<TPeriodJobPtr> jobs;
    for (ui32 i = 0; i < jobCnt; ++i) {
        if (i == specialJobId) {
            jobs.push_back(new TTestPeriodJobWithException("TestJob_" + ToString(i), TDuration::MilliSeconds(100)));
        } else {
            jobs.push_back(new TTestPeriodJob("TestJob_" + ToString(i), TDuration::MilliSeconds(100)));
        }

        periodJobWorker.AddJob(jobs.back());
    }

    periodJobWorker.Start();
    Sleep(TDuration::MilliSeconds(600));
    periodJobWorker.Stop();

    for (ui32 i = 0; i < jobCnt; ++i) {
        if (i == specialJobId) {
            UNIT_ASSERT(((TTestPeriodJobWithException*)jobs[i].Get())->GetCalls() > 1);
        } else {
            UNIT_ASSERT(((TTestPeriodJob*)jobs[i].Get())->GetCalls() > 1);
        }
    }
}

Y_UNIT_TEST(TestJobDeadlock) {
    class TTestPeriodJobWithDeadlock : public IPeriodJob {
    public:
        TTestPeriodJobWithDeadlock(
            const TString& name
            , const TDuration& period
        )
            : IPeriodJob(name, period, logger.SpawnFrame())
            , Calls_(0)
            , DeadLock_(true)
        {}

        void Run() override {
            AtomicAdd(Calls_, 1);
            while (AtomicGet(DeadLock_)) {
                Sleep(TDuration::MilliSeconds(500));
            }
        }

        ui32 GetCalls() const {
            return AtomicGet(Calls_);
        }

        void SetDeadLock(bool flag) {
            AtomicSet(DeadLock_, flag);
        }

    private:
        TAtomic Calls_;
        TAtomic DeadLock_;
    };

    TPeriodJobWorker periodJobWorker(MAIN_LOOP_PERIOD, logger.SpawnFrame());

    constexpr ui32 specialJobId = 5;
    constexpr ui32 jobCnt = 10;
    TVector<TPeriodJobPtr> jobs;
    for (ui32 i = 0; i < jobCnt; ++i) {
        if (i == specialJobId) {
            jobs.push_back(new TTestPeriodJobWithDeadlock("TestJob_" + ToString(i), TDuration::MilliSeconds(100)));
        } else {
            jobs.push_back(new TTestPeriodJob("TestJob_" + ToString(i), TDuration::MilliSeconds(100)));
        }

        periodJobWorker.AddJob(jobs.back());
    }

    periodJobWorker.Start();
    Sleep(TDuration::MilliSeconds(600));

    for (ui32 i = 0; i < jobCnt; ++i) {
        if (i == specialJobId) {
            UNIT_ASSERT_EQUAL(((TTestPeriodJobWithDeadlock*)jobs[i].Get())->GetCalls(), 1);
        } else {
            UNIT_ASSERT(((TTestPeriodJob*)jobs[i].Get())->GetCalls() > 1);
        }
    }

    ((TTestPeriodJobWithDeadlock*)jobs[specialJobId].Get())->SetDeadLock(false);
    periodJobWorker.Stop();
}

}

} // namespace NInfra::NPodAgent::NPeriodJobWorkerTest
