#include "status_and_ticker_holder.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/mock_node.h>
#include <infra/pod_agent/libs/pod_agent/unistat_object_helper/unistat_object_helper.h>
#include <infra/pod_agent/libs/system_logs_sender/mock_system_logs_session.h>

#include <util/generic/queue.h>

namespace NInfra::NPodAgent {

TLoggerConfig GetMockLoggerConfig() {
    TLoggerConfig config;
    config.SetBackend(TLoggerConfig_ELogBackend_STDERR);
    return config;
}

TLogger TStatusNTickerHolder::MOCK_LOGGER({GetMockLoggerConfig()});

TString TStatusNTickerHolder::GetStaticResourceTreeId(const TString& staticResourceDownloadHash) {
    return STATIC_RESOURCE_TREE_PREFIX + staticResourceDownloadHash;
}

TString TStatusNTickerHolder::GetLayerTreeId(const TString& layerDownloadHash) {
    return LAYER_TREE_PREFIX + layerDownloadHash;
}

TString TStatusNTickerHolder::GetVolumeTreeId(const TString& volumeId) {
    return VOLUME_TREE_PREFIX + volumeId;
}

TString TStatusNTickerHolder::GetBoxTreeId(const TString& boxId) {
    return BOX_TREE_PREFIX + boxId;
}

TString TStatusNTickerHolder::GetWorkloadTreeId(const TString& workloadId) {
    return WORKLOAD_TREE_PREFIX + workloadId;
}

TTreePtr TStatusNTickerHolder::GetEmptyTree(const TString& treeId) {
    return new TTree(TStatusNTickerHolder::MOCK_LOGGER, treeId, new TMockNode({1, "empty_tree_root"}, TTickResult(ENodeStatus::SUCCESS)));
}

TBoxStatusRepositoryPtr TStatusNTickerHolder::GetBoxStatusRepository() const {
    return StatusRepository_->GetBoxStatusRepository();
}

TLayerStatusRepositoryPtr TStatusNTickerHolder::GetLayerStatusRepository() const {
    return StatusRepository_->GetLayerStatusRepository();
}

TStaticResourceStatusRepositoryPtr TStatusNTickerHolder::GetStaticResourceStatusRepository() const {
    return StatusRepository_->GetStaticResourceStatusRepository();
}

TVolumeStatusRepositoryPtr TStatusNTickerHolder::GetVolumeStatusRepository() const {
    return StatusRepository_->GetVolumeStatusRepository();
}

TWorkloadStatusRepositoryPtr TStatusNTickerHolder::GetWorkloadStatusRepository() const {
    return StatusRepository_->GetWorkloadStatusRepository();
}

TWorkloadStatusRepositoryInternalPtr TStatusNTickerHolder::GetWorkloadStatusRepositoryInternal() const {
    return WorkloadStatusRepositoryInternal_;
}

TUpdateHolderPtr TStatusNTickerHolder::GetUpdateHolder() const {
    return StatusRepository_->GetUpdateHolder();
}

void TStatusNTickerHolder::EnableTransmitSystemLogs(bool enable) {
    TGuard<TMutex> gMutex(Mutex_);
    WorkloadSystemLogsSender_->MarkEnabled(enable);
    BoxSystemLogsSender_->MarkEnabled(enable);
}

ISystemLogsSenderPtr TStatusNTickerHolder::GetWorkloadSystemLogsSender() const {
    return WorkloadSystemLogsSender_;
}

ISystemLogsSenderPtr TStatusNTickerHolder::GetBoxSystemLogsSender() const {
    return BoxSystemLogsSender_;
}

void TStatusNTickerHolder::StartTicker() {
    Ticker_->Start();
}

void TStatusNTickerHolder::StopTicker() {
    Ticker_->Stop();
}

void TStatusNTickerHolder::UpdateActiveDownloadContainersLimit(ui32 activeDownloadContainersLimit) {
    GetStaticResourceStatusRepository()->UpdateActiveDownloadContainersLimit(activeDownloadContainersLimit);
    GetLayerStatusRepository()->UpdateActiveDownloadContainersLimit(activeDownloadContainersLimit);
}

void TStatusNTickerHolder::UpdateActiveVerifyContainersLimit(ui32 activeVerifyContainersLimit) {
    GetStaticResourceStatusRepository()->UpdateActiveVerifyContainersLimit(activeVerifyContainersLimit);
    GetLayerStatusRepository()->UpdateActiveVerifyContainersLimit(activeVerifyContainersLimit);
}

void TStatusNTickerHolder::SetStatusRepositorySpecId(const TString& id) {
    StatusRepository_->SetSpecId(id);
}

API::TPodAgentStatus TStatusNTickerHolder::GetStatusRepositoryTotalStatus(bool conditionsOnly) const {
    return StatusRepository_->GetTotalStatus(conditionsOnly);
}

void TStatusNTickerHolder::SetStatusRepositorySpecTimestamp(ui64 specTimestamp) {
    StatusRepository_->SetSpecTimestamp(specTimestamp);
}

void TStatusNTickerHolder::SetStatusRepositoryRevision(ui32 revision) {
    StatusRepository_->SetRevision(revision);
}

void TStatusNTickerHolder::SetStatusRepositoryTargetState(API::EPodAgentTargetState targetState) {
    StatusRepository_->SetTargetState(targetState);
}

void TStatusNTickerHolder::UpdateBoxIp6Subnet112Base(const TString& ip6Subnet112Base) {
    GetBoxStatusRepository()->UpdateIp6Subnet112Base(ip6Subnet112Base);
}

TString TStatusNTickerHolder::GetBoxIp6Subnet112Base() const {
    return GetBoxStatusRepository()->GetIp6Subnet112Base();
}

bool TStatusNTickerHolder::HasBoxIp6Address(const TString& address) {
    return GetBoxStatusRepository()->HasIp6Address(address);
}

bool TStatusNTickerHolder::HasCacheLayer(const TString& cacheLayerId, ui32 revision) const {
    TGuard<TMutex> gMutex(Mutex_);
    return GetLayerStatusRepository()->HasCacheObject(cacheLayerId, revision);
}

TString TStatusNTickerHolder::GetCacheLayerHash(const TString& cacheLayerId, ui32 revision) const {
    TGuard<TMutex> gMutex(Mutex_);
    return GetLayerStatusRepository()->GetCacheObjectHash(cacheLayerId, revision);
}

TVector<TStatusRepositoryCommon::TCacheObject> TStatusNTickerHolder::GetCacheLayerIdsAndRevisions() const {
    TGuard<TMutex> gMutex(Mutex_);
    return GetLayerStatusRepository()->GetCacheObjectIdsAndRevisions();
}

void TStatusNTickerHolder::AddCacheLayer(const TUpdateHolder::TLayerTarget& target) {
    Y_ENSURE(
        GetLayerTreeId(target.Meta_.DownloadHash_) == target.Tree_->GetTreeId()
        , "treeId '" << target.Tree_->GetTreeId()
            << "' does not match to DownloadHash_ '"
            << target.Meta_.DownloadHash_ << "'"
    );

    TGuard<TMutex> gMutex(Mutex_);

    Y_ENSURE(
        !GetLayerStatusRepository()->HasCacheObject(target.Meta_.Id_, target.Meta_.Revision_)
        , "CacheLayer '" << target.Meta_.Id_
            << "' with revision '" << target.Meta_.Revision_
            << "' already added"
    );
    Y_ENSURE(
        !Ticker_->HasTree(target.Tree_->GetTreeId()) || Ticker_->GetTreeHash(target.Tree_->GetTreeId()) == target.Meta_.DownloadHash_
        , "Ticker already has tree with id: '" << target.Tree_->GetTreeId() << "' but with a different hash"
    );

    GetLayerStatusRepository()->AddCacheObject(target.Meta_);
    TUnistatObjectHelper::Instance().AddCacheObject(
        target.Meta_
    );
    Ticker_->AddTree(target.Tree_, target.Meta_.DownloadHash_);
}

void TStatusNTickerHolder::RemoveCacheLayer(const TString& cacheLayerId, ui32 revision) {
    TGuard<TMutex> gMutex(Mutex_);

    Y_ENSURE(
        GetLayerStatusRepository()->HasCacheObject(cacheLayerId, revision)
        , "Layer '" << cacheLayerId
            << "' with revision '" << ToString(revision)
            << "' not found in LayerStatusRepository"
    );
    const TString treeId = GetLayerTreeId(GetLayerStatusRepository()->GetCacheObjectHash(cacheLayerId, revision));
    Y_ENSURE(
        Ticker_->HasTree(treeId)
        , "LayerStatusRepository and Ticker unsynchronized: Tree '"
            << treeId << "' found in LayerStatusRepository but not found in Ticker"
    );

    Ticker_->RemoveTree(treeId);
    TUnistatObjectHelper::Instance().RemoveCacheObject(
        cacheLayerId
        , revision
        , NStatusRepositoryTypes::EObjectType::LAYER
    );
    GetLayerStatusRepository()->RemoveCacheObject(cacheLayerId, revision);
}

bool TStatusNTickerHolder::HasLayer(const TString& layerId) const {
    TGuard<TMutex> gMutex(Mutex_);
    return GetLayerStatusRepository()->HasObject(layerId);
}

bool TStatusNTickerHolder::HasLayerDownloadHash(const TString& layerDownloadHash) const {
    TGuard<TMutex> gMutex(Mutex_);
    return GetLayerStatusRepository()->HasObjectHash(layerDownloadHash);
}

TVector<TString> TStatusNTickerHolder::GetLayerIds() const {
    TGuard<TMutex> gMutex(Mutex_);
    return GetLayerStatusRepository()->GetObjectIds();
}

void TStatusNTickerHolder::AddLayer(const TUpdateHolder::TLayerTarget& target) {
    Y_ENSURE(
        GetLayerTreeId(target.Meta_.DownloadHash_) == target.Tree_->GetTreeId()
        , "treeId '" << target.Tree_->GetTreeId()
            << "' does not match to DownloadHash_ '"
            << target.Meta_.DownloadHash_ << "'"
    );

    TGuard<TMutex> gMutex(Mutex_);

    Y_ENSURE(
        !GetLayerStatusRepository()->HasObject(target.Meta_.Id_)
        , "Layer '" << target.Meta_.Id_ << "' already added"
    );
    Y_ENSURE(
        !Ticker_->HasTree(target.Tree_->GetTreeId()) || Ticker_->GetTreeHash(target.Tree_->GetTreeId()) == target.Meta_.DownloadHash_
        , "Ticker already has tree with id: '" << target.Tree_->GetTreeId() << "' but with a different hash"
    );

    GetLayerStatusRepository()->AddObject(target.Meta_);
    TUnistatObjectHelper::Instance().AddObject(
        target.Meta_
    );
    Ticker_->AddTree(target.Tree_, target.Meta_.DownloadHash_);
}

void TStatusNTickerHolder::RemoveLayer(const TString& layerId) {
    TGuard<TMutex> gMutex(Mutex_);

    Y_ENSURE(
        GetLayerStatusRepository()->HasObject(layerId)
        , "Layer '" << layerId << "' not found in LayerStatusRepository"
    );
    const TString treeId = GetLayerTreeId(GetLayerStatusRepository()->GetObjectHash(layerId));
    Y_ENSURE(
        Ticker_->HasTree(treeId)
        , "LayerStatusRepository and Ticker unsynchronized: Tree '"
            << treeId << "' found in LayerStatusRepository but not found in Ticker"
    );

    CurrentGraph_.RemoveOutEdges(TGraph::TNode(layerId, TGraph::ENodeType::LAYER));
    TargetGraph_.RemoveOutEdges(TGraph::TNode(layerId, TGraph::ENodeType::LAYER));
    Ticker_->RemoveTree(treeId);
    TUnistatObjectHelper::Instance().RemoveObject(
        layerId
        , NStatusRepositoryTypes::EObjectType::LAYER
    );
    GetLayerStatusRepository()->RemoveObject(layerId);
}

void TStatusNTickerHolder::AddLayerWithTargetCheck(const TUpdateHolder::TLayerTarget& target) {
    Y_ENSURE(
        GetLayerTreeId(target.Meta_.DownloadHash_) == target.Tree_->GetTreeId()
        , "treeId '" << target.Tree_->GetTreeId()
            << "' does not match to DownloadHash_ '"
            << target.Meta_.DownloadHash_ << "'"
    );

    TGuard<TMutex> gMutex(Mutex_);

    Y_ENSURE(
        !GetLayerStatusRepository()->HasObject(target.Meta_.Id_)
        , "Layer '" << target.Meta_.Id_ << "' already added"
    );
    Y_ENSURE(
        !Ticker_->HasTree(target.Tree_->GetTreeId()) || Ticker_->GetTreeHash(target.Tree_->GetTreeId()) == target.Meta_.DownloadHash_
        , "Ticker already has tree with id: '" << target.Tree_->GetTreeId() << "' but with a different hash"
    );

    // 0 is a fake revision for waiting childs targets
    GetLayerStatusRepository()->AddObject(
        TLayerMeta(
            target.Meta_.Id_
            , 0
            , 0
            , target.Meta_.DownloadHash_
            , target.Meta_.RemoveSourceFileAfterImport_
        )
    );
    TUnistatObjectHelper::Instance().AddObject(
        target.Meta_
    );

    // We merge layers by hash and layer hasn't childs
    // We can't add fake tree because we wiil have two trees with same id but
    // with different hashes
    Ticker_->AddTree(target.Tree_, target.Meta_.DownloadHash_);
    UpdateLayerTarget(target);
    TryToUpdateLayerFromTarget(target.Meta_.Id_);
}

void TStatusNTickerHolder::SetLayerTargetRemove(const TString& layerId) {
    TGuard<TMutex> gMutex(Mutex_);

    Y_ENSURE(
        GetLayerStatusRepository()->HasObject(layerId)
        , "Layer '" << layerId << "' not found in LayerStatusRepository"
    );

    GetUpdateHolder()->SetLayerTarget(TUpdateHolder::TLayerTarget::GetTargetRemove(layerId));
    TargetGraph_.RemoveOutEdges(TGraph::TNode(layerId, TGraph::ENodeType::LAYER));
}

void TStatusNTickerHolder::UpdateLayerTarget(const TUpdateHolder::TLayerTarget& target) {
    Y_ENSURE(
        GetLayerTreeId(target.Meta_.DownloadHash_) == target.Tree_->GetTreeId()
        , "treeId '" << target.Tree_->GetTreeId()
            << "' does not match to DownloadHash_ '"
            << target.Meta_.DownloadHash_ << "'"
    );

    TGuard<TMutex> gMutex(Mutex_);

    Y_ENSURE(
        GetLayerStatusRepository()->HasObject(target.Meta_.Id_)
        , "Layer '" << target.Meta_.Id_ << "' not found in LayerStatusRepository"
    );
    Y_ENSURE(
        Ticker_->HasTree(GetLayerTreeId(GetLayerStatusRepository()->GetObjectHash(target.Meta_.Id_)))
        , "LayerStatusRepository and Ticker unsynchronized: Tree '"
            << target.Tree_->GetTreeId() << "' found in LayerStatusRepository but not found in Ticker"
    );

    if (!GetUpdateHolder()->GetUpdateHolderTarget()->LayerHasTarget(target.Meta_.Id_)
        && target.Meta_.DownloadHash_ == GetLayerStatusRepository()->GetObjectHash(target.Meta_.Id_)
        && target.Meta_.RemoveSourceFileAfterImport_ == GetLayerStatusRepository()->GetObjectRemoveSourceFileAfterImportFlag(target.Meta_.Id_)
        && CheckChildsHaveNoTargets(CurrentGraph_, TGraph::TNode(target.Meta_.Id_, TGraph::ENodeType::LAYER))
    ) {

        GetLayerStatusRepository()->UpdateObjectSpecTimestamp(target.Meta_.Id_, target.Meta_.SpecTimestamp_);
        GetLayerStatusRepository()->UpdateObjectRevision(target.Meta_.Id_, target.Meta_.Revision_);
    } else {
        GetUpdateHolder()->SetLayerTarget(target);

        TargetGraph_.RemoveOutEdges(TGraph::TNode(target.Meta_.Id_, TGraph::ENodeType::LAYER));
    }
}

TStatusNTickerHolder::TUpdateObjectResult TStatusNTickerHolder::TryToUpdateLayerFromTarget(const TString& layerId) {
    TGuard<TMutex> gMutex(Mutex_);

    if (!GetLayerStatusRepository()->HasObject(layerId)
        || !GetUpdateHolder()->GetUpdateHolderTarget()->LayerHasTarget(layerId)
    ) {
        return TStatusNTickerHolder::TUpdateObjectResult(TUpdateObjectResult::EUpdateResult::NO_TARGET, "");
    }
    if (const auto result = CheckReadyForUpdate(TGraph::TNode(layerId, TGraph::ENodeType::LAYER)); !result) {
        return TStatusNTickerHolder::TUpdateObjectResult(TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, result.Error());
    }

    // NOTE: order is important
    RemoveLayer(layerId);
    TUpdateHolder::TLayerTarget target = GetUpdateHolder()->GetAndRemoveLayerTarget(layerId);

    if (!target.TargetRemove_) {
        AddLayer(target);
        return TStatusNTickerHolder::TUpdateObjectResult(TUpdateObjectResult::EUpdateResult::UPDATED, "");
    } else {
        return TStatusNTickerHolder::TUpdateObjectResult(TUpdateObjectResult::EUpdateResult::REMOVED, "");
    }
}

bool TStatusNTickerHolder::HasCacheStaticResource(const TString& cacheStaticResourceId, ui32 revision) const {
    TGuard<TMutex> gMutex(Mutex_);
    return GetStaticResourceStatusRepository()->HasCacheObject(cacheStaticResourceId, revision);
}

TString TStatusNTickerHolder::GetCacheStaticResourceHash(const TString& cacheStaticResourceId, ui32 revision) const {
    TGuard<TMutex> gMutex(Mutex_);
    return GetStaticResourceStatusRepository()->GetCacheObjectHash(cacheStaticResourceId, revision);
}

TVector<TStatusRepositoryCommon::TCacheObject> TStatusNTickerHolder::GetCacheStaticResourceIdsAndRevisions() const {
    TGuard<TMutex> gMutex(Mutex_);
    return GetStaticResourceStatusRepository()->GetCacheObjectIdsAndRevisions();
}

void TStatusNTickerHolder::AddCacheStaticResource(const TUpdateHolder::TStaticResourceTarget& target) {
    Y_ENSURE(
        GetStaticResourceTreeId(target.Meta_.DownloadHash_) == target.Tree_->GetTreeId()
        , "treeId '" << target.Tree_->GetTreeId()
            << "' does not match to DownloadHash_ '"
            << target.Meta_.DownloadHash_ << "'"
    );

    TGuard<TMutex> gMutex(Mutex_);

    Y_ENSURE(
        !GetStaticResourceStatusRepository()->HasCacheObject(target.Meta_.Id_, target.Meta_.Revision_)
        , "CacheStaticResource '" << target.Meta_.Id_
            << "' with revision '" << target.Meta_.Revision_
            << "' already added"
    );
    Y_ENSURE(
        !Ticker_->HasTree(target.Tree_->GetTreeId()) || Ticker_->GetTreeHash(target.Tree_->GetTreeId()) == target.Meta_.DownloadHash_
        , "Ticker already has tree with id: '" << target.Tree_->GetTreeId() << "' but with a different hash"
    );

    GetStaticResourceStatusRepository()->AddCacheObject(target.Meta_);
    TUnistatObjectHelper::Instance().AddCacheObject(
        target.Meta_
    );
    Ticker_->AddTree(target.Tree_, target.Meta_.DownloadHash_);
}

void TStatusNTickerHolder::RemoveCacheStaticResource(const TString& cacheStaticResourceId, ui32 revision) {
    TGuard<TMutex> gMutex(Mutex_);

    Y_ENSURE(
        GetStaticResourceStatusRepository()->HasCacheObject(cacheStaticResourceId, revision)
        , "StaticResource '" << cacheStaticResourceId
            << "' with revision '" << ToString(revision)
            << "' not found in StaticResourceStatusRepository"
    );
    const TString treeId = GetStaticResourceTreeId(GetStaticResourceStatusRepository()->GetCacheObjectHash(cacheStaticResourceId, revision));
    Y_ENSURE(
        Ticker_->HasTree(treeId)
        , "StaticResourceStatusRepository and Ticker unsynchronized: Tree '"
            << treeId << "' found in StaticResourceStatusRepository but not found in Ticker"
    );

    Ticker_->RemoveTree(treeId);
    TUnistatObjectHelper::Instance().RemoveCacheObject(
        cacheStaticResourceId
        , revision
        , NStatusRepositoryTypes::EObjectType::STATIC_RESOURCE
    );
    GetStaticResourceStatusRepository()->RemoveCacheObject(cacheStaticResourceId, revision);
}

bool TStatusNTickerHolder::HasStaticResource(const TString& staticResourceId) const {
    TGuard<TMutex> gMutex(Mutex_);
    return GetStaticResourceStatusRepository()->HasObject(staticResourceId);
}

bool TStatusNTickerHolder::HasStaticResourceDownloadHash(const TString& staticResourceDownloadHash) const {
    TGuard<TMutex> gMutex(Mutex_);
    return GetStaticResourceStatusRepository()->HasObjectHash(staticResourceDownloadHash);
}

TVector<TString> TStatusNTickerHolder::GetStaticResourceIds() const {
    TGuard<TMutex> gMutex(Mutex_);
    return GetStaticResourceStatusRepository()->GetObjectIds();
}

void TStatusNTickerHolder::AddStaticResource(const TUpdateHolder::TStaticResourceTarget& target) {
    Y_ENSURE(
        GetStaticResourceTreeId(target.Meta_.DownloadHash_) == target.Tree_->GetTreeId()
        , "treeId '" << target.Tree_->GetTreeId()
            << "' does not match to DownloadHash_ '"
            << target.Meta_.DownloadHash_ << "'"
    );

    TGuard<TMutex> gMutex(Mutex_);

    Y_ENSURE(
        !GetStaticResourceStatusRepository()->HasObject(target.Meta_.Id_)
        , "StaticResource '" << target.Meta_.Id_ << "' already added"
    );
    Y_ENSURE(
        !Ticker_->HasTree(target.Tree_->GetTreeId()) || Ticker_->GetTreeHash(target.Tree_->GetTreeId()) == target.Meta_.DownloadHash_
        , "Ticker already has tree with id: '" << target.Tree_->GetTreeId() << "' but with a different hash"
    );

    GetStaticResourceStatusRepository()->AddObject(target.Meta_);
    TUnistatObjectHelper::Instance().AddObject(
        target.Meta_
    );
    Ticker_->AddTree(target.Tree_, target.Meta_.DownloadHash_);
}

void TStatusNTickerHolder::RemoveStaticResource(const TString& staticResourceId) {
    TGuard<TMutex> gMutex(Mutex_);

    Y_ENSURE(
        GetStaticResourceStatusRepository()->HasObject(staticResourceId)
        , "StaticResource '" << staticResourceId << "' not found in StaticResourceStatusRepository"
    );
    const TString treeId = GetStaticResourceTreeId(GetStaticResourceStatusRepository()->GetObjectHash(staticResourceId));
    Y_ENSURE(
        Ticker_->HasTree(treeId)
        , "StaticResourceStatusRepository and Ticker unsynchronized: Tree '"
            << treeId << "' found in StaticResourceStatusRepository but not found in Ticker"
    );

    CurrentGraph_.RemoveOutEdges(TGraph::TNode(staticResourceId, TGraph::ENodeType::STATIC_RESOURCE));
    TargetGraph_.RemoveOutEdges(TGraph::TNode(staticResourceId, TGraph::ENodeType::STATIC_RESOURCE));
    Ticker_->RemoveTree(treeId);
    TUnistatObjectHelper::Instance().RemoveObject(
        staticResourceId
        , NStatusRepositoryTypes::EObjectType::STATIC_RESOURCE
    );
    GetStaticResourceStatusRepository()->RemoveObject(staticResourceId);
}

void TStatusNTickerHolder::AddStaticResourceWithTargetCheck(const TUpdateHolder::TStaticResourceTarget& target) {
    Y_ENSURE(
        GetStaticResourceTreeId(target.Meta_.DownloadHash_) == target.Tree_->GetTreeId()
        , "treeId '" << target.Tree_->GetTreeId()
            << "' does not match to DownloadHash_ '"
            << target.Meta_.DownloadHash_ << "'"
    );

    TGuard<TMutex> gMutex(Mutex_);

    Y_ENSURE(
        !GetStaticResourceStatusRepository()->HasObject(target.Meta_.Id_)
        , "StaticResource '" << target.Meta_.Id_ << "' already added"
    );
    Y_ENSURE(
        !Ticker_->HasTree(target.Tree_->GetTreeId()) || Ticker_->GetTreeHash(target.Tree_->GetTreeId()) == target.Meta_.DownloadHash_
        , "Ticker already has tree with id: '" << target.Tree_->GetTreeId() << "' but with a different hash"
    );

    // 0 is a fake revision for waiting childs targets
    GetStaticResourceStatusRepository()->AddObject(
        TStaticResourceMeta(
            target.Meta_.Id_
            , 0
            , 0
            , target.Meta_.DownloadHash_
            , target.Meta_.CheckPeriodMs_
        )
    );
    TUnistatObjectHelper::Instance().AddObject(
        target.Meta_
    );

    // We merge staticResources by hash and staticResource hasn't childs
    // We can't add fake tree because we wiil have two trees with same id but
    // with different hashes
    Ticker_->AddTree(target.Tree_, target.Meta_.DownloadHash_);
    UpdateStaticResourceTarget(target);
    TryToUpdateStaticResourceFromTarget(target.Meta_.Id_);
}

void TStatusNTickerHolder::SetStaticResourceTargetRemove(const TString& staticResourceId) {
    TGuard<TMutex> gMutex(Mutex_);

    Y_ENSURE(
        GetStaticResourceStatusRepository()->HasObject(staticResourceId)
        , "StaticResource '" << staticResourceId << "' not found in StaticResourceStatusRepository"
    );

    GetUpdateHolder()->SetStaticResourceTarget(TUpdateHolder::TStaticResourceTarget::GetTargetRemove(staticResourceId));
    TargetGraph_.RemoveOutEdges(TGraph::TNode(staticResourceId, TGraph::ENodeType::STATIC_RESOURCE));
}

void TStatusNTickerHolder::UpdateStaticResourceTarget(const TUpdateHolder::TStaticResourceTarget& target) {
    Y_ENSURE(
        GetStaticResourceTreeId(target.Meta_.DownloadHash_) == target.Tree_->GetTreeId()
        , "treeId '" << target.Tree_->GetTreeId()
            << "' does not match to DownloadHash_ '"
            << target.Meta_.DownloadHash_ << "'"
    );

    TGuard<TMutex> gMutex(Mutex_);

    Y_ENSURE(
        GetStaticResourceStatusRepository()->HasObject(target.Meta_.Id_)
        , "StaticResource '" << target.Meta_.Id_ << "' not found in StaticResourceStatusRepository"
    );
    Y_ENSURE(
        Ticker_->HasTree(GetStaticResourceTreeId(GetStaticResourceStatusRepository()->GetObjectHash(target.Meta_.Id_)))
        , "StaticResourceStatusRepository and Ticker unsynchronized: Tree '"
            << target.Tree_->GetTreeId() << "' found in StaticResourceStatusRepository but not found in Ticker"
    );

    if (!GetUpdateHolder()->GetUpdateHolderTarget()->StaticResourceHasTarget(target.Meta_.Id_)
        && target.Meta_.DownloadHash_ == GetStaticResourceStatusRepository()->GetObjectHash(target.Meta_.Id_)
        && target.Meta_.CheckPeriodMs_ == GetStaticResourceStatusRepository()->GetObjectCheckPeriodMs(target.Meta_.Id_)
        && CheckChildsHaveNoTargets(CurrentGraph_, TGraph::TNode(target.Meta_.Id_, TGraph::ENodeType::STATIC_RESOURCE))
    ) {

        GetStaticResourceStatusRepository()->UpdateObjectSpecTimestamp(target.Meta_.Id_, target.Meta_.SpecTimestamp_);
        GetStaticResourceStatusRepository()->UpdateObjectRevision(target.Meta_.Id_, target.Meta_.Revision_);
    } else {
        GetUpdateHolder()->SetStaticResourceTarget(target);

        TargetGraph_.RemoveOutEdges(TGraph::TNode(target.Meta_.Id_, TGraph::ENodeType::STATIC_RESOURCE));
    }
}

TStatusNTickerHolder::TUpdateObjectResult TStatusNTickerHolder::TryToUpdateStaticResourceFromTarget(const TString& staticResourceId) {
    TGuard<TMutex> gMutex(Mutex_);

    if (!GetStaticResourceStatusRepository()->HasObject(staticResourceId)
        || !GetUpdateHolder()->GetUpdateHolderTarget()->StaticResourceHasTarget(staticResourceId)
    ) {
        return TStatusNTickerHolder::TUpdateObjectResult(TUpdateObjectResult::EUpdateResult::NO_TARGET, "");
    }
    if (const auto result = CheckReadyForUpdate(TGraph::TNode(staticResourceId, TGraph::ENodeType::STATIC_RESOURCE)); !result) {
        return TStatusNTickerHolder::TUpdateObjectResult(TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, result.Error());
    }

    // NOTE: order is important
    RemoveStaticResource(staticResourceId);
    TUpdateHolder::TStaticResourceTarget target = GetUpdateHolder()->GetAndRemoveStaticResourceTarget(staticResourceId);

    if (!target.TargetRemove_) {
        AddStaticResource(target);
        return TStatusNTickerHolder::TUpdateObjectResult(TUpdateObjectResult::EUpdateResult::UPDATED, "");
    } else {
        return TStatusNTickerHolder::TUpdateObjectResult(TUpdateObjectResult::EUpdateResult::REMOVED, "");
    }
}

bool TStatusNTickerHolder::HasVolume(const TString& volumeId) const {
    TGuard<TMutex> gMutex(Mutex_);
    return GetVolumeStatusRepository()->HasObject(volumeId);
}

TVector<TString> TStatusNTickerHolder::GetVolumeIds() const {
    TGuard<TMutex> gMutex(Mutex_);
    return GetVolumeStatusRepository()->GetObjectIds();
}

void TStatusNTickerHolder::AddVolume(const TUpdateHolder::TVolumeTarget& target) {
    Y_ENSURE(
        GetVolumeTreeId(target.Meta_.Id_) == target.Tree_->GetTreeId()
        , "treeId '" << target.Tree_->GetTreeId()
            << "' does not match to volumeId '" << target.Meta_.Id_ << "'"
    );

    TGuard<TMutex> gMutex(Mutex_);

    Y_ENSURE(
        !GetVolumeStatusRepository()->HasObject(target.Meta_.Id_)
        , "Volume '" << target.Meta_.Id_ << "' already added"
    );
    Y_ENSURE(
        !Ticker_->HasTree(target.Tree_->GetTreeId())
        , "VolumeStatusRepository and Ticker unsynchronized: Tree '"
            << target.Tree_->GetTreeId() << "' found in Ticker but not found in VolumeStatusRepository"
    );

    for (const TString& layerRef : target.Meta_.LayerRefs_) {
        Y_ENSURE(
            GetLayerStatusRepository()->HasObject(layerRef)
            , "Volume '" << target.Meta_.Id_ << "' layerRef '" << layerRef << "' not found in LayerStatusRepository"
        );
    }

    for (const TString& staticResourceRef : target.Meta_.StaticResourceRefs_) {
        Y_ENSURE(
            GetStaticResourceStatusRepository()->HasObject(staticResourceRef)
            , "Volume '" << target.Meta_.Id_ << "' staticResourceRef '" << staticResourceRef << "' not found in StaticResourceStatusRepository"
        );
    }

    GetVolumeStatusRepository()->AddObject(target.Meta_);
    TUnistatObjectHelper::Instance().AddObject(
        target.Meta_
    );
    Ticker_->AddTree(target.Tree_, target.Hash_);

    for (const TString& layerRef : target.Meta_.LayerRefs_) {
        CurrentGraph_.AddEdge(TGraph::TNode(target.Meta_.Id_, TGraph::ENodeType::VOLUME), TGraph::TNode(layerRef, TGraph::ENodeType::LAYER));
        TargetGraph_.AddEdge(TGraph::TNode(target.Meta_.Id_, TGraph::ENodeType::VOLUME), TGraph::TNode(layerRef, TGraph::ENodeType::LAYER));
    }

    for (const TString& staticResourceRef : target.Meta_.StaticResourceRefs_) {
        CurrentGraph_.AddEdge(TGraph::TNode(target.Meta_.Id_, TGraph::ENodeType::VOLUME), TGraph::TNode(staticResourceRef, TGraph::ENodeType::STATIC_RESOURCE));
        TargetGraph_.AddEdge(TGraph::TNode(target.Meta_.Id_, TGraph::ENodeType::VOLUME), TGraph::TNode(staticResourceRef, TGraph::ENodeType::STATIC_RESOURCE));
    }
}

void TStatusNTickerHolder::RemoveVolume(const TString& volumeId) {
    TGuard<TMutex> gMutex(Mutex_);
    const TString treeId = GetVolumeTreeId(volumeId);

    Y_ENSURE(
        GetVolumeStatusRepository()->HasObject(volumeId)
        , "Volume '" << volumeId << "' not found in VolumeStatusRepository"
    );
    Y_ENSURE(
        Ticker_->HasTree(treeId)
        , "VolumeStatusRepository and Ticker unsynchronized: Tree '"
            << treeId << "' found in VolumeStatusRepository but not found in Ticker"
    );

    CurrentGraph_.RemoveOutEdges(TGraph::TNode(volumeId, TGraph::ENodeType::VOLUME));
    TargetGraph_.RemoveOutEdges(TGraph::TNode(volumeId, TGraph::ENodeType::VOLUME));
    Ticker_->RemoveTree(treeId);
    TUnistatObjectHelper::Instance().RemoveObject(
        volumeId
        , NStatusRepositoryTypes::EObjectType::VOLUME
    );
    GetVolumeStatusRepository()->RemoveObject(volumeId);
}


void TStatusNTickerHolder::AddVolumeWithTargetCheck(const TUpdateHolder::TVolumeTarget& target) {
    Y_ENSURE(
        GetVolumeTreeId(target.Meta_.Id_) == target.Tree_->GetTreeId()
        , "treeId '" << target.Tree_->GetTreeId()
            << "' does not match to volumeId '" << target.Meta_.Id_ << "'"
    );

    TGuard<TMutex> gMutex(Mutex_);

    Y_ENSURE(
        !GetVolumeStatusRepository()->HasObject(target.Meta_.Id_)
        , "Volume '" << target.Meta_.Id_ << "' already added"
    );
    Y_ENSURE(
        !Ticker_->HasTree(target.Tree_->GetTreeId())
        , "VolumeStatusRepository and Ticker unsynchronized: Tree '"
            << target.Tree_->GetTreeId() << "' found in Ticker but not found in VolumeStatusRepository"
    );

    for (const TString& layerRef : target.Meta_.LayerRefs_) {
        Y_ENSURE(
            GetLayerStatusRepository()->HasObject(layerRef)
            , "Volume '" << target.Meta_.Id_ << "' layerRef '" << layerRef << "' not found in LayerStatusRepository"
        );
    }

    for (const TString& staticResourceRef : target.Meta_.StaticResourceRefs_) {
        Y_ENSURE(
            GetStaticResourceStatusRepository()->HasObject(staticResourceRef)
            , "Volume '" << target.Meta_.Id_ << "' staticResourceRef '" << staticResourceRef << "' not found in StaticResourceStatusRepository"
        );
    }

    // 0 is a fake revision for waiting childs targets
    GetVolumeStatusRepository()->AddObject(
        TVolumeMeta(
            target.Meta_.Id_
            , 0
            , 0
            , target.Meta_.LayerRefs_
            , target.Meta_.StaticResourceRefs_
        )
    );
    TUnistatObjectHelper::Instance().AddObject(
        target.Meta_
    );
    Ticker_->AddTree(GetEmptyTree(GetVolumeTreeId(target.Meta_.Id_)), "");
    UpdateVolumeTarget(target);
    TryToUpdateVolumeFromTarget(target.Meta_.Id_);
}

void TStatusNTickerHolder::SetVolumeTargetRemove(const TString& volumeId) {
    TGuard<TMutex> gMutex(Mutex_);

    Y_ENSURE(GetVolumeStatusRepository()->HasObject(volumeId), "Volume '" << volumeId << "' not found in VolumeStatusRepository");

    GetUpdateHolder()->SetVolumeTarget(TUpdateHolder::TVolumeTarget::GetTargetRemove(volumeId));
    TargetGraph_.RemoveOutEdges(TGraph::TNode(volumeId, TGraph::ENodeType::VOLUME));
}

void TStatusNTickerHolder::UpdateVolumeTarget(const TUpdateHolder::TVolumeTarget& target) {
    Y_ENSURE(
        GetVolumeTreeId(target.Meta_.Id_) == target.Tree_->GetTreeId()
        , "treeId '" << target.Tree_->GetTreeId()
            << "' does not match to volumeId '" << target.Meta_.Id_ << "'"
    );

    TGuard<TMutex> gMutex(Mutex_);

    Y_ENSURE(
        GetVolumeStatusRepository()->HasObject(target.Meta_.Id_)
        , "Volume '" << target.Meta_.Id_ << "' not found in VolumeStatusRepository"
    );
    Y_ENSURE(
        Ticker_->HasTree(target.Tree_->GetTreeId())
        , "VolumeStatusRepository and Ticker unsynchronized: Tree '"
            << target.Tree_->GetTreeId() << "' found in VolumeStatusRepository but not found in Ticker"
    );

    for (const TString& layerRef : target.Meta_.LayerRefs_) {
        Y_ENSURE(
            GetLayerStatusRepository()->HasObject(layerRef)
            , "Volume '" << target.Meta_.Id_ << "' layerRef '" << layerRef << "' not found in LayerStatusRepository"
        );
    }

    for (const TString& staticResourceRef : target.Meta_.StaticResourceRefs_) {
        Y_ENSURE(
                GetStaticResourceStatusRepository()->HasObject(staticResourceRef)
        , "Volume '" << target.Meta_.Id_ << "' staticResourceRef '" << staticResourceRef << "' not found in StaticResourceStatusRepository"
        );
    }

    if (!GetUpdateHolder()->GetUpdateHolderTarget()->VolumeHasTarget(target.Meta_.Id_)
        && target.Hash_ == Ticker_->GetTreeHash(target.Tree_->GetTreeId())
        && CheckChildsHaveNoTargets(CurrentGraph_, TGraph::TNode(target.Meta_.Id_, TGraph::ENodeType::VOLUME))
    ) {

        GetVolumeStatusRepository()->UpdateObjectSpecTimestamp(target.Meta_.Id_, target.Meta_.SpecTimestamp_);
        GetVolumeStatusRepository()->UpdateObjectRevision(target.Meta_.Id_, target.Meta_.Revision_);
    } else {
        GetUpdateHolder()->SetVolumeTarget(target);

        TargetGraph_.RemoveOutEdges(TGraph::TNode(target.Meta_.Id_, TGraph::ENodeType::VOLUME));

        for (const TString& layerRef : target.Meta_.LayerRefs_) {
            TargetGraph_.AddEdge(TGraph::TNode(target.Meta_.Id_, TGraph::ENodeType::VOLUME), TGraph::TNode(layerRef, TGraph::ENodeType::LAYER));
        }

        for (const TString& staticResourceRef : target.Meta_.StaticResourceRefs_) {
            TargetGraph_.AddEdge(TGraph::TNode(target.Meta_.Id_, TGraph::ENodeType::VOLUME), TGraph::TNode(staticResourceRef, TGraph::ENodeType::STATIC_RESOURCE));
        }
    }
}

TStatusNTickerHolder::TUpdateObjectResult TStatusNTickerHolder::TryToUpdateVolumeFromTarget(const TString& volumeId) {
    TGuard<TMutex> gMutex(Mutex_);

    if (!GetVolumeStatusRepository()->HasObject(volumeId)
        || !GetUpdateHolder()->GetUpdateHolderTarget()->VolumeHasTarget(volumeId)
    ) {
        return TStatusNTickerHolder::TUpdateObjectResult(TUpdateObjectResult::EUpdateResult::NO_TARGET, "");
    }
    if (const auto result = CheckReadyForUpdate(TGraph::TNode(volumeId, TGraph::ENodeType::VOLUME)); !result) {
        return TStatusNTickerHolder::TUpdateObjectResult(TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, result.Error());
    }

    // NOTE: order is important
    RemoveVolume(volumeId);
    TUpdateHolder::TVolumeTarget target = GetUpdateHolder()->GetAndRemoveVolumeTarget(volumeId);

    if (!target.TargetRemove_) {
        AddVolume(target);
        return TStatusNTickerHolder::TUpdateObjectResult(TUpdateObjectResult::EUpdateResult::UPDATED, "");
    } else {
        return TStatusNTickerHolder::TUpdateObjectResult(TUpdateObjectResult::EUpdateResult::REMOVED, "");
    }
}

bool TStatusNTickerHolder::HasBox(const TString& boxId) const {
    TGuard<TMutex> gMutex(Mutex_);
    return GetBoxStatusRepository()->HasObject(boxId);
}

TVector<TString> TStatusNTickerHolder::GetBoxIds() const {
    TGuard<TMutex> gMutex(Mutex_);
    return GetBoxStatusRepository()->GetObjectIds();
}

void TStatusNTickerHolder::AddBox(const TUpdateHolder::TBoxTarget& target) {
    Y_ENSURE(
        GetBoxTreeId(target.Meta_.Id_) == target.Tree_->GetTreeId()
        , "treeId '" << target.Tree_->GetTreeId()
            << "' does not match to boxId '" << target.Meta_.Id_ << "'"
    );

    TGuard<TMutex> gMutex(Mutex_);

    Y_ENSURE(
        !GetBoxStatusRepository()->HasObject(target.Meta_.Id_)
        , "Box '" << target.Meta_.Id_ << "' already added"
    );
    Y_ENSURE(
        !Ticker_->HasTree(target.Tree_->GetTreeId())
        , "BoxStatusRepository and Ticker unsynchronized: Tree '"
            << target.Tree_->GetTreeId() << "' found in Ticker but not found in BoxStatusRepository"
    );

    for (const TString& rootfsLayerRef : target.Meta_.RootfsLayerRefs_) {
        Y_ENSURE(
            GetLayerStatusRepository()->HasObject(rootfsLayerRef)
            , "Box '" << target.Meta_.Id_ << "' rootfsLayerRef '" << rootfsLayerRef << "' not found in LayerStatusRepository"
        );
    }
    for (const TString& staticResourceRef : target.Meta_.StaticResourceRefs_) {
        Y_ENSURE(
            GetStaticResourceStatusRepository()->HasObject(staticResourceRef)
            , "Box '" << target.Meta_.Id_ << "' staticResourceRef '" << staticResourceRef << "' not found in StaticResourceStatusRepository"
        );
    }
    for (const TString& volumeRef : target.Meta_.VolumeRefs_) {
        Y_ENSURE(
            GetVolumeStatusRepository()->HasObject(volumeRef)
            , "Box '" << target.Meta_.Id_ << "' volumeRef '" << volumeRef << "' not found in VolumeStatusRepository"
        );
    }

    GetBoxStatusRepository()->AddObject(target.Meta_);
    TUnistatObjectHelper::Instance().AddObject(
        target.Meta_
    );
    BoxSystemLogsSender_->Add(target.Meta_.Id_, new TMockSystemLogsSession());
    Ticker_->AddTree(target.Tree_, target.Hash_);

    for (const TString& rootfsLayerRef : target.Meta_.RootfsLayerRefs_) {
        CurrentGraph_.AddEdge(TGraph::TNode(target.Meta_.Id_, TGraph::ENodeType::BOX), TGraph::TNode(rootfsLayerRef, TGraph::ENodeType::LAYER));
        TargetGraph_.AddEdge(TGraph::TNode(target.Meta_.Id_, TGraph::ENodeType::BOX), TGraph::TNode(rootfsLayerRef, TGraph::ENodeType::LAYER));
    }
    for (const TString& staticResourceRef : target.Meta_.StaticResourceRefs_) {
        CurrentGraph_.AddEdge(TGraph::TNode(target.Meta_.Id_, TGraph::ENodeType::BOX), TGraph::TNode(staticResourceRef, TGraph::ENodeType::STATIC_RESOURCE));
        TargetGraph_.AddEdge(TGraph::TNode(target.Meta_.Id_, TGraph::ENodeType::BOX), TGraph::TNode(staticResourceRef, TGraph::ENodeType::STATIC_RESOURCE));
    }
    for (const TString& volumeRef : target.Meta_.VolumeRefs_) {
        CurrentGraph_.AddEdge(TGraph::TNode(target.Meta_.Id_, TGraph::ENodeType::BOX), TGraph::TNode(volumeRef, TGraph::ENodeType::VOLUME));
        TargetGraph_.AddEdge(TGraph::TNode(target.Meta_.Id_, TGraph::ENodeType::BOX), TGraph::TNode(volumeRef, TGraph::ENodeType::VOLUME));
    }
}

void TStatusNTickerHolder::RemoveBox(const TString& boxId) {
    TGuard<TMutex> gMutex(Mutex_);
    const TString treeId = GetBoxTreeId(boxId);

    Y_ENSURE(
        GetBoxStatusRepository()->HasObject(boxId)
        , "Box '" << boxId << "' not found in BoxStatusRepository"
    );
    Y_ENSURE(
        Ticker_->HasTree(treeId)
        , "BoxStatusRepository and Ticker unsynchronized: Tree '"
            << treeId << "' found in BoxStatusRepository but not found in Ticker"
    );

    CurrentGraph_.RemoveOutEdges(TGraph::TNode(boxId, TGraph::ENodeType::BOX));
    TargetGraph_.RemoveOutEdges(TGraph::TNode(boxId, TGraph::ENodeType::BOX));
    Ticker_->RemoveTree(treeId);
    TUnistatObjectHelper::Instance().RemoveObject(
        boxId
        , NStatusRepositoryTypes::EObjectType::BOX
    );
    BoxSystemLogsSender_->Remove(boxId);
    GetBoxStatusRepository()->RemoveObject(boxId);
}

void TStatusNTickerHolder::AddBoxWithTargetCheck(const TUpdateHolder::TBoxTarget& target) {
    Y_ENSURE(
        GetBoxTreeId(target.Meta_.Id_) == target.Tree_->GetTreeId()
        , "treeId '" << target.Tree_->GetTreeId()
            << "' does not match to boxId '" << target.Meta_.Id_ << "'"
    );

    TGuard<TMutex> gMutex(Mutex_);

    Y_ENSURE(
        !GetBoxStatusRepository()->HasObject(target.Meta_.Id_)
        , "Box '" << target.Meta_.Id_ << "' already added"
    );
    Y_ENSURE(
        !Ticker_->HasTree(target.Tree_->GetTreeId())
        , "BoxStatusRepository and Ticker unsynchronized: Tree '"
            << target.Tree_->GetTreeId() << "' found in Ticker but not found in BoxStatusRepository"
    );

    for (const TString& rootfsLayerRef : target.Meta_.RootfsLayerRefs_) {
        Y_ENSURE(
            GetLayerStatusRepository()->HasObject(rootfsLayerRef)
            , "Box '" << target.Meta_.Id_ << "' rootfsLayerRef '" << rootfsLayerRef << "' not found in LayerStatusRepository"
        );
    }
    for (const TString& staticResourceRef : target.Meta_.StaticResourceRefs_) {
        Y_ENSURE(
            GetStaticResourceStatusRepository()->HasObject(staticResourceRef)
            , "Box '" << target.Meta_.Id_ << "' staticResourceRef '" << staticResourceRef << "' not found in StaticResourceStatusRepository"
        );
    }
    for (const TString& volumeRef : target.Meta_.VolumeRefs_) {
        Y_ENSURE(
            GetVolumeStatusRepository()->HasObject(volumeRef)
            , "Box '" << target.Meta_.Id_ << "' volumeRef '" << volumeRef << "' not found in VolumeStatusRepository"
        );
    }

    // 0 is a fake revision for waiting childs targets
    GetBoxStatusRepository()->AddObject(
        TBoxMeta(
            target.Meta_.Id_
            , 0
            , 0
            , target.Meta_.RootfsLayerRefs_
            , target.Meta_.StaticResourceRefs_
            , target.Meta_.VolumeRefs_
            , target.Meta_.MetaContainer_
            , target.Meta_.InitContainers_
            , target.Meta_.BoxSpecificType_
        )
    );
    TUnistatObjectHelper::Instance().AddObject(
        target.Meta_
    );
    Ticker_->AddTree(GetEmptyTree(GetBoxTreeId(target.Meta_.Id_)), "");
    UpdateBoxTarget(target);
    TryToUpdateBoxFromTarget(target.Meta_.Id_);
}

void TStatusNTickerHolder::SetBoxTargetRemove(const TString& boxId) {
    TGuard<TMutex> gMutex(Mutex_);

    Y_ENSURE(
        GetBoxStatusRepository()->HasObject(boxId)
        , "Box '" << boxId << "' not found in BoxStatusRepository"
    );

    GetUpdateHolder()->SetBoxTarget(TUpdateHolder::TBoxTarget::GetTargetRemove(boxId));
    TargetGraph_.RemoveOutEdges(TGraph::TNode(boxId, TGraph::ENodeType::BOX));
}

void TStatusNTickerHolder::UpdateBoxTarget(const TUpdateHolder::TBoxTarget& target) {
    Y_ENSURE(
        GetBoxTreeId(target.Meta_.Id_) == target.Tree_->GetTreeId()
        , "treeId '" << target.Tree_->GetTreeId()
            << "' does not match to boxId '" << target.Meta_.Id_ << "'"
    );

    TGuard<TMutex> gMutex(Mutex_);

    Y_ENSURE(
        GetBoxStatusRepository()->HasObject(target.Meta_.Id_)
        , "Box '" << target.Meta_.Id_ << "' not found in BoxStatusRepository"
    );
    Y_ENSURE(
        Ticker_->HasTree(target.Tree_->GetTreeId())
        , "BoxStatusRepository and Ticker unsynchronized: Tree '"
            << target.Tree_->GetTreeId() << "' found in BoxStatusRepository but not found in Ticker"
    );

    for (const TString& rootfsLayerRef : target.Meta_.RootfsLayerRefs_) {
        Y_ENSURE(
            GetLayerStatusRepository()->HasObject(rootfsLayerRef)
            , "Box '" << target.Meta_.Id_ << "' rootfsLayerRef '" << rootfsLayerRef << "' not found in LayerStatusRepository"
        );
    }
    for (const TString& staticResourceRef : target.Meta_.StaticResourceRefs_) {
        Y_ENSURE(
            GetStaticResourceStatusRepository()->HasObject(staticResourceRef)
            , "Box '" << target.Meta_.Id_ << "' staticResourceRef '" << staticResourceRef << "' not found in StaticResourceStatusRepository"
        );
    }
    for (const TString& volumeRef : target.Meta_.VolumeRefs_) {
        Y_ENSURE(
            GetVolumeStatusRepository()->HasObject(volumeRef)
            , "Box '" << target.Meta_.Id_ << "' volumeRef '" << volumeRef << "' not found in VolumeStatusRepository"
        );
    }

    if (!GetUpdateHolder()->GetUpdateHolderTarget()->BoxHasTarget(target.Meta_.Id_)
        && target.Hash_ == Ticker_->GetTreeHash(target.Tree_->GetTreeId())
        && CheckChildsHaveNoTargets(CurrentGraph_, TGraph::TNode(target.Meta_.Id_, TGraph::ENodeType::BOX))
    ) {

        GetBoxStatusRepository()->UpdateObjectSpecTimestamp(target.Meta_.Id_, target.Meta_.SpecTimestamp_);
        GetBoxStatusRepository()->UpdateObjectRevision(target.Meta_.Id_, target.Meta_.Revision_);
    } else {
        GetUpdateHolder()->SetBoxTarget(target);

        TargetGraph_.RemoveOutEdges(TGraph::TNode(target.Meta_.Id_, TGraph::ENodeType::BOX));

        for (const TString& rootfsLayerRef : target.Meta_.RootfsLayerRefs_) {
            TargetGraph_.AddEdge(TGraph::TNode(target.Meta_.Id_, TGraph::ENodeType::BOX), TGraph::TNode(rootfsLayerRef, TGraph::ENodeType::LAYER));
        }
        for (const TString& staticResourceRef : target.Meta_.StaticResourceRefs_) {
            TargetGraph_.AddEdge(TGraph::TNode(target.Meta_.Id_, TGraph::ENodeType::BOX), TGraph::TNode(staticResourceRef, TGraph::ENodeType::STATIC_RESOURCE));
        }
        for (const TString& volumeRef : target.Meta_.VolumeRefs_) {
            TargetGraph_.AddEdge(TGraph::TNode(target.Meta_.Id_, TGraph::ENodeType::BOX), TGraph::TNode(volumeRef, TGraph::ENodeType::VOLUME));
        }
    }
}

TStatusNTickerHolder::TUpdateObjectResult TStatusNTickerHolder::TryToUpdateBoxFromTarget(const TString& boxId) {
    TGuard<TMutex> gMutex(Mutex_);

    if (!GetBoxStatusRepository()->HasObject(boxId)
        || !GetUpdateHolder()->GetUpdateHolderTarget()->BoxHasTarget(boxId)
    ) {
        return TStatusNTickerHolder::TUpdateObjectResult(TUpdateObjectResult::EUpdateResult::NO_TARGET, "");
    }
    if (const auto result = CheckReadyForUpdate(TGraph::TNode(boxId, TGraph::ENodeType::BOX)); !result) {
        return TStatusNTickerHolder::TUpdateObjectResult(TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, result.Error());
    }

    // NOTE: order is important
    RemoveBox(boxId);
    TUpdateHolder::TBoxTarget target = GetUpdateHolder()->GetAndRemoveBoxTarget(boxId);

    if (!target.TargetRemove_) {
        AddBox(target);
        return TStatusNTickerHolder::TUpdateObjectResult(TUpdateObjectResult::EUpdateResult::UPDATED, "");
    } else {
        return TStatusNTickerHolder::TUpdateObjectResult(TUpdateObjectResult::EUpdateResult::REMOVED, "");
    }
}

bool TStatusNTickerHolder::HasWorkload(const TString& workloadId) const {
    TGuard<TMutex> gMutex(Mutex_);
    return GetWorkloadStatusRepository()->HasObject(workloadId);
}

TVector<TString> TStatusNTickerHolder::GetWorkloadIds() const {
    TGuard<TMutex> gMutex(Mutex_);
    return GetWorkloadStatusRepository()->GetObjectIds();
}

void TStatusNTickerHolder::AddWorkload(const TUpdateHolder::TWorkloadTarget& target) {
    Y_ENSURE(
        GetWorkloadTreeId(target.Meta_.Id_) == target.Tree_->GetTreeId()
        , "treeId '" << target.Tree_->GetTreeId()
            << "' does not match to workloadId '" << target.Meta_.Id_ << "'"
    );

    TGuard<TMutex> gMutex(Mutex_);

    Y_ENSURE(
        !GetWorkloadStatusRepository()->HasObject(target.Meta_.Id_)
        , "Workload '" << target.Meta_.Id_ << "' already added"
    );
    Y_ENSURE(
        !WorkloadStatusRepositoryInternal_->HasObject(target.Meta_.Id_)
        , "WorkloadStatusRepository and WorkloadStatusRepositoryInternal unsynchronized: Workload '"
            << target.Meta_.Id_ << "' found in WorkloadStatusRepositoryInternal but not found in WorkloadStatusRepository"
    );
    Y_ENSURE(
        !Ticker_->HasTree(target.Tree_->GetTreeId())
        , "WorkloadStatusRepository and Ticker unsynchronized: Tree '"
            << target.Tree_->GetTreeId() << "' found in Ticker but not found in WorkloadStatusRepository"
    );
    if (IsBoxAgentMode_) {
        Y_ENSURE(
            target.Meta_.BoxRef_.empty()
            , "PodAgent is in box agent mode. Workload must not have boxRef but it has: '" << target.Meta_.BoxRef_ << "'"
        );
    } else {
        Y_ENSURE(
            GetBoxStatusRepository()->HasObject(target.Meta_.BoxRef_)
            , "Workload '" << target.Meta_.Id_ << "' boxRef '"
                << target.Meta_.BoxRef_ << "' not found in BoxStatusRepository"
        );
    }

    GetWorkloadStatusRepository()->AddObject(target.Meta_);
    WorkloadStatusRepositoryInternal_->AddObject(target.Meta_.Id_);
    TUnistatObjectHelper::Instance().AddObject(
        target.Meta_
    );
    WorkloadSystemLogsSender_->Add(target.Meta_.Id_, new TMockSystemLogsSession());
    Ticker_->AddTree(target.Tree_, target.Hash_);
    if (!IsBoxAgentMode_) {
        CurrentGraph_.AddEdge(TGraph::TNode(target.Meta_.Id_, TGraph::ENodeType::WORKLOAD), TGraph::TNode(target.Meta_.BoxRef_, TGraph::ENodeType::BOX));
        TargetGraph_.AddEdge(TGraph::TNode(target.Meta_.Id_, TGraph::ENodeType::WORKLOAD), TGraph::TNode(target.Meta_.BoxRef_, TGraph::ENodeType::BOX));
    }
}

void TStatusNTickerHolder::RemoveWorkload(const TString& workloadId) {
    TGuard<TMutex> gMutex(Mutex_);
    const TString treeId = GetWorkloadTreeId(workloadId);

    Y_ENSURE(
        GetWorkloadStatusRepository()->HasObject(workloadId)
        , "Workload '" << workloadId << "' not found in WorkloadStatusRepository"
    );
    Y_ENSURE(
        WorkloadStatusRepositoryInternal_->HasObject(workloadId)
        , "WorkloadStatusRepository and WorkloadStatusRepositoryInternal unsynchronized: Workload '"
            << workloadId << "' found in WorkloadStatusRepository but not found in WorkloadStatusRepositoryInternal"
    );
    Y_ENSURE(
        Ticker_->HasTree(treeId)
        , "WorkloadStatusRepository and Ticker unsynchronized: Tree '"
            << treeId << "' found in WorkloadStatusRepository but not found in Ticker"
    );

    CurrentGraph_.RemoveOutEdges(TGraph::TNode(workloadId, TGraph::ENodeType::WORKLOAD));
    TargetGraph_.RemoveOutEdges(TGraph::TNode(workloadId, TGraph::ENodeType::WORKLOAD));
    Ticker_->RemoveTree(treeId);
    TUnistatObjectHelper::Instance().RemoveObject(
        workloadId
        , NStatusRepositoryTypes::EObjectType::WORKLOAD
    );
    WorkloadSystemLogsSender_->Remove(workloadId);
    WorkloadStatusRepositoryInternal_->RemoveObject(workloadId);
    GetWorkloadStatusRepository()->RemoveObject(workloadId);
}

void TStatusNTickerHolder::AddWorkloadWithTargetCheck(const TUpdateHolder::TWorkloadTarget& target) {
    Y_ENSURE(
        GetWorkloadTreeId(target.Meta_.Id_) == target.Tree_->GetTreeId()
        , "treeId '" << target.Tree_->GetTreeId()
            << "' does not match to workloadId '" << target.Meta_.Id_ << "'"
    );

    TGuard<TMutex> gMutex(Mutex_);

    Y_ENSURE(
        !GetWorkloadStatusRepository()->HasObject(target.Meta_.Id_)
        , "Workload '" << target.Meta_.Id_ << "' already added"
    );
    Y_ENSURE(
        !WorkloadStatusRepositoryInternal_->HasObject(target.Meta_.Id_)
        , "WorkloadStatusRepository and WorkloadStatusRepositoryInternal unsynchronized: Workload '"
            << target.Meta_.Id_ << "' found in WorkloadStatusRepositoryInternal but not found in WorkloadStatusRepository"
    );
    Y_ENSURE(
        !Ticker_->HasTree(target.Tree_->GetTreeId())
        , "WorkloadStatusRepository and Ticker unsynchronized: Tree '"
            << target.Tree_->GetTreeId() << "' found in Ticker but not found in WorkloadStatusRepository"
    );
    if (IsBoxAgentMode_) {
        Y_ENSURE(
            target.Meta_.BoxRef_.empty()
            , "PodAgent is in box agent mode. Workload must not have boxRef but it has: '" << target.Meta_.BoxRef_ << "'"
        );
    } else {
        Y_ENSURE(
            GetBoxStatusRepository()->HasObject(target.Meta_.BoxRef_)
            , "Workload '" << target.Meta_.Id_ << "' boxRef '"
                << target.Meta_.BoxRef_ << "' not found in BoxStatusRepository"
        );
    }

    // 0 is a fake revision for waiting childs targets
    GetWorkloadStatusRepository()->AddObject(
        TWorkloadMeta(
            target.Meta_.Id_
            , 0
            , 0
            , target.Meta_.BoxRef_

            , target.Meta_.StartContainer_
            , target.Meta_.InitContainers_

            , target.Meta_.Readiness_
            , target.Meta_.Liveness_
            , target.Meta_.Stop_
            , target.Meta_.Destroy_
        )
    );
    WorkloadStatusRepositoryInternal_->AddObject(target.Meta_.Id_);
    TUnistatObjectHelper::Instance().AddObject(
        target.Meta_
    );
    Ticker_->AddTree(GetEmptyTree(GetWorkloadTreeId(target.Meta_.Id_)), "");
    // Fake workload must be in REMOVED state
    // Otherwise workload never update
    GetWorkloadStatusRepository()->UpdateObjectState(target.Meta_.Id_, API::EWorkloadState_REMOVED);
    UpdateWorkloadTarget(target);
    TryToUpdateWorkloadFromTarget(target.Meta_.Id_);
}

void TStatusNTickerHolder::SetWorkloadTargetRemove(const TString& workloadId) {
    TGuard<TMutex> gMutex(Mutex_);

    Y_ENSURE(
        GetWorkloadStatusRepository()->HasObject(workloadId)
        , "Workload '" << workloadId << "' not found in WorkloadStatusRepository"
    );

    GetUpdateHolder()->SetWorkloadTarget(TUpdateHolder::TWorkloadTarget::GetTargetRemove(workloadId));
    WorkloadStatusRepositoryInternal_->UpdateObjectToBeDestroyedByHook(workloadId, true);
    TargetGraph_.RemoveOutEdges(TGraph::TNode(workloadId, TGraph::ENodeType::WORKLOAD));
}

void TStatusNTickerHolder::UpdateWorkloadTarget(const TUpdateHolder::TWorkloadTarget& target) {
    Y_ENSURE(
        GetWorkloadTreeId(target.Meta_.Id_) == target.Tree_->GetTreeId()
        , "treeId '" << target.Tree_->GetTreeId()
            << "' does not match to workloadId '" << target.Meta_.Id_ << "'"
    );

    TGuard<TMutex> gMutex(Mutex_);

    Y_ENSURE(
        GetWorkloadStatusRepository()->HasObject(target.Meta_.Id_)
        , "Workload '" << target.Meta_.Id_ << "' not found in WorkloadStatusRepository"
    );
    Y_ENSURE(
        WorkloadStatusRepositoryInternal_->HasObject(target.Meta_.Id_)
        , "WorkloadStatusRepository and WorkloadStatusRepositoryInternal unsynchronized: Workload '"
            << target.Meta_.Id_ << "' found in WorkloadStatusRepository but not found in WorkloadStatusRepositoryInternal"
    );
    Y_ENSURE(
        Ticker_->HasTree(target.Tree_->GetTreeId())
        , "WorkloadStatusRepository and Ticker unsynchronized: Tree '" << target.Tree_->GetTreeId()
            << "' found in WorkloadStatusRepository but not found in Ticker"
    );
    if (IsBoxAgentMode_) {
        Y_ENSURE(
            target.Meta_.BoxRef_.empty()
            , "PodAgent is in box agent mode. Workload must not have boxRef but it has: '" << target.Meta_.BoxRef_ << "'"
        );
    } else {
        Y_ENSURE(
            GetBoxStatusRepository()->HasObject(target.Meta_.BoxRef_)
            , "Workload '" << target.Meta_.Id_ << "' boxRef '"
                << target.Meta_.BoxRef_ << "' not found in BoxStatusRepository"
        );
    }

    if (!GetUpdateHolder()->GetUpdateHolderTarget()->WorkloadHasTarget(target.Meta_.Id_)
        && target.Hash_ == Ticker_->GetTreeHash(target.Tree_->GetTreeId())
        && CheckChildsHaveNoTargets(CurrentGraph_, TGraph::TNode(target.Meta_.Id_, TGraph::ENodeType::WORKLOAD))
    ) {

        GetWorkloadStatusRepository()->UpdateObjectSpecTimestamp(target.Meta_.Id_, target.Meta_.SpecTimestamp_);
        GetWorkloadStatusRepository()->UpdateObjectRevision(target.Meta_.Id_, target.Meta_.Revision_);
    } else {
        GetUpdateHolder()->SetWorkloadTarget(target);

        TargetGraph_.RemoveOutEdges(TGraph::TNode(target.Meta_.Id_, TGraph::ENodeType::WORKLOAD));
        if (!IsBoxAgentMode_) {
            TargetGraph_.AddEdge(TGraph::TNode(target.Meta_.Id_, TGraph::ENodeType::WORKLOAD), TGraph::TNode(target.Meta_.BoxRef_, TGraph::ENodeType::BOX));
        }
    }
}

TStatusNTickerHolder::TUpdateObjectResult TStatusNTickerHolder::TryToUpdateWorkloadFromTarget(const TString& workloadId) {
    TGuard<TMutex> gMutex(Mutex_);

    if (!GetWorkloadStatusRepository()->HasObject(workloadId)
        || !GetUpdateHolder()->GetUpdateHolderTarget()->WorkloadHasTarget(workloadId)
    ) {
        return TStatusNTickerHolder::TUpdateObjectResult(TUpdateObjectResult::EUpdateResult::NO_TARGET, "");
    }
    if (const auto result = CheckReadyForUpdate(TGraph::TNode(workloadId, TGraph::ENodeType::WORKLOAD)); !result) {
        return TStatusNTickerHolder::TUpdateObjectResult(TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, result.Error());
    }

    // TODO(chegoryu) remove this
    // Hotfix for DEPLOY-376
    API::TWorkloadStatus workloadStatus = GetWorkloadStatusRepository()->GetObjectStatus(workloadId, GetUpdateHolder()->GetUpdateHolderTarget());

    // NOTE: order is important
    RemoveWorkload(workloadId);
    TUpdateHolder::TWorkloadTarget target = GetUpdateHolder()->GetAndRemoveWorkloadTarget(workloadId);

    if (!target.TargetRemove_) {
        AddWorkload(target);

        UpdateWorkloadTargetState(workloadId, workloadStatus.target_state());
        return TStatusNTickerHolder::TUpdateObjectResult(TUpdateObjectResult::EUpdateResult::UPDATED, "");
    }
    else {
        return TStatusNTickerHolder::TUpdateObjectResult(TUpdateObjectResult::EUpdateResult::REMOVED, "");
    }
}

void TStatusNTickerHolder::UpdateWorkloadTargetState(const TString& workloadId, API::EWorkloadTargetState state) {
    TGuard<TMutex> gMutex(Mutex_);
    GetWorkloadStatusRepository()->UpdateObjectTargetState(workloadId, state);
}

bool TStatusNTickerHolder::HasTarget(const TGraph::TNode& node) const {
    TGuard<TMutex> gMutex(Mutex_);

    switch (node.Type_) {
        case TGraph::ENodeType::STATIC_RESOURCE:
            return GetUpdateHolder()->GetUpdateHolderTarget()->StaticResourceHasTarget(node.Id_);
        case TGraph::ENodeType::LAYER:
            return GetUpdateHolder()->GetUpdateHolderTarget()->LayerHasTarget(node.Id_);
        case TGraph::ENodeType::VOLUME:
            return GetUpdateHolder()->GetUpdateHolderTarget()->VolumeHasTarget(node.Id_);
        case TGraph::ENodeType::BOX:
            return GetUpdateHolder()->GetUpdateHolderTarget()->BoxHasTarget(node.Id_);
        case TGraph::ENodeType::WORKLOAD:
            return GetUpdateHolder()->GetUpdateHolderTarget()->WorkloadHasTarget(node.Id_);
        default:
            ythrow yexception() << "Unknown node type";
    }
}

bool TStatusNTickerHolder::HasTargetRemove(const TGraph::TNode& node) const {
    TGuard<TMutex> gMutex(Mutex_);

    switch (node.Type_) {
        case TGraph::ENodeType::STATIC_RESOURCE:
            return GetUpdateHolder()->GetUpdateHolderTarget()->StaticResourceHasTargetRemove(node.Id_);
        case TGraph::ENodeType::LAYER:
            return GetUpdateHolder()->GetUpdateHolderTarget()->LayerHasTargetRemove(node.Id_);
        case TGraph::ENodeType::VOLUME:
            return GetUpdateHolder()->GetUpdateHolderTarget()->VolumeHasTargetRemove(node.Id_);
        case TGraph::ENodeType::BOX:
            return GetUpdateHolder()->GetUpdateHolderTarget()->BoxHasTargetRemove(node.Id_);
        case TGraph::ENodeType::WORKLOAD:
            return GetUpdateHolder()->GetUpdateHolderTarget()->WorkloadHasTargetRemove(node.Id_);
        default:
            ythrow yexception() << "Unknown node type";
    }
}

TExpected<void, TString> TStatusNTickerHolder::CheckReadyForUpdate(const TGraph::TNode& node) const {
    TGuard<TMutex> gMutex(Mutex_);

    // check all target refs
    if (const auto result = CheckChildsHaveNoTargets(TargetGraph_, node); !result) {
        return result;
    }

    // check that workloads REMOVED
    {
        TSet<TGraph::TGraph::TNode> used;
        TQueue<TGraph::TGraph::TNode> inQueue;
        used.insert(node);
        inQueue.push(node);

        while (!inQueue.empty()) {
            TGraph::TNode current = inQueue.front();
            inQueue.pop();

            if (current.Type_ == TGraph::ENodeType::WORKLOAD) {
                if (GetWorkloadStatusRepository()->GetObjectStatus(current.Id_).state() != API::EWorkloadState_REMOVED) {
                    return TStringBuilder() << "'" << current.ToString() << "' not removed";
                }
                // We can update if workload shouldn't be destroyed by hook or workload in fake revision
                if (WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook(current.Id_) && GetWorkloadStatusRepository()->GetObjectStatus(current.Id_).spec_timestamp() != 0) {
                    return TStringBuilder() << "'" << current.ToString() << "' not destroyed by hook";
                }
            }

            for (const TGraph::TNode& to : CurrentGraph_.GetInEdges(current)) {
                if (!used.contains(to)) {
                    used.insert(to);
                    inQueue.push(to);
                }
            }
        }

    }

    if (HasTargetRemove(node) && !CurrentGraph_.GetInEdges(node).empty()) {
        return TStringBuilder() << "'" << node.ToString() << "' has target remove and has in-edges";
    } else {
        return TExpected<void, TString>::DefaultSuccess();
    }
}

TExpected<void, TString> TStatusNTickerHolder::CheckChildsHaveNoTargets(const TGraph& graph, const TGraph::TNode& node) const {
    TGuard<TMutex> gMutex(Mutex_);

    TSet<TGraph::TNode> used;
    TQueue<TGraph::TNode> inQueue;
    used.insert(node);
    inQueue.push(node);

    while (!inQueue.empty()) {
        TGraph::TNode currentNode = inQueue.front();
        inQueue.pop();

        if (currentNode != node && HasTarget(currentNode)) {
            return TStringBuilder() << "'" << currentNode.ToString() << "' has target";
        }

        for (const TGraph::TNode& to : graph.GetOutEdges(currentNode)) {
            if (!used.contains(to)) {
                used.insert(to);
                inQueue.push(to);
            }
        }
    }

    return TExpected<void, TString>::DefaultSuccess();
}

} // namespace NInfra::NPodAgent
