#pragma once

#include "graph.h"

#include <infra/pod_agent/libs/behaviour/bt/core/tree.h>
#include <infra/pod_agent/libs/behaviour/tickers/mtp_period_ticker.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/status_repository.h>
#include <infra/pod_agent/libs/pod_agent/status_repository_internal/workload_status_repository_internal.h>
#include <infra/pod_agent/libs/pod_agent/update_holder/update_holder.h>

#include <infra/pod_agent/libs/system_logs_sender/system_logs_sender.h>

#include <util/system/mutex.h>

namespace NInfra::NPodAgent {

/**
    Thread-safe StatusNTickerHolder
*/
class TStatusNTickerHolder: public TAtomicRefCount<TStatusNTickerHolder> {
public:
    struct TUpdateObjectResult {
        enum EUpdateResult {
            NO_TARGET       = 0   /* "no_target" */,
            REMOVED         = 1   /* "removed" */,
            UPDATED         = 2   /* "updated" */,
            WAITING_UPDATE  = 3   /* "waiting_update" */
        };

        TUpdateObjectResult(const EUpdateResult& result, const TString& message)
            : Result_(result)
            , Message_(message)
        {}

        bool operator==(const TUpdateObjectResult& other) const {
            return this->Result_ == other.Result_ && this->Message_ == other.Message_;
        }

        const EUpdateResult Result_;
        const TString Message_;
    };

public:
    TStatusNTickerHolder(
        TMtpPeriodTickerPtr ticker
        , TStatusRepositoryPtr statusRepository
        , TWorkloadStatusRepositoryInternalPtr workloadStatusRepositoryInternal
        , ISystemLogsSenderPtr workloadSystemLogsSender
        , ISystemLogsSenderPtr boxSystemLogsSender
        , const bool isBoxAgentMode
    )
        : Ticker_(ticker)
        , StatusRepository_(statusRepository)
        , WorkloadStatusRepositoryInternal_(workloadStatusRepositoryInternal)
        , WorkloadSystemLogsSender_(workloadSystemLogsSender)
        , BoxSystemLogsSender_(boxSystemLogsSender)
        , IsBoxAgentMode_(isBoxAgentMode)
    {}

    static TString GetLayerTreeId(const TString& layerDownloadHash);
    static TString GetStaticResourceTreeId(const TString& staticResourceDownloadHash);
    static TString GetVolumeTreeId(const TString& volumeId);
    static TString GetBoxTreeId(const TString& boxId);
    static TString GetWorkloadTreeId(const TString& workloadId);

    // SystemLogs
    void EnableTransmitSystemLogs(bool enable);
    ISystemLogsSenderPtr GetWorkloadSystemLogsSender() const;
    ISystemLogsSenderPtr GetBoxSystemLogsSender() const;

    // TStatusRepository
    API::TPodAgentStatus GetStatusRepositoryTotalStatus(bool conditionsOnly) const;
    void SetStatusRepositorySpecTimestamp(ui64 specTimestamp);
    void SetStatusRepositoryRevision(ui32 revision);
    void SetStatusRepositorySpecId(const TString& id);
    void SetStatusRepositoryTargetState(API::EPodAgentTargetState targetState);

    TBoxStatusRepositoryPtr GetBoxStatusRepository() const;
    TLayerStatusRepositoryPtr GetLayerStatusRepository() const;
    TStaticResourceStatusRepositoryPtr GetStaticResourceStatusRepository() const;
    TVolumeStatusRepositoryPtr GetVolumeStatusRepository() const;
    TWorkloadStatusRepositoryPtr GetWorkloadStatusRepository() const;

    // Ticker
    void StartTicker();
    void StopTicker();

    // ResourceGang
    void UpdateActiveDownloadContainersLimit(ui32 activeDownloadContainersLimit);
    void UpdateActiveVerifyContainersLimit(ui32 activeDownloadContainersLimit);

    // TUpdateHolder
    TUpdateHolderPtr GetUpdateHolder() const;

    // Internal status
    TWorkloadStatusRepositoryInternalPtr GetWorkloadStatusRepositoryInternal() const;

    // Box subnet
    void UpdateBoxIp6Subnet112Base(const TString& ip6Subnet112Base);
    TString GetBoxIp6Subnet112Base() const;
    bool HasBoxIp6Address(const TString& address);

    // CacheLayer
    void AddCacheLayer(const TUpdateHolder::TLayerTarget& target);
    bool HasCacheLayer(const TString& cacheLayerId, ui32 revision) const;
    void RemoveCacheLayer(const TString& cacheLayerId, ui32 revision);
    TString GetCacheLayerHash(const TString& cacheLayerId, ui32 revision) const;
    TVector<TStatusRepositoryCommon::TCacheObject> GetCacheLayerIdsAndRevisions() const;

    // Layer
    bool HasLayer(const TString& layerId) const;
    bool HasLayerDownloadHash(const TString& layerDownloadHash) const;
    TVector<TString> GetLayerIds() const;
    void AddLayer(const TUpdateHolder::TLayerTarget& target);
    void RemoveLayer(const TString& layerId);
    void AddLayerWithTargetCheck(const TUpdateHolder::TLayerTarget& target);
    void SetLayerTargetRemove(const TString& layerId);
    // If layer already has target spec_timestamp or layerDownloadHash does not matches or some depends changed update target spec_timestamp
    void UpdateLayerTarget(const TUpdateHolder::TLayerTarget& target);
    // WARNING: Must call under UpdateSpecMutex_
    // Remove old and add new layer if old layer workloads in REMOVED state and new dependes haven't target spec_timestamp
    // return true if update
    // If layer not exists do nothing
    TUpdateObjectResult TryToUpdateLayerFromTarget(const TString& layerId);

    // CacheStaticResource
    void AddCacheStaticResource(const TUpdateHolder::TStaticResourceTarget& target);
    bool HasCacheStaticResource(const TString& cacheStaticResourceId, ui32 revision) const;
    void RemoveCacheStaticResource(const TString& cacheStaticResourceId, ui32 revision);
    TString GetCacheStaticResourceHash(const TString& cacheStaticResourceId, ui32 revision) const;
    TVector<TStatusRepositoryCommon::TCacheObject> GetCacheStaticResourceIdsAndRevisions() const;

    // StaticResource
    bool HasStaticResource(const TString& staticResourceId) const;
    bool HasStaticResourceDownloadHash(const TString& staticResourceDownloadHash) const;
    TVector<TString> GetStaticResourceIds() const;
    void AddStaticResource(const TUpdateHolder::TStaticResourceTarget& target);
    void RemoveStaticResource(const TString& staticResourceId);
    void AddStaticResourceWithTargetCheck(const TUpdateHolder::TStaticResourceTarget& target);
    void SetStaticResourceTargetRemove(const TString& staticResourceId);
    // If static resource already has target spec_timestamp or staticResourceDownloadHash does not matches or checkPeriodMs changed or some depends changed update target spec_timestamp
    void UpdateStaticResourceTarget(const TUpdateHolder::TStaticResourceTarget& target);
    // Remove old and add new resource if old layer workloads in REMOVED state and new dependes haven't target spec_timestamp
    // return true if update
    // If resource not exists do nothing
    TUpdateObjectResult TryToUpdateStaticResourceFromTarget(const TString& staticResourceId);

    // Volume
    bool HasVolume(const TString& volumeId) const;
    TVector<TString> GetVolumeIds() const;
    void AddVolume(const TUpdateHolder::TVolumeTarget& target);
    void RemoveVolume(const TString& volumeId);
    void AddVolumeWithTargetCheck(const TUpdateHolder::TVolumeTarget& target);
    void SetVolumeTargetRemove(const TString& volumeId);
    // If volume already has target spec_timestamp or treeHash does not matches or some depends changed update target spec_timestamp
    void UpdateVolumeTarget(const TUpdateHolder::TVolumeTarget& target);
    // WARNING: Must call under UpdateSpecMutex_
    // Remove old and add new volume if old volume workloads in REMOVED state and new dependes haven't target spec_timestamp
    // return true if update
    // If volume not exists do nothing
    TUpdateObjectResult TryToUpdateVolumeFromTarget(const TString& volumeId);

    // Box
    bool HasBox(const TString& boxId) const;
    TVector<TString> GetBoxIds() const;
    void AddBox(const TUpdateHolder::TBoxTarget& target);
    void RemoveBox(const TString& boxId);
    void AddBoxWithTargetCheck(const TUpdateHolder::TBoxTarget& target);
    void SetBoxTargetRemove(const TString& boxId);
    // If box already has target spec_timestamp or treeHash does not matches or some depends changed update target spec_timestamp
    void UpdateBoxTarget(const TUpdateHolder::TBoxTarget& target);
    // WARNING: Must call under UpdateSpecMutex_
    // Remove old and add new box if old box workloads in REMOVED state and new dependes haven't target spec_timestamp
    // return true if update
    // If box not exists do nothing
    TUpdateObjectResult TryToUpdateBoxFromTarget(const TString& boxId);

    // Workload
    bool HasWorkload(const TString& workloadId) const;
    TVector<TString> GetWorkloadIds() const;
    void AddWorkload(const TUpdateHolder::TWorkloadTarget& target);
    void RemoveWorkload(const TString& workloadId);
    void AddWorkloadWithTargetCheck(const TUpdateHolder::TWorkloadTarget& target);
    void UpdateWorkloadTargetState(const TString& workloadId, API::EWorkloadTargetState state);
    void SetWorkloadTargetRemove(const TString& workloadId);
    // If workload already has target spec_timestamp or treeHash does not matches update target spec_timestamp
    void UpdateWorkloadTarget(const TUpdateHolder::TWorkloadTarget& target);
    // WARNING: Must call under UpdateSpecMutex_
    // Remove old and add new workload if old workload in REMOVED state and has target spec_timestamp
    // return true if update
    // If workload not exists do nothing
    TUpdateObjectResult TryToUpdateWorkloadFromTarget(const TString& workloadId);

protected:
    TTreePtr GetEmptyTree(const TString& treeId);

    bool HasTarget(const TGraph::TNode& node) const;
    bool HasTargetRemove(const TGraph::TNode& node) const;
    TExpected<void, TString> CheckReadyForUpdate(const TGraph::TNode& node) const;
    TExpected<void, TString> CheckChildsHaveNoTargets(const TGraph& graph, const TGraph::TNode& node) const;

protected:
    static constexpr const char* STATIC_RESOURCE_TREE_PREFIX = "StaticResourceGang_";
    static constexpr const char* LAYER_TREE_PREFIX = "ResourceGang_";
    static constexpr const char* VOLUME_TREE_PREFIX = "Volume_";
    static constexpr const char* BOX_TREE_PREFIX = "Box_";
    static constexpr const char* WORKLOAD_TREE_PREFIX = "Workload_";

    static TLogger MOCK_LOGGER;

    TGraph CurrentGraph_;
    TGraph TargetGraph_;

    TMtpPeriodTickerPtr Ticker_;
    TStatusRepositoryPtr StatusRepository_;

    TWorkloadStatusRepositoryInternalPtr WorkloadStatusRepositoryInternal_;
    
    ISystemLogsSenderPtr WorkloadSystemLogsSender_;
    ISystemLogsSenderPtr BoxSystemLogsSender_;

    TMutex Mutex_;

    const bool IsBoxAgentMode_;
};

using TStatusNTickerHolderPtr = TIntrusivePtr<TStatusNTickerHolder>;

} // namespace NInfra::NPodAgent
