#include "status_and_ticker_holder.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/update_holder/test_lib/test_functions.h>

#include <infra/pod_agent/libs/system_logs_sender/sidecars_system_logs_filter_impl.h>
#include <infra/pod_agent/libs/system_logs_sender/system_logs_sender_impl.h>
#include <infra/pod_agent/libs/system_logs_sender/system_logs_session_collection_impl.h>

#include <google/protobuf/util/message_differencer.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NStatusNTickerHolderTest {

static TLogger logger({});

class TTestStatusNTickerHolder : public TStatusNTickerHolder {
public:
    TTestStatusNTickerHolder(
        TMtpPeriodTickerPtr ticker
        , TStatusRepositoryPtr statusRepository
        , TWorkloadStatusRepositoryInternalPtr workloadStatusRepositoryInternal
        , const bool isBoxAgentMode = false
    )
        : TStatusNTickerHolder(
              ticker
              , statusRepository 
              , workloadStatusRepositoryInternal
              , new TSystemLogsSender(new TSystemLogsSessionCollection(), new TFakeThreadPool(), new TSystemLogsStatistics(), new TSidecarsSystemLogsFilterImpl())
              , new TSystemLogsSender(new TSystemLogsSessionCollection(), new TFakeThreadPool(), new TSystemLogsStatistics(), new TSidecarsSystemLogsFilterImpl())
              , isBoxAgentMode
          )
    {}

    const TGraph& GetCurrentGraph() {
        return CurrentGraph_;
    }

    const TGraph& GetTargetGraph() {
        return TargetGraph_;
    }
};

using TTestStatusNTickerHolderPtr = TIntrusivePtr<TTestStatusNTickerHolder>;

class ITestStatusNTickerHolderCanon {
public:
    ITestStatusNTickerHolderCanon(const bool isBoxAgentMode = false)
        : Ticker_(new TMtpPeriodTicker(TBehaviorTickerConfig{}, logger.SpawnFrame(), logger.SpawnFrame()))
        , UpdateHolder_(new TUpdateHolder())
        , BoxStatusRepository_(new TBoxStatusRepository())
        , LayerStatusRepository_(new TLayerStatusRepository())
        , StaticResourceStatusRepository_(new TStaticResourceStatusRepository())
        , VolumeStatusRepository_(new TVolumeStatusRepository())
        , WorkloadStatusRepository_(new TWorkloadStatusRepository())
        , WorkloadStatusRepositoryInternal_(new TWorkloadStatusRepositoryInternal())
        , IsBoxAgentMode_(isBoxAgentMode)
   {
        StatusRepository_ = new TStatusRepository(
            UpdateHolder_
            , BoxStatusRepository_
            , LayerStatusRepository_
            , StaticResourceStatusRepository_
            , VolumeStatusRepository_
            , WorkloadStatusRepository_
        );
        StatusNTickerHolder_ = new TTestStatusNTickerHolder(
            Ticker_
            , StatusRepository_
            , WorkloadStatusRepositoryInternal_
            , isBoxAgentMode
        );
    }

    virtual ~ITestStatusNTickerHolderCanon() = default;

    TUpdateHolder::TWorkloadTarget CreateWorkloadTargetSimple(
        const TString& workloadId
        , const TString& hash = "hash"
        , ui32 readinessType = 2
        , ui32 boxId = 0
        , ui32 seed = 1
        , bool withContainers = false
        , bool correctTreeId = true
    ) {
        if (!IsBoxAgentMode_) {
            return NObjectTargetTestLib::CreateWorkloadTargetSimple(workloadId, hash, readinessType, boxId, seed, withContainers, correctTreeId);
        } else {
            return NObjectTargetTestLib::CreateWorkloadTargetWithoutBox(workloadId, hash, readinessType, seed, withContainers, correctTreeId);
        }
    }

    void DoTest() {
        Test();
    }

protected:
    virtual void Test() = 0;

protected:
    TMtpPeriodTickerPtr Ticker_;
    TUpdateHolderPtr UpdateHolder_;

    TBoxStatusRepositoryPtr BoxStatusRepository_;
    TLayerStatusRepositoryPtr LayerStatusRepository_;
    TStaticResourceStatusRepositoryPtr StaticResourceStatusRepository_;
    TVolumeStatusRepositoryPtr VolumeStatusRepository_;
    TWorkloadStatusRepositoryPtr WorkloadStatusRepository_;

    TWorkloadStatusRepositoryInternalPtr WorkloadStatusRepositoryInternal_;

    TStatusRepositoryPtr StatusRepository_;
    TTestStatusNTickerHolderPtr StatusNTickerHolder_;
    const bool IsBoxAgentMode_;
};

Y_UNIT_TEST_SUITE(TStatusNTickerHolderTestSuite) {

Y_UNIT_TEST(TestAddCacheLayer) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddCacheLayer(NObjectTargetTestLib::CreateCacheLayerTargetSimple("cache_layer1", "cache_layer1_download_hash", 1));

            UNIT_ASSERT(StatusNTickerHolder_->HasCacheLayer("cache_layer1", 1));
            UNIT_ASSERT(StatusNTickerHolder_->HasLayerDownloadHash("cache_layer1_download_hash"));
            UNIT_ASSERT(StatusNTickerHolder_->GetCacheLayerHash("cache_layer1", 1) == "cache_layer1_download_hash");
            UNIT_ASSERT(LayerStatusRepository_->HasCacheObject("cache_layer1", 1));
            UNIT_ASSERT(LayerStatusRepository_->HasObjectHash("cache_layer1_download_hash"));
            UNIT_ASSERT(LayerStatusRepository_->GetObjectHashKeepSourceFileAfterImportCounter("cache_layer1_download_hash") == 0);
            UNIT_ASSERT(Ticker_->HasTree(TStatusNTickerHolder::GetLayerTreeId("cache_layer1_download_hash")));
            UNIT_ASSERT(LayerStatusRepository_->GetCacheObjectStatus("cache_layer1", 1).state() == API::ELayerState_UNKNOWN);

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddCacheLayer(NObjectTargetTestLib::CreateCacheLayerTargetSimple("cache_layer1", "cache_layer1_download_hash", 1))
                , yexception
                , "Layer 'cache_layer1' with revision '1' already added"
            );

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddCacheLayer(NObjectTargetTestLib::CreateCacheLayerTargetSimple("cache_layer2", "cache_layer2_download_hash", 1, 0, true, false))
                , yexception
                , "treeId 'random_prefix_cache_layer2_download_hash' does not match to DownloadHash_ 'cache_layer2_download_hash'"
            );

            Ticker_->AddTree(GetEmptyTree(TStatusNTickerHolder::GetLayerTreeId("cache_layer3_download_hash")), "hash");
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddCacheLayer(NObjectTargetTestLib::CreateCacheLayerTargetSimple("cache_layer3", "cache_layer3_download_hash", 1))
                , yexception
                , "Ticker already has tree with id:"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveCacheLayer) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddCacheLayer(NObjectTargetTestLib::CreateCacheLayerTargetSimple("cache_layer1", "cache_layer1_download_hash", 1));

            UNIT_ASSERT(StatusNTickerHolder_->HasCacheLayer("cache_layer1", 1));
            UNIT_ASSERT(StatusNTickerHolder_->HasLayerDownloadHash("cache_layer1_download_hash"));
            UNIT_ASSERT(StatusNTickerHolder_->GetCacheLayerHash("cache_layer1", 1) == "cache_layer1_download_hash");
            UNIT_ASSERT(LayerStatusRepository_->HasCacheObject("cache_layer1", 1));
            UNIT_ASSERT(LayerStatusRepository_->HasObjectHash("cache_layer1_download_hash"));
            UNIT_ASSERT(LayerStatusRepository_->GetObjectHashKeepSourceFileAfterImportCounter("cache_layer1_download_hash") == 0);
            UNIT_ASSERT(Ticker_->HasTree(TStatusNTickerHolder::GetLayerTreeId("cache_layer1_download_hash")));
            UNIT_ASSERT(LayerStatusRepository_->GetCacheObjectStatus("cache_layer1", 1).state() == API::ELayerState_UNKNOWN);

            StatusNTickerHolder_->RemoveCacheLayer("cache_layer1", 1);

            UNIT_ASSERT(!StatusNTickerHolder_->HasCacheLayer("cache_layer1", 1));
            UNIT_ASSERT(!LayerStatusRepository_->HasCacheObject("cache_layer1", 1));
            UNIT_ASSERT(!LayerStatusRepository_->HasObjectHash("cache_layer1_download_hash"));
            UNIT_ASSERT(!Ticker_->HasTree(TStatusNTickerHolder::GetLayerTreeId("cache_layer1_download_hash")));

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->RemoveCacheLayer("cache_layer2", 1)
                , yexception
                , "Layer 'cache_layer2' with revision '1' not found in LayerStatusRepository"
            );

            StatusNTickerHolder_->AddCacheLayer(NObjectTargetTestLib::CreateCacheLayerTargetSimple("cache_layer2", "cache_layer2_download_hash", 1));
            Ticker_->RemoveTree(TStatusNTickerHolder::GetLayerTreeId("cache_layer2_download_hash"));
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->RemoveCacheLayer("cache_layer2", 1)
                , yexception
                , "LayerStatusRepository and Ticker unsynchronized: Tree"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestAddLayer) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));

            UNIT_ASSERT(StatusNTickerHolder_->HasLayer("layer1"));
            UNIT_ASSERT(StatusNTickerHolder_->HasLayerDownloadHash("layer1_download_hash"));
            UNIT_ASSERT(LayerStatusRepository_->HasObject("layer1"));
            UNIT_ASSERT(LayerStatusRepository_->HasObjectHash("layer1_download_hash"));
            UNIT_ASSERT(LayerStatusRepository_->GetObjectRemoveSourceFileAfterImportFlag("layer1"));
            UNIT_ASSERT(LayerStatusRepository_->GetObjectHashKeepSourceFileAfterImportCounter("layer1_download_hash") == 0);
            UNIT_ASSERT(Ticker_->HasTree(TStatusNTickerHolder::GetLayerTreeId("layer1_download_hash")));
            UNIT_ASSERT(LayerStatusRepository_->GetObjectStatus("layer1").state() == API::ELayerState_UNKNOWN);

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"))
                , yexception
                , "Layer 'layer1' already added"
            );

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer2", "layer2_download_hash", 0, true, false))
                , yexception
                , "treeId 'random_prefix_layer2_download_hash' does not match to DownloadHash_ 'layer2_download_hash'"
            );

            Ticker_->AddTree(GetEmptyTree(TStatusNTickerHolder::GetLayerTreeId("layer3_download_hash")), "hash");
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer3", "layer3_download_hash"))
                , yexception
                , "Ticker already has tree with id:"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveLayer) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));

            UNIT_ASSERT(StatusNTickerHolder_->HasLayer("layer1"));
            UNIT_ASSERT(StatusNTickerHolder_->HasLayerDownloadHash("layer1_download_hash"));
            UNIT_ASSERT(LayerStatusRepository_->HasObject("layer1"));
            UNIT_ASSERT(LayerStatusRepository_->HasObjectHash("layer1_download_hash"));
            UNIT_ASSERT(LayerStatusRepository_->GetObjectRemoveSourceFileAfterImportFlag("layer1"));
            UNIT_ASSERT(LayerStatusRepository_->GetObjectHashKeepSourceFileAfterImportCounter("layer1_download_hash") == 0);
            UNIT_ASSERT(Ticker_->HasTree(TStatusNTickerHolder::GetLayerTreeId("layer1_download_hash")));
            UNIT_ASSERT(LayerStatusRepository_->GetObjectStatus("layer1").state() == API::ELayerState_UNKNOWN);

            StatusNTickerHolder_->RemoveLayer("layer1");

            UNIT_ASSERT(!StatusNTickerHolder_->HasLayer("layer1"));
            UNIT_ASSERT(!LayerStatusRepository_->HasObject("layer1"));
            UNIT_ASSERT(!LayerStatusRepository_->HasObjectHash("layer1_download_hash"));
            UNIT_ASSERT(!Ticker_->HasTree(TStatusNTickerHolder::GetLayerTreeId("layer1_download_hash")));

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->RemoveLayer("layer2")
                , yexception
                , "Layer 'layer2' not found in LayerStatusRepository"
            );

            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer2", "layer2_download_hash"));
            Ticker_->RemoveTree(TStatusNTickerHolder::GetLayerTreeId("layer2_download_hash"));
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->RemoveLayer("layer2")
                , yexception
                , "LayerStatusRepository and Ticker unsynchronized: Tree"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestAddLayerWithTargetCheckAsAddLayer) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddLayerWithTargetCheck(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));

            UNIT_ASSERT(StatusNTickerHolder_->HasLayer("layer1"));
            UNIT_ASSERT(StatusNTickerHolder_->HasLayerDownloadHash("layer1_download_hash"));
            UNIT_ASSERT(LayerStatusRepository_->HasObject("layer1"));
            UNIT_ASSERT(LayerStatusRepository_->HasObjectHash("layer1_download_hash"));
            UNIT_ASSERT(LayerStatusRepository_->GetObjectRemoveSourceFileAfterImportFlag("layer1"));
            UNIT_ASSERT(LayerStatusRepository_->GetObjectHashKeepSourceFileAfterImportCounter("layer1_download_hash") == 0);
            UNIT_ASSERT(Ticker_->HasTree(TStatusNTickerHolder::GetLayerTreeId("layer1_download_hash")));
            UNIT_ASSERT(LayerStatusRepository_->GetObjectStatus("layer1").state() == API::ELayerState_UNKNOWN);

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddLayerWithTargetCheck(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"))
                , yexception
                , "Layer 'layer1' already added"
            );

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddLayerWithTargetCheck(NObjectTargetTestLib::CreateLayerTargetSimple("layer2", "layer2_download_hash", 0, true, false))
                , yexception
                , "treeId 'random_prefix_layer2_download_hash' does not match to DownloadHash_ 'layer2_download_hash'"
            );

            Ticker_->AddTree(GetEmptyTree(TStatusNTickerHolder::GetLayerTreeId("layer3_download_hash")), "hash");
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddLayerWithTargetCheck(NObjectTargetTestLib::CreateLayerTargetSimple("layer3", "layer3_download_hash"))
                , yexception
                , "Ticker already has tree with id:"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestLayerTargetNoChange) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));

            StatusNTickerHolder_->UpdateLayerTarget(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->LayerHasTarget("layer1"));

            StatusNTickerHolder_->UpdateLayerTarget(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash", 42));
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->LayerHasTarget("layer1"));
            UNIT_ASSERT(LayerStatusRepository_->GetObjectStatus("layer1").spec_timestamp() == 42);
            UNIT_ASSERT(LayerStatusRepository_->GetObjectStatus("layer1").revision() == 43);
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateLayerFromTarget("layer2").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::NO_TARGET);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestLayerTargetChangeRemoveSourceFileAfterImportFlag) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash", 1, false));

            TUpdateHolder::TLayerTarget newTarget = NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash", 2, true);
            StatusNTickerHolder_->UpdateLayerTarget(newTarget);
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->LayerHasTarget("layer1"));

            TUpdateHolder::TLayerTarget layerTarget = UpdateHolder_->GetAndRemoveLayerTarget("layer1");
            UNIT_ASSERT(layerTarget == newTarget);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestLayerTargetWithUpdateHolder) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));

            TUpdateHolder::TLayerTarget newTarget1 = NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash_other1", 1);
            TUpdateHolder::TLayerTarget newTarget2 = NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash_other2", 3);
            StatusNTickerHolder_->UpdateLayerTarget(newTarget1);
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->LayerHasTarget("layer1"));

            StatusNTickerHolder_->UpdateLayerTarget(newTarget2);
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->LayerHasTarget("layer1"));

            TUpdateHolder::TLayerTarget layerTarget = UpdateHolder_->GetAndRemoveLayerTarget("layer1");
            UNIT_ASSERT(layerTarget == newTarget2);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestLayerTargetWithWorkload) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetWithDependence("box1", "hash", {"layer1"}));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload1", "hash", "box1"));
            StatusNTickerHolder_->UpdateLayerTarget(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->LayerHasTarget("layer1"));

            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").rootfs_layer_refs().size()
                , 1
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").rootfs_layer_refs().size()
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").rootfs_layer_refs(0)
                , "layer1"
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").rootfs_layer_refs(0)
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").static_resource_refs().size()
                , 0
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").static_resource_refs().size()
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").volume_refs().size()
                , 0
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").volume_refs().size()
            );

            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_REMOVED);
            StatusNTickerHolder_->UpdateWorkloadTarget(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload1", "other_hash", "box1"));
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->WorkloadHasTarget("workload1"));
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateLayerFromTarget("layer1").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::NO_TARGET);
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->LayerHasTarget("layer1"));
            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_ACTIVE);

            StatusNTickerHolder_->UpdateLayerTarget(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash_other"));
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->LayerHasTarget("layer1"));
            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateLayerFromTarget("layer1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'workload workload1' not removed")
            );
            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateWorkloadFromTarget("workload1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'layer layer1' has target")
            );

            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_REMOVED);
            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateWorkloadFromTarget("workload1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'layer layer1' has target")
            );
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateLayerFromTarget("layer1").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::UPDATED);
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->LayerHasTarget("layer1"));
            UNIT_ASSERT(Ticker_->GetTreeHash(TStatusNTickerHolder::GetLayerTreeId("layer1_download_hash_other")) == "layer1_download_hash_other");
            UNIT_ASSERT(LayerStatusRepository_->GetObjectStatus("layer1").spec_timestamp() == 1);
            UNIT_ASSERT(LayerStatusRepository_->GetObjectStatus("layer1").revision() == 2);
            UNIT_ASSERT(!LayerStatusRepository_->HasObjectHash("layer1_download_hash"));
            UNIT_ASSERT(LayerStatusRepository_->HasObjectHash("layer1_download_hash_other"));
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateWorkloadFromTarget("workload1").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::UPDATED);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestLayerTargetExceptions) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->UpdateLayerTarget(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash", 0, true, false))
                , yexception
                , "treeId 'random_prefix_layer1_download_hash' does not match to DownloadHash_ 'layer1_download_hash'"
            );
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->UpdateLayerTarget(NObjectTargetTestLib::CreateLayerTargetSimple("layer2", "layer2_download_hash"))
                , yexception
                , "Layer 'layer2' not found in LayerStatusRepository"
            );

            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer2", "layer2_download_hash"));
            Ticker_->RemoveTree(TStatusNTickerHolder::GetLayerTreeId("layer2_download_hash"));
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->UpdateLayerTarget(NObjectTargetTestLib::CreateLayerTargetSimple("layer2", "layer2_download_hash"))
                , yexception
                , "LayerStatusRepository and Ticker unsynchronized: Tree"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestLayerTargetRemove) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetWithDependence("box1", "hash", {"layer1"}));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload1", "hash", "box1"));

            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").rootfs_layer_refs().size()
                , 1
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").rootfs_layer_refs().size()
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").rootfs_layer_refs(0)
                , "layer1"
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").rootfs_layer_refs(0)
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").static_resource_refs().size()
                , 0
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").static_resource_refs().size()
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").volume_refs().size()
                , 0
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").volume_refs().size()
            );

            UNIT_ASSERT(StatusNTickerHolder_->HasLayer("layer1"));
            StatusNTickerHolder_->SetLayerTargetRemove("layer1");
            UNIT_ASSERT(StatusNTickerHolder_->HasLayer("layer1"));

            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateLayerFromTarget("layer1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'workload workload1' not removed")
            );
            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_REMOVED);
            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateLayerFromTarget("layer1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'layer layer1' has target remove and has in-edges")
            );

            StatusNTickerHolder_->UpdateBoxTarget(NObjectTargetTestLib::CreateBoxTargetSimple("box1", "other_hash"));
            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateLayerFromTarget("layer1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'layer layer1' has target remove and has in-edges")
            );
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateBoxFromTarget("box1").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::UPDATED);
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateLayerFromTarget("layer1").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::REMOVED);
            UNIT_ASSERT(!StatusNTickerHolder_->HasLayer("layer1"));

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->SetLayerTargetRemove("layer2")
                , yexception
                , "Layer 'layer2' not found in LayerStatusRepository"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestAddCacheStaticResource) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddCacheStaticResource(NObjectTargetTestLib::CreateCacheStaticResourceTargetSimple("cache_static_resource1", "cache_static_resource1_download_hash", 1));

            UNIT_ASSERT(StatusNTickerHolder_->HasCacheStaticResource("cache_static_resource1", 1));
            UNIT_ASSERT(StatusNTickerHolder_->HasStaticResourceDownloadHash("cache_static_resource1_download_hash"));
            UNIT_ASSERT(StatusNTickerHolder_->GetCacheStaticResourceHash("cache_static_resource1", 1) == "cache_static_resource1_download_hash");
            UNIT_ASSERT(StaticResourceStatusRepository_->HasCacheObject("cache_static_resource1", 1));
            UNIT_ASSERT(StaticResourceStatusRepository_->HasObjectHash("cache_static_resource1_download_hash"));
            UNIT_ASSERT(StaticResourceStatusRepository_->GetObjectHashMinCheckPeriodMs("cache_static_resource1_download_hash") == 2);
            UNIT_ASSERT(Ticker_->HasTree(TStatusNTickerHolder::GetStaticResourceTreeId("cache_static_resource1_download_hash")));
            UNIT_ASSERT(StaticResourceStatusRepository_->GetCacheObjectStatus("cache_static_resource1", 1).state() == API::EStaticResourceState_UNKNOWN);

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddCacheStaticResource(NObjectTargetTestLib::CreateCacheStaticResourceTargetSimple("cache_static_resource1", "cache_static_resource1_download_hash", 1))
                , yexception
                , "StaticResource 'cache_static_resource1' with revision '1' already added"
            );

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddCacheStaticResource(NObjectTargetTestLib::CreateCacheStaticResourceTargetSimple("cache_static_resource2", "cache_static_resource2_download_hash", 1, 0, false))
                , yexception
                , "treeId 'random_prefix_cache_static_resource2_download_hash' does not match to DownloadHash_ 'cache_static_resource2_download_hash'"
            );

            Ticker_->AddTree(GetEmptyTree(TStatusNTickerHolder::GetStaticResourceTreeId("cache_static_resource3_download_hash")), "hash");
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddCacheStaticResource(NObjectTargetTestLib::CreateCacheStaticResourceTargetSimple("cache_static_resource3", "cache_static_resource3_download_hash", 1))
                , yexception
                , "Ticker already has tree with id:"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveCacheStaticResource) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddCacheStaticResource(NObjectTargetTestLib::CreateCacheStaticResourceTargetSimple("cache_static_resource1", "cache_static_resource1_download_hash", 1));

            UNIT_ASSERT(StatusNTickerHolder_->HasCacheStaticResource("cache_static_resource1", 1));
            UNIT_ASSERT(StatusNTickerHolder_->HasStaticResourceDownloadHash("cache_static_resource1_download_hash"));
            UNIT_ASSERT(StatusNTickerHolder_->GetCacheStaticResourceHash("cache_static_resource1", 1) == "cache_static_resource1_download_hash");
            UNIT_ASSERT(StaticResourceStatusRepository_->HasCacheObject("cache_static_resource1", 1));
            UNIT_ASSERT(StaticResourceStatusRepository_->HasObjectHash("cache_static_resource1_download_hash"));
            UNIT_ASSERT(StaticResourceStatusRepository_->GetObjectHashMinCheckPeriodMs("cache_static_resource1_download_hash") == 2);
            UNIT_ASSERT(Ticker_->HasTree(TStatusNTickerHolder::GetStaticResourceTreeId("cache_static_resource1_download_hash")));
            UNIT_ASSERT(StaticResourceStatusRepository_->GetCacheObjectStatus("cache_static_resource1", 1).state() == API::EStaticResourceState_UNKNOWN);

            StatusNTickerHolder_->RemoveCacheStaticResource("cache_static_resource1", 1);

            UNIT_ASSERT(!StatusNTickerHolder_->HasCacheStaticResource("cache_static_resource1", 1));
            UNIT_ASSERT(!StaticResourceStatusRepository_->HasCacheObject("cache_static_resource1", 1));
            UNIT_ASSERT(!StaticResourceStatusRepository_->HasObjectHash("cache_static_resource1_download_hash"));
            UNIT_ASSERT(!Ticker_->HasTree(TStatusNTickerHolder::GetStaticResourceTreeId("cache_static_resource1_download_hash")));

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->RemoveCacheStaticResource("cache_static_resource2", 1)
                , yexception
                , "StaticResource 'cache_static_resource2' with revision '1' not found in StaticResourceStatusRepository"
            );

            StatusNTickerHolder_->AddCacheStaticResource(NObjectTargetTestLib::CreateCacheStaticResourceTargetSimple("cache_static_resource2", "cache_static_resource2_download_hash", 1));
            Ticker_->RemoveTree(TStatusNTickerHolder::GetStaticResourceTreeId("cache_static_resource2_download_hash"));
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->RemoveCacheStaticResource("cache_static_resource2", 1)
                , yexception
                , "StaticResourceStatusRepository and Ticker unsynchronized: Tree"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestAddStaticResource) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash"));

            UNIT_ASSERT(StatusNTickerHolder_->HasStaticResource("static_resource1"));
            UNIT_ASSERT(StatusNTickerHolder_->HasStaticResourceDownloadHash("static_resource1_download_hash"));
            UNIT_ASSERT(StaticResourceStatusRepository_->HasObject("static_resource1"));
            UNIT_ASSERT(StaticResourceStatusRepository_->HasObjectHash("static_resource1_download_hash"));
            UNIT_ASSERT(StaticResourceStatusRepository_->GetObjectCheckPeriodMs("static_resource1") == 3);
            UNIT_ASSERT(StaticResourceStatusRepository_->GetObjectHashMinCheckPeriodMs("static_resource1_download_hash") == 3);
            UNIT_ASSERT(Ticker_->HasTree(TStatusNTickerHolder::GetStaticResourceTreeId("static_resource1_download_hash")));
            UNIT_ASSERT(StaticResourceStatusRepository_->GetObjectStatus("static_resource1").state() == API::EStaticResourceState_UNKNOWN);

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash"))
                , yexception
                , "StaticResource 'static_resource1' already added"
            );

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource2", "static_resource2_download_hash", 0, false))
                , yexception
                , "treeId 'random_prefix_static_resource2_download_hash' does not match to DownloadHash_ 'static_resource2_download_hash'"
            );

            Ticker_->AddTree(GetEmptyTree(TStatusNTickerHolder::GetStaticResourceTreeId("static_resource3_download_hash")), "hash");
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource3", "static_resource3_download_hash"))
                , yexception
                , "Ticker already has tree with id:"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveStaticResource) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash"));

            UNIT_ASSERT(StatusNTickerHolder_->HasStaticResource("static_resource1"));
            UNIT_ASSERT(StatusNTickerHolder_->HasStaticResourceDownloadHash("static_resource1_download_hash"));
            UNIT_ASSERT(StaticResourceStatusRepository_->HasObject("static_resource1"));
            UNIT_ASSERT(StaticResourceStatusRepository_->HasObjectHash("static_resource1_download_hash"));
            UNIT_ASSERT(StaticResourceStatusRepository_->GetObjectCheckPeriodMs("static_resource1") == 3);
            UNIT_ASSERT(StaticResourceStatusRepository_->GetObjectHashMinCheckPeriodMs("static_resource1_download_hash") == 3);
            UNIT_ASSERT(Ticker_->HasTree(TStatusNTickerHolder::GetStaticResourceTreeId("static_resource1_download_hash")));
            UNIT_ASSERT(StaticResourceStatusRepository_->GetObjectStatus("static_resource1").state() == API::EStaticResourceState_UNKNOWN);

            StatusNTickerHolder_->RemoveStaticResource("static_resource1");

            UNIT_ASSERT(!StatusNTickerHolder_->HasStaticResource("static_resource1"));
            UNIT_ASSERT(!StaticResourceStatusRepository_->HasObject("static_resource1"));
            UNIT_ASSERT(!StaticResourceStatusRepository_->HasObjectHash("static_resource1_download_hash"));
            UNIT_ASSERT(!Ticker_->HasTree(TStatusNTickerHolder::GetStaticResourceTreeId("static_resource1_download_hash")));

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->RemoveStaticResource("static_resource2")
                , yexception
                , "StaticResource 'static_resource2' not found in StaticResourceStatusRepository"
            );

            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource2", "static_resource2_download_hash"));
            Ticker_->RemoveTree(TStatusNTickerHolder::GetStaticResourceTreeId("static_resource2_download_hash"));
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->RemoveStaticResource("static_resource2")
                , yexception
                , "StaticResourceStatusRepository and Ticker unsynchronized: Tree"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestAddStaticResourceWithTargetCheckAsAddStaticResource) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddStaticResourceWithTargetCheck(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash"));

            UNIT_ASSERT(StatusNTickerHolder_->HasStaticResource("static_resource1"));
            UNIT_ASSERT(StatusNTickerHolder_->HasStaticResourceDownloadHash("static_resource1_download_hash"));
            UNIT_ASSERT(StaticResourceStatusRepository_->HasObject("static_resource1"));
            UNIT_ASSERT(StaticResourceStatusRepository_->HasObjectHash("static_resource1_download_hash"));
            UNIT_ASSERT(StaticResourceStatusRepository_->GetObjectCheckPeriodMs("static_resource1") == 3);
            UNIT_ASSERT(StaticResourceStatusRepository_->GetObjectHashMinCheckPeriodMs("static_resource1_download_hash") == 3);
            UNIT_ASSERT(Ticker_->HasTree(TStatusNTickerHolder::GetStaticResourceTreeId("static_resource1_download_hash")));
            UNIT_ASSERT(StaticResourceStatusRepository_->GetObjectStatus("static_resource1").state() == API::EStaticResourceState_UNKNOWN);

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddStaticResourceWithTargetCheck(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash"))
                , yexception
                , "StaticResource 'static_resource1' already added"
            );

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddStaticResourceWithTargetCheck(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource2", "static_resource2_download_hash", 0, false))
                , yexception
                , "treeId 'random_prefix_static_resource2_download_hash' does not match to DownloadHash_ 'static_resource2_download_hash'"
            );

            Ticker_->AddTree(GetEmptyTree(TStatusNTickerHolder::GetStaticResourceTreeId("static_resource3_download_hash")), "hash");
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddStaticResourceWithTargetCheck(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource3", "static_resource3_download_hash"))
                , yexception
                , "Ticker already has tree with id:"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestStaticResourceTargetNoChange) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash"));

            StatusNTickerHolder_->UpdateStaticResourceTarget(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash"));
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->StaticResourceHasTarget("static_resource1"));

            StatusNTickerHolder_->UpdateStaticResourceTarget(
                TUpdateHolder::TStaticResourceTarget(
                    TStaticResourceMeta(
                        "static_resource1"
                        , 42
                        , 43
                        , "static_resource1_download_hash"
                        , 3
                    )
                    , GetEmptyTree(TStatusNTickerHolder::GetStaticResourceTreeId("static_resource1_download_hash"))
                )
            );
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->StaticResourceHasTarget("static_resource1"));
            UNIT_ASSERT(StaticResourceStatusRepository_->GetObjectStatus("static_resource1").spec_timestamp() == 42);
            UNIT_ASSERT(StaticResourceStatusRepository_->GetObjectStatus("static_resource1").revision() == 43);
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateStaticResourceFromTarget("static_resource2").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::NO_TARGET);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestStaticResourceTargetChangeCheckPeriod) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash", 1));

            TUpdateHolder::TStaticResourceTarget newTarget = NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash", 4);
            StatusNTickerHolder_->UpdateStaticResourceTarget(newTarget);
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->StaticResourceHasTarget("static_resource1"));

            TUpdateHolder::TStaticResourceTarget staticResourceTarget = UpdateHolder_->GetAndRemoveStaticResourceTarget("static_resource1");
            UNIT_ASSERT(staticResourceTarget == newTarget);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestStaticResourceTargetWithUpdateHolder) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash"));

            TUpdateHolder::TStaticResourceTarget newTarget1 = NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash_other1", 1);
            TUpdateHolder::TStaticResourceTarget newTarget2 = NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash_other2", 3);
            StatusNTickerHolder_->UpdateStaticResourceTarget(newTarget1);
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->StaticResourceHasTarget("static_resource1"));

            StatusNTickerHolder_->UpdateStaticResourceTarget(newTarget2);
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->StaticResourceHasTarget("static_resource1"));

            TUpdateHolder::TStaticResourceTarget staticResourceTarget = UpdateHolder_->GetAndRemoveStaticResourceTarget("static_resource1");
            UNIT_ASSERT(staticResourceTarget == newTarget2);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestStaticResourceTargetWithWorkload) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash"));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetWithDependence("box1", "hash", {}, {"static_resource1"}));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload1", "hash", "box1"));
            StatusNTickerHolder_->UpdateStaticResourceTarget(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash"));
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->StaticResourceHasTarget("static_resource1"));

            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").rootfs_layer_refs().size()
                , 0
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").rootfs_layer_refs().size()
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").static_resource_refs().size()
                , 1
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").static_resource_refs().size()
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").static_resource_refs(0)
                , "static_resource1"
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").static_resource_refs(0)
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").volume_refs().size()
                , 0
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").volume_refs().size()
            );

            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_REMOVED);
            StatusNTickerHolder_->UpdateWorkloadTarget(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload1", "other_hash", "box1"));
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->WorkloadHasTarget("workload1"));
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateStaticResourceFromTarget("static_resource1").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::NO_TARGET);
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->StaticResourceHasTarget("static_resource1"));
            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_ACTIVE);

            TTreePtr newTree = GetEmptyTree(TStatusNTickerHolder::GetStaticResourceTreeId("static_resource1_download_hash_other"));
            TUpdateHolder::TStaticResourceTarget newTarget = NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash_other");
            StatusNTickerHolder_->UpdateStaticResourceTarget(newTarget);
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->StaticResourceHasTarget("static_resource1"));
            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateStaticResourceFromTarget("static_resource1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'workload workload1' not removed")
            );
            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateWorkloadFromTarget("workload1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'static_resource static_resource1' has target")
            );

            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_REMOVED);
            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateWorkloadFromTarget("workload1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'static_resource static_resource1' has target")
            );
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateStaticResourceFromTarget("static_resource1").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::UPDATED);
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->StaticResourceHasTarget("static_resource1"));
            UNIT_ASSERT(Ticker_->GetTreeHash(TStatusNTickerHolder::GetStaticResourceTreeId("static_resource1_download_hash_other")) == "static_resource1_download_hash_other");
            UNIT_ASSERT(StaticResourceStatusRepository_->GetObjectStatus("static_resource1").spec_timestamp() == 1);
            UNIT_ASSERT(StaticResourceStatusRepository_->GetObjectStatus("static_resource1").revision() == 2);
            UNIT_ASSERT(!StaticResourceStatusRepository_->HasObjectHash("static_resource1_download_hash"));
            UNIT_ASSERT(StaticResourceStatusRepository_->HasObjectHash("static_resource1_download_hash_other"));
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateWorkloadFromTarget("workload1").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::UPDATED);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestStaticResourceTargetExceptions) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash"));

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->UpdateStaticResourceTarget(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash", 0, false))
                , yexception
                , "treeId 'random_prefix_static_resource1_download_hash' does not match to DownloadHash_ 'static_resource1_download_hash'"
            );
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->UpdateStaticResourceTarget(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource2", "static_resource2_download_hash"))
                , yexception
                , "StaticResource 'static_resource2' not found in StaticResourceStatusRepository"
            );

            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource2", "static_resource2_download_hash"));
            Ticker_->RemoveTree(TStatusNTickerHolder::GetStaticResourceTreeId("static_resource2_download_hash"));
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->UpdateStaticResourceTarget(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource2", "static_resource2_download_hash"))
                , yexception
                , "StaticResourceStatusRepository and Ticker unsynchronized: Tree"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestStaticResourceTargetRemove) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash"));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetWithDependence("box1", "hash", {}, {"static_resource1"}));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload1", "hash", "box1"));

            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").rootfs_layer_refs().size()
                , 0
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").rootfs_layer_refs().size()
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").static_resource_refs().size()
                , 1
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").static_resource_refs().size()
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").static_resource_refs(0)
                , "static_resource1"
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").static_resource_refs(0)
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").volume_refs().size()
                , 0
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").volume_refs().size()
            );

            UNIT_ASSERT(StatusNTickerHolder_->HasStaticResource("static_resource1"));
            StatusNTickerHolder_->SetStaticResourceTargetRemove("static_resource1");
            UNIT_ASSERT(StatusNTickerHolder_->HasStaticResource("static_resource1"));

            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateStaticResourceFromTarget("static_resource1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'workload workload1' not removed")
            );
            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_REMOVED);
            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateStaticResourceFromTarget("static_resource1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'static_resource static_resource1' has target remove and has in-edges")
            );

            StatusNTickerHolder_->UpdateBoxTarget(NObjectTargetTestLib::CreateBoxTargetSimple("box1", "other_hash"));
            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateStaticResourceFromTarget("static_resource1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'static_resource static_resource1' has target remove and has in-edges")
            );
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateBoxFromTarget("box1").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::UPDATED);
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateStaticResourceFromTarget("static_resource1").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::REMOVED);
            UNIT_ASSERT(!StatusNTickerHolder_->HasStaticResource("static_resource1"));

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->SetStaticResourceTargetRemove("static_resource2")
                , yexception
                , "StaticResource 'static_resource2' not found in StaticResourceStatusRepository"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestAddVolume) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));
            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash"));

            StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetWithDependence("volume1", "hash", {"layer1"}, {"static_resource1"}));

            UNIT_ASSERT(StatusNTickerHolder_->HasVolume("volume1"));
            UNIT_ASSERT(VolumeStatusRepository_->HasObject("volume1"));
            UNIT_ASSERT(Ticker_->HasTree(TStatusNTickerHolder::GetVolumeTreeId("volume1")));

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("volume1"))
                , yexception
                , "Volume 'volume1' already added"
            );

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("volume2", "hash", 0, 0, false))
                , yexception
                , "treeId 'random_prefix_volume2' does not match to volumeId 'volume2'"
            );

            Ticker_->AddTree(GetEmptyTree(TStatusNTickerHolder::GetVolumeTreeId("volume3")), "hash");
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("volume3"))
                , yexception
                , "VolumeStatusRepository and Ticker unsynchronized: Tree"
            );
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetWithDependence("volume4", "hash", {"layer1", "no_layer"}, {"static_resource1"}))
                , yexception
                , "Volume 'volume4' layerRef 'no_layer' not found in LayerStatusRepository"
            );

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetWithDependence("volume4", "hash", {"layer1"}, {"static_resource1", "no_static_resource"}))
                , yexception
                , "Volume 'volume4' staticResourceRef 'no_static_resource' not found in StaticResourceStatusRepository"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveVolume) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("volume1"));

            UNIT_ASSERT(StatusNTickerHolder_->HasVolume("volume1"));
            UNIT_ASSERT(VolumeStatusRepository_->HasObject("volume1"));
            UNIT_ASSERT(Ticker_->HasTree(TStatusNTickerHolder::GetVolumeTreeId("volume1")));

            StatusNTickerHolder_->RemoveVolume("volume1");

            UNIT_ASSERT(!StatusNTickerHolder_->HasVolume("volume1"));
            UNIT_ASSERT(!VolumeStatusRepository_->HasObject("volume1"));
            UNIT_ASSERT(!Ticker_->HasTree(TStatusNTickerHolder::GetVolumeTreeId("volume1")));

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->RemoveVolume("volume2")
                , yexception
                , "Volume 'volume2' not found in VolumeStatusRepository"
            );

            StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("volume2"));
            Ticker_->RemoveTree(TStatusNTickerHolder::GetVolumeTreeId("volume2"));
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->RemoveVolume("volume2")
                , yexception
                , "VolumeStatusRepository and Ticker unsynchronized: Tree"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestAddVolumeWithTargetCheckAsAddVolume) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));
            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "layer1_download_hash"));
            StatusNTickerHolder_->AddVolumeWithTargetCheck(NObjectTargetTestLib::CreatePersistentVolumeTargetWithDependence("volume1", "hash", {"layer1"}, {"static_resource1"}));

            UNIT_ASSERT(StatusNTickerHolder_->HasVolume("volume1"));
            UNIT_ASSERT(VolumeStatusRepository_->HasObject("volume1"));
            UNIT_ASSERT(Ticker_->HasTree(TStatusNTickerHolder::GetVolumeTreeId("volume1")));

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddVolumeWithTargetCheck(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("volume1"))
                , yexception
                , "Volume 'volume1' already added"
            );

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddVolumeWithTargetCheck(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("volume2", "hash", 0, 0, false))
                , yexception
                , "treeId 'random_prefix_volume2' does not match to volumeId 'volume2'"
            );

            Ticker_->AddTree(GetEmptyTree(TStatusNTickerHolder::GetVolumeTreeId("volume3")), "hash");
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddVolumeWithTargetCheck(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("volume3"))
                , yexception
                , "VolumeStatusRepository and Ticker unsynchronized: Tree"
            );
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddVolumeWithTargetCheck(NObjectTargetTestLib::CreatePersistentVolumeTargetWithDependence("volume4", "hash", {"layer1", "no_layer"}, {"static_resource1"}))
                , yexception
                , "Volume 'volume4' layerRef 'no_layer' not found in LayerStatusRepository"
            );

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddVolumeWithTargetCheck(NObjectTargetTestLib::CreatePersistentVolumeTargetWithDependence("volume4", "hash", {"layer1"}, {"static_resource1", "no_static_resource"}))
                , yexception
                , "Volume 'volume4' staticResourceRef 'no_static_resource' not found in StaticResourceStatusRepository"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestAddVolumeWithTargetCheck) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));
            UpdateHolder_->SetLayerTarget(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash_other", 2));

            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash"));
            UpdateHolder_->SetStaticResourceTarget(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash_other", 2));

            StatusNTickerHolder_->AddVolumeWithTargetCheck(NObjectTargetTestLib::CreatePersistentVolumeTargetWithDependence("volume1", "hash", {"layer1"}, {"static_resource1"}, 2));

            UNIT_ASSERT(StatusNTickerHolder_->HasVolume("volume1"));
            UNIT_ASSERT(VolumeStatusRepository_->HasObject("volume1"));
            UNIT_ASSERT(Ticker_->HasTree(TStatusNTickerHolder::GetVolumeTreeId("volume1")));

            UNIT_ASSERT(VolumeStatusRepository_->GetObjectStatus("volume1").spec_timestamp() == 0);
            UNIT_ASSERT(VolumeStatusRepository_->GetObjectStatus("volume1").revision() == 0);
            UNIT_ASSERT(Ticker_->GetTreeHash(TStatusNTickerHolder::GetVolumeTreeId("volume1")) == "");
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->VolumeHasTarget("volume1"));
            UNIT_ASSERT(UpdateHolder_->GetAndRemoveVolumeTarget("volume1").Hash_ == "hash");
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestVolumeTargetNoChange) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("volume1"));

            StatusNTickerHolder_->UpdateVolumeTarget(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("volume1"));
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->VolumeHasTarget("volume1"));

            StatusNTickerHolder_->UpdateVolumeTarget(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("volume1", "hash", 0, 42));
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->VolumeHasTarget("volume1"));
            UNIT_ASSERT(VolumeStatusRepository_->GetObjectStatus("volume1").spec_timestamp() == 42);
            UNIT_ASSERT(VolumeStatusRepository_->GetObjectStatus("volume1").revision() == 43);
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateVolumeFromTarget("volume2").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::NO_TARGET);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestVolumeTargetWithUpdateHolder) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));
            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash"));

            StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("volume1"));

            TUpdateHolder::TVolumeTarget newTarget1 = NObjectTargetTestLib::CreatePersistentVolumeTargetWithDependence("volume1", "other_hash", {"layer1"}, {"static_resource1"}, 1);
            TUpdateHolder::TVolumeTarget newTarget2 = NObjectTargetTestLib::CreatePersistentVolumeTargetWithDependence("volume1", "other_hash", {"layer1"}, {"static_resource1"}, 3);
            StatusNTickerHolder_->UpdateVolumeTarget(newTarget1);
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->VolumeHasTarget("volume1"));

            StatusNTickerHolder_->UpdateVolumeTarget(newTarget2);
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->VolumeHasTarget("volume1"));

            TUpdateHolder::TVolumeTarget volumeTarget = UpdateHolder_->GetAndRemoveVolumeTarget("volume1");
            UNIT_ASSERT(volumeTarget == newTarget2);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestVolumeTargetWithWorkload) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("volume1", 0, 0));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetWithDependence("box1", "hash", {}, {}, {"volume1"}));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload1", "hash", "box1"));
            StatusNTickerHolder_->UpdateVolumeTarget(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("volume1", 0, 0));
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->VolumeHasTarget("volume1"));

            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").rootfs_layer_refs().size()
                , 0
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").rootfs_layer_refs().size()
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").static_resource_refs().size()
                , 0
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").static_resource_refs().size()
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").volume_refs().size()
                , 1
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").volume_refs().size()
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").volume_refs(0)
                , "volume1"
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").volume_refs(0)
            );

            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_REMOVED);
            StatusNTickerHolder_->UpdateWorkloadTarget(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload1", "other_hash", "box1"));
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->WorkloadHasTarget("workload1"));
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateVolumeFromTarget("volume1").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::NO_TARGET);
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->VolumeHasTarget("volume1"));
            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_ACTIVE);

            TUpdateHolder::TVolumeTarget newTarget = NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("volume1", "other_hash", 0, 1);
            StatusNTickerHolder_->UpdateVolumeTarget(newTarget);
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->VolumeHasTarget("volume1"));
            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateVolumeFromTarget("volume1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'workload workload1' not removed")
            );
            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateWorkloadFromTarget("workload1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'volume volume1' has target")
            );

            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_REMOVED);
            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateWorkloadFromTarget("workload1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'volume volume1' has target")
            );
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateVolumeFromTarget("volume1").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::UPDATED);
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->VolumeHasTarget("volume1"));
            UNIT_ASSERT(Ticker_->GetTreeHash(TStatusNTickerHolder::GetVolumeTreeId("volume1")) == "other_hash");
            UNIT_ASSERT(VolumeStatusRepository_->GetObjectStatus("volume1").spec_timestamp() == 1);
            UNIT_ASSERT(VolumeStatusRepository_->GetObjectStatus("volume1").revision() == 2);
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateWorkloadFromTarget("workload1").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::UPDATED);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestVolumeTargetWithDepends) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));
            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash"));
            
            StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetWithDependence("volume1", "hash", {"layer1"}, {"static_resource1"}, 0));


            StatusNTickerHolder_->UpdateLayerTarget(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash_other"));
            StatusNTickerHolder_->UpdateStaticResourceTarget(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash_other"));
            StatusNTickerHolder_->UpdateVolumeTarget(NObjectTargetTestLib::CreatePersistentVolumeTargetWithDependence("volume1", "other_hash", {"layer1"}, {"static_resource1"}, 1));
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->LayerHasTarget("layer1"));
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->StaticResourceHasTarget("static_resource1"));
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->VolumeHasTarget("volume1"));

            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateVolumeFromTarget("volume1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'layer layer1' has target")
            );
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateLayerFromTarget("layer1").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::UPDATED);

            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateVolumeFromTarget("volume1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'static_resource static_resource1' has target")
            );

            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateStaticResourceFromTarget("static_resource1").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::UPDATED);

            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateVolumeFromTarget("volume1").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::UPDATED);
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->VolumeHasTarget("volume1"));
            UNIT_ASSERT(Ticker_->GetTreeHash(TStatusNTickerHolder::GetVolumeTreeId("volume1")) == "other_hash");
            UNIT_ASSERT(VolumeStatusRepository_->GetObjectStatus("volume1").spec_timestamp() == 1);
            UNIT_ASSERT(VolumeStatusRepository_->GetObjectStatus("volume1").revision() == 2);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestVolumeTargetExceptions) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));
            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash"));
            StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetWithDependence("volume1", "hash", {"layer1"}, {"static_resource1"}));

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->UpdateVolumeTarget(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("volume1", "hash", 0, 0, false))
                , yexception
                , "treeId 'random_prefix_volume1' does not match to volumeId 'volume1'"
            );
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->UpdateVolumeTarget(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("volume2"))
                , yexception
                , "Volume 'volume2' not found in VolumeStatusRepository"
            );
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->UpdateVolumeTarget(NObjectTargetTestLib::CreatePersistentVolumeTargetWithDependence("volume1", "hash", {"layer1", "no_layer"}, {"static_resource1"}))
                , yexception
                , "Volume 'volume1' layerRef 'no_layer' not found in LayerStatusRepository"
            );

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->UpdateVolumeTarget(NObjectTargetTestLib::CreatePersistentVolumeTargetWithDependence("volume1", "hash", {"layer1"}, {"static_resource1", "no_static_resource"}))
                , yexception
                , "Volume 'volume1' staticResourceRef 'no_static_resource' not found in StaticResourceStatusRepository"
            );

            StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("volume2"));
            Ticker_->RemoveTree(TStatusNTickerHolder::GetVolumeTreeId("volume2"));
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->UpdateVolumeTarget(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("volume2", "other_hash"))
                , yexception
                , "VolumeStatusRepository and Ticker unsynchronized: Tree"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestVolumeTargetRemove) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("volume1"));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetWithDependence("box1", "hash", {}, {}, {"volume1"}));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload1", "hash", "box1"));

            UNIT_ASSERT(StatusNTickerHolder_->HasVolume("volume1"));
            StatusNTickerHolder_->SetVolumeTargetRemove("volume1");
            UNIT_ASSERT(StatusNTickerHolder_->HasVolume("volume1"));

            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateVolumeFromTarget("volume1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'workload workload1' not removed")
            );
            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_REMOVED);
            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateVolumeFromTarget("volume1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'volume volume1' has target remove and has in-edges")
            );

            StatusNTickerHolder_->UpdateBoxTarget(NObjectTargetTestLib::CreateBoxTargetSimple("box1", "other_hash"));
            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateVolumeFromTarget("volume1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'volume volume1' has target remove and has in-edges")
            );
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateBoxFromTarget("box1").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::UPDATED);
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateVolumeFromTarget("volume1").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::REMOVED);
            UNIT_ASSERT(!StatusNTickerHolder_->HasVolume("volume1"));

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->SetVolumeTargetRemove("volume2")
                , yexception
                , "Volume 'volume2' not found in VolumeStatusRepository"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestAddBox) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            UNIT_ASSERT(!StatusNTickerHolder_->GetBoxSystemLogsSender()->Has("box1"));

            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash"));
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));
            StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("volume1"));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetWithDependence("box1", "hash", {"layer1"}, {"static_resource1"}, {"volume1"}));

            UNIT_ASSERT(StatusNTickerHolder_->GetBoxSystemLogsSender()->Has("box1"));
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").rootfs_layer_refs().size()
                , 1
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").rootfs_layer_refs().size()
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").rootfs_layer_refs(0)
                , "layer1"
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").rootfs_layer_refs(0)
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").static_resource_refs().size()
                , 1
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").static_resource_refs().size()
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").static_resource_refs(0)
                , "static_resource1"
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").static_resource_refs(0)
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").volume_refs().size()
                , 1
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").volume_refs().size()
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").volume_refs(0)
                , "volume1"
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").volume_refs(0)
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").inits_size()
                , 3
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").inits_size()
            );

            UNIT_ASSERT(StatusNTickerHolder_->HasBox("box1"));
            UNIT_ASSERT(BoxStatusRepository_->HasObject("box1"));
            UNIT_ASSERT(Ticker_->HasTree(TStatusNTickerHolder::GetBoxTreeId("box1")));

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("box1"))
                , yexception
                , "Box 'box1' already added"
            );

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("box2", "hash", 0, 0, false))
                , yexception
                , "treeId 'random_prefix_box2' does not match to boxId 'box2'"
            );

            Ticker_->AddTree(GetEmptyTree(TStatusNTickerHolder::GetBoxTreeId("box3")), "hash");
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("box3"))
                , yexception
                , "BoxStatusRepository and Ticker unsynchronized: Tree"
            );

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetWithDependence("box4", "hash", {"layer1", "no_layer"}))
                , yexception
                , "Box 'box4' rootfsLayerRef 'no_layer' not found in LayerStatusRepository"
            );
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetWithDependence("box4", "hash", {}, {"static_resource1", "no_static_resource"}, {}))
                , yexception
                , "Box 'box4' staticResourceRef 'no_static_resource' not found in StaticResourceStatusRepository"
            );

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetWithDependence("box4", "hash", {}, {}, {"volume1", "no_volume"}))
                , yexception
                , "Box 'box4' volumeRef 'no_volume' not found in VolumeStatusRepository"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveBox) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("box1"));
            
            UNIT_ASSERT(StatusNTickerHolder_->GetBoxSystemLogsSender()->Has("box1"));
            UNIT_ASSERT(StatusNTickerHolder_->HasBox("box1"));
            UNIT_ASSERT(BoxStatusRepository_->HasObject("box1"));
            UNIT_ASSERT(Ticker_->HasTree(TStatusNTickerHolder::GetBoxTreeId("box1")));

            StatusNTickerHolder_->RemoveBox("box1");

            UNIT_ASSERT(!StatusNTickerHolder_->GetBoxSystemLogsSender()->Has("box1"));
            UNIT_ASSERT(!StatusNTickerHolder_->HasBox("box1"));
            UNIT_ASSERT(!BoxStatusRepository_->HasObject("box1"));
            UNIT_ASSERT(!Ticker_->HasTree(TStatusNTickerHolder::GetBoxTreeId("box1")));

            UNIT_ASSERT_EXCEPTION_CONTAINS(StatusNTickerHolder_->RemoveBox("box2"), yexception, "Box 'box2' not found in BoxStatusRepository");

            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("box2"));
            Ticker_->RemoveTree(TStatusNTickerHolder::GetBoxTreeId("box2"));
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->RemoveBox("box2")
                , yexception
                , "BoxStatusRepository and Ticker unsynchronized: Tree"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestAddBoxWithTargetCheckAsAddBox) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash"));
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));
            StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("volume1"));
            StatusNTickerHolder_->AddBoxWithTargetCheck(NObjectTargetTestLib::CreateBoxTargetWithDependence("box1", "hash", {"layer1"}, {"static_resource1"}, {"volume1"}));

            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").rootfs_layer_refs().size()
                , 1
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").rootfs_layer_refs().size()
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").rootfs_layer_refs(0)
                , "layer1"
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").rootfs_layer_refs(0)
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").static_resource_refs().size()
                , 1
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").static_resource_refs().size()
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").static_resource_refs(0)
                , "static_resource1"
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").static_resource_refs(0)
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").volume_refs().size()
                , 1
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").volume_refs().size()
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").volume_refs(0)
                , "volume1"
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").volume_refs(0)
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").inits_size()
                , 3
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").inits_size()
            );

            UNIT_ASSERT(StatusNTickerHolder_->HasBox("box1"));
            UNIT_ASSERT(BoxStatusRepository_->HasObject("box1"));
            UNIT_ASSERT(Ticker_->HasTree(TStatusNTickerHolder::GetBoxTreeId("box1")));

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddBoxWithTargetCheck(NObjectTargetTestLib::CreateBoxTargetSimple("box1"))
                , yexception
                , "Box 'box1' already added"
            );

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddBoxWithTargetCheck(NObjectTargetTestLib::CreateBoxTargetSimple("box2", "hash", 0, 0, false))
                , yexception
                , "treeId 'random_prefix_box2' does not match to boxId 'box2'"
            );

            Ticker_->AddTree(GetEmptyTree(TStatusNTickerHolder::GetBoxTreeId("box3")), "hash");
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddBoxWithTargetCheck(NObjectTargetTestLib::CreateBoxTargetSimple("box3"))
                , yexception
                , "BoxStatusRepository and Ticker unsynchronized: Tree"
            );

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddBoxWithTargetCheck(NObjectTargetTestLib::CreateBoxTargetWithDependence("box4", "hash", {"layer1", "no_layer"}))
                , yexception
                , "Box 'box4' rootfsLayerRef 'no_layer' not found in LayerStatusRepository"
            );
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetWithDependence("box4", "hash", {}, {"static_resource1", "no_static_resource"}, {}))
                , yexception
                , "Box 'box4' staticResourceRef 'no_static_resource' not found in StaticResourceStatusRepository"
            );

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetWithDependence("box4", "hash", {}, {}, {"volume1", "no_volume"}))
                , yexception
                , "Box 'box4' volumeRef 'no_volume' not found in VolumeStatusRepository"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestAddBoxWithTargetCheck) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash"));
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));
            UpdateHolder_->SetLayerTarget(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash_other"));
            StatusNTickerHolder_->AddBoxWithTargetCheck(NObjectTargetTestLib::CreateBoxTargetWithDependence("box1", "hash", {"layer1"}, {"static_resource1"}));

            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").rootfs_layer_refs().size()
                , 1
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").rootfs_layer_refs().size()
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").rootfs_layer_refs(0)
                , "layer1"
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").rootfs_layer_refs(0)
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").static_resource_refs().size()
                , 1
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").static_resource_refs().size()
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").static_resource_refs(0)
                , "static_resource1"
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").static_resource_refs(0)
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").volume_refs().size()
                , 0
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").volume_refs().size()
            );
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").inits_size()
                , 3
                , StatusNTickerHolder_->GetBoxStatusRepository()->GetObjectStatus("box1").inits_size()
            );

            UNIT_ASSERT(StatusNTickerHolder_->HasBox("box1"));
            UNIT_ASSERT(BoxStatusRepository_->HasObject("box1"));
            UNIT_ASSERT(Ticker_->HasTree(TStatusNTickerHolder::GetBoxTreeId("box1")));

            UNIT_ASSERT(BoxStatusRepository_->GetObjectStatus("box1").spec_timestamp() == 0);
            UNIT_ASSERT(BoxStatusRepository_->GetObjectStatus("box1").revision() == 0);
            UNIT_ASSERT(Ticker_->GetTreeHash(TStatusNTickerHolder::GetBoxTreeId("box1")) == "");
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->BoxHasTarget("box1"));

            TUpdateHolder::TBoxTarget boxTarget = UpdateHolder_->GetAndRemoveBoxTarget("box1");
            UNIT_ASSERT(boxTarget.Hash_ == "hash");
            UNIT_ASSERT(boxTarget.Meta_.InitContainers_.size() == 3u);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestBoxTargetNoChange) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("box1"));

            StatusNTickerHolder_->UpdateBoxTarget(NObjectTargetTestLib::CreateBoxTargetSimple("box1"));
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->BoxHasTarget("box1"));

            StatusNTickerHolder_->UpdateBoxTarget(NObjectTargetTestLib::CreateBoxTargetSimple("box1", "hash", 0, 42));
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->BoxHasTarget("box1"));
            UNIT_ASSERT(BoxStatusRepository_->GetObjectStatus("box1").spec_timestamp() == 42);
            UNIT_ASSERT(BoxStatusRepository_->GetObjectStatus("box1").revision() == 43);
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateBoxFromTarget("box2").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::NO_TARGET);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestBoxTargetWithUpdateHolder) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash"));
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));
            StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("volume1"));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("box1"));

            TUpdateHolder::TBoxTarget newTarget1 = NObjectTargetTestLib::CreateBoxTargetWithDependence("box1", "other_hash", {"layer1"}, {"static_resource1"}, {"volume1"}, 1);
            TUpdateHolder::TBoxTarget newTarget2 = NObjectTargetTestLib::CreateBoxTargetWithDependence("box1", "other_hash", {"layer1"}, {"static_resource1"}, {"volume1"}, 3);
            StatusNTickerHolder_->UpdateBoxTarget(newTarget1);
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->BoxHasTarget("box1"));

            StatusNTickerHolder_->UpdateBoxTarget(newTarget2);
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->BoxHasTarget("box1"));

            TUpdateHolder::TBoxTarget boxTarget = UpdateHolder_->GetAndRemoveBoxTarget("box1");
            UNIT_ASSERT(boxTarget == newTarget2);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestBoxTargetWithWorkload) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("box1"));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload1", "hash", "box1"));
            StatusNTickerHolder_->UpdateBoxTarget(NObjectTargetTestLib::CreateBoxTargetWithDependence("box1"));
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->BoxHasTarget("box1"));

            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_REMOVED);
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateBoxFromTarget("box1").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::NO_TARGET);
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->BoxHasTarget("box1"));
            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_ACTIVE);

            TUpdateHolder::TBoxTarget newTarget = NObjectTargetTestLib::CreateBoxTargetSimple("box1", "other_hash", 0, 3);
            StatusNTickerHolder_->UpdateBoxTarget(newTarget);
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->BoxHasTarget("box1"));
            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateBoxFromTarget("box1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'workload workload1' not removed")
            );

            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_REMOVED);
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateBoxFromTarget("box1").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::UPDATED);
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->BoxHasTarget("box1"));
            UNIT_ASSERT(Ticker_->GetTreeHash(TStatusNTickerHolder::GetBoxTreeId("box1")) == "other_hash");
            UNIT_ASSERT(BoxStatusRepository_->GetObjectStatus("box1").spec_timestamp() == 3);
            UNIT_ASSERT(BoxStatusRepository_->GetObjectStatus("box1").revision() == 4);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestBoxTargetWithDepends) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash"));
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));
            StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("volume1"));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetWithDependence("box1", "hash", {"layer1"}, {"static_resource1"}, {"volume1"}));

            StatusNTickerHolder_->UpdateStaticResourceTarget(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash_other"));
            StatusNTickerHolder_->UpdateLayerTarget(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));
            StatusNTickerHolder_->UpdateVolumeTarget(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("volume1", "other_hash"));
            StatusNTickerHolder_->UpdateBoxTarget(NObjectTargetTestLib::CreateBoxTargetWithDependence("box1", "other_hash", {"layer1"}, {}, {"volume1"}, 3));
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->LayerHasTarget("layer1"));
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->VolumeHasTarget("volume1"));
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->BoxHasTarget("box1"));

            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateBoxFromTarget("box1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'volume volume1' has target")
            );
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateVolumeFromTarget("volume1").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::UPDATED);
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateBoxFromTarget("box1").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::UPDATED);
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->BoxHasTarget("box1"));
            UNIT_ASSERT(Ticker_->GetTreeHash(TStatusNTickerHolder::GetBoxTreeId("box1")) == "other_hash");
            UNIT_ASSERT(BoxStatusRepository_->GetObjectStatus("box1").spec_timestamp() == 3);
            UNIT_ASSERT(BoxStatusRepository_->GetObjectStatus("box1").revision() == 4);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestBoxTargetExceptions) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash"));
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));
            StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("volume1"));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("box1"));

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->UpdateBoxTarget(NObjectTargetTestLib::CreateBoxTargetSimple("box1", "hash", 0, 0, false))
                , yexception
                , "treeId 'random_prefix_box1' does not match to boxId 'box1'"
            );
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->UpdateBoxTarget(NObjectTargetTestLib::CreateBoxTargetSimple("box2"))
                , yexception
                , "Box 'box2' not found in BoxStatusRepository"
            );
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->UpdateBoxTarget(NObjectTargetTestLib::CreateBoxTargetWithDependence("box1", "hash", {"layer1", "no_layer"}))
                , yexception
                , "Box 'box1' rootfsLayerRef 'no_layer' not found in LayerStatusRepository"
            );
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->UpdateBoxTarget(NObjectTargetTestLib::CreateBoxTargetWithDependence("box1", "hash", {}, {"static_resource1", "no_static_resource"}))
                , yexception
                , "Box 'box1' staticResourceRef 'no_static_resource' not found in StaticResourceStatusRepository"
            );
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->UpdateBoxTarget(NObjectTargetTestLib::CreateBoxTargetWithDependence("box1", "hash", {}, {}, {"volume1", "no_volume"}))
                , yexception
                , "Box 'box1' volumeRef 'no_volume' not found in VolumeStatusRepository"
            );

            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("box2"));
            Ticker_->RemoveTree(TStatusNTickerHolder::GetBoxTreeId("box2"));
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->UpdateBoxTarget(NObjectTargetTestLib::CreateBoxTargetSimple("box2"))
                , yexception
                , "BoxStatusRepository and Ticker unsynchronized: Tree"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestBoxTargetRemove) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("box1"));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("box2"));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload1", "hash", "box1"));

            UNIT_ASSERT(StatusNTickerHolder_->HasBox("box1"));
            StatusNTickerHolder_->SetBoxTargetRemove("box1");
            UNIT_ASSERT(StatusNTickerHolder_->HasBox("box1"));

            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateBoxFromTarget("box1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'workload workload1' not removed")
            );
            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_REMOVED);
            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateBoxFromTarget("box1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'box box1' has target remove and has in-edges")
            );

            StatusNTickerHolder_->UpdateWorkloadTarget(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload1", "other_hash", "box2"));
            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateBoxFromTarget("box1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'box box1' has target remove and has in-edges")
            );
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateWorkloadFromTarget("workload1").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::UPDATED);
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateBoxFromTarget("box1").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::REMOVED);
            UNIT_ASSERT(!StatusNTickerHolder_->HasBox("box1"));

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->SetBoxTargetRemove("box3")
                , yexception
                , "Box 'box3' not found in BoxStatusRepository"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestEnableDisableLogsTransmitting) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            UNIT_ASSERT(!StatusNTickerHolder_->GetWorkloadSystemLogsSender()->GetMarkEnabled());
            UNIT_ASSERT(!StatusNTickerHolder_->GetBoxSystemLogsSender()->GetMarkEnabled());

            StatusNTickerHolder_->EnableTransmitSystemLogs(true);

            UNIT_ASSERT(StatusNTickerHolder_->GetWorkloadSystemLogsSender()->GetMarkEnabled());
            UNIT_ASSERT(StatusNTickerHolder_->GetBoxSystemLogsSender()->GetMarkEnabled());
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestAddWorkload) {
    class TTest : public ITestStatusNTickerHolderCanon {
    public:
        TTest(const bool isBoxMode) : ITestStatusNTickerHolderCanon(isBoxMode)
        {}

    protected:
        void Test() override {
            UNIT_ASSERT(!StatusNTickerHolder_->GetWorkloadSystemLogsSender()->Has("workload1"));

            if (!IsBoxAgentMode_) {
                StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("boxRef0"));
            }
            StatusNTickerHolder_->AddWorkload(CreateWorkloadTargetSimple("workload1"));

            UNIT_ASSERT(StatusNTickerHolder_->GetWorkloadSystemLogsSender()->Has("workload1"));
            UNIT_ASSERT(StatusNTickerHolder_->HasWorkload("workload1"));
            UNIT_ASSERT(WorkloadStatusRepository_->HasObject("workload1"));
            UNIT_ASSERT(WorkloadStatusRepositoryInternal_->HasObject("workload1"));
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetWorkloadStatusRepository()->GetObjectStatus("workload1").init_size()
                , 3
                , StatusNTickerHolder_->GetWorkloadStatusRepository()->GetObjectStatus("workload1").init_size()
            );

            UNIT_ASSERT(Ticker_->HasTree(TStatusNTickerHolder::GetWorkloadTreeId("workload1")));

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddWorkload(CreateWorkloadTargetSimple("workload1"))
                , yexception
                , "Workload 'workload1' already added"
            );
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddWorkload(CreateWorkloadTargetSimple("workload2", "hash", 1, 0, 0, false, false))
                , yexception
                , "treeId 'random_prefix_workload2' does not match to workloadId 'workload2'"
            );

            Ticker_->AddTree(GetEmptyTree(TStatusNTickerHolder::GetWorkloadTreeId("workload3")), "hash");
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddWorkload(CreateWorkloadTargetSimple("workload3"))
                , yexception
                , "WorkloadStatusRepository and Ticker unsynchronized: Tree"
            );

            if (!IsBoxAgentMode_) {
                UNIT_ASSERT_EXCEPTION_CONTAINS(
                    StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload4", "hash", "noRef"))
                    , yexception
                    , "Workload 'workload4' boxRef 'noRef' not found in BoxStatusRepository"
                );
            }

            WorkloadStatusRepositoryInternal_->AddObject("workload5");
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddWorkload(CreateWorkloadTargetSimple("workload5"))
                , yexception
                , "WorkloadStatusRepository and WorkloadStatusRepositoryInternal"
            );
        }
    };

    for (const bool isBoxMode : {false, true}) {
        TTest test(isBoxMode);
        test.DoTest();
    }
}

Y_UNIT_TEST(TestRemoveWorkload) {
    class TTest : public ITestStatusNTickerHolderCanon {
    public:
        TTest(const bool isBoxMode) : ITestStatusNTickerHolderCanon(isBoxMode)
        {}

    protected:
        void Test() override {
            if (!IsBoxAgentMode_) {
                StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("boxRef0"));
            }
            StatusNTickerHolder_->AddWorkload(CreateWorkloadTargetSimple("workload1"));


            UNIT_ASSERT(StatusNTickerHolder_->GetWorkloadSystemLogsSender()->Has("workload1"));
            UNIT_ASSERT(StatusNTickerHolder_->HasWorkload("workload1"));
            UNIT_ASSERT(WorkloadStatusRepository_->HasObject("workload1"));
            UNIT_ASSERT(WorkloadStatusRepositoryInternal_->HasObject("workload1"));
            UNIT_ASSERT(Ticker_->HasTree(TStatusNTickerHolder::GetWorkloadTreeId("workload1")));

            StatusNTickerHolder_->RemoveWorkload("workload1");

            UNIT_ASSERT(!StatusNTickerHolder_->GetWorkloadSystemLogsSender()->Has("workload1"));
            UNIT_ASSERT(!StatusNTickerHolder_->HasWorkload("workload1"));
            UNIT_ASSERT(!WorkloadStatusRepository_->HasObject("workload1"));
            UNIT_ASSERT(!WorkloadStatusRepositoryInternal_->HasObject("workload1"));
            UNIT_ASSERT(!Ticker_->HasTree(TStatusNTickerHolder::GetWorkloadTreeId("workload1")));

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->RemoveWorkload("workload2")
                , yexception
                , "Workload 'workload2' not found in WorkloadStatusRepository"
            );

            StatusNTickerHolder_->AddWorkload(CreateWorkloadTargetSimple("workload2"));
            Ticker_->RemoveTree(TStatusNTickerHolder::GetWorkloadTreeId("workload2"));
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->RemoveWorkload("workload2")
                , yexception
                , "WorkloadStatusRepository and Ticker unsynchronized: Tree"
            );

            StatusNTickerHolder_->AddWorkload(CreateWorkloadTargetSimple("workload3"));
            WorkloadStatusRepositoryInternal_->RemoveObject("workload3");
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->RemoveWorkload("workload3")
                , yexception
                , "WorkloadStatusRepository and WorkloadStatusRepositoryInternal unsynchronized: Workload"
            );
        }
    };

    for (const bool isBoxMode : {false, true}) {
        TTest test(isBoxMode);
        test.DoTest();
    }
}

Y_UNIT_TEST(TestAddWorkloadWithTargetCheckAsAddWorkload) {
    class TTest : public ITestStatusNTickerHolderCanon {
    public:
        TTest(const bool isBoxMode) : ITestStatusNTickerHolderCanon(isBoxMode)
        {}

    protected:
        void Test() override {
            if (!IsBoxAgentMode_) {
                StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("boxRef0"));
            }
            StatusNTickerHolder_->AddWorkloadWithTargetCheck(CreateWorkloadTargetSimple("workload1"));

            UNIT_ASSERT(StatusNTickerHolder_->HasWorkload("workload1"));
            UNIT_ASSERT(WorkloadStatusRepository_->HasObject("workload1"));
            UNIT_ASSERT(WorkloadStatusRepositoryInternal_->HasObject("workload1"));
            UNIT_ASSERT_EQUAL_C(
                StatusNTickerHolder_->GetWorkloadStatusRepository()->GetObjectStatus("workload1").init_size()
                , 3
                , StatusNTickerHolder_->GetWorkloadStatusRepository()->GetObjectStatus("workload1").init_size()
            );

            UNIT_ASSERT(Ticker_->HasTree(TStatusNTickerHolder::GetWorkloadTreeId("workload1")));

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddWorkloadWithTargetCheck(CreateWorkloadTargetSimple("workload1"))
                , yexception
                , "Workload 'workload1' already added"
            );
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddWorkloadWithTargetCheck(CreateWorkloadTargetSimple("workload2", "hash", 1, 0, 0, false, false))
                , yexception
                , "treeId 'random_prefix_workload2' does not match to workloadId 'workload2'"
            );

            Ticker_->AddTree(GetEmptyTree(TStatusNTickerHolder::GetWorkloadTreeId("workload3")), "hash");
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddWorkloadWithTargetCheck(CreateWorkloadTargetSimple("workload3"))
                , yexception
                , "WorkloadStatusRepository and Ticker unsynchronized: Tree"
            );

            if (!IsBoxAgentMode_) {
                UNIT_ASSERT_EXCEPTION_CONTAINS(
                    StatusNTickerHolder_->AddWorkloadWithTargetCheck(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload4", "hash", "noRef"))
                    , yexception
                    , "Workload 'workload4' boxRef 'noRef' not found in BoxStatusRepository"
                );
            }

            WorkloadStatusRepositoryInternal_->AddObject("workload5");
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->AddWorkloadWithTargetCheck(CreateWorkloadTargetSimple("workload5"))
                , yexception
                , "WorkloadStatusRepository and WorkloadStatusRepositoryInternal"
            );
        }
    };

    for (const bool isBoxMode : {false, true}) {
        TTest test(isBoxMode);
        test.DoTest();
    }
}

Y_UNIT_TEST(TestAddWorkloadWithTargetCheck) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            // We don't check this test with box agent mode because this test check that workload is not active due to box hash
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("boxRef0"));
            UpdateHolder_->SetBoxTarget(NObjectTargetTestLib::CreateBoxTargetSimple("boxRef0", "other_hash"));
            StatusNTickerHolder_->AddWorkloadWithTargetCheck(CreateWorkloadTargetSimple("workload1"));

            UNIT_ASSERT(StatusNTickerHolder_->HasWorkload("workload1"));
            UNIT_ASSERT(WorkloadStatusRepository_->HasObject("workload1"));
            UNIT_ASSERT(WorkloadStatusRepositoryInternal_->HasObject("workload1"));
            UNIT_ASSERT(Ticker_->HasTree(TStatusNTickerHolder::GetWorkloadTreeId("workload1")));

            UNIT_ASSERT(WorkloadStatusRepository_->GetObjectStatus("workload1").spec_timestamp() == 0);
            UNIT_ASSERT(WorkloadStatusRepository_->GetObjectStatus("workload1").revision() == 0);
            UNIT_ASSERT(WorkloadStatusRepository_->GetObjectStatus("workload1").state() == API::EWorkloadState_REMOVED);
            if (IsBoxAgentMode_) {
                UNIT_ASSERT(WorkloadStatusRepository_->GetObjectStatus("workload1").box_ref() == "");
            } else {
                UNIT_ASSERT(WorkloadStatusRepository_->GetObjectStatus("workload1").box_ref() == "boxRef0");
            }
            UNIT_ASSERT(Ticker_->GetTreeHash(TStatusNTickerHolder::GetWorkloadTreeId("workload1")) == "");
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->WorkloadHasTarget("workload1"));

            TUpdateHolder::TWorkloadTarget workloadTarget = UpdateHolder_->GetAndRemoveWorkloadTarget("workload1");
            UNIT_ASSERT(workloadTarget.Hash_ == "hash");
            UNIT_ASSERT(!std::holds_alternative<TWorkloadMeta::TEmptyInfo>(workloadTarget.Meta_.Readiness_));
            UNIT_ASSERT(workloadTarget.Meta_.InitContainers_.size() == 3u);
        }
    };
}

Y_UNIT_TEST(TestWorkloadUpdateTargetState) {
    class TTest : public ITestStatusNTickerHolderCanon {
    public:
        TTest(const bool isBoxMode) : ITestStatusNTickerHolderCanon(isBoxMode)
        {}

    protected:
        void Test() override {
            if (!IsBoxAgentMode_) {
                StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("boxRef0"));
            }
            StatusNTickerHolder_->AddWorkload(CreateWorkloadTargetSimple("workload1"));


            StatusNTickerHolder_->UpdateWorkloadTargetState("workload1", API::EWorkloadTarget_ACTIVE);
            UNIT_ASSERT_EQUAL(WorkloadStatusRepository_->GetObjectTargetState("workload1"), API::EWorkloadTarget_ACTIVE);

            StatusNTickerHolder_->UpdateWorkloadTargetState("workload1", API::EWorkloadTarget_REMOVED);
            UNIT_ASSERT_EQUAL(WorkloadStatusRepository_->GetObjectTargetState("workload1"), API::EWorkloadTarget_REMOVED);
        }
    };

    for (const bool isBoxMode : {false, true}) {
        TTest test(isBoxMode);
        test.DoTest();
    }
}

Y_UNIT_TEST(TestWorkloadTargetNoChange) {
    class TTest : public ITestStatusNTickerHolderCanon {
    public:
        TTest(const bool isBoxMode) : ITestStatusNTickerHolderCanon(isBoxMode)
        {}

    protected:
        void Test() override {
            if (!IsBoxAgentMode_) {
                StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("boxRef0"));
            }
            StatusNTickerHolder_->AddWorkload(CreateWorkloadTargetSimple("workload1"));

            StatusNTickerHolder_->UpdateWorkloadTarget(CreateWorkloadTargetSimple("workload1"));
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->WorkloadHasTarget("workload1"));

            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_REMOVED);
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateWorkloadFromTarget("workload1").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::NO_TARGET);
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->WorkloadHasTarget("workload1"));

            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_ACTIVE);
            StatusNTickerHolder_->UpdateWorkloadTarget(CreateWorkloadTargetSimple("workload1", "hash", 1, 0, 42));
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->WorkloadHasTarget("workload1"));
            UNIT_ASSERT(WorkloadStatusRepository_->GetObjectStatus("workload1").spec_timestamp() == 42);
            UNIT_ASSERT(WorkloadStatusRepository_->GetObjectStatus("workload1").revision() == 43);
            if (IsBoxAgentMode_) {
                UNIT_ASSERT(WorkloadStatusRepository_->GetObjectStatus("workload1").box_ref() == "");
            } else {
                UNIT_ASSERT(WorkloadStatusRepository_->GetObjectStatus("workload1").box_ref() == "boxRef0");
            }
        }
    };

    for (const bool isBoxMode : {false, true}) {
        TTest test(isBoxMode);
        test.DoTest();
    }
}

Y_UNIT_TEST(TestWorkloadTargetWithUpdateHolder) {
    class TTest : public ITestStatusNTickerHolderCanon {
    public:
        TTest(const bool isBoxMode) : ITestStatusNTickerHolderCanon(isBoxMode)
        {}

    protected:
        void Test() override {
            if (!IsBoxAgentMode_) {
                StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("boxRef0"));
            }
            StatusNTickerHolder_->AddWorkload(CreateWorkloadTargetSimple("workload1"));

            TUpdateHolder::TWorkloadTarget newTarget1 = CreateWorkloadTargetSimple("workload1", "other_hash", 1, 0, 1);
            TUpdateHolder::TWorkloadTarget newTarget2 = CreateWorkloadTargetSimple("workload1", "other_hash", 1, 0, 4);
            StatusNTickerHolder_->UpdateWorkloadTarget(newTarget1);
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->WorkloadHasTarget("workload1"));

            StatusNTickerHolder_->UpdateWorkloadTarget(newTarget2);
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->WorkloadHasTarget("workload1"));
            TUpdateHolder::TWorkloadTarget workloadTarget = UpdateHolder_->GetAndRemoveWorkloadTarget("workload1");
            UNIT_ASSERT(workloadTarget == newTarget2);
        }
    };

    for (const bool isBoxMode : {false, true}) {
        TTest test(isBoxMode);
        test.DoTest();
    }
}

Y_UNIT_TEST(TestWorkloadTargetWithDepends) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("boxRef0"));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetSimple("workload1"));

            StatusNTickerHolder_->UpdateWorkloadTarget(NObjectTargetTestLib::CreateWorkloadTargetSimple("workload1", "other_hash", 1, 0, 2));
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->WorkloadHasTarget("workload1"));

            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_ACTIVE);
            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateWorkloadFromTarget("workload1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'workload workload1' not removed")
            );
            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_REMOVED);
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateWorkloadFromTarget("workload1").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::UPDATED);
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->WorkloadHasTarget("workload1"));
            UNIT_ASSERT(Ticker_->GetTreeHash(TStatusNTickerHolder::GetWorkloadTreeId("workload1")) == "other_hash");
            UNIT_ASSERT(WorkloadStatusRepository_->GetObjectStatus("workload1").spec_timestamp() == 2);
            UNIT_ASSERT(WorkloadStatusRepository_->GetObjectStatus("workload1").revision() == 3);
            UNIT_ASSERT(WorkloadStatusRepository_->GetObjectStatus("workload1").box_ref() == "boxRef0");
            UNIT_ASSERT(WorkloadStatusRepository_->GetObjectStatus("workload1").init_size() == 4);

            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("box1"));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload2", "hash", "box1"));

            UpdateHolder_->SetBoxTarget(NObjectTargetTestLib::CreateBoxTargetSimple("box1", "other_hash"));
            StatusNTickerHolder_->UpdateWorkloadTarget(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload2", "other_hash", "box1"));
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->WorkloadHasTarget("workload2"));

            WorkloadStatusRepository_->UpdateObjectState("workload2", API::EWorkloadState_REMOVED);
            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateWorkloadFromTarget("workload2")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'box box1' has target")
            );

            UpdateHolder_->GetAndRemoveBoxTarget("box1");
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateWorkloadFromTarget("workload2").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::UPDATED);

            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("box2"));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload3", "hash", "box2"));

            UpdateHolder_->SetBoxTarget(NObjectTargetTestLib::CreateBoxTargetSimple("box2", "other_hash"));
            StatusNTickerHolder_->UpdateWorkloadTarget(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload3", "other_hash", "box1"));
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->WorkloadHasTarget("workload3"));

            WorkloadStatusRepository_->UpdateObjectState("workload3", API::EWorkloadState_REMOVED);
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateWorkloadFromTarget("workload3").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::UPDATED);

            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetWithDependence("box3", "hash", {"layer1"}));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload4", "hash", "box3"));
            StatusNTickerHolder_->UpdateLayerTarget(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash_other"));
            WorkloadStatusRepository_->UpdateObjectState("workload4", API::EWorkloadState_ACTIVE);
            StatusNTickerHolder_->UpdateWorkloadTarget(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload4", "other_hash", "box3", true, 4));
            UNIT_ASSERT(WorkloadStatusRepository_->GetObjectStatus("workload4").spec_timestamp() == 1);
            UNIT_ASSERT(WorkloadStatusRepository_->GetObjectStatus("workload4").revision() == 2);
            UNIT_ASSERT(WorkloadStatusRepository_->GetObjectStatus("workload4").box_ref() == "box3");
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadTargetExceptions) {
    class TTest : public ITestStatusNTickerHolderCanon {
    public:
        TTest(const bool isBoxMode) : ITestStatusNTickerHolderCanon(isBoxMode)
        {}

    protected:
        void Test() override {
            if (!IsBoxAgentMode_) {
                StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("boxRef0"));
            }
            StatusNTickerHolder_->AddWorkload(CreateWorkloadTargetSimple("workload1"));

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->UpdateWorkloadTarget(CreateWorkloadTargetSimple("workload1", "hash", 1, 0, 0, false, false))
                , yexception
                , "treeId 'random_prefix_workload1' does not match to workloadId 'workload1'"
            );
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->UpdateWorkloadTarget(CreateWorkloadTargetSimple("workload2"))
                , yexception
                , "Workload 'workload2' not found in WorkloadStatusRepository"
            );
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateWorkloadFromTarget("workload2").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::NO_TARGET);

            StatusNTickerHolder_->AddWorkload(CreateWorkloadTargetSimple("workload2"));
            Ticker_->RemoveTree(TStatusNTickerHolder::GetWorkloadTreeId("workload2"));
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->UpdateWorkloadTarget(CreateWorkloadTargetSimple("workload2"))
                , yexception
                , "WorkloadStatusRepository and Ticker unsynchronized: Tree"
            );

            if (!IsBoxAgentMode_) {
                UNIT_ASSERT_EXCEPTION_CONTAINS(
                    StatusNTickerHolder_->UpdateWorkloadTarget(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload1", "hash", "noRef"))
                    , yexception
                    , "Workload 'workload1' boxRef 'noRef' not found in BoxStatusRepository"
                );
            }

            StatusNTickerHolder_->AddWorkload(CreateWorkloadTargetSimple("workload3"));
            WorkloadStatusRepositoryInternal_->RemoveObject("workload3");
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->UpdateWorkloadTarget(CreateWorkloadTargetSimple("workload3"))
                , yexception
                , "WorkloadStatusRepository and WorkloadStatusRepositoryInternal"
            );
        }
    };

    for (const bool isBoxMode : {false, true}) {
        TTest test(isBoxMode);
        test.DoTest();
    }
}

Y_UNIT_TEST(TestWorkloadTargetRemoveWhenWorkloadDestroyNotificationIsFalse) {
    class TTest : public ITestStatusNTickerHolderCanon {
    public:
        TTest(const bool isBoxMode) : ITestStatusNTickerHolderCanon(isBoxMode)
        {}

    protected:
        void Test() override {
            if (!IsBoxAgentMode_) {
                StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("boxRef0"));
            }
            StatusNTickerHolder_->AddWorkload(CreateWorkloadTargetSimple("workload1"));

            UNIT_ASSERT(WorkloadStatusRepositoryInternal_->HasObject("workload1"));
            UNIT_ASSERT(!WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook("workload1"));

            UNIT_ASSERT(StatusNTickerHolder_->HasWorkload("workload1"));
            StatusNTickerHolder_->SetWorkloadTargetRemove("workload1");
            UNIT_ASSERT(StatusNTickerHolder_->HasWorkload("workload1"));

            UNIT_ASSERT(WorkloadStatusRepositoryInternal_->HasObject("workload1"));
            UNIT_ASSERT(WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook("workload1"));

            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateWorkloadFromTarget("workload1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'workload workload1' not removed")
            );
            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_REMOVED);
            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateWorkloadFromTarget("workload1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'workload workload1' not destroyed by hook")
            );
            WorkloadStatusRepositoryInternal_->UpdateObjectToBeDestroyedByHook("workload1", false);
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateWorkloadFromTarget("workload1").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::REMOVED);
            UNIT_ASSERT(!StatusNTickerHolder_->HasWorkload("workload1"));
            UNIT_ASSERT(!WorkloadStatusRepositoryInternal_->HasObject("workload1"));

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                StatusNTickerHolder_->SetWorkloadTargetRemove("workload2")
                , yexception
                , "Workload 'workload2' not found in WorkloadStatusRepository"
            );
        }
    };

    for (const bool isBoxMode : {false, true}) {
        TTest test(isBoxMode);
        test.DoTest();
    }
}


Y_UNIT_TEST(TestWorkloadTargetRemoveWhenWorkloadDestroyNotificationIsTrue) {
    class TTest : public ITestStatusNTickerHolderCanon {
    public:
        TTest(const bool isBoxMode) : ITestStatusNTickerHolderCanon(isBoxMode)
        {}

    protected:
        void Test() override {
            if (!IsBoxAgentMode_) {
                StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("boxRef0"));
            }
            StatusNTickerHolder_->AddWorkload(CreateWorkloadTargetSimple("workload1"));

            UNIT_ASSERT(WorkloadStatusRepositoryInternal_->HasObject("workload1"));
            UNIT_ASSERT(!WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook("workload1"));

            UNIT_ASSERT(StatusNTickerHolder_->HasWorkload("workload1"));
            StatusNTickerHolder_->SetWorkloadTargetRemove("workload1");
            UNIT_ASSERT(StatusNTickerHolder_->HasWorkload("workload1"));

            UNIT_ASSERT(WorkloadStatusRepositoryInternal_->HasObject("workload1"));
            UNIT_ASSERT(WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook("workload1"));

            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateWorkloadFromTarget("workload1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'workload workload1' not removed")
            );
            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_REMOVED);
            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateWorkloadFromTarget("workload1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'workload workload1' not destroyed by hook")
            );
            WorkloadStatusRepositoryInternal_->UpdateObjectToBeDestroyedByHook("workload1", true);
            UNIT_ASSERT_EQUAL(
                StatusNTickerHolder_->TryToUpdateWorkloadFromTarget("workload1")
                , TStatusNTickerHolder::TUpdateObjectResult(TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE, "'workload workload1' not destroyed by hook")
            );
            UNIT_ASSERT(StatusNTickerHolder_->HasWorkload("workload1"));
            UNIT_ASSERT(WorkloadStatusRepositoryInternal_->HasObject("workload1"));

            StatusNTickerHolder_->AddWorkload(CreateWorkloadTargetSimple("workload2", "hash", 1, 0, 0));
            StatusNTickerHolder_->SetWorkloadTargetRemove("workload2");
            WorkloadStatusRepository_->UpdateObjectState("workload2", API::EWorkloadState_REMOVED);
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateWorkloadFromTarget("workload2").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::REMOVED);
        }
    };

    for (const bool isBoxMode : {false, true}) {
        TTest test(isBoxMode);
        test.DoTest();
    }
}

Y_UNIT_TEST(TestGets) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->SetStatusRepositorySpecTimestamp(42);
            StatusNTickerHolder_->SetStatusRepositoryRevision(43);
            StatusNTickerHolder_->SetStatusRepositorySpecId("test_id");
            StatusNTickerHolder_->SetStatusRepositoryTargetState(API::EPodAgentTargetState_ACTIVE);
            StatusNTickerHolder_->UpdateBoxIp6Subnet112Base("2a02:6b8:c08:68a3:0:696:6937:");
            UNIT_ASSERT(StatusNTickerHolder_->GetBoxStatusRepository() == BoxStatusRepository_);
            UNIT_ASSERT(StatusNTickerHolder_->GetLayerStatusRepository() == LayerStatusRepository_);
            UNIT_ASSERT(StatusNTickerHolder_->GetStaticResourceStatusRepository() == StaticResourceStatusRepository_);
            UNIT_ASSERT(StatusNTickerHolder_->GetVolumeStatusRepository() == VolumeStatusRepository_);
            UNIT_ASSERT(StatusNTickerHolder_->GetWorkloadStatusRepository() == WorkloadStatusRepository_);
            UNIT_ASSERT(StatusNTickerHolder_->GetBoxIp6Subnet112Base() == "2a02:6b8:c08:68a3:0:696:6937:");
            UNIT_ASSERT(
                google::protobuf::util::MessageDifferencer::Equals(
                    StatusNTickerHolder_->GetStatusRepositoryTotalStatus(false)
                    , StatusRepository_->GetTotalStatus(false)
                )
            );
            UNIT_ASSERT(
                google::protobuf::util::MessageDifferencer::Equals(
                    StatusNTickerHolder_->GetStatusRepositoryTotalStatus(true)
                    , StatusRepository_->GetTotalStatus(true)
                )
            );
            UNIT_ASSERT(StatusRepository_->GetSpecId() == "test_id");
            UNIT_ASSERT(BoxStatusRepository_->GetIp6Subnet112Base() == "2a02:6b8:c08:68a3:0:696:6937:");

            {
                UNIT_ASSERT(StatusNTickerHolder_->GetCacheLayerIdsAndRevisions().empty());
                StatusNTickerHolder_->AddCacheLayer(NObjectTargetTestLib::CreateCacheLayerTargetSimple("cache_layer1", "cache_layer1_download_hash", 1));

                UNIT_ASSERT(StatusNTickerHolder_->HasCacheLayer("cache_layer1", 1));
                UNIT_ASSERT(StatusNTickerHolder_->GetCacheLayerHash("cache_layer1", 1) == "cache_layer1_download_hash");
                UNIT_ASSERT(LayerStatusRepository_->HasCacheObject("cache_layer1", 1));
                UNIT_ASSERT(Ticker_->HasTree(TStatusNTickerHolder::GetLayerTreeId("cache_layer1_download_hash")));
                UNIT_ASSERT(LayerStatusRepository_->GetCacheObjectStatus("cache_layer1", 1).state() == API::ELayerState_UNKNOWN);
                UNIT_ASSERT(StatusNTickerHolder_->GetCacheLayerIdsAndRevisions() == TVector<TStatusRepositoryCommon::TCacheObject>({TStatusRepositoryCommon::TCacheObject("cache_layer1", 1)}));
            }

            {
                UNIT_ASSERT(StatusNTickerHolder_->GetLayerIds().empty());
                StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));

                UNIT_ASSERT(StatusNTickerHolder_->HasLayer("layer1"));
                UNIT_ASSERT(LayerStatusRepository_->HasObject("layer1"));
                UNIT_ASSERT(Ticker_->HasTree(TStatusNTickerHolder::GetLayerTreeId("layer1_download_hash")));
                UNIT_ASSERT(LayerStatusRepository_->GetObjectStatus("layer1").state() == API::ELayerState_UNKNOWN);
                UNIT_ASSERT(StatusNTickerHolder_->GetLayerIds() == TVector<TString>({"layer1"}));
            }

            {
                UNIT_ASSERT(StatusNTickerHolder_->GetCacheStaticResourceIdsAndRevisions().empty());
                StatusNTickerHolder_->AddCacheStaticResource(NObjectTargetTestLib::CreateCacheStaticResourceTargetSimple("cache_static_resource1", "cache_static_resource1_download_hash", 1));

                UNIT_ASSERT(StatusNTickerHolder_->HasCacheStaticResource("cache_static_resource1", 1));
                UNIT_ASSERT(StatusNTickerHolder_->GetCacheStaticResourceHash("cache_static_resource1", 1) == "cache_static_resource1_download_hash");
                UNIT_ASSERT(StaticResourceStatusRepository_->HasCacheObject("cache_static_resource1", 1));
                UNIT_ASSERT(Ticker_->HasTree(TStatusNTickerHolder::GetStaticResourceTreeId("cache_static_resource1_download_hash")));
                UNIT_ASSERT(StaticResourceStatusRepository_->GetCacheObjectStatus("cache_static_resource1", 1).state() == API::EStaticResourceState_UNKNOWN);
                UNIT_ASSERT(StatusNTickerHolder_->GetCacheStaticResourceIdsAndRevisions() == TVector<TStatusRepositoryCommon::TCacheObject>({TStatusRepositoryCommon::TCacheObject("cache_static_resource1", 1)}));
            }

            {
                UNIT_ASSERT(StatusNTickerHolder_->GetStaticResourceIds().empty());
                StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash"));

                UNIT_ASSERT(StatusNTickerHolder_->HasStaticResource("static_resource1"));
                UNIT_ASSERT(StaticResourceStatusRepository_->HasObject("static_resource1"));
                UNIT_ASSERT(Ticker_->HasTree(TStatusNTickerHolder::GetStaticResourceTreeId("static_resource1_download_hash")));
                UNIT_ASSERT(StaticResourceStatusRepository_->GetObjectStatus("static_resource1").state() == API::EStaticResourceState_UNKNOWN);
                UNIT_ASSERT(StatusNTickerHolder_->GetStaticResourceIds() == TVector<TString>({"static_resource1"}));
            }

            {
                UNIT_ASSERT(StatusNTickerHolder_->GetVolumeIds().empty());
                StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("volume1"));

                UNIT_ASSERT(StatusNTickerHolder_->HasVolume("volume1"));
                UNIT_ASSERT(VolumeStatusRepository_->HasObject("volume1"));
                UNIT_ASSERT(Ticker_->HasTree(TStatusNTickerHolder::GetVolumeTreeId("volume1")));
                UNIT_ASSERT(StatusNTickerHolder_->GetVolumeIds() == TVector<TString>({"volume1"}));
            }

            {
                UNIT_ASSERT(StatusNTickerHolder_->GetBoxIds().empty());
                UNIT_ASSERT(!StatusNTickerHolder_->HasBoxIp6Address("2a02:6b8:c08:68a3:0:696:6937:1"));
                StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("box1"));
                BoxStatusRepository_->UpdateObjectIpAddress("box1", "2a02:6b8:c08:68a3:0:696:6937:1");

                UNIT_ASSERT(StatusNTickerHolder_->HasBox("box1"));
                UNIT_ASSERT(BoxStatusRepository_->HasObject("box1"));
                UNIT_ASSERT(Ticker_->HasTree(TStatusNTickerHolder::GetBoxTreeId("box1")));
                UNIT_ASSERT(StatusNTickerHolder_->GetBoxIds() == TVector<TString>({"box1"}));
                UNIT_ASSERT(StatusNTickerHolder_->HasBoxIp6Address("2a02:6b8:c08:68a3:0:696:6937:1"));
            }

            {
                UNIT_ASSERT(StatusNTickerHolder_->GetWorkloadIds().empty());
                StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("boxRef0"));
                StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetSimple("workload1"));

                UNIT_ASSERT(StatusNTickerHolder_->HasWorkload("workload1"));
                UNIT_ASSERT(WorkloadStatusRepository_->HasObject("workload1"));
                UNIT_ASSERT(Ticker_->HasTree(TStatusNTickerHolder::GetWorkloadTreeId("workload1")));
                UNIT_ASSERT(StatusNTickerHolder_->GetWorkloadIds() == TVector<TString>({"workload1"}));
            }
       }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestGraphBuild) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            const TMap<TGraph::TNode, TSet<TGraph::TNode>> allOutEdgesCurrentBegin = {
                {TGraph::TNode("volume1", TGraph::ENodeType::VOLUME), {TGraph::TNode("layer1", TGraph::ENodeType::LAYER), TGraph::TNode("layer3", TGraph::ENodeType::LAYER)}}
                , {TGraph::TNode("volume2", TGraph::ENodeType::VOLUME), {TGraph::TNode("layer2", TGraph::ENodeType::LAYER)}}
                , {TGraph::TNode("volume3", TGraph::ENodeType::VOLUME), {TGraph::TNode("layer1", TGraph::ENodeType::LAYER)}}
                , {TGraph::TNode("box1", TGraph::ENodeType::BOX), {TGraph::TNode("layer1", TGraph::ENodeType::LAYER), TGraph::TNode("layer2", TGraph::ENodeType::LAYER), TGraph::TNode("static_resource1", TGraph::ENodeType::STATIC_RESOURCE), TGraph::TNode("volume2", TGraph::ENodeType::VOLUME)}}
                , {TGraph::TNode("box2", TGraph::ENodeType::BOX), {TGraph::TNode("layer3", TGraph::ENodeType::LAYER), TGraph::TNode("static_resource2", TGraph::ENodeType::STATIC_RESOURCE), TGraph::TNode("volume1", TGraph::ENodeType::VOLUME), TGraph::TNode("volume3", TGraph::ENodeType::VOLUME)}}
                , {TGraph::TNode("box3", TGraph::ENodeType::BOX), {TGraph::TNode("layer1", TGraph::ENodeType::LAYER), TGraph::TNode("static_resource3", TGraph::ENodeType::STATIC_RESOURCE), TGraph::TNode("volume1", TGraph::ENodeType::VOLUME)}}
                , {TGraph::TNode("workload1", TGraph::ENodeType::WORKLOAD), {TGraph::TNode("box1", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("workload2", TGraph::ENodeType::WORKLOAD), {TGraph::TNode("box1", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("workload3", TGraph::ENodeType::WORKLOAD), {TGraph::TNode("box2", TGraph::ENodeType::BOX)}}
            };

            const TMap<TGraph::TNode, TSet<TGraph::TNode>> allInEdgesCurrentBegin = {
                {TGraph::TNode("layer1", TGraph::ENodeType::LAYER), {TGraph::TNode("volume1", TGraph::ENodeType::VOLUME), TGraph::TNode("volume3", TGraph::ENodeType::VOLUME), TGraph::TNode("box1", TGraph::ENodeType::BOX), TGraph::TNode("box3", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("layer2", TGraph::ENodeType::LAYER), {TGraph::TNode("volume2", TGraph::ENodeType::VOLUME), TGraph::TNode("box1", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("layer3", TGraph::ENodeType::LAYER), {TGraph::TNode("volume1", TGraph::ENodeType::VOLUME), TGraph::TNode("box2", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("static_resource1", TGraph::ENodeType::STATIC_RESOURCE), {TGraph::TNode("box1", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("static_resource2", TGraph::ENodeType::STATIC_RESOURCE), {TGraph::TNode("box2", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("static_resource3", TGraph::ENodeType::STATIC_RESOURCE), {TGraph::TNode("box3", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("volume1", TGraph::ENodeType::VOLUME), {TGraph::TNode("box2", TGraph::ENodeType::BOX), TGraph::TNode("box3", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("volume2", TGraph::ENodeType::VOLUME), {TGraph::TNode("box1", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("volume3", TGraph::ENodeType::VOLUME), {TGraph::TNode("box2", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("box1", TGraph::ENodeType::BOX), {TGraph::TNode("workload1", TGraph::ENodeType::WORKLOAD), TGraph::TNode("workload2", TGraph::ENodeType::WORKLOAD)}}
                , {TGraph::TNode("box2", TGraph::ENodeType::BOX), {TGraph::TNode("workload3", TGraph::ENodeType::WORKLOAD)}}
            };

            const TMap<TGraph::TNode, TSet<TGraph::TNode>> allOutEdgesTargetBegin = allOutEdgesCurrentBegin;
            const TMap<TGraph::TNode, TSet<TGraph::TNode>> allInEdgesTargetBegin = allInEdgesCurrentBegin;

            const TMap<TGraph::TNode, TSet<TGraph::TNode>> allOutEdgesCurrentEnd = {
                {TGraph::TNode("volume1", TGraph::ENodeType::VOLUME), {TGraph::TNode("layer1", TGraph::ENodeType::LAYER), TGraph::TNode("layer3", TGraph::ENodeType::LAYER)}}
                , {TGraph::TNode("volume2", TGraph::ENodeType::VOLUME), {TGraph::TNode("layer2", TGraph::ENodeType::LAYER)}}
                , {TGraph::TNode("box1", TGraph::ENodeType::BOX), {TGraph::TNode("layer1", TGraph::ENodeType::LAYER), TGraph::TNode("layer2", TGraph::ENodeType::LAYER), TGraph::TNode("static_resource1", TGraph::ENodeType::STATIC_RESOURCE), TGraph::TNode("volume2", TGraph::ENodeType::VOLUME)}}
                , {TGraph::TNode("box2", TGraph::ENodeType::BOX), {TGraph::TNode("layer3", TGraph::ENodeType::LAYER), TGraph::TNode("static_resource2", TGraph::ENodeType::STATIC_RESOURCE), TGraph::TNode("volume1", TGraph::ENodeType::VOLUME), TGraph::TNode("volume3", TGraph::ENodeType::VOLUME)}}
                , {TGraph::TNode("workload1", TGraph::ENodeType::WORKLOAD), {TGraph::TNode("box1", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("workload3", TGraph::ENodeType::WORKLOAD), {TGraph::TNode("box2", TGraph::ENodeType::BOX)}}
            };

            const TMap<TGraph::TNode, TSet<TGraph::TNode>> allInEdgesCurrentEnd = {
                {TGraph::TNode("layer1", TGraph::ENodeType::LAYER), {TGraph::TNode("volume1", TGraph::ENodeType::VOLUME), TGraph::TNode("box1", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("layer2", TGraph::ENodeType::LAYER), {TGraph::TNode("volume2", TGraph::ENodeType::VOLUME), TGraph::TNode("box1", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("layer3", TGraph::ENodeType::LAYER), {TGraph::TNode("volume1", TGraph::ENodeType::VOLUME), TGraph::TNode("box2", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("static_resource1", TGraph::ENodeType::STATIC_RESOURCE), {TGraph::TNode("box1", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("static_resource2", TGraph::ENodeType::STATIC_RESOURCE), {TGraph::TNode("box2", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("volume1", TGraph::ENodeType::VOLUME), {TGraph::TNode("box2", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("volume2", TGraph::ENodeType::VOLUME), {TGraph::TNode("box1", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("volume3", TGraph::ENodeType::VOLUME), {TGraph::TNode("box2", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("box1", TGraph::ENodeType::BOX), {TGraph::TNode("workload1", TGraph::ENodeType::WORKLOAD)}}
                , {TGraph::TNode("box2", TGraph::ENodeType::BOX), {TGraph::TNode("workload3", TGraph::ENodeType::WORKLOAD)}}
            };

            const TMap<TGraph::TNode, TSet<TGraph::TNode>> allOutEdgesTargetEnd = {
                {TGraph::TNode("volume1", TGraph::ENodeType::VOLUME), {TGraph::TNode("layer2", TGraph::ENodeType::LAYER)}}
                , {TGraph::TNode("volume2", TGraph::ENodeType::VOLUME), {TGraph::TNode("layer2", TGraph::ENodeType::LAYER)}}
                , {TGraph::TNode("box1", TGraph::ENodeType::BOX), {TGraph::TNode("layer1", TGraph::ENodeType::LAYER), TGraph::TNode("layer2", TGraph::ENodeType::LAYER), TGraph::TNode("static_resource1", TGraph::ENodeType::STATIC_RESOURCE), TGraph::TNode("volume2", TGraph::ENodeType::VOLUME)}}
                , {TGraph::TNode("box2", TGraph::ENodeType::BOX), {TGraph::TNode("layer1", TGraph::ENodeType::LAYER), TGraph::TNode("static_resource2", TGraph::ENodeType::STATIC_RESOURCE), TGraph::TNode("volume2", TGraph::ENodeType::VOLUME)}}
                , {TGraph::TNode("workload1", TGraph::ENodeType::WORKLOAD), {TGraph::TNode("box2", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("workload3", TGraph::ENodeType::WORKLOAD), {TGraph::TNode("box2", TGraph::ENodeType::BOX)}}
            };

            const TMap<TGraph::TNode, TSet<TGraph::TNode>> allInEdgesTargetEnd = {
                {TGraph::TNode("layer1", TGraph::ENodeType::LAYER), {TGraph::TNode("box1", TGraph::ENodeType::BOX), TGraph::TNode("box2", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("layer2", TGraph::ENodeType::LAYER), {TGraph::TNode("volume1", TGraph::ENodeType::VOLUME), TGraph::TNode("volume2", TGraph::ENodeType::VOLUME), TGraph::TNode("box1", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("static_resource1", TGraph::ENodeType::STATIC_RESOURCE), {TGraph::TNode("box1", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("static_resource2", TGraph::ENodeType::STATIC_RESOURCE), {TGraph::TNode("box2", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("volume2", TGraph::ENodeType::VOLUME), {TGraph::TNode("box1", TGraph::ENodeType::BOX), TGraph::TNode("box2", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("box2", TGraph::ENodeType::BOX), {TGraph::TNode("workload1", TGraph::ENodeType::WORKLOAD), TGraph::TNode("workload3", TGraph::ENodeType::WORKLOAD)}}
            };

            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash"));
            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource2", "static_resource2_download_hash"));
            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource3", "static_resource3_download_hash"));
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer2", "layer2_download_hash"));
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer3", "layer3_download_hash"));
            StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetWithDependence("volume1", "hash", {"layer1", "layer3"}));
            StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetWithDependence("volume2", "hash", {"layer2"}));
            StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetWithDependence("volume3", "hash", {"layer1"}));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetWithDependence("box1", "hash", {"layer1", "layer2"}, {"static_resource1"}, {"volume2"}));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetWithDependence("box2", "hash", {"layer3"}, {"static_resource2"}, {"volume1", "volume3"}));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetWithDependence("box3", "hash", {"layer1"}, {"static_resource3"}, {"volume1"}));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload1", "hash", "box1"));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload2", "hash", "box1"));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload3", "hash", "box2"));

            UNIT_ASSERT_EQUAL(StatusNTickerHolder_->GetCurrentGraph().GetAllOutEdges(), allOutEdgesCurrentBegin);
            UNIT_ASSERT_EQUAL(StatusNTickerHolder_->GetCurrentGraph().GetAllInEdges(), allInEdgesCurrentBegin);
            UNIT_ASSERT_EQUAL(StatusNTickerHolder_->GetTargetGraph().GetAllOutEdges(), allOutEdgesTargetBegin);
            UNIT_ASSERT_EQUAL(StatusNTickerHolder_->GetTargetGraph().GetAllInEdges(), allInEdgesTargetBegin);

            UNIT_ASSERT_EQUAL(StatusNTickerHolder_->TryToUpdateWorkloadFromTarget("workload1").Result_, TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::NO_TARGET);
            UNIT_ASSERT_EQUAL(StatusNTickerHolder_->TryToUpdateBoxFromTarget("box2").Result_, TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::NO_TARGET);
            UNIT_ASSERT_EQUAL(StatusNTickerHolder_->TryToUpdateVolumeFromTarget("volume3").Result_, TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::NO_TARGET);

            StatusNTickerHolder_->RemoveWorkload("workload2");
            StatusNTickerHolder_->UpdateWorkloadTarget(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload3", "other_hash", "box1"));
            StatusNTickerHolder_->UpdateWorkloadTarget(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload1", "other_hash", "box2"));
            StatusNTickerHolder_->UpdateWorkloadTarget(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload3", "other_hash", "box2"));
            StatusNTickerHolder_->RemoveBox("box3");
            StatusNTickerHolder_->UpdateBoxTarget(NObjectTargetTestLib::CreateBoxTargetWithDependence("box2", "other_hash", {"layer1"}, {"static_resource2"}, {"volume1"}));
            StatusNTickerHolder_->RemoveVolume("volume3");
            StatusNTickerHolder_->UpdateBoxTarget(NObjectTargetTestLib::CreateBoxTargetWithDependence("box2", "other_hash", {"layer1"}, {"static_resource2"}, {"volume2"}));
            StatusNTickerHolder_->UpdateVolumeTarget(NObjectTargetTestLib::CreatePersistentVolumeTargetWithDependence("volume1", "other_hash", {"layer1"}));
            StatusNTickerHolder_->RemoveLayer("layer3");
            StatusNTickerHolder_->UpdateVolumeTarget(NObjectTargetTestLib::CreatePersistentVolumeTargetWithDependence("volume1", "other_hash", {"layer2"}));

            UNIT_ASSERT_EQUAL(StatusNTickerHolder_->TryToUpdateWorkloadFromTarget("workload1").Result_, TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE);
            UNIT_ASSERT_EQUAL(StatusNTickerHolder_->TryToUpdateWorkloadFromTarget("workload3").Result_, TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE);
            UNIT_ASSERT_EQUAL(StatusNTickerHolder_->TryToUpdateBoxFromTarget("box1").Result_, TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::NO_TARGET);
            UNIT_ASSERT_EQUAL(StatusNTickerHolder_->TryToUpdateBoxFromTarget("box2").Result_, TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE);
            UNIT_ASSERT_EQUAL(StatusNTickerHolder_->TryToUpdateVolumeFromTarget("volume1").Result_, TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::WAITING_UPDATE);
            UNIT_ASSERT_EQUAL(StatusNTickerHolder_->TryToUpdateVolumeFromTarget("volume2").Result_, TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::NO_TARGET);
            UNIT_ASSERT_EQUAL(StatusNTickerHolder_->TryToUpdateVolumeFromTarget("volume3").Result_, TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::NO_TARGET);

            UNIT_ASSERT_EQUAL(StatusNTickerHolder_->GetCurrentGraph().GetAllOutEdges(), allOutEdgesCurrentEnd);
            UNIT_ASSERT_EQUAL(StatusNTickerHolder_->GetCurrentGraph().GetAllInEdges(), allInEdgesCurrentEnd);
            UNIT_ASSERT_EQUAL(StatusNTickerHolder_->GetTargetGraph().GetAllOutEdges(), allOutEdgesTargetEnd);
            UNIT_ASSERT_EQUAL(StatusNTickerHolder_->GetTargetGraph().GetAllInEdges(), allInEdgesTargetEnd);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestGraphBuildWithTargetRemove) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            const TMap<TGraph::TNode, TSet<TGraph::TNode>> allOutEdgesCurrentBegin = {
                {TGraph::TNode("volume1", TGraph::ENodeType::VOLUME), {TGraph::TNode("layer1", TGraph::ENodeType::LAYER)}}
                , {TGraph::TNode("volume2", TGraph::ENodeType::VOLUME), {TGraph::TNode("layer1", TGraph::ENodeType::LAYER)}}
                , {TGraph::TNode("box1", TGraph::ENodeType::BOX), {TGraph::TNode("layer1", TGraph::ENodeType::LAYER), TGraph::TNode("static_resource1", TGraph::ENodeType::STATIC_RESOURCE), TGraph::TNode("volume1", TGraph::ENodeType::VOLUME)}}
                , {TGraph::TNode("box2", TGraph::ENodeType::BOX), {TGraph::TNode("layer1", TGraph::ENodeType::LAYER), TGraph::TNode("static_resource1", TGraph::ENodeType::STATIC_RESOURCE), TGraph::TNode("volume1", TGraph::ENodeType::VOLUME), TGraph::TNode("volume2", TGraph::ENodeType::VOLUME)}}
                , {TGraph::TNode("workload1", TGraph::ENodeType::WORKLOAD), {TGraph::TNode("box1", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("workload2", TGraph::ENodeType::WORKLOAD), {TGraph::TNode("box2", TGraph::ENodeType::BOX)}}
            };

            const TMap<TGraph::TNode, TSet<TGraph::TNode>> allInEdgesCurrentBegin = {
                {TGraph::TNode("layer1", TGraph::ENodeType::LAYER), {TGraph::TNode("volume1", TGraph::ENodeType::VOLUME), TGraph::TNode("volume2", TGraph::ENodeType::VOLUME), TGraph::TNode("box1", TGraph::ENodeType::BOX), TGraph::TNode("box2", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("static_resource1", TGraph::ENodeType::STATIC_RESOURCE), {TGraph::TNode("box1", TGraph::ENodeType::BOX), TGraph::TNode("box2", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("volume1", TGraph::ENodeType::VOLUME), {TGraph::TNode("box1", TGraph::ENodeType::BOX), TGraph::TNode("box2", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("volume2", TGraph::ENodeType::VOLUME), {TGraph::TNode("box2", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("box1", TGraph::ENodeType::BOX), {TGraph::TNode("workload1", TGraph::ENodeType::WORKLOAD)}}
                , {TGraph::TNode("box2", TGraph::ENodeType::BOX), {TGraph::TNode("workload2", TGraph::ENodeType::WORKLOAD)}}
            };

            const TMap<TGraph::TNode, TSet<TGraph::TNode>> allOutEdgesTargetBegin = allOutEdgesCurrentBegin;
            const TMap<TGraph::TNode, TSet<TGraph::TNode>> allInEdgesTargetBegin = allInEdgesCurrentBegin;

            const TMap<TGraph::TNode, TSet<TGraph::TNode>> allOutEdgesCurrentEnd = allOutEdgesCurrentBegin;
            const TMap<TGraph::TNode, TSet<TGraph::TNode>> allInEdgesCurrentEnd = allInEdgesCurrentBegin;

            const TMap<TGraph::TNode, TSet<TGraph::TNode>> allOutEdgesTargetEnd = {
                {TGraph::TNode("volume1", TGraph::ENodeType::VOLUME), {TGraph::TNode("layer1", TGraph::ENodeType::LAYER)}}
                , {TGraph::TNode("box1", TGraph::ENodeType::BOX), {TGraph::TNode("layer1", TGraph::ENodeType::LAYER), TGraph::TNode("static_resource1", TGraph::ENodeType::STATIC_RESOURCE),  TGraph::TNode("volume1", TGraph::ENodeType::VOLUME)}}
                , {TGraph::TNode("workload1", TGraph::ENodeType::WORKLOAD), {TGraph::TNode("box1", TGraph::ENodeType::BOX)}}
            };

            const TMap<TGraph::TNode, TSet<TGraph::TNode>> allInEdgesTargetEnd = {
                {TGraph::TNode("layer1", TGraph::ENodeType::LAYER), {TGraph::TNode("volume1", TGraph::ENodeType::VOLUME), TGraph::TNode("box1", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("static_resource1", TGraph::ENodeType::STATIC_RESOURCE), {TGraph::TNode("box1", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("volume1", TGraph::ENodeType::VOLUME), {TGraph::TNode("box1", TGraph::ENodeType::BOX)}}
                , {TGraph::TNode("box1", TGraph::ENodeType::BOX), {TGraph::TNode("workload1", TGraph::ENodeType::WORKLOAD)}}
            };

            StatusNTickerHolder_->AddStaticResource(NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource1", "static_resource1_download_hash"));
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("layer1", "layer1_download_hash"));
            StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetWithDependence("volume1", "hash", {"layer1"}));
            StatusNTickerHolder_->AddVolume(NObjectTargetTestLib::CreatePersistentVolumeTargetWithDependence("volume2", "hash", {"layer1"}));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetWithDependence("box1", "hash", {"layer1"}, {"static_resource1"}, {"volume1"}));
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetWithDependence("box2", "hash", {"layer1"}, {"static_resource1"}, {"volume1", "volume2"}));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload1", "hash", "box1"));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload2", "hash", "box2"));

            UNIT_ASSERT_EQUAL(StatusNTickerHolder_->GetCurrentGraph().GetAllOutEdges(), allOutEdgesCurrentBegin);
            UNIT_ASSERT_EQUAL(StatusNTickerHolder_->GetCurrentGraph().GetAllInEdges(), allInEdgesCurrentBegin);
            UNIT_ASSERT_EQUAL(StatusNTickerHolder_->GetTargetGraph().GetAllOutEdges(), allOutEdgesTargetBegin);
            UNIT_ASSERT_EQUAL(StatusNTickerHolder_->GetTargetGraph().GetAllInEdges(), allInEdgesTargetBegin);

            StatusNTickerHolder_->SetVolumeTargetRemove("volume2");
            StatusNTickerHolder_->SetBoxTargetRemove("box2");
            StatusNTickerHolder_->SetWorkloadTargetRemove("workload2");

            UNIT_ASSERT_EQUAL(StatusNTickerHolder_->GetCurrentGraph().GetAllOutEdges(), allOutEdgesCurrentEnd);
            UNIT_ASSERT_EQUAL(StatusNTickerHolder_->GetCurrentGraph().GetAllInEdges(), allInEdgesCurrentEnd);
            UNIT_ASSERT_EQUAL(StatusNTickerHolder_->GetTargetGraph().GetAllOutEdges(), allOutEdgesTargetEnd);
            UNIT_ASSERT_EQUAL(StatusNTickerHolder_->GetTargetGraph().GetAllInEdges(), allInEdgesTargetEnd);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestHotfixForMutableWorkloadFields) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            StatusNTickerHolder_->AddBox(NObjectTargetTestLib::CreateBoxTargetSimple("box1"));
            StatusNTickerHolder_->AddWorkload(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload1", "hash1", "box1"));

            UNIT_ASSERT_EQUAL(WorkloadStatusRepository_->GetObjectTargetState("workload1"), API::EWorkloadTarget_ACTIVE);

            StatusNTickerHolder_->UpdateWorkloadTargetState("workload1", API::EWorkloadTarget_REMOVED);
            UNIT_ASSERT_EQUAL(WorkloadStatusRepository_->GetObjectTargetState("workload1"), API::EWorkloadTarget_REMOVED);

            StatusNTickerHolder_->UpdateWorkloadTarget(NObjectTargetTestLib::CreateWorkloadTargetWithDependence("workload1", "other_hash", "box1"));
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->WorkloadHasTarget("workload1"));

            WorkloadStatusRepository_->UpdateObjectState("workload1", API::EWorkloadState_REMOVED);
            UNIT_ASSERT(StatusNTickerHolder_->TryToUpdateWorkloadFromTarget("workload1").Result_ == TStatusNTickerHolder::TUpdateObjectResult::EUpdateResult::UPDATED);
            UNIT_ASSERT_EQUAL(WorkloadStatusRepository_->GetObjectTargetState("workload1"), API::EWorkloadTarget_REMOVED);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestUpdateActiveDownloadContainersLimit) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            UNIT_ASSERT_EQUAL_C(
                StaticResourceStatusRepository_->GetActiveDownloadContainersLimit()
                , 0
                , StaticResourceStatusRepository_->GetActiveDownloadContainersLimit()
            );
            UNIT_ASSERT_EQUAL_C(
                LayerStatusRepository_->GetActiveDownloadContainersLimit()
                , 0
                , LayerStatusRepository_->GetActiveDownloadContainersLimit()
            );

            StatusNTickerHolder_->UpdateActiveDownloadContainersLimit(3);

            UNIT_ASSERT_EQUAL_C(
                StaticResourceStatusRepository_->GetActiveDownloadContainersLimit()
                , 3
                , StaticResourceStatusRepository_->GetActiveDownloadContainersLimit()
            );
            UNIT_ASSERT_EQUAL_C(
                LayerStatusRepository_->GetActiveDownloadContainersLimit()
                , 3
                , LayerStatusRepository_->GetActiveDownloadContainersLimit()
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestUpdateActiveVerifyContainersLimit) {
    class TTest : public ITestStatusNTickerHolderCanon {
    protected:
        void Test() override {
            UNIT_ASSERT_EQUAL_C(
                StaticResourceStatusRepository_->GetActiveVerifyContainersLimit()
                , 0
                , StaticResourceStatusRepository_->GetActiveVerifyContainersLimit()
            );
            UNIT_ASSERT_EQUAL_C(
                LayerStatusRepository_->GetActiveVerifyContainersLimit()
                , 0
                , LayerStatusRepository_->GetActiveVerifyContainersLimit()
            );

            StatusNTickerHolder_->UpdateActiveVerifyContainersLimit(3);

            UNIT_ASSERT_EQUAL_C(
                StaticResourceStatusRepository_->GetActiveVerifyContainersLimit()
                , 3
                , StaticResourceStatusRepository_->GetActiveVerifyContainersLimit()
            );
            UNIT_ASSERT_EQUAL_C(
                LayerStatusRepository_->GetActiveVerifyContainersLimit()
                , 3
                , LayerStatusRepository_->GetActiveVerifyContainersLimit()
            );
        }
    };

    TTest test;
    test.DoTest();
}

}

} // namespace NInfra::NPodAgent::NStatusNTickerHolderTest
