#include "box_status_repository.h"

#include "support_functions.h"

#include <util/string/builder.h>

namespace NInfra::NPodAgent {

void TBoxStatusRepository::UpdateSpecTimestamp(TInstant currentTime) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    for (auto& [objectId, info] : Objects_) {
        TGuard<TMutex> g(GetObjectMutex(objectId));
        RefreshObjectStatusConditions(info.Status_, info.Status_, currentTime);
    }
}

void TBoxStatusRepository::IncrementContainerSystemFailureCounter(const NStatusRepositoryTypes::TContainerDescription& container) {
    if (container.ContainerType_ == NStatusRepositoryTypes::TContainerDescription::EContainerType::META) {
        CheckContainerObjectType(container);
        IncrementObjectFailCounter(container.ObjectIdOrHash_);
    } else {
        TStatusRepositoryCommon::IncrementContainerSystemFailureCounter(container);
    }
}

void TBoxStatusRepository::UpdateContainerState(const NStatusRepositoryTypes::TContainerDescription& container, API::EContainerState state) {
    if (container.ContainerType_ == NStatusRepositoryTypes::TContainerDescription::EContainerType::META) {
        Y_ENSURE(state == API::EContainerState_SYSTEM_FAILURE, "you can only call UpdateContainerState with EContainerState_SYSTEM_FAILURE for box meta container");
    } else {
        TStatusRepositoryCommon::UpdateContainerState(container, state);
    }
}

void TBoxStatusRepository::PatchTotalStatus(API::TPodAgentStatus& status, TUpdateHolderTargetPtr updateHolderTarget, bool conditionsOnly) const {
    Y_ENSURE(updateHolderTarget, "UpdateHolderTarget not provided for TBoxStatusRepository::PatchTotalStatus");

    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);

    for (const auto& [objectId, info] : Objects_) {
        API::TBoxStatus* st = status.add_boxes();
        *st = GetObjectStatusNoGlobalLock(objectId, updateHolderTarget, conditionsOnly);
        NSupport::PatchAllNonReadyConditions(status, *st);
    }
}

bool TBoxStatusRepository::NeedLongTickPeriod(const TString& objectIdOrHash, TUpdateHolderTargetPtr updateHolderTarget) const {
    Y_ENSURE(updateHolderTarget, "UpdateHolderTarget not provided for TBoxStatusRepository::NeedLongTickPeriod");

    auto status = GetObjectStatus(objectIdOrHash, updateHolderTarget);

    if (status.ready().status() == API::EConditionStatus_TRUE) {
        return true;
    }

    switch (status.state()) {
        case API::EBoxState_WAITING_FOR_VOLUMES:
        case API::EBoxState_WAITING_FOR_ROOTFS_LAYERS:
        case API::EBoxState_WAITING_FOR_STATIC_RESOURCES:
            return TInstant::Now() - NSupport::ToInstant(status.in_progress().last_transition_time()) > ACTIVE_WAIT_TIME;
        case API::EBoxState_INVALID:
            return true;
        default:
            return false;
    }
}

void TBoxStatusRepository::UpdateContainerFailReason(const NStatusRepositoryTypes::TContainerDescription& container, const TString& failReason) {
    CheckContainerObjectType(container);
    if (container.ContainerType_ == NStatusRepositoryTypes::TContainerDescription::EContainerType::META) {
        UpdateObjectFailedMessage(
            container.ObjectIdOrHash_
            , TStringBuilder() << "Object " << ToString(container.ContainerType_) << " container FailReason: " << failReason
        );
    } else {
        TStatusRepositoryCommon::UpdateContainerFailReason(container, failReason);
    }
}

const TLightRWLock& TBoxStatusRepository::GetGlobalContainerLock() const {
    return GlobalObjectLock_;
}

const TMutex& TBoxStatusRepository::GetLocalContainerLock(const NStatusRepositoryTypes::TContainerDescription& container) const {
    if (container.ContainerType_ == NStatusRepositoryTypes::TContainerDescription::EContainerType::META) {
        ythrow yexception() << "Object meta container does not have its own lock";
    }

    if (container.ContainerType_ == NStatusRepositoryTypes::TContainerDescription::EContainerType::INIT) {
        return GetObjectMutex(container.ObjectIdOrHash_);
    }

    ythrow yexception() << "Box does not have '" << ToString(container.ContainerType_) << "' container";
}

API::TContainerStatus* TBoxStatusRepository::GetMutableContainerStatus(const NStatusRepositoryTypes::TContainerDescription& container) {
     // Must lock before call
    if (container.ContainerType_ == NStatusRepositoryTypes::TContainerDescription::EContainerType::META) {
        ythrow yexception() << "Object meta container does not have its own status";
    }

    if (container.ContainerType_ == NStatusRepositoryTypes::TContainerDescription::EContainerType::INIT) {
        API::TBoxStatus& objectStatus = Objects_.at(container.ObjectIdOrHash_).Status_;
        Y_ENSURE(
            container.InitNum_ < (ui32)objectStatus.inits().size()
            , "Too big init num for box '" << container.ObjectIdOrHash_
            << "' InitSize = " << objectStatus.inits().size()
            << ", InitNum = " << container.InitNum_
         );

        return objectStatus.mutable_inits(container.InitNum_);
    }

    ythrow yexception() << "Box does not have '" << ToString(container.ContainerType_) << "' container";
}

const API::TContainerStatus* TBoxStatusRepository::GetContainerStatus(const NStatusRepositoryTypes::TContainerDescription& container) {
     // Must lock before call
     return GetMutableContainerStatus(container);
}

void TBoxStatusRepository::AddObject(const TBoxMeta& objectMeta) {
    TWriteGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    auto [ptr, result] = Objects_.insert(
        {
            objectMeta.Id_
            , TObjectInfo()
        }
    );
    API::TBoxStatus& objectStatus = ptr->second.Status_;

    objectStatus.set_id(objectMeta.Id_);
    objectStatus.set_spec_timestamp(objectMeta.SpecTimestamp_);
    objectStatus.set_revision(objectMeta.Revision_);

    for (const auto& rootfsLayerRef : objectMeta.RootfsLayerRefs_) {
        *(objectStatus.add_rootfs_layer_refs()) = rootfsLayerRef;
    }
    for (const auto& staticResourceRef : objectMeta.StaticResourceRefs_) {
        *(objectStatus.add_static_resource_refs()) = staticResourceRef;
    }
    for (const auto& volumeRef : objectMeta.VolumeRefs_) {
        *(objectStatus.add_volume_refs()) = volumeRef;
    }
    for (const TString& initContainer : objectMeta.InitContainers_) {
        auto* initContainerStatus = objectStatus.add_inits();
        initContainerStatus->set_container_name(initContainer);
    }
    objectStatus.set_container_name(objectMeta.MetaContainer_);
    objectStatus.set_specific_type(objectMeta.BoxSpecificType_);
}

bool TBoxStatusRepository::HasObject(const TString& objectId) const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    return Objects_.FindPtr(objectId);
}

TVector<TString> TBoxStatusRepository::GetObjectIds() const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);

    TVector<TString> result;
    for (const auto& [objectId, info] : Objects_) {
        result.push_back(objectId);
    }
    return result;
}

void TBoxStatusRepository::RemoveObject(const TString& objectId) {
    TWriteGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TWriteGuardBase<TLightRWLock> ipGuard(Ip6AddressesLock_);

    if (const TString& ip6Address = Objects_.at(objectId).Status_.ip6_address(); ip6Address != "") {
        ActiveIp6Addresses_.erase(ip6Address);
    }

    Objects_.erase(objectId);
}

void TBoxStatusRepository::UpdateObjectRevision(const TString& objectId, ui32 revision) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    API::TBoxStatus& box = Objects_.at(objectId).Status_;
    if (box.revision() != revision) {
        box.set_revision(revision);
        TInstant now = TInstant::Now();
        RefreshObjectStatusConditions(box, box, now);
    }
}

void TBoxStatusRepository::UpdateObjectSpecTimestamp(const TString& objectId, ui64 specTimestamp) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    API::TBoxStatus& box = Objects_.at(objectId).Status_;
    if (box.spec_timestamp() != specTimestamp) {
        box.set_spec_timestamp(specTimestamp);
        TInstant now = TInstant::Now();
        RefreshObjectStatusConditions(box, box, now);
    }
}

TStatusRepositoryCommon::TObjectState TBoxStatusRepository::UpdateObjectState(const TString& objectIdOrHash, TObjectState state) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectIdOrHash));
    API::TBoxStatus& box = Objects_.at(objectIdOrHash).Status_;
    TStatusRepositoryCommon::TObjectState oldState = box.state();
    if (oldState != state) {
        box.set_state(std::get<API::EBoxState>(state));
        TInstant now = TInstant::Now();
        RefreshObjectStatusConditions(box, box, now);
    }
    return oldState;
}

void TBoxStatusRepository::IncrementObjectFailCounter(const TString& objectIdOrHash) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectIdOrHash));
    API::TBoxStatus& status = Objects_.at(objectIdOrHash).Status_;
    status.set_fail_counter(status.fail_counter() + 1);
}

void TBoxStatusRepository::UpdateObjectIpAddress(const TString& objectIdOrHash, const TString& address) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TWriteGuardBase<TLightRWLock> ipGuard(Ip6AddressesLock_);
    TGuard<TMutex> g(GetObjectMutex(objectIdOrHash));

    API::TBoxStatus& status = Objects_.at(objectIdOrHash).Status_;

    if (status.ip6_address() != address) {
        Y_ENSURE(!ActiveIp6Addresses_.contains(address), "Two different objects use same ip '" << address << "'");

        if (status.ip6_address() != "") {
            ActiveIp6Addresses_.erase(status.ip6_address());
        }

        status.set_ip6_address(address);

        if (address != "") {
            ActiveIp6Addresses_.insert(address);
        }
    }
}

void TBoxStatusRepository::UpdateObjectFailedMessage(const TString& objectIdOrHash, const TString& failedMessage) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectIdOrHash));
    Objects_.at(objectIdOrHash).Status_.mutable_failed()->set_message(NSupport::Truncate(failedMessage));
}

TVector<TString> TBoxStatusRepository::GetObjectIdsByHash(const TString& /* objectIdOrHash */) {
    // An empty vector is returned in order to have no duplicates of id in logs
    return {};
}

TVector<TStatusRepositoryCommon::TCacheObject> TBoxStatusRepository::GetCacheObjectIdsAndRevisionsByHash(const TString& /* objectIdOrHash */) {
    // No CacheObjects in this status repository
    return {};
}

void TBoxStatusRepository::UpdateIp6Subnet112Base(const TString& ip6Subnet112Base) {
    TWriteGuardBase<TLightRWLock> ipGuard(Ip6AddressesLock_);

    if (ip6Subnet112Base != "") {
        Y_ENSURE(ip6Subnet112Base.EndsWith(":"), "Box IpSubnet112Base must ends with ':', but actual value is '" << ip6Subnet112Base << "'");
    }

    Ip6Subnet112Base_ = ip6Subnet112Base;
}

TString TBoxStatusRepository::GetIp6Subnet112Base() const {
    TReadGuardBase<TLightRWLock> ipGuard(Ip6AddressesLock_);

    return Ip6Subnet112Base_;
}

bool TBoxStatusRepository::HasIp6Address(const TString& address) const {
    TReadGuardBase<TLightRWLock> ipGuard(Ip6AddressesLock_);

    return ActiveIp6Addresses_.contains(address);
}

API::TBoxStatus TBoxStatusRepository::GetObjectStatus(const TString& objectId, TUpdateHolderTargetPtr updateHolderTarget) const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    return GetObjectStatusNoGlobalLock(objectId, updateHolderTarget, false);
}

API::TBoxStatus TBoxStatusRepository::GetObjectStatusNoGlobalLock(
    const TString& objectId
    , TUpdateHolderTargetPtr updateHolderTarget
    , bool conditionsOnly
) const {
    TGuard<TMutex> g(GetObjectMutex(objectId));

    // WARNING: Do not copy full status here
    const API::TBoxStatus& fullStatus = Objects_.at(objectId).Status_;

    API::TBoxStatus status;
    if (conditionsOnly) {
        // Set object id only
        status.set_id(objectId);
    } else {
        status.CopyFrom(fullStatus);
    }

    bool changingSpecTimestamp = updateHolderTarget ? updateHolderTarget->BoxHasTarget(objectId) : false;
    RefreshObjectStatusConditions(status, fullStatus, TInstant::Zero(), false, changingSpecTimestamp);

    return status;
}

void TBoxStatusRepository::RefreshObjectStatusConditions(
    API::TBoxStatus& targetStatus
    , const API::TBoxStatus& box
    , const TInstant& now
    , bool refreshTime
    , bool changingSpecTimestamp
) {
    RefreshObjectStatusReady(
        *targetStatus.mutable_ready()
        , box
        , now
        , refreshTime
        , changingSpecTimestamp
    );
    RefreshObjectStatusInProgress(
        *targetStatus.mutable_in_progress()
        , box
        , now
        , refreshTime
        , changingSpecTimestamp
    );
    RefreshObjectStatusFailed(
        *targetStatus.mutable_failed()
        , box
        , now
        , refreshTime
    );
}

void TBoxStatusRepository::RefreshObjectStatusReady(
    API::TCondition& ready
    , const API::TBoxStatus& box
    , const TInstant& now
    , bool refreshTime
    , bool changingSpecTimestamp
) {
    NSupport::CopyConditonStatusAndTimestamp(ready, box.ready());
    API::EConditionStatus prevStatus = ready.status();

    ready.set_reason(NSupport::ExtractState(API::EBoxState_Name(box.state())));
    ready.set_message("");
    if (changingSpecTimestamp) {
        ready.set_message(NSupport::APPLYING_OBJECT_SPEC_MESSAGE);
    }

    switch (box.state()) {
        case API::EBoxState_READY:
            if (!changingSpecTimestamp) {
                ready.set_status(API::EConditionStatus_TRUE);
            } else {
                ready.set_status(API::EConditionStatus_FALSE);
                NSupport::SetApplyingObjectSpecState(ready);
            }
            break;
        case API::EBoxState_WAITING_FOR_ROOTFS_LAYERS:
        case API::EBoxState_WAITING_FOR_STATIC_RESOURCES:
        case API::EBoxState_CREATING_ROOTFS_VOLUME:
        case API::EBoxState_CREATING_ROOT_CONTAINER:
        case API::EBoxState_WAITING_FOR_VOLUMES:
        case API::EBoxState_LINKING_VOLUMES:
        case API::EBoxState_INIT_PROCESSES:
        case API::EBoxState_REMOVING:
        case API::EBoxState_REMOVED:
        case API::EBoxState_INVALID:
            ready.set_status(API::EConditionStatus_FALSE);
            break;
        case API::EBoxState_UNKNOWN:
            ready.set_status(API::EConditionStatus_UNKNOWN);
            break;
        // We don't want to use default
        // So we need this to handle all cases
        case API::EBoxState_INT_MIN_SENTINEL_DO_NOT_USE_:
        case API::EBoxState_INT_MAX_SENTINEL_DO_NOT_USE_:
            ready.set_status(API::EConditionStatus_UNKNOWN);
            break;
    }

    if (refreshTime && prevStatus != ready.status()) {
        *ready.mutable_last_transition_time() = NSupport::ToTimestamp(now);
    }
}

void TBoxStatusRepository::RefreshObjectStatusInProgress(
    API::TCondition& inProgress
    , const API::TBoxStatus& box
    , const TInstant& now
    , bool refreshTime
    , bool changingSpecTimestamp
) {
    NSupport::CopyConditonStatusAndTimestamp(inProgress, box.in_progress());
    API::EConditionStatus prevStatus = inProgress.status();

    inProgress.set_reason(NSupport::ExtractState(API::EBoxState_Name(box.state())));
    inProgress.set_message("");
    if (changingSpecTimestamp) {
        inProgress.set_message(NSupport::APPLYING_OBJECT_SPEC_MESSAGE);
    }

    switch (box.state()) {
        case API::EBoxState_WAITING_FOR_ROOTFS_LAYERS:
        case API::EBoxState_WAITING_FOR_STATIC_RESOURCES:
        case API::EBoxState_CREATING_ROOTFS_VOLUME:
        case API::EBoxState_CREATING_ROOT_CONTAINER:
        case API::EBoxState_WAITING_FOR_VOLUMES:
        case API::EBoxState_LINKING_VOLUMES:
        case API::EBoxState_INIT_PROCESSES:
        case API::EBoxState_REMOVING:
            inProgress.set_status(API::EConditionStatus_TRUE);
            break;
        case API::EBoxState_READY:
            if (!changingSpecTimestamp) {
                inProgress.set_status(API::EConditionStatus_FALSE);
            } else {
                inProgress.set_status(API::EConditionStatus_TRUE);
                NSupport::SetApplyingObjectSpecState(inProgress);
            }
            break;
        case API::EBoxState_REMOVED:
        case API::EBoxState_INVALID:
            inProgress.set_status(API::EConditionStatus_FALSE);
            break;
        case API::EBoxState_UNKNOWN:
            inProgress.set_status(API::EConditionStatus_UNKNOWN);
            break;
        // We don't want to use default
        // So we need this to handle all cases
        case API::EBoxState_INT_MIN_SENTINEL_DO_NOT_USE_:
        case API::EBoxState_INT_MAX_SENTINEL_DO_NOT_USE_:
            inProgress.set_status(API::EConditionStatus_UNKNOWN);
            break;
    }

    if (refreshTime && prevStatus != inProgress.status()) {
        *inProgress.mutable_last_transition_time() = NSupport::ToTimestamp(now);
    }
}

void TBoxStatusRepository::RefreshObjectStatusFailed(
    API::TCondition& failed
    , const API::TBoxStatus& box
    , const TInstant& now
    , bool refreshTime
) {
    NSupport::CopyConditonStatusAndTimestamp(failed, box.failed());
    API::EConditionStatus prevStatus = failed.status();

    switch (box.state()) {
        case API::EBoxState_WAITING_FOR_ROOTFS_LAYERS:
        case API::EBoxState_WAITING_FOR_STATIC_RESOURCES:
        case API::EBoxState_CREATING_ROOTFS_VOLUME:
        case API::EBoxState_CREATING_ROOT_CONTAINER:
        case API::EBoxState_WAITING_FOR_VOLUMES:
        case API::EBoxState_LINKING_VOLUMES:
        case API::EBoxState_INIT_PROCESSES:
        case API::EBoxState_REMOVING:
        case API::EBoxState_REMOVED:
            // UNKNOWN -> FALSE
            // TRUE and FALSE do not change
            if (failed.status() == API::EConditionStatus_UNKNOWN) {
                failed.set_status(API::EConditionStatus_FALSE);
            }
            break;
        case API::EBoxState_READY:
            failed.set_status(API::EConditionStatus_FALSE);
            failed.set_reason(NSupport::ExtractState(API::EBoxState_Name(box.state())));
            break;
        case API::EBoxState_INVALID:
            failed.set_status(API::EConditionStatus_TRUE);
            failed.set_reason(NSupport::ExtractState(API::EBoxState_Name(box.state())));
            break;
        case API::EBoxState_UNKNOWN:
            failed.set_status(API::EConditionStatus_UNKNOWN);
            failed.set_reason(NSupport::ExtractState(API::EBoxState_Name(box.state())));
            break;
        // We don't want to use default
        // So we need this to handle all cases
        case API::EBoxState_INT_MIN_SENTINEL_DO_NOT_USE_:
        case API::EBoxState_INT_MAX_SENTINEL_DO_NOT_USE_:
            failed.set_status(API::EConditionStatus_UNKNOWN);
            failed.set_reason(NSupport::ExtractState(API::EBoxState_Name(box.state())));
            break;
    }

    if (refreshTime && prevStatus != failed.status()) {
        *failed.mutable_last_transition_time() = NSupport::ToTimestamp(now);
    }
}

const TMutex& TBoxStatusRepository::GetObjectMutex(const TString& objectId) const {
    const TObjectInfo* ptr = Objects_.FindPtr(objectId);
    Y_ENSURE(ptr, "Object '" << objectId << "' not found at TBoxStatusRepository");
    return ptr->Mutex_;
}

} // namespace NInfra::NPodAgent
