#pragma once

#include "status_repository_common.h"

#include <infra/pod_agent/libs/pod_agent/object_meta/object_meta.h>

#include <util/generic/hash_set.h>
#include <util/generic/map.h>
#include <util/generic/vector.h>

namespace NInfra::NPodAgent {

/**
    Thread-safe box status repository
*/
class TBoxStatusRepository: public TStatusRepositoryCommon {
private:
    struct TObjectInfo {
        TMutex Mutex_;
        API::TBoxStatus Status_;
    };

public:
    TBoxStatusRepository()
        : TStatusRepositoryCommon(NStatusRepositoryTypes::EObjectType::BOX)
    {}

    virtual void UpdateSpecTimestamp(TInstant currentTime) override final;
    virtual void PatchTotalStatus(API::TPodAgentStatus& status, TUpdateHolderTargetPtr updateHolderTarget, bool conditionsOnly) const override final;
    virtual bool NeedLongTickPeriod(const TString& objectIdOrHash, TUpdateHolderTargetPtr updateHolderTarget) const override final;

    // Container methods
    virtual void UpdateContainerFailReason(const NStatusRepositoryTypes::TContainerDescription& container, const TString& failReason) override final;
    virtual void IncrementContainerSystemFailureCounter(const NStatusRepositoryTypes::TContainerDescription& container) override final;
    virtual void UpdateContainerState(const NStatusRepositoryTypes::TContainerDescription& container, API::EContainerState state) override final;

    // Common object methods
    virtual void UpdateObjectRevision(const TString& objectId, ui32 revision) override final;
    virtual void UpdateObjectSpecTimestamp(const TString& objectId, ui64 specTimestamp) override final;
    virtual TObjectState UpdateObjectState(const TString& objectIdOrHash, TObjectState state) override final;
    virtual void UpdateObjectFailedMessage(const TString& objectIdOrHash, const TString& failedMessage) override final;
    virtual void IncrementObjectFailCounter(const TString& objectIdOrHash) override final;
    virtual void UpdateObjectIpAddress(const TString& objectIdOrHash, const TString& address) override final;
    virtual TVector<TString> GetObjectIdsByHash(const TString& objectIdOrHash) override final;
    virtual TVector<TCacheObject> GetCacheObjectIdsAndRevisionsByHash(const TString& objectIdOrHash) override final;

    // Box specific methods
    void AddObject(const TBoxMeta& objectMeta);
    void RemoveObject(const TString& objectId);
    bool HasObject(const TString& objectId) const;

    API::TBoxStatus GetObjectStatus(const TString& objectId, TUpdateHolderTargetPtr updateHolderTarget = nullptr) const;
    TVector<TString> GetObjectIds() const;

    void UpdateIp6Subnet112Base(const TString& ip6Subnet112Base);
    TString GetIp6Subnet112Base() const;
    bool HasIp6Address(const TString& address) const;

private:
    virtual const TLightRWLock& GetGlobalContainerLock() const override final;
    virtual const TMutex& GetLocalContainerLock(const NStatusRepositoryTypes::TContainerDescription& container) const override final;
    virtual API::TContainerStatus* GetMutableContainerStatus(const NStatusRepositoryTypes::TContainerDescription& container) override final;
    virtual const API::TContainerStatus* GetContainerStatus(const NStatusRepositoryTypes::TContainerDescription& container) override final;

    const TMutex& GetObjectMutex(const TString &objectId) const;
    API::TBoxStatus GetObjectStatusNoGlobalLock(
        const TString& objectId
        , TUpdateHolderTargetPtr updateHolderTarget
        , bool conditionsOnly
    ) const;

    static void RefreshObjectStatusConditions(
        API::TBoxStatus& targetStatus
        , const API::TBoxStatus& box
        , const TInstant& now
        , bool refreshTime = true
        , bool changingSpecTimestamp = false
    );
    static void RefreshObjectStatusReady(
        API::TCondition& ready
        , const API::TBoxStatus& box
        , const TInstant& now
        , bool refreshTime
        , bool changingSpecTimestamp
    );
    static void RefreshObjectStatusInProgress(
        API::TCondition& inProgress
        , const API::TBoxStatus& box
        , const TInstant& now
        , bool refreshTime
        , bool changingSpecTimestamp
    );
    static void RefreshObjectStatusFailed(
        API::TCondition& failed
        , const API::TBoxStatus& box
        , const TInstant& now
        , bool refreshTime
    );

private:
    THashSet<TString> ActiveIp6Addresses_;
    TString Ip6Subnet112Base_; // first 112 bits of subnet in format '2a02:6b8:c08:68a3:0:696:6937:'
    TLightRWLock Ip6AddressesLock_;

    TMap<TString, TObjectInfo> Objects_;
    TLightRWLock GlobalObjectLock_;
};

using TBoxStatusRepositoryPtr = TIntrusivePtr<TBoxStatusRepository>;

} // namespace NInfra::NPodAgent
