#include "box_status_repository.h"
#include "support_functions.h"
#include "test_functions.h"

#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/update_holder/test_lib/test_functions.h>

#include <google/protobuf/util/message_differencer.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NBoxStatusRepositoryTest {

Y_UNIT_TEST_SUITE(BoxStatusRepositorySuite) {

Y_UNIT_TEST(TestGetObjectType) {
    TBoxStatusRepositoryPtr holder = new TBoxStatusRepository();
    const TString id = "my_box";
    holder->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(id));

    UNIT_ASSERT_EQUAL_C(holder->GetObjectType(), NStatusRepositoryTypes::EObjectType::BOX, ToString(holder->GetObjectType()));
}

Y_UNIT_TEST(TestUpdateHolderNotProvided) {
    TBoxStatusRepositoryPtr holder = new TBoxStatusRepository();
    const TString id = "my_box";
    holder->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(id));

    API::TPodAgentStatus patchStatus;
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->PatchTotalStatus(patchStatus, nullptr, false), yexception, "UpdateHolderTarget not provided");
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->NeedLongTickPeriod(id, nullptr), yexception, "UpdateHolderTarget not provided");
}

Y_UNIT_TEST(TestAddBox) {
    TBoxStatusRepositoryPtr holder = new TBoxStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString id = "my_box";
    holder->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(id));
    API::TBoxStatus status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL(API::EBoxState_UNKNOWN, status.state());
    UNIT_ASSERT_EQUAL("box_meta_container_name", status.container_name());
    UNIT_ASSERT_EQUAL("box_specific_type", status.specific_type());
}

Y_UNIT_TEST(TestBoxPatchTotalStatus) {
    TBoxStatusRepositoryPtr holder = new TBoxStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString id = "my_box";

    holder->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(id));
    // Changing something in the status so that it is not equal to the default one
    holder->UpdateObjectState(id, API::EBoxState_READY);
    API::TBoxStatus status = holder->GetObjectStatus(id);

    {
        API::TPodAgentStatus patchStatus;
        holder->PatchTotalStatus(patchStatus, updateHolder->GetUpdateHolderTarget(), false);
        UNIT_ASSERT_EQUAL_C(
            patchStatus.boxes(0).id()
            , id
            , patchStatus.boxes(0).id()
        );
        UNIT_ASSERT_EQUAL_C(
            patchStatus.boxes(0).state()
            , API::EBoxState_READY
            , API::EBoxState_Name(patchStatus.boxes(0).state())
        );
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(status, patchStatus.boxes(0)));
    }

    {
        API::TPodAgentStatus patchStatus;
        holder->PatchTotalStatus(patchStatus, updateHolder->GetUpdateHolderTarget(), true);
        UNIT_ASSERT_EQUAL_C(
            patchStatus.boxes(0).id()
            , id
            , patchStatus.boxes(0).id()
        );
        UNIT_ASSERT_EQUAL_C(
            patchStatus.boxes(0).state()
            // conditionsOnly = true
            , API::EBoxState_UNKNOWN
            , API::EBoxState_Name(patchStatus.boxes(0).state())
        );
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(status.ready(), patchStatus.boxes(0).ready()));
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(status.in_progress(), patchStatus.boxes(0).in_progress()));
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(status.failed(), patchStatus.boxes(0).failed()));
    }
}

Y_UNIT_TEST(TestBoxRevisionAndSpecTimestamp) {
    TBoxStatusRepositoryPtr holder = new TBoxStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString id = "my_box";

    holder->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(id, 0, 1));
    API::TBoxStatus status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL_C(status.spec_timestamp(), 1, status.spec_timestamp());
    UNIT_ASSERT_EQUAL_C(status.revision(), 2, status.revision());

    holder->UpdateObjectSpecTimestamp(id, 3);
    holder->UpdateObjectRevision(id, 4);
    status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL_C(status.spec_timestamp(), 3, status.spec_timestamp());
    UNIT_ASSERT_EQUAL_C(status.revision(), 4, status.revision());
}

Y_UNIT_TEST(TestHasBox) {
    TBoxStatusRepositoryPtr holder = new TBoxStatusRepository();
    TString id = "my_box";
    UNIT_ASSERT(!holder->HasObject(id));
    holder->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(id));
    UNIT_ASSERT(holder->HasObject(id));
}

Y_UNIT_TEST(TestUpdateBoxFailedMessage) {
    TBoxStatusRepositoryPtr holder = new TBoxStatusRepository();
    const TString id = "my_box";
    TString message = "message-reason";
    holder->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(id));
    holder->UpdateObjectFailedMessage(id, message);
    API::TBoxStatus status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL(message, status.failed().message());
}

Y_UNIT_TEST(TestUpdateBoxState) {
    TBoxStatusRepositoryPtr holder = new TBoxStatusRepository();
    const TString id = "my_box";
    holder->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(id));
    holder->UpdateObjectState(id, API::EBoxState_READY);
    API::TBoxStatus status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL(API::EBoxState_READY, status.state());
}

Y_UNIT_TEST(TestAddBoxWithRefs) {
    TBoxStatusRepositoryPtr holder = new TBoxStatusRepository();
    TString boxId = "my_box";
    TVector<TString> layerRefs = {"test_layer_one", "test_layer_two"};
    TVector<TString> staticResourceRefs = {"test_resource_one", "test_resource_two"};
    TVector<TString> volumeRefs = {"test_volume_one", "test_volume_two"};
    holder->AddObject(NObjectMetaTestLib::CreateBoxMetaWithDependence(boxId, layerRefs, staticResourceRefs, volumeRefs));
    API::TBoxStatus status = holder->GetObjectStatus(boxId);

    TSet<TString> checkStaticResourceRefs(status.static_resource_refs().begin(), status.static_resource_refs().end());
    TSet<TString> checkLayerRefs(status.rootfs_layer_refs().begin(), status.rootfs_layer_refs().end());
    TSet<TString> checkVolumeRefs(status.volume_refs().begin(), status.volume_refs().end());

    for (auto& layerRef : layerRefs) {
        UNIT_ASSERT(checkLayerRefs.contains(layerRef));
    }
    UNIT_ASSERT_EQUAL_C(checkLayerRefs.size(), layerRefs.size(), checkLayerRefs.size());

    for (auto& staticResourceRef : staticResourceRefs) {
        UNIT_ASSERT(checkStaticResourceRefs.contains(staticResourceRef));
    }
    UNIT_ASSERT_EQUAL_C(checkStaticResourceRefs.size(), staticResourceRefs.size(), checkStaticResourceRefs.size());

    for (auto& volumeRef : volumeRefs) {
        UNIT_ASSERT(checkVolumeRefs.contains(volumeRef));
    }
    UNIT_ASSERT_EQUAL_C(checkVolumeRefs.size(), volumeRefs.size(), checkVolumeRefs.size());
}

Y_UNIT_TEST(TestUpdateBoxStateAndReadyStatus) {
    TBoxStatusRepositoryPtr holder = new TBoxStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString id = "my_box";
    holder->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(id));

    holder->UpdateObjectState(id, API::EBoxState_UNKNOWN);
    API::TBoxStatus status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EBoxState_UNKNOWN, status.state());
    auto ready = status.ready();
    UNIT_ASSERT_EQUAL(ready.last_transition_time().seconds(), 0);
    UNIT_ASSERT_EQUAL(ready.last_transition_time().nanos(), 0);
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_UNKNOWN);
    UNIT_ASSERT(!holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(id, API::EBoxState_INVALID);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EBoxState_INVALID, status.state());
    ready = status.ready();
    UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT(holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(id, API::EBoxState_REMOVED);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EBoxState_REMOVED, status.state());
    ready = status.ready();
    UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT(!holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(id, API::EBoxState_READY);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EBoxState_READY, status.state());
    ready = status.ready();
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT_EQUAL(NSupport::ExtractState(API::EBoxState_Name(API::EBoxState_READY)), ready.reason());
    UNIT_ASSERT_EQUAL("", ready.message());
    UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
    UNIT_ASSERT(holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateSpecTimestamp(TInstant::Now());
    updateHolder->SetBoxTarget(NObjectTargetTestLib::CreateBoxTargetSimple(id));
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EBoxState_READY, status.state());
    ready  = status.ready();
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT_EQUAL(NSupport::APPLYING_OBJECT_SPEC, ready.reason());
    UNIT_ASSERT_EQUAL(NSupport::APPLYING_OBJECT_SPEC_MESSAGE, ready.message());
    UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
    UNIT_ASSERT(!holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL(API::EBoxState_READY, status.state());
    ready  = status.ready();
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT_EQUAL(NSupport::ExtractState(API::EBoxState_Name(API::EBoxState_READY)), ready.reason());
    UNIT_ASSERT_EQUAL("", ready.message());
    UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
}

Y_UNIT_TEST(TestUpdateBoxStateAndInProgressStatus) {
    TBoxStatusRepositoryPtr holder = new TBoxStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString id = "my_box";
    holder->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(id));

    holder->UpdateObjectState(id, API::EBoxState_UNKNOWN);
    API::TBoxStatus status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EBoxState_UNKNOWN, status.state());
    auto inProgress = status.in_progress();
    UNIT_ASSERT_EQUAL(inProgress.last_transition_time().seconds(), 0);
    UNIT_ASSERT_EQUAL(inProgress.last_transition_time().nanos(), 0);
    UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_UNKNOWN);

    holder->UpdateObjectState(id, API::EBoxState_READY);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EBoxState_READY, status.state());
    inProgress = status.in_progress();
    UNIT_ASSERT(inProgress.last_transition_time().seconds() != 0 || inProgress.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_FALSE);

    holder->UpdateObjectState(id, API::EBoxState_WAITING_FOR_ROOTFS_LAYERS);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EBoxState_WAITING_FOR_ROOTFS_LAYERS, status.state());
    inProgress = status.in_progress();
    UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT_EQUAL(NSupport::ExtractState(API::EBoxState_Name(API::EBoxState_WAITING_FOR_ROOTFS_LAYERS)), inProgress.reason());
    UNIT_ASSERT_EQUAL("", inProgress.message());
    UNIT_ASSERT(inProgress.last_transition_time().seconds() != 0 || inProgress.last_transition_time().nanos() != 0);

    holder->UpdateObjectState(id, API::EBoxState_READY);
    holder->UpdateSpecTimestamp(TInstant::Now());
    updateHolder->SetBoxTarget(NObjectTargetTestLib::CreateBoxTargetSimple(id));
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EBoxState_READY, status.state());
    inProgress = status.in_progress();
    UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT_EQUAL(NSupport::APPLYING_OBJECT_SPEC, inProgress.reason());
    UNIT_ASSERT_EQUAL(NSupport::APPLYING_OBJECT_SPEC_MESSAGE, inProgress.message());
    UNIT_ASSERT(inProgress.last_transition_time().seconds() != 0 || inProgress.last_transition_time().nanos() != 0);

    status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL(API::EBoxState_READY, status.state());
    inProgress = status.in_progress();
    UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT_EQUAL(NSupport::ExtractState(API::EBoxState_Name(API::EBoxState_READY)), inProgress.reason());
    UNIT_ASSERT_EQUAL("", inProgress.message());
    UNIT_ASSERT(inProgress.last_transition_time().seconds() != 0 || inProgress.last_transition_time().nanos() != 0);
}

Y_UNIT_TEST(TestUpdateBoxStateAndFailedStatus) {
    TBoxStatusRepositoryPtr holder = new TBoxStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString id = "my_box";
    holder->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(id));

    holder->UpdateObjectState(id, API::EBoxState_UNKNOWN);
    API::TBoxStatus status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EBoxState_UNKNOWN, status.state());
    auto failed = status.failed();
    UNIT_ASSERT_EQUAL(failed.last_transition_time().seconds(), 0);
    UNIT_ASSERT_EQUAL(failed.last_transition_time().nanos(), 0);
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_UNKNOWN);
    UNIT_ASSERT(!holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(id, API::EBoxState_REMOVED);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EBoxState_REMOVED, status.state());
    failed = status.failed();
    UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT(!holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(id, API::EBoxState_READY);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EBoxState_READY, status.state());
    failed = status.failed();
    UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT(holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(id, API::EBoxState_INVALID);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EBoxState_INVALID, status.state());
    failed = status.failed();
    UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT(holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(id, API::EBoxState_REMOVED);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EBoxState_REMOVED, status.state());
    failed = status.failed();
    UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT(!holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(id, API::EBoxState_READY);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EBoxState_READY, status.state());
    failed = status.failed();
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT_EQUAL(NSupport::ExtractState(API::EBoxState_Name(API::EBoxState_READY)), failed.reason());
    UNIT_ASSERT_EQUAL("", failed.message());
    UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);
    UNIT_ASSERT(holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateSpecTimestamp(TInstant::Now());
    updateHolder->SetBoxTarget(NObjectTargetTestLib::CreateBoxTargetSimple(id));
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EBoxState_READY, status.state());
    failed  = status.failed();
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT_EQUAL(NSupport::ExtractState(API::EBoxState_Name(API::EBoxState_READY)), failed.reason());
    UNIT_ASSERT_EQUAL("", failed.message());
    UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);
    UNIT_ASSERT(!holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL(API::EBoxState_READY, status.state());
    failed  = status.failed();
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT_EQUAL(NSupport::ExtractState(API::EBoxState_Name(API::EBoxState_READY)), failed.reason());
    UNIT_ASSERT_EQUAL("", failed.message());
    UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);
}

Y_UNIT_TEST(TestUpdateBoxNotFound) {
    TBoxStatusRepositoryPtr holder = new TBoxStatusRepository();
    const TString id = "my_box";
    holder->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(id));
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->UpdateObjectState(id + "_other", API::EBoxState_READY), yexception, "not found at TBoxStatusRepository");
}

Y_UNIT_TEST(TestGetBoxIds) {
    TBoxStatusRepositoryPtr holder = new TBoxStatusRepository();
    const TString id = "my_box";
    holder->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(id));

    UNIT_ASSERT_EQUAL(
        holder->GetObjectIds()
        , TVector<TString>(
            {
                id
            }
        )
    );

    // Must be empty
    UNIT_ASSERT_EQUAL(
        holder->GetObjectIdsByHash(id)
        , TVector<TString>()
    );
    UNIT_ASSERT_EQUAL(
        holder->GetCacheObjectIdsAndRevisionsByHash(id)
        , TVector<TStatusRepositoryCommon::TCacheObject>()
    );
}

Y_UNIT_TEST(TestBoxFailCounter) {
    TBoxStatusRepositoryPtr holder = new TBoxStatusRepository();
    const TString id = "my_box";
    const size_t cntIter = 10;

    holder->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(id));
    for (size_t i = 0; i < cntIter; ++i) {
        holder->IncrementObjectFailCounter(id);
    }

    UNIT_ASSERT_EQUAL(holder->GetObjectStatus(id).fail_counter(), cntIter);
}

Y_UNIT_TEST(TestBoxMetaContainer) {
    TBoxStatusRepositoryPtr holder = new TBoxStatusRepository();
    const TString id = "my_box";
    const NStatusRepositoryTypes::TContainerDescription container(
        id
        , NStatusRepositoryTypes::EObjectType::BOX
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::META
    );

    holder->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(id));
    holder->UpdateContainerFailReason(container, "Some fail reason");
    holder->IncrementContainerSystemFailureCounter(container);

    auto boxStatus = holder->GetObjectStatus(id);
    UNIT_ASSERT_STRING_CONTAINS(boxStatus.failed().message(), "Some fail reason");
    UNIT_ASSERT_STRING_CONTAINS(boxStatus.failed().message(), ToString(NStatusRepositoryTypes::TContainerDescription::EContainerType::META));
    UNIT_ASSERT_EQUAL_C(boxStatus.fail_counter(), 1, boxStatus.fail_counter());

    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->IncrementContainerPositiveReturnCodeCounter(container), yexception, "own lock");
}

Y_UNIT_TEST(TestBoxInitContainer) {
    TBoxStatusRepositoryPtr holder = new TBoxStatusRepository();
    const TString id = "my_box";
    const size_t initSize = 10;
    const NStatusRepositoryTypes::TContainerDescription container(
        id
        , NStatusRepositoryTypes::EObjectType::BOX
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::INIT
        , 2
    );

    holder->AddObject(
        TBoxMeta(
            id
            , 0
            , 0
            , {}
            , {}
            , {}
            , ""
            , NObjectMetaTestLib::GenerateInitContainerNames(initSize)
            , ""
        )
    );
    UNIT_ASSERT_EQUAL_C(holder->GetObjectStatus(id).inits().size(), initSize, holder->GetObjectStatus(id).inits().size());

    holder->UpdateContainerFailReason(container, "Some fail reason");
    holder->IncrementContainerSystemFailureCounter(container);

    auto containerStatus = holder->GetObjectStatus(id).inits(2);
    UNIT_ASSERT_EQUAL_C(containerStatus.current().fail_reason(), "Some fail reason", containerStatus.current().fail_reason());
    UNIT_ASSERT_EQUAL_C(containerStatus.system_failure_counter(), 1, containerStatus.system_failure_counter());
}

Y_UNIT_TEST(TestBoxInitContainerBigInitNumber) {
    TBoxStatusRepositoryPtr holder = new TBoxStatusRepository();
    const TString id = "my_box";
    const NStatusRepositoryTypes::TContainerDescription container(
        id
        , NStatusRepositoryTypes::EObjectType::BOX
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::INIT
        , 2
    );

    holder->AddObject(
        TBoxMeta(
            id
            , 0
            , 0
            , {}
            , {}
            , {}
            , ""
            , NObjectMetaTestLib::GenerateInitContainerNames(0)
            , ""
        )
    );

    UNIT_ASSERT_EXCEPTION_CONTAINS(
        holder->UpdateContainerFailReason(container, "Some fail reason")
        , yexception
        , "Too big init num for box 'my_box' InitSize = 0, InitNum = 2"
    );
    UNIT_ASSERT_EQUAL_C(holder->GetObjectStatus(id).inits().size(), 0, holder->GetObjectStatus(id).inits().size());
}

Y_UNIT_TEST(TestBoxSubnet112) {
    TBoxStatusRepositoryPtr holder = new TBoxStatusRepository();
    const TString ip6Subnet112 = "2a02:6b8:c08:68a3:0:696:6937:";

    UNIT_ASSERT_EQUAL_C(holder->GetIp6Subnet112Base(), "", holder->GetIp6Subnet112Base());
    holder->UpdateIp6Subnet112Base("2a02:6b8:c08:68a3:0:696:6937:");
    UNIT_ASSERT_EQUAL_C(holder->GetIp6Subnet112Base(), ip6Subnet112, holder->GetIp6Subnet112Base());

    UNIT_ASSERT_EXCEPTION_CONTAINS(
        holder->UpdateIp6Subnet112Base("bad_ip")
        , yexception
        , "must ends with"
    );

    UNIT_ASSERT_EQUAL_C(holder->GetIp6Subnet112Base(), ip6Subnet112, holder->GetIp6Subnet112Base());
    holder->UpdateIp6Subnet112Base("");
    UNIT_ASSERT_EQUAL_C(holder->GetIp6Subnet112Base(), "", holder->GetIp6Subnet112Base());
}

Y_UNIT_TEST(TestBoxIp6Address) {
    TBoxStatusRepositoryPtr holder = new TBoxStatusRepository();
    const size_t cntIter = 10;
    const TString boxId = "my_box_";
    const TString boxIp6 = "my_box_ip6_";

    for (size_t i = 0; i < cntIter; ++i) {
        holder->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(boxId + ToString(i)));
        UNIT_ASSERT(!holder->HasIp6Address(""));
    }

    for (size_t i = 0; i < cntIter; ++i) {
        UNIT_ASSERT(!holder->HasIp6Address(boxIp6 + ToString(i)));
        UNIT_ASSERT(!holder->HasIp6Address(""));

        // Update twice to verify that assigning the old value will not result in an error
        holder->UpdateObjectIpAddress(boxId + ToString(i), boxIp6 + ToString(i));
        holder->UpdateObjectIpAddress(boxId + ToString(i), boxIp6 + ToString(i));

        UNIT_ASSERT(holder->HasIp6Address(boxIp6 + ToString(i)));
    }

    UNIT_ASSERT(!holder->HasIp6Address(""));

    for (size_t i = 0; i < cntIter; ++i) {
        UNIT_ASSERT(holder->HasIp6Address(boxIp6 + ToString(i)));
        holder->RemoveObject(boxId + ToString(i));
        UNIT_ASSERT(!holder->HasIp6Address(boxIp6 + ToString(i)));
    }

    holder->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(boxId + "1"));
    holder->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(boxId + "2"));
    holder->UpdateObjectIpAddress(boxId + "1", boxIp6 + "1");
    UNIT_ASSERT_EXCEPTION_CONTAINS(
        holder->UpdateObjectIpAddress(boxId + "2", boxIp6 + "1")
        , yexception
        , "use same ip"
    );
}

}

} // namespace NInfra::NPodAgent::NBoxStatusRepositoryTest
