#include "layer_status_repository.h"

#include "support_functions.h"

#include <util/string/builder.h>

namespace NInfra::NPodAgent {

void TLayerStatusRepository::UpdateSpecTimestamp(TInstant currentTime) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    for (auto& [objectHash, info] : Hashes_) {
        TGuard<TMutex> g(GetObjectHashMutex(objectHash));
        RefreshObjectStatusConditions(info.Status_, info.Status_, currentTime);
    }
}

void TLayerStatusRepository::IncrementContainerSystemFailureCounter(const NStatusRepositoryTypes::TContainerDescription& container) {
    CheckContainerObjectType(container);
    IncrementObjectFailCounter(container.ObjectIdOrHash_);
}

void TLayerStatusRepository::UpdateContainerState(const NStatusRepositoryTypes::TContainerDescription& /* container */, API::EContainerState state) {
    Y_ENSURE(state == API::EContainerState_SYSTEM_FAILURE, "you can only call UpdateContainerState with EContainerState_SYSTEM_FAILURE for layer");
}

void TLayerStatusRepository::PatchTotalStatus(API::TPodAgentStatus& status, TUpdateHolderTargetPtr updateHolderTarget, bool conditionsOnly) const {
    Y_ENSURE(updateHolderTarget, "UpdateHolderTarget not provided for TLayerStatusRepository::PatchTotalStatus");

    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);

    API::TResourceGangStatus* resourceCacheStatus = status.mutable_resource_cache();
    for (const auto& [cacheObject, info]: CacheObjects_) {
        API::TLayerStatus* st = resourceCacheStatus->add_layers();
        *st = GetCacheObjectStatusNoGlobalLock(cacheObject.Id_, cacheObject.Revision_, conditionsOnly);
        NSupport::PatchAllNonReadyConditions(status, *st);
    }

    API::TResourceGangStatus* resourceGangStatus = status.mutable_resource_gang();
    for (const auto& [objectId, info] : Objects_) {
        API::TLayerStatus* st = resourceGangStatus->add_layers();
        *st = GetObjectStatusNoGlobalLock(objectId, updateHolderTarget, conditionsOnly);
        NSupport::PatchAllNonReadyConditions(status, *st);
    }
}

bool TLayerStatusRepository::NeedLongTickPeriod(const TString& objectIdOrHash, TUpdateHolderTargetPtr updateHolderTarget) const {
    Y_ENSURE(updateHolderTarget, "UpdateHolderTarget not provided for TLayerStatusRepository::NeedLongTickPeriod");

    auto status = GetObjectHashStatus(objectIdOrHash);

    return status.ready().status() == API::EConditionStatus_TRUE
        || status.state() == API::ELayerState_INVALID;
}

void TLayerStatusRepository::UpdateContainerFailReason(const NStatusRepositoryTypes::TContainerDescription& container, const TString& failReason) {
    CheckContainerObjectType(container);
    UpdateObjectFailedMessage(
        container.ObjectIdOrHash_
        , TStringBuilder() << "Object " << ToString(container.ContainerType_) << " container FailReason: " << failReason
    );
}

void TLayerStatusRepository::AddCacheObject(const TLayerMeta& cacheObjectMeta) {
    TWriteGuardBase<TLightRWLock> guard(GlobalObjectLock_);

    TCacheObject cacheObject = TCacheObject(cacheObjectMeta.Id_, cacheObjectMeta.Revision_);

    CacheObjects_.insert(
        {
            cacheObject
            , TCacheObjectInfo(
                cacheObjectMeta.DownloadHash_
                , cacheObjectMeta.RemoveSourceFileAfterImport_
            )
        }
    );
    THashInfo& objectHashInfo = Hashes_[cacheObjectMeta.DownloadHash_];
    objectHashInfo.CacheObjectIdsAndRevisions_.insert(cacheObject);
    if (!cacheObjectMeta.RemoveSourceFileAfterImport_) {
        ++objectHashInfo.KeepSourceFileAfterImportCounter_;
    }
}

void TLayerStatusRepository::RemoveCacheObject(const TString& cacheObjectId, ui32 revision) {
    TWriteGuardBase<TLightRWLock> guard(GlobalObjectLock_);

    TCacheObject cacheObject = TCacheObject(cacheObjectId, revision);
    const TCacheObjectInfo& cacheObjectInfo = CacheObjects_.at(cacheObject);
    const TString& objectHash = cacheObjectInfo.ObjectHash_;

    THashInfo& objectHashInfo = Hashes_.at(objectHash);

    if (!cacheObjectInfo.RemoveSourceFileAfterImport_) {
        --objectHashInfo.KeepSourceFileAfterImportCounter_;
    }

    objectHashInfo.CacheObjectIdsAndRevisions_.erase(cacheObject);
    // If this is the last reference clear status
    if (objectHashInfo.ObjectIds_.empty() && objectHashInfo.CacheObjectIdsAndRevisions_.empty()) {
        Hashes_.erase(objectHash);
    }

    CacheObjects_.erase(cacheObject);
}

API::TLayerStatus TLayerStatusRepository::GetCacheObjectStatus(const TString& cacheObjectId, ui32 revision) const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    return GetCacheObjectStatusNoGlobalLock(cacheObjectId, revision, false);
}

API::TLayerStatus TLayerStatusRepository::GetCacheObjectStatusNoGlobalLock(
    const TString& cacheObjectId
    , ui32 revision
    , bool conditionsOnly
) const {
    TGuard<TMutex> g(GetCacheObjectMutex(cacheObjectId, revision));

    const TString& objectHash = CacheObjects_.at(TCacheObject(cacheObjectId, revision)).ObjectHash_;
    TGuard<TMutex> gh(GetObjectHashMutex(objectHash));
    // WARNING: Do not copy full status here
    const API::TLayerStatus& fullStatus = Hashes_.at(objectHash).Status_;

    API::TLayerStatus status;
    if (!conditionsOnly) {
        status.CopyFrom(fullStatus);
    }
    // Always add real id and revision to status
    status.set_id(cacheObjectId);
    status.set_revision(revision);

    RefreshObjectStatusConditions(status, fullStatus, TInstant::Zero(), false, false);

    return status;
}

bool TLayerStatusRepository::HasCacheObject(const TString& cacheObjectId, ui32 revision) const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    return CacheObjects_.FindPtr(TCacheObject(cacheObjectId, revision));
}

TVector<TStatusRepositoryCommon::TCacheObject> TLayerStatusRepository::GetCacheObjectIdsAndRevisions() const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);

    TVector<TCacheObject> result;
    for (const auto& [cacheObject, info]: CacheObjects_) {
        result.push_back(cacheObject);
    }
    return result;
}

TString TLayerStatusRepository::GetCacheObjectHash(const TString& cacheObjectId, ui32 revision) const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetCacheObjectMutex(cacheObjectId, revision));
    return CacheObjects_.at(TCacheObject(cacheObjectId, revision)).ObjectHash_;
}

void TLayerStatusRepository::AddObject(const TLayerMeta& objectMeta) {
    TWriteGuardBase<TLightRWLock> guard(GlobalObjectLock_);

    Objects_.insert(
        {
            objectMeta.Id_
            , TObjectInfo(
                objectMeta.DownloadHash_
                , objectMeta.RemoveSourceFileAfterImport_
                , objectMeta.SpecTimestamp_
                , objectMeta.Revision_
            )
        }
    );
    THashInfo& objectHashInfo = Hashes_[objectMeta.DownloadHash_];
    objectHashInfo.ObjectIds_.insert(objectMeta.Id_);
    if (!objectMeta.RemoveSourceFileAfterImport_) {
        ++objectHashInfo.KeepSourceFileAfterImportCounter_;
    }
}

void TLayerStatusRepository::RemoveObject(const TString& objectId) {
    TWriteGuardBase<TLightRWLock> guard(GlobalObjectLock_);

    const TObjectInfo& objectInfo = Objects_.at(objectId);
    const TString& objectHash = objectInfo.ObjectHash_;

    THashInfo& objectHashInfo = Hashes_.at(objectHash);

    if (!objectInfo.RemoveSourceFileAfterImport_) {
        --objectHashInfo.KeepSourceFileAfterImportCounter_;
    }

    objectHashInfo.ObjectIds_.erase(objectId);
    // If this is the last reference clear status
    if (objectHashInfo.ObjectIds_.empty() && objectHashInfo.CacheObjectIdsAndRevisions_.empty()) {
        Hashes_.erase(objectHash);
    }

    Objects_.erase(objectId);
}

bool TLayerStatusRepository::HasObjectHash(const TString& objectHash) const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    return Hashes_.FindPtr(objectHash);
}

TString TLayerStatusRepository::GetObjectHash(const TString& objectId) const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    return Objects_.at(objectId).ObjectHash_;
}

bool TLayerStatusRepository::GetObjectRemoveSourceFileAfterImportFlag(const TString& objectId) const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    return Objects_.at(objectId).RemoveSourceFileAfterImport_;
}

void TLayerStatusRepository::UpdateObjectRevision(const TString& objectId, ui32 revision) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    Objects_.at(objectId).Revision_ = revision;
}

void TLayerStatusRepository::UpdateObjectSpecTimestamp(const TString& objectId, ui64 specTimestamp) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    Objects_.at(objectId).SpecTimestamp_ = specTimestamp;
}

TStatusRepositoryCommon::TObjectState TLayerStatusRepository::UpdateObjectState(const TString& objectIdOrHash, TObjectState state) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectHashMutex(objectIdOrHash));
    API::TLayerStatus& layer = Hashes_.at(objectIdOrHash).Status_;
    TStatusRepositoryCommon::TObjectState oldState = layer.state();
    if (oldState != state) {
        layer.set_state(std::get<API::ELayerState>(state));
        TInstant now = TInstant::Now();
        RefreshObjectStatusConditions(layer, layer, now);
    }
    return oldState;
}

void TLayerStatusRepository::IncrementObjectFailCounter(const TString& objectIdOrHash) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectHashMutex(objectIdOrHash));
    API::TLayerStatus& status = Hashes_.at(objectIdOrHash).Status_;
    status.set_fail_counter(status.fail_counter() + 1);
}

void TLayerStatusRepository::UpdateObjectFailedMessage(const TString& objectIdOrHash, const TString& failedMessage) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectHashMutex(objectIdOrHash));
    Hashes_.at(objectIdOrHash).Status_.mutable_failed()->set_message(NSupport::Truncate(failedMessage));
}

void TLayerStatusRepository::UpdateObjectVerifyAttempt(const TString& objectIdOrHash) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectHashMutex(objectIdOrHash));
    API::TLayerStatus& status = Hashes_.at(objectIdOrHash).Status_;
    status.set_verification_attempts_counter(status.verification_attempts_counter() + 1);
}

void TLayerStatusRepository::UpdateObjectDownloadAttempt(const TString& objectIdOrHash) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectHashMutex(objectIdOrHash));
    API::TLayerStatus& status = Hashes_.at(objectIdOrHash).Status_;
    status.set_download_attempts_counter(status.download_attempts_counter() + 1);
}

void TLayerStatusRepository::UpdateObjectDownloadProgress(const TString& objectIdOrHash, ui32 percent) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectHashMutex(objectIdOrHash));
    Hashes_.at(objectIdOrHash).Status_.mutable_download_progress()->set_percent(percent);
}

API::TLayerStatus TLayerStatusRepository::GetObjectStatus(const TString& objectId, TUpdateHolderTargetPtr updateHolderTarget) const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    return GetObjectStatusNoGlobalLock(objectId, updateHolderTarget, false);
}

API::TLayerStatus TLayerStatusRepository::GetObjectStatusNoGlobalLock(
    const TString& objectId
    , TUpdateHolderTargetPtr updateHolderTarget
    , bool conditionsOnly
) const {
    TGuard<TMutex> g(GetObjectMutex(objectId));

    const TObjectInfo& objectInfo = Objects_.at(objectId);
    const TString& objectHash = objectInfo.ObjectHash_;
    TGuard<TMutex> gh(GetObjectHashMutex(objectHash));
    // WARNING: Do not copy full status here
    const API::TLayerStatus& fullStatus = Hashes_.at(objectHash).Status_;

    API::TLayerStatus status;
    if (!conditionsOnly) {
        status.CopyFrom(fullStatus);
        status.set_spec_timestamp(objectInfo.SpecTimestamp_);
        status.set_revision(objectInfo.Revision_);
    }
    // Always add real id to status
    status.set_id(objectId);

    bool changingSpecTimestamp = updateHolderTarget ? updateHolderTarget->LayerHasTarget(objectId) : false;
    RefreshObjectStatusConditions(status, fullStatus, TInstant::Zero(), false, changingSpecTimestamp);

    return status;
}

API::TLayerStatus TLayerStatusRepository::GetObjectHashStatus(const TString& objectHash) const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectHashMutex(objectHash));
    return Hashes_.at(objectHash).Status_;
}

ui32 TLayerStatusRepository::GetObjectHashKeepSourceFileAfterImportCounter(const TString& objectHash) const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectHashMutex(objectHash));
    return Hashes_.at(objectHash).KeepSourceFileAfterImportCounter_;
}

bool TLayerStatusRepository::HasObject(const TString& objectId) const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    return Objects_.FindPtr(objectId);
}

TVector<TString> TLayerStatusRepository::GetObjectIds() const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);

    TVector<TString> result;
    for (const auto& [objectId, info] : Objects_) {
        result.push_back(objectId);
    }
    return result;
}

TVector<TString> TLayerStatusRepository::GetObjectIdsByHash(const TString& objectIdOrHash) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);

    if (auto ptr = Hashes_.FindPtr(objectIdOrHash)) {
        return TVector<TString>(ptr->ObjectIds_.begin(), ptr->ObjectIds_.end());
    } else {
        return {};
    }
}

TVector<TStatusRepositoryCommon::TCacheObject> TLayerStatusRepository::GetCacheObjectIdsAndRevisionsByHash(const TString& objectIdOrHash) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);

    if (auto ptr = Hashes_.FindPtr(objectIdOrHash)) {
        return TVector<TStatusRepositoryCommon::TCacheObject>(
            ptr->CacheObjectIdsAndRevisions_.begin(), ptr->CacheObjectIdsAndRevisions_.end()
        );
    } else {
        return {};
    }
}

void TLayerStatusRepository::UpdateActiveDownloadContainersLimit(ui32 activeDownloadContainersLimit) {
    TWriteGuardBase<TLightRWLock> guard(ActiveDownloadContainersLimitLock_);
    ActiveDownloadContainersLimit_ = activeDownloadContainersLimit;
}

void TLayerStatusRepository::UpdateActiveVerifyContainersLimit(ui32 activeVerifyContainersLimit) {
    TWriteGuardBase<TLightRWLock> guard(ActiveVerifyContainersLimitLock_);
    ActiveVerifyContainersLimit_ = activeVerifyContainersLimit;
}

ui32 TLayerStatusRepository::GetActiveDownloadContainersLimit() const {
    TReadGuardBase<TLightRWLock> guard(ActiveDownloadContainersLimitLock_);
    return ActiveDownloadContainersLimit_;
}

ui32 TLayerStatusRepository::GetActiveVerifyContainersLimit() const {
    TReadGuardBase<TLightRWLock> guard(ActiveVerifyContainersLimitLock_);
    return ActiveVerifyContainersLimit_;
}

void TLayerStatusRepository::RefreshObjectStatusConditions(
    API::TLayerStatus& targetStatus
    , const API::TLayerStatus& layer
    , const TInstant& now
    , bool refreshTime
    , bool changingSpecTimestamp
) {
    RefreshObjectStatusReady(
        *targetStatus.mutable_ready()
        , layer
        , now
        , refreshTime
        , changingSpecTimestamp
    );
    RefreshObjectStatusInProgress(
        *targetStatus.mutable_in_progress()
        , layer
        , now
        , refreshTime
        , changingSpecTimestamp
    );
    RefreshObjectStatusFailed(
        *targetStatus.mutable_failed()
        , layer
        , now
        , refreshTime
    );
}

void TLayerStatusRepository::RefreshObjectStatusReady(
    API::TCondition& ready
    , const API::TLayerStatus& layer
    , const TInstant& now
    , bool refreshTime
    , bool changingSpecTimestamp
) {
    NSupport::CopyConditonStatusAndTimestamp(ready, layer.ready());
    API::EConditionStatus prevStatus = ready.status();

    ready.set_reason(NSupport::ExtractState(API::ELayerState_Name(layer.state())));
    ready.set_message("");
    if (changingSpecTimestamp) {
        ready.set_message(NSupport::APPLYING_OBJECT_SPEC_MESSAGE);
    }

    switch (layer.state()) {
        case API::ELayerState_READY:
            if (!changingSpecTimestamp) {
                ready.set_status(API::EConditionStatus_TRUE);
            } else {
                ready.set_status(API::EConditionStatus_FALSE);
                NSupport::SetApplyingObjectSpecState(ready);
            }
            break;
        case API::ELayerState_DOWNLOADING:
        case API::ELayerState_VERIFYING:
        case API::ELayerState_IMPORTING:
        case API::ELayerState_REMOVING:
        case API::ELayerState_REMOVED:
        case API::ELayerState_INVALID:
        case API::ELayerState_WAITING_FOR_RESOURCE_GANG_META_CONTAINER:
        case API::ELayerState_IN_QUEUE:
        case API::ELayerState_VERIFICATION_IN_QUEUE:
            ready.set_status(API::EConditionStatus_FALSE);
            break;
        case API::ELayerState_UNKNOWN:
            ready.set_status(API::EConditionStatus_UNKNOWN);
            break;
        // We don't want to use default
        // So we need this to handle all cases
        case API::ELayerState_INT_MIN_SENTINEL_DO_NOT_USE_:
        case API::ELayerState_INT_MAX_SENTINEL_DO_NOT_USE_:
            ready.set_status(API::EConditionStatus_UNKNOWN);
            break;
    }

    if (refreshTime && prevStatus != ready.status()) {
        *ready.mutable_last_transition_time() = NSupport::ToTimestamp(now);
    }
}

void TLayerStatusRepository::RefreshObjectStatusInProgress(
    API::TCondition& inProgress
    , const API::TLayerStatus& layer
    , const TInstant& now
    , bool refreshTime
    , bool changingSpecTimestamp
) {
    NSupport::CopyConditonStatusAndTimestamp(inProgress, layer.in_progress());
    API::EConditionStatus prevStatus = inProgress.status();

    inProgress.set_reason(NSupport::ExtractState(API::ELayerState_Name(layer.state())));
    inProgress.set_message("");
    if (changingSpecTimestamp) {
        inProgress.set_message(NSupport::APPLYING_OBJECT_SPEC_MESSAGE);
    }

    switch (layer.state()) {
        case API::ELayerState_DOWNLOADING:
        case API::ELayerState_VERIFYING:
        case API::ELayerState_IMPORTING:
        case API::ELayerState_REMOVING:
            inProgress.set_status(API::EConditionStatus_TRUE);
            break;
        case API::ELayerState_READY:
        case API::ELayerState_REMOVED:
            if (!changingSpecTimestamp) {
                inProgress.set_status(API::EConditionStatus_FALSE);
            } else {
                inProgress.set_status(API::EConditionStatus_TRUE);
                NSupport::SetApplyingObjectSpecState(inProgress);
            }
            break;
        case API::ELayerState_WAITING_FOR_RESOURCE_GANG_META_CONTAINER:
        case API::ELayerState_IN_QUEUE:
        case API::ELayerState_VERIFICATION_IN_QUEUE:
        case API::ELayerState_INVALID:
            inProgress.set_status(API::EConditionStatus_FALSE);
            break;
        case API::ELayerState_UNKNOWN:
            inProgress.set_status(API::EConditionStatus_UNKNOWN);
            break;
        // We don't want to use default
        // So we need this to handle all cases
        case API::ELayerState_INT_MIN_SENTINEL_DO_NOT_USE_:
        case API::ELayerState_INT_MAX_SENTINEL_DO_NOT_USE_:
            inProgress.set_status(API::EConditionStatus_UNKNOWN);
            break;
    }

    if (refreshTime && prevStatus != inProgress.status()) {
        *inProgress.mutable_last_transition_time() = NSupport::ToTimestamp(now);
    }
}

void TLayerStatusRepository::RefreshObjectStatusFailed(
    API::TCondition& failed
    , const API::TLayerStatus& layer
    , const TInstant& now
    , bool refreshTime
) {
    NSupport::CopyConditonStatusAndTimestamp(failed, layer.failed());
    API::EConditionStatus prevStatus = failed.status();

    switch (layer.state()) {
        case API::ELayerState_DOWNLOADING:
        case API::ELayerState_VERIFYING:
        case API::ELayerState_IMPORTING:
        case API::ELayerState_REMOVING:
        case API::ELayerState_REMOVED:
        case API::ELayerState_WAITING_FOR_RESOURCE_GANG_META_CONTAINER:
        case API::ELayerState_IN_QUEUE:
        case API::ELayerState_VERIFICATION_IN_QUEUE:
            // UNKNOWN -> FALSE
            // TRUE and FALSE do not change
            if (failed.status() == API::EConditionStatus_UNKNOWN) {
                failed.set_status(API::EConditionStatus_FALSE);
            }
            break;
        case API::ELayerState_READY:
            failed.set_status(API::EConditionStatus_FALSE);
            failed.set_reason(NSupport::ExtractState(API::ELayerState_Name(layer.state())));
            break;
        case API::ELayerState_INVALID:
            failed.set_status(API::EConditionStatus_TRUE);
            failed.set_reason(NSupport::ExtractState(API::ELayerState_Name(layer.state())));
            break;
        case API::ELayerState_UNKNOWN:
            failed.set_status(API::EConditionStatus_UNKNOWN);
            failed.set_reason(NSupport::ExtractState(API::ELayerState_Name(layer.state())));
            break;
        // We don't want to use default
        // So we need this to handle all cases
        case API::ELayerState_INT_MIN_SENTINEL_DO_NOT_USE_:
        case API::ELayerState_INT_MAX_SENTINEL_DO_NOT_USE_:
            failed.set_status(API::EConditionStatus_UNKNOWN);
            failed.set_reason(NSupport::ExtractState(API::ELayerState_Name(layer.state())));
            break;
    }

    if (refreshTime && prevStatus != failed.status()) {
        *failed.mutable_last_transition_time() = NSupport::ToTimestamp(now);
    }
}

const TMutex& TLayerStatusRepository::GetCacheObjectMutex(const TString& cacheObjectId, ui32 revision) const {
    const TCacheObjectInfo* ptr = CacheObjects_.FindPtr(TStatusRepositoryCommon::TCacheObject(cacheObjectId, revision));
    Y_ENSURE(ptr, "CacheObject '" << cacheObjectId << "' with revision '" << ToString(revision) << "' not found at TLayerStatusRepository");
    return ptr->Mutex_;
}

const TMutex& TLayerStatusRepository::GetObjectMutex(const TString& objectId) const {
    const TObjectInfo* ptr = Objects_.FindPtr(objectId);
    Y_ENSURE(ptr, "Object '" << objectId << "' not found at TLayerStatusRepository");
    return ptr->Mutex_;
}

const TMutex& TLayerStatusRepository::GetObjectHashMutex(const TString& objectHash) const {
    const THashInfo* ptr = Hashes_.FindPtr(objectHash);
    Y_ENSURE(ptr, "Object hash '" << objectHash << "' not found at TLayerStatusRepository");
    return ptr->Mutex_;
}

} // namespace NInfra::NPodAgent
