#pragma once

#include "status_repository_common.h"

#include <infra/pod_agent/libs/pod_agent/object_meta/object_meta.h>

#include <util/generic/vector.h>
#include <util/generic/map.h>
#include <util/generic/set.h>

namespace NInfra::NPodAgent {

/**
    Thread-safe layer status repository
*/
class TLayerStatusRepository: public TStatusRepositoryCommon {
private:
    struct TCacheObjectInfo {
        TCacheObjectInfo(
            const TString& objectHash
            , bool removeSourceFileAfterImport
        )
            : ObjectHash_(objectHash)
            , RemoveSourceFileAfterImport_(removeSourceFileAfterImport)
        {}

        TMutex Mutex_;
        const TString ObjectHash_;
        const bool RemoveSourceFileAfterImport_;
    };

    struct TObjectInfo {
        TObjectInfo(
            const TString& objectHash
            , bool removeSourceFileAfterImport
            , ui64 specTimestamp
            , ui32 revision
        )
            : ObjectHash_(objectHash)
            , RemoveSourceFileAfterImport_(removeSourceFileAfterImport)
            , SpecTimestamp_(specTimestamp)
            , Revision_(revision)
        {}

        TMutex Mutex_;
        const TString ObjectHash_;
        const bool RemoveSourceFileAfterImport_;

        ui64 SpecTimestamp_;
        ui32 Revision_;
    };

    struct THashInfo {
        TMutex Mutex_;
        API::TLayerStatus Status_;

        TSet<TString> ObjectIds_;
        TSet<TCacheObject> CacheObjectIdsAndRevisions_;

        // Despite the fact that we pass 'RemoveSourceFileAfterImport' in the spec,
        // we are interested in whether there is at least one layer that wants to save the source file,
        // so for simplicity of the code logic, we invert the meaning
        ui32 KeepSourceFileAfterImportCounter_ = 0;
    };

public:
    TLayerStatusRepository()
        : TStatusRepositoryCommon(NStatusRepositoryTypes::EObjectType::LAYER)
        , ActiveDownloadContainersLimit_(0)
        , ActiveVerifyContainersLimit_(0)
    {}

    virtual void UpdateSpecTimestamp(TInstant currentTime) override final;
    virtual void PatchTotalStatus(API::TPodAgentStatus& status, TUpdateHolderTargetPtr updateHolderTarget, bool conditionsOnly) const override final;
    virtual bool NeedLongTickPeriod(const TString& objectIdOrHash, TUpdateHolderTargetPtr updateHolderTarget) const override final;

    // Container methods
    virtual void UpdateContainerFailReason(const NStatusRepositoryTypes::TContainerDescription& container, const TString& failReason) override final;
    virtual void IncrementContainerSystemFailureCounter(const NStatusRepositoryTypes::TContainerDescription& container) override final;
    virtual void UpdateContainerState(const NStatusRepositoryTypes::TContainerDescription& container, API::EContainerState state) override final;

    // Common object methods
    virtual void UpdateObjectRevision(const TString& objectId, ui32 revision) override final;
    virtual void UpdateObjectSpecTimestamp(const TString& objectId, ui64 specTimestamp) override final;
    virtual TObjectState UpdateObjectState(const TString& objectIdOrHash, TObjectState state) override final;
    virtual void UpdateObjectDownloadProgress(const TString& objectIdOrHash, ui32 percent) override final;
    virtual void UpdateObjectFailedMessage(const TString& objectIdOrHash, const TString& failedMessage) override final;
    virtual void IncrementObjectFailCounter(const TString& objectIdOrHash) override final;
    virtual void UpdateObjectVerifyAttempt(const TString& objectIdOrHash) override final;
    virtual void UpdateObjectDownloadAttempt(const TString& objectIdOrHash) override final;
    virtual TVector<TString> GetObjectIdsByHash(const TString& objectIdOrHash) override final;
    virtual TVector<TCacheObject> GetCacheObjectIdsAndRevisionsByHash(const TString& objectIdOrHash) override final;

    // ResourceGang methods
    virtual void UpdateActiveDownloadContainersLimit(ui32 activeDownloadContainersLimit) override final;
    virtual void UpdateActiveVerifyContainersLimit(ui32 activeVerifyContainersLimit) override final;
    virtual ui32 GetActiveDownloadContainersLimit() const override final;
    virtual ui32 GetActiveVerifyContainersLimit() const override final;

    // Layer specific methods

    // CacheLayer
    void AddCacheObject(const TLayerMeta& cacheObjectMeta);
    void RemoveCacheObject(const TString& cacheObjectId, ui32 revision);
    bool HasCacheObject(const TString& cacheObjectId, ui32 revision) const;

    TString GetCacheObjectHash(const TString& cacheObjectId, ui32 revision) const;
    TVector<TCacheObject> GetCacheObjectIdsAndRevisions() const;
    API::TLayerStatus GetCacheObjectStatus(const TString& cacheObjectId, ui32 revision) const;

    // Layer
    void AddObject(const TLayerMeta& objectMeta);
    void RemoveObject(const TString& objectId);
    bool HasObject(const TString& objectId) const;

    TString GetObjectHash(const TString& objectId) const;
    bool GetObjectRemoveSourceFileAfterImportFlag(const TString& objectId) const;
    API::TLayerStatus GetObjectStatus(const TString& objectId, TUpdateHolderTargetPtr updateHolderTarget = nullptr) const;
    TVector<TString> GetObjectIds() const;

    // Hash
    bool HasObjectHash(const TString& objectHash) const;
    ui32 GetObjectHashKeepSourceFileAfterImportCounter(const TString& objectHash) const;
    API::TLayerStatus GetObjectHashStatus(const TString& objectHash) const;

private:
    // CacheLayer
    const TMutex& GetCacheObjectMutex(const TString& cacheObjectId, ui32 revision) const;
    API::TLayerStatus GetCacheObjectStatusNoGlobalLock(
        const TString& cacheObjectId
        , ui32 revision
        , bool conditionsOnly
    ) const;

    // Layer
    const TMutex& GetObjectMutex(const TString& objectId) const;
    API::TLayerStatus GetObjectStatusNoGlobalLock(
        const TString& objectId
        , TUpdateHolderTargetPtr updateHolderTarget
        , bool conditionsOnly
    ) const;

    // Hash
    const TMutex& GetObjectHashMutex(const TString& objectHash) const;

    static void RefreshObjectStatusConditions(
        API::TLayerStatus& targetStatus
        , const API::TLayerStatus& layer
        , const TInstant& now
        , bool refreshTime = true
        , bool changingSpecTimestamp = false
    );
    static void RefreshObjectStatusReady(
        API::TCondition& ready
        , const API::TLayerStatus& layer
        , const TInstant& now
        , bool refreshTime
        , bool changingSpecTimestamp
    );
    static void RefreshObjectStatusInProgress(
        API::TCondition& inProgress
        , const API::TLayerStatus& layer
        , const TInstant& now
        , bool refreshTime
        , bool changingSpecTimestamp
    );
    static void RefreshObjectStatusFailed(
        API::TCondition& failed
        , const API::TLayerStatus& layer
        , const TInstant& now
        , bool refreshTime
    );

private:
    ui32 ActiveDownloadContainersLimit_;
    ui32 ActiveVerifyContainersLimit_;
    TLightRWLock ActiveDownloadContainersLimitLock_;
    TLightRWLock ActiveVerifyContainersLimitLock_;

    TMap<TCacheObject, TCacheObjectInfo> CacheObjects_;
    TMap<TString, TObjectInfo> Objects_;
    TMap<TString, THashInfo> Hashes_;
    TLightRWLock GlobalObjectLock_;
};

using TLayerStatusRepositoryPtr = TIntrusivePtr<TLayerStatusRepository>;

} // namespace NInfra::NPodAgent
