#include "layer_status_repository.h"
#include "support_functions.h"
#include "test_functions.h"

#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/update_holder/test_lib/test_functions.h>

#include <google/protobuf/util/message_differencer.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <util/thread/pool.h>

namespace NInfra::NPodAgent::NLayerStatusRepositoryTest {

Y_UNIT_TEST_SUITE(LayerStatusRepositorySuite) {

Y_UNIT_TEST(TestGetObjectType) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    const TString id = "my_layer";
    const TString downloadHash = "my_hash";
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, downloadHash));

    UNIT_ASSERT_EQUAL_C(holder->GetObjectType(), NStatusRepositoryTypes::EObjectType::LAYER, ToString(holder->GetObjectType()));
}

Y_UNIT_TEST(TestUpdateHolderNotProvided) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    const TString id = "my_layer";
    const TString downloadHash = "my_hash";
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, downloadHash));

    API::TPodAgentStatus patchStatus;
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->PatchTotalStatus(patchStatus, nullptr, false), yexception, "UpdateHolderTarget not provided");
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->NeedLongTickPeriod(downloadHash, nullptr), yexception, "UpdateHolderTarget not provided");
}

Y_UNIT_TEST(TestActiveDownloadContainersLimit) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();

    UNIT_ASSERT_EQUAL_C(holder->GetActiveDownloadContainersLimit(), 0, holder->GetActiveDownloadContainersLimit());
    holder->UpdateActiveDownloadContainersLimit(3);
    UNIT_ASSERT_EQUAL_C(holder->GetActiveDownloadContainersLimit(), 3, holder->GetActiveDownloadContainersLimit());
}

Y_UNIT_TEST(TestActiveVerifyContainersLimit) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();

    UNIT_ASSERT_EQUAL_C(holder->GetActiveVerifyContainersLimit(), 0, holder->GetActiveVerifyContainersLimit());
    holder->UpdateActiveVerifyContainersLimit(3);
    UNIT_ASSERT_EQUAL_C(holder->GetActiveVerifyContainersLimit(), 3, holder->GetActiveVerifyContainersLimit());
}

Y_UNIT_TEST(TestAddLayer) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString id = "my_layer";
    const TString downloadHash = "my_layer_download_hash";
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, downloadHash));

    API::TLayerStatus status = holder->GetObjectStatus(id);
    UNIT_ASSERT(holder->GetObjectRemoveSourceFileAfterImportFlag(id));
    UNIT_ASSERT_EQUAL_C(
        holder->GetObjectHashKeepSourceFileAfterImportCounter(downloadHash)
        , 0
        , holder->GetObjectHashKeepSourceFileAfterImportCounter(downloadHash)
    );
    UNIT_ASSERT_EQUAL(API::ELayerState_UNKNOWN, status.state());
}

Y_UNIT_TEST(TestLayerPatchTotalStatus) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString id = "my_layer";
    const TString downloadHash = "my_layer_download_hash";

    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, downloadHash));
    // Changing something in the status so that it is not equal to the default one
    holder->UpdateObjectState(downloadHash, API::ELayerState_READY);
    API::TLayerStatus status = holder->GetObjectStatus(id);

    {
        API::TPodAgentStatus patchStatus;
        holder->PatchTotalStatus(patchStatus, updateHolder->GetUpdateHolderTarget(), false);
        UNIT_ASSERT_EQUAL_C(
            patchStatus.resource_gang().layers(0).id()
            , id
            , patchStatus.resource_gang().layers(0).id()
        );
        UNIT_ASSERT_EQUAL_C(
            patchStatus.resource_gang().layers(0).state()
            , API::ELayerState_READY
            , API::ELayerState_Name(patchStatus.resource_gang().layers(0).state())
        );
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(status, patchStatus.resource_gang().layers(0)));
    }

    {
        API::TPodAgentStatus patchStatus;
        holder->PatchTotalStatus(patchStatus, updateHolder->GetUpdateHolderTarget(), true);
        UNIT_ASSERT_EQUAL_C(
            patchStatus.resource_gang().layers(0).id()
            , id
            , patchStatus.resource_gang().layers(0).id()
        );
        UNIT_ASSERT_EQUAL_C(
            patchStatus.resource_gang().layers(0).state()
            // conditionsOnly = true
            , API::ELayerState_UNKNOWN
            , API::ELayerState_Name(patchStatus.resource_gang().layers(0).state())
        );
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(status.ready(), patchStatus.resource_gang().layers(0).ready()));
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(status.in_progress(), patchStatus.resource_gang().layers(0).in_progress()));
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(status.failed(), patchStatus.resource_gang().layers(0).failed()));
    }
}

Y_UNIT_TEST(TestLayerRevisionAndSpecTimestamp) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString id = "my_layer";
    const TString downloadHash = "my_layer_download_hash";

    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, downloadHash, 1));
    API::TLayerStatus status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL_C(status.spec_timestamp(), 1, status.spec_timestamp());
    UNIT_ASSERT_EQUAL_C(status.revision(), 2, status.revision());

    holder->UpdateObjectSpecTimestamp(id, 3);
    holder->UpdateObjectRevision(id, 4);
    status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL_C(status.spec_timestamp(), 3, status.spec_timestamp());
    UNIT_ASSERT_EQUAL_C(status.revision(), 4, status.revision());
}

Y_UNIT_TEST(TestUpdateLayerState) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    const TString id = "my_layer";
    const TString downloadHash = "my_layer_download_hash";
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, downloadHash));
    holder->UpdateObjectState(downloadHash, API::ELayerState_READY);
    API::TLayerStatus status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL(API::ELayerState_READY, status.state());
}

Y_UNIT_TEST(TestUpdateLayerStateAndReadyStatus) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString id = "my_layer";
    const TString downloadHash = "my_layer_download_hash";
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, downloadHash));

    holder->UpdateObjectState(downloadHash, API::ELayerState_UNKNOWN);
    API::TLayerStatus status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::ELayerState_UNKNOWN, status.state());
    auto ready = status.ready();
    UNIT_ASSERT_EQUAL(ready.last_transition_time().seconds(), 0);
    UNIT_ASSERT_EQUAL(ready.last_transition_time().nanos(), 0);
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_UNKNOWN);
    UNIT_ASSERT(!holder->NeedLongTickPeriod(downloadHash, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(downloadHash, API::ELayerState_INVALID);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::ELayerState_INVALID, status.state());
    ready = status.ready();
    UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT(holder->NeedLongTickPeriod(downloadHash, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(downloadHash, API::ELayerState_REMOVED);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::ELayerState_REMOVED, status.state());
    ready = status.ready();
    UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT(!holder->NeedLongTickPeriod(downloadHash, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(downloadHash, API::ELayerState_READY);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::ELayerState_READY, status.state());
    ready = status.ready();
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT_EQUAL(NSupport::ExtractState(API::ELayerState_Name(API::ELayerState_READY)), ready.reason());
    UNIT_ASSERT_EQUAL("", ready.message());
    UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
    UNIT_ASSERT(holder->NeedLongTickPeriod(downloadHash, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateSpecTimestamp(TInstant::Now());
    updateHolder->SetLayerTarget(NObjectTargetTestLib::CreateLayerTargetSimple(id, downloadHash));
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::ELayerState_READY, status.state());
    ready  = status.ready();
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT_EQUAL(NSupport::APPLYING_OBJECT_SPEC, ready.reason());
    UNIT_ASSERT_EQUAL(NSupport::APPLYING_OBJECT_SPEC_MESSAGE, ready.message());
    UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
    UNIT_ASSERT(holder->NeedLongTickPeriod(downloadHash, updateHolder->GetUpdateHolderTarget()));

    status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL(API::ELayerState_READY, status.state());
    ready  = status.ready();
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT_EQUAL(NSupport::ExtractState(API::ELayerState_Name(API::ELayerState_READY)), ready.reason());
    UNIT_ASSERT_EQUAL("", ready.message());
    UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
}

Y_UNIT_TEST(TestUpdateLayerStateAndInProgressStatus) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString id = "my_layer";
    const TString downloadHash = "my_layer_download_hash";
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, downloadHash));

    holder->UpdateObjectState(downloadHash, API::ELayerState_UNKNOWN);
    API::TLayerStatus status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::ELayerState_UNKNOWN, status.state());
    auto inProgress = status.in_progress();
    UNIT_ASSERT_EQUAL(inProgress.last_transition_time().seconds(), 0);
    UNIT_ASSERT_EQUAL(inProgress.last_transition_time().nanos(), 0);
    UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_UNKNOWN);

    holder->UpdateObjectState(downloadHash, API::ELayerState_READY);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::ELayerState_READY, status.state());
    inProgress = status.in_progress();
    UNIT_ASSERT(inProgress.last_transition_time().seconds() != 0 || inProgress.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_FALSE);

    holder->UpdateObjectState(downloadHash, API::ELayerState_DOWNLOADING);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::ELayerState_DOWNLOADING, status.state());
    inProgress = status.in_progress();
    UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT_EQUAL(NSupport::ExtractState(API::ELayerState_Name(API::ELayerState_DOWNLOADING)), inProgress.reason());
    UNIT_ASSERT_EQUAL("", inProgress.message());
    UNIT_ASSERT(inProgress.last_transition_time().seconds() != 0 || inProgress.last_transition_time().nanos() != 0);

    holder->UpdateObjectState(downloadHash, API::ELayerState_READY);
    holder->UpdateSpecTimestamp(TInstant::Now());
    updateHolder->SetLayerTarget(NObjectTargetTestLib::CreateLayerTargetSimple(id, downloadHash));
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::ELayerState_READY, status.state());
    inProgress = status.in_progress();
    UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT_EQUAL(NSupport::APPLYING_OBJECT_SPEC, inProgress.reason());
    UNIT_ASSERT_EQUAL(NSupport::APPLYING_OBJECT_SPEC_MESSAGE, inProgress.message());
    UNIT_ASSERT(inProgress.last_transition_time().seconds() != 0 || inProgress.last_transition_time().nanos() != 0);

    status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL(API::ELayerState_READY, status.state());
    inProgress = status.in_progress();
    UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT_EQUAL(NSupport::ExtractState(API::ELayerState_Name(API::ELayerState_READY)), inProgress.reason());
    UNIT_ASSERT_EQUAL("", inProgress.message());
    UNIT_ASSERT(inProgress.last_transition_time().seconds() != 0 || inProgress.last_transition_time().nanos() != 0);
}

Y_UNIT_TEST(TestUpdateLayerStateAndFailedStatus) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString id = "my_layer";
    const TString downloadHash = "my_layer_download_hash";
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, downloadHash));

    holder->UpdateObjectState(downloadHash, API::ELayerState_UNKNOWN);
    API::TLayerStatus status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::ELayerState_UNKNOWN, status.state());
    auto failed = status.failed();
    UNIT_ASSERT_EQUAL(failed.last_transition_time().seconds(), 0);
    UNIT_ASSERT_EQUAL(failed.last_transition_time().nanos(), 0);
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_UNKNOWN);
    UNIT_ASSERT(!holder->NeedLongTickPeriod(downloadHash, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(downloadHash, API::ELayerState_REMOVED);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::ELayerState_REMOVED, status.state());
    failed = status.failed();
    UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT(!holder->NeedLongTickPeriod(downloadHash, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(downloadHash, API::ELayerState_READY);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::ELayerState_READY, status.state());
    failed = status.failed();
    UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT(holder->NeedLongTickPeriod(downloadHash, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(downloadHash, API::ELayerState_INVALID);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::ELayerState_INVALID, status.state());
    failed = status.failed();
    UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT(holder->NeedLongTickPeriod(downloadHash, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(downloadHash, API::ELayerState_REMOVED);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::ELayerState_REMOVED, status.state());
    failed = status.failed();
    UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT(!holder->NeedLongTickPeriod(downloadHash, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(downloadHash, API::ELayerState_READY);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::ELayerState_READY, status.state());
    failed = status.failed();
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT_EQUAL(NSupport::ExtractState(API::ELayerState_Name(API::ELayerState_READY)), failed.reason());
    UNIT_ASSERT_EQUAL("", failed.message());
    UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);
    UNIT_ASSERT(holder->NeedLongTickPeriod(downloadHash, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateSpecTimestamp(TInstant::Now());
    updateHolder->SetLayerTarget(NObjectTargetTestLib::CreateLayerTargetSimple(id, downloadHash));
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::ELayerState_READY, status.state());
    failed  = status.failed();
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT_EQUAL(NSupport::ExtractState(API::ELayerState_Name(API::ELayerState_READY)), failed.reason());
    UNIT_ASSERT_EQUAL("", failed.message());
    UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);
    UNIT_ASSERT(holder->NeedLongTickPeriod(downloadHash, updateHolder->GetUpdateHolderTarget()));

    status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL(API::ELayerState_READY, status.state());
    failed  = status.failed();
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT_EQUAL(NSupport::ExtractState(API::ELayerState_Name(API::ELayerState_READY)), failed.reason());
    UNIT_ASSERT_EQUAL("", failed.message());
    UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);
}

Y_UNIT_TEST(TestUpdateLayerFailedMessage) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    const TString id = "my_layer";
    const TString downloadHash = "my_layer_download_hash";
    const TString message = "message-reason";
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, downloadHash));
    holder->UpdateObjectFailedMessage(downloadHash, message);
    API::TLayerStatus status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL(message, status.failed().message());
}

Y_UNIT_TEST(TestUpdateLayerVerifyAttempt) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    const TString id = "my_layer";
    const TString downloadHash = "my_layer_download_hash";
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, downloadHash));
    holder->UpdateObjectVerifyAttempt(downloadHash);
    API::TLayerStatus status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL(1, status.verification_attempts_counter());
}

Y_UNIT_TEST(TestUpdateLayerDownloadAttempt) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    const TString id = "my_layer";
    const TString downloadHash = "my_layer_download_hash";
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, downloadHash));
    holder->UpdateObjectDownloadAttempt(downloadHash);
    API::TLayerStatus status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL(1, status.download_attempts_counter());
}

Y_UNIT_TEST(TestUpdateLayerDownloadAttemptMultiThread) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    const TString id = "my_layer";
    const TString downloadHash = "my_layer_download_hash";
    const size_t threads = 8;
    const size_t cnt = 100000;
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, downloadHash));
    TThreadPool queue;
    queue.Start(threads);
    for (size_t i = 0; i < threads; ++i) {
        queue.SafeAddFunc([holder, id, downloadHash]() {
            for (size_t i = 0; i < cnt; ++i) {
                holder->UpdateObjectDownloadAttempt(downloadHash);
            }
        });
    }
    queue.Stop();
    API::TLayerStatus status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL_C(threads * cnt, status.download_attempts_counter(), "expected " << threads * cnt << " got " << status.download_attempts_counter());
}

Y_UNIT_TEST(TestUpdateLayerNotFound) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    const TString id = "my_layer";
    const TString downloadHash = "my_layer_download_hash";
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, downloadHash));
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->UpdateObjectDownloadAttempt(downloadHash + "_other"), yexception, "not found at TLayerStatusRepository");
}

Y_UNIT_TEST(TestrGetLayerIds) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    const TString layerId = "my_layer";
    const TString layerDownloadHash = "my_layer_download_hash";
    const TString cacheLayerId = "my_cache_layer";
    const TString cacheLayerDownloadHash = "my_cache_layer_download_hash";
    const ui32 cacheLayerRevision = 1;
    holder->AddCacheObject(NObjectMetaTestLib::CreateCacheLayerMetaSimple(cacheLayerId, cacheLayerDownloadHash, cacheLayerRevision));
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(layerId, layerDownloadHash));

    UNIT_ASSERT_EQUAL(
        holder->GetCacheObjectIdsAndRevisions()
        , TVector<TStatusRepositoryCommon::TCacheObject>(
            {
                TStatusRepositoryCommon::TCacheObject(cacheLayerId, cacheLayerRevision)
            }
        )
    );
    UNIT_ASSERT_EQUAL(
        holder->GetCacheObjectIdsAndRevisionsByHash(cacheLayerDownloadHash)
        , TVector<TStatusRepositoryCommon::TCacheObject>(
            {
                TStatusRepositoryCommon::TCacheObject(cacheLayerId, cacheLayerRevision)
            }
        )
    );
    UNIT_ASSERT_EQUAL(
        holder->GetCacheObjectIdsAndRevisionsByHash(cacheLayerDownloadHash + "_other")
        , TVector<TStatusRepositoryCommon::TCacheObject>()
    );

    UNIT_ASSERT_EQUAL(
        holder->GetObjectIds()
        , TVector<TString>(
            {
                layerId
            }
        )
    );
    UNIT_ASSERT_EQUAL(
        holder->GetObjectIdsByHash(layerDownloadHash)
        , TVector<TString>(
            {
                layerId
            }
        )
    );
    UNIT_ASSERT_EQUAL(
        holder->GetObjectIdsByHash(layerDownloadHash + "_other")
        , TVector<TString>()
    );
}

Y_UNIT_TEST(TestLayerFailCounter) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    const TString id = "my_layer";
    const TString downloadHash = "my_layer_download_hash";
    const size_t cntIter = 10;

    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, downloadHash));
    for (size_t i = 0; i < cntIter; ++i) {
        holder->IncrementObjectFailCounter(downloadHash);
    }

    UNIT_ASSERT_EQUAL(holder->GetObjectStatus(id).fail_counter(), cntIter);
}

Y_UNIT_TEST(TestUpdateLayerDownloadProgress) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    const TString id = "my_layer";
    const TString downloadHash = "my_layer_download_hash";
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, downloadHash));

    holder->UpdateObjectDownloadProgress(downloadHash, 57);

    auto status = holder->GetObjectStatus(id, nullptr);
    UNIT_ASSERT_EQUAL_C(status.download_progress().percent(), 57, status.download_progress().percent());
}

Y_UNIT_TEST(TestLayerContainer) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    const TString id = "my_layer";
    const TString downloadHash = "my_layer_download_hash";
    const NStatusRepositoryTypes::TContainerDescription container(
        downloadHash
        , NStatusRepositoryTypes::EObjectType::LAYER
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::DOWNLOAD
    );

    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, downloadHash));
    holder->UpdateContainerFailReason(container, "Some fail reason");
    holder->IncrementContainerSystemFailureCounter(container);

    auto layerStatus = holder->GetObjectStatus(id);
    UNIT_ASSERT_STRING_CONTAINS(layerStatus.failed().message(), "Some fail reason");
    UNIT_ASSERT_STRING_CONTAINS(layerStatus.failed().message(), ToString(NStatusRepositoryTypes::TContainerDescription::EContainerType::DOWNLOAD));
    UNIT_ASSERT_EQUAL_C(layerStatus.fail_counter(), 1, layerStatus.fail_counter());

    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->IncrementContainerPositiveReturnCodeCounter(container), yexception, "not implemented");
    UNIT_ASSERT_EXCEPTION_CONTAINS(
        holder->UpdateContainerFailReason(
            NStatusRepositoryTypes::TContainerDescription(
                downloadHash
                , NStatusRepositoryTypes::EObjectType::WORKLOAD
                , NStatusRepositoryTypes::TContainerDescription::EContainerType::DOWNLOAD
            )
            , "Some fail reason"
        )
        , yexception
        , "Expected container object type"
    );
}

Y_UNIT_TEST(TestLayerHashIndependence) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    const TString layer1 = "my_layer1";
    const TString layer2 = "my_layer2";
    const TString hash1 = "my_layer1_download_hash";
    const TString hash2 = "my_layer2_download_hash";

    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(layer1, hash1, 3, false));
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(layer2, hash2, 5, false));

    UNIT_ASSERT(holder->HasObject(layer1));
    UNIT_ASSERT(holder->HasObject(layer2));
    UNIT_ASSERT(holder->HasObjectHash(hash1));
    UNIT_ASSERT(holder->HasObjectHash(hash2));
    UNIT_ASSERT_EQUAL(holder->GetObjectHashKeepSourceFileAfterImportCounter(hash1), 1);
    UNIT_ASSERT_EQUAL(holder->GetObjectHashKeepSourceFileAfterImportCounter(hash2), 1);
    UNIT_ASSERT_EQUAL_C(holder->GetObjectHash(layer1), hash1, holder->GetObjectHash(layer1));
    UNIT_ASSERT_EQUAL_C(holder->GetObjectHash(layer2), hash2, holder->GetObjectHash(layer2));

    holder->UpdateObjectFailedMessage(hash1, hash1);
    holder->UpdateObjectFailedMessage(hash2, hash2);

    auto status1 = holder->GetObjectStatus(layer1);
    UNIT_ASSERT_EQUAL_C(status1.id(), layer1, status1.id());
    UNIT_ASSERT_EQUAL_C(status1.spec_timestamp(), 3, status1.spec_timestamp());
    UNIT_ASSERT_EQUAL_C(status1.revision(), 4, status1.revision());
    UNIT_ASSERT_EQUAL_C(status1.failed().message(), hash1, status1.failed().message());

    auto status2 = holder->GetObjectStatus(layer2);
    UNIT_ASSERT_EQUAL_C(status2.id(), layer2, status2.id());
    UNIT_ASSERT_EQUAL_C(status2.spec_timestamp(), 5, status2.spec_timestamp());
    UNIT_ASSERT_EQUAL_C(status2.revision(), 6, status2.revision());
    UNIT_ASSERT_EQUAL_C(status2.failed().message(), hash2, status2.failed().message());

    holder->RemoveObject(layer1);
    UNIT_ASSERT(!holder->HasObject(layer1));
    UNIT_ASSERT(holder->HasObject(layer2));
    UNIT_ASSERT(!holder->HasObjectHash(hash1));
    UNIT_ASSERT(holder->HasObjectHash(hash2));
    UNIT_ASSERT_EQUAL(holder->GetObjectHashKeepSourceFileAfterImportCounter(hash2), 1);

    holder->RemoveObject(layer2);
    UNIT_ASSERT(!holder->HasObject(layer1));
    UNIT_ASSERT(!holder->HasObject(layer2));
    UNIT_ASSERT(!holder->HasObjectHash(hash1));
    UNIT_ASSERT(!holder->HasObjectHash(hash2));
}

Y_UNIT_TEST(TestLayerHashDependence) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString layer1 = "my_layer1";
    const TString layer2 = "my_layer2";
    const TString hash = "my_layer_download_hash";

    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(layer1, hash, 3, false));
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(layer2, hash, 5, false));
    updateHolder->SetLayerTarget(NObjectTargetTestLib::CreateLayerTargetSimple(layer1, hash));

    UNIT_ASSERT(holder->HasObject(layer1));
    UNIT_ASSERT(holder->HasObject(layer2));
    UNIT_ASSERT(holder->HasObjectHash(hash));
    UNIT_ASSERT_EQUAL(holder->GetObjectHashKeepSourceFileAfterImportCounter(hash), 2);
    UNIT_ASSERT_EQUAL_C(holder->GetObjectHash(layer1), hash, holder->GetObjectHash(layer1));
    UNIT_ASSERT_EQUAL_C(holder->GetObjectHash(layer2), hash, holder->GetObjectHash(layer2));

    holder->UpdateObjectFailedMessage(hash, hash);

    auto status1 = holder->GetObjectStatus(layer1, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL_C(status1.id(), layer1, status1.id());
    UNIT_ASSERT_EQUAL_C(status1.spec_timestamp(), 3, status1.spec_timestamp());
    UNIT_ASSERT_EQUAL_C(status1.revision(), 4, status1.revision());
    UNIT_ASSERT_EQUAL_C(status1.failed().message(), hash, status1.failed().message());
    UNIT_ASSERT_EQUAL_C(status1.ready().message(), NSupport::APPLYING_OBJECT_SPEC_MESSAGE, status1.ready().message());

    auto status2 = holder->GetObjectStatus(layer2, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL_C(status2.id(), layer2, status2.id());
    UNIT_ASSERT_EQUAL_C(status2.spec_timestamp(), 5, status2.spec_timestamp());
    UNIT_ASSERT_EQUAL_C(status2.revision(), 6, status2.revision());
    UNIT_ASSERT_EQUAL_C(status2.failed().message(), hash, status2.failed().message());
    UNIT_ASSERT_EQUAL_C(status2.ready().message(), "", status2.ready().message());

    auto hashStatus = holder->GetObjectHashStatus(hash);
    UNIT_ASSERT_EQUAL_C(hashStatus.id(), "", hashStatus.id());
    UNIT_ASSERT_EQUAL_C(hashStatus.spec_timestamp(), 0, hashStatus.spec_timestamp());
    UNIT_ASSERT_EQUAL_C(hashStatus.revision(), 0, hashStatus.revision());
    UNIT_ASSERT_EQUAL_C(hashStatus.failed().message(), hash, hashStatus.failed().message());
    UNIT_ASSERT_EQUAL_C(hashStatus.ready().message(), "", hashStatus.ready().message());

    holder->RemoveObject(layer1);
    UNIT_ASSERT(!holder->HasObject(layer1));
    UNIT_ASSERT(holder->HasObject(layer2));
    UNIT_ASSERT(holder->HasObjectHash(hash));
    UNIT_ASSERT_EQUAL(holder->GetObjectHashKeepSourceFileAfterImportCounter(hash), 1);

    holder->RemoveObject(layer2);
    UNIT_ASSERT(!holder->HasObject(layer1));
    UNIT_ASSERT(!holder->HasObject(layer2));
    UNIT_ASSERT(!holder->HasObjectHash(hash));
}

Y_UNIT_TEST(TestAddCacheLayer) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString id = "my_cache_layer";
    const ui32 revision = 3;
    const TString downloadHash = "my_layer_download_hash";
    holder->AddCacheObject(NObjectMetaTestLib::CreateCacheLayerMetaSimple(id, downloadHash, revision));

    UNIT_ASSERT(holder->HasCacheObject(id, revision));
    UNIT_ASSERT(!holder->HasCacheObject(id, revision + 1));
    UNIT_ASSERT(holder->HasObjectHash(downloadHash));
    UNIT_ASSERT_EQUAL_C(holder->GetCacheObjectHash(id, revision), downloadHash, holder->GetCacheObjectHash(id, revision));

    API::TLayerStatus status = holder->GetCacheObjectStatus(id, revision);
    UNIT_ASSERT_EQUAL_C(status.id(), id, status.id());
    UNIT_ASSERT_EQUAL_C(status.spec_timestamp(), 0, status.spec_timestamp());
    UNIT_ASSERT_EQUAL_C(status.revision(), revision, status.revision());
    UNIT_ASSERT_EQUAL(API::ELayerState_UNKNOWN, status.state());
}

Y_UNIT_TEST(TestCacheLayerPatchTotalStatus) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString id = "my_cache_layer";
    const ui32 revision = 3;
    const TString downloadHash = "my_layer_download_hash";

    holder->AddCacheObject(NObjectMetaTestLib::CreateCacheLayerMetaSimple(id, downloadHash, revision));
    // Changing something in the status so that it is not equal to the default one
    holder->UpdateObjectState(downloadHash, API::ELayerState_READY);
    API::TLayerStatus status = holder->GetCacheObjectStatus(id, revision);

    {
        API::TPodAgentStatus patchStatus;
        holder->PatchTotalStatus(patchStatus, updateHolder->GetUpdateHolderTarget(), false);
        UNIT_ASSERT_EQUAL_C(
            patchStatus.resource_cache().layers(0).id()
            , id
            , patchStatus.resource_cache().layers(0).id()
        );
        UNIT_ASSERT_EQUAL_C(
            patchStatus.resource_cache().layers(0).revision()
            , revision
            , patchStatus.resource_cache().layers(0).revision()
        );
        UNIT_ASSERT_EQUAL_C(
            patchStatus.resource_cache().layers(0).state()
            , API::ELayerState_READY
            , API::ELayerState_Name(patchStatus.resource_cache().layers(0).state())
        );
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(status, patchStatus.resource_cache().layers(0)));
    }

    {
        API::TPodAgentStatus patchStatus;
        holder->PatchTotalStatus(patchStatus, updateHolder->GetUpdateHolderTarget(), true);
        UNIT_ASSERT_EQUAL_C(
            patchStatus.resource_cache().layers(0).id()
            , id
            , patchStatus.resource_cache().layers(0).id()
        );
        UNIT_ASSERT_EQUAL_C(
            patchStatus.resource_cache().layers(0).revision()
            , revision
            , patchStatus.resource_cache().layers(0).revision()
        );
        UNIT_ASSERT_EQUAL_C(
            patchStatus.resource_cache().layers(0).state()
            // conditionsOnly = true
            , API::ELayerState_UNKNOWN
            , API::ELayerState_Name(patchStatus.resource_cache().layers(0).state())
        );
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(status.ready(), patchStatus.resource_cache().layers(0).ready()));
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(status.in_progress(), patchStatus.resource_cache().layers(0).in_progress()));
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(status.failed(), patchStatus.resource_cache().layers(0).failed()));
    }
}

Y_UNIT_TEST(TestCacheLayerNotFound) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    const TString id = "my_cache_ayer";
    const ui32 revision = 3;
    const TString downloadHash = "my_layer_download_hash";
    holder->AddCacheObject(NObjectMetaTestLib::CreateCacheLayerMetaSimple(id, downloadHash, revision));
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->GetCacheObjectStatus(id + "_other", revision), yexception, "not found");
}

Y_UNIT_TEST(TestAddCacheLayerWithSameId) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    const TString id = "my_cache_layer";
    const ui32 revision1 = 4;
    const ui32 revision2 = 6;
    const TString downloadHash1 = "my_layer_download_hash1";
    const TString downloadHash2 = "my_layer_download_hash2";
    holder->AddCacheObject(NObjectMetaTestLib::CreateCacheLayerMetaSimple(id, downloadHash1, revision1));
    holder->AddCacheObject(NObjectMetaTestLib::CreateCacheLayerMetaSimple(id, downloadHash2, revision2));

    UNIT_ASSERT(holder->HasCacheObject(id, revision1));
    UNIT_ASSERT(holder->HasCacheObject(id, revision2));
    UNIT_ASSERT(holder->HasObjectHash(downloadHash1));
    UNIT_ASSERT(holder->HasObjectHash(downloadHash2));
    UNIT_ASSERT_EQUAL_C(holder->GetCacheObjectHash(id, revision1), downloadHash1, holder->GetCacheObjectHash(id, revision1));
    UNIT_ASSERT_EQUAL_C(holder->GetCacheObjectHash(id, revision2), downloadHash2, holder->GetCacheObjectHash(id, revision2));
    UNIT_ASSERT_EQUAL(
        holder->GetCacheObjectIdsAndRevisions()
        , TVector<TStatusRepositoryCommon::TCacheObject>({TStatusRepositoryCommon::TCacheObject(id, revision1), TStatusRepositoryCommon::TCacheObject(id, revision2)})
    );
}

Y_UNIT_TEST(TestAddCacheLayerWithSameRevision) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    const TString id1 = "my_cache_layer1";
    const TString id2 = "my_cache_layer2";
    const ui32 revision = 4;
    const TString downloadHash1 = "my_layer_download_hash1";
    const TString downloadHash2 = "my_layer_download_hash2";
    holder->AddCacheObject(NObjectMetaTestLib::CreateCacheLayerMetaSimple(id1, downloadHash1, revision));
    holder->AddCacheObject(NObjectMetaTestLib::CreateCacheLayerMetaSimple(id2, downloadHash2, revision));

    UNIT_ASSERT(holder->HasCacheObject(id1, revision));
    UNIT_ASSERT(holder->HasCacheObject(id2, revision));
    UNIT_ASSERT(holder->HasObjectHash(downloadHash1));
    UNIT_ASSERT(holder->HasObjectHash(downloadHash2));
    UNIT_ASSERT_EQUAL_C(holder->GetCacheObjectHash(id1, revision), downloadHash1, holder->GetCacheObjectHash(id1, revision));
    UNIT_ASSERT_EQUAL_C(holder->GetCacheObjectHash(id2, revision), downloadHash2, holder->GetCacheObjectHash(id2, revision));
    UNIT_ASSERT_EQUAL(holder->GetCacheObjectIdsAndRevisions()
        , TVector<TStatusRepositoryCommon::TCacheObject>({TStatusRepositoryCommon::TCacheObject(id1, revision), TStatusRepositoryCommon::TCacheObject(id2, revision)}));
}

Y_UNIT_TEST(TestCacheLayerHashIndependence) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    const TString id1 = "my_cache_layer";
    const TString id2 = "my_cache_layer";
    const ui32 revision1 = 3;
    const ui32 revision2 = 6;
    const TString downloadHash1 = "my_layer_download_hash1";
    const TString downloadHash2 = "my_layer_download_hash2";

    holder->AddCacheObject(NObjectMetaTestLib::CreateCacheLayerMetaSimple(id1, downloadHash1, revision1, 1, false));
    holder->AddCacheObject(NObjectMetaTestLib::CreateCacheLayerMetaSimple(id2, downloadHash2, revision2, 2, false));

    UNIT_ASSERT(holder->HasCacheObject(id1, revision1));
    UNIT_ASSERT(holder->HasCacheObject(id2, revision2));
    UNIT_ASSERT(holder->HasObjectHash(downloadHash1));
    UNIT_ASSERT(holder->HasObjectHash(downloadHash2));
    UNIT_ASSERT_EQUAL(holder->GetObjectHashKeepSourceFileAfterImportCounter(downloadHash1), 1);
    UNIT_ASSERT_EQUAL(holder->GetObjectHashKeepSourceFileAfterImportCounter(downloadHash2), 1);
    UNIT_ASSERT_EQUAL_C(holder->GetCacheObjectHash(id1, revision1), downloadHash1, holder->GetCacheObjectHash(id1, revision1));
    UNIT_ASSERT_EQUAL_C(holder->GetCacheObjectHash(id2, revision2), downloadHash2, holder->GetCacheObjectHash(id2, revision2));

    holder->UpdateObjectFailedMessage(downloadHash1, downloadHash1);
    holder->UpdateObjectFailedMessage(downloadHash2, downloadHash2);

    auto status1 = holder->GetCacheObjectStatus(id1, revision1);
    UNIT_ASSERT_EQUAL_C(status1.id(), id1, status1.id());
    UNIT_ASSERT_EQUAL_C(status1.spec_timestamp(), 0, status1.spec_timestamp());
    UNIT_ASSERT_EQUAL_C(status1.revision(), revision1, status1.revision());
    UNIT_ASSERT_EQUAL_C(status1.failed().message(), downloadHash1, status1.failed().message());

    auto status2 = holder->GetCacheObjectStatus(id2, revision2);
    UNIT_ASSERT_EQUAL_C(status2.id(), id2, status2.id());
    UNIT_ASSERT_EQUAL_C(status2.spec_timestamp(), 0, status2.spec_timestamp());
    UNIT_ASSERT_EQUAL_C(status2.revision(), revision2, status2.revision());
    UNIT_ASSERT_EQUAL_C(status2.failed().message(), downloadHash2, status2.failed().message());

    holder->RemoveCacheObject(id1, revision1);
    UNIT_ASSERT(!holder->HasCacheObject(id1, revision1));
    UNIT_ASSERT(holder->HasCacheObject(id2, revision2));
    UNIT_ASSERT(!holder->HasObjectHash(downloadHash1));
    UNIT_ASSERT(holder->HasObjectHash(downloadHash2));
    UNIT_ASSERT_EQUAL(holder->GetObjectHashKeepSourceFileAfterImportCounter(downloadHash2), 1);

    holder->RemoveCacheObject(id2, revision2);
    UNIT_ASSERT(!holder->HasCacheObject(id1, revision1));
    UNIT_ASSERT(!holder->HasCacheObject(id2, revision2));
    UNIT_ASSERT(!holder->HasObjectHash(downloadHash1));
    UNIT_ASSERT(!holder->HasObjectHash(downloadHash2));
}

Y_UNIT_TEST(TestCacheLayerHashDependence) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    const TString id1 = "my_cache_layer";
    const TString id2 = "my_cache_layer";
    const ui32 revision1 = 3;
    const ui32 revision2 = 6;
    const TString hash = "my_layer_download_hash";

    holder->AddCacheObject(NObjectMetaTestLib::CreateCacheLayerMetaSimple(id1, hash, revision1, 1, false));
    holder->AddCacheObject(NObjectMetaTestLib::CreateCacheLayerMetaSimple(id2, hash, revision2, 1, false));

    UNIT_ASSERT(holder->HasCacheObject(id1, revision1));
    UNIT_ASSERT(holder->HasCacheObject(id2, revision2));
    UNIT_ASSERT(holder->HasObjectHash(hash));
    UNIT_ASSERT_EQUAL(holder->GetObjectHashKeepSourceFileAfterImportCounter(hash), 2);
    UNIT_ASSERT_EQUAL_C(holder->GetCacheObjectHash(id1, revision1), hash, holder->GetCacheObjectHash(id1, revision1));
    UNIT_ASSERT_EQUAL_C(holder->GetCacheObjectHash(id2, revision2), hash, holder->GetCacheObjectHash(id2, revision2));

    holder->UpdateObjectFailedMessage(hash, hash);

    auto status1 = holder->GetCacheObjectStatus(id1, revision1);
    UNIT_ASSERT_EQUAL_C(status1.id(), id1, status1.id());
    UNIT_ASSERT_EQUAL_C(status1.spec_timestamp(), 0, status1.spec_timestamp());
    UNIT_ASSERT_EQUAL_C(status1.revision(), revision1, status1.revision());
    UNIT_ASSERT_EQUAL_C(status1.failed().message(), hash, status1.failed().message());
    UNIT_ASSERT_EQUAL_C(status1.ready().message(), "", status1.ready().message());

    auto status2 = holder->GetCacheObjectStatus(id2, revision2);
    UNIT_ASSERT_EQUAL_C(status2.id(), id2, status2.id());
    UNIT_ASSERT_EQUAL_C(status2.spec_timestamp(), 0, status2.spec_timestamp());
    UNIT_ASSERT_EQUAL_C(status2.revision(), revision2, status2.revision());
    UNIT_ASSERT_EQUAL_C(status2.failed().message(), hash, status2.failed().message());
    UNIT_ASSERT_EQUAL_C(status2.ready().message(), "", status2.ready().message());

    auto hashStatus = holder->GetObjectHashStatus(hash);
    UNIT_ASSERT_EQUAL_C(hashStatus.id(), "", hashStatus.id());
    UNIT_ASSERT_EQUAL_C(hashStatus.spec_timestamp(), 0, hashStatus.spec_timestamp());
    UNIT_ASSERT_EQUAL_C(hashStatus.revision(), 0, hashStatus.revision());
    UNIT_ASSERT_EQUAL_C(hashStatus.failed().message(), hash, hashStatus.failed().message());
    UNIT_ASSERT_EQUAL_C(hashStatus.ready().message(), "", hashStatus.ready().message());

    holder->RemoveCacheObject(id1, revision1);
    UNIT_ASSERT(!holder->HasCacheObject(id1, revision1));
    UNIT_ASSERT(holder->HasCacheObject(id2, revision2));
    UNIT_ASSERT(holder->HasObjectHash(hash));
    UNIT_ASSERT_EQUAL(holder->GetObjectHashKeepSourceFileAfterImportCounter(hash), 1);

    holder->RemoveCacheObject(id2, revision2);
    UNIT_ASSERT(!holder->HasCacheObject(id1, revision1));
    UNIT_ASSERT(!holder->HasCacheObject(id2, revision2));
    UNIT_ASSERT(!holder->HasObjectHash(hash));
}

Y_UNIT_TEST(TestCacheLayerAndLayerHashDependence) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString id1 = "my_cache_layer";
    const TString id2 = "my_cache_layer";
    const ui32 specTimestamp1 = 3;
    const ui32 revision1 = specTimestamp1 + 1;
    const ui32 revision2 = 6;
    const TString hash = "my_layer_download_hash";

    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id1, hash, specTimestamp1, false));
    holder->AddCacheObject(NObjectMetaTestLib::CreateCacheLayerMetaSimple(id2, hash, revision2, 1, false));
    updateHolder->SetLayerTarget(NObjectTargetTestLib::CreateLayerTargetSimple(id1, hash));

    UNIT_ASSERT(holder->HasObject(id1));
    UNIT_ASSERT(holder->HasCacheObject(id2, revision2));
    UNIT_ASSERT(holder->HasObjectHash(hash));
    UNIT_ASSERT_EQUAL(holder->GetObjectHashKeepSourceFileAfterImportCounter(hash), 2);
    UNIT_ASSERT_EQUAL_C(holder->GetObjectHash(id1), hash, holder->GetObjectHash(id1));
    UNIT_ASSERT_EQUAL_C(holder->GetCacheObjectHash(id2, revision2), hash, holder->GetCacheObjectHash(id2, revision2));

    holder->UpdateObjectFailedMessage(hash, hash);

    auto status1 = holder->GetObjectStatus(id1, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL_C(status1.id(), id1, status1.id());
    UNIT_ASSERT_EQUAL_C(status1.spec_timestamp(), specTimestamp1, status1.spec_timestamp());
    UNIT_ASSERT_EQUAL_C(status1.revision(), revision1, status1.revision());
    UNIT_ASSERT_EQUAL_C(status1.failed().message(), hash, status1.failed().message());
    UNIT_ASSERT_EQUAL_C(status1.ready().message(), NSupport::APPLYING_OBJECT_SPEC_MESSAGE, status1.ready().message());

    auto status2 = holder->GetCacheObjectStatus(id2, revision2);
    UNIT_ASSERT_EQUAL_C(status2.id(), id2, status2.id());
    UNIT_ASSERT_EQUAL_C(status2.spec_timestamp(), 0, status2.spec_timestamp());
    UNIT_ASSERT_EQUAL_C(status2.revision(), revision2, status2.revision());
    UNIT_ASSERT_EQUAL_C(status2.failed().message(), hash, status2.failed().message());
    UNIT_ASSERT_EQUAL_C(status2.ready().message(), "", status2.ready().message());

    auto hashStatus = holder->GetObjectHashStatus(hash);
    UNIT_ASSERT_EQUAL_C(hashStatus.id(), "", hashStatus.id());
    UNIT_ASSERT_EQUAL_C(hashStatus.spec_timestamp(), 0, hashStatus.spec_timestamp());
    UNIT_ASSERT_EQUAL_C(hashStatus.revision(), 0, hashStatus.revision());
    UNIT_ASSERT_EQUAL_C(hashStatus.failed().message(), hash, hashStatus.failed().message());
    UNIT_ASSERT_EQUAL_C(hashStatus.ready().message(), "", hashStatus.ready().message());

    holder->RemoveObject(id1);
    UNIT_ASSERT(!holder->HasObject(id1));
    UNIT_ASSERT(holder->HasCacheObject(id2, revision2));
    UNIT_ASSERT(holder->HasObjectHash(hash));
    UNIT_ASSERT_EQUAL(holder->GetObjectHashKeepSourceFileAfterImportCounter(hash), 1);

    holder->RemoveCacheObject(id2, revision2);
    UNIT_ASSERT(!holder->HasObject(id1));
    UNIT_ASSERT(!holder->HasCacheObject(id2, revision2));
    UNIT_ASSERT(!holder->HasObjectHash(hash));

    // Other remove order
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id1, hash, specTimestamp1, false));
    holder->AddCacheObject(NObjectMetaTestLib::CreateCacheLayerMetaSimple(id2, hash, revision2, 1, false));

    UNIT_ASSERT(holder->HasObject(id1));
    UNIT_ASSERT(holder->HasCacheObject(id2, revision2));
    UNIT_ASSERT(holder->HasObjectHash(hash));
    UNIT_ASSERT_EQUAL(holder->GetObjectHashKeepSourceFileAfterImportCounter(hash), 2);
    UNIT_ASSERT_EQUAL_C(holder->GetObjectHash(id1), hash, holder->GetObjectHash(id1));
    UNIT_ASSERT_EQUAL_C(holder->GetCacheObjectHash(id2, revision2), hash, holder->GetCacheObjectHash(id2, revision2));

    holder->RemoveCacheObject(id2, revision2);
    UNIT_ASSERT(holder->HasObject(id1));
    UNIT_ASSERT(!holder->HasCacheObject(id2, revision2));
    UNIT_ASSERT(holder->HasObjectHash(hash));
    UNIT_ASSERT_EQUAL(holder->GetObjectHashKeepSourceFileAfterImportCounter(hash), 1);

    holder->RemoveObject(id1);
    UNIT_ASSERT(!holder->HasObject(id1));
    UNIT_ASSERT(!holder->HasCacheObject(id2, revision2));
    UNIT_ASSERT(!holder->HasObjectHash(hash));
}

Y_UNIT_TEST(TestLayerKeepSourceFileAfterImportCounter) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    const TString layerId = "my_layer";
    const TString cacheLayerId = "my_cache_layer";
    const ui32 revision = 5;
    const TString hash = "my_layer_download_hash";
    const ui32 cntIter = 10;

    for (ui32 i = 1; i <= cntIter; ++i) {
        holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(layerId + "_" + ToString(i), hash, i, i % 2));
        holder->AddCacheObject(NObjectMetaTestLib::CreateCacheLayerMetaSimple(cacheLayerId + "_" + ToString(i), hash, revision, i, (i + 1) % 2));
        UNIT_ASSERT_EQUAL_C(
            holder->GetObjectHashKeepSourceFileAfterImportCounter(hash)
            , i
            , holder->GetObjectHashKeepSourceFileAfterImportCounter(hash)
        );
    }

    for (ui32 i = cntIter; i >= 1; --i) {
        UNIT_ASSERT_EQUAL_C(
            holder->GetObjectHashKeepSourceFileAfterImportCounter(hash)
            , i
            , holder->GetObjectHashKeepSourceFileAfterImportCounter(hash)
        );
        holder->RemoveObject(layerId + "_" + ToString(i));
        holder->RemoveCacheObject(cacheLayerId + "_" + ToString(i), revision);
    }

    const ui32 seed = 1;
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(layerId, hash, seed, true));
    UNIT_ASSERT_EQUAL_C(
        holder->GetObjectHashKeepSourceFileAfterImportCounter(hash)
        , 0
        , holder->GetObjectHashKeepSourceFileAfterImportCounter(hash)
    );
    holder->AddCacheObject(NObjectMetaTestLib::CreateCacheLayerMetaSimple(layerId, hash, revision, seed, false));
    UNIT_ASSERT_EQUAL_C(
        holder->GetObjectHashKeepSourceFileAfterImportCounter(hash)
        , 1
        , holder->GetObjectHashKeepSourceFileAfterImportCounter(hash)
    );
    holder->AddCacheObject(NObjectMetaTestLib::CreateCacheLayerMetaSimple(layerId + "_other", hash, revision, seed, true));
    UNIT_ASSERT_EQUAL_C(
        holder->GetObjectHashKeepSourceFileAfterImportCounter(hash)
        , 1
        , holder->GetObjectHashKeepSourceFileAfterImportCounter(hash)
    );
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(layerId + "_other", hash, seed, false));
    UNIT_ASSERT_EQUAL_C(
        holder->GetObjectHashKeepSourceFileAfterImportCounter(hash)
        , 2
        , holder->GetObjectHashKeepSourceFileAfterImportCounter(hash)
    );
}

}

} // namespace NInfra::NPodAgent::NLayerStatusRepositoryTest
