#include "static_resource_status_repository.h"

#include "support_functions.h"

#include <util/string/builder.h>

namespace NInfra::NPodAgent {

void TStaticResourceStatusRepository::UpdateSpecTimestamp(TInstant currentTime) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    for (auto& [objectHash, info] : Hashes_) {
        TGuard<TMutex> g(GetObjectHashMutex(objectHash));
        RefreshObjectStatusConditions(info.Status_, info.Status_, currentTime);
    }
}

void TStaticResourceStatusRepository::IncrementContainerSystemFailureCounter(const NStatusRepositoryTypes::TContainerDescription& container) {
    CheckContainerObjectType(container);
    IncrementObjectFailCounter(container.ObjectIdOrHash_);
}

void TStaticResourceStatusRepository::UpdateContainerState(const NStatusRepositoryTypes::TContainerDescription& /* container */, API::EContainerState state) {
    Y_ENSURE(state == API::EContainerState_SYSTEM_FAILURE, "you can only call UpdateContainerState with EContainerState_SYSTEM_FAILURE for static resource");
}

void TStaticResourceStatusRepository::PatchTotalStatus(API::TPodAgentStatus& status, TUpdateHolderTargetPtr updateHolderTarget, bool conditionsOnly) const {
    Y_ENSURE(updateHolderTarget, "UpdateHolderTarget not provided for TStaticResourceStatusRepository::PatchTotalStatus");

    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);

    API::TResourceGangStatus* resourceCacheStatus = status.mutable_resource_cache();
    for (const auto& [cacheObject, info]: CacheObjects_) {
        API::TStaticResourceStatus* st = resourceCacheStatus->add_static_resources();
        *st = GetCacheObjectStatusNoGlobalLock(cacheObject.Id_, cacheObject.Revision_, conditionsOnly);
        NSupport::PatchAllNonReadyConditions(status, *st);
    }

    API::TResourceGangStatus* resourceGangStatus = status.mutable_resource_gang();
    for (const auto& [objectId, info] : Objects_) {
        API::TStaticResourceStatus* st = resourceGangStatus->add_static_resources();
        *st = GetObjectStatusNoGlobalLock(objectId, updateHolderTarget, conditionsOnly);
        NSupport::PatchAllNonReadyConditions(status, *st);
    }
}

bool TStaticResourceStatusRepository::NeedLongTickPeriod(const TString& objectIdOrHash, TUpdateHolderTargetPtr updateHolderTarget) const {
    Y_ENSURE(updateHolderTarget, "UpdateHolderTarget not provided for TStaticResourceStatusRepository::NeedLongTickPeriod");

    auto status = GetObjectHashStatus(objectIdOrHash);

    return status.ready().status() == API::EConditionStatus_TRUE
        || status.state() == API::EStaticResourceState_INVALID;
}

void TStaticResourceStatusRepository::UpdateContainerFailReason(const NStatusRepositoryTypes::TContainerDescription& container, const TString& failReason) {
    CheckContainerObjectType(container);
    UpdateObjectFailedMessage(
        container.ObjectIdOrHash_
        , TStringBuilder() << "Object " << ToString(container.ContainerType_) << " container FailReason: " << failReason
    );
}

void TStaticResourceStatusRepository::AddCacheObject(const TStaticResourceMeta& cacheObjectMeta) {
    TWriteGuardBase<TLightRWLock> guard(GlobalObjectLock_);

    TCacheObject cacheObject = TCacheObject(cacheObjectMeta.Id_, cacheObjectMeta.Revision_);

    CacheObjects_.insert(
        {
            cacheObject
            , TCacheObjectInfo(
                cacheObjectMeta.DownloadHash_
                , cacheObjectMeta.CheckPeriodMs_
            )
        }
    );
    THashInfo& objectHashInfo = Hashes_[cacheObjectMeta.DownloadHash_];
    objectHashInfo.CacheObjectIdsAndRevisions_.insert(cacheObject);
    objectHashInfo.CheckPeriodsMs_.insert(cacheObjectMeta.CheckPeriodMs_);
}

void TStaticResourceStatusRepository::RemoveCacheObject(const TString& cacheObjectId, ui32 revision) {
    TWriteGuardBase<TLightRWLock> guard(GlobalObjectLock_);

    TCacheObject cacheObject = TCacheObject(cacheObjectId, revision);
    const TCacheObjectInfo& cacheObjectInfo = CacheObjects_.at(cacheObject);
    const TString& objectHash = cacheObjectInfo.ObjectHash_;

    THashInfo& objectHashInfo = Hashes_.at(objectHash);

    // WARNING: it is TMultiSet
    objectHashInfo.CheckPeriodsMs_.erase(
        objectHashInfo.CheckPeriodsMs_.find(cacheObjectInfo.CheckPeriodMs_)
    );

    objectHashInfo.CacheObjectIdsAndRevisions_.erase(cacheObject);
    // If this is the last reference clear status
    if (objectHashInfo.ObjectIds_.empty() && objectHashInfo.CacheObjectIdsAndRevisions_.empty()) {
        Hashes_.erase(objectHash);
    }

    CacheObjects_.erase(cacheObject);
}

API::TStaticResourceStatus TStaticResourceStatusRepository::GetCacheObjectStatus(const TString& cacheObjectId, ui32 revision) const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    return GetCacheObjectStatusNoGlobalLock(cacheObjectId, revision, false);
}

API::TStaticResourceStatus TStaticResourceStatusRepository::GetCacheObjectStatusNoGlobalLock(
    const TString& cacheObjectId
    , ui32 revision
    , bool conditionsOnly
) const {
    TGuard<TMutex> g(GetCacheObjectMutex(cacheObjectId, revision));

    const TString& objectHash = CacheObjects_.at(TCacheObject(cacheObjectId, revision)).ObjectHash_;
    TGuard<TMutex> gh(GetObjectHashMutex(objectHash));
    // WARNING: Do not copy full status here
    const API::TStaticResourceStatus& fullStatus = Hashes_.at(objectHash).Status_;

    API::TStaticResourceStatus status;
    if (!conditionsOnly) {
        status.CopyFrom(fullStatus);
    }
    // Always add real id and revision to status
    status.set_id(cacheObjectId);
    status.set_revision(revision);

    RefreshObjectStatusConditions(status, fullStatus, TInstant::Zero(), false, false);

    return status;
}

bool TStaticResourceStatusRepository::HasCacheObject(const TString& cacheObjectId, ui32 revision) const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    return CacheObjects_.FindPtr(TCacheObject(cacheObjectId, revision));
}

TVector<TStatusRepositoryCommon::TCacheObject> TStaticResourceStatusRepository::GetCacheObjectIdsAndRevisions() const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);

    TVector<TCacheObject> result;
    for (const auto& [cacheObject, info]: CacheObjects_) {
        result.push_back(cacheObject);
    }
    return result;
}

TString TStaticResourceStatusRepository::GetCacheObjectHash(const TString& cacheObjectId, ui32 revision) const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetCacheObjectMutex(cacheObjectId, revision));
    return CacheObjects_.at(TCacheObject(cacheObjectId, revision)).ObjectHash_;
}

void TStaticResourceStatusRepository::AddObject(const TStaticResourceMeta& objectMeta) {
    TWriteGuardBase<TLightRWLock> guard(GlobalObjectLock_);

    Objects_.insert(
        {
            objectMeta.Id_
            , TObjectInfo(
                objectMeta.DownloadHash_
                , objectMeta.CheckPeriodMs_
                , objectMeta.SpecTimestamp_
                , objectMeta.Revision_
            )
        }
    );
    THashInfo& objectHashInfo = Hashes_[objectMeta.DownloadHash_];
    objectHashInfo.ObjectIds_.insert(objectMeta.Id_);
    objectHashInfo.CheckPeriodsMs_.insert(objectMeta.CheckPeriodMs_);
}

void TStaticResourceStatusRepository::RemoveObject(const TString& objectId) {
    TWriteGuardBase<TLightRWLock> guard(GlobalObjectLock_);

    const TObjectInfo& objectInfo = Objects_.at(objectId);
    const TString& objectHash = objectInfo.ObjectHash_;

    THashInfo& objectHashInfo = Hashes_.at(objectHash);

    // WARNING: it is TMultiSet
    objectHashInfo.CheckPeriodsMs_.erase(
        objectHashInfo.CheckPeriodsMs_.find(objectInfo.CheckPeriodMs_)
    );

    objectHashInfo.ObjectIds_.erase(objectId);
    // If this is the last reference clear status
    if (objectHashInfo.ObjectIds_.empty() && objectHashInfo.CacheObjectIdsAndRevisions_.empty()) {
        Hashes_.erase(objectHash);
    }

    Objects_.erase(objectId);
}

bool TStaticResourceStatusRepository::HasObjectHash(const TString& objectHash) const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    return Hashes_.FindPtr(objectHash);
}

TString TStaticResourceStatusRepository::GetObjectHash(const TString& objectId) const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    return Objects_.at(objectId).ObjectHash_;
}

ui64 TStaticResourceStatusRepository::GetObjectCheckPeriodMs(const TString& objectId) const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    return Objects_.at(objectId).CheckPeriodMs_;
}

void TStaticResourceStatusRepository::UpdateObjectRevision(const TString& objectId, ui32 revision) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    Objects_.at(objectId).Revision_ = revision;
}

void TStaticResourceStatusRepository::UpdateObjectSpecTimestamp(const TString& objectId, ui64 specTimestamp) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    Objects_.at(objectId).SpecTimestamp_ = specTimestamp;
}

TStatusRepositoryCommon::TObjectState TStaticResourceStatusRepository::UpdateObjectState(const TString& objectIdOrHash, TObjectState state) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectHashMutex(objectIdOrHash));
    API::TStaticResourceStatus& staticResource = Hashes_[objectIdOrHash].Status_;
    TStatusRepositoryCommon::TObjectState oldState = staticResource.state();
    if (oldState != state) {
        staticResource.set_state(std::get<API::EStaticResourceState>(state));
        TInstant now = TInstant::Now();
        RefreshObjectStatusConditions(staticResource, staticResource, now);
    }
    return oldState;
}

void TStaticResourceStatusRepository::IncrementObjectFailCounter(const TString& objectIdOrHash) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectHashMutex(objectIdOrHash));
    API::TStaticResourceStatus& status = Hashes_.at(objectIdOrHash).Status_;
    status.set_fail_counter(status.fail_counter() + 1);
}

void TStaticResourceStatusRepository::UpdateObjectFailedMessage(const TString& objectIdOrHash, const TString& failedMessage) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectHashMutex(objectIdOrHash));
    Hashes_.at(objectIdOrHash).Status_.mutable_failed()->set_message(NSupport::Truncate(failedMessage));
}

void TStaticResourceStatusRepository::UpdateObjectVerifyAttempt(const TString& objectIdOrHash) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectHashMutex(objectIdOrHash));
    API::TStaticResourceStatus& status = Hashes_.at(objectIdOrHash).Status_;
    status.set_verification_attempts_counter(status.verification_attempts_counter() + 1);
}

void TStaticResourceStatusRepository::UpdateObjectDownloadAttempt(const TString& objectIdOrHash) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectHashMutex(objectIdOrHash));
    API::TStaticResourceStatus& status = Hashes_.at(objectIdOrHash).Status_;
    status.set_download_attempts_counter(status.download_attempts_counter() + 1);
}

void TStaticResourceStatusRepository::UpdateObjectDownloadProgress(const TString& objectIdOrHash, ui32 percent) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectHashMutex(objectIdOrHash));
    Hashes_.at(objectIdOrHash).Status_.mutable_download_progress()->set_percent(percent);
}

API::TStaticResourceStatus TStaticResourceStatusRepository::GetObjectStatus(const TString& objectId, TUpdateHolderTargetPtr updateHolderTarget) const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    return GetObjectStatusNoGlobalLock(objectId, updateHolderTarget, false);
}

API::TStaticResourceStatus TStaticResourceStatusRepository::GetObjectStatusNoGlobalLock(
    const TString& objectId
    , TUpdateHolderTargetPtr updateHolderTarget
    , bool conditionsOnly
) const {
    TGuard<TMutex> g(GetObjectMutex(objectId));

    const TObjectInfo& objectInfo = Objects_.at(objectId);
    const TString& objectHash = objectInfo.ObjectHash_;
    TGuard<TMutex> gh(GetObjectHashMutex(objectHash));
    // WARNING: Do not copy full status here
    const API::TStaticResourceStatus& fullStatus = Hashes_.at(objectHash).Status_;

    API::TStaticResourceStatus status;
    if (!conditionsOnly) {
        status.CopyFrom(fullStatus);
        status.set_spec_timestamp(objectInfo.SpecTimestamp_);
        status.set_revision(objectInfo.Revision_);
    }
    // Always add real id to status
    status.set_id(objectId);

    bool changingSpecTimestamp = updateHolderTarget ? updateHolderTarget->StaticResourceHasTarget(objectId) : false;
    RefreshObjectStatusConditions(status, fullStatus, TInstant::Zero(), false, changingSpecTimestamp);

    return status;
}

API::TStaticResourceStatus TStaticResourceStatusRepository::GetObjectHashStatus(const TString& objectHash) const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectHashMutex(objectHash));
    return Hashes_.at(objectHash).Status_;
}

ui64 TStaticResourceStatusRepository::GetObjectHashMinCheckPeriodMs(const TString& objectHash) const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectHashMutex(objectHash));
    return *Hashes_.at(objectHash).CheckPeriodsMs_.begin();
}

bool TStaticResourceStatusRepository::HasObject(const TString& objectId) const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    return Objects_.FindPtr(objectId);
}

TVector<TString> TStaticResourceStatusRepository::GetObjectIds() const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);

    TVector<TString> result;
    for (const auto& [objectId, info] : Objects_) {
        result.push_back(objectId);
    }
    return result;
}

TVector<TString> TStaticResourceStatusRepository::GetObjectIdsByHash(const TString& objectIdOrHash) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);

    if (auto ptr = Hashes_.FindPtr(objectIdOrHash)) {
        return TVector<TString>(ptr->ObjectIds_.begin(), ptr->ObjectIds_.end());
    } else {
        return {};
    }
}

TVector<TStatusRepositoryCommon::TCacheObject> TStaticResourceStatusRepository::GetCacheObjectIdsAndRevisionsByHash(const TString& objectIdOrHash) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);

    if (auto ptr = Hashes_.FindPtr(objectIdOrHash)) {
        return TVector<TStatusRepositoryCommon::TCacheObject>(
            ptr->CacheObjectIdsAndRevisions_.begin(), ptr->CacheObjectIdsAndRevisions_.end()
        );
    } else {
        return {};
    }
}

void TStaticResourceStatusRepository::UpdateActiveDownloadContainersLimit(ui32 activeDownloadContainersLimit) {
    TWriteGuardBase<TLightRWLock> guard(ActiveDownloadContainersLimitLock_);
    ActiveDownloadContainersLimit_ = activeDownloadContainersLimit;
}

void TStaticResourceStatusRepository::UpdateActiveVerifyContainersLimit(ui32 activeVerifyContainersLimit) {
    TWriteGuardBase<TLightRWLock> guard(ActiveVerifyContainersLimitLock_);
    ActiveVerifyContainersLimit_ = activeVerifyContainersLimit;
}

ui32 TStaticResourceStatusRepository::GetActiveDownloadContainersLimit() const {
    TReadGuardBase<TLightRWLock> guard(ActiveDownloadContainersLimitLock_);
    return ActiveDownloadContainersLimit_;
}

ui32 TStaticResourceStatusRepository::GetActiveVerifyContainersLimit() const {
    TReadGuardBase<TLightRWLock> guard(ActiveVerifyContainersLimitLock_);
    return ActiveVerifyContainersLimit_;
}

void TStaticResourceStatusRepository::RefreshObjectStatusConditions(
    API::TStaticResourceStatus& targetStatus
    , const API::TStaticResourceStatus& staticResource
    , const TInstant& now
    , bool refreshTime
    , bool changingSpecTimestamp
) {
    RefreshObjectStatusReady(
        *targetStatus.mutable_ready()
        , staticResource
        , now
        , refreshTime
        , changingSpecTimestamp
    );
    RefreshObjectStatusInProgress(
        *targetStatus.mutable_in_progress()
        , staticResource
        , now
        , refreshTime
        , changingSpecTimestamp
    );
    RefreshObjectStatusFailed(
        *targetStatus.mutable_failed()
        , staticResource
        , now
        , refreshTime
    );
}

void TStaticResourceStatusRepository::RefreshObjectStatusReady(
    API::TCondition& ready
    , const API::TStaticResourceStatus& staticResource
    , const TInstant& now
    , bool refreshTime
    , bool changingSpecTimestamp
) {
    NSupport::CopyConditonStatusAndTimestamp(ready, staticResource.ready());
    API::EConditionStatus prevStatus = ready.status();

    ready.set_reason(NSupport::ExtractState(API::EStaticResourceState_Name(staticResource.state())));
    ready.set_message("");
    if (changingSpecTimestamp) {
        ready.set_message(NSupport::APPLYING_OBJECT_SPEC_MESSAGE);
    }

    switch (staticResource.state()) {
        case API::EStaticResourceState_READY:
            if (!changingSpecTimestamp) {
                ready.set_status(API::EConditionStatus_TRUE);
            } else {
                ready.set_status(API::EConditionStatus_FALSE);
                NSupport::SetApplyingObjectSpecState(ready);
            }
            break;
        case API::EStaticResourceState_DOWNLOADING:
        case API::EStaticResourceState_VERIFYING:
        case API::EStaticResourceState_REMOVING:
        case API::EStaticResourceState_REMOVED:
        case API::EStaticResourceState_INVALID:
        case API::EStaticResourceState_WAITING_FOR_RESOURCE_GANG_META_CONTAINER:
        case API::EStaticResourceState_IN_QUEUE:
        case API::EStaticResourceState_VERIFICATION_IN_QUEUE:
            ready.set_status(API::EConditionStatus_FALSE);
            break;
        case API::EStaticResourceState_UNKNOWN:
            ready.set_status(API::EConditionStatus_UNKNOWN);
            break;
        // We don't want to use default
        // So we need this to handle all cases
        case API::EStaticResourceState_INT_MIN_SENTINEL_DO_NOT_USE_:
        case API::EStaticResourceState_INT_MAX_SENTINEL_DO_NOT_USE_:
            ready.set_status(API::EConditionStatus_UNKNOWN);
            break;
    }

    if (refreshTime && prevStatus != ready.status()) {
        *ready.mutable_last_transition_time() = NSupport::ToTimestamp(now);
    }
}

void TStaticResourceStatusRepository::RefreshObjectStatusInProgress(
    API::TCondition& inProgress
    , const API::TStaticResourceStatus& staticResource
    , const TInstant& now
    , bool refreshTime
    , bool changingSpecTimestamp
) {
    NSupport::CopyConditonStatusAndTimestamp(inProgress, staticResource.in_progress());
    API::EConditionStatus prevStatus = inProgress.status();

    inProgress.set_reason(NSupport::ExtractState(API::EStaticResourceState_Name(staticResource.state())));
    inProgress.set_message("");
    if (changingSpecTimestamp) {
        inProgress.set_message(NSupport::APPLYING_OBJECT_SPEC_MESSAGE);
    }

    switch (staticResource.state()) {
        case API::EStaticResourceState_DOWNLOADING:
        case API::EStaticResourceState_VERIFYING:
        case API::EStaticResourceState_REMOVING:
            inProgress.set_status(API::EConditionStatus_TRUE);
            break;
        case API::EStaticResourceState_READY:
        case API::EStaticResourceState_REMOVED:
            if (!changingSpecTimestamp) {
                inProgress.set_status(API::EConditionStatus_FALSE);
            } else {
                inProgress.set_status(API::EConditionStatus_TRUE);
                NSupport::SetApplyingObjectSpecState(inProgress);
            }
            break;
        case API::EStaticResourceState_WAITING_FOR_RESOURCE_GANG_META_CONTAINER:
        case API::EStaticResourceState_IN_QUEUE:
        case API::EStaticResourceState_VERIFICATION_IN_QUEUE:
        case API::EStaticResourceState_INVALID:
            inProgress.set_status(API::EConditionStatus_FALSE);
            break;
        case API::EStaticResourceState_UNKNOWN:
            inProgress.set_status(API::EConditionStatus_UNKNOWN);
            break;
        // We don't want to use default
        // So we need this to handle all cases
        case API::EStaticResourceState_INT_MIN_SENTINEL_DO_NOT_USE_:
        case API::EStaticResourceState_INT_MAX_SENTINEL_DO_NOT_USE_:
            inProgress.set_status(API::EConditionStatus_UNKNOWN);
            break;
    }

    if (refreshTime && prevStatus != inProgress.status()) {
        *inProgress.mutable_last_transition_time() = NSupport::ToTimestamp(now);
    }
}

void TStaticResourceStatusRepository::RefreshObjectStatusFailed(
    API::TCondition& failed
    , const API::TStaticResourceStatus& staticResource
    , const TInstant& now
    , bool refreshTime
) {
    NSupport::CopyConditonStatusAndTimestamp(failed, staticResource.failed());
    API::EConditionStatus prevStatus = failed.status();

    switch (staticResource.state()) {
        case API::EStaticResourceState_DOWNLOADING:
        case API::EStaticResourceState_VERIFYING:
        case API::EStaticResourceState_REMOVING:
        case API::EStaticResourceState_REMOVED:
        case API::EStaticResourceState_WAITING_FOR_RESOURCE_GANG_META_CONTAINER:
        case API::EStaticResourceState_IN_QUEUE:
        case API::EStaticResourceState_VERIFICATION_IN_QUEUE:
            // UNKNOWN -> FALSE
            // TRUE and FALSE do not change
            if (failed.status() == API::EConditionStatus_UNKNOWN) {
                failed.set_status(API::EConditionStatus_FALSE);
            }
            break;
        case API::EStaticResourceState_READY:
            failed.set_status(API::EConditionStatus_FALSE);
            failed.set_reason(NSupport::ExtractState(API::EStaticResourceState_Name(staticResource.state())));
            break;
        case API::EStaticResourceState_INVALID:
            failed.set_status(API::EConditionStatus_TRUE);
            failed.set_reason(NSupport::ExtractState(API::EStaticResourceState_Name(staticResource.state())));
            break;
        case API::EStaticResourceState_UNKNOWN:
            failed.set_status(API::EConditionStatus_UNKNOWN);
            failed.set_reason(NSupport::ExtractState(API::EStaticResourceState_Name(staticResource.state())));
            break;
        // We don't want to use default
        // So we need this to handle all cases
        case API::EStaticResourceState_INT_MIN_SENTINEL_DO_NOT_USE_:
        case API::EStaticResourceState_INT_MAX_SENTINEL_DO_NOT_USE_:
            failed.set_status(API::EConditionStatus_UNKNOWN);
            failed.set_reason(NSupport::ExtractState(API::EStaticResourceState_Name(staticResource.state())));
            break;
    }

    if (refreshTime && prevStatus != failed.status()) {
        *failed.mutable_last_transition_time() = NSupport::ToTimestamp(now);
    }
}

const TMutex& TStaticResourceStatusRepository::GetCacheObjectMutex(const TString& cacheObjectId, ui32 revision) const {
    const TCacheObjectInfo* ptr = CacheObjects_.FindPtr(TStatusRepositoryCommon::TCacheObject(cacheObjectId, revision));
    Y_ENSURE(ptr, "CacheObject '" << cacheObjectId << "' with revision '" << ToString(revision) << "' not found at TStaticResourceStatusRepository");
    return ptr->Mutex_;
}

const TMutex& TStaticResourceStatusRepository::GetObjectMutex(const TString& objectId) const {
    const TObjectInfo* ptr = Objects_.FindPtr(objectId);
    Y_ENSURE(ptr, "Object '" << objectId << "' not found at TStaticResourceStatusRepository");
    return ptr->Mutex_;
}

const TMutex& TStaticResourceStatusRepository::GetObjectHashMutex(const TString& objectHash) const {
    const THashInfo* ptr = Hashes_.FindPtr(objectHash);
    Y_ENSURE(ptr, "Object hash '" << objectHash << "' not found at TStaticResourceStatusRepository");
    return ptr->Mutex_;
}

} // namespace NInfra::NPodAgent
