#pragma once

#include "status_repository_common.h"

#include <infra/pod_agent/libs/pod_agent/object_meta/object_meta.h>

#include <util/generic/vector.h>
#include <util/generic/map.h>
#include <util/generic/set.h>

namespace NInfra::NPodAgent {

/**
    Thread-safe static resource status repository
*/
class TStaticResourceStatusRepository: public TStatusRepositoryCommon {
private:
    struct TCacheObjectInfo {
        TCacheObjectInfo(
            const TString& objectHash
            , ui64 checkPeriodMs
        )
            : ObjectHash_(objectHash)
            , CheckPeriodMs_(checkPeriodMs)
        {}

        TMutex Mutex_;
        const TString ObjectHash_;
        const ui64 CheckPeriodMs_;
    };

    struct TObjectInfo {
        TObjectInfo(
            const TString& objectHash
            , ui64 checkPeriodMs
            , ui64 specTimestamp
            , ui32 revision
        )
            : ObjectHash_(objectHash)
            , CheckPeriodMs_(checkPeriodMs)
            , SpecTimestamp_(specTimestamp)
            , Revision_(revision)
        {}

        TMutex Mutex_;
        const TString ObjectHash_;
        const ui64 CheckPeriodMs_;

        ui64 SpecTimestamp_;
        ui32 Revision_;
    };

    struct THashInfo {
        TMutex Mutex_;
        API::TStaticResourceStatus Status_;

        TSet<TString> ObjectIds_;
        TSet<TCacheObject> CacheObjectIdsAndRevisions_;
        TMultiSet<ui64> CheckPeriodsMs_;
    };

public:
    TStaticResourceStatusRepository()
        : TStatusRepositoryCommon(NStatusRepositoryTypes::EObjectType::STATIC_RESOURCE)
        , ActiveDownloadContainersLimit_(0)
        , ActiveVerifyContainersLimit_(0)
    {}

    virtual void UpdateSpecTimestamp(TInstant currentTime) override final;
    virtual void PatchTotalStatus(API::TPodAgentStatus& status, TUpdateHolderTargetPtr updateHolderTarget, bool conditionsOnly) const override final;
    virtual bool NeedLongTickPeriod(const TString& objectIdOrHash, TUpdateHolderTargetPtr updateHolderTarget) const override final;

    // Container methods
    virtual void UpdateContainerFailReason(const NStatusRepositoryTypes::TContainerDescription& container, const TString& failReason) override final;
    virtual void IncrementContainerSystemFailureCounter(const NStatusRepositoryTypes::TContainerDescription& container) override final;
    virtual void UpdateContainerState(const NStatusRepositoryTypes::TContainerDescription& container, API::EContainerState state) override final;

    // Common object methods
    virtual void UpdateObjectRevision(const TString& objectId, ui32 revision) override final;
    virtual void UpdateObjectSpecTimestamp(const TString& objectId, ui64 specTimestamp) override final;
    virtual TObjectState UpdateObjectState(const TString& objectIdOrHash, TObjectState state) override final;
    virtual void UpdateObjectDownloadProgress(const TString& objectIdOrHash, ui32 percent) override final;
    virtual void UpdateObjectFailedMessage(const TString& objectIdOrHash, const TString& failedMessage) override final;
    virtual void IncrementObjectFailCounter(const TString& objectIdOrHash) override final;
    virtual void UpdateObjectVerifyAttempt(const TString& objectIdOrHash) override final;
    virtual void UpdateObjectDownloadAttempt(const TString& objectIdOrHash) override final;
    virtual TVector<TString> GetObjectIdsByHash(const TString& objectIdOrHash) override final;
    virtual TVector<TCacheObject> GetCacheObjectIdsAndRevisionsByHash(const TString& objectIdOrHash) override final;

    // ResourceGang methods
    virtual void UpdateActiveDownloadContainersLimit(ui32 activeDownloadContainersLimit) override final;
    virtual void UpdateActiveVerifyContainersLimit(ui32 activeVerifyContainersLimit) override final;
    virtual ui32 GetActiveDownloadContainersLimit() const override final;
    virtual ui32 GetActiveVerifyContainersLimit() const override final;

    // StaticResource specific methods

    // CacheStaticResource
    void AddCacheObject(const TStaticResourceMeta& cacheObjectMeta);
    void RemoveCacheObject(const TString& cacheObjectId, ui32 revision);
    bool HasCacheObject(const TString& cacheObjectId, ui32 revision) const;

    TString GetCacheObjectHash(const TString& cacheObjectId, ui32 revision) const;
    TVector<TCacheObject> GetCacheObjectIdsAndRevisions() const;
    API::TStaticResourceStatus GetCacheObjectStatus(const TString& cacheObjectId, ui32 revision) const;

    // StaticResource
    void AddObject(const TStaticResourceMeta& objectMeta);
    void RemoveObject(const TString& objectId);
    bool HasObject(const TString& objectId) const;

    TString GetObjectHash(const TString& objectId) const;
    ui64 GetObjectCheckPeriodMs(const TString& objectId) const;
    TVector<TString> GetObjectIds() const;
    API::TStaticResourceStatus GetObjectStatus(const TString& objectId, TUpdateHolderTargetPtr updateHolderTarget = nullptr) const;

    // Hash
    bool HasObjectHash(const TString& objectHash) const;
    ui64 GetObjectHashMinCheckPeriodMs(const TString& objectHash) const;
    API::TStaticResourceStatus GetObjectHashStatus(const TString& objectHash) const;

private:
    // CacheStaticResource
    const TMutex& GetCacheObjectMutex(const TString& cacheObjectId, ui32 revision) const;
    API::TStaticResourceStatus GetCacheObjectStatusNoGlobalLock(
        const TString& cacheObjectId
        , ui32 revision
        , bool conditionsOnly
    ) const;

    // StaticResource
    const TMutex& GetObjectMutex(const TString& objectId) const;
    API::TStaticResourceStatus GetObjectStatusNoGlobalLock(
        const TString& objectId
        , TUpdateHolderTargetPtr updateHolderTarget
        , bool conditionsOnly
    ) const;

    // Hash
    const TMutex& GetObjectHashMutex(const TString& objectHash) const;

    static void RefreshObjectStatusConditions(
        API::TStaticResourceStatus& targetStatus
        , const API::TStaticResourceStatus& staticResource
        , const TInstant& now
        , bool refreshTime = true
        , bool changingSpecTimestamp = false
    );
    static void RefreshObjectStatusReady(
        API::TCondition& ready
        , const API::TStaticResourceStatus& staticResource
        , const TInstant& now
        , bool refreshTime
        , bool changingSpecTimestamp
    );
    static void RefreshObjectStatusInProgress(
        API::TCondition& inProgress
        , const API::TStaticResourceStatus& staticResource
        , const TInstant& now
        , bool refreshTime
        , bool changingSpecTimestamp
    );
    static void RefreshObjectStatusFailed(
        API::TCondition& failed
        , const API::TStaticResourceStatus& staticResource
        , const TInstant& now
        , bool refreshTime
    );

private:
    ui32 ActiveDownloadContainersLimit_;
    ui32 ActiveVerifyContainersLimit_;
    TLightRWLock ActiveDownloadContainersLimitLock_;
    TLightRWLock ActiveVerifyContainersLimitLock_;

    TMap<TCacheObject, TCacheObjectInfo> CacheObjects_;
    TMap<TString, TObjectInfo> Objects_;
    TMap<TString, THashInfo> Hashes_;
    TLightRWLock GlobalObjectLock_;
};

using TStaticResourceStatusRepositoryPtr = TIntrusivePtr<TStaticResourceStatusRepository>;

} // namespace NInfra::NPodAgent
