#include "status_repository.h"

#include "support_functions.h"

namespace NInfra::NPodAgent {

TUpdateHolderPtr TStatusRepository::GetUpdateHolder() const {
    return UpdateHolder_;
}

TBoxStatusRepositoryPtr TStatusRepository::GetBoxStatusRepository() const {
    return BoxStatusRepository_;
}

TLayerStatusRepositoryPtr TStatusRepository::GetLayerStatusRepository() const {
    return LayerStatusRepository_;
}

TStaticResourceStatusRepositoryPtr TStatusRepository::GetStaticResourceStatusRepository() const {
    return StaticResourceStatusRepository_;
}

TVolumeStatusRepositoryPtr TStatusRepository::GetVolumeStatusRepository() const {
    return VolumeStatusRepository_;
}

TWorkloadStatusRepositoryPtr TStatusRepository::GetWorkloadStatusRepository() const {
    return WorkloadStatusRepository_;
}

void TStatusRepository::SetSpecTimestamp(ui64 specTimestamp) {
    TWriteGuardBase<TLightRWLock> guard(TotalStatusLock_);
    TInstant currentTime = TInstant::Now();

    BoxStatusRepository_->UpdateSpecTimestamp(currentTime);
    LayerStatusRepository_->UpdateSpecTimestamp(currentTime);
    StaticResourceStatusRepository_->UpdateSpecTimestamp(currentTime);
    VolumeStatusRepository_->UpdateSpecTimestamp(currentTime);
    WorkloadStatusRepository_->UpdateSpecTimestamp(currentTime);

    SpecTimestamp_ = specTimestamp;
}

void TStatusRepository::SetRevision(ui32 revision) {
    TWriteGuardBase<TLightRWLock> guard(TotalStatusLock_);
    TInstant currentTime = TInstant::Now();

    BoxStatusRepository_->UpdateSpecTimestamp(currentTime);
    LayerStatusRepository_->UpdateSpecTimestamp(currentTime);
    StaticResourceStatusRepository_->UpdateSpecTimestamp(currentTime);
    VolumeStatusRepository_->UpdateSpecTimestamp(currentTime);
    WorkloadStatusRepository_->UpdateSpecTimestamp(currentTime);

    Revision_ = revision;
}

void TStatusRepository::SetSpecId(const TString& id) {
    TWriteGuardBase<TLightRWLock> guard(TotalStatusLock_);
    SpecId_ = id;
}

void TStatusRepository::SetTargetState(const API::EPodAgentTargetState targetState) {
    TWriteGuardBase<TLightRWLock> guard(TotalStatusLock_);
    TargetState_ = targetState;
}

API::EPodAgentTargetState TStatusRepository::GetTargetState() const {
    TReadGuardBase<TLightRWLock> guard(TotalStatusLock_);
    return TargetState_;
}

TString TStatusRepository::GetSpecId() const {
    TReadGuardBase<TLightRWLock> guard(TotalStatusLock_);
    return SpecId_;
}

API::TPodAgentStatus TStatusRepository::GetTotalStatus(bool conditionsOnly) {
    API::TPodAgentStatus status;

    {
        TReadGuardBase<TLightRWLock> guard(TotalStatusLock_);
        status.set_spec_timestamp(SpecTimestamp_);
        status.set_revision(Revision_);
        status.set_id(SpecId_);
        status.set_target_state(TargetState_);
    }

    API::TCondition* ready = status.mutable_ready();
    ready->set_status(API::EConditionStatus_TRUE);
    ready->set_reason("WORKLOADS_READY");
    ready->set_message("");
    API::TCondition* inProgress = status.mutable_in_progress();
    inProgress->set_status(API::EConditionStatus_FALSE);
    inProgress->set_reason("ALL_READY");
    inProgress->set_message("");
    API::TCondition* failed = status.mutable_failed();
    failed->set_status(API::EConditionStatus_FALSE);
    failed->set_reason("ALL_OK");
    failed->set_message("");

    BoxStatusRepository_->PatchTotalStatus(status, UpdateHolder_->GetUpdateHolderTarget(), conditionsOnly);
    LayerStatusRepository_->PatchTotalStatus(status, UpdateHolder_->GetUpdateHolderTarget(), conditionsOnly);
    StaticResourceStatusRepository_->PatchTotalStatus(status, UpdateHolder_->GetUpdateHolderTarget(), conditionsOnly);
    VolumeStatusRepository_->PatchTotalStatus(status, UpdateHolder_->GetUpdateHolderTarget(), conditionsOnly);
    WorkloadStatusRepository_->PatchTotalStatus(status, UpdateHolder_->GetUpdateHolderTarget(), conditionsOnly);

    TInstant currentTime = TInstant::Now();
    status.set_host_timestamp(currentTime.Seconds());

    PushConditionSignals(status);

    return status;
}

void TStatusRepository::PushConditionSignals(const API::TPodAgentStatus& status) {
    NSupport::PushAllPodSignals(status);

    for (const auto& cacheLayerStatus : status.resource_cache().layers()) {
        NSupport::PushAllCacheObjectSignals(
            cacheLayerStatus
            , NStatusRepositoryTypes::EObjectType::LAYER
        );
    }

    for (const auto& cacheStaticResourceStatus : status.resource_cache().static_resources()) {
        NSupport::PushAllCacheObjectSignals(
            cacheStaticResourceStatus
            , NStatusRepositoryTypes::EObjectType::LAYER
        );
    }

    for (const auto& boxStatus : status.boxes()) {
        NSupport::PushAllObjectSignals(
            boxStatus
            , NStatusRepositoryTypes::EObjectType::BOX
        );
    }

    for (const auto& layerStatus : status.resource_gang().layers()) {
        NSupport::PushAllObjectSignals(
            layerStatus
            , NStatusRepositoryTypes::EObjectType::LAYER
        );
    }

    for (const auto& staticResourceStatus : status.resource_gang().static_resources()) {
        NSupport::PushAllObjectSignals(
            staticResourceStatus
            , NStatusRepositoryTypes::EObjectType::STATIC_RESOURCE
        );
    }

    for (const auto& volumeStatus : status.volumes()) {
        NSupport::PushAllObjectSignals(
            volumeStatus
            , NStatusRepositoryTypes::EObjectType::VOLUME
        );
    }

    for (const auto& workloadStatus : status.workloads()) {
        NSupport::PushAllObjectSignals(
            workloadStatus
            , NStatusRepositoryTypes::EObjectType::WORKLOAD
        );
    }
}

} // namespace NInfra::NPodAgent
