#include "status_repository_common.h"

#include "support_functions.h"

#include <infra/pod_agent/libs/pod_agent/unistat_object_helper/unistat_object_helper.h>

namespace NInfra::NPodAgent {

NStatusRepositoryTypes::EObjectType TStatusRepositoryCommon::GetObjectType() {
    return ObjectType_;
}

void TStatusRepositoryCommon::UpdateContainerFailReason(const NStatusRepositoryTypes::TContainerDescription& container, const TString& failReason) {
    CheckContainerObjectType(container);
    TReadGuardBase<TLightRWLock> guard(GetGlobalContainerLock());
    TGuard<TMutex> g(GetLocalContainerLock(container));
    GetMutableContainerStatus(container)->mutable_current()->set_fail_reason(NSupport::Truncate(failReason));
}

void TStatusRepositoryCommon::IncrementContainerSystemFailureCounter(const NStatusRepositoryTypes::TContainerDescription& container) {
    CheckContainerObjectType(container);
    TReadGuardBase<TLightRWLock> guard(GetGlobalContainerLock());
    TGuard<TMutex> g(GetLocalContainerLock(container));

    API::TContainerStatus* containerStatus = GetMutableContainerStatus(container);
    ui32 attempts = containerStatus->system_failure_counter();
    containerStatus->set_system_failure_counter(attempts + 1);

    TUnistatObjectHelper::Instance().PushContainerSignal(
        container
        , TUnistatObjectHelper::EContainerSignalType::SYSTEM_FAILURE
        , 1
    );
}

void TStatusRepositoryCommon::ClearContainerStatus(const NStatusRepositoryTypes::TContainerDescription& container) {
    CheckContainerObjectType(container);
    TReadGuardBase<TLightRWLock> guard(GetGlobalContainerLock());
    TGuard<TMutex> g(GetLocalContainerLock(container));
    GetMutableContainerStatus(container)->Clear();
}

TStatusRepositoryCommon::TObjectState TStatusRepositoryCommon::UpdateContainerOwnerState(const NStatusRepositoryTypes::TContainerDescription& container, const TObjectState& state) {
    CheckContainerObjectType(container);
    return UpdateObjectState(container.ObjectIdOrHash_, state);
}

void TStatusRepositoryCommon::UpdateContainerState(const NStatusRepositoryTypes::TContainerDescription& container, API::EContainerState state) {
    CheckContainerObjectType(container);
    TReadGuardBase<TLightRWLock> guard(GetGlobalContainerLock());
    TGuard<TMutex> g(GetLocalContainerLock(container));
    GetMutableContainerStatus(container)->mutable_current()->set_state(state);
}

void TStatusRepositoryCommon::UpdateContainerStderr(const NStatusRepositoryTypes::TContainerDescription& container, const TString& stderr) {
    CheckContainerObjectType(container);
    TReadGuardBase<TLightRWLock> guard(GetGlobalContainerLock());
    TGuard<TMutex> g(GetLocalContainerLock(container));
    GetMutableContainerStatus(container)->mutable_current()->set_stderr_tail(stderr);
}

void TStatusRepositoryCommon::UpdateContainerStdout(const NStatusRepositoryTypes::TContainerDescription& container, const TString& stdout) {
    CheckContainerObjectType(container);
    TReadGuardBase<TLightRWLock> guard(GetGlobalContainerLock());
    TGuard<TMutex> g(GetLocalContainerLock(container));
    GetMutableContainerStatus(container)->mutable_current()->set_stdout_tail(stdout);
}

void TStatusRepositoryCommon::UpdateContainerStartTime(const NStatusRepositoryTypes::TContainerDescription& container, const TInstant& time) {
    CheckContainerObjectType(container);
    TReadGuardBase<TLightRWLock> guard(GetGlobalContainerLock());
    TGuard<TMutex> g(GetLocalContainerLock(container));
    *GetMutableContainerStatus(container)->mutable_current()->mutable_start_time() = NSupport::ToTimestamp(time);
}

void TStatusRepositoryCommon::UpdateContainerDeathTime(const NStatusRepositoryTypes::TContainerDescription& container, const TInstant& time) {
    CheckContainerObjectType(container);
    TReadGuardBase<TLightRWLock> guard(GetGlobalContainerLock());
    TGuard<TMutex> g(GetLocalContainerLock(container));
    *GetMutableContainerStatus(container)->mutable_current()->mutable_death_time() = NSupport::ToTimestamp(time);
}

void TStatusRepositoryCommon::UpdateContainerReturnCode(const NStatusRepositoryTypes::TContainerDescription& container, i32 code) {
    CheckContainerObjectType(container);
    TReadGuardBase<TLightRWLock> guard(GetGlobalContainerLock());
    TGuard<TMutex> g(GetLocalContainerLock(container));
    GetMutableContainerStatus(container)->mutable_current()->set_return_code(code);
}

void TStatusRepositoryCommon::IncrementContainerOomCounter(const NStatusRepositoryTypes::TContainerDescription& container) {
    CheckContainerObjectType(container);
    TReadGuardBase<TLightRWLock> guard(GetGlobalContainerLock());
    TGuard<TMutex> g(GetLocalContainerLock(container));

    API::TContainerStatus* containerStatus = GetMutableContainerStatus(container);
    ui32 attempts = containerStatus->oom_counter();
    containerStatus->set_oom_counter(attempts + 1);

    TUnistatObjectHelper::Instance().PushContainerSignal(
        container
        , TUnistatObjectHelper::EContainerSignalType::OOM
        , 1
    );
}

void TStatusRepositoryCommon::IncrementContainerZeroReturnCodeCounter(const NStatusRepositoryTypes::TContainerDescription& container) {
    CheckContainerObjectType(container);
    TReadGuardBase<TLightRWLock> guard(GetGlobalContainerLock());
    TGuard<TMutex> g(GetLocalContainerLock(container));

    API::TContainerStatus* containerStatus = GetMutableContainerStatus(container);
    ui32 attempts = containerStatus->zero_return_code_counter();
    containerStatus->set_zero_return_code_counter(attempts + 1);

    TUnistatObjectHelper::Instance().PushContainerSignal(
        container
        , TUnistatObjectHelper::EContainerSignalType::ZERO_RETURN_CODE
        , 1
    );
}

void TStatusRepositoryCommon::IncrementContainerPositiveReturnCodeCounter(const NStatusRepositoryTypes::TContainerDescription& container) {
    CheckContainerObjectType(container);
    TReadGuardBase<TLightRWLock> guard(GetGlobalContainerLock());
    TGuard<TMutex> g(GetLocalContainerLock(container));

    API::TContainerStatus* containerStatus = GetMutableContainerStatus(container);
    ui32 attempts = containerStatus->positive_return_code_counter();
    containerStatus->set_positive_return_code_counter(attempts + 1);

    TUnistatObjectHelper::Instance().PushContainerSignal(
        container
        , TUnistatObjectHelper::EContainerSignalType::POSITIVE_RETURN_CODE
        , 1
    );
}

void TStatusRepositoryCommon::IncrementContainerKilledExternallyCounter(const NStatusRepositoryTypes::TContainerDescription& container) {
    CheckContainerObjectType(container);
    TReadGuardBase<TLightRWLock> guard(GetGlobalContainerLock());
    TGuard<TMutex> g(GetLocalContainerLock(container));

    API::TContainerStatus* containerStatus = GetMutableContainerStatus(container);
    ui32 attempts = containerStatus->killed_externally_counter();
    containerStatus->set_killed_externally_counter(attempts + 1);

    TUnistatObjectHelper::Instance().PushContainerSignal(
        container
        , TUnistatObjectHelper::EContainerSignalType::KILLED_EXTERNALLY
        , 1
    );
}

void TStatusRepositoryCommon::IncrementContainerTimeoutCounter(const NStatusRepositoryTypes::TContainerDescription& container) {
    CheckContainerObjectType(container);
    TReadGuardBase<TLightRWLock> guard(GetGlobalContainerLock());
    TGuard<TMutex> g(GetLocalContainerLock(container));

    API::TContainerStatus* containerStatus = GetMutableContainerStatus(container);
    ui32 attempts = containerStatus->timeout_counter();
    containerStatus->set_timeout_counter(attempts + 1);

    TUnistatObjectHelper::Instance().PushContainerSignal(
        container
        , TUnistatObjectHelper::EContainerSignalType::TIMEOUT
        , 1
    );
}

void TStatusRepositoryCommon::UpdateContainerConsecutiveFailuresAndSuccessesCounter(const NStatusRepositoryTypes::TContainerDescription& container, i32 code) {
    CheckContainerObjectType(container);
    TReadGuardBase<TLightRWLock> guard(GetGlobalContainerLock());
    TGuard<TMutex> g(GetLocalContainerLock(container));

    API::TContainerStatus* containerStatus = GetMutableContainerStatus(container);
    if (code != 0) {
        containerStatus->mutable_time_limit()->set_consecutive_successes_counter(0);
        ui32 attempts = containerStatus->time_limit().consecutive_failures_counter();
        containerStatus->mutable_time_limit()->set_consecutive_failures_counter(attempts + 1);
    } else {
        containerStatus->mutable_time_limit()->set_consecutive_failures_counter(0);
        ui32 attempts = containerStatus->time_limit().consecutive_successes_counter();
        containerStatus->mutable_time_limit()->set_consecutive_successes_counter(attempts + 1);
    }

    TUnistatObjectHelper::Instance().PushContainerSignal(
        container
        , TUnistatObjectHelper::EContainerSignalType::EXITED
        , 1
    );
}

API::TContainerStatus::TAttemptFeedback TStatusRepositoryCommon::CaptureContainerStatus(const NStatusRepositoryTypes::TContainerDescription& container) {
    CheckContainerObjectType(container);
    TReadGuardBase<TLightRWLock> guard(GetGlobalContainerLock());
    TGuard<TMutex> g(GetLocalContainerLock(container));
    if (GetContainerStatus(container)->has_current()) {
        const auto& currentAttempt = GetContainerStatus(container)->current();

        *GetMutableContainerStatus(container)->mutable_last() = currentAttempt;
        if (currentAttempt.state() != API::EContainerState_EXITED || currentAttempt.return_code() != 0) {
            *GetMutableContainerStatus(container)->mutable_last_failed() = currentAttempt;
        }

        GetMutableContainerStatus(container)->clear_current();
    }
    return GetContainerStatus(container)->last();
}

TInstant TStatusRepositoryCommon::GetContainerStartTime(const NStatusRepositoryTypes::TContainerDescription& container) {
    CheckContainerObjectType(container);
    TReadGuardBase<TLightRWLock> guard(GetGlobalContainerLock());
    TGuard<TMutex> g(GetLocalContainerLock(container));
    return NSupport::ToInstant(GetContainerStatus(container)->last().start_time());
}

ui32 TStatusRepositoryCommon::GetContainerConsecutiveSuccessesCounter(const NStatusRepositoryTypes::TContainerDescription& container) {
    CheckContainerObjectType(container);
    TReadGuardBase<TLightRWLock> guard(GetGlobalContainerLock());
    TGuard<TMutex> g(GetLocalContainerLock(container));
    return GetContainerStatus(container)->time_limit().consecutive_successes_counter();
}

ui32 TStatusRepositoryCommon::GetContainerConsecutiveFailuresCounter(const NStatusRepositoryTypes::TContainerDescription& container) {
    CheckContainerObjectType(container);
    TReadGuardBase<TLightRWLock> guard(GetGlobalContainerLock());
    TGuard<TMutex> g(GetLocalContainerLock(container));
    return GetContainerStatus(container)->time_limit().consecutive_failures_counter();
}

ui32 TStatusRepositoryCommon::GetContainerZeroReturnCodeCounter(const NStatusRepositoryTypes::TContainerDescription& container) {
    CheckContainerObjectType(container);
    TReadGuardBase<TLightRWLock> guard(GetGlobalContainerLock());
    TGuard<TMutex> g(GetLocalContainerLock(container));
    return GetContainerStatus(container)->zero_return_code_counter();
}

ui32 TStatusRepositoryCommon::GetContainerPositiveReturnCodeCounter(const NStatusRepositoryTypes::TContainerDescription& container) {
    CheckContainerObjectType(container);
    TReadGuardBase<TLightRWLock> guard(GetGlobalContainerLock());
    TGuard<TMutex> g(GetLocalContainerLock(container));
    return GetContainerStatus(container)->positive_return_code_counter();
}

ui32 TStatusRepositoryCommon::GetContainerOomCounter(const NStatusRepositoryTypes::TContainerDescription& container) {
    CheckContainerObjectType(container);
    TReadGuardBase<TLightRWLock> guard(GetGlobalContainerLock());
    TGuard<TMutex> g(GetLocalContainerLock(container));
    return GetContainerStatus(container)->oom_counter();
}

ui32 TStatusRepositoryCommon::GetContainerSystemFailureCounter(const NStatusRepositoryTypes::TContainerDescription& container) {
    CheckContainerObjectType(container);
    TReadGuardBase<TLightRWLock> guard(GetGlobalContainerLock());
    TGuard<TMutex> g(GetLocalContainerLock(container));
    return GetContainerStatus(container)->system_failure_counter();
}

ui32 TStatusRepositoryCommon::GetContainerKilledExternallyCounter(const NStatusRepositoryTypes::TContainerDescription& container) {
    CheckContainerObjectType(container);
    TReadGuardBase<TLightRWLock> guard(GetGlobalContainerLock());
    TGuard<TMutex> g(GetLocalContainerLock(container));
    return GetContainerStatus(container)->killed_externally_counter();
}

TString TStatusRepositoryCommon::GetContainerFailReason(const NStatusRepositoryTypes::TContainerDescription& container) {
    CheckContainerObjectType(container);
    TReadGuardBase<TLightRWLock> guard(GetGlobalContainerLock());
    TGuard<TMutex> g(GetLocalContainerLock(container));
    return GetContainerStatus(container)->current().fail_reason();
}

void TStatusRepositoryCommon::CheckContainerObjectType(const NStatusRepositoryTypes::TContainerDescription& container) const {
    Y_ENSURE(container.ObjectType_ == ObjectType_, "Expected container object type: " << ToString(ObjectType_) << ", but got container with object type: " << ToString(container.ObjectType_));
}

} // namespace NInfra::NPodAgent
