#pragma once

#include <infra/pod_agent/libs/pod_agent/status_repository/types/status_repository_types.h>
#include <infra/pod_agent/libs/pod_agent/update_holder/update_holder.h>
#include <infra/pod_agent/libs/service_iface/protos/service.pb.h>

#include <library/cpp/threading/light_rw_lock/lightrwlock.h>

#include <util/string/cast.h>
#include <util/system/mutex.h>
#include <util/datetime/base.h>
#include <util/generic/variant.h>
#include <util/generic/vector.h>

namespace NInfra::NPodAgent {

/**
    Thread-safe common part of status repository
*/
class TStatusRepositoryCommon: public TAtomicRefCount<TStatusRepositoryCommon> {
public:
    struct TCacheObject {
        TCacheObject(const TString& id, ui32 revision)
            : Id_(id)
            , Revision_(revision)
        {}

        TString Id_;
        ui32 Revision_;

        bool operator<(const TCacheObject& other) const {
            return Id_ < other.Id_ || (Id_ == other.Id_ && Revision_ < other.Revision_);
        }

        bool operator==(const TCacheObject& other) const {
            return Id_ == other.Id_ && Revision_ == other.Revision_;
        }
    };

    using TObjectState = std::variant<API::EBoxState, API::ELayerState, API::EStaticResourceState, API::EVolumeState, API::EWorkloadState>;

public:
    TStatusRepositoryCommon(NStatusRepositoryTypes::EObjectType objectType)
        : ObjectType_(objectType)
    {}

    virtual ~TStatusRepositoryCommon() = default;

    virtual void UpdateSpecTimestamp(TInstant currentTime) = 0;
    virtual void PatchTotalStatus(API::TPodAgentStatus& status, TUpdateHolderTargetPtr updateHolderTarget, bool conditionsOnly) const = 0;
    virtual bool NeedLongTickPeriod(const TString& objectIdOrHash, TUpdateHolderTargetPtr updateHolderTarget) const = 0;

    // Common object methods
    virtual void UpdateObjectRevision(const TString& /* objectId */, ui32 /* revision */) {
        ythrow yexception() << "UpdateObjectRevision not implemented";
    }

    virtual void UpdateObjectSpecTimestamp(const TString& /* objectId */, ui64 /* specTimestamp */) {
        ythrow yexception() << "UpdateObjectSpecTimestamp not implemented";
    }

    virtual TObjectState UpdateObjectState(const TString& /* objectIdOrHash */, TObjectState /* state */) {
        ythrow yexception() << "UpdateObjectState not implemented";
    }

    virtual void UpdateObjectDownloadProgress(const TString& /* objectIdOrHash */, ui32 /* percent */) {
        ythrow yexception() << "UpdateObjectDownloadProgress not implemented";
    }

    virtual void UpdateObjectFailedMessage(const TString& /* objectIdOrHash */, const TString& /* failedMessage */) {
        ythrow yexception() << "UpdateObjectFailedMessage not implemented";
    }

    virtual void IncrementObjectFailCounter(const TString& /* objectIdOrHash */) {
        ythrow yexception() << "IncrementObjectFailCounter not implemented";
    }

    virtual void UpdateObjectVerifyAttempt(const TString& /* objectIdOrHash */) {
        ythrow yexception() << "UpdateObjectVerifyAttempt not implemented";
    }

    virtual void UpdateObjectDownloadAttempt(const TString& /* objectIdOrHash */) {
        ythrow yexception() << "UpdateObjectDownloadAttempt not implemented";
    }

    virtual void UpdateObjectIpAddress(const TString& /* objectIdOrHash */, const TString& /* address */) {
        ythrow yexception() << "UpdateObjectIpAddress not implemented";
    }

    virtual TVector<TString> GetObjectIdsByHash(const TString& /* objectIdOrHash */) {
        ythrow yexception() << "GetObjectIdsByHash not implemented";
    }

    virtual TVector<TCacheObject> GetCacheObjectIdsAndRevisionsByHash(const TString& /* objectIdOrHash */) {
        ythrow yexception() << "GetCacheObjectIdsAndRevisionsByHash not implemented";
    }

    // ResourceGang methods
    virtual void UpdateActiveDownloadContainersLimit(ui32 /* activeDownloadContainersLimit */) {
        ythrow yexception() << "UpdateActiveDownloadContainersLimit not implemented";
    }
    virtual void UpdateActiveVerifyContainersLimit(ui32 /* activeVerifyContainersLimit */) {
        ythrow yexception() << "UpdateActiveVerifyContainersLimit not implemented";
    }
    virtual ui32 GetActiveDownloadContainersLimit() const {
        ythrow yexception() << "GetActiveDownloadContainersLimit not implemented";
    }
    virtual ui32 GetActiveVerifyContainersLimit() const {
        ythrow yexception() << "GetActiveVerifyContainersLimit not implemented";
    }

    NStatusRepositoryTypes::EObjectType GetObjectType();

    // Container methods

    // Для Layer, StaticResource и BoxMetaContainer поддерживается только функции
    // UpdateContainerFailReason, IncrementContainerFailCounter и UpdateContainerState,
    // поскольку у download, verify и meta контейнеров нет собственного статуса, то будут вызваны
    // UpdateObjectFailedMessage и IncrementObjectFailCounter, а state проставляться не будет
    // Остальные функции кинут yexception

    virtual void IncrementContainerSystemFailureCounter(const NStatusRepositoryTypes::TContainerDescription& container);
    virtual void UpdateContainerFailReason(const NStatusRepositoryTypes::TContainerDescription& container, const TString& failReason);
    virtual void UpdateContainerState(const NStatusRepositoryTypes::TContainerDescription& container, API::EContainerState state);

    void ClearContainerStatus(const NStatusRepositoryTypes::TContainerDescription& container);

    TObjectState UpdateContainerOwnerState(const NStatusRepositoryTypes::TContainerDescription& container, const TObjectState& state);
    void UpdateContainerStderr(const NStatusRepositoryTypes::TContainerDescription& container, const TString& stderr);
    void UpdateContainerStdout(const NStatusRepositoryTypes::TContainerDescription& container, const TString& stdout);
    void UpdateContainerStartTime(const NStatusRepositoryTypes::TContainerDescription& container, const TInstant& time);
    void UpdateContainerDeathTime(const NStatusRepositoryTypes::TContainerDescription& container, const TInstant& time);
    void UpdateContainerReturnCode(const NStatusRepositoryTypes::TContainerDescription& container, i32 code);

    void IncrementContainerOomCounter(const NStatusRepositoryTypes::TContainerDescription& container);
    void IncrementContainerZeroReturnCodeCounter(const NStatusRepositoryTypes::TContainerDescription& container);
    void IncrementContainerPositiveReturnCodeCounter(const NStatusRepositoryTypes::TContainerDescription& container);
    void IncrementContainerKilledExternallyCounter(const NStatusRepositoryTypes::TContainerDescription& container);
    void IncrementContainerTimeoutCounter(const NStatusRepositoryTypes::TContainerDescription& container);
    void UpdateContainerConsecutiveFailuresAndSuccessesCounter(const NStatusRepositoryTypes::TContainerDescription& container, i32 code);

    API::TContainerStatus::TAttemptFeedback CaptureContainerStatus(const NStatusRepositoryTypes::TContainerDescription& container);

    TInstant GetContainerStartTime(const NStatusRepositoryTypes::TContainerDescription& container);

    ui32 GetContainerConsecutiveSuccessesCounter(const NStatusRepositoryTypes::TContainerDescription& container);
    ui32 GetContainerConsecutiveFailuresCounter(const NStatusRepositoryTypes::TContainerDescription& container);
    ui32 GetContainerZeroReturnCodeCounter(const NStatusRepositoryTypes::TContainerDescription& container);
    ui32 GetContainerPositiveReturnCodeCounter(const NStatusRepositoryTypes::TContainerDescription& container);
    ui32 GetContainerOomCounter(const NStatusRepositoryTypes::TContainerDescription& container);
    ui32 GetContainerKilledExternallyCounter(const NStatusRepositoryTypes::TContainerDescription& container);
    ui32 GetContainerSystemFailureCounter(const NStatusRepositoryTypes::TContainerDescription& container);

    TString GetContainerFailReason(const NStatusRepositoryTypes::TContainerDescription& container);

protected:
    // Container methods
    virtual const TLightRWLock& GetGlobalContainerLock() const {
        ythrow yexception() << "GetGlobalContainerLock not implemented";
    }

    virtual const TMutex& GetLocalContainerLock(const NStatusRepositoryTypes::TContainerDescription& /* container */) const {
        ythrow yexception() << "GetLocalContainerLock not implemented";
    }

    virtual API::TContainerStatus* GetMutableContainerStatus(const NStatusRepositoryTypes::TContainerDescription& /* container */) {
        ythrow yexception() << "GetMutableContainerStatus not implemented";
    }

    virtual const API::TContainerStatus* GetContainerStatus(const NStatusRepositoryTypes::TContainerDescription& /* container */) {
        ythrow yexception() << "GetContainerStatus not implemented";
    }

    void CheckContainerObjectType(const NStatusRepositoryTypes::TContainerDescription& container) const;

protected:
    const NStatusRepositoryTypes::EObjectType ObjectType_;
    inline const static TDuration ACTIVE_WAIT_TIME = TDuration::Minutes(1);
};

using TStatusRepositoryCommonPtr = TIntrusivePtr<TStatusRepositoryCommon>;

} // namespace NInfra::NPodAgent
