#include "status_repository_common.h"
#include "test_functions.h"

#include <google/protobuf/util/message_differencer.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NStatusRepositoryCommonTest {

// StatusRepository for for exactly one workload only with STOP container
class TTestStatusRepository: public TStatusRepositoryCommon {
public:
    TTestStatusRepository()
        : TStatusRepositoryCommon(NStatusRepositoryTypes::EObjectType::WORKLOAD)
    {}

    void UpdateSpecTimestamp(TInstant /* currentTime */) final {
    }

    void PatchTotalStatus(API::TPodAgentStatus& status, TUpdateHolderTargetPtr /* updateHolderTarget */, bool /* conditionsOnly */) const final {
        TReadGuardBase<TLightRWLock> guard(GlobalWorkloadLock_);
        API::TWorkloadStatus* st = status.add_workloads();
        TGuard<TMutex> g(WorkloadStatusMutex_);
        *st = WorkloadStatus_;
    }

    bool NeedLongTickPeriod(const TString& /* objectIdOrHash */, TUpdateHolderTargetPtr /* updateHolderTarget */) const final {
        return true;
    }

    TObjectState UpdateObjectState(const TString& /* objectIdOrHash */, TObjectState state) final {
        auto oldState = WorkloadStatus_.state();
        WorkloadStatus_.set_state(std::get<API::EWorkloadState>(state));
        return oldState;
    }

    static API::TWorkloadStatus GetWorkloadStatus(TStatusRepositoryCommonPtr statusRepository) {
        API::TPodAgentStatus status;
        statusRepository->PatchTotalStatus(status, nullptr, false);
        return status.workloads(0);
    }

    static API::TContainerStatus GetWorkloadStopContainerStatus(TStatusRepositoryCommonPtr statusRepository) {
        return GetWorkloadStatus(statusRepository).stop_status().container_status();
    }

protected:
    const TLightRWLock& GetGlobalContainerLock() const final {
        return GlobalWorkloadLock_;
    }

    const TMutex& GetLocalContainerLock(const NStatusRepositoryTypes::TContainerDescription& /* container */) const final {
        return WorkloadStatusMutex_;
    }

    API::TContainerStatus* GetMutableContainerStatus(const NStatusRepositoryTypes::TContainerDescription& /* container */) final {
        return WorkloadStatus_.mutable_stop_status()->mutable_container_status();
    }

    const API::TContainerStatus* GetContainerStatus(const NStatusRepositoryTypes::TContainerDescription& container) final {
        return GetMutableContainerStatus(container);
    }

protected:
    API::TWorkloadStatus WorkloadStatus_;
    TMutex WorkloadStatusMutex_;
    TLightRWLock GlobalWorkloadLock_;
};

Y_UNIT_TEST_SUITE(StatusRepositoryCommonSuite) {

Y_UNIT_TEST(TestContainerMethods) {
    TStatusRepositoryCommonPtr holder = new TTestStatusRepository();
    NStatusRepositoryTypes::TContainerDescription container(
        "my_workload"
        , NStatusRepositoryTypes::EObjectType::WORKLOAD
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::STOP
    );

    UNIT_ASSERT_EQUAL_C(holder->GetContainerConsecutiveSuccessesCounter(container), 0, holder->GetContainerConsecutiveSuccessesCounter(container));
    UNIT_ASSERT_EQUAL_C(holder->GetContainerConsecutiveFailuresCounter(container), 0, holder->GetContainerConsecutiveFailuresCounter(container));
    UNIT_ASSERT_EQUAL_C(holder->GetContainerZeroReturnCodeCounter(container), 0, holder->GetContainerZeroReturnCodeCounter(container));
    UNIT_ASSERT_EQUAL_C(holder->GetContainerPositiveReturnCodeCounter(container), 0, holder->GetContainerPositiveReturnCodeCounter(container));
    UNIT_ASSERT_EQUAL_C(holder->GetContainerOomCounter(container), 0, holder->GetContainerOomCounter(container));
    UNIT_ASSERT_EQUAL_C(holder->GetContainerKilledExternallyCounter(container), 0, holder->GetContainerKilledExternallyCounter(container));
    UNIT_ASSERT_EQUAL_C(holder->GetContainerSystemFailureCounter(container), 0, holder->GetContainerSystemFailureCounter(container));

    holder->UpdateContainerConsecutiveFailuresAndSuccessesCounter(container, 0);
    holder->IncrementContainerZeroReturnCodeCounter(container);
    holder->IncrementContainerPositiveReturnCodeCounter(container);
    holder->IncrementContainerOomCounter(container);
    holder->IncrementContainerKilledExternallyCounter(container);
    holder->IncrementContainerSystemFailureCounter(container);

    UNIT_ASSERT_EQUAL_C(holder->GetContainerConsecutiveSuccessesCounter(container), 1, holder->GetContainerConsecutiveSuccessesCounter(container));
    UNIT_ASSERT_EQUAL_C(holder->GetContainerConsecutiveFailuresCounter(container), 0, holder->GetContainerConsecutiveFailuresCounter(container));
    UNIT_ASSERT_EQUAL_C(holder->GetContainerZeroReturnCodeCounter(container), 1, holder->GetContainerZeroReturnCodeCounter(container));
    UNIT_ASSERT_EQUAL_C(holder->GetContainerPositiveReturnCodeCounter(container), 1, holder->GetContainerPositiveReturnCodeCounter(container));
    UNIT_ASSERT_EQUAL_C(holder->GetContainerOomCounter(container), 1, holder->GetContainerOomCounter(container));
    UNIT_ASSERT_EQUAL_C(holder->GetContainerKilledExternallyCounter(container), 1, holder->GetContainerKilledExternallyCounter(container));
    UNIT_ASSERT_EQUAL_C(holder->GetContainerSystemFailureCounter(container), 1, holder->GetContainerSystemFailureCounter(container));

    holder->UpdateContainerConsecutiveFailuresAndSuccessesCounter(container, 123456);

    UNIT_ASSERT_EQUAL_C(holder->GetContainerConsecutiveSuccessesCounter(container), 0, holder->GetContainerConsecutiveSuccessesCounter(container));
    UNIT_ASSERT_EQUAL_C(holder->GetContainerConsecutiveFailuresCounter(container), 1, holder->GetContainerConsecutiveFailuresCounter(container));

    holder->UpdateContainerConsecutiveFailuresAndSuccessesCounter(container, 123456);

    UNIT_ASSERT_EQUAL_C(holder->GetContainerConsecutiveSuccessesCounter(container), 0, holder->GetContainerConsecutiveSuccessesCounter(container));
    UNIT_ASSERT_EQUAL_C(holder->GetContainerConsecutiveFailuresCounter(container), 2, holder->GetContainerConsecutiveFailuresCounter(container));

    holder->UpdateContainerStartTime(container, TInstant::Max());
    holder->CaptureContainerStatus(container);

    UNIT_ASSERT_EQUAL(holder->GetContainerStartTime(container), TInstant::Max());
}

Y_UNIT_TEST(TestClearContainerStatus) {
    TStatusRepositoryCommonPtr holder = new TTestStatusRepository();
    NStatusRepositoryTypes::TContainerDescription container(
        "my_workload"
        , NStatusRepositoryTypes::EObjectType::WORKLOAD
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::STOP
    );

    holder->UpdateContainerOwnerState(container, API::EWorkloadState_ACTIVE);
    holder->UpdateContainerStdout(container, "test");
    holder->UpdateContainerStderr(container, "test");
    holder->UpdateContainerFailReason(container, "test");
    holder->UpdateContainerConsecutiveFailuresAndSuccessesCounter(container, 0);
    holder->UpdateContainerState(container, API::EContainerState_EXITED);

    holder->IncrementContainerOomCounter(container);
    holder->IncrementContainerKilledExternallyCounter(container);
    holder->IncrementContainerZeroReturnCodeCounter(container);
    holder->IncrementContainerPositiveReturnCodeCounter(container);
    holder->IncrementContainerTimeoutCounter(container);
    holder->IncrementContainerSystemFailureCounter(container);

    {
        API::TContainerStatus containerStatus = TTestStatusRepository::GetWorkloadStopContainerStatus(holder);

        UNIT_ASSERT_EQUAL(TTestStatusRepository::GetWorkloadStatus(holder).state(), API::EWorkloadState_ACTIVE);
        UNIT_ASSERT_EQUAL_C(containerStatus.current().stdout_tail(), "test", containerStatus.current().stdout_tail());
        UNIT_ASSERT_EQUAL_C(containerStatus.current().stderr_tail(), "test", containerStatus.current().stderr_tail());
        UNIT_ASSERT_EQUAL_C(containerStatus.time_limit().consecutive_successes_counter(), 1, containerStatus.time_limit().consecutive_successes_counter());
        UNIT_ASSERT_EQUAL_C(containerStatus.time_limit().consecutive_failures_counter(), 0, containerStatus.time_limit().consecutive_failures_counter());
        UNIT_ASSERT_EQUAL(containerStatus.current().state(), API::EContainerState_EXITED);

        UNIT_ASSERT_EQUAL_C(containerStatus.oom_counter(), 1, containerStatus.oom_counter());
        UNIT_ASSERT_EQUAL_C(containerStatus.killed_externally_counter(), 1, containerStatus.killed_externally_counter());
        UNIT_ASSERT_EQUAL_C(containerStatus.zero_return_code_counter(), 1, containerStatus.zero_return_code_counter());
        UNIT_ASSERT_EQUAL_C(containerStatus.positive_return_code_counter(), 1, containerStatus.positive_return_code_counter());
        UNIT_ASSERT_EQUAL_C(containerStatus.timeout_counter(), 1, containerStatus.timeout_counter());
        UNIT_ASSERT_EQUAL_C(containerStatus.system_failure_counter(), 1, containerStatus.system_failure_counter());
    }

    holder->ClearContainerStatus(container);

    {
        API::TContainerStatus containerStatus = TTestStatusRepository::GetWorkloadStopContainerStatus(holder);

        UNIT_ASSERT_EQUAL(TTestStatusRepository::GetWorkloadStatus(holder).state(), API::EWorkloadState_ACTIVE);
        UNIT_ASSERT_EQUAL_C(containerStatus.current().stdout_tail(), "", containerStatus.current().stdout_tail());
        UNIT_ASSERT_EQUAL_C(containerStatus.current().stderr_tail(), "", containerStatus.current().stderr_tail());
        UNIT_ASSERT_EQUAL_C(containerStatus.time_limit().consecutive_successes_counter(), 0, containerStatus.time_limit().consecutive_successes_counter());
        UNIT_ASSERT_EQUAL_C(containerStatus.time_limit().consecutive_failures_counter(), 0, containerStatus.time_limit().consecutive_failures_counter());
        UNIT_ASSERT_EQUAL(containerStatus.current().state(), API::EContainerState_UNKNOWN);

        UNIT_ASSERT_EQUAL_C(containerStatus.oom_counter(), 0, containerStatus.oom_counter());
        UNIT_ASSERT_EQUAL_C(containerStatus.killed_externally_counter(), 0, containerStatus.killed_externally_counter());
        UNIT_ASSERT_EQUAL_C(containerStatus.zero_return_code_counter(), 0, containerStatus.zero_return_code_counter());
        UNIT_ASSERT_EQUAL_C(containerStatus.positive_return_code_counter(), 0, containerStatus.positive_return_code_counter());
        UNIT_ASSERT_EQUAL_C(containerStatus.timeout_counter(), 0, containerStatus.timeout_counter());
        UNIT_ASSERT_EQUAL_C(containerStatus.system_failure_counter(), 0, containerStatus.system_failure_counter());
    }

    UNIT_ASSERT_EQUAL(holder->UpdateContainerOwnerState(container, API::EWorkloadState_REMOVED), TStatusRepositoryCommon::TObjectState(API::EWorkloadState_ACTIVE));
}

Y_UNIT_TEST(TestCheckContainerObjectType) {
    TStatusRepositoryCommonPtr holder = new TTestStatusRepository();
    NStatusRepositoryTypes::TContainerDescription container(
        "my_workload"
        , NStatusRepositoryTypes::EObjectType::BOX
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::STOP
    );

    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->UpdateContainerStartTime(container, TInstant::Max()), yexception, "Expected container object type");
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->UpdateContainerOwnerState(container, API::EWorkloadState_ACTIVE), yexception, "Expected container object type");
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->UpdateContainerStdout(container, "test"), yexception, "Expected container object type");
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->UpdateContainerStderr(container, "test"), yexception, "Expected container object type");
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->UpdateContainerFailReason(container, "test"), yexception, "Expected container object type");
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->UpdateContainerConsecutiveFailuresAndSuccessesCounter(container, 0), yexception, "Expected container object type");
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->UpdateContainerState(container, API::EContainerState_EXITED), yexception, "Expected container object type");

    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->IncrementContainerOomCounter(container), yexception, "Expected container object type");
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->IncrementContainerKilledExternallyCounter(container), yexception, "Expected container object type");
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->IncrementContainerZeroReturnCodeCounter(container), yexception, "Expected container object type");
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->IncrementContainerPositiveReturnCodeCounter(container), yexception, "Expected container object type");
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->IncrementContainerTimeoutCounter(container), yexception, "Expected container object type");
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->IncrementContainerSystemFailureCounter(container), yexception, "Expected container object type");

    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->GetContainerConsecutiveSuccessesCounter(container), yexception, "Expected container object type");
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->GetContainerConsecutiveFailuresCounter(container), yexception, "Expected container object type");
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->GetContainerZeroReturnCodeCounter(container), yexception, "Expected container object type");
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->GetContainerPositiveReturnCodeCounter(container), yexception, "Expected container object type");
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->GetContainerOomCounter(container), yexception, "Expected container object type");
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->GetContainerKilledExternallyCounter(container), yexception, "Expected container object type");
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->GetContainerStartTime(container), yexception, "Expected container object type");

    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->CaptureContainerStatus(container), yexception, "Expected container object type");
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->ClearContainerStatus(container), yexception, "Expected container object type");
}

Y_UNIT_TEST(TestCaptureContainerStatus) {
    auto test = [](
        std::function<
            void(
                TStatusRepositoryCommonPtr& holder
                , const NStatusRepositoryTypes::TContainerDescription& container
            )
        > initCurrentAttempt
        , bool isFailedAttempt
    ) {
        TStatusRepositoryCommonPtr holder = new TTestStatusRepository();
        const NStatusRepositoryTypes::TContainerDescription::EContainerType containerType = NStatusRepositoryTypes::TContainerDescription::EContainerType::STOP;
        NStatusRepositoryTypes::TContainerDescription container(
            "my_workload"
            , NStatusRepositoryTypes::EObjectType::WORKLOAD
            , containerType
        );

        initCurrentAttempt(holder, container);

        auto workloadStatus = TTestStatusRepository::GetWorkloadStatus(holder);
        auto curStatus = GetWorkloadContainerHookStatus(workloadStatus, containerType).current();
        auto lastStatus = GetWorkloadContainerHookStatus(workloadStatus, containerType).last();
        auto lastFailedStatus = GetWorkloadContainerHookStatus(workloadStatus, containerType).last_failed();
        API::TContainerStatus::TAttemptFeedback emptyStatus;

        UNIT_ASSERT(!google::protobuf::util::MessageDifferencer::Equals(curStatus, lastStatus));
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(emptyStatus, lastStatus));
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(emptyStatus, lastFailedStatus));

        // First iteration check that capture work as expected
        // Second iteration check that empty current doesn't push to last
        for (size_t i = 0; i < 2; ++i) {
            holder->CaptureContainerStatus(container);

            workloadStatus = TTestStatusRepository::GetWorkloadStatus(holder);
            auto newCurStatus = GetWorkloadContainerHookStatus(workloadStatus, containerType).current();
            auto newLastStatus = GetWorkloadContainerHookStatus(workloadStatus, containerType).last();
            auto newLastFailedStatus = GetWorkloadContainerHookStatus(workloadStatus, containerType).last_failed();

            UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(emptyStatus, newCurStatus));
            UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(curStatus, newLastStatus));
            UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(isFailedAttempt ? curStatus : emptyStatus, newLastFailedStatus));
        }
    };

    test(
        [](
            TStatusRepositoryCommonPtr& holder
            , const NStatusRepositoryTypes::TContainerDescription& container
        ) {
            holder->UpdateContainerState(container, API::EContainerState_EXITED);
            holder->UpdateContainerFailReason(container, "fail_reason");
            holder->UpdateContainerStderr(container, "stderr_text");
            holder->UpdateContainerStdout(container, "stdout_text");
            holder->UpdateContainerReturnCode(container, 0);
        }
        , false
    );

    test(
        [](
            TStatusRepositoryCommonPtr& holder
            , const NStatusRepositoryTypes::TContainerDescription& container
        ) {
            holder->UpdateContainerState(container, API::EContainerState_EXITED);
            holder->UpdateContainerReturnCode(container, 123);
        }
        , true
    );

    test(
        [](
            TStatusRepositoryCommonPtr& holder
            , const NStatusRepositoryTypes::TContainerDescription& container
        ) {
            holder->UpdateContainerState(container, API::EContainerState_TIMEOUT);
            holder->UpdateContainerReturnCode(container, 0);
        }
        , true
    );
}

}

} // namespace NInfra::NPodAgent::NStatusRepositoryCommonTest

