#include "status_repository.h"
#include "test_functions.h"

#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NStatusRepositoryTest {

Y_UNIT_TEST_SUITE(StatusRepositorySuite) {

Y_UNIT_TEST(TestSetAndGet) {
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    TBoxStatusRepositoryPtr boxStatusRepository = new TBoxStatusRepository();
    TLayerStatusRepositoryPtr layerStatusRepository = new TLayerStatusRepository();
    TStaticResourceStatusRepositoryPtr staticResourceStatusRepository = new TStaticResourceStatusRepository();
    TVolumeStatusRepositoryPtr volumeStatusRepository = new TVolumeStatusRepository();
    TWorkloadStatusRepositoryPtr workloadStatusRepository = new TWorkloadStatusRepository();
    TStatusRepositoryPtr holder = new TStatusRepository(
        updateHolder
        , boxStatusRepository
        , layerStatusRepository
        , staticResourceStatusRepository
        , volumeStatusRepository
        , workloadStatusRepository
    );

    const ui64 specTimestamp = 41;
    const ui32 revision = 42;
    const TString id = "id";
    const API::EPodAgentTargetState targetState = API::EPodAgentTargetState_ACTIVE;
    holder->SetSpecTimestamp(specTimestamp);
    holder->SetRevision(revision);
    holder->SetSpecId(id);
    holder->SetTargetState(targetState);

    API::TPodAgentStatus status = holder->GetTotalStatus(false);
    UNIT_ASSERT_EQUAL(specTimestamp, status.spec_timestamp());
    UNIT_ASSERT_EQUAL(revision, status.revision());
    UNIT_ASSERT_EQUAL(id, status.id());
    UNIT_ASSERT_EQUAL(id, holder->GetSpecId());
    UNIT_ASSERT_EQUAL(targetState, status.target_state());

    UNIT_ASSERT_EQUAL(updateHolder, holder->GetUpdateHolder());
    UNIT_ASSERT_EQUAL(boxStatusRepository, holder->GetBoxStatusRepository());
    UNIT_ASSERT_EQUAL(layerStatusRepository, holder->GetLayerStatusRepository());
    UNIT_ASSERT_EQUAL(staticResourceStatusRepository, holder->GetStaticResourceStatusRepository());
    UNIT_ASSERT_EQUAL(volumeStatusRepository, holder->GetVolumeStatusRepository());
    UNIT_ASSERT_EQUAL(workloadStatusRepository, holder->GetWorkloadStatusRepository());
}

Y_UNIT_TEST(TestPodAgentStatusConditions) {
    for (ui32 conditionsOnly = 0; conditionsOnly < 2; ++conditionsOnly) {
        TUpdateHolderPtr updateHolder = new TUpdateHolder();
        TBoxStatusRepositoryPtr boxStatusRepository = new TBoxStatusRepository();
        TLayerStatusRepositoryPtr layerStatusRepository = new TLayerStatusRepository();
        TStaticResourceStatusRepositoryPtr staticResourceStatusRepository = new TStaticResourceStatusRepository();
        TVolumeStatusRepositoryPtr volumeStatusRepository = new TVolumeStatusRepository();
        TWorkloadStatusRepositoryPtr workloadStatusRepository = new TWorkloadStatusRepository();
        TStatusRepositoryPtr holder = new TStatusRepository(
            updateHolder
            , boxStatusRepository
            , layerStatusRepository
            , staticResourceStatusRepository
            , volumeStatusRepository
            , workloadStatusRepository
        );

        const TString workloadId = "my_workload";
        const TString boxId = "my_box";
        workloadStatusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));
        const TString cacheLayerId = "my_cache_layer";
        const TString cacheLayerDownloadHash = "my_cache_layer_download_hash";
        layerStatusRepository->AddCacheObject(NObjectMetaTestLib::CreateLayerMetaSimple(cacheLayerId, cacheLayerDownloadHash));
        const TString layerId = "my_layer";
        const TString layerDownloadHash = "my_layer_download_hash";
        layerStatusRepository->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(layerId, layerDownloadHash));
        const TString cacheStaticResourceId = "my_cache_static_resource";
        const TString cacheStaticResourceDownloadHash = "my_cache_static_resource_download_hash";
        staticResourceStatusRepository->AddCacheObject(NObjectMetaTestLib::CreateStaticResourceMetaSimple(cacheStaticResourceId, cacheStaticResourceDownloadHash));
        const TString staticResourceId = "my_static_resource";
        const TString staticResourceDownloadHash = "my_static_resource_download_hash";
        staticResourceStatusRepository->AddObject(NObjectMetaTestLib::CreateStaticResourceMetaSimple(staticResourceId, staticResourceDownloadHash));
        const TString boxId1 = "my_first_box";
        boxStatusRepository->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(boxId1));
        const TString boxId2 = "my_second_box";
        boxStatusRepository->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(boxId2));
        const TString volumeId = "my_volume";
        volumeStatusRepository->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple(volumeId));

        auto validateConditionsOnly = [](const API::TPodAgentStatus& status, bool conditionsOnly) -> void {
            if (conditionsOnly) {
                UNIT_ASSERT_EQUAL_C(
                    status.boxes(0).state()
                    , API::EBoxState_UNKNOWN
                    , API::EBoxState_Name(status.boxes(0).state())
                );

                UNIT_ASSERT_EQUAL_C(
                    status.resource_gang().layers(0).state()
                    , API::ELayerState_UNKNOWN
                    , API::ELayerState_Name(status.resource_gang().layers(0).state())
                );
                UNIT_ASSERT_EQUAL_C(
                    status.resource_gang().layers(0).state()
                    , API::ELayerState_UNKNOWN
                    , API::ELayerState_Name(status.resource_cache().layers(0).state())
                );

                UNIT_ASSERT_EQUAL_C(
                    status.resource_gang().static_resources(0).state()
                    , API::EStaticResourceState_UNKNOWN
                    , API::EStaticResourceState_Name(status.resource_gang().static_resources(0).state())
                );
                UNIT_ASSERT_EQUAL_C(
                    status.resource_gang().static_resources(0).state()
                    , API::EStaticResourceState_UNKNOWN
                    , API::EStaticResourceState_Name(status.resource_cache().static_resources(0).state())
                );

                UNIT_ASSERT_EQUAL_C(
                    status.volumes(0).state()
                    , API::EVolumeState_UNKNOWN
                    , API::EVolumeState_Name(status.volumes(0).state())
                );

                UNIT_ASSERT_EQUAL_C(
                    status.workloads(0).state()
                    , API::EWorkloadState_UNKNOWN
                    , API::EWorkloadState_Name(status.workloads(0).state())
                );
            }
        };

        workloadStatusRepository->UpdateObjectState(workloadId, API::EWorkloadState_UNKNOWN);
        layerStatusRepository->UpdateObjectState(cacheLayerDownloadHash, API::ELayerState_UNKNOWN);
        layerStatusRepository->UpdateObjectState(layerDownloadHash, API::ELayerState_UNKNOWN);
        staticResourceStatusRepository->UpdateObjectState(cacheStaticResourceDownloadHash, API::EStaticResourceState_UNKNOWN);
        staticResourceStatusRepository->UpdateObjectState(staticResourceDownloadHash, API::EStaticResourceState_UNKNOWN);
        boxStatusRepository->UpdateObjectState(boxId1, API::EBoxState_UNKNOWN);
        boxStatusRepository->UpdateObjectState(boxId2, API::EBoxState_UNKNOWN);
        volumeStatusRepository->UpdateObjectState(volumeId, API::EVolumeState_UNKNOWN);
        API::TPodAgentStatus status = holder->GetTotalStatus((bool)conditionsOnly);
        auto ready = status.ready();
        auto inProgress = status.in_progress();
        auto failed = status.failed();

        validateConditionsOnly(status, (bool)conditionsOnly);
        UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_UNKNOWN);
        UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_UNKNOWN);
        UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_UNKNOWN);
        UNIT_ASSERT_EQUAL(ready.last_transition_time().seconds(), 0);
        UNIT_ASSERT_EQUAL(ready.last_transition_time().nanos(), 0);
        UNIT_ASSERT_EQUAL(inProgress.last_transition_time().seconds(), 0);
        UNIT_ASSERT_EQUAL(inProgress.last_transition_time().nanos(), 0);
        UNIT_ASSERT_EQUAL(failed.last_transition_time().seconds(), 0);
        UNIT_ASSERT_EQUAL(failed.last_transition_time().nanos(), 0);

        workloadStatusRepository->UpdateObjectState(workloadId, API::EWorkloadState_ACTIVE);
        layerStatusRepository->UpdateObjectState(cacheLayerDownloadHash, API::ELayerState_READY);
        layerStatusRepository->UpdateObjectState(layerDownloadHash, API::ELayerState_READY);
        boxStatusRepository->UpdateObjectState(boxId1, API::EBoxState_READY);
        boxStatusRepository->UpdateObjectState(boxId2, API::EBoxState_LINKING_VOLUMES);
        volumeStatusRepository->UpdateObjectState(volumeId, API::EVolumeState_UNKNOWN);
        status = holder->GetTotalStatus((bool)conditionsOnly);
        ready = status.ready();
        inProgress = status.in_progress();
        failed = status.failed();

        validateConditionsOnly(status, (bool)conditionsOnly);
        UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_TRUE);
        UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_UNKNOWN);
        UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_UNKNOWN);

        workloadStatusRepository->UpdateObjectState(workloadId, API::EWorkloadState_ACTIVE);
        layerStatusRepository->UpdateObjectState(cacheLayerDownloadHash, API::ELayerState_READY);
        layerStatusRepository->UpdateObjectState(layerDownloadHash, API::ELayerState_READY);
        staticResourceStatusRepository->UpdateObjectState(cacheStaticResourceDownloadHash, API::EStaticResourceState_READY);
        staticResourceStatusRepository->UpdateObjectState(staticResourceDownloadHash, API::EStaticResourceState_READY);
        boxStatusRepository->UpdateObjectState(boxId1, API::EBoxState_READY);
        boxStatusRepository->UpdateObjectState(boxId2, API::EBoxState_READY);
        volumeStatusRepository->UpdateObjectState(volumeId, API::EVolumeState_CREATING);
        status = holder->GetTotalStatus((bool)conditionsOnly);
        ready = status.ready();
        inProgress = status.in_progress();
        failed = status.failed();

        validateConditionsOnly(status, (bool)conditionsOnly);
        UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_TRUE);
        UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_TRUE);
        UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_FALSE);
        UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
        UNIT_ASSERT(inProgress.last_transition_time().seconds() != 0 || inProgress.last_transition_time().nanos() != 0);
        UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);

        workloadStatusRepository->UpdateObjectState(workloadId, API::EWorkloadState_ACTIVE);
        layerStatusRepository->UpdateObjectState(cacheLayerDownloadHash, API::ELayerState_READY);
        layerStatusRepository->UpdateObjectState(layerDownloadHash, API::ELayerState_READY);
        staticResourceStatusRepository->UpdateObjectState(cacheStaticResourceDownloadHash, API::EStaticResourceState_READY);
        staticResourceStatusRepository->UpdateObjectState(staticResourceDownloadHash, API::EStaticResourceState_READY);
        boxStatusRepository->UpdateObjectState(boxId1, API::EBoxState_READY);
        boxStatusRepository->UpdateObjectState(boxId2, API::EBoxState_WAITING_FOR_ROOTFS_LAYERS);
        volumeStatusRepository->UpdateObjectState(volumeId, API::EVolumeState_READY);
        status = holder->GetTotalStatus((bool)conditionsOnly);
        ready = status.ready();
        inProgress = status.in_progress();
        failed = status.failed();

        validateConditionsOnly(status, (bool)conditionsOnly);
        UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_TRUE);
        UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_TRUE);
        UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_FALSE);
        UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
        UNIT_ASSERT(inProgress.last_transition_time().seconds() != 0 || inProgress.last_transition_time().nanos() != 0);
        UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);

        workloadStatusRepository->UpdateObjectState(workloadId, API::EWorkloadState_ACTIVE);
        layerStatusRepository->UpdateObjectState(cacheLayerDownloadHash, API::ELayerState_READY);
        layerStatusRepository->UpdateObjectState(layerDownloadHash, API::ELayerState_VERIFYING);
        staticResourceStatusRepository->UpdateObjectState(cacheStaticResourceDownloadHash, API::EStaticResourceState_READY);
        staticResourceStatusRepository->UpdateObjectState(staticResourceDownloadHash, API::EStaticResourceState_VERIFYING);
        boxStatusRepository->UpdateObjectState(boxId1, API::EBoxState_READY);
        boxStatusRepository->UpdateObjectState(boxId2, API::EBoxState_READY);
        volumeStatusRepository->UpdateObjectState(volumeId, API::EVolumeState_READY);
        status = holder->GetTotalStatus((bool)conditionsOnly);
        ready = status.ready();
        inProgress = status.in_progress();
        failed = status.failed();

        validateConditionsOnly(status, (bool)conditionsOnly);
        UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_TRUE);
        UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_TRUE);
        UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_FALSE);
        UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
        UNIT_ASSERT(inProgress.last_transition_time().seconds() != 0 || inProgress.last_transition_time().nanos() != 0);
        UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);

        workloadStatusRepository->UpdateObjectState(workloadId, API::EWorkloadState_ACTIVE);
        layerStatusRepository->UpdateObjectState(cacheLayerDownloadHash, API::ELayerState_VERIFYING);
        layerStatusRepository->UpdateObjectState(layerDownloadHash, API::ELayerState_READY);
        staticResourceStatusRepository->UpdateObjectState(cacheStaticResourceDownloadHash, API::EStaticResourceState_READY);
        staticResourceStatusRepository->UpdateObjectState(staticResourceDownloadHash, API::EStaticResourceState_READY);
        boxStatusRepository->UpdateObjectState(boxId1, API::EBoxState_READY);
        boxStatusRepository->UpdateObjectState(boxId2, API::EBoxState_READY);
        volumeStatusRepository->UpdateObjectState(volumeId, API::EVolumeState_READY);
        status = holder->GetTotalStatus((bool)conditionsOnly);
        ready = status.ready();
        inProgress = status.in_progress();
        failed = status.failed();

        validateConditionsOnly(status, (bool)conditionsOnly);
        UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_TRUE);
        UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_TRUE);
        UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_FALSE);
        UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
        UNIT_ASSERT(inProgress.last_transition_time().seconds() != 0 || inProgress.last_transition_time().nanos() != 0);
        UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);

        workloadStatusRepository->UpdateObjectState(workloadId, API::EWorkloadState_ACTIVE);
        layerStatusRepository->UpdateObjectState(cacheLayerDownloadHash, API::ELayerState_READY);
        layerStatusRepository->UpdateObjectState(layerDownloadHash, API::ELayerState_READY);
        staticResourceStatusRepository->UpdateObjectState(cacheStaticResourceDownloadHash, API::EStaticResourceState_VERIFYING);
        staticResourceStatusRepository->UpdateObjectState(staticResourceDownloadHash, API::EStaticResourceState_READY);
        boxStatusRepository->UpdateObjectState(boxId1, API::EBoxState_READY);
        boxStatusRepository->UpdateObjectState(boxId2, API::EBoxState_READY);
        volumeStatusRepository->UpdateObjectState(volumeId, API::EVolumeState_READY);
        status = holder->GetTotalStatus((bool)conditionsOnly);
        ready = status.ready();
        inProgress = status.in_progress();
        failed = status.failed();

        validateConditionsOnly(status, (bool)conditionsOnly);
        UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_TRUE);
        UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_TRUE);
        UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_FALSE);
        UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
        UNIT_ASSERT(inProgress.last_transition_time().seconds() != 0 || inProgress.last_transition_time().nanos() != 0);
        UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);

        workloadStatusRepository->UpdateObjectState(workloadId, API::EWorkloadState_ACTIVATING);
        layerStatusRepository->UpdateObjectState(cacheLayerDownloadHash, API::ELayerState_READY);
        layerStatusRepository->UpdateObjectState(layerDownloadHash, API::ELayerState_READY);
        staticResourceStatusRepository->UpdateObjectState(cacheStaticResourceDownloadHash, API::EStaticResourceState_READY);
        staticResourceStatusRepository->UpdateObjectState(staticResourceDownloadHash, API::EStaticResourceState_READY);
        boxStatusRepository->UpdateObjectState(boxId1, API::EBoxState_READY);
        boxStatusRepository->UpdateObjectState(boxId2, API::EBoxState_READY);
        volumeStatusRepository->UpdateObjectState(volumeId, API::EVolumeState_READY);
        status = holder->GetTotalStatus((bool)conditionsOnly);
        ready = status.ready();
        inProgress = status.in_progress();
        failed = status.failed();

        validateConditionsOnly(status, (bool)conditionsOnly);
        UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_FALSE);
        UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_TRUE);
        UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_FALSE);
        UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
        UNIT_ASSERT(inProgress.last_transition_time().seconds() != 0 || inProgress.last_transition_time().nanos() != 0);
        UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);

        workloadStatusRepository->UpdateObjectState(workloadId, API::EWorkloadState_ACTIVATING);
        layerStatusRepository->UpdateObjectState(cacheLayerDownloadHash, API::ELayerState_READY);
        layerStatusRepository->UpdateObjectState(layerDownloadHash, API::ELayerState_READY);
        staticResourceStatusRepository->UpdateObjectState(cacheStaticResourceDownloadHash, API::EStaticResourceState_READY);
        staticResourceStatusRepository->UpdateObjectState(staticResourceDownloadHash, API::EStaticResourceState_READY);
        boxStatusRepository->UpdateObjectState(boxId1, API::EBoxState_READY);
        boxStatusRepository->UpdateObjectState(boxId2, API::EBoxState_READY);
        volumeStatusRepository->UpdateObjectState(volumeId, API::EVolumeState_READY);
        status = holder->GetTotalStatus((bool)conditionsOnly);
        ready = status.ready();
        inProgress = status.in_progress();
        failed = status.failed();

        validateConditionsOnly(status, (bool)conditionsOnly);
        UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_FALSE);
        UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_TRUE);
        UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_FALSE);
        UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
        UNIT_ASSERT(inProgress.last_transition_time().seconds() != 0 || inProgress.last_transition_time().nanos() != 0);
        UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);

        workloadStatusRepository->UpdateObjectState(workloadId, API::EWorkloadState_ACTIVE);
        layerStatusRepository->UpdateObjectState(cacheLayerDownloadHash, API::ELayerState_READY);
        layerStatusRepository->UpdateObjectState(layerDownloadHash, API::ELayerState_READY);
        staticResourceStatusRepository->UpdateObjectState(cacheStaticResourceDownloadHash, API::EStaticResourceState_READY);
        staticResourceStatusRepository->UpdateObjectState(staticResourceDownloadHash, API::EStaticResourceState_READY);
        boxStatusRepository->UpdateObjectState(boxId1, API::EBoxState_READY);
        boxStatusRepository->UpdateObjectState(boxId2, API::EBoxState_INVALID);
        volumeStatusRepository->UpdateObjectState(volumeId, API::EVolumeState_READY);
        status = holder->GetTotalStatus((bool)conditionsOnly);
        ready = status.ready();
        inProgress = status.in_progress();
        failed = status.failed();

        validateConditionsOnly(status, (bool)conditionsOnly);
        UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_TRUE);
        UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_FALSE);
        UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_TRUE);
        UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
        UNIT_ASSERT(inProgress.last_transition_time().seconds() != 0 || inProgress.last_transition_time().nanos() != 0);
        UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);

        workloadStatusRepository->UpdateObjectState(workloadId, API::EWorkloadState_ACTIVE);
        layerStatusRepository->UpdateObjectState(cacheLayerDownloadHash, API::ELayerState_READY);
        layerStatusRepository->UpdateObjectState(layerDownloadHash, API::ELayerState_READY);
        staticResourceStatusRepository->UpdateObjectState(cacheStaticResourceDownloadHash, API::EStaticResourceState_READY);
        staticResourceStatusRepository->UpdateObjectState(staticResourceDownloadHash, API::EStaticResourceState_READY);
        boxStatusRepository->UpdateObjectState(boxId1, API::EBoxState_READY);
        boxStatusRepository->UpdateObjectState(boxId2, API::EBoxState_READY);
        volumeStatusRepository->UpdateObjectState(volumeId, API::EVolumeState_READY);
        status = holder->GetTotalStatus((bool)conditionsOnly);
        ready = status.ready();
        inProgress = status.in_progress();
        failed = status.failed();

        validateConditionsOnly(status, (bool)conditionsOnly);
        UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_TRUE);
        UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_FALSE);
        UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_FALSE);
        UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
        UNIT_ASSERT(inProgress.last_transition_time().seconds() != 0 || inProgress.last_transition_time().nanos() != 0);
        UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);
    }
}

Y_UNIT_TEST(TestGetTotalStatusPerformance) {
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    TBoxStatusRepositoryPtr boxStatusRepository = new TBoxStatusRepository();
    TLayerStatusRepositoryPtr layerStatusRepository = new TLayerStatusRepository();
    TStaticResourceStatusRepositoryPtr staticResourceStatusRepository = new TStaticResourceStatusRepository();
    TVolumeStatusRepositoryPtr volumeStatusRepository = new TVolumeStatusRepository();
    TWorkloadStatusRepositoryPtr workloadStatusRepository = new TWorkloadStatusRepository();
    TStatusRepositoryPtr holder = new TStatusRepository(
        updateHolder
        , boxStatusRepository
        , layerStatusRepository
        , staticResourceStatusRepository
        , volumeStatusRepository
        , workloadStatusRepository
    );

    // Generate status for default pod
    const ui32 boxes = 4;
    const ui32 layersPerBox = 10;
    const ui32 staticResourcesPerBox = 10;
    const ui32 volumesPerBox = 10;
    const ui32 workloadsPerBox = 5;

    const ui32 defaultMessageLen = 900;
    const TString defaultMessage = TString(defaultMessageLen - 1, 'a') + "_";

    const TVector<NStatusRepositoryTypes::TContainerDescription::EContainerType> workloadContainerTypes = {
        NStatusRepositoryTypes::TContainerDescription::EContainerType::START
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::READINESS
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::LIVENESS
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::STOP
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::DESTROY
    };

    for (ui32 boxId = 0; boxId < boxes; ++boxId) {
        const TString boxIdStr = "box_" + ToString(boxId);

        TVector<TString> layerRefs;
        for (ui32 layerId = 0; layerId < layersPerBox; ++layerId) {
            const TString layerIdStr = "layer_" + ToString(boxId) + ToString(layerId);
            const TString layerDownloadHash = layerIdStr + "_downloadHash";
            layerRefs.push_back(layerIdStr);

            layerStatusRepository->AddObject(
                NObjectMetaTestLib::CreateLayerMetaSimple(
                    layerIdStr
                    , layerDownloadHash
                )
            );

            layerStatusRepository->UpdateObjectFailedMessage(layerDownloadHash, defaultMessage + layerIdStr);
        }

        TVector<TString> staticResourceRefs;
        for (ui32 staticResourceId = 0; staticResourceId < staticResourcesPerBox; ++staticResourceId) {
            const TString staticResourceIdStr = "static_resource_" + ToString(boxId) + ToString(staticResourceId);
            const TString staticResourceDownloadHash = staticResourceIdStr + "_downloadHash";
            staticResourceRefs.push_back(staticResourceIdStr);

            staticResourceStatusRepository->AddObject(
                NObjectMetaTestLib::CreateStaticResourceMetaSimple(
                    staticResourceIdStr
                    , staticResourceDownloadHash
                )
            );

            staticResourceStatusRepository->UpdateObjectFailedMessage(staticResourceDownloadHash, defaultMessage + staticResourceIdStr);
        }

        TVector<TString> volumeRefs;
        for (ui32 volumeId = 0; volumeId < volumesPerBox; ++volumeId) {
            const TString volumeIdStr = "volume_" + ToString(boxId) + ToString(volumeId);
            volumeRefs.push_back(volumeIdStr);

            volumeStatusRepository->AddObject(
                NObjectMetaTestLib::CreateVolumeMetaWithDependence(
                    volumeIdStr
                    , layerRefs
                    , staticResourceRefs
                )
            );

            volumeStatusRepository->UpdateObjectFailedMessage(volumeIdStr, defaultMessage + boxIdStr);
        }

        {
            boxStatusRepository->AddObject(
                NObjectMetaTestLib::CreateBoxMetaWithDependence(
                    boxIdStr
                    , layerRefs
                    , staticResourceRefs
                    , volumeRefs
                    , 3
                )
            );

            boxStatusRepository->UpdateObjectFailedMessage(boxIdStr, defaultMessage);
        }

        for (ui32 workloadId = 0; workloadId < workloadsPerBox; ++workloadId) {
            const TString workloadIdStr = "workload_" + ToString(boxId) + "_" + ToString(workloadId);

            workloadStatusRepository->AddObject(
                NObjectMetaTestLib::CreateWorkloadMetaWithDependence(
                    workloadIdStr
                    , boxIdStr
                    , 1
                    , 3
                    , true
                )
            );

            for (const auto& containerType : workloadContainerTypes) {
                workloadStatusRepository->UpdateContainerFailReason(
                    NStatusRepositoryTypes::TContainerDescription(
                        workloadIdStr
                        , NStatusRepositoryTypes::EObjectType::WORKLOAD
                        , containerType
                    )
                    , defaultMessage + workloadIdStr + "_" + ToString(containerType)
                );
            }
        }
    }

    const ui32 cntIter = 3000;
    for (ui32 conditionsOnly = 0; conditionsOnly < 2; ++conditionsOnly) {
        TInstant startTime = TInstant::Now();
        ui32 maxMessageSize = 0;
        for (ui32 i = 0; i < cntIter; ++i) {
            maxMessageSize = Max(maxMessageSize, (ui32)holder->GetTotalStatus((bool)conditionsOnly).ByteSizeLong());
        }
        TInstant endTime = TInstant::Now();
        double timePerGet = (double)(endTime - startTime).MilliSeconds() / cntIter;

        const TString metricPrefix = conditionsOnly
            ? "conditions_only_"
            : "full_status_"
        ;

        if (conditionsOnly) {
            // Double margin check at the time of writing the test
            UNIT_ASSERT_C(maxMessageSize < 15 * 1000, maxMessageSize);
        }

        UNIT_ADD_METRIC(metricPrefix + "time_per_get", timePerGet);
        UNIT_ADD_METRIC(metricPrefix + "max_message_size_bytes", maxMessageSize);
    }
}

}

} // namespace NInfra::NPodAgent::NStatusRepositoryTest
