#include "support_functions.h"

#include <infra/pod_agent/libs/common/common.h>
#include <util/string/builder.h>

namespace NInfra::NPodAgent::NSupport {

TUnistatObjectHelper::EConditionStatus ConvertConditionStatus(API::EConditionStatus conditionStatus) {
    switch (conditionStatus) {
        case API::EConditionStatus_UNKNOWN:
            return TUnistatObjectHelper::EConditionStatus::UNKNOWN;
        case API::EConditionStatus_TRUE:
            return TUnistatObjectHelper::EConditionStatus::TRUE;
        case API::EConditionStatus_FALSE:
            return TUnistatObjectHelper::EConditionStatus::FALSE;
        // We don't want to use default
        // So we need this to handle all cases
        case API::EConditionStatus_INT_MIN_SENTINEL_DO_NOT_USE_:
        case API::EConditionStatus_INT_MAX_SENTINEL_DO_NOT_USE_:
            return TUnistatObjectHelper::EConditionStatus::UNKNOWN;
    }
}

void PushAllPodSignals(const API::TPodAgentStatus& status) {
    TUnistatObjectHelper::Instance().PushPodConditionSignal(
        TUnistatObjectHelper::EConditionSignalType::READY
        , ConvertConditionStatus(status.ready().status())
    );
    TUnistatObjectHelper::Instance().PushPodConditionSignal(
        TUnistatObjectHelper::EConditionSignalType::IN_PROGRESS
        , ConvertConditionStatus(status.in_progress().status())
    );
    TUnistatObjectHelper::Instance().PushPodConditionSignal(
        TUnistatObjectHelper::EConditionSignalType::FAILED
        , ConvertConditionStatus(status.failed().status())
    );
}

TInstant ToInstant(const google::protobuf::Timestamp& timestamp) {
    return TInstant::MicroSeconds(TimestampToMicros(timestamp));
}

TString Truncate(const TString& stringToTruncate) {
    size_t size = stringToTruncate.size();
    if (size <= 2 * MAX_TRUNCATE_SIZE + 3) {
        return stringToTruncate;
    }

    return TStringBuilder() << stringToTruncate.substr(0, MAX_TRUNCATE_SIZE) << "..." << stringToTruncate.substr(size - MAX_TRUNCATE_SIZE, MAX_TRUNCATE_SIZE);
}

TString ExtractState(const TString& state) {
    size_t lengthOfPrefix = strlen("State");
    size_t firstUnderline = state.find_first_of('_');

    // Ensures that the argument is correct
    Y_ENSURE(firstUnderline != TString::npos, "invalid argument, cannot find _ in '" << state << "'");
    Y_ENSURE(state[0] == 'E', "invalid argument, first character of '"<< state <<"' is not equal to E");
    Y_ENSURE(firstUnderline >= lengthOfPrefix && state.substr(firstUnderline - lengthOfPrefix, lengthOfPrefix) == "State", "invalid argument, cannot find 'State' before underline in '" << state << "'");

    return state.substr(firstUnderline + 1);
}

void SetApplyingObjectSpecState(API::TCondition& condition) {
    condition.set_reason(APPLYING_OBJECT_SPEC);
}

google::protobuf::Timestamp ToTimestamp(const TInstant& time) {
    google::protobuf::Timestamp timestamp;
    timestamp.set_seconds(time.Seconds());
    timestamp.set_nanos(time.NanoSecondsOfSecond());
    return timestamp;
}

void CopyConditonStatusAndTimestamp(API::TCondition& target, const API::TCondition& source) {
    target.set_status(source.status());
    target.mutable_last_transition_time()->CopyFrom(source.last_transition_time());
}

} // namespace NInfra::NPodAgent::NSupport
