#pragma once

#include <infra/pod_agent/libs/service_iface/protos/service.pb.h>

#include <infra/pod_agent/libs/pod_agent/unistat_object_helper/unistat_object_helper.h>

#include <google/protobuf/timestamp.pb.h>
#include <util/datetime/base.h>

namespace NInfra::NPodAgent::NSupport {

const size_t MAX_TRUNCATE_SIZE = 512; // Half of the maximum size of the string in Truncate function
const TString APPLYING_OBJECT_SPEC = "APPLYING_OBJECT_SPEC";
const TString APPLYING_OBJECT_SPEC_MESSAGE = "Changing spec_timestamp";

template<API::EConditionStatus STRONG>
void PatchCondition(API::TCondition* target, const API::TCondition& correction) {
    google::protobuf::Timestamp oldTime = target->last_transition_time();
    switch (correction.status()) {
        case API::EConditionStatus_UNKNOWN:
            *target = correction;
            break;
        case STRONG:
            switch (target->status()) {
                case API::EConditionStatus_UNKNOWN:
                case STRONG:
                    break;
                default:
                    *target = correction;
                    break;
            }
            break;
        default:
            break;
    }
    *target->mutable_last_transition_time() = (
        oldTime.seconds() < correction.last_transition_time().seconds() ||
        oldTime.seconds() == correction.last_transition_time().seconds() &&
        oldTime.nanos() < correction.last_transition_time().nanos()
        ? correction.last_transition_time() : oldTime);
}

template<typename T>
void PatchAllNonReadyConditions(API::TPodAgentStatus& objectStatus, const T& correctionStatus) {
    PatchCondition<API::EConditionStatus_TRUE>(objectStatus.mutable_in_progress(), correctionStatus.in_progress());
    PatchCondition<API::EConditionStatus_TRUE>(objectStatus.mutable_failed(), correctionStatus.failed());
}

TUnistatObjectHelper::EConditionStatus ConvertConditionStatus(API::EConditionStatus conditionStatus);

void PushAllPodSignals(const API::TPodAgentStatus& status);

template<typename T>
void PushAllCacheObjectSignals(
    const T& status
    , NStatusRepositoryTypes::EObjectType objectType
) {
    TUnistatObjectHelper::Instance().PushCacheObjectConditionSignal(
        status.id()
        , status.revision()
        , objectType
        , TUnistatObjectHelper::EConditionSignalType::READY
        , ConvertConditionStatus(status.ready().status())
    );
    TUnistatObjectHelper::Instance().PushCacheObjectConditionSignal(
        status.id()
        , status.revision()
        , objectType
        , TUnistatObjectHelper::EConditionSignalType::IN_PROGRESS
        , ConvertConditionStatus(status.in_progress().status())
    );
    TUnistatObjectHelper::Instance().PushCacheObjectConditionSignal(
        status.id()
        , status.revision()
        , objectType
        , TUnistatObjectHelper::EConditionSignalType::FAILED
        , ConvertConditionStatus(status.failed().status())
    );
}

template<typename T>
void PushAllObjectSignals(
    const T& status
    , NStatusRepositoryTypes::EObjectType objectType
) {
    TUnistatObjectHelper::Instance().PushObjectConditionSignal(
        status.id()
        , objectType
        , TUnistatObjectHelper::EConditionSignalType::READY
        , ConvertConditionStatus(status.ready().status())
    );
    TUnistatObjectHelper::Instance().PushObjectConditionSignal(
        status.id()
        , objectType
        , TUnistatObjectHelper::EConditionSignalType::IN_PROGRESS
        , ConvertConditionStatus(status.in_progress().status())
    );
    TUnistatObjectHelper::Instance().PushObjectConditionSignal(
        status.id()
        , objectType
        , TUnistatObjectHelper::EConditionSignalType::FAILED
        , ConvertConditionStatus(status.failed().status())
    );
}

TInstant ToInstant(const google::protobuf::Timestamp& timestamp);
google::protobuf::Timestamp ToTimestamp(const TInstant& time);
TString Truncate(const TString& stringToTruncate);
TString ExtractState(const TString& state);
void SetApplyingObjectSpecState(API::TCondition& condition);

// Sometimes a message is too large and we don't want to copy it
void CopyConditonStatusAndTimestamp(API::TCondition& target, const API::TCondition& source);

} // namespace NInfra::NPodAgent::NSupport
