#include "test_functions.h"

#include <util/generic/serialized_enum.h>

namespace NInfra::NPodAgent {

template <typename TType>
void TestAllTypes(std::function<void(TType)> callback) {
    const auto enums = GetEnumAllValues<TType>();
    for (const auto& it : enums) {
        callback(it);
    }
}

void TestNetworkHooksAllTypes(std::function<void(NStatusRepositoryTypes::ENetworkHookType, NStatusRepositoryTypes::EHookBackend)> callback) {
    auto testHttp = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType) {
        callback(networkHookType, NStatusRepositoryTypes::EHookBackend::HTTP);
    };
    TestHttpHookAllTypes(testHttp);

    auto testTcp = [&](NStatusRepositoryTypes::ENetworkHookType networkHookType) {
        callback(networkHookType, NStatusRepositoryTypes::EHookBackend::TCP);
    };
    TestTcpHookAllTypes(testTcp);
}

void TestHttpHookAllTypes(std::function<void(NStatusRepositoryTypes::ENetworkHookType)> callback) {
    TestAllTypes(callback);
}

void TestTcpHookAllTypes(std::function<void(NStatusRepositoryTypes::ENetworkHookType)> callback) {
    auto wrapped = [&](NStatusRepositoryTypes::ENetworkHookType type) {
        if (type != NStatusRepositoryTypes::ENetworkHookType::STOP && type != NStatusRepositoryTypes::ENetworkHookType::DESTROY) {
            callback(type);
        }
    };

    TestAllTypes<NStatusRepositoryTypes::ENetworkHookType>(wrapped);
}

void TestWorkloadContainerHookAllTypes(std::function<void(NStatusRepositoryTypes::TContainerDescription::EContainerType)> callback) {
    auto wrapped = [&](NStatusRepositoryTypes::TContainerDescription::EContainerType containerType) {
        if (containerType == NStatusRepositoryTypes::TContainerDescription::EContainerType::READINESS
            || containerType == NStatusRepositoryTypes::TContainerDescription::EContainerType::LIVENESS
            || containerType == NStatusRepositoryTypes::TContainerDescription::EContainerType::STOP
            || containerType == NStatusRepositoryTypes::TContainerDescription::EContainerType::DESTROY) {
            callback(containerType);
        }
    };

    TestAllTypes<NStatusRepositoryTypes::TContainerDescription::EContainerType>(wrapped);
}

API::TContainerStatus GetWorkloadContainerHookStatus(const API::TWorkloadStatus& status, NStatusRepositoryTypes::TContainerDescription::EContainerType containerType) {
    switch (containerType) {
        case NStatusRepositoryTypes::TContainerDescription::EContainerType::READINESS:
            return status.readiness_status().container_status();
        case NStatusRepositoryTypes::TContainerDescription::EContainerType::LIVENESS:
            return status.liveness_status().container_status();
        case NStatusRepositoryTypes::TContainerDescription::EContainerType::STOP:
            return status.stop_status().container_status();
        case NStatusRepositoryTypes::TContainerDescription::EContainerType::DESTROY:
            return status.destroy_status().container_status();
        case NStatusRepositoryTypes::TContainerDescription::EContainerType::INIT:
            ythrow yexception() << "Init container is not hook container";
        case NStatusRepositoryTypes::TContainerDescription::EContainerType::START:
            ythrow yexception() << "Start container is not hook container";
        case NStatusRepositoryTypes::TContainerDescription::EContainerType::META:
        case NStatusRepositoryTypes::TContainerDescription::EContainerType::DOWNLOAD:
        case NStatusRepositoryTypes::TContainerDescription::EContainerType::VERIFY:
            ythrow yexception() << "Workload does not have " << ToString(containerType) << " container";
    }
}

API::THttpGetStatus GetHttpHookStatus(const API::TWorkloadStatus& status, NStatusRepositoryTypes::ENetworkHookType type) {
    switch (type) {
        case NStatusRepositoryTypes::ENetworkHookType::READINESS:
            return status.readiness_status().http_get_status();
        case NStatusRepositoryTypes::ENetworkHookType::LIVENESS:
            return status.liveness_status().http_get_status();
        case NStatusRepositoryTypes::ENetworkHookType::STOP:
            return status.stop_status().http_get_status();
        case NStatusRepositoryTypes::ENetworkHookType::DESTROY:
            return status.destroy_status().http_get_status();
    }
}

API::TTcpCheckStatus GetTcpCheckStatus(const API::TWorkloadStatus& status, NStatusRepositoryTypes::ENetworkHookType type) {
    switch (type) {
        case NStatusRepositoryTypes::ENetworkHookType::READINESS:
            return status.readiness_status().tcp_check_status();
        case NStatusRepositoryTypes::ENetworkHookType::LIVENESS:
            return status.liveness_status().tcp_check_status();
        case NStatusRepositoryTypes::ENetworkHookType::STOP:
            ythrow yexception() << "Stop status not provided for tcp protocol";
        case NStatusRepositoryTypes::ENetworkHookType::DESTROY:
            ythrow yexception() << "Destroy status not provided for tcp protocol";
    }
}

API::TUnixSignalStatus GetStopUnixSignalStatus(const API::TWorkloadStatus& status) {
    return status.stop_status().unix_signal_status();
}

} // namespace NInfra::NPodAgent
