#include "volume_status_repository.h"

#include "support_functions.h"

namespace NInfra::NPodAgent {

void TVolumeStatusRepository::UpdateSpecTimestamp(TInstant currentTime) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    for (auto& [objectId, info] : Objects_) {
        TGuard<TMutex> g(GetObjectMutex(objectId));
        RefreshObjectStatusConditions(info.Status_, info.Status_, currentTime);
    }
}

void TVolumeStatusRepository::PatchTotalStatus(API::TPodAgentStatus& status, TUpdateHolderTargetPtr updateHolderTarget, bool conditionsOnly) const {
    Y_ENSURE(updateHolderTarget, "UpdateHolderTarget not provided for TVolumeStatusRepository::PatchTotalStatus");

    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);

    for (const auto& [objectId, info] : Objects_) {
        API::TVolumeStatus* st = status.add_volumes();
        *st = GetObjectStatusNoGlobalLock(objectId, updateHolderTarget, conditionsOnly);
        NSupport::PatchAllNonReadyConditions(status, *st);
    }
}

bool TVolumeStatusRepository::NeedLongTickPeriod(const TString& objectIdOrHash, TUpdateHolderTargetPtr updateHolderTarget) const {
    Y_ENSURE(updateHolderTarget, "UpdateHolderTarget not provided for TVolumeStatusRepository::NeedLongTickPeriod");

    auto status = GetObjectStatus(objectIdOrHash, updateHolderTarget);

    return status.ready().status() == API::EConditionStatus_TRUE
        || status.state() == API::EVolumeState_WAITING_FOR_LAYERS && TInstant::Now() - NSupport::ToInstant(status.in_progress().last_transition_time()) > ACTIVE_WAIT_TIME
        || status.state() == API::EVolumeState_INVALID;
}

void TVolumeStatusRepository::AddObject(const TVolumeMeta& objectMeta) {
    TWriteGuardBase<TLightRWLock> guard(GlobalObjectLock_);

    auto [ptr, result] = Objects_.insert(
        {
            objectMeta.Id_
            , TObjectInfo()
        }
    );
    API::TVolumeStatus& objectStatus = ptr->second.Status_;

    objectStatus.set_id(objectMeta.Id_);
    objectStatus.set_spec_timestamp(objectMeta.SpecTimestamp_);
    objectStatus.set_revision(objectMeta.Revision_);
}

void TVolumeStatusRepository::RemoveObject(const TString& objectId) {
    TWriteGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    Objects_.erase(objectId);
}

bool TVolumeStatusRepository::HasObject(const TString& objectId) const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    return Objects_.FindPtr(objectId);
}

API::TVolumeStatus TVolumeStatusRepository::GetObjectStatus(const TString& objectId, TUpdateHolderTargetPtr updateHolderTarget) const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    return GetObjectStatusNoGlobalLock(objectId, updateHolderTarget, false);
}

API::TVolumeStatus TVolumeStatusRepository::GetObjectStatusNoGlobalLock(
    const TString& objectId
    , TUpdateHolderTargetPtr updateHolderTarget
    , bool conditionsOnly
) const {
    TGuard<TMutex> g(GetObjectMutex(objectId));

    // WARNING: Do not copy full status here
    const API::TVolumeStatus& fullStatus = Objects_.at(objectId).Status_;

    API::TVolumeStatus status;
    if (conditionsOnly) {
        // Set object id only
        status.set_id(objectId);
    } else {
        status.CopyFrom(fullStatus);
    }

    bool changingSpecTimestamp = updateHolderTarget ? updateHolderTarget->VolumeHasTarget(objectId) : false;
    RefreshObjectStatusConditions(status, fullStatus, TInstant::Zero(), false, changingSpecTimestamp);

    return status;
}

TVector<TString> TVolumeStatusRepository::GetObjectIds() const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);

    TVector<TString> result;
    for (const auto& [objectId, info] : Objects_) {
        result.push_back(objectId);
    }
    return result;
}

void TVolumeStatusRepository::UpdateObjectRevision(const TString& objectId, ui32 revision) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    API::TVolumeStatus& volume = Objects_.at(objectId).Status_;
    if (volume.revision() != revision) {
        volume.set_revision(revision);
        TInstant now = TInstant::Now();
        RefreshObjectStatusConditions(volume, volume, now);
    }
}

void TVolumeStatusRepository::UpdateObjectSpecTimestamp(const TString& objectId, ui64 specTimestamp) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    API::TVolumeStatus& volume = Objects_.at(objectId).Status_;
    if (volume.spec_timestamp() != specTimestamp) {
        volume.set_spec_timestamp(specTimestamp);
        TInstant now = TInstant::Now();
        RefreshObjectStatusConditions(volume, volume, now);
    }
}

TStatusRepositoryCommon::TObjectState TVolumeStatusRepository::UpdateObjectState(const TString& objectIdOrHash, TObjectState state) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectIdOrHash));
    API::TVolumeStatus& volume = Objects_.at(objectIdOrHash).Status_;
    TStatusRepositoryCommon::TObjectState oldState = volume.state();
    if (oldState != state) {
        volume.set_state(std::get<API::EVolumeState>(state));
        TInstant now = TInstant::Now();
        RefreshObjectStatusConditions(volume, volume, now);
    }
    return oldState;
}

void TVolumeStatusRepository::UpdateObjectFailedMessage(const TString& objectIdOrHash, const TString& failedMessage) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectIdOrHash));
    Objects_.at(objectIdOrHash).Status_.mutable_failed()->set_message(NSupport::Truncate(failedMessage));
}

void TVolumeStatusRepository::IncrementObjectFailCounter(const TString& objectIdOrHash) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectIdOrHash));
    API::TVolumeStatus& status = Objects_.at(objectIdOrHash).Status_;
    status.set_fail_counter(status.fail_counter() + 1);
}

TVector<TString> TVolumeStatusRepository::GetObjectIdsByHash(const TString& /* objectIdOrHash */) {
    // An empty vector is returned in order to have no duplicates of id in logs
    return {};
}

TVector<TStatusRepositoryCommon::TCacheObject> TVolumeStatusRepository::GetCacheObjectIdsAndRevisionsByHash(const TString& /* objectIdOrHash */) {
    // No CacheObjects in this status repository
    return {};
}

void TVolumeStatusRepository::RefreshObjectStatusConditions(
    API::TVolumeStatus& targetStatus
    , const API::TVolumeStatus& volume
    , const TInstant& now
    , bool refreshTime
    , bool changingSpecTimestamp
) {
    RefreshObjectStatusReady(
        *targetStatus.mutable_ready()
        , volume
        , now
        , refreshTime
        , changingSpecTimestamp
    );
    RefreshObjectStatusInProgress(
        *targetStatus.mutable_in_progress()
        , volume
        , now
        , refreshTime
        , changingSpecTimestamp
    );
    RefreshObjectStatusFailed(
        *targetStatus.mutable_failed()
        , volume
        , now
        , refreshTime
    );
}

void TVolumeStatusRepository::RefreshObjectStatusReady(
    API::TCondition& ready
    , const API::TVolumeStatus& volume
    , const TInstant& now
    , bool refreshTime
    , bool changingSpecTimestamp
) {
    NSupport::CopyConditonStatusAndTimestamp(ready, volume.ready());
    API::EConditionStatus prevStatus = ready.status();

    ready.set_reason(NSupport::ExtractState(API::EVolumeState_Name(volume.state())));
    ready.set_message("");
    if (changingSpecTimestamp) {
        ready.set_message(NSupport::APPLYING_OBJECT_SPEC_MESSAGE);
    }

    switch (volume.state()) {
        case API::EVolumeState_READY:
            if (!changingSpecTimestamp) {
                ready.set_status(API::EConditionStatus_TRUE);
            } else {
                ready.set_status(API::EConditionStatus_FALSE);
                NSupport::SetApplyingObjectSpecState(ready);
            }
            break;
        case API::EVolumeState_WAITING_FOR_LAYERS:
        case API::EVolumeState_WAITING_FOR_STATIC_RESOURCES:
        case API::EVolumeState_CREATING:
        case API::EVolumeState_REMOVING:
        case API::EVolumeState_REMOVED:
        case API::EVolumeState_INVALID:
            ready.set_status(API::EConditionStatus_FALSE);
            break;
        case API::EVolumeState_UNKNOWN:
            ready.set_status(API::EConditionStatus_UNKNOWN);
            break;
        // We don't want to use default
        // So we need this to handle all cases
        case API::EVolumeState_INT_MIN_SENTINEL_DO_NOT_USE_:
        case API::EVolumeState_INT_MAX_SENTINEL_DO_NOT_USE_:
            ready.set_status(API::EConditionStatus_UNKNOWN);
            break;
    }

    if (refreshTime && prevStatus != ready.status()) {
        *ready.mutable_last_transition_time() = NSupport::ToTimestamp(now);
    }
}

void TVolumeStatusRepository::RefreshObjectStatusInProgress(
    API::TCondition& inProgress
    , const API::TVolumeStatus& volume
    , const TInstant& now
    , bool refreshTime
    , bool changingSpecTimestamp
) {
    NSupport::CopyConditonStatusAndTimestamp(inProgress, volume.in_progress());
    API::EConditionStatus prevStatus = inProgress.status();

    inProgress.set_reason(NSupport::ExtractState(API::EVolumeState_Name(volume.state())));
    inProgress.set_message("");
    if (changingSpecTimestamp) {
        inProgress.set_message(NSupport::APPLYING_OBJECT_SPEC_MESSAGE);
    }

    switch (volume.state()) {
        case API::EVolumeState_WAITING_FOR_LAYERS:
        case API::EVolumeState_WAITING_FOR_STATIC_RESOURCES:
        case API::EVolumeState_CREATING:
        case API::EVolumeState_REMOVING:
            inProgress.set_status(API::EConditionStatus_TRUE);
            break;
        case API::EVolumeState_READY:
        case API::EVolumeState_REMOVED:
            if (!changingSpecTimestamp) {
                inProgress.set_status(API::EConditionStatus_FALSE);
            } else {
                inProgress.set_status(API::EConditionStatus_TRUE);
                NSupport::SetApplyingObjectSpecState(inProgress);
            }
            break;
        case API::EVolumeState_INVALID:
            inProgress.set_status(API::EConditionStatus_FALSE);
            break;
        case API::EVolumeState_UNKNOWN:
            inProgress.set_status(API::EConditionStatus_UNKNOWN);
            break;
        // We don't want to use default
        // So we need this to handle all cases
        case API::EVolumeState_INT_MIN_SENTINEL_DO_NOT_USE_:
        case API::EVolumeState_INT_MAX_SENTINEL_DO_NOT_USE_:
            inProgress.set_status(API::EConditionStatus_UNKNOWN);
            break;
    }

    if (refreshTime && prevStatus != inProgress.status()) {
        *inProgress.mutable_last_transition_time() = NSupport::ToTimestamp(now);
    }
}

void TVolumeStatusRepository::RefreshObjectStatusFailed(
    API::TCondition& failed
    , const API::TVolumeStatus& volume
    , const TInstant& now
    , bool refreshTime
) {
    NSupport::CopyConditonStatusAndTimestamp(failed, volume.failed());
    API::EConditionStatus prevStatus = failed.status();

    switch (volume.state()) {
        case API::EVolumeState_WAITING_FOR_LAYERS:
        case API::EVolumeState_WAITING_FOR_STATIC_RESOURCES:
        case API::EVolumeState_CREATING:
        case API::EVolumeState_REMOVING:
        case API::EVolumeState_REMOVED:
            // UNKNOWN -> FALSE
            // TRUE and FALSE do not change
            if (failed.status() == API::EConditionStatus_UNKNOWN) {
                failed.set_status(API::EConditionStatus_FALSE);
            }
            break;
        case API::EVolumeState_READY:
            failed.set_status(API::EConditionStatus_FALSE);
            failed.set_reason(NSupport::ExtractState(API::EVolumeState_Name(volume.state())));
            break;
        case API::EVolumeState_INVALID:
            failed.set_status(API::EConditionStatus_TRUE);
            failed.set_reason(NSupport::ExtractState(API::EVolumeState_Name(volume.state())));
            break;
        case API::EVolumeState_UNKNOWN:
            failed.set_status(API::EConditionStatus_UNKNOWN);
            failed.set_reason(NSupport::ExtractState(API::EVolumeState_Name(volume.state())));
            break;
        // We don't want to use default
        // So we need this to handle all cases
        case API::EVolumeState_INT_MIN_SENTINEL_DO_NOT_USE_:
        case API::EVolumeState_INT_MAX_SENTINEL_DO_NOT_USE_:
            failed.set_status(API::EConditionStatus_UNKNOWN);
            failed.set_reason(NSupport::ExtractState(API::EVolumeState_Name(volume.state())));
            break;
    }

    if (refreshTime && prevStatus != failed.status()) {
        *failed.mutable_last_transition_time() = NSupport::ToTimestamp(now);
    }
}

const TMutex& TVolumeStatusRepository::GetObjectMutex(const TString& objectId) const {
    const TObjectInfo* ptr = Objects_.FindPtr(objectId);
    Y_ENSURE(ptr, "Volume '" << objectId << "' not found at TVolumeStatusRepository");
    return ptr->Mutex_;
}

} // namespace NInfra::NPodAgent
