#pragma once

#include "status_repository_common.h"

#include <infra/pod_agent/libs/pod_agent/object_meta/object_meta.h>

#include <util/generic/vector.h>
#include <util/generic/map.h>
#include <util/generic/set.h>

namespace NInfra::NPodAgent {

/**
    Thread-safe volume status repository
*/
class TVolumeStatusRepository: public TStatusRepositoryCommon {
private:
    struct TObjectInfo {
        TMutex Mutex_;
        API::TVolumeStatus Status_;
    };

public:
    TVolumeStatusRepository()
        : TStatusRepositoryCommon(NStatusRepositoryTypes::EObjectType::VOLUME)
    {}

    virtual void UpdateSpecTimestamp(TInstant currentTime) override final;
    virtual void PatchTotalStatus(API::TPodAgentStatus& status, TUpdateHolderTargetPtr updateHolderTarget, bool conditionsOnly) const override final;
    virtual bool NeedLongTickPeriod(const TString& objectIdOrHash, TUpdateHolderTargetPtr updateHolderTarget) const override final;

    // Common object methods
    virtual void UpdateObjectRevision(const TString& objectId, ui32 revision) override final;
    virtual void UpdateObjectSpecTimestamp(const TString& objectId, ui64 specTimestamp) override final;
    virtual TObjectState UpdateObjectState(const TString& objectIdOrHash, TObjectState state) override final;
    virtual void UpdateObjectFailedMessage(const TString& objectIdOrHash, const TString& failedMessage) override final;
    virtual void IncrementObjectFailCounter(const TString& objectIdOrHash) override final;
    virtual TVector<TString> GetObjectIdsByHash(const TString& objectIdOrHash) override final;
    virtual TVector<TCacheObject> GetCacheObjectIdsAndRevisionsByHash(const TString& objectIdOrHash) override final;

    // Volume specific methods
    void AddObject(const TVolumeMeta& objectMeta);
    void RemoveObject(const TString& objectId);
    bool HasObject(const TString& objectId) const;

    API::TVolumeStatus GetObjectStatus(const TString& objectId, TUpdateHolderTargetPtr updateHolderTarget = nullptr) const;
    TVector<TString> GetObjectIds() const;

private:
    const TMutex& GetObjectMutex(const TString& objectId) const;
    API::TVolumeStatus GetObjectStatusNoGlobalLock(
        const TString& objectId
        , TUpdateHolderTargetPtr updateHolderTarget
        , bool conditionsOnly
    ) const;

    static void RefreshObjectStatusConditions(
        API::TVolumeStatus& targetStatus
        , const API::TVolumeStatus& volume
        , const TInstant& now
        , bool refreshTime = true
        , bool changingSpecTimestamp = false
    );
    static void RefreshObjectStatusReady(
        API::TCondition& ready
        , const API::TVolumeStatus& volume
        , const TInstant& now
        , bool refreshTime
        , bool changingSpecTimestamp
    );
    static void RefreshObjectStatusInProgress(
        API::TCondition& inProgress
        , const API::TVolumeStatus& volume
        , const TInstant& now
        , bool refreshTime
        , bool changingSpecTimestamp
    );
    static void RefreshObjectStatusFailed(
        API::TCondition& failed
        , const API::TVolumeStatus& volume
        , const TInstant& now
        , bool refreshTime
    );

private:
    TMap<TString, TObjectInfo> Objects_;
    TLightRWLock GlobalObjectLock_;
};

using TVolumeStatusRepositoryPtr = TIntrusivePtr<TVolumeStatusRepository>;

} // namespace NInfra::NPodAgent
