#include "volume_status_repository.h"
#include "support_functions.h"
#include "test_functions.h"

#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/update_holder/test_lib/test_functions.h>

#include <google/protobuf/util/message_differencer.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <util/string/join.h>

namespace NInfra::NPodAgent::NVolumeStatusRepositoryTest {

Y_UNIT_TEST_SUITE(VolumeStatusRepositorySuite) {

Y_UNIT_TEST(TestGetObjectType) {
    TVolumeStatusRepositoryPtr holder = new TVolumeStatusRepository();
    const TString id = "my_volume";
    holder->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple(id));

    UNIT_ASSERT_EQUAL_C(holder->GetObjectType(), NStatusRepositoryTypes::EObjectType::VOLUME, ToString(holder->GetObjectType()));
}

Y_UNIT_TEST(TestUpdateHolderNotProvided) {
    TVolumeStatusRepositoryPtr holder = new TVolumeStatusRepository();
    const TString id = "my_volume";
    holder->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple(id));

    API::TPodAgentStatus patchStatus;
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->PatchTotalStatus(patchStatus, nullptr, false), yexception, "UpdateHolderTarget not provided");
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->NeedLongTickPeriod(id, nullptr), yexception, "UpdateHolderTarget not provided");
}

Y_UNIT_TEST(TestAddVolume) {
    TVolumeStatusRepositoryPtr holder = new TVolumeStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString id = "my_volume";
    holder->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple(id));
    API::TVolumeStatus status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL(API::EVolumeState_UNKNOWN, status.state());
}

Y_UNIT_TEST(TestVolumePatchTotalStatus) {
    TVolumeStatusRepositoryPtr holder = new TVolumeStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString id = "my_volume";

    holder->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple(id));
    // Changing something in the status so that it is not equal to the default one
    holder->UpdateObjectState(id, API::EVolumeState_READY);
    API::TVolumeStatus status = holder->GetObjectStatus(id);

    {
        API::TPodAgentStatus patchStatus;
        holder->PatchTotalStatus(patchStatus, updateHolder->GetUpdateHolderTarget(), false);
        UNIT_ASSERT_EQUAL_C(
            patchStatus.volumes(0).id()
            , id
            , patchStatus.volumes(0).id()
        );
        UNIT_ASSERT_EQUAL_C(
            patchStatus.volumes(0).state()
            , API::EVolumeState_READY
            , API::EVolumeState_Name(patchStatus.volumes(0).state())
        );
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(status, patchStatus.volumes(0)));
    }

    {
        API::TPodAgentStatus patchStatus;
        holder->PatchTotalStatus(patchStatus, updateHolder->GetUpdateHolderTarget(), true);
        UNIT_ASSERT_EQUAL_C(
            patchStatus.volumes(0).id()
            , id
            , patchStatus.volumes(0).id()
        );
        UNIT_ASSERT_EQUAL_C(
            patchStatus.volumes(0).state()
            // conditionsOnly = true
            , API::EVolumeState_UNKNOWN
            , API::EVolumeState_Name(patchStatus.volumes(0).state())
        );
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(status.ready(), patchStatus.volumes(0).ready()));
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(status.in_progress(), patchStatus.volumes(0).in_progress()));
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(status.failed(), patchStatus.volumes(0).failed()));
    }
}

Y_UNIT_TEST(TestVolumeRevisionAndSpecTimestamp) {
    TVolumeStatusRepositoryPtr holder = new TVolumeStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString id = "my_volume";

    holder->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple(id, 0, 1));
    API::TVolumeStatus status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL_C(status.spec_timestamp(), 1, status.spec_timestamp());
    UNIT_ASSERT_EQUAL_C(status.revision(), 2, status.revision());

    holder->UpdateObjectSpecTimestamp(id, 3);
    holder->UpdateObjectRevision(id, 4);
    status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL_C(status.spec_timestamp(), 3, status.spec_timestamp());
    UNIT_ASSERT_EQUAL_C(status.revision(), 4, status.revision());
}

Y_UNIT_TEST(TestUpdateVolumeState) {
    TVolumeStatusRepositoryPtr holder = new TVolumeStatusRepository();
    const TString id = "my_volume";
    holder->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple(id));
    holder->UpdateObjectState(id, API::EVolumeState_READY);
    API::TVolumeStatus status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL(API::EVolumeState_READY, status.state());
}

Y_UNIT_TEST(TestUpdateVolumeFailedMessage) {
    TVolumeStatusRepositoryPtr holder = new TVolumeStatusRepository();
    const TString id = "my_volume";
    const TString message = "message-reason";
    holder->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple(id));
    holder->UpdateObjectFailedMessage(id, message);
    API::TVolumeStatus status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL(message, status.failed().message());
}

Y_UNIT_TEST(TestUpdateVolumeStateAndReadyStatus) {
    TVolumeStatusRepositoryPtr holder = new TVolumeStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString id = "my_volume";
    holder->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple(id));

    holder->UpdateObjectState(id, API::EVolumeState_UNKNOWN);
    API::TVolumeStatus status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EVolumeState_UNKNOWN, status.state());
    auto ready = status.ready();
    UNIT_ASSERT_EQUAL(ready.last_transition_time().seconds(), 0);
    UNIT_ASSERT_EQUAL(ready.last_transition_time().nanos(), 0);
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_UNKNOWN);
    UNIT_ASSERT(!holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(id, API::EVolumeState_INVALID);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EVolumeState_INVALID, status.state());
    ready = status.ready();
    UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT(holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(id, API::EVolumeState_REMOVED);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EVolumeState_REMOVED, status.state());
    ready = status.ready();
    UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT(!holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(id, API::EVolumeState_WAITING_FOR_STATIC_RESOURCES);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EVolumeState_WAITING_FOR_STATIC_RESOURCES, status.state());
    ready = status.ready();
    UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT(!holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(id, API::EVolumeState_READY);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EVolumeState_READY, status.state());
    ready = status.ready();
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT_EQUAL(NSupport::ExtractState(API::EVolumeState_Name(API::EVolumeState_READY)), ready.reason());
    UNIT_ASSERT_EQUAL("", ready.message());
    UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
    UNIT_ASSERT(holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateSpecTimestamp(TInstant::Now());
    updateHolder->SetVolumeTarget(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple(id));
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EVolumeState_READY, status.state());
    ready  = status.ready();
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT_EQUAL(NSupport::APPLYING_OBJECT_SPEC, ready.reason());
    UNIT_ASSERT_EQUAL(NSupport::APPLYING_OBJECT_SPEC_MESSAGE, ready.message());
    UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
    UNIT_ASSERT(!holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL(API::EVolumeState_READY, status.state());
    ready  = status.ready();
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT_EQUAL(NSupport::ExtractState(API::EVolumeState_Name(API::EVolumeState_READY)), ready.reason());
    UNIT_ASSERT_EQUAL("", ready.message());
    UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
}

Y_UNIT_TEST(TestUpdateVolumeStateAndInProgressStatus) {
    TVolumeStatusRepositoryPtr holder = new TVolumeStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString id = "my_volume";
    holder->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple(id));

    holder->UpdateObjectState(id, API::EVolumeState_UNKNOWN);
    API::TVolumeStatus status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EVolumeState_UNKNOWN, status.state());
    auto inProgress = status.in_progress();
    UNIT_ASSERT_EQUAL(inProgress.last_transition_time().seconds(), 0);
    UNIT_ASSERT_EQUAL(inProgress.last_transition_time().nanos(), 0);
    UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_UNKNOWN);

    holder->UpdateObjectState(id, API::EVolumeState_READY);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EVolumeState_READY, status.state());
    inProgress = status.in_progress();
    UNIT_ASSERT(inProgress.last_transition_time().seconds() != 0 || inProgress.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_FALSE);

    holder->UpdateObjectState(id, API::EVolumeState_CREATING);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EVolumeState_CREATING, status.state());
    inProgress = status.in_progress();
    UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT_EQUAL(NSupport::ExtractState(API::EVolumeState_Name(API::EVolumeState_CREATING)), inProgress.reason());
    UNIT_ASSERT_EQUAL("", inProgress.message());
    UNIT_ASSERT(inProgress.last_transition_time().seconds() != 0 || inProgress.last_transition_time().nanos() != 0);

    holder->UpdateObjectState(id, API::EVolumeState_WAITING_FOR_STATIC_RESOURCES);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EVolumeState_WAITING_FOR_STATIC_RESOURCES, status.state());
    inProgress = status.in_progress();
    UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT_EQUAL(NSupport::ExtractState(API::EVolumeState_Name(API::EVolumeState_WAITING_FOR_STATIC_RESOURCES)), inProgress.reason());
    UNIT_ASSERT_EQUAL("", inProgress.message());
    UNIT_ASSERT(inProgress.last_transition_time().seconds() != 0 || inProgress.last_transition_time().nanos() != 0);

    holder->UpdateObjectState(id, API::EVolumeState_READY);
    holder->UpdateSpecTimestamp(TInstant::Now());
    updateHolder->SetVolumeTarget(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple(id));
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EVolumeState_READY, status.state());
    inProgress = status.in_progress();
    UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT_EQUAL(NSupport::APPLYING_OBJECT_SPEC, inProgress.reason());
    UNIT_ASSERT_EQUAL(NSupport::APPLYING_OBJECT_SPEC_MESSAGE, inProgress.message());
    UNIT_ASSERT(inProgress.last_transition_time().seconds() != 0 || inProgress.last_transition_time().nanos() != 0);

    status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL(API::EVolumeState_READY, status.state());
    inProgress = status.in_progress();
    UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT_EQUAL(NSupport::ExtractState(API::EVolumeState_Name(API::EVolumeState_READY)), inProgress.reason());
    UNIT_ASSERT_EQUAL("", inProgress.message());
    UNIT_ASSERT(inProgress.last_transition_time().seconds() != 0 || inProgress.last_transition_time().nanos() != 0);
}

Y_UNIT_TEST(TestUpdateVolumeStateAndFailedStatus) {
    TVolumeStatusRepositoryPtr holder = new TVolumeStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString id = "my_volume";
    holder->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple(id));

    holder->UpdateObjectState(id, API::EVolumeState_UNKNOWN);
    API::TVolumeStatus status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EVolumeState_UNKNOWN, status.state());
    auto failed = status.failed();
    UNIT_ASSERT_EQUAL(failed.last_transition_time().seconds(), 0);
    UNIT_ASSERT_EQUAL(failed.last_transition_time().nanos(), 0);
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_UNKNOWN);
    UNIT_ASSERT(!holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(id, API::EVolumeState_REMOVED);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EVolumeState_REMOVED, status.state());
    failed = status.failed();
    UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT(!holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));
 
    holder->UpdateObjectState(id, API::EVolumeState_WAITING_FOR_STATIC_RESOURCES);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EVolumeState_WAITING_FOR_STATIC_RESOURCES, status.state());
    failed = status.failed();
    UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT(!holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(id, API::EVolumeState_READY);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EVolumeState_READY, status.state());
    failed = status.failed();
    UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT(holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(id, API::EVolumeState_INVALID);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EVolumeState_INVALID, status.state());
    failed = status.failed();
    UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT(holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(id, API::EVolumeState_REMOVED);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EVolumeState_REMOVED, status.state());
    failed = status.failed();
    UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT(!holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(id, API::EVolumeState_READY);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EVolumeState_READY, status.state());
    failed = status.failed();
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT_EQUAL(NSupport::ExtractState(API::EVolumeState_Name(API::EVolumeState_READY)), failed.reason());
    UNIT_ASSERT_EQUAL("", failed.message());
    UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);
    UNIT_ASSERT(holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateSpecTimestamp(TInstant::Now());
    updateHolder->SetVolumeTarget(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple(id));
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EVolumeState_READY, status.state());
    failed  = status.failed();
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT_EQUAL(NSupport::ExtractState(API::EVolumeState_Name(API::EVolumeState_READY)), failed.reason());
    UNIT_ASSERT_EQUAL("", failed.message());
    UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);
    UNIT_ASSERT(!holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL(API::EVolumeState_READY, status.state());
    failed  = status.failed();
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT_EQUAL(NSupport::ExtractState(API::EVolumeState_Name(API::EVolumeState_READY)), failed.reason());
    UNIT_ASSERT_EQUAL("", failed.message());
    UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);
}

Y_UNIT_TEST(TestUpdateVolumeNotFound) {
    TVolumeStatusRepositoryPtr holder = new TVolumeStatusRepository();
    const TString id = "my_volume";
    holder->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple(id));
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->UpdateObjectState(id + "_other", API::EVolumeState_READY), yexception, "not found at TVolumeStatusRepository");
}

Y_UNIT_TEST(TestHasVolume) {
    TVolumeStatusRepositoryPtr holder = new TVolumeStatusRepository();
    const TString id = "my_volume";
    UNIT_ASSERT(!holder->HasObject(id));
    holder->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple(id));
    UNIT_ASSERT(holder->HasObject(id));
}

Y_UNIT_TEST(TestGetVolumeIds) {
    TVolumeStatusRepositoryPtr holder = new TVolumeStatusRepository();
    const TString id = "my_volume";
    holder->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple(id));

    UNIT_ASSERT_EQUAL(
        holder->GetObjectIds()
        , TVector<TString>(
            {
                id
            }
        )
    );

    // Must be empty
    UNIT_ASSERT_EQUAL(
        holder->GetObjectIdsByHash(id)
        , TVector<TString>()
    );
    UNIT_ASSERT_EQUAL(
        holder->GetCacheObjectIdsAndRevisionsByHash(id)
        , TVector<TStatusRepositoryCommon::TCacheObject>()
    );
}

Y_UNIT_TEST(TestFailCounter) {
    TVolumeStatusRepositoryPtr holder = new TVolumeStatusRepository();
    const TString id = "my_volume";
    const size_t cntIter = 10;

    holder->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple(id));
    for (size_t i = 0; i < cntIter; ++i) {
        holder->IncrementObjectFailCounter("my_volume");
    }

    UNIT_ASSERT_EQUAL(holder->GetObjectStatus("my_volume").fail_counter(), cntIter);
}

}

} // namespace NInfra::NPodAgent::NVolumeStatusRepositoryTest
