#include "workload_status_repository.h"

#include "support_functions.h"

namespace NInfra::NPodAgent {

void TWorkloadStatusRepository::UpdateSpecTimestamp(TInstant currentTime) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    for (auto& [objectId, info] : Objects_) {
        TGuard<TMutex> g(GetObjectMutex(objectId));
        RefreshObjectStatusConditions(info.Status_, info.Status_, currentTime);
    }
}

void TWorkloadStatusRepository::PatchTotalStatus(API::TPodAgentStatus& status, TUpdateHolderTargetPtr updateHolderTarget, bool conditionsOnly) const {
    Y_ENSURE(updateHolderTarget, "UpdateHolderTarget not provided for TWorkloadStatusRepository::PatchTotalStatus");

    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);

    for (const auto& [objectId, info] : Objects_) {
        API::TWorkloadStatus* workloadStatus = status.add_workloads();
        *workloadStatus = GetObjectStatusNoGlobalLock(objectId, updateHolderTarget, conditionsOnly);
        // We do not patch status failed/in progress if workload has empty readiness
        if (workloadStatus->ready().reason() != "NO_READINESS") {
            NSupport::PatchAllNonReadyConditions(status, *workloadStatus);
        }
        // Only workloads affect ready
        NSupport::PatchCondition<API::EConditionStatus_FALSE>(status.mutable_ready(), workloadStatus->ready());
    }
}

bool TWorkloadStatusRepository::NeedLongTickPeriod(const TString& objectIdOrHash, TUpdateHolderTargetPtr updateHolderTarget) const {
    Y_ENSURE(updateHolderTarget, "UpdateHolderTarget not provided for TWorkloadStatusRepository::NeedLongTickPeriod");

    auto status = GetObjectStatus(objectIdOrHash, updateHolderTarget);
    bool hasReadiness = status.readiness_status().has_readiness();
    bool changingSpecTimestamp = updateHolderTarget->WorkloadHasTarget(objectIdOrHash);

    return (status.ready().status() == API::EConditionStatus_TRUE && hasReadiness)
        || (status.state() == API::EWorkloadState_ACTIVE && status.target_state() == API::EWorkloadTarget_ACTIVE && !hasReadiness && !changingSpecTimestamp)
        || (status.state() == API::EWorkloadState_REMOVED && status.target_state() == API::EWorkloadTarget_REMOVED && !hasReadiness && !changingSpecTimestamp)
        || status.state() == API::EWorkloadState_DEAD
        || status.state() == API::EWorkloadState_INVALID
        || status.state() == API::EWorkloadState_WAITING_FOR_BOX && TInstant::Now() - NSupport::ToInstant(status.in_progress().last_transition_time()) > ACTIVE_WAIT_TIME
    ;
}

const TLightRWLock& TWorkloadStatusRepository::GetGlobalContainerLock() const {
    return GlobalObjectLock_;
}

const TMutex& TWorkloadStatusRepository::GetLocalContainerLock(const NStatusRepositoryTypes::TContainerDescription& container) const {
    return GetObjectMutex(container.ObjectIdOrHash_);
}

API::TContainerStatus* TWorkloadStatusRepository::GetMutableContainerStatus(const NStatusRepositoryTypes::TContainerDescription& container) {
    // Must lock before call
    API::TWorkloadStatus& objectStatus = Objects_.at(container.ObjectIdOrHash_).Status_;

    switch (container.ContainerType_) {
        case NStatusRepositoryTypes::TContainerDescription::EContainerType::START:
            return objectStatus.mutable_start();
        case NStatusRepositoryTypes::TContainerDescription::EContainerType::READINESS:
            return objectStatus.mutable_readiness_status()->mutable_container_status();
        case NStatusRepositoryTypes::TContainerDescription::EContainerType::STOP:
            return objectStatus.mutable_stop_status()->mutable_container_status();
        case NStatusRepositoryTypes::TContainerDescription::EContainerType::DESTROY:
            return objectStatus.mutable_destroy_status()->mutable_container_status();
        case NStatusRepositoryTypes::TContainerDescription::EContainerType::LIVENESS:
            return objectStatus.mutable_liveness_status()->mutable_container_status();
        case NStatusRepositoryTypes::TContainerDescription::EContainerType::INIT:
            Y_ENSURE(
                container.InitNum_ < (ui32)objectStatus.init().size()
                , "Too big init num for workload '" << container.ObjectIdOrHash_
                << "' InitSize = " << objectStatus.init().size()
                << ", InitNum = " << container.InitNum_
             );
            return objectStatus.mutable_init(container.InitNum_);
        case NStatusRepositoryTypes::TContainerDescription::EContainerType::META:
        case NStatusRepositoryTypes::TContainerDescription::EContainerType::DOWNLOAD:
        case NStatusRepositoryTypes::TContainerDescription::EContainerType::VERIFY:
            ythrow yexception() << "Workload does not have '" << ToString(container.ContainerType_) << "' container";
    }
}

const API::TContainerStatus* TWorkloadStatusRepository::GetContainerStatus(const NStatusRepositoryTypes::TContainerDescription& container) {
     // Must lock before call
     return GetMutableContainerStatus(container);
}

void TWorkloadStatusRepository::AddObject(const TWorkloadMeta& objectMeta) {
    TWriteGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    auto [ptr, result] = Objects_.insert(
        {
            objectMeta.Id_
            , TObjectInfo()
        }
    );
    API::TWorkloadStatus& objectStatus = ptr->second.Status_;

    objectStatus.set_id(objectMeta.Id_);
    objectStatus.set_box_ref(objectMeta.BoxRef_);
    objectStatus.set_spec_timestamp(objectMeta.SpecTimestamp_);
    objectStatus.set_revision(objectMeta.Revision_);

    objectStatus.mutable_readiness_status()->set_has_readiness(!std::holds_alternative<TWorkloadMeta::TEmptyInfo>(objectMeta.Readiness_));
    objectStatus.mutable_liveness_status()->set_has_liveness(!std::holds_alternative<TWorkloadMeta::TEmptyInfo>(objectMeta.Liveness_));
    objectStatus.mutable_stop_status()->set_has_stop(!std::holds_alternative<TWorkloadMeta::TEmptyInfo>(objectMeta.Stop_));
    objectStatus.mutable_destroy_status()->set_has_destroy(!std::holds_alternative<TWorkloadMeta::TEmptyInfo>(objectMeta.Destroy_));

    objectStatus.mutable_start()->set_container_name(objectMeta.StartContainer_.ContainerName_);
    for (const TWorkloadMeta::TContainerInfo& initContainer : objectMeta.InitContainers_) {
        auto* initContainerStatus = objectStatus.add_init();
        initContainerStatus->set_container_name(initContainer.ContainerName_);
    }

    if (std::holds_alternative<TWorkloadMeta::TContainerInfo>(objectMeta.Readiness_)) {
        objectStatus.mutable_readiness_status()->mutable_container_status()->set_container_name(std::get<TWorkloadMeta::TContainerInfo>(objectMeta.Readiness_).ContainerName_);
    }
    if (std::holds_alternative<TWorkloadMeta::TContainerInfo>(objectMeta.Liveness_)) {
        objectStatus.mutable_liveness_status()->mutable_container_status()->set_container_name(std::get<TWorkloadMeta::TContainerInfo>(objectMeta.Liveness_).ContainerName_);
    }
    if (std::holds_alternative<TWorkloadMeta::TContainerInfo>(objectMeta.Stop_)) {
        objectStatus.mutable_stop_status()->mutable_container_status()->set_container_name(std::get<TWorkloadMeta::TContainerInfo>(objectMeta.Stop_).ContainerName_);
    }
    if (std::holds_alternative<TWorkloadMeta::TContainerInfo>(objectMeta.Destroy_)) {
        objectStatus.mutable_destroy_status()->mutable_container_status()->set_container_name(std::get<TWorkloadMeta::TContainerInfo>(objectMeta.Destroy_).ContainerName_);
    }
}

void TWorkloadStatusRepository::RemoveObject(const TString& objectId) {
    TWriteGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    Objects_.erase(objectId);
}

API::TWorkloadStatus TWorkloadStatusRepository::GetObjectStatus(const TString& objectId, TUpdateHolderTargetPtr updateHolderTarget) const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    return GetObjectStatusNoGlobalLock(objectId, updateHolderTarget, false);
}

API::TWorkloadStatus TWorkloadStatusRepository::GetObjectStatusNoGlobalLock(
    const TString& objectId
    , TUpdateHolderTargetPtr updateHolderTarget
    , bool conditionsOnly
) const {
    TGuard<TMutex> g(GetObjectMutex(objectId));

    // WARNING: Do not copy full status here
    const API::TWorkloadStatus& fullStatus = Objects_.at(objectId).Status_;

    API::TWorkloadStatus status;
    if (conditionsOnly) {
        // Set object id only
        status.set_id(objectId);
    } else {
        status.CopyFrom(fullStatus);
    }

    bool changingSpecTimestamp = updateHolderTarget ? updateHolderTarget->WorkloadHasTarget(objectId) : false;
    RefreshObjectStatusConditions(status, fullStatus, TInstant::Zero(), false, changingSpecTimestamp);

    return status;
}

bool TWorkloadStatusRepository::HasObject(const TString& objectId) const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    return Objects_.FindPtr(objectId);
}

TVector<TString> TWorkloadStatusRepository::GetObjectIds() const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);

    TVector<TString> result;
    for (const auto& [objectId, info] : Objects_) {
        result.push_back(objectId);
    }
    return result;
}

API::EWorkloadTargetState TWorkloadStatusRepository::GetObjectTargetState(const TString& objectId) const {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    return Objects_.at(objectId).Status_.target_state();
}

void TWorkloadStatusRepository::UpdateObjectRevision(const TString& objectIdOrHash, ui32 revision) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectIdOrHash));
    API::TWorkloadStatus& workload = Objects_.at(objectIdOrHash).Status_;
    if (workload.revision() != revision) {
        workload.set_revision(revision);
        TInstant now = TInstant::Now();
        RefreshObjectStatusConditions(workload, workload, now);
    }
}

void TWorkloadStatusRepository::UpdateObjectSpecTimestamp(const TString& objectIdOrHash, ui64 specTimestamp) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectIdOrHash));
    API::TWorkloadStatus& workload = Objects_.at(objectIdOrHash).Status_;
    if (workload.spec_timestamp() != specTimestamp) {
        workload.set_spec_timestamp(specTimestamp);
        TInstant now = TInstant::Now();
        RefreshObjectStatusConditions(workload, workload, now);
    }
}

TStatusRepositoryCommon::TObjectState TWorkloadStatusRepository::UpdateObjectState(const TString& objectIdOrHash, TObjectState state) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectIdOrHash));
    API::TWorkloadStatus& workload = Objects_.at(objectIdOrHash).Status_;
    TStatusRepositoryCommon::TObjectState oldState = workload.state();
    if (oldState != state) {
        workload.set_state(std::get<API::EWorkloadState>(state));
        TInstant now = TInstant::Now();
        RefreshObjectStatusConditions(workload, workload, now);
    }
    return oldState;
}

TVector<TString> TWorkloadStatusRepository::GetObjectIdsByHash(const TString& /* objectIdOrHash */) {
    // An empty vector is returned in order to have no duplicates of id in logs
    return {};
}

TVector<TStatusRepositoryCommon::TCacheObject> TWorkloadStatusRepository::GetCacheObjectIdsAndRevisionsByHash(const TString& /* objectIdOrHash */) {
    // No CacheObjects in this status repository
    return {};
}

void TWorkloadStatusRepository::UpdateObjectTargetState(const TString& objectId, API::EWorkloadTargetState state) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    API::TWorkloadStatus& workload = Objects_.at(objectId).Status_;
    if (workload.target_state() != state) {
        workload.set_target_state(state);
        TInstant now = TInstant::Now();
        RefreshObjectStatusConditions(workload, workload, now);
    }
}

template <>
API::THttpGetStatus& TWorkloadStatusRepository::MutableHookStatus<API::THttpGetStatus>(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    switch (networkHookType) {
        case NStatusRepositoryTypes::ENetworkHookType::READINESS:
            return *Objects_.at(objectId).Status_.mutable_readiness_status()->mutable_http_get_status();
        case NStatusRepositoryTypes::ENetworkHookType::LIVENESS:
            return *Objects_.at(objectId).Status_.mutable_liveness_status()->mutable_http_get_status();
        case NStatusRepositoryTypes::ENetworkHookType::STOP:
            return *Objects_.at(objectId).Status_.mutable_stop_status()->mutable_http_get_status();
        case NStatusRepositoryTypes::ENetworkHookType::DESTROY:
            return *Objects_.at(objectId).Status_.mutable_destroy_status()->mutable_http_get_status();
    }
}

template <>
API::TTcpCheckStatus& TWorkloadStatusRepository::MutableHookStatus<API::TTcpCheckStatus>(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    switch (networkHookType) {
        case NStatusRepositoryTypes::ENetworkHookType::READINESS:
            return *Objects_.at(objectId).Status_.mutable_readiness_status()->mutable_tcp_check_status();
        case NStatusRepositoryTypes::ENetworkHookType::LIVENESS:
            return *Objects_.at(objectId).Status_.mutable_liveness_status()->mutable_tcp_check_status();
        case NStatusRepositoryTypes::ENetworkHookType::STOP:
            ythrow yexception() << "Stop hook is not provided for tcp protocol";
        case NStatusRepositoryTypes::ENetworkHookType::DESTROY:
            ythrow yexception() << "Destroy hook is not provided for tcp protocol";
    }
}

template <>
const API::THttpGetStatus& TWorkloadStatusRepository::GetHookStatus<API::THttpGetStatus>(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    switch (networkHookType) {
        case NStatusRepositoryTypes::ENetworkHookType::READINESS:
            return Objects_.at(objectId).Status_.readiness_status().http_get_status();
        case NStatusRepositoryTypes::ENetworkHookType::LIVENESS:
            return Objects_.at(objectId).Status_.liveness_status().http_get_status();
        case NStatusRepositoryTypes::ENetworkHookType::STOP:
            return Objects_.at(objectId).Status_.stop_status().http_get_status();
        case NStatusRepositoryTypes::ENetworkHookType::DESTROY:
            return Objects_.at(objectId).Status_.destroy_status().http_get_status();
    }
}

template <>
const API::TTcpCheckStatus& TWorkloadStatusRepository::GetHookStatus<API::TTcpCheckStatus>(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    switch (networkHookType) {
        case NStatusRepositoryTypes::ENetworkHookType::READINESS:
            return Objects_.at(objectId).Status_.readiness_status().tcp_check_status();
        case NStatusRepositoryTypes::ENetworkHookType::LIVENESS:
            return Objects_.at(objectId).Status_.liveness_status().tcp_check_status();
        case NStatusRepositoryTypes::ENetworkHookType::STOP:
            ythrow yexception() << "Stop hook is not provided for tcp protocol";
        case NStatusRepositoryTypes::ENetworkHookType::DESTROY:
            ythrow yexception() << "Destroy hook is not provided for tcp protocol";
    }
}

template <class TNetworkHookStatus>
void TWorkloadStatusRepository::UpdateNetworkHookStartTime(const TString& objectId, const TInstant& time, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));

    *MutableHookStatus<TNetworkHookStatus>(objectId, networkHookType).mutable_current()->mutable_start_time() = NSupport::ToTimestamp(time);
}

template <class TNetworkHookStatus>
void TWorkloadStatusRepository::UpdateNetworkHookDeathTime(const TString& objectId, const TInstant& time, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));

    *MutableHookStatus<TNetworkHookStatus>(objectId, networkHookType).mutable_current()->mutable_death_time() = NSupport::ToTimestamp(time);
}

template <class TNetworkHookStatus>
TInstant TWorkloadStatusRepository::GetNetworkHookStartTime(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    return NSupport::ToInstant(GetHookStatus<TNetworkHookStatus>(objectId, networkHookType).last().start_time());
}

template <class TNetworkHookStatus, class ENetworkHookState>
void TWorkloadStatusRepository::UpdateNetworkHookState(const TString& objectId, ENetworkHookState checkState, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    MutableHookStatus<TNetworkHookStatus>(objectId, networkHookType).mutable_current()->set_state(checkState);
}

template <class TNetworkHookStatus>
void TWorkloadStatusRepository::UpdateNetworkHookConsecutiveFailuresAndSuccessesCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType, bool success) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    TNetworkHookStatus& status = MutableHookStatus<TNetworkHookStatus>(objectId, networkHookType);

    if (success) {
        ui32 prev = status.time_limit().consecutive_successes_counter();
        status.mutable_time_limit()->set_consecutive_successes_counter(prev + 1);
        status.mutable_time_limit()->set_consecutive_failures_counter(0);
    } else {
        ui32 prev = status.time_limit().consecutive_failures_counter();
        status.mutable_time_limit()->set_consecutive_failures_counter(prev + 1);
        status.mutable_time_limit()->set_consecutive_successes_counter(0);
    }
}

template <class TNetworkHookStatus>
void TWorkloadStatusRepository::UpdateNetworkHookFailReason(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType, const TString& reason) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    MutableHookStatus<TNetworkHookStatus>(objectId, networkHookType).mutable_current()->set_fail_reason(NSupport::Truncate(reason));
}

template <class TNetworkHookStatus, NStatusRepositoryTypes::EHookBackend backend>
void TWorkloadStatusRepository::IncrementNetworkHookSuccessCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    TNetworkHookStatus& status = MutableHookStatus<TNetworkHookStatus>(objectId, networkHookType);

    status.set_success_counter(status.success_counter() + 1);

    TUnistatObjectHelper::Instance().PushNetworkSignal(
        objectId
        , backend
        , networkHookType
        , TUnistatObjectHelper::ENetworkSignalType::SUCCESS
        , 1
    );
}

template <class TNetworkHookStatus, NStatusRepositoryTypes::EHookBackend backend>
void TWorkloadStatusRepository::IncrementNetworkHookErrorCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    TNetworkHookStatus& status = MutableHookStatus<TNetworkHookStatus>(objectId, networkHookType);

    status.set_error_counter(status.error_counter() + 1);

    TUnistatObjectHelper::Instance().PushNetworkSignal(
        objectId
        , backend
        , networkHookType
        , TUnistatObjectHelper::ENetworkSignalType::ERROR
        , 1
    );
}

template <class TNetworkHookStatus, NStatusRepositoryTypes::EHookBackend backend>
void TWorkloadStatusRepository::IncrementNetworkHookTimeoutCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    TNetworkHookStatus& status = MutableHookStatus<TNetworkHookStatus>(objectId, networkHookType);

    status.set_timeout_counter(status.timeout_counter() + 1);

    TUnistatObjectHelper::Instance().PushNetworkSignal(
        objectId
        , backend
        , networkHookType
        , TUnistatObjectHelper::ENetworkSignalType::TIMEOUT
        , 1
    );
}

template <class TNetworkHookStatus>
ui32 TWorkloadStatusRepository::GetNetworkHookConsecutiveSuccessesCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    return GetHookStatus<TNetworkHookStatus>(objectId, networkHookType).time_limit().consecutive_successes_counter();
}

template <class TNetworkHookStatus>
ui32 TWorkloadStatusRepository::GetNetworkHookConsecutiveFailuresCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    return GetHookStatus<TNetworkHookStatus>(objectId, networkHookType).time_limit().consecutive_failures_counter();
}

template <class TNetworkHookStatus>
ui32 TWorkloadStatusRepository::GetNetworkHookSuccessCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    return GetHookStatus<TNetworkHookStatus>(objectId, networkHookType).success_counter();
}

template <class TNetworkHookStatus>
ui32 TWorkloadStatusRepository::GetNetworkHookErrorCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    return GetHookStatus<TNetworkHookStatus>(objectId, networkHookType).error_counter();
}

template <class TNetworkHookStatus>
ui32 TWorkloadStatusRepository::GetNetworkHookTimeoutCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    return GetHookStatus<TNetworkHookStatus>(objectId, networkHookType).timeout_counter();
}

template <class TNetworkHookStatus, class TReturnType, typename TAttemptStateType, TAttemptStateType SuccessState>
TReturnType TWorkloadStatusRepository::CaptureNetworkHookStatus(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    if (GetHookStatus<TNetworkHookStatus>(objectId, networkHookType).has_current()) {
        const auto& currentAttempt = GetHookStatus<TNetworkHookStatus>(objectId, networkHookType).current();

        *MutableHookStatus<TNetworkHookStatus>(objectId, networkHookType).mutable_last() = currentAttempt;
        if (currentAttempt.state() != SuccessState) {
            *MutableHookStatus<TNetworkHookStatus>(objectId, networkHookType).mutable_last_failed() = currentAttempt;
        }

        MutableHookStatus<TNetworkHookStatus>(objectId, networkHookType).clear_current();
    }
    return GetHookStatus<TNetworkHookStatus>(objectId, networkHookType).last();
}

void TWorkloadStatusRepository::UpdateHttpHookStartTime(const TString& objectId, const TInstant& time, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    UpdateNetworkHookStartTime<API::THttpGetStatus>(objectId, time, networkHookType);
}

void TWorkloadStatusRepository::UpdateHttpHookDeathTime(const TString& objectId, const TInstant& time, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    UpdateNetworkHookDeathTime<API::THttpGetStatus>(objectId, time, networkHookType);
}

TInstant TWorkloadStatusRepository::GetHttpHookStartTime(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    return GetNetworkHookStartTime<API::THttpGetStatus>(objectId, networkHookType);
}

void TWorkloadStatusRepository::UpdateHttpHookState(const TString& objectId, API::EHttpGetState checkState, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    UpdateNetworkHookState<API::THttpGetStatus, API::EHttpGetState>(objectId, checkState, networkHookType);
}

void TWorkloadStatusRepository::UpdateHttpHookConsecutiveFailuresAndSuccessesCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType, bool success) {
    UpdateNetworkHookConsecutiveFailuresAndSuccessesCounter<API::THttpGetStatus>(objectId, networkHookType, success);
}

void TWorkloadStatusRepository::UpdateHttpHookFailReason(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType, const TString& reason) {
    UpdateNetworkHookFailReason<API::THttpGetStatus>(objectId, networkHookType, reason);
}

void TWorkloadStatusRepository::UpdateHttpHookInnerFailReason(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType, const TString& reason) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    MutableHookStatus<API::THttpGetStatus>(objectId, networkHookType).mutable_current()->set_inner_fail_reason(reason);
}

ui32 TWorkloadStatusRepository::GetHttpHookConsecutiveSuccessesCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    return GetNetworkHookConsecutiveSuccessesCounter<API::THttpGetStatus>(objectId, networkHookType);
}

ui32 TWorkloadStatusRepository::GetHttpHookConsecutiveFailuresCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    return GetNetworkHookConsecutiveFailuresCounter<API::THttpGetStatus>(objectId, networkHookType);
}

void TWorkloadStatusRepository::UpdateTcpCheckStartTime(const TString& objectId, const TInstant& time, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    UpdateNetworkHookStartTime<API::TTcpCheckStatus>(objectId, time, networkHookType);
}

void TWorkloadStatusRepository::UpdateTcpCheckDeathTime(const TString& objectId, const TInstant& time, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    UpdateNetworkHookDeathTime<API::TTcpCheckStatus>(objectId, time, networkHookType);
}

TInstant TWorkloadStatusRepository::GetTcpCheckStartTime(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    return GetNetworkHookStartTime<API::TTcpCheckStatus>(objectId, networkHookType);
}

void TWorkloadStatusRepository::UpdateTcpCheckState(const TString& objectId, API::ETcpCheckState checkState, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    UpdateNetworkHookState<API::TTcpCheckStatus, API::ETcpCheckState>(objectId, checkState, networkHookType);
}

void TWorkloadStatusRepository::UpdateTcpCheckConsecutiveFailuresAndSuccessesCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType, bool success) {
    UpdateNetworkHookConsecutiveFailuresAndSuccessesCounter<API::TTcpCheckStatus>(objectId, networkHookType, success);
}

void TWorkloadStatusRepository::IncrementTcpCheckSuccessCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    IncrementNetworkHookSuccessCounter<API::TTcpCheckStatus, NStatusRepositoryTypes::EHookBackend::TCP>(objectId, networkHookType);
}

void TWorkloadStatusRepository::IncrementTcpCheckErrorCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    IncrementNetworkHookErrorCounter<API::TTcpCheckStatus, NStatusRepositoryTypes::EHookBackend::TCP>(objectId, networkHookType);
}

void TWorkloadStatusRepository::IncrementTcpCheckTimeoutCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    IncrementNetworkHookTimeoutCounter<API::TTcpCheckStatus, NStatusRepositoryTypes::EHookBackend::TCP>(objectId, networkHookType);
}

void TWorkloadStatusRepository::UpdateTcpCheckFailReason(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType, const TString& reason) {
    UpdateNetworkHookFailReason<API::TTcpCheckStatus>(objectId, networkHookType, reason);
}

ui32 TWorkloadStatusRepository::GetTcpCheckConsecutiveSuccessesCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    return GetNetworkHookConsecutiveSuccessesCounter<API::TTcpCheckStatus>(objectId, networkHookType);
}

ui32 TWorkloadStatusRepository::GetTcpCheckConsecutiveFailuresCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    return GetNetworkHookConsecutiveFailuresCounter<API::TTcpCheckStatus>(objectId, networkHookType);
}

ui32 TWorkloadStatusRepository::GetTcpCheckSuccessCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    return GetNetworkHookSuccessCounter<API::TTcpCheckStatus>(objectId, networkHookType);
}

ui32 TWorkloadStatusRepository::GetTcpCheckErrorCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    return GetNetworkHookErrorCounter<API::TTcpCheckStatus>(objectId, networkHookType);
}

ui32 TWorkloadStatusRepository::GetTcpCheckTimeoutCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    return GetNetworkHookTimeoutCounter<API::TTcpCheckStatus>(objectId, networkHookType);
}

void TWorkloadStatusRepository::IncrementHttpRequestsSuccessCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    IncrementNetworkHookSuccessCounter<API::THttpGetStatus, NStatusRepositoryTypes::EHookBackend::HTTP>(objectId, networkHookType);
}

void TWorkloadStatusRepository::IncrementHttpRequestsErrorCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    IncrementNetworkHookErrorCounter<API::THttpGetStatus, NStatusRepositoryTypes::EHookBackend::HTTP>(objectId, networkHookType);
}

void TWorkloadStatusRepository::IncrementHttpRequestsTimeoutCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    IncrementNetworkHookTimeoutCounter<API::THttpGetStatus, NStatusRepositoryTypes::EHookBackend::HTTP>(objectId, networkHookType);
}

void TWorkloadStatusRepository::IncrementHttpRequestsWrongAnswerCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    API::THttpGetStatus& status = MutableHookStatus<API::THttpGetStatus>(objectId, networkHookType);

    status.set_wrong_answer_counter(status.wrong_answer_counter() + 1);

    TUnistatObjectHelper::Instance().PushNetworkSignal(
        objectId
        , NStatusRepositoryTypes::EHookBackend::HTTP
        , networkHookType
        , TUnistatObjectHelper::ENetworkSignalType::WRONG_ANSWER
        , 1
    );
}

ui32 TWorkloadStatusRepository::GetHttpRequestsSuccessCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    return GetNetworkHookSuccessCounter<API::THttpGetStatus>(objectId, networkHookType);
}

ui32 TWorkloadStatusRepository::GetHttpRequestsErrorCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    return GetNetworkHookErrorCounter<API::THttpGetStatus>(objectId, networkHookType);
}

ui32 TWorkloadStatusRepository::GetHttpRequestsTimeoutCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    return GetNetworkHookTimeoutCounter<API::THttpGetStatus>(objectId, networkHookType);
}

ui32 TWorkloadStatusRepository::GetHttpRequestsWrongAnswerCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    return GetHookStatus<API::THttpGetStatus>(objectId, networkHookType).wrong_answer_counter();
}

API::THttpGetStatus::TAttemptFeedback TWorkloadStatusRepository::CaptureHttpHookStatus(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    return CaptureNetworkHookStatus<API::THttpGetStatus, API::THttpGetStatus::TAttemptFeedback, API::EHttpGetState, API::EHttpGetState_SUCCESS>(objectId, networkHookType);
}

API::TTcpCheckStatus::TAttemptFeedback TWorkloadStatusRepository::CaptureTcpHookStatus(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType) {
    return CaptureNetworkHookStatus<API::TTcpCheckStatus, API::TTcpCheckStatus::TAttemptFeedback, API::ETcpCheckState, API::ETcpCheckState_SUCCESS>(objectId, networkHookType);
}

void TWorkloadStatusRepository::UpdateStopUnixSignalSendTime(const TString& objectId, const TInstant& time) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    API::TUnixSignalStatus& status = MutableStopUnixSignalStatus(objectId);

    *status.mutable_current()->mutable_send_time() = NSupport::ToTimestamp(time);
}

void TWorkloadStatusRepository::UpdateStopUnixSignalState(const TString& objectId, API::EUnixSignalState state) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    API::TUnixSignalStatus& status = MutableStopUnixSignalStatus(objectId);

    status.mutable_current()->set_state(state);
}

void TWorkloadStatusRepository::UpdateStopUnixSignalConsecutiveFailuresAndSuccessesCounter(const TString& objectId, bool success) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    API::TUnixSignalStatus& status = MutableStopUnixSignalStatus(objectId);

    if (success) {
        ui32 prev = status.time_limit().consecutive_successes_counter();
        status.mutable_time_limit()->set_consecutive_successes_counter(prev + 1);
        status.mutable_time_limit()->set_consecutive_failures_counter(0);
    } else {
        ui32 prev = status.time_limit().consecutive_failures_counter();
        status.mutable_time_limit()->set_consecutive_failures_counter(prev + 1);
        status.mutable_time_limit()->set_consecutive_successes_counter(0);
    }
}

void TWorkloadStatusRepository::UpdateStopUnixSignalFailReason(const TString& objectId, const TString& reason) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    API::TUnixSignalStatus& status = MutableStopUnixSignalStatus(objectId);

    status.mutable_current()->set_fail_reason(NSupport::Truncate(reason));
}

void TWorkloadStatusRepository::IncrementStopUnixSignalSuccessCounter(const TString& objectId) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    API::TUnixSignalStatus& status = MutableStopUnixSignalStatus(objectId);

    status.set_success_counter(status.success_counter() + 1);
}

void TWorkloadStatusRepository::IncrementStopUnixSignalErrorCounter(const TString& objectId) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    API::TUnixSignalStatus& status = MutableStopUnixSignalStatus(objectId);

    status.set_error_counter(status.error_counter() + 1);
}

TInstant TWorkloadStatusRepository::GetStopUnixSignalSendTime(const TString& objectId) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    const API::TUnixSignalStatus& status = GetStopUnixSignalStatus(objectId);

    return NSupport::ToInstant(status.last().send_time());
}

ui32 TWorkloadStatusRepository::GetStopUnixSignalConsecutiveSuccessesCounter(const TString& objectId) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    const API::TUnixSignalStatus& status = GetStopUnixSignalStatus(objectId);

    return status.time_limit().consecutive_successes_counter();
}

ui32 TWorkloadStatusRepository::GetStopUnixSignalConsecutiveFailuresCounter(const TString& objectId) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    const API::TUnixSignalStatus& status = GetStopUnixSignalStatus(objectId);

    return status.time_limit().consecutive_failures_counter();
}

ui32 TWorkloadStatusRepository::GetStopUnixSignalSuccessCounter(const TString& objectId) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    const API::TUnixSignalStatus& status = GetStopUnixSignalStatus(objectId);

    return status.success_counter();
}

ui32 TWorkloadStatusRepository::GetStopUnixSignalErrorCounter(const TString& objectId) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    const API::TUnixSignalStatus& status = GetStopUnixSignalStatus(objectId);

    return status.error_counter();
}

API::TUnixSignalStatus::TAttemptFeedback TWorkloadStatusRepository::CaptureStopUnixSignalStatus(const TString& objectId) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    API::TUnixSignalStatus& status = MutableStopUnixSignalStatus(objectId);

    if (status.has_current()) {
        *status.mutable_last() = status.current();
        if (status.current().state() != API::EUnixSignalState_SUCCESS) {
            *status.mutable_last_failed() = status.current();
        }

        status.clear_current();
    }

    return status.last();
}

void TWorkloadStatusRepository::ClearStopStatus(const TString& objectId) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    API::TWorkloadStatus& status = Objects_.at(objectId).Status_;

    // We want to clear everything except the flag indicating a stop policy and container name
    bool hasStop = status.stop_status().has_stop();
    TString containerName = status.stop_status().container_status().container_name();

    status.clear_stop_status();
    status.mutable_stop_status()->set_has_stop(hasStop);
    if (!containerName.empty()) {
        status.mutable_stop_status()->mutable_container_status()->set_container_name(containerName);
    }
}

void TWorkloadStatusRepository::ClearDestroyStatus(const TString& objectId) {
    TReadGuardBase<TLightRWLock> guard(GlobalObjectLock_);
    TGuard<TMutex> g(GetObjectMutex(objectId));
    API::TWorkloadStatus& status = Objects_.at(objectId).Status_;

    // We want to clear everything except the flag indicating a destroy policy and container name
    bool hasDestroy = status.destroy_status().has_destroy();
    TString containerName = status.destroy_status().container_status().container_name();

    status.clear_destroy_status();
    status.mutable_destroy_status()->set_has_destroy(hasDestroy);
    if (!containerName.empty()) {
        status.mutable_destroy_status()->mutable_container_status()->set_container_name(containerName);
    }
}

API::TUnixSignalStatus& TWorkloadStatusRepository::MutableStopUnixSignalStatus(const TString& objectId) {
    // Must lock before call
    return *Objects_.at(objectId).Status_.mutable_stop_status()->mutable_unix_signal_status();
}

const API::TUnixSignalStatus& TWorkloadStatusRepository::GetStopUnixSignalStatus(const TString& objectId) {
    // Must lock before call
    return Objects_.at(objectId).Status_.stop_status().unix_signal_status();
}

void TWorkloadStatusRepository::RefreshObjectStatusConditions(
    API::TWorkloadStatus& targetStatus
    , const API::TWorkloadStatus& workload
    , const TInstant& now
    , bool refreshTime
    , bool changingSpecTimestamp
) {
    RefreshObjectStatusReady(
        *targetStatus.mutable_ready()
        , workload
        , now
        , refreshTime
        , changingSpecTimestamp
    );
    RefreshObjectStatusInProgress(
        *targetStatus.mutable_in_progress()
        , workload
        , now
        , refreshTime
        , changingSpecTimestamp
    );
    RefreshObjectStatusFailed(
        *targetStatus.mutable_failed()
        , workload
        , now
        , refreshTime
    );
}

void TWorkloadStatusRepository::RefreshObjectStatusReady(
    API::TCondition& ready
    , const API::TWorkloadStatus& workload
    , const TInstant& now
    , bool refreshTime
    , bool changingSpecTimestamp
) {
    NSupport::CopyConditonStatusAndTimestamp(ready, workload.ready());
    API::EConditionStatus prevStatus = ready.status();

    ready.set_message("");
    if (changingSpecTimestamp) {
        ready.set_message(NSupport::APPLYING_OBJECT_SPEC_MESSAGE);
    }
    // Even if there are no container or http/tcp for readiness check, it will be true when we have check of start container alive.
    if (workload.readiness_status().has_readiness()) {
        ready.set_reason(NSupport::ExtractState(API::EWorkloadState_Name(workload.state())));

        switch (workload.state()) {
            case API::EWorkloadState_ACTIVE:
                if (workload.target_state() == API::EWorkloadTarget_ACTIVE && !changingSpecTimestamp) {
                    ready.set_status(API::EConditionStatus_TRUE);
                } else {
                    ready.set_status(API::EConditionStatus_FALSE);
                    NSupport::SetApplyingObjectSpecState(ready);
                }
                break;
            case API::EWorkloadState_REMOVED:
                if (workload.target_state() == API::EWorkloadTarget_REMOVED && !changingSpecTimestamp) {
                    ready.set_status(API::EConditionStatus_TRUE);
                } else {
                    ready.set_status(API::EConditionStatus_FALSE);
                    NSupport::SetApplyingObjectSpecState(ready);
                }
                break;
            case API::EWorkloadState_ACTIVATING:
            case API::EWorkloadState_DEACTIVATED_BY_LIVENESS:
            case API::EWorkloadState_DEACTIVATING:
            case API::EWorkloadState_DEAD:
            case API::EWorkloadState_INIT_PROCESSES:
            case API::EWorkloadState_INVALID:
            case API::EWorkloadState_REMOVING:
            case API::EWorkloadState_READINESS_FAILED:
            case API::EWorkloadState_WAITING_FOR_BOX:
                ready.set_status(API::EConditionStatus_FALSE);
                break;
            case API::EWorkloadState_FAILED_TO_START_READINESS:
            case API::EWorkloadState_FAILED_TO_START_LIVENESS:
                // Probably we do not want to change FALSE status to UNKNOWN
                // However now it's always changes to UNKNOWN for symmetry with READY
                ready.set_status(API::EConditionStatus_UNKNOWN);
                break;
            case API::EWorkloadState_UNKNOWN:
                ready.set_status(API::EConditionStatus_UNKNOWN);
                break;
            // We don't want to use default
            // So we need this to handle all cases
            case API::EWorkloadState_INT_MIN_SENTINEL_DO_NOT_USE_:
            case API::EWorkloadState_INT_MAX_SENTINEL_DO_NOT_USE_:
                ready.set_status(API::EConditionStatus_UNKNOWN);
                break;
        }
    } else {
        ready.set_status(API::EConditionStatus_TRUE);
        ready.set_reason("NO_READINESS");
    }

    if (refreshTime && prevStatus != ready.status()) {
        *ready.mutable_last_transition_time() = NSupport::ToTimestamp(now);
    }
}

void TWorkloadStatusRepository::RefreshObjectStatusInProgress(
    API::TCondition& inProgress
    , const API::TWorkloadStatus& workload
    , const TInstant& now
    , bool refreshTime
    , bool changingSpecTimestamp
) {
    NSupport::CopyConditonStatusAndTimestamp(inProgress, workload.in_progress());
    API::EConditionStatus prevStatus = inProgress.status();

    inProgress.set_reason(NSupport::ExtractState(API::EWorkloadState_Name(workload.state())));
    inProgress.set_message("");
    if (changingSpecTimestamp) {
        inProgress.set_message(NSupport::APPLYING_OBJECT_SPEC_MESSAGE);
    }

    switch (workload.state()) {
        case API::EWorkloadState_ACTIVATING:
        case API::EWorkloadState_DEACTIVATED_BY_LIVENESS:
        case API::EWorkloadState_DEACTIVATING:
        case API::EWorkloadState_INIT_PROCESSES:
        case API::EWorkloadState_REMOVING:
        case API::EWorkloadState_READINESS_FAILED:
        case API::EWorkloadState_WAITING_FOR_BOX:
        case API::EWorkloadState_FAILED_TO_START_READINESS:
        case API::EWorkloadState_FAILED_TO_START_LIVENESS:
            inProgress.set_status(API::EConditionStatus_TRUE);
            break;
        case API::EWorkloadState_ACTIVE:
            if (workload.target_state() == API::EWorkloadTarget_ACTIVE && !changingSpecTimestamp) {
                inProgress.set_status(API::EConditionStatus_FALSE);
            } else {
                inProgress.set_status(API::EConditionStatus_TRUE);
                NSupport::SetApplyingObjectSpecState(inProgress);
            }
            break;
        case API::EWorkloadState_REMOVED:
            if (workload.target_state() == API::EWorkloadTarget_REMOVED && !changingSpecTimestamp) {
                inProgress.set_status(API::EConditionStatus_FALSE);
            } else {
                inProgress.set_status(API::EConditionStatus_TRUE);
                NSupport::SetApplyingObjectSpecState(inProgress);
            }
            break;
        case API::EWorkloadState_DEAD:
        case API::EWorkloadState_INVALID:
            inProgress.set_status(API::EConditionStatus_FALSE);
            break;
        case API::EWorkloadState_UNKNOWN:
            inProgress.set_status(API::EConditionStatus_UNKNOWN);
            break;
        // We don't want to use default
        // So we need this to handle all cases
        case API::EWorkloadState_INT_MIN_SENTINEL_DO_NOT_USE_:
        case API::EWorkloadState_INT_MAX_SENTINEL_DO_NOT_USE_:
            inProgress.set_status(API::EConditionStatus_UNKNOWN);
            break;
    }

    if (refreshTime && prevStatus != inProgress.status()) {
        *inProgress.mutable_last_transition_time() = NSupport::ToTimestamp(now);
    }
}

void TWorkloadStatusRepository::RefreshObjectStatusFailed(
    API::TCondition& failed
    , const API::TWorkloadStatus& workload
    , const TInstant& now
    , bool refreshTime
) {
    NSupport::CopyConditonStatusAndTimestamp(failed, workload.failed());
    API::EConditionStatus prevStatus = failed.status();

    switch (workload.state()) {
        case API::EWorkloadState_REMOVED:
        case API::EWorkloadState_ACTIVATING:
        case API::EWorkloadState_DEACTIVATED_BY_LIVENESS:
        case API::EWorkloadState_DEACTIVATING:
        case API::EWorkloadState_READINESS_FAILED:
        case API::EWorkloadState_INIT_PROCESSES:
        case API::EWorkloadState_REMOVING:
        case API::EWorkloadState_WAITING_FOR_BOX:
            // UNKNOWN -> FALSE
            // TRUE and FALSE do not change
            if (failed.status() == API::EConditionStatus_UNKNOWN) {
                failed.set_status(API::EConditionStatus_FALSE);
            }
            break;
        case API::EWorkloadState_ACTIVE:
            failed.set_status(API::EConditionStatus_FALSE);
            failed.set_reason(NSupport::ExtractState(API::EWorkloadState_Name(workload.state())));
            break;
        case API::EWorkloadState_INVALID:
        case API::EWorkloadState_DEAD:
        case API::EWorkloadState_FAILED_TO_START_READINESS:
        case API::EWorkloadState_FAILED_TO_START_LIVENESS:
            failed.set_status(API::EConditionStatus_TRUE);
            failed.set_reason(NSupport::ExtractState(API::EWorkloadState_Name(workload.state())));
            break;
        case API::EWorkloadState_UNKNOWN:
            failed.set_status(API::EConditionStatus_UNKNOWN);
            failed.set_reason(NSupport::ExtractState(API::EWorkloadState_Name(workload.state())));
            break;
        // We don't want to use default
        // So we need this to handle all cases
        case API::EWorkloadState_INT_MIN_SENTINEL_DO_NOT_USE_:
        case API::EWorkloadState_INT_MAX_SENTINEL_DO_NOT_USE_:
            failed.set_status(API::EConditionStatus_UNKNOWN);
            failed.set_reason(NSupport::ExtractState(API::EWorkloadState_Name(workload.state())));
            break;
    }

    if (refreshTime && prevStatus != failed.status()) {
        *failed.mutable_last_transition_time() = NSupport::ToTimestamp(now);
    }
}

const TMutex& TWorkloadStatusRepository::GetObjectMutex(const TString& objectId) const {
    const TObjectInfo* ptr = Objects_.FindPtr(objectId);
    Y_ENSURE(ptr, "Object '" << objectId << "' not found at TWorkloadStatusRepository");
    return ptr->Mutex_;
}

} // namespace NInfra::NPodAgent
