#pragma once

#include "status_repository_common.h"

#include <infra/pod_agent/libs/pod_agent/object_meta/object_meta.h>

#include <util/generic/vector.h>
#include <util/generic/map.h>
#include <util/generic/set.h>

namespace NInfra::NPodAgent {

/**
    Thread-safe workload status repository
*/
class TWorkloadStatusRepository: public TStatusRepositoryCommon {
private:
    struct TObjectInfo {
        TMutex Mutex_;
        API::TWorkloadStatus Status_;
    };

public:
    TWorkloadStatusRepository()
        : TStatusRepositoryCommon(NStatusRepositoryTypes::EObjectType::WORKLOAD)
    {}

    virtual void UpdateSpecTimestamp(TInstant currentTime) override final;
    virtual void PatchTotalStatus(API::TPodAgentStatus& status, TUpdateHolderTargetPtr updateHolderTarget, bool conditionsOnly) const override final;
    virtual bool NeedLongTickPeriod(const TString& objectIdOrHash, TUpdateHolderTargetPtr updateHolderTarget) const override final;

    // Common object methods
    virtual void UpdateObjectRevision(const TString& objectIdOrHash, ui32 revision) override final;
    virtual void UpdateObjectSpecTimestamp(const TString& objectId, ui64 specTimestamp) override final;
    virtual TObjectState UpdateObjectState(const TString& objectIdOrHash, TObjectState state) override final;
    virtual TVector<TString> GetObjectIdsByHash(const TString& objectIdOrHash) override final;
    virtual TVector<TCacheObject> GetCacheObjectIdsAndRevisionsByHash(const TString& objectIdOrHash) override final;

    // Workload specific methods

    // Workload
    void AddObject(const TWorkloadMeta& objectMeta);
    void RemoveObject(const TString& objectId);
    bool HasObject(const TString& objectId) const;

    API::TWorkloadStatus GetObjectStatus(const TString& objectId, TUpdateHolderTargetPtr updateHolderTarget = nullptr) const;
    API::EWorkloadTargetState GetObjectTargetState(const TString& objectId) const;
    TVector<TString> GetObjectIds() const;

    void UpdateObjectTargetState(const TString& objectId, API::EWorkloadTargetState state);

    // Http
    void UpdateHttpHookStartTime(const TString& objectId, const TInstant& time, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    void UpdateHttpHookDeathTime(const TString& objectId, const TInstant& time, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    void UpdateHttpHookState(const TString& objectId, API::EHttpGetState checkState, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    void UpdateHttpHookConsecutiveFailuresAndSuccessesCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType, bool success);
    void UpdateHttpHookFailReason(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType, const TString& reason);
    void UpdateHttpHookInnerFailReason(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType, const TString& reason);
    void IncrementHttpRequestsSuccessCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    void IncrementHttpRequestsErrorCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    void IncrementHttpRequestsTimeoutCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    void IncrementHttpRequestsWrongAnswerCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);

    TInstant GetHttpHookStartTime(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    ui32 GetHttpHookConsecutiveSuccessesCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    ui32 GetHttpHookConsecutiveFailuresCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    ui32 GetHttpRequestsSuccessCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    ui32 GetHttpRequestsErrorCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    ui32 GetHttpRequestsTimeoutCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    ui32 GetHttpRequestsWrongAnswerCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);

    API::THttpGetStatus::TAttemptFeedback CaptureHttpHookStatus(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);

    // Tcp
    void UpdateTcpCheckStartTime(const TString& objectId, const TInstant& time, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    void UpdateTcpCheckDeathTime(const TString& objectId, const TInstant& time, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    void UpdateTcpCheckState(const TString& objectId, API::ETcpCheckState checkState, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    void UpdateTcpCheckConsecutiveFailuresAndSuccessesCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType, bool success);
    void UpdateTcpCheckFailReason(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType, const TString& reason);
    void IncrementTcpCheckSuccessCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    void IncrementTcpCheckErrorCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    void IncrementTcpCheckTimeoutCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);

    TInstant GetTcpCheckStartTime(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    ui32 GetTcpCheckConsecutiveSuccessesCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    ui32 GetTcpCheckConsecutiveFailuresCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    ui32 GetTcpCheckSuccessCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    ui32 GetTcpCheckErrorCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    ui32 GetTcpCheckTimeoutCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);

    API::TTcpCheckStatus::TAttemptFeedback CaptureTcpHookStatus(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);

    // Unix signal
    void UpdateStopUnixSignalSendTime(const TString& objectId, const TInstant& time);
    void UpdateStopUnixSignalState(const TString& objectId, API::EUnixSignalState state);
    void UpdateStopUnixSignalConsecutiveFailuresAndSuccessesCounter(const TString& objectId, bool success);
    void UpdateStopUnixSignalFailReason(const TString& objectId, const TString& reason);

    void IncrementStopUnixSignalSuccessCounter(const TString& objectId);
    void IncrementStopUnixSignalErrorCounter(const TString& objectId);

    TInstant GetStopUnixSignalSendTime(const TString& objectId);
    ui32 GetStopUnixSignalConsecutiveSuccessesCounter(const TString& objectId);
    ui32 GetStopUnixSignalConsecutiveFailuresCounter(const TString& objectId);
    ui32 GetStopUnixSignalSuccessCounter(const TString& objectId);
    ui32 GetStopUnixSignalErrorCounter(const TString& objectId);

    API::TUnixSignalStatus::TAttemptFeedback CaptureStopUnixSignalStatus(const TString& objectId);

    // Full hook/policy status
    void ClearStopStatus(const TString& objectId);
    void ClearDestroyStatus(const TString& objectId);

private:
    virtual const TLightRWLock& GetGlobalContainerLock() const override final;
    virtual const TMutex& GetLocalContainerLock(const NStatusRepositoryTypes::TContainerDescription& container) const override final;
    virtual API::TContainerStatus* GetMutableContainerStatus(const NStatusRepositoryTypes::TContainerDescription& container) override final;
    virtual const API::TContainerStatus* GetContainerStatus(const NStatusRepositoryTypes::TContainerDescription& container) override final;

    const TMutex& GetObjectMutex(const TString &objectId) const;
    API::TWorkloadStatus GetObjectStatusNoGlobalLock(
        const TString& objectId
        , TUpdateHolderTargetPtr updateHolderTarget
        , bool conditionsOnly
   ) const;

    static void RefreshObjectStatusConditions(
        API::TWorkloadStatus& targetStatus
        , const API::TWorkloadStatus& workload
        , const TInstant& now
        , bool refreshTime = true
        , bool changingSpecTimestamp = false
    );
    static void RefreshObjectStatusReady(
        API::TCondition& ready
        , const API::TWorkloadStatus& workload
        , const TInstant& now
        , bool refreshTime
        , bool changingSpecTimestamp
    );
    static void RefreshObjectStatusInProgress(
        API::TCondition& inProgress
        , const API::TWorkloadStatus& workload
        , const TInstant& now
        , bool refreshTime
        , bool changingSpecTimestamp
    );
    static void RefreshObjectStatusFailed(
        API::TCondition& failed
        , const API::TWorkloadStatus& workload
        , const TInstant& now
        , bool refreshTime
    );

    // Network
    template <class TNetworkHookStatus>
    TNetworkHookStatus& MutableHookStatus(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    template <class TNetworkHookStatus>
    const TNetworkHookStatus& GetHookStatus(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);

    template <class TNetworkHookStatus>
    void UpdateNetworkHookStartTime(const TString& objectId, const TInstant& time, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    template <class TNetworkHookStatus>
    void UpdateNetworkHookDeathTime(const TString& objectId, const TInstant& time, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    template <class TNetworkHookStatus>
    TInstant GetNetworkHookStartTime(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    template <class TNetworkHookStatus, class ENetworkHookState>
    void UpdateNetworkHookState(const TString& objectId, ENetworkHookState checkState, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    template <class TNetworkHookStatus>
    void UpdateNetworkHookConsecutiveFailuresAndSuccessesCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType, bool success);
    template <class TNetworkHookStatus>
    void UpdateNetworkHookFailReason(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType, const TString& reason);
    template <class TNetworkHookStatus, NStatusRepositoryTypes::EHookBackend backend>
    void IncrementNetworkHookSuccessCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    template <class TNetworkHookStatus, NStatusRepositoryTypes::EHookBackend backend>
    void IncrementNetworkHookErrorCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    template <class TNetworkHookStatus, NStatusRepositoryTypes::EHookBackend backend>
    void IncrementNetworkHookTimeoutCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    template <class TNetworkHookStatus>
    ui32 GetNetworkHookConsecutiveSuccessesCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    template <class TNetworkHookStatus>
    ui32 GetNetworkHookConsecutiveFailuresCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    template <class TNetworkHookStatus>
    ui32 GetNetworkHookSuccessCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    template <class TNetworkHookStatus>
    ui32 GetNetworkHookErrorCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    template <class TNetworkHookStatus>
    ui32 GetNetworkHookTimeoutCounter(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);
    template <class TNetworkHookStatus, class TReturnType, typename TAttemptStateType, TAttemptStateType SuccessState>
    TReturnType CaptureNetworkHookStatus(const TString& objectId, NStatusRepositoryTypes::ENetworkHookType networkHookType);

    // Unix signal
    API::TUnixSignalStatus& MutableStopUnixSignalStatus(const TString& objectId);
    const API::TUnixSignalStatus& GetStopUnixSignalStatus(const TString& objectId);

private:
    TMap<TString, TObjectInfo> Objects_;
    TLightRWLock GlobalObjectLock_;
};

using TWorkloadStatusRepositoryPtr = TIntrusivePtr<TWorkloadStatusRepository>;

} // namespace NInfra::NPodAgent
