#include "workload_status_repository.h"
#include "support_functions.h"
#include "test_functions.h"

#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/update_holder/test_lib/test_functions.h>

#include <google/protobuf/util/message_differencer.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NWorkloadStatusRepositoryTest {

static const TString boxId = "my_box";

Y_UNIT_TEST_SUITE(WorkloadStatusRepositorySuite) {

Y_UNIT_TEST(TestGetObjectType) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    const TString id = "my_workload";
    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

    UNIT_ASSERT_EQUAL_C(holder->GetObjectType(), NStatusRepositoryTypes::EObjectType::WORKLOAD, ToString(holder->GetObjectType()));
}

Y_UNIT_TEST(TestUpdateHolderNotProvided) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    const TString id = "my_workload";
    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

    API::TPodAgentStatus patchStatus;
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->PatchTotalStatus(patchStatus, nullptr, false), yexception, "UpdateHolderTarget not provided");
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->NeedLongTickPeriod(id, nullptr), yexception, "UpdateHolderTarget not provided");
}

Y_UNIT_TEST(TestAddWorkload) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString id = "my_workload";

    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));
    API::TWorkloadStatus status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL(API::EWorkloadState_UNKNOWN, status.state());
    UNIT_ASSERT_EQUAL_C(
        status.start().container_name()
        , "start_container"
        , status.start().container_name()
    );
    UNIT_ASSERT_EQUAL_C(
        status.readiness_status().container_status().container_name()
        , ""
        , status.readiness_status().container_status().container_name()
    );
    UNIT_ASSERT_EQUAL_C(
        status.liveness_status().container_status().container_name()
        , ""
        , status.liveness_status().container_status().container_name()
    );
    UNIT_ASSERT_EQUAL_C(
        status.stop_status().container_status().container_name()
        , ""
        , status.stop_status().container_status().container_name()
    );
    UNIT_ASSERT_EQUAL_C(
        status.destroy_status().container_status().container_name()
        , ""
        , status.destroy_status().container_status().container_name()
    );
}

Y_UNIT_TEST(TestAddWorkloadWithContainersInStatus) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString id = "my_workload";

    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id, 1, 0, 0, true));
    API::TWorkloadStatus status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL(API::EWorkloadState_UNKNOWN, status.state());
        UNIT_ASSERT_EQUAL_C(
        status.start().container_name()
        , "start_container"
        , status.start().container_name()
    );
    UNIT_ASSERT_EQUAL_C(
        status.readiness_status().container_status().container_name()
        , "readiness_container"
        , status.readiness_status().container_status().container_name()
    );
    UNIT_ASSERT_EQUAL_C(
        status.liveness_status().container_status().container_name()
        , "liveness_container"
        , status.liveness_status().container_status().container_name()
    );
    UNIT_ASSERT_EQUAL_C(
        status.stop_status().container_status().container_name()
        , "stop_container"
        , status.stop_status().container_status().container_name()
    );
    UNIT_ASSERT_EQUAL_C(
        status.destroy_status().container_status().container_name()
        , "destroy_container"
        , status.destroy_status().container_status().container_name()
    );
}

Y_UNIT_TEST(TestWorkloadPatchTotalStatus) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString id = "my_workload";

    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));
    // Changing something in the status so that it is not equal to the default one
    holder->UpdateObjectState(id, API::EWorkloadState_ACTIVE);
    API::TWorkloadStatus status = holder->GetObjectStatus(id);

    {
        API::TPodAgentStatus patchStatus;
        holder->PatchTotalStatus(patchStatus, updateHolder->GetUpdateHolderTarget(), false);
        UNIT_ASSERT_EQUAL_C(
            patchStatus.workloads(0).id()
            , id
            , patchStatus.workloads(0).id()
        );
        UNIT_ASSERT_EQUAL_C(
            patchStatus.workloads(0).state()
            , API::EWorkloadState_ACTIVE
            , API::EWorkloadState_Name(patchStatus.workloads(0).state())
        );
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(status, patchStatus.workloads(0)));
    }

    {
        API::TPodAgentStatus patchStatus;
        holder->PatchTotalStatus(patchStatus, updateHolder->GetUpdateHolderTarget(), true);
        UNIT_ASSERT_EQUAL_C(
            patchStatus.workloads(0).id()
            , id
            , patchStatus.workloads(0).id()
        );
        UNIT_ASSERT_EQUAL_C(
            patchStatus.workloads(0).state()
            // conditionsOnly = true
            , API::EWorkloadState_UNKNOWN
            , API::EWorkloadState_Name(patchStatus.workloads(0).state())
        );
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(status.ready(), patchStatus.workloads(0).ready()));
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(status.in_progress(), patchStatus.workloads(0).in_progress()));
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(status.failed(), patchStatus.workloads(0).failed()));
    }
}

Y_UNIT_TEST(TestWorkloadWithoutReainessPatchTotalStatus) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString id = "my_workload";

    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id, 0));
    // Failed status of workload is not patched to pod status if workload fails
    {
        holder->UpdateObjectState(id, API::EWorkloadState_INVALID);
        API::TWorkloadStatus status = holder->GetObjectStatus(id);
        API::TPodAgentStatus patchStatus;
        patchStatus.mutable_failed()->set_status(API::EConditionStatus_FALSE);
        patchStatus.mutable_failed()->set_reason("ALL_OK");
        holder->PatchTotalStatus(patchStatus, updateHolder->GetUpdateHolderTarget(), false);

        const auto workloadStatus = patchStatus.workloads(0);
        UNIT_ASSERT_EQUAL_C(workloadStatus.id(), id, workloadStatus.id());
        UNIT_ASSERT_EQUAL_C(
            workloadStatus.state()
            , API::EWorkloadState_INVALID
            , API::EWorkloadState_Name(workloadStatus.state())
        );

        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(status, workloadStatus));
        UNIT_ASSERT(!google::protobuf::util::MessageDifferencer::Equals(status.failed(), patchStatus.failed()));

        UNIT_ASSERT_EQUAL(patchStatus.failed().status(), API::EConditionStatus_FALSE);
        UNIT_ASSERT_EQUAL_C(patchStatus.failed().reason(), "ALL_OK", patchStatus.failed().reason());
        UNIT_ASSERT_EQUAL(workloadStatus.failed().status(), API::EConditionStatus_TRUE);
        UNIT_ASSERT_EQUAL_C(workloadStatus.failed().reason(), "INVALID", workloadStatus.failed().reason());
    }
    // In progress status of workload is not patched to pod status if workload is in progress
    {
        holder->UpdateObjectState(id, API::EWorkloadState_ACTIVATING);
        API::TWorkloadStatus status = holder->GetObjectStatus(id);
        API::TPodAgentStatus patchStatus;
        patchStatus.mutable_in_progress()->set_status(API::EConditionStatus_FALSE);
        patchStatus.mutable_in_progress()->set_reason("ALL_OK");
        holder->PatchTotalStatus(patchStatus, updateHolder->GetUpdateHolderTarget(), false);

        const auto workloadStatus = patchStatus.workloads(0);
        UNIT_ASSERT_EQUAL_C(workloadStatus.id(), id, workloadStatus.id());
        UNIT_ASSERT_EQUAL_C(
            workloadStatus.state()
            , API::EWorkloadState_ACTIVATING
            , API::EWorkloadState_Name(workloadStatus.state())
        );

        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(status, workloadStatus));
        UNIT_ASSERT(!google::protobuf::util::MessageDifferencer::Equals(status.in_progress(), patchStatus.in_progress()));

        UNIT_ASSERT_EQUAL(patchStatus.in_progress().status(), API::EConditionStatus_FALSE);
        UNIT_ASSERT_EQUAL_C(patchStatus.in_progress().reason(), "ALL_OK", patchStatus.in_progress().reason());
        UNIT_ASSERT_EQUAL(workloadStatus.in_progress().status(), API::EConditionStatus_TRUE);
        UNIT_ASSERT_EQUAL_C(workloadStatus.in_progress().reason(), "ACTIVATING", workloadStatus.in_progress().reason());
    }
}

Y_UNIT_TEST(TestWorkloadRevisionAndSpecTimestamp) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString id = "my_workload";

    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));
    API::TWorkloadStatus status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL_C(status.spec_timestamp(), 1, status.spec_timestamp());
    UNIT_ASSERT_EQUAL_C(status.revision(), 2, status.revision());

    holder->UpdateObjectSpecTimestamp(id, 3);
    holder->UpdateObjectRevision(id, 4);
    status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL_C(status.spec_timestamp(), 3, status.spec_timestamp());
    UNIT_ASSERT_EQUAL_C(status.revision(), 4, status.revision());
}

Y_UNIT_TEST(TestUpdateWorkloadState) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    const TString id = "my_workload";
    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));
    holder->UpdateObjectState(id, API::EWorkloadState_ACTIVE);
    API::TWorkloadStatus status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL(API::EWorkloadState_ACTIVE, status.state());
}

Y_UNIT_TEST(TestWorkloadReadyNoReadiness) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString id = "my_workload";
    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id, 0));

    holder->UpdateObjectState(id, API::EWorkloadState_UNKNOWN);
    API::TWorkloadStatus status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EWorkloadState_UNKNOWN, status.state());
    auto ready = status.ready();
    UNIT_ASSERT_EQUAL(ready.last_transition_time().seconds(), 0);
    UNIT_ASSERT_EQUAL(ready.last_transition_time().nanos(), 0);
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT_EQUAL("NO_READINESS", ready.reason());
    UNIT_ASSERT(!holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(id, API::EWorkloadState_INVALID);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EWorkloadState_INVALID, status.state());
    ready = status.ready();
    UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT_EQUAL("NO_READINESS", ready.reason());
    UNIT_ASSERT(holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(id, API::EWorkloadState_ACTIVE);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EWorkloadState_ACTIVE, status.state());
    ready = status.ready();
    UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT_EQUAL("NO_READINESS", ready.reason());
    UNIT_ASSERT(holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(id, API::EWorkloadState_REMOVED);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EWorkloadState_REMOVED, status.state());
    ready = status.ready();
    UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT_EQUAL("NO_READINESS", ready.reason());
    UNIT_ASSERT(!holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectTargetState(id, API::EWorkloadTarget_REMOVED);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EWorkloadState_REMOVED, status.state());
    ready = status.ready();
    UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT_EQUAL("NO_READINESS", ready.reason());
    UNIT_ASSERT(holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateSpecTimestamp(TInstant::Now());
    updateHolder->SetWorkloadTarget(NObjectTargetTestLib::CreateWorkloadTargetSimple(id));
    holder->UpdateObjectTargetState(id, API::EWorkloadTarget_ACTIVE);
    holder->UpdateObjectState(id, API::EWorkloadState_ACTIVE);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EWorkloadState_ACTIVE, status.state());
    ready  = status.ready();
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT_EQUAL("NO_READINESS", ready.reason());
    UNIT_ASSERT_EQUAL("Changing spec_timestamp", ready.message());
    UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
    UNIT_ASSERT(!holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));
}

Y_UNIT_TEST(TestUpdateWorkloadStateAndReadyStatus) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString id = "my_workload";
    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

    holder->UpdateObjectState(id, API::EWorkloadState_UNKNOWN);
    API::TWorkloadStatus status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EWorkloadState_UNKNOWN, status.state());
    auto ready = status.ready();
    UNIT_ASSERT_EQUAL(ready.last_transition_time().seconds(), 0);
    UNIT_ASSERT_EQUAL(ready.last_transition_time().nanos(), 0);
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_UNKNOWN);
    UNIT_ASSERT(!holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(id, API::EWorkloadState_INVALID);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EWorkloadState_INVALID, status.state());
    ready = status.ready();
    UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT(holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(id, API::EWorkloadState_REMOVED);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EWorkloadState_REMOVED, status.state());
    ready = status.ready();
    UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT(!holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(id, API::EWorkloadState_ACTIVE);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EWorkloadState_ACTIVE, status.state());
    ready = status.ready();
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT_EQUAL(NSupport::ExtractState(API::EWorkloadState_Name(API::EWorkloadState_ACTIVE)), ready.reason());
    UNIT_ASSERT_EQUAL("", ready.message());
    UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
    UNIT_ASSERT(holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(id, API::EWorkloadState_FAILED_TO_START_READINESS);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EWorkloadState_FAILED_TO_START_READINESS, status.state());
    ready = status.ready();
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_UNKNOWN);
    UNIT_ASSERT_EQUAL(NSupport::ExtractState(API::EWorkloadState_Name(API::EWorkloadState_FAILED_TO_START_READINESS)), ready.reason());
    UNIT_ASSERT_EQUAL("", ready.message());
    UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);

    holder->UpdateObjectState(id, API::EWorkloadState_ACTIVE);
    holder->UpdateSpecTimestamp(TInstant::Now());
    updateHolder->SetWorkloadTarget(NObjectTargetTestLib::CreateWorkloadTargetSimple(id));
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EWorkloadState_ACTIVE, status.state());
    ready  = status.ready();
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT_EQUAL(NSupport::APPLYING_OBJECT_SPEC, ready.reason());
    UNIT_ASSERT_EQUAL(NSupport::APPLYING_OBJECT_SPEC_MESSAGE, ready.message());
    UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
    UNIT_ASSERT(!holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL(API::EWorkloadState_ACTIVE, status.state());
    ready  = status.ready();
    UNIT_ASSERT_EQUAL(ready.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT_EQUAL(NSupport::ExtractState(API::EWorkloadState_Name(API::EWorkloadState_ACTIVE)), ready.reason());
    UNIT_ASSERT_EQUAL("", ready.message());
    UNIT_ASSERT(ready.last_transition_time().seconds() != 0 || ready.last_transition_time().nanos() != 0);
}

Y_UNIT_TEST(TestUpdateWorkloadStateAndInProgressStatus) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString id = "my_workload";
    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

    holder->UpdateObjectState(id, API::EWorkloadState_UNKNOWN);
    API::TWorkloadStatus status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EWorkloadState_UNKNOWN, status.state());
    auto inProgress = status.in_progress();
    UNIT_ASSERT_EQUAL(inProgress.last_transition_time().seconds(), 0);
    UNIT_ASSERT_EQUAL(inProgress.last_transition_time().nanos(), 0);
    UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_UNKNOWN);

    holder->UpdateObjectState(id, API::EWorkloadState_ACTIVE);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EWorkloadState_ACTIVE, status.state());
    inProgress = status.in_progress();
    UNIT_ASSERT(inProgress.last_transition_time().seconds() != 0 || inProgress.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_FALSE);

    holder->UpdateObjectState(id, API::EWorkloadState_ACTIVATING);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EWorkloadState_ACTIVATING, status.state());
    inProgress = status.in_progress();
    UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT_EQUAL(NSupport::ExtractState(API::EWorkloadState_Name(API::EWorkloadState_ACTIVATING)), inProgress.reason());
    UNIT_ASSERT_EQUAL("", inProgress.message());
    UNIT_ASSERT(inProgress.last_transition_time().seconds() != 0 || inProgress.last_transition_time().nanos() != 0);

    holder->UpdateObjectState(id, API::EWorkloadState_ACTIVE);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EWorkloadState_ACTIVE, status.state());
    inProgress = status.in_progress();
    UNIT_ASSERT(inProgress.last_transition_time().seconds() != 0 || inProgress.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_FALSE);

    holder->UpdateSpecTimestamp(TInstant::Now());
    updateHolder->SetWorkloadTarget(NObjectTargetTestLib::CreateWorkloadTargetSimple(id));
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EWorkloadState_ACTIVE, status.state());
    inProgress = status.in_progress();
    UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT_EQUAL(NSupport::APPLYING_OBJECT_SPEC, inProgress.reason());
    UNIT_ASSERT_EQUAL(NSupport::APPLYING_OBJECT_SPEC_MESSAGE, inProgress.message());
    UNIT_ASSERT(inProgress.last_transition_time().seconds() != 0 || inProgress.last_transition_time().nanos() != 0);

    status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL(API::EWorkloadState_ACTIVE, status.state());
    inProgress = status.in_progress();
    UNIT_ASSERT_EQUAL(inProgress.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT_EQUAL(NSupport::ExtractState(API::EWorkloadState_Name(API::EWorkloadState_ACTIVE)), inProgress.reason());
    UNIT_ASSERT_EQUAL("", inProgress.message());
    UNIT_ASSERT(inProgress.last_transition_time().seconds() != 0 || inProgress.last_transition_time().nanos() != 0);
}

Y_UNIT_TEST(TestUpdateWorkloadStateAndFailedStatus) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    const TString id = "my_workload";
    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

    holder->UpdateObjectState(id, API::EWorkloadState_UNKNOWN);
    API::TWorkloadStatus status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EWorkloadState_UNKNOWN, status.state());
    auto failed = status.failed();
    UNIT_ASSERT_EQUAL(failed.last_transition_time().seconds(), 0);
    UNIT_ASSERT_EQUAL(failed.last_transition_time().nanos(), 0);
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_UNKNOWN);
    UNIT_ASSERT(!holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(id, API::EWorkloadState_REMOVED);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EWorkloadState_REMOVED, status.state());
    failed = status.failed();
    UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT(!holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(id, API::EWorkloadState_ACTIVE);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EWorkloadState_ACTIVE, status.state());
    failed = status.failed();
    UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT(holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(id, API::EWorkloadState_INVALID);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EWorkloadState_INVALID, status.state());
    failed = status.failed();
    UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT(holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(id, API::EWorkloadState_REMOVED);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EWorkloadState_REMOVED, status.state());
    failed = status.failed();
    UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_TRUE);
    UNIT_ASSERT(!holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateObjectState(id, API::EWorkloadState_ACTIVE);
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EWorkloadState_ACTIVE, status.state());
    failed = status.failed();
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT_EQUAL(NSupport::ExtractState(API::EWorkloadState_Name(API::EWorkloadState_ACTIVE)), failed.reason());
    UNIT_ASSERT_EQUAL("", failed.message());
    UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);
    UNIT_ASSERT(holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    holder->UpdateSpecTimestamp(TInstant::Now());
    updateHolder->SetWorkloadTarget(NObjectTargetTestLib::CreateWorkloadTargetSimple(id));
    status = holder->GetObjectStatus(id, updateHolder->GetUpdateHolderTarget());
    UNIT_ASSERT_EQUAL(API::EWorkloadState_ACTIVE, status.state());
    failed  = status.failed();
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT_EQUAL(NSupport::ExtractState(API::EWorkloadState_Name(API::EWorkloadState_ACTIVE)), failed.reason());
    UNIT_ASSERT_EQUAL("", failed.message());
    UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);
    UNIT_ASSERT(!holder->NeedLongTickPeriod(id, updateHolder->GetUpdateHolderTarget()));

    status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL(API::EWorkloadState_ACTIVE, status.state());
    failed  = status.failed();
    UNIT_ASSERT_EQUAL(failed.status(), API::EConditionStatus_FALSE);
    UNIT_ASSERT_EQUAL(NSupport::ExtractState(API::EWorkloadState_Name(API::EWorkloadState_ACTIVE)), failed.reason());
    UNIT_ASSERT_EQUAL("", failed.message());
    UNIT_ASSERT(failed.last_transition_time().seconds() != 0 || failed.last_transition_time().nanos() != 0);
}

Y_UNIT_TEST(TestUpdateWorkloadNotFound) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    const TString id = "my_workload";
    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));
    UNIT_ASSERT_EXCEPTION_CONTAINS(holder->UpdateObjectState(id + "_other", API::EWorkloadState_WAITING_FOR_BOX), yexception, "not found at TWorkloadStatusRepository");
}

Y_UNIT_TEST(TestWorkloadTargetState) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    const TString id = "my_workload";

    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

    holder->UpdateObjectTargetState(id, API::EWorkloadTarget_ACTIVE);
    UNIT_ASSERT_EQUAL(holder->GetObjectTargetState(id), API::EWorkloadTarget_ACTIVE);

     holder->UpdateObjectTargetState(id, API::EWorkloadTarget_REMOVED);
    UNIT_ASSERT_EQUAL(holder->GetObjectTargetState(id), API::EWorkloadTarget_REMOVED);
}

Y_UNIT_TEST(TestGetWorkloadIds) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    const TString id = "my_workload";
    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

    UNIT_ASSERT_EQUAL(
        holder->GetObjectIds()
        , TVector<TString>(
            {
                id
            }
        )
    );

    // Must be empty
    UNIT_ASSERT_EQUAL(
        holder->GetObjectIdsByHash(id)
        , TVector<TString>()
    );
    UNIT_ASSERT_EQUAL(
        holder->GetCacheObjectIdsAndRevisionsByHash(id)
        , TVector<TStatusRepositoryCommon::TCacheObject>()
    );
}

Y_UNIT_TEST(TestWorkloadCaptureContainerStatus) {
    auto test = [](
        NStatusRepositoryTypes::TContainerDescription::EContainerType containerType
        , std::function<
            void(
                TWorkloadStatusRepositoryPtr& holder
                , const NStatusRepositoryTypes::TContainerDescription& container
            )
        > initCurrentAttempt
        , bool isFailedAttempt
    ) {
        const TString id = "my_workload";
        TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
        holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));
        NStatusRepositoryTypes::TContainerDescription container(
            id
            , NStatusRepositoryTypes::EObjectType::WORKLOAD
            , containerType
        );

        initCurrentAttempt(holder, container);

        auto workloadStatus = holder->GetObjectStatus(id);
        auto curStatus = GetWorkloadContainerHookStatus(workloadStatus, containerType).current();
        auto lastStatus = GetWorkloadContainerHookStatus(workloadStatus, containerType).last();
        auto lastFailedStatus = GetWorkloadContainerHookStatus(workloadStatus, containerType).last_failed();
        API::TContainerStatus::TAttemptFeedback emptyStatus;

        UNIT_ASSERT(!google::protobuf::util::MessageDifferencer::Equals(curStatus, lastStatus));
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(emptyStatus, lastStatus));
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(emptyStatus, lastFailedStatus));

        // First iteration check that capture work as expected
        // Second iteration check that empty current doesn't push to last
        for (size_t i = 0; i < 2; ++i) {
            holder->CaptureContainerStatus(container);

            workloadStatus = holder->GetObjectStatus(id);
            auto newCurStatus = GetWorkloadContainerHookStatus(workloadStatus, containerType).current();
            auto newLastStatus = GetWorkloadContainerHookStatus(workloadStatus, containerType).last();
            auto newLastFailedStatus = GetWorkloadContainerHookStatus(workloadStatus, containerType).last_failed();

            UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(emptyStatus, newCurStatus));
            UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(curStatus, newLastStatus));
            UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(isFailedAttempt ? curStatus : emptyStatus, newLastFailedStatus));
        }
    };

    TestWorkloadContainerHookAllTypes(
        [&test](NStatusRepositoryTypes::TContainerDescription::EContainerType containerType) {
            test(
                containerType
                , [](
                    TWorkloadStatusRepositoryPtr& holder
                    , const NStatusRepositoryTypes::TContainerDescription& container
                ) {
                    holder->UpdateContainerState(container, API::EContainerState_EXITED);
                    holder->UpdateContainerFailReason(container, "fail_reason");
                    holder->UpdateContainerStderr(container, "stderr_text");
                    holder->UpdateContainerStdout(container, "stdout_text");
                    holder->UpdateContainerReturnCode(container, 0);
                }
                , false
            );
        }
    );

    TestWorkloadContainerHookAllTypes(
        [&test](NStatusRepositoryTypes::TContainerDescription::EContainerType containerType) {
            test(
                containerType
                , [](
                    TWorkloadStatusRepositoryPtr& holder
                    , const NStatusRepositoryTypes::TContainerDescription& container
                ) {
                    holder->UpdateContainerState(container, API::EContainerState_EXITED);
                    holder->UpdateContainerReturnCode(container, 123);
                }
                , true
            );
        }
    );

    TestWorkloadContainerHookAllTypes(
        [&test](NStatusRepositoryTypes::TContainerDescription::EContainerType containerType) {
            test(
                containerType
                , [](
                    TWorkloadStatusRepositoryPtr& holder
                    , const NStatusRepositoryTypes::TContainerDescription& container
                ) {
                    holder->UpdateContainerState(container, API::EContainerState_TIMEOUT);
                    holder->UpdateContainerReturnCode(container, 0);
                }
                , true
            );
        }
    );
}

Y_UNIT_TEST(TestWorkloadInitContainer) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    const TString id = "my_workload";
    const size_t initSize = 10;
    const NStatusRepositoryTypes::TContainerDescription container(
        id
        , NStatusRepositoryTypes::EObjectType::WORKLOAD
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::INIT
        , 2
    );

    holder->AddObject(
        TWorkloadMeta(
            id
            , 0
            , 0
            , boxId

            , TWorkloadMeta::TContainerInfo("")
            , NObjectMetaTestLib::GenerateInitContainerInfo(initSize)

            , TWorkloadMeta::TEmptyInfo()
            , TWorkloadMeta::TEmptyInfo()
            , TWorkloadMeta::TEmptyInfo()
            , TWorkloadMeta::TEmptyInfo()
        )
    );
    UNIT_ASSERT_EQUAL_C(holder->GetObjectStatus(id).init().size(), initSize, holder->GetObjectStatus(id).init().size());

    holder->UpdateContainerFailReason(container, "Some fail reason");
    holder->IncrementContainerSystemFailureCounter(container);

    auto containerStatus = holder->GetObjectStatus(id).init(2);
    UNIT_ASSERT_EQUAL_C(containerStatus.current().fail_reason(), "Some fail reason", containerStatus.current().fail_reason());
    UNIT_ASSERT_EQUAL_C(containerStatus.system_failure_counter(), 1, containerStatus.system_failure_counter());
}

Y_UNIT_TEST(TestWorkloadInitContainerBigInitNumber) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    const TString id = "my_workload";
    const NStatusRepositoryTypes::TContainerDescription container(
        id
        , NStatusRepositoryTypes::EObjectType::WORKLOAD
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::INIT
        , 2
    );

    holder->AddObject(
        TWorkloadMeta(
            id
            , 0
            , 0
            , boxId

            , TWorkloadMeta::TContainerInfo("")
            , NObjectMetaTestLib::GenerateInitContainerInfo(0)

            , TWorkloadMeta::TEmptyInfo()
            , TWorkloadMeta::TEmptyInfo()
            , TWorkloadMeta::TEmptyInfo()
            , TWorkloadMeta::TEmptyInfo()
        )
    );

    UNIT_ASSERT_EXCEPTION_CONTAINS(
        holder->UpdateContainerFailReason(container, "Some fail reason")
        , yexception
        , "Too big init num for workload 'my_workload' InitSize = 0, InitNum = 2"
    );
    UNIT_ASSERT_EQUAL_C(holder->GetObjectStatus(id).init().size(), 0, holder->GetObjectStatus(id).init().size());
}

Y_UNIT_TEST(TestWorkloadLastHttpHookTime) {
    auto test = [](NStatusRepositoryTypes::ENetworkHookType type) {
        TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
        const TString id = "my_workload";

        holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

        holder->UpdateHttpHookStartTime(id, TInstant::MicroSeconds(123), type);
        holder->CaptureHttpHookStatus(id, type);
        UNIT_ASSERT_EQUAL(holder->GetHttpHookStartTime(id, type).MicroSeconds(), 123);

        TInstant time = TInstant::Now();
        holder->UpdateHttpHookStartTime(id, time, type);
        holder->CaptureHttpHookStatus(id, type);
        UNIT_ASSERT_EQUAL(holder->GetHttpHookStartTime(id, type), time);
    };

    TestHttpHookAllTypes(test);
}

Y_UNIT_TEST(TestWorkloadSetHttpHookState) {
    auto test = [](NStatusRepositoryTypes::ENetworkHookType type) {
        TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
        const TString id = "my_workload";

        holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

        holder->UpdateHttpHookState(id, API::EHttpGetState::EHttpGetState_RUNNING, type);
        auto status = holder->GetObjectStatus(id);
        UNIT_ASSERT_EQUAL(GetHttpHookStatus(status, type).current().state(), API::EHttpGetState::EHttpGetState_RUNNING);

        holder->UpdateHttpHookState(id, API::EHttpGetState::EHttpGetState_FAILURE, type);
        status = holder->GetObjectStatus(id);
        UNIT_ASSERT_EQUAL(GetHttpHookStatus(status, type).current().state(), API::EHttpGetState::EHttpGetState_FAILURE);
    };

    TestHttpHookAllTypes(test);
}

Y_UNIT_TEST(TestWorkloadHttpHookConsecutiveFailuresAndSuccessesCounter) {
    auto test = [](NStatusRepositoryTypes::ENetworkHookType type) {
        TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
        const TString id = "my_workload";

        holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));
        UNIT_ASSERT_EQUAL(holder->GetHttpHookConsecutiveSuccessesCounter(id, type), 0);
        UNIT_ASSERT_EQUAL(holder->GetHttpHookConsecutiveFailuresCounter(id, type), 0);

        holder->UpdateHttpHookConsecutiveFailuresAndSuccessesCounter(id, type, true);
        UNIT_ASSERT_EQUAL(holder->GetHttpHookConsecutiveSuccessesCounter(id, type), 1);
        UNIT_ASSERT_EQUAL(holder->GetHttpHookConsecutiveFailuresCounter(id, type), 0);

        holder->UpdateHttpHookConsecutiveFailuresAndSuccessesCounter(id, type, true);
        UNIT_ASSERT_EQUAL(holder->GetHttpHookConsecutiveSuccessesCounter(id, type), 2);
        UNIT_ASSERT_EQUAL(holder->GetHttpHookConsecutiveFailuresCounter(id, type), 0);

        holder->UpdateHttpHookConsecutiveFailuresAndSuccessesCounter(id, type, false);
        UNIT_ASSERT_EQUAL(holder->GetHttpHookConsecutiveSuccessesCounter(id, type), 0);
        UNIT_ASSERT_EQUAL(holder->GetHttpHookConsecutiveFailuresCounter(id, type), 1);

        holder->UpdateHttpHookConsecutiveFailuresAndSuccessesCounter(id, type, false);
        UNIT_ASSERT_EQUAL(holder->GetHttpHookConsecutiveSuccessesCounter(id, type), 0);
        UNIT_ASSERT_EQUAL(holder->GetHttpHookConsecutiveFailuresCounter(id, type), 2);
    };

    TestHttpHookAllTypes(test);
}

Y_UNIT_TEST(TestWorkloadUpdateHttpHookInnerFailReason) {
    auto test = [](NStatusRepositoryTypes::ENetworkHookType networkHookType) {
        TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
        const TString id = "my_workload";
        holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

        auto reason = GetHttpHookStatus(holder->GetObjectStatus(id), networkHookType).current().inner_fail_reason();
        UNIT_ASSERT_EQUAL_C(reason, "", reason);

        holder->UpdateHttpHookInnerFailReason(id, networkHookType, "abc");
        reason = GetHttpHookStatus(holder->GetObjectStatus(id), networkHookType).current().inner_fail_reason();
        UNIT_ASSERT_EQUAL_C(reason, "abc", reason);

        holder->UpdateHttpHookInnerFailReason(id, networkHookType, "qwe");
        reason = GetHttpHookStatus(holder->GetObjectStatus(id), networkHookType).current().inner_fail_reason();
        UNIT_ASSERT_EQUAL_C(reason, "qwe", reason);
    };

    TestHttpHookAllTypes(test);
}

Y_UNIT_TEST(TestWorkloadCaptureHttpHookStatus) {
    auto test = [](
        NStatusRepositoryTypes::ENetworkHookType networkHookType
        , std::function<
            void(
                TWorkloadStatusRepositoryPtr& holder
                , const TString& workloadId
                , NStatusRepositoryTypes::ENetworkHookType networkHookType
            )
        > initCurrentAttempt
        , bool isFailedAttempt
    ) {
        TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
        const TString id = "my_workload";
        holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

        initCurrentAttempt(holder, id, networkHookType);

        auto workloadStatus = holder->GetObjectStatus(id);
        auto curStatus = GetHttpHookStatus(workloadStatus, networkHookType).current();
        auto lastStatus = GetHttpHookStatus(workloadStatus, networkHookType).last();
        auto lastFailedStatus = GetHttpHookStatus(workloadStatus, networkHookType).last_failed();
        API::THttpGetStatus::TAttemptFeedback emptyStatus;

        UNIT_ASSERT(!google::protobuf::util::MessageDifferencer::Equals(curStatus, lastStatus));
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(emptyStatus, lastStatus));
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(emptyStatus, lastFailedStatus));

        // First iteration check that capture work as expected
        // Second iteration check that empty current doesn't push to last
        for (size_t i = 0; i < 2; ++i) {
            holder->CaptureHttpHookStatus(id, networkHookType);

            workloadStatus = holder->GetObjectStatus(id);
            auto newCurStatus = GetHttpHookStatus(workloadStatus, networkHookType).current();
            auto newLastStatus = GetHttpHookStatus(workloadStatus, networkHookType).last();
            auto newLastFailedStatus = GetHttpHookStatus(workloadStatus, networkHookType).last_failed();

            UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(curStatus, newLastStatus));
            UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(emptyStatus, newCurStatus));
            UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(isFailedAttempt ? curStatus : emptyStatus, newLastFailedStatus));
        }
    };

    TestHttpHookAllTypes(
        [&test](NStatusRepositoryTypes::ENetworkHookType networkHookType) {
            test(
                networkHookType
                , [](
                    TWorkloadStatusRepositoryPtr& holder
                    , const TString& workloadId
                    , NStatusRepositoryTypes::ENetworkHookType networkHookType
                ) {
                    holder->UpdateHttpHookState(workloadId, API::EHttpGetState::EHttpGetState_SUCCESS, networkHookType);
                    holder->UpdateHttpHookFailReason(workloadId, networkHookType, "fail_reason");
                }
                , false
            );
        }
    );

    TestHttpHookAllTypes(
        [&test](NStatusRepositoryTypes::ENetworkHookType networkHookType) {
            test(
                networkHookType
                , [](
                    TWorkloadStatusRepositoryPtr& holder
                    , const TString& workloadId
                    , NStatusRepositoryTypes::ENetworkHookType networkHookType
                ) {
                    holder->UpdateHttpHookState(workloadId, API::EHttpGetState::EHttpGetState_FAILURE, networkHookType);
                    holder->UpdateHttpHookFailReason(workloadId, networkHookType, "fail_reason");
                }
                , true
            );
        }
    );
}

Y_UNIT_TEST(TestWorkloadHttpRequestsCounters) {
    auto test = [](NStatusRepositoryTypes::ENetworkHookType type) {
        TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
        const TString id = "my_workload";
        const ui32 cntIter = 10;

        holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

        for (ui32 i = 0; i < cntIter; ++i) {
            holder->IncrementHttpRequestsSuccessCounter(id, type);
            UNIT_ASSERT_EQUAL(holder->GetHttpRequestsSuccessCounter(id, type), i + 1);
        }

        for (ui32 i = 0; i < cntIter; ++i) {
            holder->IncrementHttpRequestsErrorCounter(id, type);
            UNIT_ASSERT_EQUAL(holder->GetHttpRequestsErrorCounter(id, type), i + 1);
        }

        for (ui32 i = 0; i < cntIter; ++i) {
            holder->IncrementHttpRequestsTimeoutCounter(id, type);
            UNIT_ASSERT_EQUAL(holder->GetHttpRequestsTimeoutCounter(id, type), i + 1);
        }

        for (ui32 i = 0; i < cntIter; ++i) {
            holder->IncrementHttpRequestsWrongAnswerCounter(id, type);
            UNIT_ASSERT_EQUAL(holder->GetHttpRequestsWrongAnswerCounter(id, type), i + 1);
        }
    };

    TestHttpHookAllTypes(test);
}

Y_UNIT_TEST(TestWorkloadTcpCheckMethods) {
    auto test = [](NStatusRepositoryTypes::ENetworkHookType type) {
        TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
        const TString id = "my_workload";

        holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

        holder->UpdateTcpCheckStartTime(id, TInstant::MicroSeconds(123), type);
        holder->CaptureTcpHookStatus(id, type);
        UNIT_ASSERT_EQUAL(holder->GetTcpCheckStartTime(id, type).MicroSeconds(), 123);

        TInstant time = TInstant::Now();
        holder->UpdateTcpCheckStartTime(id, time, type);
        holder->CaptureTcpHookStatus(id, type);
        UNIT_ASSERT_EQUAL(holder->GetTcpCheckStartTime(id, type), time);

        holder->UpdateTcpCheckState(id, API::ETcpCheckState::ETcpCheckState_RUNNING, type);
        auto status = holder->GetObjectStatus(id);
        UNIT_ASSERT_EQUAL(GetTcpCheckStatus(status, type).current().state(), API::ETcpCheckState::ETcpCheckState_RUNNING);

        holder->UpdateTcpCheckConsecutiveFailuresAndSuccessesCounter(id, type, false);
        status = holder->GetObjectStatus(id);
        UNIT_ASSERT_EQUAL(holder->GetTcpCheckConsecutiveSuccessesCounter(id, type), 0);
        UNIT_ASSERT_EQUAL(holder->GetTcpCheckConsecutiveFailuresCounter(id, type), 1);
        UNIT_ASSERT_EQUAL(GetTcpCheckStatus(status, type).time_limit().consecutive_successes_counter(), 0);
        UNIT_ASSERT_EQUAL(GetTcpCheckStatus(status, type).time_limit().consecutive_failures_counter(), 1);

        holder->UpdateTcpCheckConsecutiveFailuresAndSuccessesCounter(id, type, true);
        status = holder->GetObjectStatus(id);
        UNIT_ASSERT_EQUAL(holder->GetTcpCheckConsecutiveSuccessesCounter(id, type), 1);
        UNIT_ASSERT_EQUAL(holder->GetTcpCheckConsecutiveFailuresCounter(id, type), 0);
        UNIT_ASSERT_EQUAL(GetTcpCheckStatus(status, type).time_limit().consecutive_successes_counter(), 1);
        UNIT_ASSERT_EQUAL(GetTcpCheckStatus(status, type).time_limit().consecutive_failures_counter(), 0);
    };

    TestTcpHookAllTypes(test);
}

Y_UNIT_TEST(TestWorkloadCaptureTcpHookStatus) {
    auto test = [](
        NStatusRepositoryTypes::ENetworkHookType networkHookType
        , std::function<
            void(
                TWorkloadStatusRepositoryPtr& holder
                , const TString& workloadId
                , NStatusRepositoryTypes::ENetworkHookType networkHookType
            )
        > initCurrentAttempt
        , bool isFailedAttempt
    ) {
        TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
        const TString id = "my_workload";
        holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

        initCurrentAttempt(holder, id, networkHookType);

        auto workloadStatus = holder->GetObjectStatus(id);
        auto curStatus = GetTcpCheckStatus(workloadStatus, networkHookType).current();
        auto lastStatus = GetTcpCheckStatus(workloadStatus, networkHookType).last();
        auto lastFailedStatus = GetTcpCheckStatus(workloadStatus, networkHookType).last_failed();
        API::TTcpCheckStatus::TAttemptFeedback emptyStatus;

        UNIT_ASSERT(!google::protobuf::util::MessageDifferencer::Equals(curStatus, lastStatus));
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(emptyStatus, lastStatus));
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(emptyStatus, lastFailedStatus));

        // First iteration check that capture work as expected
        // Second iteration check that empty current doesn't push to last
        for (size_t i = 0; i < 2; ++i) {
            holder->CaptureTcpHookStatus(id, networkHookType);

            workloadStatus = holder->GetObjectStatus(id);
            auto newCurStatus = GetTcpCheckStatus(workloadStatus, networkHookType).current();
            auto newLastStatus = GetTcpCheckStatus(workloadStatus, networkHookType).last();
            auto newLastFailedStatus = GetTcpCheckStatus(workloadStatus, networkHookType).last_failed();

            UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(curStatus, newLastStatus));
            UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(emptyStatus, newCurStatus));
            UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(isFailedAttempt ? curStatus : emptyStatus, newLastFailedStatus));
        }
    };

    TestTcpHookAllTypes(
        [&test](NStatusRepositoryTypes::ENetworkHookType networkHookType) {
            test(
                networkHookType
                , [](
                    TWorkloadStatusRepositoryPtr& holder
                    , const TString& workloadId
                    , NStatusRepositoryTypes::ENetworkHookType networkHookType
                ) {
                    holder->UpdateTcpCheckState(workloadId, API::ETcpCheckState::ETcpCheckState_SUCCESS, networkHookType);
                    holder->UpdateTcpCheckFailReason(workloadId, networkHookType, "fail_reason");
                }
                , false
            );
        }
    );

    TestTcpHookAllTypes(
        [&test](NStatusRepositoryTypes::ENetworkHookType networkHookType) {
            test(
                networkHookType
                , [](
                    TWorkloadStatusRepositoryPtr& holder
                    , const TString& workloadId
                    , NStatusRepositoryTypes::ENetworkHookType networkHookType
                ) {
                    holder->UpdateTcpCheckState(workloadId, API::ETcpCheckState::ETcpCheckState_FAILURE, networkHookType);
                    holder->UpdateTcpCheckFailReason(workloadId, networkHookType, "fail_reason");
                }
                , true
            );
        }
    );
}

Y_UNIT_TEST(TestWorkloadTcpCheckCounters) {
    auto test = [](NStatusRepositoryTypes::ENetworkHookType type) {
        TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
        const TString id = "my_workload";
        const ui32 cntIter = 10;

        holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

        for (ui32 i = 0; i < cntIter; ++i) {
            holder->IncrementTcpCheckSuccessCounter(id, type);
            UNIT_ASSERT_EQUAL(holder->GetTcpCheckSuccessCounter(id, type), i + 1);
        }

        for (ui32 i = 0; i < cntIter; ++i) {
            holder->IncrementTcpCheckErrorCounter(id, type);
            UNIT_ASSERT_EQUAL(holder->GetTcpCheckErrorCounter(id, type), i + 1);
        }

        for (ui32 i = 0; i < cntIter; ++i) {
            holder->IncrementTcpCheckTimeoutCounter(id, type);
            UNIT_ASSERT_EQUAL(holder->GetTcpCheckTimeoutCounter(id, type), i + 1);
        }
    };

    TestTcpHookAllTypes(test);
}

Y_UNIT_TEST(TestWorkloadStopUnixSignalMethods) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    const TString id = "my_workload";

    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

    holder->UpdateStopUnixSignalSendTime(id, TInstant::MicroSeconds(123));
    holder->CaptureStopUnixSignalStatus(id);
    UNIT_ASSERT_EQUAL(holder->GetStopUnixSignalSendTime(id).MicroSeconds(), 123);

    TInstant time = TInstant::Now();
    holder->UpdateStopUnixSignalSendTime(id, time);
    holder->CaptureStopUnixSignalStatus(id);
    UNIT_ASSERT_EQUAL(holder->GetStopUnixSignalSendTime(id), time);

    holder->UpdateStopUnixSignalState(id, API::EUnixSignalState::EUnixSignalState_SUCCESS);
    auto status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL(GetStopUnixSignalStatus(status).current().state(), API::EUnixSignalState::EUnixSignalState_SUCCESS);

    holder->UpdateStopUnixSignalFailReason(id, "fail_reason");
    status = holder->GetObjectStatus(id);
    UNIT_ASSERT_EQUAL(GetStopUnixSignalStatus(status).current().fail_reason(), "fail_reason");
}

Y_UNIT_TEST(TestWorkloadStopUnixSignalCounters) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    const TString id = "my_workload";
    const ui32 cntIter = 10;

    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

    for (ui32 i = 0; i < cntIter; ++i) {
        holder->IncrementStopUnixSignalSuccessCounter(id);
        auto status = holder->GetObjectStatus(id);
        UNIT_ASSERT_EQUAL(holder->GetStopUnixSignalSuccessCounter(id), i + 1);
        UNIT_ASSERT_EQUAL(GetStopUnixSignalStatus(status).success_counter(), i + 1);
    }

    for (ui32 i = 0; i < cntIter; ++i) {
        holder->IncrementStopUnixSignalErrorCounter(id);
        auto status = holder->GetObjectStatus(id);
        UNIT_ASSERT_EQUAL(holder->GetStopUnixSignalErrorCounter(id), i + 1);
        UNIT_ASSERT_EQUAL(GetStopUnixSignalStatus(status).error_counter(), i + 1);
    }

    for (ui32 i = 0; i < cntIter; ++i) {
        holder->UpdateStopUnixSignalConsecutiveFailuresAndSuccessesCounter(id, false);
        auto status = holder->GetObjectStatus(id);
        UNIT_ASSERT_EQUAL(holder->GetStopUnixSignalConsecutiveSuccessesCounter(id), 0);
        UNIT_ASSERT_EQUAL(holder->GetStopUnixSignalConsecutiveFailuresCounter(id), i + 1);
        UNIT_ASSERT_EQUAL(GetStopUnixSignalStatus(status).time_limit().consecutive_successes_counter(), 0);
        UNIT_ASSERT_EQUAL(GetStopUnixSignalStatus(status).time_limit().consecutive_failures_counter(), i + 1);
    }

    for (ui32 i = 0; i < cntIter; ++i) {
        holder->UpdateStopUnixSignalConsecutiveFailuresAndSuccessesCounter(id, true);
        auto status = holder->GetObjectStatus(id);
        UNIT_ASSERT_EQUAL(holder->GetStopUnixSignalConsecutiveSuccessesCounter(id), i + 1);
        UNIT_ASSERT_EQUAL(holder->GetStopUnixSignalConsecutiveFailuresCounter(id), 0);
        UNIT_ASSERT_EQUAL(GetStopUnixSignalStatus(status).time_limit().consecutive_successes_counter(), i + 1);
        UNIT_ASSERT_EQUAL(GetStopUnixSignalStatus(status).time_limit().consecutive_failures_counter(), 0);
    }
}

Y_UNIT_TEST(TestWorkloadCaptureStopUnixSignalStatus) {
    auto test = [](
        std::function<
            void(
                TWorkloadStatusRepositoryPtr& holder
                , const TString& workloadId
            )
        > initCurrentAttempt
        , bool isFailedAttempt
    ) {
        TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
        const TString id = "my_workload";
        holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

        initCurrentAttempt(holder, id);

        auto workloadStatus = holder->GetObjectStatus(id);
        auto curStatus = GetStopUnixSignalStatus(workloadStatus).current();
        auto lastStatus = GetStopUnixSignalStatus(workloadStatus).last();
        auto lastFailedStatus = GetStopUnixSignalStatus(workloadStatus).last_failed();
        API::TUnixSignalStatus::TAttemptFeedback emptyStatus;

        UNIT_ASSERT(!google::protobuf::util::MessageDifferencer::Equals(curStatus, lastStatus));
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(emptyStatus, lastStatus));
        UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(emptyStatus, lastFailedStatus));

        // First iteration check that capture work as expected
        // Second iteration check that empty current doesn't push to last
        for (size_t i = 0; i < 2; ++i) {
            holder->CaptureStopUnixSignalStatus(id);

            workloadStatus = holder->GetObjectStatus(id);
            auto newCurStatus = GetStopUnixSignalStatus(workloadStatus).current();
            auto newLastStatus = GetStopUnixSignalStatus(workloadStatus).last();
            auto newLastFailedStatus = GetStopUnixSignalStatus(workloadStatus).last_failed();

            UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(curStatus, newLastStatus));
            UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(emptyStatus, newCurStatus));
            UNIT_ASSERT(google::protobuf::util::MessageDifferencer::Equals(isFailedAttempt ? curStatus : emptyStatus, newLastFailedStatus));
        }
    };

    test(
        [](
            TWorkloadStatusRepositoryPtr& holder
            , const TString& workloadId
        ) {
            holder->UpdateStopUnixSignalState(workloadId, API::EUnixSignalState::EUnixSignalState_SUCCESS);
            holder->UpdateStopUnixSignalFailReason(workloadId, "fail_reason");
        }
        , false
    );

    test(
        [](
            TWorkloadStatusRepositoryPtr& holder
            , const TString& workloadId
        ) {
            holder->UpdateStopUnixSignalState(workloadId, API::EUnixSignalState::EUnixSignalState_FAILURE);
            holder->UpdateStopUnixSignalFailReason(workloadId, "fail_reason");
        }
        , true
    );
}

Y_UNIT_TEST(TestWorkloadClearStopStatus) {
    {
        // Test with stop
        TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
        TUpdateHolderPtr updateHolder = new TUpdateHolder();
        const TString id = "my_workload";

        holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id, 1, 0, 0, true));

        NStatusRepositoryTypes::TContainerDescription stopContainer(
            id
            , NStatusRepositoryTypes::EObjectType::WORKLOAD
            , NStatusRepositoryTypes::TContainerDescription::EContainerType::STOP
        );
        holder->IncrementContainerSystemFailureCounter(stopContainer);

        API::TStopStatus stopStatus = holder->GetObjectStatus(id).stop_status();
        UNIT_ASSERT(stopStatus.has_stop());
        UNIT_ASSERT_EQUAL_C(
            stopStatus.container_status().container_name()
            , "stop_container"
            , stopStatus.container_status().container_name()
        );
        UNIT_ASSERT_EQUAL_C(
            stopStatus.container_status().system_failure_counter()
            , 1
            , stopStatus.container_status().system_failure_counter()
        );

        holder->ClearStopStatus(id);

        stopStatus = holder->GetObjectStatus(id).stop_status();
        UNIT_ASSERT(stopStatus.has_stop());
        UNIT_ASSERT_EQUAL_C(
            stopStatus.container_status().container_name()
            , "stop_container"
            , stopStatus.container_status().container_name()
        );
        UNIT_ASSERT_EQUAL_C(
            stopStatus.container_status().system_failure_counter()
            , 0
            , stopStatus.container_status().system_failure_counter()
        );
    }

    {
        // Test without stop
        TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
        TUpdateHolderPtr updateHolder = new TUpdateHolder();
        const TString id = "my_workload";

        holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

        API::TStopStatus stopStatus = holder->GetObjectStatus(id).stop_status();
        UNIT_ASSERT(!stopStatus.has_stop());

        holder->ClearStopStatus(id);

        stopStatus = holder->GetObjectStatus(id).stop_status();
        UNIT_ASSERT(!stopStatus.has_stop());
    }
}

Y_UNIT_TEST(TestWorkloadClearDestroyStatus) {
    {
        // Test with destroy
        TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
        TUpdateHolderPtr updateHolder = new TUpdateHolder();
        const TString id = "my_workload";

        holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id, 1, 0, 0, true));

        NStatusRepositoryTypes::TContainerDescription destroyContainer(
            id
            , NStatusRepositoryTypes::EObjectType::WORKLOAD
            , NStatusRepositoryTypes::TContainerDescription::EContainerType::DESTROY
        );
        holder->IncrementContainerSystemFailureCounter(destroyContainer);

        API::TDestroyStatus destroyStatus = holder->GetObjectStatus(id).destroy_status();
        UNIT_ASSERT(destroyStatus.has_destroy());
        UNIT_ASSERT_EQUAL_C(
            destroyStatus.container_status().container_name()
            , "destroy_container"
            , destroyStatus.container_status().container_name()
        );
        UNIT_ASSERT_EQUAL_C(
            destroyStatus.container_status().system_failure_counter()
            , 1
            , destroyStatus.container_status().system_failure_counter()
        );

        holder->ClearDestroyStatus(id);

        destroyStatus = holder->GetObjectStatus(id).destroy_status();
        UNIT_ASSERT(destroyStatus.has_destroy());
        UNIT_ASSERT_EQUAL_C(
            destroyStatus.container_status().container_name()
            , "destroy_container"
            , destroyStatus.container_status().container_name()
        );
        UNIT_ASSERT_EQUAL_C(
            destroyStatus.container_status().system_failure_counter()
            , 0
            , destroyStatus.container_status().system_failure_counter()
        );
    }

    {
        // Test without destroy
        TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
        TUpdateHolderPtr updateHolder = new TUpdateHolder();
        const TString id = "my_workload";

        holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

        API::TDestroyStatus destroyStatus = holder->GetObjectStatus(id).destroy_status();
        UNIT_ASSERT(!destroyStatus.has_destroy());

        holder->ClearDestroyStatus(id);

        destroyStatus = holder->GetObjectStatus(id).destroy_status();
        UNIT_ASSERT(!destroyStatus.has_destroy());
    }
}

}

} // namespace NInfra::NPodAgent::NWorkloadStatusRepositoryTest
