#include "container_status_repository_internal.h"

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NContainerStatusRepositoryInternalTest {

Y_UNIT_TEST_SUITE(ContainerStatusRepositoryInternalSuite) {

class TTestStatusRepositoryInternal: public IContainerStatusRepositoryInternal {
public:
    struct TWorkloadContainerCompare {
        bool operator()(const NStatusRepositoryTypes::TContainerDescription& a, const NStatusRepositoryTypes::TContainerDescription& b) const {
            return a.ObjectIdOrHash_ < b.ObjectIdOrHash_
                || (a.ObjectIdOrHash_ == b.ObjectIdOrHash_ && a.ObjectType_ < b.ObjectType_)
                || (a.ObjectIdOrHash_ == b.ObjectIdOrHash_ && a.ContainerType_ < b.ContainerType_)
                || (a.ObjectIdOrHash_ == b.ObjectIdOrHash_ && a.ContainerType_ == b.ContainerType_ && a.ContainerType_ == NStatusRepositoryTypes::TContainerDescription::EContainerType::INIT && a.InitNum_ < b.InitNum_);
        }
    };

public:
    TTestStatusRepositoryInternal() = default;

    void AddContainer(const NStatusRepositoryTypes::TContainerDescription& container) {
        TWriteGuardBase<TLightRWLock> guard(GlobalLock_);
        Containers_[container];
        ContainerLocks_[container];
    }

private:
    TContainerStatusInternal& GetContainerStatus(const NStatusRepositoryTypes::TContainerDescription& container) final {
        return Containers_.at(container);
    }

    const TContainerStatusInternal& GetContainerStatus(const NStatusRepositoryTypes::TContainerDescription& container) const final {
        return Containers_.at(container);
    }

    const TMutex& GetContainerMutex(const NStatusRepositoryTypes::TContainerDescription& container) const final {
        return ContainerLocks_.at(container);
    }

private:
    TMap<NStatusRepositoryTypes::TContainerDescription, TContainerStatusInternal, TWorkloadContainerCompare> Containers_;
    TMap<NStatusRepositoryTypes::TContainerDescription, TMutex, TWorkloadContainerCompare> ContainerLocks_;
};

using TTestStatusRepositoryInternalPtr = TIntrusivePtr<TTestStatusRepositoryInternal>;


Y_UNIT_TEST(TestStartTime) {
    TTestStatusRepositoryInternalPtr statusRepositoryInternal = new TTestStatusRepositoryInternal();
    TContainerStatusRepositoryInternalPtr containerStatusRepositoryInternal = statusRepositoryInternal;

    NStatusRepositoryTypes::TContainerDescription container1 = NStatusRepositoryTypes::TContainerDescription(
        "WorkloadId"
        , NStatusRepositoryTypes::EObjectType::WORKLOAD
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::START
    );
    NStatusRepositoryTypes::TContainerDescription container2 = NStatusRepositoryTypes::TContainerDescription(
        "WorkloadId"
        , NStatusRepositoryTypes::EObjectType::WORKLOAD
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::STOP
    );

    statusRepositoryInternal->AddContainer(container1);
    statusRepositoryInternal->AddContainer(container2);

    UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerStartTime(container1), TInstant::Zero(), containerStatusRepositoryInternal->GetContainerStartTime(container1));
    UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerStartTime(container2), TInstant::Zero(), containerStatusRepositoryInternal->GetContainerStartTime(container2));
    containerStatusRepositoryInternal->SetContainerStartTime(container1, TInstant::Max());
    UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerStartTime(container1), TInstant::Max(), containerStatusRepositoryInternal->GetContainerStartTime(container1));
    UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerStartTime(container2), TInstant::Zero(), containerStatusRepositoryInternal->GetContainerStartTime(container2));
}

Y_UNIT_TEST(TestCounters) {
    const size_t cntIter = 10;
    TTestStatusRepositoryInternalPtr statusRepositoryInternal = new TTestStatusRepositoryInternal();
    TContainerStatusRepositoryInternalPtr containerStatusRepositoryInternal = statusRepositoryInternal;

    NStatusRepositoryTypes::TContainerDescription container1 = NStatusRepositoryTypes::TContainerDescription(
        "WorkloadId"
        , NStatusRepositoryTypes::EObjectType::WORKLOAD
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::START
    );
    NStatusRepositoryTypes::TContainerDescription container2 = NStatusRepositoryTypes::TContainerDescription(
        "WorkloadId"
        , NStatusRepositoryTypes::EObjectType::WORKLOAD
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::STOP
    );

    statusRepositoryInternal->AddContainer(container1);
    statusRepositoryInternal->AddContainer(container2);

    UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerZeroReturnCodeCounter(container1), 0, containerStatusRepositoryInternal->GetContainerZeroReturnCodeCounter(container1));
    UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerPositiveReturnCodeCounter(container1), 0, containerStatusRepositoryInternal->GetContainerPositiveReturnCodeCounter(container1));
    UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerOomCounter(container1), 0, containerStatusRepositoryInternal->GetContainerOomCounter(container1));
    UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerKilledExternallyCounter(container1), 0, containerStatusRepositoryInternal->GetContainerKilledExternallyCounter(container1));
    UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerConsecutiveSuccessesCounter(container1), 0, containerStatusRepositoryInternal->GetContainerConsecutiveSuccessesCounter(container1));
    UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerConsecutiveFailuresCounter(container1), 0, containerStatusRepositoryInternal->GetContainerConsecutiveFailuresCounter(container1));

    for (size_t i = 0; i < cntIter; ++i) {
        containerStatusRepositoryInternal->IncrementContainerZeroReturnCodeCounter(container1);

        UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerZeroReturnCodeCounter(container1), i + 1, containerStatusRepositoryInternal->GetContainerZeroReturnCodeCounter(container1));
        UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerPositiveReturnCodeCounter(container1), 0, containerStatusRepositoryInternal->GetContainerPositiveReturnCodeCounter(container1));
        UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerOomCounter(container1), 0, containerStatusRepositoryInternal->GetContainerOomCounter(container1));
        UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerKilledExternallyCounter(container1), 0, containerStatusRepositoryInternal->GetContainerKilledExternallyCounter(container1));
        UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerConsecutiveSuccessesCounter(container1), i + 1, containerStatusRepositoryInternal->GetContainerConsecutiveSuccessesCounter(container1));
        UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerConsecutiveFailuresCounter(container1), 0, containerStatusRepositoryInternal->GetContainerConsecutiveFailuresCounter(container1));
    }

    for (size_t i = 0; i < cntIter; ++i) {
        containerStatusRepositoryInternal->IncrementContainerPositiveReturnCodeCounter(container1);

        UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerZeroReturnCodeCounter(container1), cntIter, containerStatusRepositoryInternal->GetContainerZeroReturnCodeCounter(container1));
        UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerPositiveReturnCodeCounter(container1), i + 1, containerStatusRepositoryInternal->GetContainerPositiveReturnCodeCounter(container1));
        UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerOomCounter(container1), 0, containerStatusRepositoryInternal->GetContainerOomCounter(container1));
        UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerKilledExternallyCounter(container1), 0, containerStatusRepositoryInternal->GetContainerKilledExternallyCounter(container1));
        UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerConsecutiveSuccessesCounter(container1), 0, containerStatusRepositoryInternal->GetContainerConsecutiveSuccessesCounter(container1));
        UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerConsecutiveFailuresCounter(container1), i + 1, containerStatusRepositoryInternal->GetContainerConsecutiveFailuresCounter(container1));
    }

    for (size_t i = 0; i < cntIter; ++i) {
        containerStatusRepositoryInternal->IncrementContainerOomCounter(container1);

        UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerZeroReturnCodeCounter(container1), cntIter, containerStatusRepositoryInternal->GetContainerZeroReturnCodeCounter(container1));
        UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerPositiveReturnCodeCounter(container1), cntIter, containerStatusRepositoryInternal->GetContainerPositiveReturnCodeCounter(container1));
        UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerOomCounter(container1), i + 1, containerStatusRepositoryInternal->GetContainerOomCounter(container1));
        UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerKilledExternallyCounter(container1), 0, containerStatusRepositoryInternal->GetContainerKilledExternallyCounter(container1));
        UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerConsecutiveSuccessesCounter(container1), 0, containerStatusRepositoryInternal->GetContainerConsecutiveSuccessesCounter(container1));
        UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerConsecutiveFailuresCounter(container1), cntIter + i + 1, containerStatusRepositoryInternal->GetContainerConsecutiveFailuresCounter(container1));
    }

    for (size_t i = 0; i < cntIter; ++i) {
        containerStatusRepositoryInternal->IncrementContainerKilledExternallyCounter(container1);

        UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerZeroReturnCodeCounter(container1), cntIter, containerStatusRepositoryInternal->GetContainerZeroReturnCodeCounter(container1));
        UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerPositiveReturnCodeCounter(container1), cntIter, containerStatusRepositoryInternal->GetContainerPositiveReturnCodeCounter(container1));
        UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerOomCounter(container1), cntIter, containerStatusRepositoryInternal->GetContainerOomCounter(container1));
        UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerKilledExternallyCounter(container1), i + 1, containerStatusRepositoryInternal->GetContainerKilledExternallyCounter(container1));
        UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerConsecutiveSuccessesCounter(container1), 0, containerStatusRepositoryInternal->GetContainerConsecutiveSuccessesCounter(container1));
        UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerConsecutiveFailuresCounter(container1), 2 * cntIter + i + 1, containerStatusRepositoryInternal->GetContainerConsecutiveFailuresCounter(container1));
    }

    // make sure the container2 is not affected
    UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerZeroReturnCodeCounter(container2), 0, containerStatusRepositoryInternal->GetContainerZeroReturnCodeCounter(container2));
    UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerPositiveReturnCodeCounter(container2), 0, containerStatusRepositoryInternal->GetContainerPositiveReturnCodeCounter(container2));
    UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerOomCounter(container2), 0, containerStatusRepositoryInternal->GetContainerOomCounter(container2));
    UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerKilledExternallyCounter(container2), 0, containerStatusRepositoryInternal->GetContainerKilledExternallyCounter(container2));
    UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerConsecutiveSuccessesCounter(container2), 0, containerStatusRepositoryInternal->GetContainerConsecutiveSuccessesCounter(container2));
    UNIT_ASSERT_EQUAL_C(containerStatusRepositoryInternal->GetContainerConsecutiveFailuresCounter(container2), 0, containerStatusRepositoryInternal->GetContainerConsecutiveFailuresCounter(container2));
}

}

}; // namespace NInfra::NPodAgent::NContainerStatusRepositoryInternalTest
