#pragma once
#include "container_status_repository_internal.h"
#include "container_status_holder_internal.h"

#include <util/generic/map.h>
#include <util/system/mutex.h>

namespace NInfra::NPodAgent {

enum EContainerStatusRepositoryType {
    LAYER_CONTAINER_STATUS_REPOSITORY               /* "TLayerContainerStatusRepositoryInternal"          */,
    STATIC_RESOURCE_CONTAINER_STATUS_REPOSITORY     /* "TStaticContainerResourceStatusRepositoryInternal" */,
    BOX_CONTAINER_STATUS_REPOSITORY                 /* "TBoxContainerStatusRepositoryInternal"            */,
    WORKLOAD_CONTAINER_STATUS_REPOSITORY            /* "TWorkloadContainerStatusRepositoryInternal"       */,
};

template<EContainerStatusRepositoryType ContainerStatusRepositoryType>
class TObjectContainerStatusRepositoryInternal: public IContainerStatusRepositoryInternal {
public:
    TObjectContainerStatusRepositoryInternal() = default;

    void AddObject(const TString& objectId) {
        TWriteGuardBase<TLightRWLock> guard(GlobalLock_);
        Objects_[objectId];
        ObjectLocks_[objectId];
    }

    void RemoveObject(const TString& objectId) {
        TWriteGuardBase<TLightRWLock> guard(GlobalLock_);
        ObjectLocks_.erase(objectId);
        Objects_.erase(objectId);
    }

    bool HasObject(const TString& objectId) const {
        TReadGuardBase<TLightRWLock> guard(GlobalLock_);
        return ObjectLocks_.FindPtr(objectId);
    }

private:
    const TMutex& GetObjectMutex(const TString& objectId) const {
        TReadGuardBase<TLightRWLock> guard(GlobalLock_);
        const TMutex* ptr = ObjectLocks_.FindPtr(objectId);
        Y_ENSURE(ptr, "Object '" << objectId << "' not found at " << ToString(ContainerStatusRepositoryType));
        return *ptr;
    }

    virtual TContainerStatusInternal& GetContainerStatus(const NStatusRepositoryTypes::TContainerDescription& container) override final {
        TReadGuardBase<TLightRWLock> guard(GlobalLock_);
        return Objects_.at(container.ObjectIdOrHash_).GetObjectContainerStatus(container);
    }

    virtual const TContainerStatusInternal& GetContainerStatus(const NStatusRepositoryTypes::TContainerDescription& container) const override final {
        TReadGuardBase<TLightRWLock> guard(GlobalLock_);
        return Objects_.at(container.ObjectIdOrHash_).GetObjectContainerStatus(container);
    }

    virtual const TMutex& GetContainerMutex(const NStatusRepositoryTypes::TContainerDescription& container) const override final {
        TReadGuardBase<TLightRWLock> guard(GlobalLock_);
        return GetObjectMutex(container.ObjectIdOrHash_);
    }

private:
    TMap<TString, TContainerStatusHolderInternal> Objects_; // ObjectId -> Status
    TMap<TString, TMutex> ObjectLocks_;
};

using TLayerContainerStatusRepositoryInternal = TObjectContainerStatusRepositoryInternal<EContainerStatusRepositoryType::LAYER_CONTAINER_STATUS_REPOSITORY>;
using TStaticResourceContainerStatusRepositoryInternal = TObjectContainerStatusRepositoryInternal<EContainerStatusRepositoryType::STATIC_RESOURCE_CONTAINER_STATUS_REPOSITORY>;
using TBoxContainerStatusRepositoryInternal = TObjectContainerStatusRepositoryInternal<EContainerStatusRepositoryType::BOX_CONTAINER_STATUS_REPOSITORY>;
using TWorkloadContainerStatusRepositoryInternal = TObjectContainerStatusRepositoryInternal<EContainerStatusRepositoryType::WORKLOAD_CONTAINER_STATUS_REPOSITORY>;

using TLayerContainerStatusRepositoryInternalPtr = TIntrusivePtr<TLayerContainerStatusRepositoryInternal>;
using TStaticResourceContainerStatusRepositoryInternalPtr = TIntrusivePtr<TStaticResourceContainerStatusRepositoryInternal>;
using TBoxContainerStatusRepositoryInternalPtr = TIntrusivePtr<TBoxContainerStatusRepositoryInternal>;
using TWorkloadContainerStatusRepositoryInternalPtr = TIntrusivePtr<TWorkloadContainerStatusRepositoryInternal>;

} // namespace NInfra::NPodAgent
