#pragma once

#include <library/cpp/threading/light_rw_lock/lightrwlock.h>

#include <util/generic/map.h>
#include <util/generic/set.h>
#include <util/generic/vector.h>
#include <util/string/cast.h>
#include <util/system/mutex.h>

namespace NInfra::NPodAgent {

enum EStatusRepositoryType {
    LAYER_STATUS_REPOSITORY               /* "TLayerStatusRepositoryInternal"          */,
    STATIC_RESOURCE_STATUS_REPOSITORY     /* "TStaticResourceStatusRepositoryInternal" */,
    VOLUME_STATUS_REPOSITORY              /* "TVolumeStatusRepositoryInternal"         */,
    BOX_STATUS_REPOSITORY                 /* "TBoxStatusRepositoryInternal"            */,
    WORKLOAD_STATUS_REPOSITORY            /* "TWorkloadStatusRepositoryInternal"       */,
};

template<class StatusType, EStatusRepositoryType StatusRepositoryType>
class TObjectStatusRepositoryInternal: public TAtomicRefCount<TObjectStatusRepositoryInternal<StatusType, StatusRepositoryType>> {
public:
    TObjectStatusRepositoryInternal() = default;
    virtual ~TObjectStatusRepositoryInternal() = default;

    void AddObject(const TString& objectId) {
        TWriteGuardBase<TLightRWLock> guard(GlobalLock_);
        Objects_[objectId];
        ObjectLocks_[objectId];
    }

    void RemoveObject(const TString& objectId) {
        TWriteGuardBase<TLightRWLock> guard(GlobalLock_);
        ObjectLocks_.erase(objectId);
        Objects_.erase(objectId);
    }

    bool HasObject(const TString& objectId) const {
        TReadGuardBase<TLightRWLock> guard(GlobalLock_);
        return ObjectLocks_.FindPtr(objectId);
    }

protected:
    const TMutex& GetObjectMutex(const TString& objectId) const {
        TReadGuardBase<TLightRWLock> guard(GlobalLock_);
        const TMutex* ptr = ObjectLocks_.FindPtr(objectId);
        Y_ENSURE(ptr, "Object '" << objectId << "' not found at " << ToString(StatusRepositoryType));
        return *ptr;
    }

protected:
    TLightRWLock GlobalLock_;

    TMap<TString, StatusType> Objects_; // ObjectId -> Status
    TMap<TString, TMutex> ObjectLocks_;
};

template<class StatusType, EStatusRepositoryType StatusRepositoryType>
class TObjectStatusRepositoryInternalWithHashes: public TAtomicRefCount<TObjectStatusRepositoryInternalWithHashes<StatusType, StatusRepositoryType>> {
public:
    TObjectStatusRepositoryInternalWithHashes() = default;
    virtual ~TObjectStatusRepositoryInternalWithHashes() = default;

    void AddObjectWithHash(const TString& objectId, const TString& hash) {
        TWriteGuardBase<TLightRWLock> guard(GlobalLock_);
        Objects_[objectId].SetHash(hash);
        ObjectLocks_[objectId];
        Hashes_[hash].insert(objectId);
    }

    void RemoveObject(const TString& objectId) {
        TWriteGuardBase<TLightRWLock> guard(GlobalLock_);
        TSet<TString>& ids = Hashes_[Objects_.at(objectId).GetHash()];
        ids.erase(objectId);
        if (ids.empty()) {
            Hashes_.erase(Objects_.at(objectId).GetHash());
        }
        ObjectLocks_.erase(objectId);
        Objects_.erase(objectId);
    }

    bool HasObject(const TString& objectId) const {
        TReadGuardBase<TLightRWLock> guard(GlobalLock_);
        return ObjectLocks_.FindPtr(objectId);
    }

    bool HasHash(const TString& hash) const {
        TReadGuardBase<TLightRWLock> guard(GlobalLock_);
        return Hashes_.FindPtr(hash);
    }

    TVector<TString> GetObjectIdsByHash(const TString& hash) const {
        TReadGuardBase<TLightRWLock> guard(GlobalLock_);
        auto ptr = Hashes_.FindPtr(hash);
        if (ptr) {
            return TVector<TString>(ptr->begin(), ptr->end());
        } else {
            return TVector<TString>();
        }
    }

    TString GetObjectHash(const TString& objectId) const {
        TReadGuardBase<TLightRWLock> guard(GlobalLock_);
        TGuard<TMutex> g(GetObjectMutex(objectId));
        return Objects_.at(objectId).GetHash();
    }

protected:
    const TMutex& GetObjectMutex(const TString& objectId) const {
        TReadGuardBase<TLightRWLock> guard(GlobalLock_);
        const TMutex* ptr = ObjectLocks_.FindPtr(objectId);
        Y_ENSURE(ptr, "Object '" << objectId << "' not found at " << ToString(StatusRepositoryType));
        return *ptr;
    }

protected:
    TLightRWLock GlobalLock_;

    TMap<TString, StatusType> Objects_; // ObjectId -> Status
    TMap<TString, TMutex> ObjectLocks_;
    TMap<TString, TSet<TString>> Hashes_; // Hash -> ObjectIds
};

} // namespace NInfra::NPodAgent
