#include "box_tree_generator.h"

#include "object_tree.h"
#include "support_functions.h"

#include <infra/pod_agent/libs/behaviour/loaders/behavior3_editor_json_reader.h>
#include <infra/pod_agent/libs/behaviour/loaders/behavior3_template_resolver.h>
#include <infra/pod_agent/libs/pod_agent/cache_file/cache_file.h>
#include <infra/pod_agent/libs/pod_agent/trees_generators/proto_hasher.h>
#include <infra/pod_agent/libs/util/string_utils.h>

#include <library/cpp/digest/md5/md5.h>
#include <library/cpp/yson/node/node_io.h>

#include <util/string/join.h>

namespace NInfra::NPodAgent {

namespace {

static const TString SKYNET_ACTIVE_PATH = "/skynet";
static const TString SKYNET_BASE_SUFFIX = "/base/";
static size_t SKYNET_HASH_LEN = 40;

static const TString YT_BIND_ANNOTATIONS_KEY = "pod.rbind.yt";
static const TString BASESEARCH_BIND_ANNOTATIONS_KEY = "pod.rbind.basesearch";

static const TString NAT64_RESOLV_CONF = TStringBuilder()
    << "search search.yandex.net."
    << ";nameserver 2a02:6b8:0:3400::5005"
    << ";options timeout:1 attempts:1"
;

static const TString NAT64_LOCAL_RESOLV_CONF = TStringBuilder()
    << "search search.yandex.net."
    << ";nameserver fd64::1"
    << ";nameserver 2a02:6b8:0:3400::5005"
    << ";options timeout:1 attempts:1"
;

TString GenerateTVMToolLocalToken(TLogFramePtr logFrame) {
    TVector<TString> seeds = {"/proc/sys/kernel/random/boot_id", "/sys/class/net/veth/address"};
    TStringBuilder sum;
    for (auto& it : seeds) {
        try {
            sum << TUnbufferedFileInput(it).ReadAll();
        } catch (const yexception& e) {
            logFrame->LogEvent(ELogPriority::TLOG_WARNING, NLogEvent::TTVMTokenError(e.what()));
        }
    }
    return MD5::Calc(sum);
}

TIp6 GenerateIp6From112Subnet(const TIp6& subnetStart, ui16 offset) {
    TIp6 result = subnetStart;
    result.Data[14] = offset >> 8;
    result.Data[15] = offset;
    return result;
}

ui16 GetOffsetFromCache(::google::protobuf::Map<TString, ui32>& cache, const TString& id) {
    auto ptr = cache.find(id);
    if (ptr != cache.end()) {
        return ptr->second;
    }
    // find unused offset for id
    THashSet<ui16> used;
    for (auto& it : cache) {
        used.insert(it.second);
    }

    // start from 1, 0 is special for all 112 subnet
    for (ui32 cur = 1; cur < (1 << 16); ++cur) {
        if (!used.contains(cur)) {
            return cache[id] = cur;
        }
    }
    ythrow yexception() << "couldn't find free IP6 for box '" << id << "', subnet has no free IP6";
}

TString ResolveSkynetBindPath() {
    // resolve /skynet link to <skynet_path>/base/<skynet_version_hash>[.<something>]
    const TString activeSkynetPath = RealPath(SKYNET_ACTIVE_PATH);

    // cut <skynet_version_hash>[.<something>] suffix
    const size_t lastSlash = activeSkynetPath.find_last_of('/');
    const TStringBuf suffix = TStringBuf(activeSkynetPath).substr(lastSlash + 1, activeSkynetPath.length() - lastSlash - 1);
    TStringBuf tmp = TStringBuf(activeSkynetPath).substr(0, activeSkynetPath.length() - suffix.length());
    Y_ENSURE(
        tmp.EndsWith(SKYNET_BASE_SUFFIX) && (suffix.length() == SKYNET_HASH_LEN || (suffix.length() > SKYNET_HASH_LEN && suffix[SKYNET_HASH_LEN] == '.'))
        , "/skynet link resolved to '" << activeSkynetPath << "'"
        << " doesn't ends with '" << SKYNET_BASE_SUFFIX << "<" << SKYNET_HASH_LEN << "_char_hash>[.something]'"
    );

    // cut /base/ suffix
    return TString(tmp.substr(0, tmp.length() - SKYNET_BASE_SUFFIX.length()));
}

bool CheckBindAnnotation(
    const NYT::NYTree::NProto::TAttributeDictionary& annotations
    , const TString& annotationKey
) {
    bool haveBind = false;
    bool haveBindAnnotation = false;
    for (auto& it : annotations.attributes()) {
        if (it.key() == annotationKey) {
            Y_ENSURE(!haveBindAnnotation, "Two or more annotations with key: '" << annotationKey << "'");
            haveBindAnnotation = true;
            haveBind = NYT::NodeFromYsonString(it.value()).ConvertTo<bool>();
        }
    }

    return haveBind;
}

TMaybe<TString> ResolveBindPath(const TString& path) {
    if (NFs::Exists(path)) {
        return RealPath(path);
    } else {
        return Nothing();
    }
}

}

TBoxTreeGenerator::TBoxTreeGenerator(
    TLogger& logger
    , TPathHolderPtr pathHolder
    , const TBehavior3& boxTreeTemplate
    , TAsyncIpClientPtr ipClient
    , TAsyncPortoClientPtr porto
    , TPosixWorkerPtr posixWorker
    , TStatusNTickerHolderPtr statusNTickerHolder
    , TTemplateBTStoragePtr templateBTStorage
    , const TString& cacheFileName
    , const TString& hostname
    , const TString& ytPath
    , const TString& baseSearchPath
    , const TString& publicVolumePath
    , const TString& publicVolumeMountPath
    , const TString& podAgentBinaryFilePathInBox
    , const TString& newtworkDevice
)
    : Logger_(logger)
    , PathHolder_(pathHolder)
    , BoxTreeTemplate_(boxTreeTemplate)
    , CacheFileName_(cacheFileName)
    , SkynetBindPath_(ResolveSkynetBindPath())
    , YtPath_(ytPath)
    , BaseSearchPath_(baseSearchPath)
    , PublicVolumePath_(RealPath(publicVolumePath)) // Must always exist
    , PublicVolumeMountPath_(publicVolumeMountPath)
    , PodAgentBinaryFilePathInBox_(podAgentBinaryFilePathInBox)
    , IpClient_(ipClient)
    , Porto_(porto)
    , PosixWorker_(posixWorker)
    , StatusNTickerHolder_(statusNTickerHolder)
    , TemplateBTStorage_(templateBTStorage)
    , LocalHostName_(hostname)
    , NetworkDevice_(newtworkDevice)
    , TVMToolLocalToken_(GenerateTVMToolLocalToken(logger.SpawnFrame()))
{}

TBoxTreeGenerator::TBoxesToAdd TBoxTreeGenerator::UpdateSpec(
    const google::protobuf::RepeatedPtrField<API::TBox>& boxes
    , API::EPodAgentTargetState podAgentTargetState
    , const TLayerTreeGenerator::TLayersToAdd& layers
    , const TStaticResourceTreeGenerator::TStaticResourcesToAdd& staticResources
    , const TVolumeTreeGenerator::TVolumesToAdd& volumes
    , const TString& podId
    , const API::TNodeMeta& nodeMeta
    , const API::TGpuManagerMeta& gpuManagerMeta
    , const NYP::NClient::NApi::NProto::TPodStatus::TDns& dns
    , const google::protobuf::RepeatedPtrField<NYP::NClient::NApi::NProto::TPodStatus::TIP6SubnetAllocation>& subnets
    , const google::protobuf::RepeatedPtrField<NYP::NClient::NApi::NProto::TPodStatus::TIP6AddressAllocation>& ip6AddressAllocations
    , const NSecret::TSecretMap& secretMap
    , const NYT::NYTree::NProto::TAttributeDictionary& annotations
    , const double cpuToVcpuFactor
    , ui64 specTimestamp
    , ui32 revision
    , TLogFramePtr logFrame
    , bool useEnvSecret
    , bool autoDecodeBase64Secrets
) {
    TBoxTreeGenerator::TBoxesToAdd boxesToAdd;

    NTreesGenerators::TBoxesCache boxCache = LoadBoxCache(logFrame);

    // set ipv6 address for each box from given boxes_subnet (if given)
    TBoxIpInfo boxIpInfo = GetBoxIpInfo(boxes, subnets, boxCache);

    // Add boxes to internal state only in ACTIVE or SUSPENDED pod_agent target states
    // WARNING: We do not validate box specs in other cases (for example in REMOVED target state)
    if (
        podAgentTargetState == API::EPodAgentTargetState_ACTIVE
        || podAgentTargetState == API::EPodAgentTargetState_SUSPENDED
    ) {
        // check annotations for pod.rbind.yt
        TMaybe<TString> ytBindPath = Nothing();
        bool haveYtBind = CheckBindAnnotation(annotations, YT_BIND_ANNOTATIONS_KEY);
        if (haveYtBind) {
            ytBindPath = ResolveBindPath(YtPath_);
            Y_ENSURE(ytBindPath.Defined(), "Yt bind annotation provided, but yt path doesn't exist");
        }

        // check annotations for pod.rbind.basesearch
        TMaybe<TString> baseSearchBindPath = Nothing();
        bool haveBaseSearchBind = CheckBindAnnotation(annotations, BASESEARCH_BIND_ANNOTATIONS_KEY);
        if (haveBaseSearchBind) {
            baseSearchBindPath = ResolveBindPath(BaseSearchPath_);
            Y_ENSURE(baseSearchBindPath.Defined(), "Basesearch bind annotation provided, but basesearch path doesn't exist");
        }

        const TMap<TString, TString> systemEnv = NSupport::GetCommonSystemEnvironment(
            TVMToolLocalToken_
            , podId
            , nodeMeta
            , gpuManagerMeta
            , dns
            , ip6AddressAllocations
        );

        TMap<TString, TVector<TString>> staticResourceHashToIdsMountedToReadOnlyVolumes;
        TMap<TString, TVector<TString>> staticResourceHashToIdsMountedToWritableVolumes;

        for (const auto& box : boxes) {
            const TString& boxId = box.id();

            Y_ENSURE(!boxId.empty(), "One of boxes has empty id");
            Y_ENSURE(!boxesToAdd.Boxes_.contains(boxId)
                , "Pod agent spec boxes' ids are not unique: "
                << "box " << Quote(boxId) << " occurs twice"
            );

            try {
                boxesToAdd.Boxes_.insert(
                    {
                        boxId
                        , GetBoxToAdd(
                            box
                            , layers
                            , staticResources
                            , volumes
                            , systemEnv
                            , secretMap
                            , boxCache
                            , boxIpInfo
                            , ytBindPath
                            , baseSearchBindPath
                            , cpuToVcpuFactor
                            , specTimestamp
                            , revision
                            , useEnvSecret
                            , staticResourceHashToIdsMountedToReadOnlyVolumes
                            , staticResourceHashToIdsMountedToWritableVolumes
                            , autoDecodeBase64Secrets
                        )
                    }
                );
            } catch (yexception& e) {
                throw e << " at box " << Quote(boxId);
            }
        }

        ValidateStaticResourcesMountedToReadOnlyAndWritableVolumeSimultaneously(
            staticResourceHashToIdsMountedToReadOnlyVolumes
            , staticResourceHashToIdsMountedToWritableVolumes
        );
    }

    if (boxIpInfo.HaveIp_) {
        boxesToAdd.Ip6Subnet112Base_ = Ip6ToString(GenerateIp6From112Subnet(boxIpInfo.Ip6Subnet_, 0));
        boxesToAdd.Ip6Subnet112Base_.pop_back(); // remove last "0"
    } else {
        boxesToAdd.Ip6Subnet112Base_ = "";
    }

    return boxesToAdd;
}

TBoxTreeGenerator::TBoxToAdd TBoxTreeGenerator::GetBoxToAdd(
    const API::TBox& box
    , const TLayerTreeGenerator::TLayersToAdd& layers
    , const TStaticResourceTreeGenerator::TStaticResourcesToAdd& staticResources
    , const TVolumeTreeGenerator::TVolumesToAdd& volumes
    , const TMap<TString, TString>& systemEnv
    , const NSecret::TSecretMap& secretMap
    , NTreesGenerators::TBoxesCache& boxCache
    , const TBoxIpInfo& boxIpInfo
    , const TMaybe<TString>& ytBindPath
    , const TMaybe<TString>& baseSearchBindPath
    , const double cpuToVcpuFactor
    , ui64 specTimestamp
    , ui32 revision
    , bool useEnvSecret
    , TMap<TString, TVector<TString>>& staticResourceHashToIdsMountedToReadOnlyVolumes
    , TMap<TString, TVector<TString>>& staticResourceHashToIdsMountedToWritableVolumes
    , bool autoDecodeBase64Secrets
) const {
    const TString& boxId = box.id();

    const ui16 boxIp6SubnetOffset = boxIpInfo.HaveIp_ ? GetOffsetFromCache(*boxCache.MutableIP6SubnetOffset(), boxId) : (ui16)0;
    const TString boxIp6Address = boxIpInfo.HaveIp_ ? Ip6ToString(GenerateIp6From112Subnet(boxIpInfo.Ip6Subnet_, boxIp6SubnetOffset)) : "";

    const TString treeHash = GetBoxHash(
        box
        , layers
        , staticResources
        , volumes
        , systemEnv
        , secretMap
        , ytBindPath
        , baseSearchBindPath
        , boxIp6Address
        , useEnvSecret
    );

    TVector<TString> layerRefs;
    TVector<TString> staticResourceRefs;
    TVector<TString> volumeRefs;
    TVector<TString> initContainers;

    TBehavior3 resolvedTree = BoxTreeTemplate_;
    TMap<TString, TString> cachedReplace;
    {
        TMap<TString, TString> replace;
        replace["BOX_ID"] = boxId;

        replace["YT_VOLUME_PATH"] = PathHolder_->GetBoxRootfsPath(boxId) + "/yt";
        replace["YT_VOLUME_STORAGE"] = ytBindPath.GetOrElse("");
        replace["YT_ENABLED"] = (ytBindPath.Defined() ? "true" : "");

        replace["BASESEARCH_VOLUME_PATH"] = PathHolder_->GetBoxRootfsPath(boxId) + "/basesearch";
        replace["BASESEARCH_VOLUME_STORAGE"] = baseSearchBindPath.GetOrElse("");
        replace["BASESEARCH_ENABLED"] = (baseSearchBindPath.Defined() ? "true" : "");

        replace["SKYNET_VOLUME_PATH"] = PathHolder_->GetBoxRootfsPath(boxId) + "/place/berkanavt/supervisor";
        replace["SKYNET_VOLUME_STORAGE"] = SkynetBindPath_;
        replace["SKYNET_ENABLED"] = (box.bind_skynet() ? "true" : "");

        replace["BOX_CONTAINER"] = PathHolder_->GetBoxContainer(boxId);
        replace.merge(
            NSupport::FillComputeResourcesReplaceMap(
                box.compute_resources()
                , "BOX_"
                , PathHolder_
                , cpuToVcpuFactor
            )
        );

        Y_ENSURE(PathHolder_->HasVirtualDisk(box.virtual_disk_id_ref()), "Unknown virtual disk ref: '" << box.virtual_disk_id_ref() << "'");
        const TString place = PathHolder_->GetPlaceFromVirtualDisk(box.virtual_disk_id_ref());
        replace["ROOTFS_VOLUME_PLACE"] = place;
        // Because we have isolated box inside isolated pod we can't simply use some porto magic like "///",
        // we need to specify box meta container place with dom0 prefix
        replace["BOX_CONTAINER_PLACE"] = PathHolder_->GetDom0PlaceFromVirtualDisk(box.virtual_disk_id_ref());

        switch (box.resolv_conf()) {
            case API::EResolvConf_DEFAULT:
                replace["BOX_RESOLV_CONF"] = "default";
                break;
            case API::EResolvConf_KEEP:
                replace["BOX_RESOLV_CONF"] = "keep";
                break;
            case API::EResolvConf_NAT64:
                replace["BOX_RESOLV_CONF"] = NAT64_RESOLV_CONF;
                break;
            case API::EResolvConf_NAT64_LOCAL:
                replace["BOX_RESOLV_CONF"] = NAT64_LOCAL_RESOLV_CONF;
                break;
            // No default to get compilation error when some case missed
            case API::EResolvConf_INT_MIN_SENTINEL_DO_NOT_USE_:
            case API::EResolvConf_INT_MAX_SENTINEL_DO_NOT_USE_:
                ythrow yexception() << "Invalid resolv conf " << API::EResolvConf_Name(box.resolv_conf());
        }

        NSupport::TEnvironmentList envList = NSupport::GetEnvironmentList(
            box.env()
            , secretMap
            , systemEnv
            , useEnvSecret
            , autoDecodeBase64Secrets
        );

        replace["BOX_HOSTNAME"] = LocalHostName_;
        replace["BOX_CAPABILITIES_AMBIENT"] = NSupport::CONTAINER_CAPABILITIES_AMBIENT;
        replace["BOX_ENVIRONMENT"] = envList.PublicEnv_;
        replace["BOX_SECRET_ENVIRONMENT"] = envList.SecretEnv_;

        replace["NETWORK_DEVICE"] = NetworkDevice_;
        replace["BOX_IP6_ADDRESS"] = boxIp6Address;

        replace["ROOTFS_VOLUME_PATH"] = PathHolder_->GetBoxRootfsPath(boxId);
        replace["ROOTFS_VOLUME_STORAGE"] = PathHolder_->GetBoxRootfsPersistentStorage(boxId);
        replace["ROOTFS_QUOTA_BYTES"] = ToString(box.rootfs().quota_bytes());

        auto rootfsVolumeCreationMode = box.rootfs().create_mode();

        Y_ENSURE(
            rootfsVolumeCreationMode == API::EVolumeCreateMode_READ_ONLY || rootfsVolumeCreationMode == API::EVolumeCreateMode_READ_WRITE
            , "Incorrect creation mode for rootfs volume " << Quote(API::EVolumeCreateMode_Name(rootfsVolumeCreationMode))
        );

        replace["ROOTFS_READ_ONLY"] = (rootfsVolumeCreationMode == API::EVolumeCreateMode_READ_ONLY ? "true" : "");

        auto boxContainerIsolationMode = box.isolation_mode();

        Y_ENSURE(
            boxContainerIsolationMode == API::EContainerIsolationMode_NO_ISOLATION || boxContainerIsolationMode == API::EContainerIsolationMode_CHILD_ONLY
            , "Incorrect box container isolation mode " << Quote(API::EContainerIsolationMode_Name(boxContainerIsolationMode))
        );

        replace["BOX_ENABLE_PORTO"] = (boxContainerIsolationMode == API::EContainerIsolationMode_CHILD_ONLY ? "child-only" : "");

        replace["BOX_CGROUP_FS_MOUNT_TYPE"] = NSupport::CgroupFsMountModeToString(box.cgroup_fs_mount_mode());

        TVector<TString> layerNames;
        TVector<TString> layerDownloadHashes;
        Y_ENSURE(box.rootfs().layer_refs().size(), "layers ids not provided for rootfs");
        for (const auto& layerId: box.rootfs().layer_refs()) {
            Y_ENSURE(layers.Layers_.contains(layerId), "Unknown layer: " << Quote(layerId));
            Y_ENSURE(
                Find(layerRefs, layerId) == layerRefs.end()
                , "Two equal layers: " << Quote(layerId)
            );
            Y_ENSURE(layers.Layers_.at(layerId).VirtualDiskIdRef_ == box.virtual_disk_id_ref()
                , "Box " << Quote(boxId)
                << " with virtual disk ref " << Quote(box.virtual_disk_id_ref())
                << " refers to layer " << Quote(layerId)
                << " with another virtual disk ref " << Quote(layers.Layers_.at(layerId).VirtualDiskIdRef_)
            );

            TString layerName = PathHolder_->GetLayerNameFromHash(layers.Layers_.at(layerId).Target_.Meta_.DownloadHash_);
            layerRefs.push_back(layerId);
            layerNames.push_back(layerName);
            layerDownloadHashes.push_back(layers.Layers_.at(layerId).Target_.Meta_.DownloadHash_);
        }

        TVector<TString> allMountPoints;
        TVector<TString> staticResourceOriginalPaths;
        TVector<TString> staticResourceMountPoints;
        for(const auto& staticResource: box.static_resources()) {
            const TString& staticResourceId = staticResource.resource_ref();
            Y_ENSURE(staticResources.StaticResources_.contains(staticResourceId), "Unknown static resource: " << Quote(staticResourceId));
            Y_ENSURE(staticResources.StaticResources_.at(staticResourceId).VirtualDiskIdRef_ == box.virtual_disk_id_ref()
                , "Box " << Quote(boxId)
                << " with virtual disk ref " << Quote(box.virtual_disk_id_ref())
                << " refers to static resource " << Quote(staticResourceId)
                << " with another virtual disk ref " << Quote(staticResources.StaticResources_.at(staticResourceId).VirtualDiskIdRef_)
            );

            staticResourceRefs.push_back(staticResourceId);

            TString staticResourceDownloadHash = staticResources.StaticResources_.at(staticResourceId).Target_.Meta_.DownloadHash_;
            staticResourceOriginalPaths.push_back(PathHolder_->GetFinalStaticResourcePathFromHash(staticResourceDownloadHash, place));

            TString staticResourceMountPoint = NSupport::NormalizeMountPoint(staticResource.mount_point());
            allMountPoints.push_back(PathHolder_->GetBoxRootfsPath(boxId) + "/" + staticResourceMountPoint);
            staticResourceMountPoints.push_back(staticResourceMountPoint);

            if (rootfsVolumeCreationMode == API::EVolumeCreateMode_READ_ONLY) {
                AddStaticResourceHashAndIdToMap(staticResourceHashToIdsMountedToReadOnlyVolumes, staticResourceDownloadHash, staticResourceId);
            } else if (rootfsVolumeCreationMode == API::EVolumeCreateMode_READ_WRITE) {
                AddStaticResourceHashAndIdToMap(staticResourceHashToIdsMountedToWritableVolumes, staticResourceDownloadHash, staticResourceId);
            }
        }

        replace["STATIC_RESOURCE_PATHS"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator(staticResourceMountPoints);
        replace["STATIC_RESOURCE_ORIGINAL_PATHS"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator(staticResourceOriginalPaths);
        replace["LAYER_NAME_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator(layerNames);
        replace["LAYER_DOWNLOAD_HASH_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator(layerDownloadHashes);

        TVector<TString> mountVolumePathList;
        TVector<TString> mountVolumeLinkPathList;
        TVector<TString> mountVolumeReadOnlyModeList;
        TVector<TString> mountVolumeTreeHashList;
        TVector<TString> rbindVolumePathList;
        TVector<TString> rbindVolumeStorageList;
        TVector<TString> rbindVolumeReadOnlyModeList;
        TVector<TString> nonPersistentVolumeIdList;
        TVector<TString> nonPersistentVolumePathList;
        for (const auto& mountedVolume : box.volumes()) {
            Y_ENSURE(
                mountedVolume.volume_type_case() != API::TMountedVolume::VolumeTypeCase::VOLUME_TYPE_NOT_SET
                , "Volume type not set for mounted volume"
            );

            if (mountedVolume.volume_type_case() == API::TMountedVolume::VolumeTypeCase::kVolumeRef) {
                const TString& volumeId = mountedVolume.volume_ref();
                Y_ENSURE(
                    mountedVolume.mode() == API::EVolumeMountMode_READ_ONLY || mountedVolume.mode() == API::EVolumeMountMode_READ_WRITE
                    , "Incorrect mount mode for volume " << Quote(volumeId)
                );

                Y_ENSURE(volumes.Volumes_.contains(volumeId), "Unknown volume " << Quote(volumeId));

                volumeRefs.push_back(volumeId);
                mountVolumePathList.push_back(PathHolder_->GetVolumePath(volumeId));
                mountVolumeTreeHashList.push_back(volumes.Volumes_.at(volumeId).Target_.Hash_);
                mountVolumeReadOnlyModeList.push_back(mountedVolume.mode() == API::EVolumeMountMode_READ_ONLY ? "true" : "false");

                const TString volumeLinkPath = PathHolder_->GetBoxRootfsPath(boxId) + "/" + NSupport::NormalizeMountPoint(mountedVolume.mount_point());
                allMountPoints.push_back(volumeLinkPath);
                mountVolumeLinkPathList.push_back(volumeLinkPath);

                if (!volumes.Volumes_.at(volumeId).Target_.IsPersistent_) {
                    nonPersistentVolumeIdList.push_back(volumeId);
                    nonPersistentVolumePathList.push_back(PathHolder_->GetVolumePath(volumeId));
                }

                if (mountedVolume.mode() == API::EVolumeMountMode_READ_ONLY) {
                    AddStaticResourceHashAndIdsToMap(
                        staticResourceHashToIdsMountedToReadOnlyVolumes
                        , volumes.Volumes_.at(volumeId).Target_.Meta_.StaticResourceRefs_
                        , staticResources.StaticResources_
                    );
                }  else if (mountedVolume.mode() == API::EVolumeMountMode_READ_WRITE) {
                    AddStaticResourceHashAndIdsToMap(
                        staticResourceHashToIdsMountedToWritableVolumes
                        , volumes.Volumes_.at(volumeId).Target_.Meta_.StaticResourceRefs_
                        , staticResources.StaticResources_
                    );
                }

            } else if (mountedVolume.volume_type_case() == API::TMountedVolume::VolumeTypeCase::kRbindVolumeRef) {
                const TString& rbindVolumeRef = mountedVolume.rbind_volume_ref();
                Y_ENSURE(
                    !rbindVolumeRef.empty()
                    , "Rbind volume ref is empty"
                );
                Y_ENSURE(
                    rbindVolumeRef.find('/') == TString::npos
                    , "Rbind volume ref " << Quote(rbindVolumeRef) << " contains '/'"
                );

                Y_ENSURE(
                    mountedVolume.mode() == API::EVolumeMountMode_READ_ONLY || mountedVolume.mode() == API::EVolumeMountMode_READ_WRITE
                    , "Incorrect mount mode for rbind volume " << Quote(rbindVolumeRef)
                );

                rbindVolumeStorageList.push_back(PathHolder_->GetRbindVolumeStorage(rbindVolumeRef));
                rbindVolumeReadOnlyModeList.push_back(mountedVolume.mode() == API::EVolumeMountMode_READ_ONLY ? "true" : "false");

                const TString rbindVolumePath = PathHolder_->GetBoxRootfsPath(boxId) + "/" + NSupport::NormalizeMountPoint(mountedVolume.mount_point());
                allMountPoints.push_back(rbindVolumePath);
                rbindVolumePathList.push_back(rbindVolumePath);
            }
        }

        {
            // Public volume
            mountVolumePathList.push_back(PublicVolumePath_);
            // Public volume hash is empty
            mountVolumeTreeHashList.push_back("");
            // Mount public volume in read only mode
            mountVolumeReadOnlyModeList.push_back("true");

            const TString volumeLinkPath = PathHolder_->GetBoxRootfsPath(boxId) + "/" + NSupport::NormalizeMountPoint(PublicVolumeMountPath_);
            allMountPoints.push_back(volumeLinkPath);
            mountVolumeLinkPathList.push_back(volumeLinkPath);
        }

        replace["MOUNT_VOLUME_PATH_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator(mountVolumePathList);
        replace["MOUNT_VOLUME_LINK_PATH_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator(mountVolumeLinkPathList);
        replace["MOUNT_VOLUME_RO_MODE_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator(mountVolumeReadOnlyModeList);
        replace["MOUNT_VOLUME_TREE_HASH_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator(mountVolumeTreeHashList);
        replace["NON_PERSISTENT_MOUNT_VOLUME_ID_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator(nonPersistentVolumeIdList);
        replace["NON_PERSISTENT_MOUNT_VOLUME_PATH_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator(nonPersistentVolumePathList);

        replace["RBIND_VOLUME_PATH_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator(rbindVolumePathList);
        replace["RBIND_VOLUME_STORAGE_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator(rbindVolumeStorageList);
        replace["RBIND_VOLUME_RO_MODE_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator(rbindVolumeReadOnlyModeList);

        NSupport::ValidateMountPoints(allMountPoints);

        {
            for (size_t i = 0; i < box.initSize(); ++i) {
                initContainers.push_back(PathHolder_->GetBoxInitContainer(boxId, i));
            }

            replace.merge(
                NSupport::FillInitContainersReplaceMap(
                    box.init()
                    , initContainers
                    , PathHolder_->GetBoxRootfsPath(boxId)
                    , PodAgentBinaryFilePathInBox_
                    , {} // Inherit env from box meta container
                    , secretMap
                    , PathHolder_
                    , cpuToVcpuFactor
                    , useEnvSecret
                )
            );
            replace["INIT_ULIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({box.initSize(), ""});
        }

        replace["TREE_HASH"] = treeHash;

        cachedReplace = replace;
        TBehavior3TemplateResolver().Resolve(resolvedTree, replace);
    }

    return TBoxToAdd{
        TUpdateHolder::TBoxTarget(
            TBoxMeta(
                boxId
                , specTimestamp
                , revision
                , layerRefs
                , staticResourceRefs
                , volumeRefs
                , PathHolder_->GetBoxContainer(boxId)
                , initContainers
                , box.specific_type()
            )
            , new TObjectTree(
                Logger_
                , TStatusNTickerHolder::GetBoxTreeId(boxId)
                , TBehavior3EditorJsonReader(resolvedTree)
                    .WithIpClient(IpClient_)
                    .WithPorto(Porto_)
                    .WithPosixWorker(PosixWorker_)
                    .WithBoxStatusRepository(StatusNTickerHolder_->GetBoxStatusRepository())
                    .WithLayerStatusRepository(StatusNTickerHolder_->GetLayerStatusRepository())
                    .WithStaticResourceStatusRepository(StatusNTickerHolder_->GetStaticResourceStatusRepository())
                    .WithVolumeStatusRepository(StatusNTickerHolder_->GetVolumeStatusRepository())
                    .WithTemplateBTStorage(TemplateBTStorage_)
                    .WithSystemLogsSender(StatusNTickerHolder_->GetBoxSystemLogsSender())
                    .BuildRootNode()
                , boxId
                , StatusNTickerHolder_->GetUpdateHolder()->GetUpdateHolderTarget()
                , StatusNTickerHolder_->GetBoxStatusRepository()
            )
            , treeHash
        )
        , boxIp6SubnetOffset
        , cachedReplace
    };
}

void TBoxTreeGenerator::GenAndSafeBoxCache(
    const TBoxesToAdd& boxes
    , TLogFramePtr logFrame
) {
    NTreesGenerators::TBoxesCache boxCache;

    for (const auto& it : boxes.Boxes_) {
        boxCache.MutableIP6SubnetOffset()->insert({it.first, it.second.Ip6SubnetOffset_});
    }
    boxCache.set_version(TBoxTreeGenerator::BOX_CACHE_VERSION);

    try {
        NCacheFile::SaveToFileJson(CacheFileName_, boxCache);
    } catch (const yexception& e) {
        logFrame->LogEvent(ELogPriority::TLOG_WARNING, NLogEvent::TBoxTreeGeneratorCacheWriteError(e.what()));
    }
}

NTreesGenerators::TBoxesCache TBoxTreeGenerator::LoadBoxCache(
    TLogFramePtr logFrame
) {
    NTreesGenerators::TBoxesCache boxCache;
    try {
        boxCache = NCacheFile::LoadFromFileJson<NTreesGenerators::TBoxesCache>(CacheFileName_);
    } catch (const yexception& e) {
        logFrame->LogEvent(ELogPriority::TLOG_WARNING, NLogEvent::TBoxTreeGeneratorCacheReadError(e.what()));
    }

    if (boxCache.version() != TBoxTreeGenerator::BOX_CACHE_VERSION) {
        boxCache = NTreesGenerators::TBoxesCache(); // clear box cache
        boxCache.set_version(TBoxTreeGenerator::BOX_CACHE_VERSION);
    }

    return boxCache;
}

TBoxTreeGenerator::TBoxIpInfo TBoxTreeGenerator::GetBoxIpInfo(
    const google::protobuf::RepeatedPtrField<API::TBox>& boxes
    , const google::protobuf::RepeatedPtrField<NYP::NClient::NApi::NProto::TPodStatus::TIP6SubnetAllocation>& subnets
    , NTreesGenerators::TBoxesCache& boxCache
) {
    TBoxTreeGenerator::TBoxIpInfo boxIpInfo{
        false
        , {}
    };

    for (auto& it : subnets) {
        for (auto& attr: it.labels().attributes()) {
            if (attr.key() == "id" && attr.value() == "\x01\x18"/*yson header*/ "boxes_subnet") {
                Y_ENSURE(!boxIpInfo.HaveIp_, "Got several subnets with 'boxes_subnet' label");
                boxIpInfo.HaveIp_ = true;
                const TString& addr = it.subnet();
                Y_ENSURE(addr.EndsWith("/112"), "Subnet should end with '/112', got '" << addr << "'");
                TString subnetStart = addr.substr(0, addr.length() - 4); // cut '/112' tail
                boxIpInfo.Ip6Subnet_ = Ip6FromString(subnetStart.c_str());
            }
        }
    }

    THashSet<TString> boxesIds;

    if (boxIpInfo.HaveIp_) {
        // Add all boxes from current spec
        for (const auto& box : boxes) {
            boxesIds.insert(box.id());
        }
    }

    // Add all boxes from current internal state
    for (const auto& boxId : StatusNTickerHolder_->GetBoxIds()) {
        boxesIds.insert(boxId);
    }

    auto& ipMap = *(boxCache.MutableIP6SubnetOffset());
    for (auto it = ipMap.begin(); it != ipMap.end(); /* pass */) { // remove old ids from cache
        if (boxesIds.contains(it->first)) {
            ++it;
        } else {
            it = ipMap.erase(it);
        }
    }

    return boxIpInfo;
}

void TBoxTreeGenerator::RemoveBoxes(const TBoxesToAdd& boxes) {
    for (const TString& boxId : StatusNTickerHolder_->GetBoxIds()) {
        if (!boxes.Boxes_.contains(boxId)) {
            StatusNTickerHolder_->SetBoxTargetRemove(boxId);
        }
    }
}

void TBoxTreeGenerator::AddBoxes(
    const TBoxesToAdd& boxes
    , TLogFramePtr logFrame
) {
    for (const auto& it : boxes.Boxes_) {
        const TString& boxId = it.first;
        const auto& boxData = it.second;

        if (!StatusNTickerHolder_->HasBox(boxId)) {
            StatusNTickerHolder_->AddBoxWithTargetCheck(boxData.Target_);
        }

        StatusNTickerHolder_->UpdateBoxTarget(boxData.Target_);
    }

    GenAndSafeBoxCache(boxes, logFrame);
    StatusNTickerHolder_->UpdateBoxIp6Subnet112Base(boxes.Ip6Subnet112Base_);
}

void TBoxTreeGenerator::AddStaticResourceHashAndIdToMap(
    TMap<TString, TVector<TString>>& resourceHashToIds
    , const TString& resourceHash
    , const TString& resourceId
) {
    auto resourceIt = resourceHashToIds.find(resourceHash);
    if (resourceIt == resourceHashToIds.end()) {
        resourceHashToIds[resourceHash] = {resourceId};
    } else {
        resourceIt->second.push_back(resourceId);
    }
}

void TBoxTreeGenerator::AddStaticResourceHashAndIdsToMap(
    TMap<TString, TVector<TString>>& resourceHashToIds
    , const TVector<TString>& staticResourceRefs
    , const TMap<TString, TStaticResourceTreeGenerator::TStaticResourceToAdd>& staticResources
) {
    for (const auto& staticResourceId : staticResourceRefs) {
        TString staticResourceDownloadHash = staticResources.at(staticResourceId).Target_.Meta_.DownloadHash_;
        AddStaticResourceHashAndIdToMap(resourceHashToIds, staticResourceDownloadHash, staticResourceId);
    }
}

void TBoxTreeGenerator::ValidateStaticResourcesMountedToReadOnlyAndWritableVolumeSimultaneously(
    const TMap<TString, TVector<TString>>& staticResourceHashToIdsMountedToReadOnlyVolumes
    , const TMap<TString, TVector<TString>>& staticResourceHashToIdsMountedToWritableVolumes
) {
    TSet<TString> hashesMountedToReadOnlyVolumes;
    TSet<TString> hashesMountedToWritableVolumes;
    transform(
        staticResourceHashToIdsMountedToReadOnlyVolumes.begin()
        , staticResourceHashToIdsMountedToReadOnlyVolumes.end()
        , std::inserter(hashesMountedToReadOnlyVolumes, hashesMountedToReadOnlyVolumes.end())
        , [](const auto& pair) { return pair.first; }
    );
    transform(
        staticResourceHashToIdsMountedToWritableVolumes.begin()
        , staticResourceHashToIdsMountedToWritableVolumes.end()
        , std::inserter(hashesMountedToWritableVolumes, hashesMountedToWritableVolumes.end())
        , [](const auto& pair) { return pair.first; }
    );

    TSet<TString> hashesMountedToReadOnlyAndWritableVolumes;
    set_intersection(
        hashesMountedToReadOnlyVolumes.begin(), hashesMountedToReadOnlyVolumes.end()
        , hashesMountedToWritableVolumes.begin(), hashesMountedToWritableVolumes.end()
        , inserter(hashesMountedToReadOnlyAndWritableVolumes, hashesMountedToReadOnlyAndWritableVolumes.begin())
    );

    if (!hashesMountedToReadOnlyAndWritableVolumes.empty()) {
        TStringBuilder errorDescription = TStringBuilder() << "Static resources list, bound at once to read only and writable volumes: " << Endl;
        for (const auto& resourceHash : hashesMountedToReadOnlyAndWritableVolumes) {
            TVector<TString> idsMountedToReadOnlyVolumes = staticResourceHashToIdsMountedToReadOnlyVolumes.at(resourceHash);
            TVector<TString> idsMountedToWritableVolumes = staticResourceHashToIdsMountedToWritableVolumes.at(resourceHash);
            TSet<TString> idsMountedToReadOnlyAndWritableVolumes(idsMountedToReadOnlyVolumes.begin(), idsMountedToReadOnlyVolumes.end());
            idsMountedToReadOnlyAndWritableVolumes.insert(idsMountedToWritableVolumes.begin(), idsMountedToWritableVolumes.end());
            errorDescription << "hash: " << resourceHash << ", ids: ";
            errorDescription << JoinSeq(", ", idsMountedToReadOnlyAndWritableVolumes) << Endl;
        }
        throw yexception() << errorDescription;
    }
}

} // namespace NInfra::NPodAgent
