#pragma once

#include "object_tree.h"

#include "layer_tree_generator.h"
#include "static_resource_tree_generator.h"
#include "volume_tree_generator.h"

#include <infra/pod_agent/libs/behaviour/template_bt_storage/template_bt_storage.h>
#include <infra/pod_agent/libs/ip_client/async_client.h>
#include <infra/pod_agent/libs/path_util/path_holder.h>
#include <infra/pod_agent/libs/pod_agent/status_and_ticker_holder/status_and_ticker_holder.h>
#include <infra/pod_agent/libs/pod_agent/trees_generators/protos/boxes_cache.pb.h>
#include <infra/pod_agent/libs/porto_client/async_client.h>
#include <infra/pod_agent/libs/posix_worker/posix_worker.h>

#include <util/draft/ip.h>

namespace NInfra::NPodAgent {

/**
    Creates behavior tree for each new box from provided TreeTemplate
    Removes unneeded trees
    Ticks trees via TMtpPeriodTicker
    Method Wait() waits for signal from Shutdown() and waits the TMtpPeriodTicker
    Method Stop() calls {Shutdown(); Wait();}
*/
class TBoxTreeGenerator {
public:
    TBoxTreeGenerator(
        TLogger& logger
        , TPathHolderPtr pathHolder
        , const TBehavior3& boxTreeTemplate
        , TAsyncIpClientPtr ipClient
        , TAsyncPortoClientPtr porto
        , TPosixWorkerPtr posixWorker
        , TStatusNTickerHolderPtr statusNTickerHolder
        , TTemplateBTStoragePtr templateBTStorage
        , const TString& cacheFileName
        , const TString& hostname
        , const TString& ytPath
        , const TString& baseSearchPath
        , const TString& publicVolumePath
        , const TString& publicVolumeMountPath
        , const TString& podAgentBinaryFilePathInBox
        , const TString& newtworkDevice
    );

    ~TBoxTreeGenerator() {}

public:
    struct TBoxToAdd {
        TUpdateHolder::TBoxTarget Target_;
        ui16 Ip6SubnetOffset_;
        // need for tests
        TMap<TString, TString> TreeReplaceMap_;
    };

    struct TBoxesToAdd {
        TMap<TString, TBoxToAdd> Boxes_;
        TString Ip6Subnet112Base_; // first 112 bits of subnet in format '2a02:6b8:c08:68a3:0:696:6937:'
    };

    struct TBoxIpInfo {
        bool HaveIp_;
        TIp6 Ip6Subnet_;
    };

public:
    TBoxTreeGenerator::TBoxesToAdd UpdateSpec(
        const google::protobuf::RepeatedPtrField<API::TBox>& boxes
        , API::EPodAgentTargetState podAgentTargetState
        , const TLayerTreeGenerator::TLayersToAdd& layers
        , const TStaticResourceTreeGenerator::TStaticResourcesToAdd& staticResources
        , const TVolumeTreeGenerator::TVolumesToAdd& volumes
        , const TString& podId
        , const API::TNodeMeta& nodeMeta
        , const API::TGpuManagerMeta& gpuManagerMeta
        , const NYP::NClient::NApi::NProto::TPodStatus::TDns& dns
        , const google::protobuf::RepeatedPtrField<NYP::NClient::NApi::NProto::TPodStatus::TIP6SubnetAllocation>& subnets
        , const google::protobuf::RepeatedPtrField<NYP::NClient::NApi::NProto::TPodStatus::TIP6AddressAllocation>& ip6AddressAllocations
        , const NSecret::TSecretMap& secretMap
        // pass pod.rbind.yt to tree generator
        // This is done through annotations to speed up development DEPLOY-477
        // TODO do it in other way
        , const NYT::NYTree::NProto::TAttributeDictionary& annotations
        , const double cpuToVcpuFactor
        , ui64 specTimestamp
        , ui32 revision
        , TLogFramePtr logFrame
        , bool useEnvSecret = false
        , bool autoDecodeBase64Secrets = false
    );

    void RemoveBoxes(const TBoxesToAdd& toAdd);
    void AddBoxes(
        const TBoxesToAdd& toAdd
        , TLogFramePtr logFrame
    );

private:
    TBoxToAdd GetBoxToAdd(
        const API::TBox& box
        , const TLayerTreeGenerator::TLayersToAdd& layers
        , const TStaticResourceTreeGenerator::TStaticResourcesToAdd& staticResources
        , const TVolumeTreeGenerator::TVolumesToAdd& volumes
        , const TMap<TString, TString>& systemEnv
        , const NSecret::TSecretMap& secretMap
        , NTreesGenerators::TBoxesCache& boxCache
        , const TBoxIpInfo& boxIpInfo
        , const TMaybe<TString>& ytBindPath
        , const TMaybe<TString>& baseSearchBindPath
        , const double cpuToVcpuFactor
        , ui64 specTimestamp
        , ui32 revision
        , bool useEnvSecret
        , TMap<TString, TVector<TString>>& staticResourceHashToIdsMountedToReadOnlyVolumes
        , TMap<TString, TVector<TString>>& staticResourceHashToIdsMountedToWritableVolumes
        , bool autoDecodeBase64Secrets
    ) const;

    void GenAndSafeBoxCache(
        const TBoxesToAdd& boxes
        , TLogFramePtr logFrame
    );

    NTreesGenerators::TBoxesCache LoadBoxCache(
        TLogFramePtr logFrame
    );

    TBoxIpInfo GetBoxIpInfo(
        const google::protobuf::RepeatedPtrField<API::TBox>& boxes
        , const google::protobuf::RepeatedPtrField<NYP::NClient::NApi::NProto::TPodStatus::TIP6SubnetAllocation>& subnets
        , NTreesGenerators::TBoxesCache& boxCache
    );

private:
    static void AddStaticResourceHashAndIdToMap(
        TMap<TString, TVector<TString>>& resourceHashToIds
        , const TString& resourceHash
        , const TString& resourceId
    );

    static void AddStaticResourceHashAndIdsToMap(
        TMap<TString, TVector<TString>>& resourceHashToIds
        , const TVector<TString>& staticResourceRefs
        , const TMap<TString, TStaticResourceTreeGenerator::TStaticResourceToAdd>& staticResources
    );

    static void ValidateStaticResourcesMountedToReadOnlyAndWritableVolumeSimultaneously(
        const TMap<TString, TVector<TString>>& staticResourceHashToIdsMountedToReadOnlyVolumes
        , const TMap<TString, TVector<TString>>& staticResourceHashToIdsMountedToWritableVolumes
    );

private:
    TLogger& Logger_;
    TPathHolderPtr PathHolder_;
    const TBehavior3 BoxTreeTemplate_;
    const TString CacheFileName_;
    const TString SkynetBindPath_;
    const TString YtPath_;
    const TString BaseSearchPath_;
    const TString PublicVolumePath_;
    const TString PublicVolumeMountPath_;
    const TString PodAgentBinaryFilePathInBox_;

    TAsyncIpClientPtr IpClient_;
    TAsyncPortoClientPtr Porto_;
    TPosixWorkerPtr PosixWorker_;
    TStatusNTickerHolderPtr StatusNTickerHolder_;
    TTemplateBTStoragePtr TemplateBTStorage_;
    const TString LocalHostName_;
    const TString NetworkDevice_;
    const TString TVMToolLocalToken_;

private:
    static const ui64 BOX_CACHE_VERSION = 3;
};

} // namespace NInfra::NPodAgent
