#include "box_tree_generator.h"

#include "support_functions.h"
#include "test_canon.h"
#include "test_functions.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/test_functions.h>
#include <infra/pod_agent/libs/behaviour/loaders/behavior3_editor_json_reader.h>
#include <infra/pod_agent/libs/ip_client/mock_client.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/update_holder/test_lib/test_functions.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>
#include <infra/pod_agent/libs/util/string_utils.h>

#include <library/cpp/yson/node/node_io.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <util/system/fs.h>
#include <util/system/thread.h>

namespace NInfra::NPodAgent::NTreeGeneratorTest {

class ITestBoxTreeGeneratorCanon: public ITestTreeGeneratorCanon {
public:
    ITestBoxTreeGeneratorCanon()
        : ITestTreeGeneratorCanon()
        , PublicVolumePath_(NFs::CurrentWorkingDirectory() + "/public_volume")
        , PublicVolumeMountPath_("/pod_agent_public")
    {}

protected:
    virtual TString GetYtPath() {
        return "";
    }

    virtual TString GetBaseSearchPath() {
        return "";
    }

    void CreateWorker() final {
        // BoxTreeGenerator assumes that this directory always exist
        NFs::MakeDirectory(PublicVolumePath_);

        TMap<TString, TString> properties;
        properties["object_id_or_hash"] = "<RSLV:BOX_ID>";
        properties["state"] = "EBoxState_READY";
        properties["object_type"] = "box";
        Worker_.Reset(new TBoxTreeGenerator(
            logger
            , PathHolder_
            , GetTestTemplateTree (
                properties
                , "FeedbackObjectState"
                , "FeedbackObjectState(box, EBoxState_READY)"
            )
            , new TAsyncIpClient(new TMockIpClient(), new TFakeThreadPool())
            , new TAsyncPortoClient(new TMockPortoClient(), new TFakeThreadPool())
            , new TPosixWorker(new TFakeThreadPool())
            , StatusNTickerHolder_
            , nullptr
            , "boxes_cache.json"
            , "" /* hostname */
            , GetYtPath()
            , GetBaseSearchPath()
            , PublicVolumePath_
            , PublicVolumeMountPath_
            , "/pod_agent_public/pod_agent" /* podAgentBinaryFilePathInBox */
            , "" /* networkDevice */
        ));
    }

    TBoxTreeGenerator::TBoxesToAdd UpdateSpec(
        const ::google::protobuf::RepeatedPtrField<API::TBox>& boxes
        , API::EPodAgentTargetState podAgentTargetState
        , const TLayerTreeGenerator::TLayersToAdd& layers
        , const TStaticResourceTreeGenerator::TStaticResourcesToAdd& staticResources
        , const TVolumeTreeGenerator::TVolumesToAdd& volumes
        , const NSecret::TSecretMap& secretMap
        , const NYT::NYTree::NProto::TAttributeDictionary annotations
        , ui64 specTimestamp
        , ui32 revision
        , bool autoDecodeBase64Secrets = false
    ) {
        TBoxTreeGenerator::TBoxesToAdd newSpec = Worker_->UpdateSpec(
            boxes
            , podAgentTargetState
            , layers
            , staticResources
            , volumes
            , "podId"
            , {} /* nodeMeta */
            , {} /* gpuManagerMeta */
            , {} /* dns */
            , {} /* subnets */
            , {} /* ip6AddressAllocations */
            , secretMap
            , annotations
            , 1.0 /* CpuToVcpuFactor */
            , specTimestamp
            , revision
            , logger.SpawnFrame()
            , false /* useEnvSecret */
            , autoDecodeBase64Secrets
        );
        Worker_->RemoveBoxes(newSpec);
        Worker_->AddBoxes(newSpec, logger.SpawnFrame());
        return newSpec;
    }

    static TLayerTreeGenerator::TLayersToAdd GetLayerToAdd(const TString& layerId, const TString& virtualDiskRef = "") {
        TLayerTreeGenerator::TLayersToAdd layers;
        layers.Layers_.insert(
            {
                layerId
                , {
                    NObjectTargetTestLib::CreateLayerTargetSimple(layerId, layerId + "_download_hash_other", 3)
                    , "layer_tree_hash"
                    , virtualDiskRef
                }
            }
        );
        return layers;
    }

    static TString EscapeForGenerator(const TVector<TString>& sequence) {
        return TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator(sequence);
    }

    TString MakeVolumeLinkPath(const TString& boxId, const TString& mountPoint) {
        return PathHolder_->GetBoxRootfsPath(boxId) + "/" + NSupport::NormalizeMountPoint(mountPoint);
    }

    static void AddVolume(
        TMap<TString, TVolumeTreeGenerator::TVolumeToAdd>& volumes
        , const TString& volumeId
        , const TString& hash
        , bool isPersistent = true
    ) {
        volumes.insert(
            {
                volumeId
                , {
                    isPersistent
                        ? NObjectTargetTestLib::CreatePersistentVolumeTargetSimple(volumeId, hash)
                        : NObjectTargetTestLib::CreateNonPersistentVolumeTargetSimple(volumeId, hash)
                }
            }
        );
    }

protected:
    THolder<TBoxTreeGenerator> Worker_;
    TString PublicVolumePath_;
    TString PublicVolumeMountPath_;
};

Y_UNIT_TEST_SUITE(BoxTreeGeneratorSuite) {

Y_UNIT_TEST(TestSimpleTree) {
    class TTest : public ITestBoxTreeGeneratorCanon {
    public:
        TTest() : ITestBoxTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TPodAgentSpec spec;

            API::TBox* box = spec.add_boxes();
            box->set_id("my_box");
            box->set_virtual_disk_id_ref("virtual_disk");

            API::TRootfsVolume* rootfs = box->mutable_rootfs();
            rootfs->add_layer_refs("my_layer");

            API::TMountedStaticResource* staticResource = box->add_static_resources();
            staticResource->set_resource_ref("my_static_resource");
            staticResource->set_mount_point("static_resource_mount_point");

            LayerStatusRepository_->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple("my_layer", "my_layer_download_hash"));
            TLayerTreeGenerator::TLayersToAdd layers = GetLayerToAdd("my_layer", "virtual_disk");

            StaticResourceStatusRepository_->AddObject(NObjectMetaTestLib::CreateStaticResourceMetaSimple("my_static_resource", "my_static_resource_download_hash"));
            TStaticResourceTreeGenerator::TStaticResourcesToAdd staticResources;
            staticResources.StaticResources_.insert(
                {
                    "my_static_resource"
                    , {
                        NObjectTargetTestLib::CreateStaticResourceTargetSimple("my_static_resource", "my_static_resource_download_hash")
                        , "static_resource_tree_hash"
                        , "virtual_disk"
                        , {}
                    }
                }
            );

            TVolumeTreeGenerator::TVolumesToAdd volumes;

            auto boxCopy = spec.add_boxes();
            boxCopy->CopyFrom(*box);

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, staticResources, volumes, {}, {}, 0, 0)
                , yexception
                , "boxes' ids are not unique: box 'my_box' occurs twice"
            );

            *boxCopy->mutable_id() += "_other";
            auto boxesToAdd = UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, staticResources, volumes, {}, {}, 0, 0);

            *boxCopy->mutable_id() = "";
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, staticResources, volumes, {}, {}, 0, 0)
                , yexception
                , "One of boxes has empty id"
            );

            *boxCopy->mutable_id() = *box->mutable_id() + "_other";
            boxesToAdd = UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, staticResources, volumes, {}, {}, 0, 0);
            layers.Layers_.at("my_layer").VirtualDiskIdRef_ = "";
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, staticResources, volumes, {}, {}, 0, 0)
                , yexception
                , "another virtual disk ref"
            );

            layers.Layers_.at("my_layer").VirtualDiskIdRef_ = "bad_virtual_disk";
            staticResources.StaticResources_.at("my_static_resource").VirtualDiskIdRef_ = "bad_virtual_disk";
            box->set_virtual_disk_id_ref("bad_virtual_disk");
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, staticResources, volumes, {}, {}, 0, 0)
                , yexception
                , "Unknown virtual disk ref"
            );

            layers.Layers_.at("my_layer").VirtualDiskIdRef_ = "virtual_disk";
            staticResources.StaticResources_.at("my_static_resource").VirtualDiskIdRef_ = "virtual_disk";
            box->set_virtual_disk_id_ref("virtual_disk");
            boxesToAdd = UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, staticResources, volumes, {}, {}, 0, 0);

            staticResources.StaticResources_.at("my_static_resource").VirtualDiskIdRef_ = "bad_virtual_disk";
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, staticResources, volumes, {}, {}, 0, 0)
                , yexception
                , "another virtual disk ref"
            );

            Sleep(TDuration::MilliSeconds(500));
            UNIT_ASSERT_EQUAL(API::EBoxState_READY, BoxStatusRepository_->GetObjectStatus("my_box").state());
            UNIT_ASSERT(boxesToAdd.Boxes_.at("my_box").Target_.Tree_->GetUseLongTickPeriod());
        }

    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestEmptyInitCommandLine) {
    class TTest : public ITestBoxTreeGeneratorCanon {
    public:
        TTest() : ITestBoxTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TPodAgentSpec spec;

            API::TBox* box = spec.add_boxes();
            box->set_id("my_box");

            API::TRootfsVolume* rootfs = box->mutable_rootfs();
            rootfs->add_layer_refs("my_layer");

            API::TMountedVolume* mountedVolume = box->add_volumes();
            mountedVolume->set_volume_ref("my_volume");
            mountedVolume->set_mount_point("mount_point");
            VolumeStatusRepository_->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple("my_volume"));

            API::TUtilityContainer* init = box->add_init();
            init->mutable_time_limit()->set_min_restart_period_ms(60000);
            init->mutable_time_limit()->set_max_restart_period_ms(60000);
            init->mutable_time_limit()->set_max_execution_time_ms(60000);

            LayerStatusRepository_->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple("my_layer", "my_layer_download_hash"));
            TLayerTreeGenerator::TLayersToAdd layers = GetLayerToAdd("my_layer");

            TVolumeTreeGenerator::TVolumesToAdd volumes;
            volumes.Volumes_.insert(
                {
                    "my_volume"
                    , {
                        NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("my_volume", "volume_tree_hash")
                    }
                }
            );

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, {}, volumes, {}, {}, 0, 0)
                , yexception
                , "init container command_line is empty at box 'my_box'"
            );
            init->set_command_line("echo init");
            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, {}, volumes, {}, {}, 0, 0));
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestRbindVolume) {
    class TTest : public ITestBoxTreeGeneratorCanon {
    public:
        TTest() : ITestBoxTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TPodAgentSpec spec;

            API::TBox* box = spec.add_boxes();
            box->set_id("my_box");

            API::TRootfsVolume* rootfs = box->mutable_rootfs();
            rootfs->add_layer_refs("my_layer");

            API::TMountedVolume* mountedVolume = box->add_volumes();
            mountedVolume->set_rbind_volume_ref("rbind_volume_ref");
            mountedVolume->set_mount_point("mount_point");

            LayerStatusRepository_->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple("my_layer", "my_layer_download_hash"));
            TLayerTreeGenerator::TLayersToAdd layers = GetLayerToAdd("my_layer");

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, {}, {}, {}, {}, 0, 0));

            mountedVolume->set_rbind_volume_ref("");
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, {}, {}, {}, {}, 0, 0)
                , yexception
                , "bind volume ref is empty at box 'my_box'"
            );

            mountedVolume->set_rbind_volume_ref("rbind_volume_ref/my_subvolume");
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, {}, {}, {}, {}, 0, 0)
                , yexception
                , "Rbind volume ref 'rbind_volume_ref/my_subvolume' contains '/' at box 'my_box'"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestChangeTarget) {
    class TTest : public ITestBoxTreeGeneratorCanon {
    public:
        TTest() : ITestBoxTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TPodAgentSpec spec;

            API::TBox* box = spec.add_boxes();
            box->set_id("my_box");

            API::TRootfsVolume* rootfs = box->mutable_rootfs();
            rootfs->add_layer_refs("my_layer");

            LayerStatusRepository_->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple("my_layer", "my_layer_download_hash"));
            TLayerTreeGenerator::TLayersToAdd layers = GetLayerToAdd("my_layer");

            TVolumeTreeGenerator::TVolumesToAdd volumes;

            for (size_t i = 0; i < 3; ++i) {
                UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, {}, volumes, {}, {}, i + 1, i + 2));
                UNIT_ASSERT(BoxStatusRepository_->HasObject("my_box"));
                UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->BoxHasTarget("my_box"));
                UNIT_ASSERT(BoxStatusRepository_->GetObjectStatus("my_box").spec_timestamp() == i + 1);
                UNIT_ASSERT(BoxStatusRepository_->GetObjectStatus("my_box").revision() == i + 2);
            }

            // change box
            API::TUtilityContainer* init = box->add_init();
            init->mutable_time_limit()->set_min_restart_period_ms(60000);
            init->mutable_time_limit()->set_max_restart_period_ms(60000);
            init->mutable_time_limit()->set_max_execution_time_ms(60000);
            init->set_command_line("echo init");

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, {}, volumes, {}, {}, 4, 5));

            UNIT_ASSERT(BoxStatusRepository_->HasObject("my_box"));
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->BoxHasTarget("my_box"));
            UNIT_ASSERT(BoxStatusRepository_->GetObjectStatus("my_box").spec_timestamp() == 3);
            UNIT_ASSERT(BoxStatusRepository_->GetObjectStatus("my_box").revision() == 4);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestAddWithTargetCheck) {
    class TTest : public ITestBoxTreeGeneratorCanon {
    public:
        TTest() : ITestBoxTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TPodAgentSpec spec;

            API::TBox* box = spec.add_boxes();
            box->set_id("my_box");

            API::TRootfsVolume* rootfs = box->mutable_rootfs();
            rootfs->add_layer_refs("my_layer");

            Holder_->SetRevision(1);
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("my_layer", "my_layer_download_hash", 1));
            StatusNTickerHolder_->UpdateLayerTarget(NObjectTargetTestLib::CreateLayerTargetSimple("my_layer", "my_layer_download_hash_other", 3));

            TLayerTreeGenerator::TLayersToAdd layers = GetLayerToAdd("my_layer");

            TVolumeTreeGenerator::TVolumesToAdd volumes;

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, {}, volumes, {}, {}, 3, 4));
            UNIT_ASSERT(BoxStatusRepository_->HasObject("my_box"));
            UNIT_ASSERT(BoxStatusRepository_->GetObjectStatus("my_box").spec_timestamp() == 0);
            UNIT_ASSERT(BoxStatusRepository_->GetObjectStatus("my_box").revision() == 0);
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->BoxHasTarget("my_box"));
            TUpdateHolder::TBoxTarget boxTarget = UpdateHolder_->GetAndRemoveBoxTarget("my_box");
            UNIT_ASSERT(boxTarget.Meta_.SpecTimestamp_ == 3);
            UNIT_ASSERT(boxTarget.Meta_.Revision_ == 4);
        }

    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestPodAgentTargetStateWithBoxTree) {
    class TTest : public ITestBoxTreeGeneratorCanon {
    public:
        TTest() : ITestBoxTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TPodAgentSpec spec;

            API::TBox* box = spec.add_boxes();
            box->set_id("my_box");

            API::TRootfsVolume* rootfs = box->mutable_rootfs();
            rootfs->add_layer_refs("my_layer");

            LayerStatusRepository_->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple("my_layer", "my_layer_download_hash"));
            TLayerTreeGenerator::TLayersToAdd layers = GetLayerToAdd("my_layer");

            TVolumeTreeGenerator::TVolumesToAdd volumes;

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, {}, volumes, {}, {}, 0, 0));
            UNIT_ASSERT(BoxStatusRepository_->HasObject("my_box"));
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->BoxHasTarget("my_box"));

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.boxes(), API::EPodAgentTargetState_SUSPENDED, layers, {}, volumes, {}, {}, 1, 1));
            UNIT_ASSERT(BoxStatusRepository_->HasObject("my_box"));
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->BoxHasTarget("my_box"));

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.boxes(), API::EPodAgentTargetState_REMOVED, layers, {}, volumes, {}, {}, 2, 2));
            UNIT_ASSERT(BoxStatusRepository_->HasObject("my_box"));
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->BoxHasTarget("my_box"));
            UNIT_ASSERT(UpdateHolder_->GetAndRemoveBoxTarget("my_box").TargetRemove_);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestSetIP6Address) {
    class TTest : public ITestBoxTreeGeneratorCanon {
    public:
        TTest() : ITestBoxTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TPodAgentSpec spec;
            API::TResourceGang* gang = spec.mutable_resources();
            API::TLayer* layer = gang->add_layers();
            layer->set_id("my_layer");
            {
                API::TBox* box = spec.add_boxes();
                box->set_id("my_box_0");
                API::TRootfsVolume* rootfs = box->mutable_rootfs();
                rootfs->add_layer_refs("my_layer");
            }
            {
                API::TBox* box = spec.add_boxes();
                box->set_id("my_box_1");
                API::TRootfsVolume* rootfs = box->mutable_rootfs();
                rootfs->add_layer_refs("my_layer");
            }

            LayerStatusRepository_->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple("my_layer", "my_layer_download_hash"));
            TLayerTreeGenerator::TLayersToAdd layers = GetLayerToAdd("my_layer");

            google::protobuf::RepeatedPtrField<NYP::NClient::NApi::NProto::TPodStatus::TIP6SubnetAllocation> subnets;
            NYP::NClient::NApi::NProto::TPodStatus::TIP6SubnetAllocation* subnet = subnets.Add();
            subnet->Setsubnet("2a02:6b8:c08:c812:0:696:af91:0/112");
            NYT::NYTree::NProto::TAttribute* label = subnet->Mutablelabels()->Addattributes();
            label->Setkey("id");
            label->Setvalue("\x01\x18"/*yson header*/ "boxes_subnet");
            subnet->Setvlan_id("backbone");

            TBoxTreeGenerator::TBoxesToAdd boxesToAdd = Worker_->UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, {}, {}, "podId", {}, {}, {}, subnets, {}, {}, {}, 1.0, 1, 2, logger.SpawnFrame());
            UNIT_ASSERT_EQUAL_C("2a02:6b8:c08:c812:0:696:af91:", boxesToAdd.Ip6Subnet112Base_, boxesToAdd.Ip6Subnet112Base_);
            {
                UNIT_ASSERT(boxesToAdd.Boxes_.contains("my_box_0"));
                auto& box = boxesToAdd.Boxes_.at("my_box_0");
                UNIT_ASSERT_EQUAL_C(1, box.Ip6SubnetOffset_, box.Ip6SubnetOffset_);
            }
            {
                UNIT_ASSERT(boxesToAdd.Boxes_.contains("my_box_1"));
                auto& box = boxesToAdd.Boxes_.at("my_box_1");
                UNIT_ASSERT_EQUAL_C(2, box.Ip6SubnetOffset_, box.Ip6SubnetOffset_);
            }
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestSecrets) {
    class TTest : public ITestBoxTreeGeneratorCanon {
    public:
        TTest() : ITestBoxTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TPodAgentSpec spec;

            API::TBox* box = spec.add_boxes();
            box->set_id("my_box");

            API::TRootfsVolume* rootfs = box->mutable_rootfs();
            rootfs->add_layer_refs("my_layer");
            LayerStatusRepository_->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple("my_layer", "my_layer_download_hash"));

            LayerStatusRepository_->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple("my_layer", "my_layer_download_hash"));
            TLayerTreeGenerator::TLayersToAdd layers = GetLayerToAdd("my_layer");

            TVolumeTreeGenerator::TVolumesToAdd volumes;

            API::TPodAgentRequest podAgentRequest;

            auto* envVar = box->add_env();
            envVar->set_name("TestSecret");
            envVar->mutable_value()->mutable_secret_env()->set_id("SecretKey");
            envVar->mutable_value()->mutable_secret_env()->set_alias("SecretAlias");

            auto* secret = podAgentRequest.add_secrets();
            secret->set_id("SecretAlias");
            auto* value = secret->add_values();
            value->set_key("SecretKey");
            value->set_value("SomeValue");

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, {}, volumes, NSecret::GetSecretMap(podAgentRequest.secrets()), {}, 0, 0));

            value->set_value("SomeOtherValue");
            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, {}, volumes, NSecret::GetSecretMap(podAgentRequest.secrets()), {}, 0, 0));

            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->BoxHasTarget("my_box"));

            value->set_value("U29tZU90aGVyVmFsdWU=");
            value->set_encoding("base64");

            auto boxesToAdd = UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, {}, volumes, NSecret::GetSecretMap(podAgentRequest.secrets()), {}, 0, 0, /* autoDecodeBase64Secrets = */ true);
            auto envList = boxesToAdd.Boxes_.at("my_box").TreeReplaceMap_.at("BOX_ENVIRONMENT");
            UNIT_ASSERT_STRING_CONTAINS_C(envList, "TestSecret=SomeOtherValue", envList);
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->BoxHasTarget("my_box"));
        }

    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestYtBind) {
    class TTest : public ITestBoxTreeGeneratorCanon {
    public:
        TTest() : ITestBoxTreeGeneratorCanon()
        {}

    protected:
        TString GetYtPath() override {
            return NFs::CurrentWorkingDirectory() + "/yt";
        }

        void Test() override {
            NYT::NYTree::NProto::TAttributeDictionary annotations;

            auto* attribute = annotations.mutable_attributes()->Add();
            NYT::TNode trueNode(true);
            attribute->set_key("pod.rbind.yt");
            attribute->set_value(NYT::NodeToYsonString(trueNode));

            API::TPodAgentSpec spec;
            API::TResourceGang* gang = spec.mutable_resources();
            API::TLayer* layer = gang->add_layers();
            layer->set_id("my_layer");

            API::TBox* box = spec.add_boxes();
            box->set_id("my_box");
            API::TRootfsVolume* rootfs = box->mutable_rootfs();
            rootfs->add_layer_refs("my_layer");

            LayerStatusRepository_->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple("my_layer", "my_layer_download_hash"));
            TLayerTreeGenerator::TLayersToAdd layers = GetLayerToAdd("my_layer");

            UNIT_ASSERT(!NFs::Exists(GetYtPath()));
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                Worker_->UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, {}, {}, "podId", {}, {}, {}, {}, {}, {}, annotations, 1.0, 1, 2, logger.SpawnFrame())
                , yexception
                , "Yt bind annotation provided, but yt path doesn't exist"
            );

            NFs::MakeDirectoryRecursive(GetYtPath());
            UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, {}, {}, "podId", {}, {}, {}, {}, {}, {}, annotations, 1.0, 1, 2, logger.SpawnFrame()));
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestBaseSearchBind) {
    class TTest : public ITestBoxTreeGeneratorCanon {
    public:
        TTest() : ITestBoxTreeGeneratorCanon()
        {}

    protected:
        TString GetBaseSearchPath() override {
            return NFs::CurrentWorkingDirectory() + "/basesearch";
        }

        void Test() override {
            NYT::NYTree::NProto::TAttributeDictionary annotations;

            auto* attribute = annotations.mutable_attributes()->Add();
            NYT::TNode trueNode(true);
            attribute->set_key("pod.rbind.basesearch");
            attribute->set_value(NYT::NodeToYsonString(trueNode));

            API::TPodAgentSpec spec;
            API::TResourceGang* gang = spec.mutable_resources();
            API::TLayer* layer = gang->add_layers();
            layer->set_id("my_layer");

            API::TBox* box = spec.add_boxes();
            box->set_id("my_box");
            API::TRootfsVolume* rootfs = box->mutable_rootfs();
            rootfs->add_layer_refs("my_layer");

            LayerStatusRepository_->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple("my_layer", "my_layer_download_hash"));
            TLayerTreeGenerator::TLayersToAdd layers = GetLayerToAdd("my_layer");

            UNIT_ASSERT(!NFs::Exists(GetBaseSearchPath()));
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                Worker_->UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, {}, {}, "podId", {}, {}, {}, {}, {}, {}, annotations, 1.0, 1, 2, logger.SpawnFrame())
                , yexception
                , "Basesearch bind annotation provided, but basesearch path doesn't exist"
            );

            NFs::MakeDirectoryRecursive(GetBaseSearchPath());
            UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, {}, {}, "podId", {}, {}, {}, {}, {}, {}, annotations, 1.0, 1, 2, logger.SpawnFrame()));
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestBoxWithCgroupFs) {
    class TTest : public ITestBoxTreeGeneratorCanon {
    public:
        TTest() : ITestBoxTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TPodAgentSpec spec;

            API::TBox* box = spec.add_boxes();
            box->set_id("my_box");
            box->set_cgroup_fs_mount_mode(API::ECgroupFsMountMode_NONE);

            API::TRootfsVolume* rootfs = box->mutable_rootfs();
            rootfs->add_layer_refs("my_layer");

            LayerStatusRepository_->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple("my_layer", "my_layer_download_hash"));
            TLayerTreeGenerator::TLayersToAdd layers = GetLayerToAdd("my_layer");

            auto boxesToAdd = Worker_->UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, {}, {}, "podId", {}, {}, {}, {}, {}, {}, {}, 1.0, 1, 2, logger.SpawnFrame());
            UNIT_ASSERT_EQUAL(boxesToAdd.Boxes_.at("my_box").TreeReplaceMap_.at("BOX_CGROUP_FS_MOUNT_TYPE"), "");

            box->set_cgroup_fs_mount_mode(API::ECgroupFsMountMode_RO);

            boxesToAdd = Worker_->UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, {}, {}, "podId", {}, {}, {}, {}, {}, {}, {}, 1.0, 1, 2, logger.SpawnFrame());
            UNIT_ASSERT_EQUAL(boxesToAdd.Boxes_.at("my_box").TreeReplaceMap_.at("BOX_CGROUP_FS_MOUNT_TYPE"), "ro");
            
            box->set_cgroup_fs_mount_mode(API::ECgroupFsMountMode_RW);

            boxesToAdd = Worker_->UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, {}, {}, "podId", {}, {}, {}, {}, {}, {}, {}, 1.0, 1, 2, logger.SpawnFrame());
            UNIT_ASSERT_EQUAL(boxesToAdd.Boxes_.at("my_box").TreeReplaceMap_.at("BOX_CGROUP_FS_MOUNT_TYPE"), "rw");
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestBoxWithVolumes) {
    class TTest : public ITestBoxTreeGeneratorCanon {
    public:
        TTest() : ITestBoxTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TPodAgentSpec spec;
            TString boxId = "my_box";
            TString layerId = "my_layer";
            TString persistentVolumeId = "persistent_volume";
            TString persistentVolumeHash = "persistent_volume_tree_hash";
            TString nonPersistentVolumeId = "non_persistent_volume";
            TString nonPersistentVolumeHash = "non_persistent_volume_tree_hash";
            TString persistentMountPoint = "mount_point1";
            TString nonPersistentMountPoint = "mount_point2";

            API::TBox* box = spec.add_boxes();
            box->set_id(boxId);

            API::TRootfsVolume* rootfs = box->mutable_rootfs();
            rootfs->add_layer_refs(layerId);

            API::TMountedVolume* mountedVolume1 = box->add_volumes();
            mountedVolume1->set_volume_ref(persistentVolumeId);
            mountedVolume1->set_mount_point(persistentMountPoint);
            VolumeStatusRepository_->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple(persistentVolumeId));

            API::TMountedVolume* mountedVolume2 = box->add_volumes();
            mountedVolume2->set_volume_ref(nonPersistentVolumeId);
            mountedVolume2->set_mount_point(nonPersistentMountPoint);
            VolumeStatusRepository_->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple(nonPersistentVolumeId));

            LayerStatusRepository_->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(layerId, "my_layer_download_hash"));
            TLayerTreeGenerator::TLayersToAdd layers = GetLayerToAdd(layerId);

            TVolumeTreeGenerator::TVolumesToAdd volumes;
            AddVolume(volumes.Volumes_, persistentVolumeId, persistentVolumeHash);
            AddVolume(volumes.Volumes_, nonPersistentVolumeId, nonPersistentVolumeHash, /* isPersistent = */ false);

            auto boxesToAdd = UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, {}, volumes, {}, {}, 0, 0);
            const auto& replace = boxesToAdd.Boxes_.at(boxId).TreeReplaceMap_;
            TMap<TString, TString> expectPartReplace = {
                {"MOUNT_VOLUME_PATH_LIST", EscapeForGenerator({
                    PathHolder_->GetVolumePath(persistentVolumeId)
                    , PathHolder_->GetVolumePath(nonPersistentVolumeId)
                    , PublicVolumePath_
                })}
                , {"MOUNT_VOLUME_TREE_HASH_LIST", EscapeForGenerator({
                    persistentVolumeHash
                    , nonPersistentVolumeHash
                    , ""
                })}
                , {"MOUNT_VOLUME_LINK_PATH_LIST", EscapeForGenerator({
                    MakeVolumeLinkPath(boxId, persistentMountPoint)
                    , MakeVolumeLinkPath(boxId, nonPersistentMountPoint)
                    , MakeVolumeLinkPath(boxId, PublicVolumeMountPath_)
                })}
                , {"MOUNT_VOLUME_RO_MODE_LIST", EscapeForGenerator({"true", "true", "true"})}
                , {"NON_PERSISTENT_MOUNT_VOLUME_ID_LIST", EscapeForGenerator({
                    nonPersistentVolumeId
                })}
                , {"NON_PERSISTENT_MOUNT_VOLUME_PATH_LIST", EscapeForGenerator({
                    PathHolder_->GetVolumePath(nonPersistentVolumeId)
                })}
            };
            for (const auto& [key, value] : expectPartReplace) {
                UNIT_ASSERT_EQUAL_C(replace.at(key), value, key + " expected " + Quote(value) + ", got " + Quote(replace.at(key)));
            }
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestBoxWithStaticResourcesMountedToVolumes) {
    class TTest : public ITestBoxTreeGeneratorCanon {
    public:
        TTest() : ITestBoxTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TPodAgentSpec spec;

            API::TBox* box = spec.add_boxes();
            box->set_id("my_box");
            box->set_virtual_disk_id_ref("virtual_disk");

            API::TRootfsVolume* rootfs = box->mutable_rootfs();
            rootfs->add_layer_refs("my_layer");

            API::TMountedStaticResource* staticResource = box->add_static_resources();
            TString staticResourceId = "my_static_resource";
            TString staticResourceDownloadHash = "my_static_resource_download_hash";
            staticResource->set_resource_ref(staticResourceId);
            staticResource->set_mount_point("static_resource_mount_point");

            LayerStatusRepository_->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple("my_layer", "my_layer_download_hash"));
            TLayerTreeGenerator::TLayersToAdd layers = GetLayerToAdd("my_layer", "virtual_disk");

            StaticResourceStatusRepository_->AddObject(NObjectMetaTestLib::CreateStaticResourceMetaSimple(staticResourceId, staticResourceDownloadHash));
            TStaticResourceTreeGenerator::TStaticResourcesToAdd staticResources;
            staticResources.StaticResources_.insert(
                {
                    staticResourceId
                    , {
                        NObjectTargetTestLib::CreateStaticResourceTargetSimple(staticResourceId, staticResourceDownloadHash)
                        , "static_resource_tree_hash"
                        , "virtual_disk"
                        , {}
                    }
                }
            );

            API::TMountedVolume* mountedVolume = box->add_volumes();
            mountedVolume->set_volume_ref("my_volume");
            mountedVolume->set_mount_point("mount_point");
            mountedVolume->set_mode(API::EVolumeMountMode_READ_WRITE);
            VolumeStatusRepository_->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple("my_volume"));

            TVolumeTreeGenerator::TVolumesToAdd volumes;
            volumes.Volumes_.insert(
                {
                    "my_volume"
                    , {
                        NObjectTargetTestLib::CreatePersistentVolumeTargetWithDependence("my_volume", "volume_tree_hash", {}, {staticResourceId})
                    }
                }
            );

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, staticResources, volumes, {}, {}, 0, 0));

            mountedVolume->set_mode(API::EVolumeMountMode_READ_ONLY);

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, staticResources, volumes, {}, {}, 0, 0)
                , yexception
                , TStringBuilder() << "Static resources list, bound at once to read only and writable volumes: " << Endl
                    << "hash: " << staticResourceDownloadHash << ", ids: " << staticResourceId
            );

            rootfs->set_create_mode(API::EVolumeCreateMode_READ_ONLY);
            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, staticResources, volumes, {}, {}, 0, 0));

            mountedVolume->set_mode(API::EVolumeMountMode_READ_WRITE);
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, staticResources, volumes, {}, {}, 0, 0)
                , yexception
                , TStringBuilder() << "Static resources list, bound at once to read only and writable volumes: " << Endl
                    << "hash: " << staticResourceDownloadHash << ", ids: " << staticResourceId
            );

            rootfs->set_create_mode(API::EVolumeCreateMode_READ_WRITE);
            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, staticResources, volumes, {}, {}, 0, 0));

            API::TMountedVolume* mountedVolume2 = box->add_volumes();
            mountedVolume2->set_volume_ref("my_volume");
            mountedVolume2->set_mount_point("mount_point_2");
            mountedVolume2->set_mode(API::EVolumeMountMode_READ_ONLY);

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, staticResources, volumes, {}, {}, 0, 0)
                , yexception
                , TStringBuilder() << "Static resources list, bound at once to read only and writable volumes: " << Endl
                    << "hash: " << staticResourceDownloadHash << ", ids: " << staticResourceId
            );

            mountedVolume2->set_mode(API::EVolumeMountMode_READ_WRITE);
            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.boxes(), API::EPodAgentTargetState_ACTIVE, layers, staticResources, volumes, {}, {}, 0, 0));
        }
    };

    TTest test;
    test.DoTest();
}

}

} // NInfra::NPodAgent::NTreeneratorTest
