#include "layer_tree_generator.h"

#include "object_tree.h"
#include "support_functions.h"

#include <infra/pod_agent/libs/behaviour/loaders/behavior3_editor_json_reader.h>
#include <infra/pod_agent/libs/behaviour/loaders/behavior3_template_resolver.h>
#include <infra/pod_agent/libs/pod_agent/trees_generators/proto_hasher.h>

#include <library/cpp/string_utils/base64/base64.h>

#include <util/folder/path.h>
#include <util/string/vector.h>
#include <util/string/split.h>

namespace NInfra::NPodAgent {

TLayerTreeGenerator::TLayersToAdd TLayerTreeGenerator::UpdateSpec(
    const API::TResourceGang& spec
    , API::EPodAgentTargetState podAgentTargetState
    , const double cpuToVcpuFactor
    , ui64 specTimestamp
    , ui32 revision
) {
    TLayerTreeGenerator::TLayersToAdd layers;

    // Add layers to internal state only in ACTIVE or SUSPENDED pod_agent target states
    // WARNING: We do not validate layer specs in other cases (for example in REMOVED target state)
    if (
        podAgentTargetState == API::EPodAgentTargetState_ACTIVE
        || podAgentTargetState == API::EPodAgentTargetState_SUSPENDED
    ) {
        API::ELayerSourceFileStoragePolicy defaultLayerSourceFileStoragePolicy = spec.default_layer_source_file_storage_policy();

        for (const auto& layer : spec.layers()) {
            const TString& layerId = layer.id();

            Y_ENSURE(!layerId.empty(), "One of layers has empty id");
            Y_ENSURE(!layers.Layers_.contains(layerId)
                , "Pod agent spec layers' ids are not unique: "
                << "layer '" << layerId << "' occurs twice"
            );

            try {
                API::ELayerSourceFileStoragePolicy layerSourceFileStoragePolicy = layer.layer_source_file_storage_policy();
                bool removeSourceFileAfterImport = RemoveSourceFileAfterImport(layerSourceFileStoragePolicy, defaultLayerSourceFileStoragePolicy);

                layers.Layers_.insert(
                    {
                        layerId
                        , GetLayerToAdd(
                            removeSourceFileAfterImport
                            , layer
                            , spec.compute_resources()
                            , cpuToVcpuFactor
                            , specTimestamp
                            , revision
                        )
                    }
                );
            } catch (yexception& e) {
                throw e << " at layer '" << layerId << "'";
            }
        }
    }

    return layers;
}

TLayerTreeGenerator::TCacheLayersToAdd TLayerTreeGenerator::UpdateResourceCache(
    const API::TPodAgentResourceCacheSpec& spec
    , API::EPodAgentTargetState podAgentTargetState
    , const API::TComputeResources& computeResources
    , const double cpuToVcpuFactor
) {
    TLayerTreeGenerator::TCacheLayersToAdd cacheLayers;

    // Add cache layers to internal state only in ACTIVE or SUSPENDED pod_agent target states
    // WARNING: We do not validate cache layer specs in other cases (for example in REMOVED target state)
    if (
        podAgentTargetState == API::EPodAgentTargetState_ACTIVE
        || podAgentTargetState == API::EPodAgentTargetState_SUSPENDED
    ) {
        API::ELayerSourceFileStoragePolicy defaultLayerSourceFileStoragePolicy = spec.default_layer_source_file_storage_policy();

        for (const auto& cacheLayer : spec.layers()) {
            const API::TLayer& layer = cacheLayer.layer();

            const TString& cacheLayerId = layer.id();
            const ui32 cacheLayerRevision = cacheLayer.revision();

            Y_ENSURE(!cacheLayerId.empty(), "One of cache layers has empty id");
            Y_ENSURE(!cacheLayers.CacheLayers_.contains(TStatusRepositoryCommon::TCacheObject(cacheLayerId, cacheLayerRevision))
                , "Pod agent resource cache layers' ids and revisions are not unique: "
                << "cache layer '" << cacheLayerId << "' with revision '" << cacheLayerRevision << "' occurs twice"
            );

            try {
                API::ELayerSourceFileStoragePolicy layerSourceFileStoragePolicy = layer.layer_source_file_storage_policy();
                bool removeSourceFileAfterImport = RemoveSourceFileAfterImport(layerSourceFileStoragePolicy, defaultLayerSourceFileStoragePolicy);

                TLayerToAdd layerToAdd = GetLayerToAdd(
                    removeSourceFileAfterImport
                    , layer
                    , computeResources
                    , cpuToVcpuFactor
                    , 0 /* specTimestamp */
                    , cacheLayerRevision
                );

                if (StatusNTickerHolder_->HasCacheLayer(cacheLayerId, cacheLayerRevision)) {
                    Y_ENSURE(layerToAdd.Target_.Meta_.DownloadHash_ == StatusNTickerHolder_->GetCacheLayerHash(cacheLayerId, cacheLayerRevision)
                        , "cache layer '" << cacheLayerId << "' with revision '" << ToString(cacheLayerRevision) << "' hash was changed");
                }

                cacheLayers.CacheLayers_.insert(
                    {
                        TStatusRepositoryCommon::TCacheObject(cacheLayerId, cacheLayerRevision)
                        , layerToAdd
                    }
                );
            } catch (yexception& e) {
                throw e << " at cache layer '" << cacheLayerId << "' with revision '" << cacheLayerRevision << "'";;
            }
        }
    }

    return cacheLayers;
}

void TLayerTreeGenerator::AddAndRemoveCacheLayers(const TCacheLayersToAdd& cacheLayers) {
    // If StatusRepository contains two layers with the same download hash, layer will still be downloaded exactly once
    // but if at least at the moment the layer was removed from the StatusRepository and added there again, it can lead to a re-download
    // so you first need to add new ones, and only then remove the old ones

    // Add new cache layers
    for (const auto& it : cacheLayers.CacheLayers_) {
        const TString& cacheLayerId = it.first.Id_;
        const ui32 cacheLayerRevision = it.first.Revision_;
        const auto& layerData = it.second;

        if (!StatusNTickerHolder_->HasCacheLayer(cacheLayerId, cacheLayerRevision)) {
            StatusNTickerHolder_->AddCacheLayer(layerData.Target_);
        }
    }

    // remove unneeded cache layers
    for (const auto& cacheLayer : StatusNTickerHolder_->GetCacheLayerIdsAndRevisions()) {
        if (!cacheLayers.CacheLayers_.contains(cacheLayer)) {
            StatusNTickerHolder_->RemoveCacheLayer(cacheLayer.Id_, cacheLayer.Revision_);
        }
    }
}

void TLayerTreeGenerator::RemoveLayers(const TLayerTreeGenerator::TLayersToAdd& layers) {
    for (const TString& layerId : StatusNTickerHolder_->GetLayerIds()) {
        if (!layers.Layers_.contains(layerId)) {
            StatusNTickerHolder_->SetLayerTargetRemove(layerId);
        }
    }
}

void TLayerTreeGenerator::AddLayers(const TLayerTreeGenerator::TLayersToAdd& layers) {
    for (const auto& it : layers.Layers_) {
        const TString& layerId = it.first;
        const auto& layerData = it.second;

        if (!StatusNTickerHolder_->HasLayer(layerId)) {
            StatusNTickerHolder_->AddLayerWithTargetCheck(layerData.Target_);
        }

        StatusNTickerHolder_->UpdateLayerTarget(layerData.Target_);
    }
}

TLayerTreeGenerator::TLayerToAdd TLayerTreeGenerator::GetLayerToAdd(
    bool removeSourceFileAfterImport
    , const API::TLayer& layer
    , const API::TComputeResources& computeResources
    , const double cpuToVcpuFactor
    , ui64 specTimestamp
    , ui32 revision
) const {
    const TString layerDownloadHash = GetLayerDownloadHash(layer, removeSourceFileAfterImport);
    const TString fullHash = GetLayerHash(layer, removeSourceFileAfterImport);

    TBehavior3 resolvedTree = LayerTreeTemplate_;
    {
        TMap<TString, TString> replace;

        replace["RESOURCE_DOWNLOAD_QUEUE_LOCK"] = NSupport::RESOURCE_DOWNLOAD_QUEUE_LOCK;
        replace["RESOURCE_VERIFY_QUEUE_LOCK"] = NSupport::RESOURCE_VERIFY_QUEUE_LOCK;

        replace["RESOURCE_GANG_CONTAINER"] = PathHolder_->GetResourceGangMetaContainer();
        replace["RESOURCE_GANG_CONTAINER_LOCK"] = NSupport::RESOURCE_GANG_CONTAINER_LOCK;
        replace.merge(
            NSupport::FillComputeResourcesReplaceMap(
                computeResources
                , "RESOURCE_GANG_"
                , PathHolder_
                , cpuToVcpuFactor
            )
        );

        // TODO(DEPLOY-2825, DEPLOY-2988) use ContainerUser/ContainerGroup
        // https://st.yandex-team.ru/DEPLOY-2825#5f44f3c6558e052f3a2d261d
        replace["LAYER_CONTAINER_USER"] = ""; // ContainerUser_;
        replace["LAYER_CONTAINER_GROUP"] = ""; // ContainerGroup_;
        replace["LAYER_DOWNLOAD_CONTAINER"] = PathHolder_->GetLayerContainerWithNameFromHash(layerDownloadHash, "download");
        replace["LAYER_VERIFY_CONTAINER"] = PathHolder_->GetLayerContainerWithNameFromHash(layerDownloadHash, "verify");

        Y_ENSURE(PathHolder_->HasVirtualDisk(layer.virtual_disk_id_ref()), "Unknown virtual disk ref: '" << layer.virtual_disk_id_ref() << "'");
        const TString place = PathHolder_->GetPlaceFromVirtualDisk(layer.virtual_disk_id_ref());
        replace["LAYER_NAME"] = PathHolder_->GetLayerNameFromHash(layerDownloadHash);
        replace["LAYER_PLACE"] = place;
        replace["LAYER_REMOVE_SOURCE_FILE_AFTER_IMPORT"] = removeSourceFileAfterImport ? "true" : "";

        replace.merge(
            NSupport::FillVerifyResourceReplaceMap(
                "layer"
                , layer.checksum()
            )
        );

        replace["LAYER_DOWNLOAD_CMD"] = GetDownloadCMD(layer);
        replace["NEED_CHECK_DOWNLOAD_PROGRESS"] =
            (layer.download_method_case() == API::TLayer::DownloadMethodCase::kUrl && NSupport::IsWgetOrSkyget(layer.url()))
            || (layer.download_method_case() == API::TLayer::DownloadMethodCase::kSkyGet)
            ? "true"
            : "false"
        ;

        replace["LAYER_DOWNLOAD_CWD"] = PathHolder_->GetLayerDirectoryFromHash(layerDownloadHash, place);
        replace["LAYER_DOWNLOAD_DIRECTORY"] = PathHolder_->GetLayerDownloadDirectoryFromHash(layerDownloadHash, place);
        replace["LAYER_FINAL_PATH"] = PathHolder_->GetFinalLayerPathFromHash(layerDownloadHash, place);

        // Set default aging_time for download container because we do not track its state after success death
        replace["LAYER_DOWNLOAD_AGING_TIME"] = "";
        replace["LAYER_VERIFY_AGING_TIME"] = ToString(NSupport::CONTAINER_AGING_TIME_SECONDS);

        replace["LAYER_DOWNLOAD_HASH"] = layerDownloadHash;

        TBehavior3TemplateResolver().Resolve(resolvedTree, replace);
    }

    return TLayerToAdd{
        TUpdateHolder::TLayerTarget(
            TLayerMeta(
                layer.id()
                , specTimestamp
                , revision
                , layerDownloadHash
                , removeSourceFileAfterImport
            )
            , new TObjectTree(
                Logger_
                , TStatusNTickerHolder::GetLayerTreeId(layerDownloadHash)
                , TBehavior3EditorJsonReader(resolvedTree)
                    .WithPorto(Porto_)
                    .WithPosixWorker(PosixWorker_)
                    .WithLayerStatusRepository(StatusNTickerHolder_->GetLayerStatusRepository())
                    .WithTemplateBTStorage(TemplateBTStorage_)
                    .WithPathHolder(PathHolder_)
                    .BuildRootNode()
                , layerDownloadHash
                , StatusNTickerHolder_->GetUpdateHolder()->GetUpdateHolderTarget()
                , StatusNTickerHolder_->GetLayerStatusRepository()
            )
        )
        , fullHash
        , layer.virtual_disk_id_ref()
    };
}

bool TLayerTreeGenerator::RemoveSourceFileAfterImport(API::ELayerSourceFileStoragePolicy layerSourceFileStoragePolicy, API::ELayerSourceFileStoragePolicy defaultLayerSourceFileStoragePolicy) {
    switch (layerSourceFileStoragePolicy) {
        case API::ELayerSourceFileStoragePolicy::ELayerSourceFileStoragePolicy_NONE: {
            switch (defaultLayerSourceFileStoragePolicy) {
                case API::ELayerSourceFileStoragePolicy::ELayerSourceFileStoragePolicy_NONE:
                    return true;
                case API::ELayerSourceFileStoragePolicy::ELayerSourceFileStoragePolicy_KEEP:
                    return false;
                case API::ELayerSourceFileStoragePolicy::ELayerSourceFileStoragePolicy_REMOVE:
                    return true;
                default:
                    ythrow yexception() << "Unknown default layer source file storage policy";
            }
        }
        case API::ELayerSourceFileStoragePolicy::ELayerSourceFileStoragePolicy_KEEP:
            return false;
        case API::ELayerSourceFileStoragePolicy::ELayerSourceFileStoragePolicy_REMOVE:
            return true;
        default:
            ythrow yexception() << "Unknown layer source file storage policy";
    }
}

TString TLayerTreeGenerator::WrapLayerDownloadCommand(const TString& command) const {
    return NSupport::WrapDownloadCommand(TStringBuilder()
        << "cd downloaded\n"
        << command << "\n"
        << "cd ..\n"
        << R"(if [[ \"$(find downloaded -not -path downloaded | wc -l)\" != \"1\" ]] || [[ \"$(find downloaded -type f | wc -l)\" != \"1\" ]] )" << "\n"
        << "then\n"
            << R"(echo \"expected one file, but found:\")" << "\n"
            << "find downloaded -not -path downloaded\n"
            << "exit 1\n"
        << "fi\n"
        << "mkdir downloaded_result\n"
        << "find downloaded -type f -exec ln -s ../'\\''{}'\\'' downloaded_result/layer_link \\;\n"
    );
}

TString TLayerTreeGenerator::GetDownloadCMD(const API::TLayer& layer) const {
    switch (layer.download_method_case()) {
        case API::TLayer::DownloadMethodCase::kUrl:
            return GetDownloadCMDFromURL(layer.url());
        case API::TLayer::DownloadMethodCase::kSkyGet:
            return GetDownloadCMDFromSkyGet(layer.sky_get());
        case API::TLayer::DownloadMethodCase::DOWNLOAD_METHOD_NOT_SET:
            ythrow yexception() << "Layer download method not set";
    }
}

TString TLayerTreeGenerator::GetDownloadCMDFromURL(const TString& url) const {
    Y_ENSURE(url, "Url not provided");

    if (url.StartsWith(NSupport::HTTP_OR_HTTPS_PREFIX)) {
        Y_ENSURE(NSupport::IsUrl(url), "String '" << url << "' doesn't fit the url pattern");
        return WrapLayerDownloadCommand(TStringBuilder()
            << "wget --no-check-certificate " << url
        );
    }
    if (url.StartsWith(NSupport::RBTORRENT_PREFIX)) {
        Y_ENSURE(NSupport::IsRbtorrent(url), "String '" << url << "' doesn't fit the rbtorrent pattern");
        return WrapLayerDownloadCommand(TStringBuilder()
            << "sky get -p " << url
        );
    }
    if (url.StartsWith(NSupport::LOCAL_PREFIX)) {
        TFsPath filepath(url.substr(NSupport::LOCAL_PREFIX.length()));
        return WrapLayerDownloadCommand(TStringBuilder()
            << "cp " << filepath.RealPath() << " ./"
        );
    }
    if (url.StartsWith(NSupport::RAW_PREFIX)) {
        TString base64FileContent = Base64Encode(url.substr(NSupport::RAW_PREFIX.size()));
        return WrapLayerDownloadCommand(TStringBuilder()
            << "echo -n '\\''" << base64FileContent << "'\\'' | base64 -d > raw_file; tar -czvf layer.tar.xz raw_file; rm raw_file;"
        );
    }

    ythrow yexception() << "Unable to parse URL '" << url << "'";
}

TString TLayerTreeGenerator::GetDownloadCMDFromSkyGet(const API::TSkyGetDownload& skyGetDownload) const {
    return WrapLayerDownloadCommand(NSupport::GetSkyGetDownloadCommand(skyGetDownload, ""));
}

} // namespace NInfra::NPodAgent
