#include "layer_tree_generator.h"
#include "test_canon.h"
#include "test_functions.h"

#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/stream/file.h>
#include <util/system/fs.h>
#include <util/system/shellcommand.h>

namespace NInfra::NPodAgent::NTreeGeneratorTest {

const TString LAYER_DOWNLOAD_COMMAND_PREFIX = "bash -e -c 'rm -rf downloaded; rm -rf downloaded_result; mkdir downloaded; cd downloaded\n";
const TString LAYER_DOWNLOAD_COMMAND_SUFFIX = TStringBuilder()
    << "cd ..\n"
    << R"(if [[ \"$(find downloaded -not -path downloaded | wc -l)\" != \"1\" ]] || [[ \"$(find downloaded -type f | wc -l)\" != \"1\" ]] )" << "\n"
    << "then\n"
        << R"(echo \"expected one file, but found:\")" << "\n"
        << "find downloaded -not -path downloaded\n"
        << "exit 1\n"
    << "fi\n"
    << "mkdir downloaded_result\n"
    << "find downloaded -type f -exec ln -s ../'\\''{}'\\'' downloaded_result/layer_link \\;\n"
    << "'"
;

class TMockLayerTreeGenerator : public TLayerTreeGenerator {
public:
    TMockLayerTreeGenerator(
        TLogger& logger
        , TPathHolderPtr pathHolder
        , const TBehavior3& layerTreeTemplate
        , TAsyncPortoClientPtr porto
        , TPosixWorkerPtr posixWorker
        , TStatusNTickerHolderPtr statusNTickerHolder
        , TTemplateBTStoragePtr templateBTStorage
        , const TString& containerUser
        , const TString& containerGroup
    ): TLayerTreeGenerator(
            logger
            , pathHolder
            , layerTreeTemplate
            , porto
            , posixWorker
            , statusNTickerHolder
            , templateBTStorage
            , containerUser
            , containerGroup
        )
    {}

    TString GetDownloadCMD(const API::TLayer& layer) const {
        return TLayerTreeGenerator::GetDownloadCMD(layer);
    }
    TString GetDownloadCMDFromURL(const TString& url) const {
        return TLayerTreeGenerator::GetDownloadCMDFromURL(url);
    }
};

using TMockLayerTreeGeneratorPtr = TSimpleSharedPtr<TMockLayerTreeGenerator>;

class ITestLayerTreeGeneratorCanon: public ITestTreeGeneratorCanon {
public:
    ITestLayerTreeGeneratorCanon() : ITestTreeGeneratorCanon()
    {}

protected:
    void CreateWorker() final {
        TMap<TString, TString> properties;
        properties["object_id_or_hash"] = "<RSLV:LAYER_DOWNLOAD_HASH>";
        properties["state"] = "ELayerState_READY";
        properties["object_type"] = "layer";
        Worker_ = new TMockLayerTreeGenerator(
            logger
            , PathHolder_
            , GetTestTemplateTree(
                properties
                , "FeedbackObjectState"
                , "FeedbackObjectState(ELayerState_READY)"
            )
            , new TAsyncPortoClient(new TMockPortoClient(), new TFakeThreadPool())
            , new TPosixWorker(new TFakeThreadPool())
            , StatusNTickerHolder_
            , nullptr
            , "ContainerUser"
            , "ContainerGroup"
        );
    }

    TLayerTreeGenerator::TLayersToAdd UpdateSpec(
        const API::TResourceGang& spec
        , API::EPodAgentTargetState podAgentTargetState
        , ui64 specTimestamp
        , ui32 revision
    ) {
        TLayerTreeGenerator::TLayersToAdd newSpec = Worker_->UpdateSpec(
            spec
            , podAgentTargetState
            , 1.0 /* CpuToVcpuFactor */
            , specTimestamp
            , revision
        );
        Worker_->RemoveLayers(newSpec);
        Worker_->AddLayers(newSpec);
        return newSpec;
    }

    void UpdateResourceCache(
        const API::TPodAgentResourceCacheSpec& spec
        , API::EPodAgentTargetState podAgentTargetState
    ) {
        auto newSpec = Worker_->UpdateResourceCache(
            spec
            , podAgentTargetState
            , API::TComputeResources()
            , 1.0 /* CpuToVcpuFactor */
        );
        Worker_->AddAndRemoveCacheLayers(newSpec);
    }

protected:
    TMockLayerTreeGeneratorPtr Worker_;
};

Y_UNIT_TEST_SUITE(LayerResourceGangWorkerSuite) {

Y_UNIT_TEST(TestRemoveSourceFileAfterImport) {
    API::ELayerSourceFileStoragePolicy nonePolicy = API::ELayerSourceFileStoragePolicy::ELayerSourceFileStoragePolicy_NONE;
    API::ELayerSourceFileStoragePolicy keepPolicy = API::ELayerSourceFileStoragePolicy::ELayerSourceFileStoragePolicy_KEEP;
    API::ELayerSourceFileStoragePolicy removePolicy = API::ELayerSourceFileStoragePolicy::ELayerSourceFileStoragePolicy_REMOVE;

    UNIT_ASSERT_EQUAL(TLayerTreeGenerator::RemoveSourceFileAfterImport(nonePolicy, nonePolicy), true);
    UNIT_ASSERT_EQUAL(TLayerTreeGenerator::RemoveSourceFileAfterImport(nonePolicy, keepPolicy), false);
    UNIT_ASSERT_EQUAL(TLayerTreeGenerator::RemoveSourceFileAfterImport(nonePolicy, removePolicy), true);

    UNIT_ASSERT_EQUAL(TLayerTreeGenerator::RemoveSourceFileAfterImport(keepPolicy, nonePolicy), false);
    UNIT_ASSERT_EQUAL(TLayerTreeGenerator::RemoveSourceFileAfterImport(keepPolicy, keepPolicy), false);
    UNIT_ASSERT_EQUAL(TLayerTreeGenerator::RemoveSourceFileAfterImport(keepPolicy, removePolicy), false);

    UNIT_ASSERT_EQUAL(TLayerTreeGenerator::RemoveSourceFileAfterImport(removePolicy, nonePolicy), true);
    UNIT_ASSERT_EQUAL(TLayerTreeGenerator::RemoveSourceFileAfterImport(removePolicy, keepPolicy), true);
    UNIT_ASSERT_EQUAL(TLayerTreeGenerator::RemoveSourceFileAfterImport(removePolicy, removePolicy), true);
}

Y_UNIT_TEST(TestSimpleLayerTree) {
    class TTest : public ITestLayerTreeGeneratorCanon {
    public:
        TTest() : ITestLayerTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TResourceGang spec;
            API::TLayer* layer = spec.add_layers();
            layer->set_id("my_layer");
            layer->set_url("raw:some_data");
            layer->set_checksum("MD5:3ea05169a95365cbf9b4ff4688cd8945");
            layer->set_virtual_disk_id_ref("virtual_disk");

            auto layerCopy = spec.add_layers();
            layerCopy->CopyFrom(*layer);

            UNIT_ASSERT_EXCEPTION_CONTAINS(UpdateSpec(spec, API::EPodAgentTargetState_ACTIVE, 0, 0), yexception, "ids are not unique");
            *layerCopy->mutable_id() += "_other";
            auto layersToAdd = UpdateSpec(spec, API::EPodAgentTargetState_ACTIVE, 0, 0);
            *layerCopy->mutable_id() = "";
            UNIT_ASSERT_EXCEPTION(UpdateSpec(spec, API::EPodAgentTargetState_ACTIVE, 0, 0), yexception);

            *layerCopy->mutable_id() = "my_layer_other";
            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec, API::EPodAgentTargetState_ACTIVE, 0, 0));
            layer->set_virtual_disk_id_ref("bad_virtual_disk");
            UNIT_ASSERT_EXCEPTION_CONTAINS(UpdateSpec(spec, API::EPodAgentTargetState_ACTIVE, 0, 0), yexception, "Unknown virtual disk ref");

            Sleep(TDuration::MilliSeconds(500));
            UNIT_ASSERT_EQUAL(API::ELayerState_READY, LayerStatusRepository_->GetObjectStatus("my_layer").state());
            UNIT_ASSERT(layersToAdd.Layers_.at("my_layer").Target_.Tree_->GetUseLongTickPeriod());

        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestSimpleCacheLayerTree) {
    class TTest : public ITestLayerTreeGeneratorCanon {
    public:
        TTest() : ITestLayerTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TPodAgentResourceCacheSpec spec;
            API::TCacheLayer* cacheLayer = spec.add_layers();
            cacheLayer->set_revision(1);
            API::TLayer* layer = cacheLayer->mutable_layer();
            layer->set_id("my_layer");
            layer->set_url("raw:some_data");
            layer->set_checksum("MD5:3ea05169a95365cbf9b4ff4688cd8945");

            auto layerCopy = spec.add_layers();
            layerCopy->CopyFrom(*cacheLayer);

            UNIT_ASSERT_EXCEPTION_CONTAINS(UpdateResourceCache(spec, API::EPodAgentTargetState_ACTIVE), yexception, "ids and revisions are not unique");
            *layerCopy->mutable_layer()->mutable_id() += "_other";
            UNIT_ASSERT_NO_EXCEPTION(UpdateResourceCache(spec, API::EPodAgentTargetState_ACTIVE));
            *layerCopy->mutable_layer()->mutable_id() = "";
            UNIT_ASSERT_EXCEPTION(UpdateResourceCache(spec, API::EPodAgentTargetState_ACTIVE), yexception);
            *layerCopy->mutable_layer()->mutable_id() = "my_layer_other";
            UNIT_ASSERT_NO_EXCEPTION(UpdateResourceCache(spec, API::EPodAgentTargetState_ACTIVE));
            *layerCopy->mutable_layer()->mutable_url() += "_other";
            UNIT_ASSERT_EXCEPTION_CONTAINS(UpdateResourceCache(spec, API::EPodAgentTargetState_ACTIVE), yexception, "hash was changed");

            Sleep(TDuration::MilliSeconds(500));
            UNIT_ASSERT_EQUAL(API::ELayerState_READY, LayerStatusRepository_->GetCacheObjectStatus("my_layer", 1).state());
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveLayerTree) {
    class TTest : public ITestLayerTreeGeneratorCanon {
    public:
        TTest() : ITestLayerTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TResourceGang spec;
            API::TLayer* layer = spec.add_layers();
            layer->set_id("my_layer");
            layer->set_url("raw:some_data");
            layer->set_checksum("MD5:3ea05169a95365cbf9b4ff4688cd8945");

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec, API::EPodAgentTargetState_ACTIVE, 0, 0));
            UNIT_ASSERT(LayerStatusRepository_->HasObject("my_layer"));
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->LayerHasTarget("my_layer"));

            layer->set_id("other_layer");

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec, API::EPodAgentTargetState_ACTIVE, 1, 1));
            UNIT_ASSERT(LayerStatusRepository_->HasObject("my_layer"));
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->LayerHasTarget("my_layer"));
            UNIT_ASSERT(UpdateHolder_->GetAndRemoveLayerTarget("my_layer").TargetRemove_);
            UNIT_ASSERT(LayerStatusRepository_->HasObject("other_layer"));
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveCacheLayerTree) {
    class TTest : public ITestLayerTreeGeneratorCanon {
    public:
        TTest() : ITestLayerTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TPodAgentResourceCacheSpec spec;
            API::TCacheLayer* cacheLayer = spec.add_layers();
            cacheLayer->set_revision(1);
            API::TLayer* layer = cacheLayer->mutable_layer();
            layer->set_id("my_layer");
            layer->set_url("raw:some_data");
            layer->set_checksum("MD5:3ea05169a95365cbf9b4ff4688cd8945");

            UNIT_ASSERT_NO_EXCEPTION(UpdateResourceCache(spec, API::EPodAgentTargetState_ACTIVE));
            UNIT_ASSERT(LayerStatusRepository_->HasCacheObject("my_layer", 1));

            layer->set_id("other_layer");

            UNIT_ASSERT_NO_EXCEPTION(UpdateResourceCache(spec, API::EPodAgentTargetState_ACTIVE));
            UNIT_ASSERT(!LayerStatusRepository_->HasCacheObject("my_layer", 1));
            UNIT_ASSERT(LayerStatusRepository_->HasCacheObject("other_layer", 1));
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestChangeTarget) {
    class TTest : public ITestLayerTreeGeneratorCanon {
    public:
        TTest() : ITestLayerTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TResourceGang spec;
            API::TLayer* layer = spec.add_layers();
            layer->set_id("my_layer");
            layer->set_url("raw:some_data");
            layer->set_checksum("MD5:3ea05169a95365cbf9b4ff4688cd8945");

            for (size_t i = 0; i < 3; ++i) {
                UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec, API::EPodAgentTargetState_ACTIVE, i + 1, i + 2));
                UNIT_ASSERT(LayerStatusRepository_->HasObject("my_layer"));
                UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->LayerHasTarget("my_layer"));
                UNIT_ASSERT(LayerStatusRepository_->GetObjectStatus("my_layer").spec_timestamp() == i + 1);
                UNIT_ASSERT(LayerStatusRepository_->GetObjectStatus("my_layer").revision() == i + 2);
            }

            // change layer
            layer->set_url("raw:some_other_data");

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec, API::EPodAgentTargetState_ACTIVE, 4, 5));

            UNIT_ASSERT(LayerStatusRepository_->HasObject("my_layer"));
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->LayerHasTarget("my_layer"));
            UNIT_ASSERT(LayerStatusRepository_->GetObjectStatus("my_layer").spec_timestamp() == 3);
            UNIT_ASSERT(LayerStatusRepository_->GetObjectStatus("my_layer").revision() == 4);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestPodAgentTargetStateWithLayerTree) {
    class TTest : public ITestLayerTreeGeneratorCanon {
    public:
        TTest() : ITestLayerTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TResourceGang spec;
            API::TLayer* layer = spec.add_layers();
            layer->set_id("my_layer");
            layer->set_url("raw:some_data");
            layer->set_checksum("MD5:3ea05169a95365cbf9b4ff4688cd8945");

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec, API::EPodAgentTargetState_ACTIVE, 0, 0));
            UNIT_ASSERT(LayerStatusRepository_->HasObject("my_layer"));
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->LayerHasTarget("my_layer"));

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec, API::EPodAgentTargetState_SUSPENDED, 1, 1));
            UNIT_ASSERT(LayerStatusRepository_->HasObject("my_layer"));
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->LayerHasTarget("my_layer"));

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec, API::EPodAgentTargetState_REMOVED, 2, 2));
            UNIT_ASSERT(LayerStatusRepository_->HasObject("my_layer"));
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->LayerHasTarget("my_layer"));
            UNIT_ASSERT(UpdateHolder_->GetAndRemoveLayerTarget("my_layer").TargetRemove_);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestPodAgentTargetStateWithCacheLayerTree) {
    class TTest : public ITestLayerTreeGeneratorCanon {
    public:
        TTest() : ITestLayerTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TPodAgentResourceCacheSpec spec;
            API::TCacheLayer* cacheLayer = spec.add_layers();
            cacheLayer->set_revision(1);
            API::TLayer* layer = cacheLayer->mutable_layer();
            layer->set_id("my_layer");
            layer->set_url("raw:some_data");
            layer->set_checksum("MD5:3ea05169a95365cbf9b4ff4688cd8945");

            UNIT_ASSERT_NO_EXCEPTION(UpdateResourceCache(spec, API::EPodAgentTargetState_ACTIVE));
            UNIT_ASSERT(LayerStatusRepository_->HasCacheObject("my_layer", 1));

            UNIT_ASSERT_NO_EXCEPTION(UpdateResourceCache(spec, API::EPodAgentTargetState_SUSPENDED));
            UNIT_ASSERT(LayerStatusRepository_->HasCacheObject("my_layer", 1));

            UNIT_ASSERT_NO_EXCEPTION(UpdateResourceCache(spec, API::EPodAgentTargetState_REMOVED));
            UNIT_ASSERT(!LayerStatusRepository_->HasCacheObject("my_layer", 1));
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestGetDownloadCMDFromURL) {
    class TTest : public ITestLayerTreeGeneratorCanon {
    public:
        TTest() : ITestLayerTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            TFsPath("layer.tar.gz").Touch();
            const TVector<std::pair<TString, TString>> validItems = {
                {
                    "https://proxy.sandbox.yandex-team.ru/586812773"
                    , TStringBuilder()
                        << LAYER_DOWNLOAD_COMMAND_PREFIX
                        << "wget --no-check-certificate https://proxy.sandbox.yandex-team.ru/586812773\n"
                        << LAYER_DOWNLOAD_COMMAND_SUFFIX
                },
                {
                    "rbtorrent:73b5ea60d641225a71571d38dd8975a4fdf86825"
                    , TStringBuilder()
                        << LAYER_DOWNLOAD_COMMAND_PREFIX
                        << "sky get -p rbtorrent:73b5ea60d641225a71571d38dd8975a4fdf86825\n"
                        << LAYER_DOWNLOAD_COMMAND_SUFFIX
                },
                {
                    "local:layer.tar.gz"
                    , TStringBuilder()
                        << LAYER_DOWNLOAD_COMMAND_PREFIX
                        << "cp " << NFs::CurrentWorkingDirectory() << "/layer.tar.gz ./\n"
                        << LAYER_DOWNLOAD_COMMAND_SUFFIX
                },
                {
                    "raw: You think you can get away, Rose? You stood your ground on that little speck called Earth, but you’re on our world now…"
                    , TStringBuilder()
                        << LAYER_DOWNLOAD_COMMAND_PREFIX
                        << "echo -n '\\''IFlvdSB0aGluayB5b3UgY2FuIGdldCBhd2F5LCBSb3NlPyBZb3Ugc3Rvb2QgeW91ciBncm91bmQgb24gdGhhdCBsaXR0bGUgc3BlY2sgY2FsbGVkIEVhcnRoLCBidXQgeW914oCZcmUgb24gb3VyIHdvcmxkIG5vd+KApg=='\\''"
                        << " | base64 -d > raw_file; tar -czvf layer.tar.xz raw_file; rm raw_file;\n"
                        << LAYER_DOWNLOAD_COMMAND_SUFFIX
                }
            };

            const TVector<TString> invalidUrls = {
                "htt\\ps://proxy.sandbox.yandex-team.ru/586812773"
                , "htts//proxy.sandbox.yandex-team.ru/586812773"
                , "https://proxy"
                , "http:///proxy.sandbox.yandex-team.ru/586812773"
                , "https:73b5ea60d641225a71571d38dd8975a4fdf86825"
                , "https://proxy.sandbox.yandex-team.ru/586812773 \"; mkdir directory; \""
                , "https:proxy.sandbox.yandex-team.ru/586812773"

                , "rtorrent:73b5ea60d641225a71571d38dd8975a4fdf86825"
                , "Rbtorrent:73b5ea60d641225a71571d38dd8975a4fdf86825"
                , "rbtorrent:73b5ea60d641225a71'571d38dd8975a4fdf86825"
                , "rbtorrent:73b5ea60d641225a7\"; pwd;\"1571d38dd8975a4fdf86825"
                , "rbtorrent:73b5Ёa60d641225a71571d38dd8975a4f"

                , "local:dir/layer.tar.gz"
                , "locl:layer.tar.gz"
                , "local:dir'layer.tar.gz"
                , "local:file;rm -rf /*"

                , "ruw: some_data"
                , "Raw: some_data"
                , "raW: some_data"
                , "raw : some_data"
                , ":raw: some_data"
                , "r\aw: some_data"

                , ""
            };

            for (auto& item : validItems) {
                auto command = Worker_->GetDownloadCMDFromURL(item.first);
                UNIT_ASSERT_EQUAL_C(command, item.second, "Commands are different, expected '" << item.second << "', got '" << command << "'");
            }

            for (auto& invalidUrl : invalidUrls) {
                UNIT_ASSERT_EXCEPTION_C(Worker_->GetDownloadCMDFromURL(invalidUrl), yexception, invalidUrl);
            }
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestGetDownloadCMDFromSkyGet) {
    class TTest : public ITestLayerTreeGeneratorCanon {
    public:
        TTest() : ITestLayerTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TLayer layer;

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                Worker_->GetDownloadCMD(layer)
                , yexception
                , "Layer download method not set"
            );

            layer.mutable_sky_get()->set_resid("rbtorrent:4cfdc2e157eefe6facb983b1d557b3a1");

            auto command = Worker_->GetDownloadCMD(layer);
            UNIT_ASSERT_EQUAL_C(
                command
                , TStringBuilder()
                    << LAYER_DOWNLOAD_COMMAND_PREFIX
                    << "sky get -p rbtorrent:4cfdc2e157eefe6facb983b1d557b3a1\n"
                    << LAYER_DOWNLOAD_COMMAND_SUFFIX
                , command
            );
        }
    };

    TTest test;
    test.DoTest();
}

}

} // NInfra::NPodAgent::NTreeGeneratorTest
