#include "object_tree.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/mock_node.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>

#include <infra/pod_agent/libs/pod_agent/status_repository/box_status_repository.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/layer_status_repository.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/static_resource_status_repository.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/volume_status_repository.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/workload_status_repository.h>

#include <infra/pod_agent/libs/pod_agent/update_holder/test_lib/test_functions.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NObjectTreeTest {

static TLogger logger({});

Y_UNIT_TEST_SUITE(ObjectTreeTestSuite) {

TTreePtr GetTestObjectTree(
    const TString& objectIdOrHash
    , TUpdateHolderTargetPtr updateHolderTarget
    , TStatusRepositoryCommonPtr statusRepositoryCommon
) {
    return new TObjectTree(
        logger
        , "my_tree"
        , new TMockNode({1, "empty_tree_root"}, TTickResult(ENodeStatus::SUCCESS))
        , objectIdOrHash
        , updateHolderTarget
        , statusRepositoryCommon
    );
}

Y_UNIT_TEST(TestWithBoxStatusRepository) {
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    TBoxStatusRepositoryPtr boxStatusRepository = new TBoxStatusRepository();
    const TString id = "my_box";

    boxStatusRepository->AddObject(NObjectMetaTestLib::CreateBoxMetaSimple(id));

    TTreePtr tree = GetTestObjectTree(
        id
        , updateHolder->GetUpdateHolderTarget()
        , boxStatusRepository
    );

    UNIT_ASSERT(!tree->GetUseLongTickPeriod());

    boxStatusRepository->UpdateObjectState(id, API::EBoxState_READY);
    UNIT_ASSERT(tree->GetUseLongTickPeriod());

    updateHolder->SetBoxTarget(NObjectTargetTestLib::CreateBoxTargetSimple(id));
    UNIT_ASSERT(!tree->GetUseLongTickPeriod());
}

Y_UNIT_TEST(TestWithLayerStatusRepository) {
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    TLayerStatusRepositoryPtr layerStatusRepository = new TLayerStatusRepository();
    const TString id = "my_layer";
    const TString downloadHash = "my_layer_download_hash";

    layerStatusRepository->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, downloadHash));

    TTreePtr tree = GetTestObjectTree(
        downloadHash
        , updateHolder->GetUpdateHolderTarget()
        , layerStatusRepository
    );

    UNIT_ASSERT(!tree->GetUseLongTickPeriod());

    layerStatusRepository->UpdateObjectState(downloadHash, API::ELayerState_READY);
    UNIT_ASSERT(tree->GetUseLongTickPeriod());

    updateHolder->SetLayerTarget(NObjectTargetTestLib::CreateLayerTargetSimple(id, downloadHash));
    UNIT_ASSERT(tree->GetUseLongTickPeriod());
}

Y_UNIT_TEST(TestWithStaticResourceStatusRepository) {
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    TStaticResourceStatusRepositoryPtr staticResourceStatusRepository = new TStaticResourceStatusRepository();
    const TString id = "my_static_resource";
    const TString downloadHash = "my_static_resource_download_hash";

    staticResourceStatusRepository->AddObject(NObjectMetaTestLib::CreateStaticResourceMetaSimple(id, downloadHash));

    TTreePtr tree = GetTestObjectTree(
        downloadHash
        , updateHolder->GetUpdateHolderTarget()
        , staticResourceStatusRepository
    );

    UNIT_ASSERT(!tree->GetUseLongTickPeriod());

    staticResourceStatusRepository->UpdateObjectState(downloadHash, API::EStaticResourceState_READY);
    UNIT_ASSERT(tree->GetUseLongTickPeriod());

    updateHolder->SetStaticResourceTarget(NObjectTargetTestLib::CreateStaticResourceTargetSimple(id, downloadHash));
    UNIT_ASSERT(tree->GetUseLongTickPeriod());
}

Y_UNIT_TEST(TestWithVolumeStatusRepository) {
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    TVolumeStatusRepositoryPtr volumeStatusRepository = new TVolumeStatusRepository();
    const TString id = "my_volume";

    volumeStatusRepository->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple(id));

    TTreePtr tree = GetTestObjectTree(
        id
        , updateHolder->GetUpdateHolderTarget()
        , volumeStatusRepository
    );

    UNIT_ASSERT(!tree->GetUseLongTickPeriod());

    volumeStatusRepository->UpdateObjectState(id, API::EVolumeState_READY);
    UNIT_ASSERT(tree->GetUseLongTickPeriod());

    updateHolder->SetVolumeTarget(NObjectTargetTestLib::CreatePersistentVolumeTargetSimple(id));
    UNIT_ASSERT(!tree->GetUseLongTickPeriod());
}

Y_UNIT_TEST(TestWithWorkloadStatusRepository) {
    TUpdateHolderPtr updateHolder = new TUpdateHolder();
    TWorkloadStatusRepositoryPtr workloadStatusRepository = new TWorkloadStatusRepository();
    const TString id = "my_workload";

    workloadStatusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

    TTreePtr tree = GetTestObjectTree(
        id
        , updateHolder->GetUpdateHolderTarget()
        , workloadStatusRepository
    );

    UNIT_ASSERT(!tree->GetUseLongTickPeriod());

    workloadStatusRepository->UpdateObjectState(id, API::EWorkloadState_ACTIVE);
    UNIT_ASSERT(tree->GetUseLongTickPeriod());

    updateHolder->SetWorkloadTarget(NObjectTargetTestLib::CreateWorkloadTargetSimple(id));
    UNIT_ASSERT(!tree->GetUseLongTickPeriod());
}

}

} // namespace NInfra::NPodAgent::NObjectTreeTest
