#include "proto_hasher.h"
#include "support_functions.h"

#include <library/cpp/digest/md5/md5.h>

#include <google/protobuf/io/coded_stream.h>
#include <google/protobuf/io/zero_copy_stream_impl_lite.h>

#include <util/generic/buffer.h>
#include <util/generic/yexception.h>
#include <util/generic/vector.h>
#include <util/string/join.h>

namespace NInfra::NPodAgent {

namespace {

TString GetSecretStringFromFiles(
    const API::TResource& resource
    , const NSecret::TSecretMap& secretMap
    , const bool autoDecodeBase64Secrets
) {
    TString secretString;
    for (auto& file : resource.files().files()) {
        if (file.has_secret_data()) {
            secretString += NSecret::GetSecretValue(file.secret_data(), secretMap, autoDecodeBase64Secrets) + "#";
        } else if (file.has_multi_secret_data()) {
            secretString += NSecret::GetMultiSecretFileContent(file.multi_secret_data(), secretMap, autoDecodeBase64Secrets) + "#";
        }
    }
    return secretString;
}

} // namespace

TString GetLayerDownloadHash(const API::TLayer& layer, bool removeSourceFileAfterImport) {
    TString stringToCalculateMd5;

    switch (layer.download_method_case()) {
        case API::TLayer::DownloadMethodCase::kUrl:
            stringToCalculateMd5 = layer.url() + layer.checksum() + layer.virtual_disk_id_ref();
            break;
        case API::TLayer::DownloadMethodCase::kSkyGet:
            stringToCalculateMd5 = GetHashFromProto(layer.sky_get()) + "#" + layer.checksum() + "#" + layer.virtual_disk_id_ref();
            break;
        case API::TLayer::DownloadMethodCase::DOWNLOAD_METHOD_NOT_SET:
            ythrow yexception() << "Layer download method not set";
    }

    // We dont use removeSourceFileAfterImport in hash when removeSourceFileAfterImport == true - for migration without layers redownload 
    if (!removeSourceFileAfterImport) {
        stringToCalculateMd5 = stringToCalculateMd5 + ToString(removeSourceFileAfterImport);
    }

    return MD5::Calc(stringToCalculateMd5);
}

TString GetStaticResourceDownloadHash(
    const API::TResource& resource
    , const NSecret::TSecretMap& secretMap
    , const bool autoDecodeBase64Secrets
) {
    TString stringToCalculateMd5;

    TString checksum = resource.verification().checksum();
    // If only disabled flag is turned off and checksum is not EMPTY:, hash will be changed.
    // If disabled flag is true, changes of checksum will not change hash.
    if (resource.verification().disabled()) {
        checksum = "EMPTY:";
    }

    switch (resource.download_method_case()) {
        case API::TResource::DownloadMethodCase::kFiles:
            stringToCalculateMd5 = JoinSeq("#", {
                GetHashFromProto(resource.files())
                , GetSecretStringFromFiles(resource, secretMap, autoDecodeBase64Secrets)
                , checksum
                , resource.virtual_disk_id_ref()
            });
            break;
        case API::TResource::DownloadMethodCase::kUrl:
            stringToCalculateMd5 = JoinSeq("#", {resource.url(), checksum, resource.virtual_disk_id_ref()});
            break;
        case API::TResource::DownloadMethodCase::kSkyGet:
            stringToCalculateMd5 = JoinSeq("#", {GetHashFromProto(resource.sky_get()), checksum, resource.virtual_disk_id_ref()});
            break;
        case API::TResource::DownloadMethodCase::DOWNLOAD_METHOD_NOT_SET:
            ythrow yexception() << "Resource download method not set";
    }

    // We don't use resource access_permissions in download hash when access permissions is UNMODIFIED - for migration without resource redownload
    if (resource.access_permissions() != API::EResourceAccessPermissions::EResourceAccessPermissions_UNMODIFIED) {
        stringToCalculateMd5 = stringToCalculateMd5 + "#" + resource.access_permissions();
    }

    // We don't use resource group_id in download hash when group_id is empty - for migration without resource redownload
    if (resource.has_group_id()) {
        stringToCalculateMd5 = stringToCalculateMd5 + "#" + ToString(resource.group_id().value());
    }

    return MD5::Calc(stringToCalculateMd5);
}

TString GetLayerHash(const API::TLayer& layer, bool removeSourceFileAfterImport) {
    // We dont use removeSourceFileAfterImport in hash when removeSourceFileAfterImport == true - for migration without layers redownload
    if (!removeSourceFileAfterImport) {
        return MD5::Calc(GetHashFromProto(layer) + ToString(removeSourceFileAfterImport));
    }

    return GetHashFromProto(layer);
}

TString GetStaticResourceHash(
    const API::TResource& resource
    , const NSecret::TSecretMap& secretMap
    , const bool autoDecodeBase64Secrets
) {
    auto resourceClone = resource;
    // If only disabled flag is turned off and checksum is not EMPTY:, hash will be changed.
    // If disabled flag is true, changes of checksum will not change hash.
    if (resourceClone.verification().disabled()) {
        resourceClone.mutable_verification()->set_checksum("EMPTY:");
        resourceClone.mutable_verification()->clear_disabled();
    }

    if (resourceClone.has_files()) {
        return MD5::Calc(GetHashFromProto(resourceClone) + GetSecretStringFromFiles(resourceClone, secretMap, autoDecodeBase64Secrets));
    } else { // Valid for url and sky_get
        return GetHashFromProto(resourceClone);
    }
}

TString GetVolumeHash(
    const API::TVolume& volume
    , const TLayerTreeGenerator::TLayersToAdd& layers
    , const TStaticResourceTreeGenerator::TStaticResourcesToAdd& staticResources
) {
    TVector<TString> layerHashes;
    for (const TString& layerId: volume.generic().layer_refs()) {
        auto it = layers.Layers_.find(layerId);
        Y_ENSURE(it != layers.Layers_.end(), "volume " << volume.id() << " refers to nonexistent layer with id " << layerId);
        layerHashes.push_back(it->second.FullHash_);
    }

    TVector<TString> staticResourceHashes;
    for (auto& staticResource: volume.static_resources()) {
        auto staticResourceId = staticResource.resource_ref();
        auto it = staticResources.StaticResources_.find(staticResourceId);
        Y_ENSURE(staticResources.StaticResources_.end() != it, "volume " << volume.id() << " refers to nonexistent static resource with id " << staticResourceId);
        staticResourceHashes.push_back(it->second.FullHash_);
    }

    return MD5::Calc(
        GetHashFromProto(volume)
        + JoinSeq("", layerHashes)
        + JoinSeq("", staticResourceHashes)
    );
}

TString GetBoxHash(
    const API::TBox& box
    , const TLayerTreeGenerator::TLayersToAdd& layers
    , const TStaticResourceTreeGenerator::TStaticResourcesToAdd& staticResources
    , const TVolumeTreeGenerator::TVolumesToAdd& volumes
    , const TMap<TString, TString>& systemEnv
    , const NSecret::TSecretMap& secretMap
    , const TMaybe<TString>& ytBindPath
    , const TMaybe<TString>& baseSearchBindPath
    , const TString& Ip6Address
    , bool useEnvSecret
    , const bool autoDecodeBase64Secrets
) {
    TVector<TString> layerHashes, staticResourceHashes, volumeHashes;
    for (auto& layerId: box.rootfs().layer_refs()) {
        auto it = layers.Layers_.find(layerId);
        Y_ENSURE(it != layers.Layers_.end(), "box " << box.id() << " rootfs volume refers to nonexistent layer with id " << layerId);
        layerHashes.push_back(it->second.FullHash_);
    }

    for (auto& staticResource: box.static_resources()) {
        auto staticResourceId = staticResource.resource_ref();
        auto it = staticResources.StaticResources_.find(staticResourceId);
        Y_ENSURE(staticResources.StaticResources_.end() != it, "box " << box.id() << " rootfs volume refers to nonexistent static resource with id " << staticResourceId);
        staticResourceHashes.push_back(it->second.FullHash_);
    }

    for (auto& mountedVolume : box.volumes()) {
        if (mountedVolume.volume_type_case() == API::TMountedVolume::VolumeTypeCase::kVolumeRef) {
            auto it = volumes.Volumes_.find(mountedVolume.volume_ref());
            Y_ENSURE(it != volumes.Volumes_.end(), "box with id " << box.id() << " refers to nonexistent volume with id " << mountedVolume.volume_ref());
            volumeHashes.push_back(it->second.Target_.Hash_);
        }
    }

    NSupport::TEnvironmentList envList = NSupport::GetEnvironmentList(
        box.env()
        , secretMap
        , systemEnv
        , useEnvSecret
        , autoDecodeBase64Secrets
    );

    return MD5::Calc(
        GetHashFromProto(box)
        + JoinSeq("", layerHashes)
        + JoinSeq("", staticResourceHashes)
        + JoinSeq("", volumeHashes)
        + envList.PublicEnv_
        + envList.SecretEnv_
        + ytBindPath.GetOrElse("")
        + baseSearchBindPath.GetOrElse("")
        + Ip6Address
    );
}

TString GetWorkloadHash(
    const API::TWorkload& workload
    , const NSecret::TSecretMap& secretMap
    , const TString& boxHash
    , bool useEnvSecret
    , const bool autoDecodeBase64Secrets
) {
    NSupport::TEnvironmentList envList = NSupport::GetEnvironmentList(
        workload.env()
        , secretMap
        , {} /* systemEnv in box hash */
        , useEnvSecret
        , autoDecodeBase64Secrets
    );

    return MD5::Calc(
        GetHashFromProto(workload)
        + boxHash
        + envList.PublicEnv_
        + envList.SecretEnv_
    );
}

TString GetHashFromProto(const NProtoBuf::MessageLite& message) {
    // magic from https://a.yandex-team.ru/arc/trunk/arcadia/contrib/libs/tf/tensorflow/core/lib/strings/proto_serialization.cc?rev=3997277#L21-31

    Y_ENSURE(message.ByteSizeLong() <= static_cast<size_t>(INT_MAX), "Fail to serialize: protobuf message too big " << message.ByteSizeLong());
    const i32 size = static_cast<i32>(message.ByteSizeLong());
    TString result = TString(size, '\0');
    NProtoBuf::protobuf::io::ArrayOutputStream arrayStream(&*result.begin(), size);
    NProtoBuf::protobuf::io::CodedOutputStream outputStream(&arrayStream);
    outputStream.SetSerializationDeterministic(true);
    message.SerializeWithCachedSizes(&outputStream);
    Y_ENSURE(!outputStream.HadError() && size == outputStream.ByteCount(), "Fail to serialize protobuf");

    return MD5::Calc(result);
}

} // namespace NInfra::NPodAgent
