#include "proto_hasher.h"

#include <infra/pod_agent/libs/pod_agent/trees_generators/secret.h>
#include <infra/pod_agent/libs/pod_agent/update_holder/test_lib/test_functions.h>

#include <library/cpp/digest/md5/md5.h>
#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NProtoHasherTest {

Y_UNIT_TEST_SUITE(ProtoHasherTestSuite) {

Y_UNIT_TEST(TestHashesNotEqual) {
    {
        API::TLayer layer;
        layer.set_id("layer_id");
        API::TLayer layer2;
        layer2.set_id("layer_id1");
        UNIT_ASSERT_UNEQUAL(GetLayerHash(layer, true), GetLayerHash(layer2, true));
        UNIT_ASSERT_EQUAL(GetLayerHash(layer, true), GetLayerHash(layer, true));
        UNIT_ASSERT_UNEQUAL(GetLayerHash(layer, true), GetLayerHash(layer, false));
    }
    {
        API::TResource resource;
        resource.set_id("resource_id");
        API::TResource resource2;
        resource2.set_id("resource_id1");
        UNIT_ASSERT_UNEQUAL(GetStaticResourceHash(resource, {}), GetStaticResourceHash(resource2, {}));
    }
    {
        API::TVolume volume;
        volume.set_id("volume_id");
        volume.mutable_generic()->add_layer_refs("layer_id");
        TLayerTreeGenerator::TLayersToAdd layers;
        layers.Layers_.insert(
            {
                "layer_id"
                , {
                    NObjectTargetTestLib::CreateLayerTargetSimple("layer_id", "layer_download_hash")
                    , "some_hash"
                    , "" /* VirtualDiskIdRef_ */
                }
            }
        );
        auto hash1 = GetVolumeHash(volume, layers, {});
        layers.Layers_.at("layer_id").FullHash_ = "other_hash";
        auto hash2 = GetVolumeHash(volume, layers, {});
        UNIT_ASSERT_UNEQUAL(hash1, hash2);
    }
    {
        API::TVolume volume;
        volume.set_id("volume_id");
        auto resource = volume.add_static_resources();
        *resource->mutable_resource_ref() = "static_resource_id";
        *resource->mutable_volume_relative_mount_point() = "folder";
        TStaticResourceTreeGenerator::TStaticResourcesToAdd staticResources;
        staticResources.StaticResources_.insert(
            {
                "static_resource_id"
                , {
                    NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource_id", "static_resource_download_hash")
                    , "some_hash"
                    , "" /* VirtualDiskIdRef_ */
                    , {}
                }
            }
        );
        auto hash1 = GetVolumeHash(volume, {}, staticResources);
        staticResources.StaticResources_.at("static_resource_id").FullHash_ = "other_hash";
        auto hash2 = GetVolumeHash(volume, {}, staticResources);
        UNIT_ASSERT_UNEQUAL(hash1, hash2);
    }
    {
        API::TBox box;
        box.set_id("box_id");
        box.mutable_rootfs()->add_layer_refs("layer_id");
        TLayerTreeGenerator::TLayersToAdd layers;
        layers.Layers_.insert(
            {
                "layer_id"
                , {
                    NObjectTargetTestLib::CreateLayerTargetSimple("layer_id", "layer_download_hash")
                    , "some_hash"
                    , "" /* VirtualDiskIdRef_ */
                }
            }
        );
        auto hash1 = GetBoxHash(box, layers, {}, {}, {}, {}, Nothing(), Nothing(), "");
        layers.Layers_.at("layer_id").FullHash_ = "other_hash";
        auto hash2 = GetBoxHash(box, layers, {}, {}, {}, {}, Nothing(), Nothing(), "");
        UNIT_ASSERT_UNEQUAL(hash1, hash2);
    }
    {
        API::TBox box;
        box.set_id("box_id");
        auto resource = box.add_static_resources();
        *resource->mutable_resource_ref() = "static_resource_id";
        *resource->mutable_mount_point() = "folder";
        TStaticResourceTreeGenerator::TStaticResourcesToAdd staticResources;
        staticResources.StaticResources_.insert(
            {
                "static_resource_id"
                , {
                    NObjectTargetTestLib::CreateStaticResourceTargetSimple("static_resource_id", "static_resource_download_hash")
                    , "some_hash"
                    , "" /* VirtualDiskIdRef_ */
                    , {}
                }
            }
        );
        auto hash1 = GetBoxHash(box, {}, {staticResources}, {}, {}, {}, Nothing(), Nothing(), "");
        staticResources.StaticResources_.at("static_resource_id").FullHash_ = "other_hash";
        auto hash2 = GetBoxHash(box, {}, {staticResources}, {}, {}, {}, Nothing(), Nothing(), "");
        UNIT_ASSERT_UNEQUAL(hash1, hash2);
    }
    {
        API::TBox box;
        box.set_id("box_id");
        box.add_volumes()->set_volume_ref("volume_id");
        TVolumeTreeGenerator::TVolumesToAdd volumes;
        volumes.Volumes_.insert(
            {
                "volume_id"
                , {
                    NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("volume_id", "some_hash")
                }
            }
        );
        auto hash1 = GetBoxHash(box, {}, {}, volumes, {}, {}, Nothing(), Nothing(), "");
        volumes.Volumes_.erase("volume_id");
        volumes.Volumes_.insert(
            {
                "volume_id"
                , {
                    NObjectTargetTestLib::CreatePersistentVolumeTargetSimple("volume_id", "other_hash")
                }
            }
        );
        auto hash2 = GetBoxHash(box, {}, {}, volumes, {}, {}, Nothing(), Nothing(), "");
        UNIT_ASSERT_UNEQUAL(hash1, hash2);
    }
    {
        API::TBox box;
        box.set_id("box_id");
        auto hash1 = GetBoxHash(box, {}, {}, {}, {{"name", "val1"}}, {}, Nothing(), Nothing(), "");
        auto hash2 = GetBoxHash(box, {}, {}, {}, {{"name", "val2"}}, {}, Nothing(), Nothing(), "");
        UNIT_ASSERT_UNEQUAL(hash1, hash2);
    }
    {
        API::TBox box;
        box.set_id("box_id");
        auto hash1 = GetBoxHash(box, {}, {}, {}, {}, {}, Nothing(), Nothing(), "");
        auto hash2 = GetBoxHash(box, {}, {}, {}, {}, {}, "ytPath", Nothing(), "");
        UNIT_ASSERT_UNEQUAL(hash1, hash2);
    }
    {
        API::TBox box;
        box.set_id("box_id");
        auto hash1 = GetBoxHash(box, {}, {}, {}, {}, {}, "ytPath1", Nothing(), "");
        auto hash2 = GetBoxHash(box, {}, {}, {}, {}, {}, "ytPath2", Nothing(), "");
        UNIT_ASSERT_UNEQUAL(hash1, hash2);
    }
    {
        API::TBox box;
        box.set_id("box_id");
        auto hash1 = GetBoxHash(box, {}, {}, {}, {}, {}, Nothing(), Nothing(), "");
        auto hash2 = GetBoxHash(box, {}, {}, {}, {}, {}, Nothing(), "baseSearchPath", "");
        UNIT_ASSERT_UNEQUAL(hash1, hash2);
    }
    {
        API::TBox box;
        box.set_id("box_id");
        auto hash1 = GetBoxHash(box, {}, {}, {}, {}, {}, Nothing(), "baseSearchPath1", "");
        auto hash2 = GetBoxHash(box, {}, {}, {}, {}, {}, Nothing(), "baseSearchPath2", "");
        UNIT_ASSERT_UNEQUAL(hash1, hash2);
    }
    {
        API::TBox box;
        box.set_id("box_id");
        auto hash1 = GetBoxHash(box, {}, {}, {}, {}, {}, Nothing(), Nothing(), "ip1");
        auto hash2 = GetBoxHash(box, {}, {}, {}, {}, {}, Nothing(), Nothing(), "ip2");
        UNIT_ASSERT_UNEQUAL(hash1, hash2);
    }
    {
        API::TWorkload workload;
        workload.set_box_ref("box_id");
        workload.set_id("workload_id");

        TString boxHash1 = "some_hash";
        auto hash1 = GetWorkloadHash(workload, {}, boxHash1);
        TString boxHash2 = "other_hash";
        auto hash2 = GetWorkloadHash(workload, {}, boxHash2);
        UNIT_ASSERT_UNEQUAL(hash1, hash2);
    }
}

Y_UNIT_TEST(TestDownloadHash) {
    {
        API::TLayer layer1;
        layer1.set_id("layer1");
        layer1.set_url("url1");
        layer1.set_checksum("checksum1");
        layer1.set_virtual_disk_id_ref("virtual_disk_1");

        API::TLayer layer2;
        layer2.set_id("layer2");
        layer2.set_url("url1");
        layer2.set_checksum("checksum1");
        layer2.set_virtual_disk_id_ref("virtual_disk_1");

        UNIT_ASSERT_EQUAL(GetLayerDownloadHash(layer1, true), GetLayerDownloadHash(layer2, true));
        UNIT_ASSERT_UNEQUAL(GetLayerDownloadHash(layer1, true), GetLayerDownloadHash(layer2, false));

        layer2.set_url("url2");
        UNIT_ASSERT_UNEQUAL(GetLayerDownloadHash(layer1, true), GetLayerDownloadHash(layer2, true));
        layer1.set_url("url2");
        UNIT_ASSERT_EQUAL(GetLayerDownloadHash(layer1, true), GetLayerDownloadHash(layer2, true));

        layer2.set_checksum("checksum2");
        UNIT_ASSERT_UNEQUAL(GetLayerDownloadHash(layer1, true), GetLayerDownloadHash(layer2, true));
        layer1.set_checksum("checksum2");
        UNIT_ASSERT_EQUAL(GetLayerDownloadHash(layer1, true), GetLayerDownloadHash(layer2, true));

        layer2.set_virtual_disk_id_ref("virtual_disk_2");
        UNIT_ASSERT_UNEQUAL(GetLayerDownloadHash(layer1, true), GetLayerDownloadHash(layer2, true));
        layer1.set_virtual_disk_id_ref("virtual_disk_2");
        UNIT_ASSERT_EQUAL(GetLayerDownloadHash(layer1, true), GetLayerDownloadHash(layer2, true));

        layer1.mutable_sky_get()->set_resid("resid1");
        UNIT_ASSERT_UNEQUAL(GetLayerDownloadHash(layer1, true), GetLayerDownloadHash(layer2, true));
        layer2.mutable_sky_get()->set_resid("resid2");
        UNIT_ASSERT_UNEQUAL(GetLayerDownloadHash(layer1, true), GetLayerDownloadHash(layer2, true));
        layer2.mutable_sky_get()->set_resid("resid1");
        UNIT_ASSERT_EQUAL(GetLayerDownloadHash(layer1, true), GetLayerDownloadHash(layer2, true));
    }

    {
        API::TResource resource1;
        resource1.set_id("resource1");
        resource1.set_url("url1");
        resource1.mutable_verification()->set_checksum("checksum1");
        resource1.mutable_verification()->set_check_period_ms(3);
        resource1.set_virtual_disk_id_ref("virtual_disk_1");

        API::TResource resource2;
        resource2.set_id("resource2");
        resource2.set_url("url1");
        resource2.mutable_verification()->set_checksum("checksum1");
        resource2.mutable_verification()->set_check_period_ms(4);
        resource2.set_virtual_disk_id_ref("virtual_disk_1");

        UNIT_ASSERT_EQUAL(GetStaticResourceDownloadHash(resource1, {}), GetStaticResourceDownloadHash(resource2, {}));

        resource2.set_url("url2");
        UNIT_ASSERT_UNEQUAL(GetStaticResourceDownloadHash(resource1, {}), GetStaticResourceDownloadHash(resource2, {}));
        resource1.set_url("url2");
        UNIT_ASSERT_EQUAL(GetStaticResourceDownloadHash(resource1, {}), GetStaticResourceDownloadHash(resource2, {}));

        resource2.set_access_permissions(API::EResourceAccessPermissions::EResourceAccessPermissions_UNMODIFIED);
        UNIT_ASSERT_EQUAL(GetStaticResourceDownloadHash(resource1, {}), GetStaticResourceDownloadHash(resource2, {}));

        resource2.set_access_permissions(API::EResourceAccessPermissions::EResourceAccessPermissions_660);
        UNIT_ASSERT_UNEQUAL(GetStaticResourceDownloadHash(resource1, {}), GetStaticResourceDownloadHash(resource2, {}));

        resource1.set_access_permissions(API::EResourceAccessPermissions::EResourceAccessPermissions_600);
        UNIT_ASSERT_UNEQUAL(GetStaticResourceDownloadHash(resource1, {}), GetStaticResourceDownloadHash(resource2, {}));

        resource1.set_access_permissions(API::EResourceAccessPermissions::EResourceAccessPermissions_UNMODIFIED);
        resource2.set_access_permissions(API::EResourceAccessPermissions::EResourceAccessPermissions_UNMODIFIED);
        UNIT_ASSERT_EQUAL(GetStaticResourceDownloadHash(resource1, {}), GetStaticResourceDownloadHash(resource2, {}));

        resource1.mutable_group_id()->set_value(0);
        UNIT_ASSERT_UNEQUAL(GetStaticResourceDownloadHash(resource1, {}), GetStaticResourceDownloadHash(resource2, {}));

        resource2.mutable_group_id()->set_value(0);
        UNIT_ASSERT_EQUAL(GetStaticResourceDownloadHash(resource1, {}), GetStaticResourceDownloadHash(resource2, {}));

        resource2.mutable_group_id()->set_value(1);
        UNIT_ASSERT_UNEQUAL(GetStaticResourceDownloadHash(resource1, {}), GetStaticResourceDownloadHash(resource2, {}));

        resource2.mutable_group_id()->set_value(0);
        UNIT_ASSERT_EQUAL(GetStaticResourceDownloadHash(resource1, {}), GetStaticResourceDownloadHash(resource2, {}));

        resource2.mutable_verification()->set_checksum("checksum2");
        UNIT_ASSERT_UNEQUAL(GetStaticResourceDownloadHash(resource1, {}), GetStaticResourceDownloadHash(resource2, {}));
        resource1.mutable_verification()->set_checksum("checksum2");
        UNIT_ASSERT_EQUAL(GetStaticResourceDownloadHash(resource1, {}), GetStaticResourceDownloadHash(resource2, {}));

        resource2.mutable_verification()->set_disabled(true);
        UNIT_ASSERT_UNEQUAL(GetStaticResourceDownloadHash(resource1, {}), GetStaticResourceDownloadHash(resource2, {}));
        resource1.mutable_verification()->set_checksum("EMPTY:");
        UNIT_ASSERT_EQUAL(GetStaticResourceDownloadHash(resource1, {}), GetStaticResourceDownloadHash(resource2, {}));

        resource2.set_virtual_disk_id_ref("virtual_disk_2");
        UNIT_ASSERT_UNEQUAL(GetStaticResourceDownloadHash(resource1, {}), GetStaticResourceDownloadHash(resource2, {}));
        resource1.set_virtual_disk_id_ref("virtual_disk_2");
        UNIT_ASSERT_EQUAL(GetStaticResourceDownloadHash(resource1, {}), GetStaticResourceDownloadHash(resource2, {}));

        API::TFile* file1 = resource1.mutable_files()->add_files();
        file1->set_file_name("file.txt");
        file1->set_raw_data("raw data");
        UNIT_ASSERT_UNEQUAL(GetStaticResourceDownloadHash(resource1, {}), GetStaticResourceDownloadHash(resource2, {}));
        API::TFile* file2 = resource2.mutable_files()->add_files();
        file2->set_file_name("file.txt");
        file2->set_raw_data("another raw data");
        UNIT_ASSERT_UNEQUAL(GetStaticResourceDownloadHash(resource1, {}), GetStaticResourceDownloadHash(resource2, {}));
        file2->set_raw_data("raw data");
        UNIT_ASSERT_EQUAL(GetStaticResourceDownloadHash(resource1, {}), GetStaticResourceDownloadHash(resource2, {}));

        resource1.mutable_sky_get()->set_resid("resid1");
        UNIT_ASSERT_UNEQUAL(GetStaticResourceDownloadHash(resource1, {}), GetStaticResourceDownloadHash(resource2, {}));
        resource2.mutable_sky_get()->set_resid("resid2");
        UNIT_ASSERT_UNEQUAL(GetStaticResourceDownloadHash(resource1, {}), GetStaticResourceDownloadHash(resource2, {}));
        resource2.mutable_sky_get()->set_resid("resid1");
        UNIT_ASSERT_EQUAL(GetStaticResourceDownloadHash(resource1, {}), GetStaticResourceDownloadHash(resource2, {}));
    }
}

Y_UNIT_TEST(TestExceptions) {
    {
        API::TVolume volume;
        volume.mutable_generic()->add_layer_refs("some_layer");
        volume.set_id("volume_id");
        UNIT_ASSERT_EXCEPTION(GetVolumeHash(volume, {}, {}), yexception);
    }
    {
        API::TBox box;
        box.mutable_rootfs()->add_layer_refs("some_layer");
        box.set_id("box_id");
        UNIT_ASSERT_EXCEPTION(GetBoxHash(box, {}, {}, {}, {}, {}, Nothing(), Nothing(), ""), yexception);
    }
    {
        API::TBox box;
        box.add_static_resources()->set_resource_ref("some_static_resource");
        box.set_id("box_id");
        UNIT_ASSERT_EXCEPTION(GetBoxHash(box, {}, {}, {}, {}, {}, Nothing(), Nothing(), ""), yexception);
    }
    {
        API::TBox box;
        box.add_volumes()->set_volume_ref("some_volume");
        box.set_id("box_id");
        UNIT_ASSERT_EXCEPTION(GetBoxHash(box, {}, {}, {}, {}, {}, Nothing(), Nothing(), ""), yexception);
    }
    {
        API::TResource resource;
        resource.set_id("resource_id");
        resource.mutable_verification()->set_checksum("checksum");
        resource.mutable_verification()->set_check_period_ms(4);
        resource.set_virtual_disk_id_ref("virtual_disk");
        UNIT_ASSERT_EXCEPTION(GetStaticResourceDownloadHash(resource, {}), yexception);
    }
    {
        API::TLayer layer;
        layer.set_id("layer_id");
        layer.set_checksum("checksum");
        layer.set_virtual_disk_id_ref("virtual_disk");
        UNIT_ASSERT_EXCEPTION(GetLayerDownloadHash(layer, true), yexception);
    }
}


Y_UNIT_TEST(TestHashWithSecrets) {
    NSecret::TSecretMap secrets1;
    NSecret::TSecretMap secrets2;
    secrets1["SecretAlias"]["SecretId"] = {
        "SomeValue"
        , ""
    };
    secrets2["SecretAlias"]["SecretId"] = {
        "T3RoZXJWYWx1ZQ=="
        , NSecret::BASE64_ENCODING
    };
    {
        API::TBox box;
        box.set_id("box_id");

        TString hash1 = GetBoxHash(box, {}, {}, {}, {}, secrets1, Nothing(), Nothing(), "");
        TString hash2 = GetBoxHash(box, {}, {}, {}, {}, secrets2, Nothing(), Nothing(), "");
        UNIT_ASSERT_EQUAL(hash1, hash2);

        API::TEnvVar* secretEnvVar = box.add_env();
        secretEnvVar->set_name("TestSecret");
        secretEnvVar->mutable_value()->mutable_secret_env()->set_alias("SecretAlias");
        secretEnvVar->mutable_value()->mutable_secret_env()->set_id("SecretId");

        hash1 = GetBoxHash(box, {}, {}, {}, {}, secrets1, Nothing(), Nothing(), "");
        hash2 = GetBoxHash(box, {}, {}, {}, {}, secrets2, Nothing(), Nothing(), "");
        UNIT_ASSERT_UNEQUAL(hash1, hash2);

        TString hashWithoutDecoding = GetBoxHash(box, {}, {}, {}, {}, secrets2, Nothing(), Nothing(), "");
        TString hashWithDecoding = GetBoxHash(box, {}, {}, {}, {}, secrets2, Nothing(), Nothing(), "", /* useEnvSecret = */ false, /* autoDecodeBase64Secrets = */ true);
        UNIT_ASSERT_UNEQUAL(hashWithoutDecoding, hashWithDecoding);
    }

    {
        API::TWorkload workload;
        workload.set_box_ref("box_id");
        workload.set_id("workload_id");

        TString boxHash = "hash";

        TString hash1 = GetWorkloadHash(workload, secrets1, boxHash);
        TString hash2 = GetWorkloadHash(workload, secrets2, boxHash);
        UNIT_ASSERT_EQUAL(hash1, hash2);

        API::TEnvVar* secretEnvVar = workload.add_env();
        secretEnvVar->set_name("TestSecret");
        secretEnvVar->mutable_value()->mutable_secret_env()->set_alias("SecretAlias");
        secretEnvVar->mutable_value()->mutable_secret_env()->set_id("SecretId");

        hash1 = GetWorkloadHash(workload, secrets1, boxHash);
        hash2 = GetWorkloadHash(workload, secrets2, boxHash);
        UNIT_ASSERT_UNEQUAL(hash1, hash2);

        TString hashWithoutDecoding = GetWorkloadHash(workload, secrets2, boxHash);
        TString hashWithDecoding = GetWorkloadHash(workload, secrets2, boxHash, /* useEnvSecret = */ false, /* autoDecodeBase64Secrets = */ true);
        UNIT_ASSERT_UNEQUAL(hashWithoutDecoding, hashWithDecoding);
    }

    {
        API::TResource resource1;
        resource1.set_id("resource_id");
        resource1.set_url("url1");
        API::TResource resource2;
        resource2.set_id("resource_id");
        resource2.set_url("url1");

        UNIT_ASSERT_EQUAL(GetStaticResourceHash(resource1, secrets1), GetStaticResourceHash(resource2, secrets2));
        UNIT_ASSERT_EQUAL(GetStaticResourceDownloadHash(resource1, secrets1), GetStaticResourceDownloadHash(resource2, secrets2));

        API::SecretSelector secret;
        secret.set_alias("SecretAlias");
        secret.set_id("SecretId");

        API::TFile file;
        *(file.mutable_secret_data()) = secret;

        *(resource1.mutable_files()->add_files()) = file;
        *(resource2.mutable_files()->add_files()) = file;

        UNIT_ASSERT_UNEQUAL(GetStaticResourceHash(resource1, secrets1), GetStaticResourceHash(resource2, secrets2));
        UNIT_ASSERT_UNEQUAL(GetStaticResourceDownloadHash(resource1, secrets1), GetStaticResourceDownloadHash(resource2, secrets2));

        UNIT_ASSERT_UNEQUAL(GetStaticResourceHash(resource1, secrets2), GetStaticResourceHash(resource2, secrets2, /* autoDecodeBase64Secrets = */ true));
        UNIT_ASSERT_UNEQUAL(GetStaticResourceDownloadHash(resource1, secrets2), GetStaticResourceDownloadHash(resource2, secrets2, /* autoDecodeBase64Secrets = */ true));
    }

    {
        API::TResource resource1;
        resource1.set_id("resource_id");
        resource1.set_url("url1");
        API::TResource resource2;
        resource2.set_id("resource_id");
        resource2.set_url("url1");

        UNIT_ASSERT_EQUAL(GetStaticResourceHash(resource1, secrets1), GetStaticResourceHash(resource2, secrets2));
        UNIT_ASSERT_EQUAL(GetStaticResourceDownloadHash(resource1, secrets1), GetStaticResourceDownloadHash(resource2, secrets2));

        API::SecretSelector secret;
        secret.set_alias("SecretAlias");
        secret.set_id("SecretId");

        API::TFile file;
        file.mutable_multi_secret_data()->set_secret_alias("SecretAlias");

        *(resource1.mutable_files()->add_files()) = file;
        *(resource2.mutable_files()->add_files()) = file;

        UNIT_ASSERT_UNEQUAL(GetStaticResourceHash(resource1, secrets1), GetStaticResourceHash(resource2, secrets2));
        UNIT_ASSERT_UNEQUAL(GetStaticResourceDownloadHash(resource1, secrets1), GetStaticResourceDownloadHash(resource2, secrets2));

        UNIT_ASSERT_UNEQUAL(GetStaticResourceHash(resource1, secrets2), GetStaticResourceHash(resource2, secrets2, /* autoDecodeBase64Secrets = */ true));
        UNIT_ASSERT_UNEQUAL(GetStaticResourceDownloadHash(resource1, secrets2), GetStaticResourceDownloadHash(resource2, secrets2, /* autoDecodeBase64Secrets = */ true));
    }
}

Y_UNIT_TEST(TestMapDeterministic) {
    const size_t iterCnt = 1000;

    TString lastHash = "";
    for (size_t i = 0; i < iterCnt; ++i) {
        API::TPodAgentRequest podAgentRequest;
        auto* secrets = podAgentRequest.mutable_secrets();

        auto* secret = secrets->Add();
        secret->set_id("SecretAlias");
        auto* secretsPayload = secret->mutable_payload();
        (*secretsPayload)["SecretId1"] = "SomeValue1";
        (*secretsPayload)["SecretId2"] = "SomeValue2";
        (*secretsPayload)["SecretId3"] = "SomeValue3";

        TString currentHash = GetHashFromProto(podAgentRequest);
        if (i != 0) {
            UNIT_ASSERT_EQUAL_C(lastHash, currentHash, "Fail at iter " << ToString(i));
        }
        lastHash = currentHash;
    }
}

Y_UNIT_TEST(TestHashWithChangingFlagOfUsingEnvSecret) {
    {
        API::TBox box;
        box.set_id("box_id");

        NSecret::TSecretMap secrets;
        secrets["SecretAlias"]["SecretId"] = {
            "SomeValue"
            , ""
        };
        API::TEnvVar* secretEnvVar = box.add_env();
        secretEnvVar->set_name("TestSecret");
        secretEnvVar->mutable_value()->mutable_secret_env()->set_alias("SecretAlias");
        secretEnvVar->mutable_value()->mutable_secret_env()->set_id("SecretId");
        API::TEnvVar* publicEnvVar = box.add_env();
        publicEnvVar->set_name("TestPublic");
        publicEnvVar->mutable_value()->mutable_literal_env()->set_value("TestValue");

        TString hash1 = GetBoxHash(box, {}, {}, {}, {}, secrets, Nothing(), Nothing(), "");
        // Secret and public anv in one list. Hash works as before using env secret
        TString hash2 = MD5::Calc(GetHashFromProto(box) + "TestPublic=TestValue;TestSecret=SomeValue");
        UNIT_ASSERT_EQUAL(hash1, hash2);

        TString hash3 = GetBoxHash(box, {}, {}, {}, {}, secrets, Nothing(), Nothing(), "", /* useEnvSecret = */ true);
        // Secret and public anv in separated lists. Hash will be changed
        TString hash4 = MD5::Calc(GetHashFromProto(box) + "TestPublic=TestValue" + "TestSecret=SomeValue");
        UNIT_ASSERT_EQUAL(hash3, hash4);
        UNIT_ASSERT_UNEQUAL(hash1, hash3);
    }

    {
        API::TWorkload workload;
        workload.set_box_ref("box_id");
        workload.set_id("workload_id");

        TString boxHash = "hash";

        NSecret::TSecretMap secrets;
        secrets["SecretAlias"]["SecretId"] = {
            "SomeValue"
            , ""
        };
        API::TEnvVar* secretEnvVar = workload.add_env();
        secretEnvVar->set_name("TestSecret");
        secretEnvVar->mutable_value()->mutable_secret_env()->set_alias("SecretAlias");
        secretEnvVar->mutable_value()->mutable_secret_env()->set_id("SecretId");
        API::TEnvVar* publicEnvVar = workload.add_env();
        publicEnvVar->set_name("TestPublic");
        publicEnvVar->mutable_value()->mutable_literal_env()->set_value("TestValue");

        TString hash1 = GetWorkloadHash(workload, secrets, boxHash);
        // Secret and public anv in one list. Hash works as before using env secret
        TString hash2 = MD5::Calc(GetHashFromProto(workload) + "hash" + "TestPublic=TestValue;TestSecret=SomeValue");
        UNIT_ASSERT_EQUAL(hash1, hash2);

        TString hash3 = GetWorkloadHash(workload, secrets, boxHash, /* useEnvSecret = */ true);
        // Secret and public anv in separated lists. Hash will be changed
        TString hash4 = MD5::Calc(GetHashFromProto(workload) + "hash" + "TestPublic=TestValue" + "TestSecret=SomeValue");
        UNIT_ASSERT_EQUAL(hash3, hash4);
        UNIT_ASSERT_UNEQUAL(hash1, hash3);
    }
}

Y_UNIT_TEST(TestStaticResourceHashWithChangingFlagOfDisablingVerification) {
    API::TResource resource1;
    resource1.set_id("resource1");
    resource1.set_url("url1");
    resource1.mutable_verification()->set_checksum("checksum1");

    API::TResource resource2 = resource1;

    UNIT_ASSERT_EQUAL(GetStaticResourceHash(resource1, {}), GetStaticResourceHash(resource2, {}));

    resource1.mutable_verification()->set_checksum("EMPTY:");
    UNIT_ASSERT_UNEQUAL(GetStaticResourceHash(resource1, {}), GetStaticResourceHash(resource2, {}));
    resource2.mutable_verification()->set_disabled(true);
    UNIT_ASSERT_EQUAL(GetStaticResourceHash(resource1, {}), GetStaticResourceHash(resource2, {}));
    resource2.mutable_verification()->clear_checksum();
    UNIT_ASSERT_EQUAL(GetStaticResourceHash(resource1, {}), GetStaticResourceHash(resource2, {}));
    resource1.mutable_verification()->set_disabled(true);
    UNIT_ASSERT_EQUAL(GetStaticResourceHash(resource1, {}), GetStaticResourceHash(resource2, {}));
    resource1.mutable_verification()->clear_checksum();
    UNIT_ASSERT_EQUAL(GetStaticResourceHash(resource1, {}), GetStaticResourceHash(resource2, {}));

    resource1.mutable_verification()->set_checksum("checksum");
    resource2.mutable_verification()->set_checksum("another_checksum");
    // Checksums do not affect hash when verification is disabled
    UNIT_ASSERT_EQUAL(GetStaticResourceHash(resource1, {}), GetStaticResourceHash(resource2, {}));
}

}

} // namespace NInfra::NPodAgent::NProtoHasherTest
