#include "secret.h"

#include <library/cpp/json/writer/json.h>
#include <library/cpp/json/writer/json_value.h>
#include <library/cpp/string_utils/base64/base64.h>
#include <library/cpp/yaml/as/tstring.h>

#include <util/string/builder.h>

namespace NInfra::NPodAgent::NSecret {

namespace NMapFormatter {

TString MapToJsonFormat(
    const TMap<TString, TSecretInfo>& map
) {
    NJson::TJsonValue jsonMap = NJson::TJsonValue(NJson::JSON_MAP);

    for (const auto& [key, secretInfo] : map) {
        jsonMap[key] = secretInfo.Value_;
    }

    return NJsonWriter::TBuf()
        .SetIndentSpaces(4)
        .WriteJsonValue(&jsonMap, true /* sortKeys */)
        .Str()
    ;
}

TString MapToJavaFormat(
    const TMap<TString, TSecretInfo>& map
) {
    TStringBuilder javaStr;

    bool firstElement = true;
    for (const auto& [key, secretInfo] : map) {
        TString escapedKey;
        TString escapedValue;

        // https://a.yandex-team.ru/arc/trunk/arcadia/passport/python/vault/cli/yav/vault_client_cli/commands/format.py?rev=6200047#L20-24
        for (const char c : key) {
            if (c == ' ' || c == '\n' || c == ':' || c == '=' || c == '!' || c == '#' || c == '\t') {
                escapedKey.push_back('\\');
            }
            escapedKey.push_back(c);
        }
        for (const char c : secretInfo.Value_) {
            if (c == '\n' || c == '!' || c == '#') {
                escapedValue.push_back('\\');
            }
            escapedValue.push_back(c);
        }

        if (!firstElement) {
            javaStr << Endl;
        }
        firstElement = false;

        javaStr << escapedKey << " = " << escapedValue;
    }

    return javaStr;
}

TString MapToYamlFormat(
    const TMap<TString, TSecretInfo>& map
) {
    YAML::Node yamlMap(YAML::NodeType::Map);

    for (const auto& [key, secretInfo] : map) {
        yamlMap[key] = secretInfo.Value_;
    }

    return TString(YAML::Dump(yamlMap));
}

} // namespace NMapFormatter

namespace {
    TString DecodeSecretValue(
        const TString& secretValue
        , const TString& secretAlias
        , const TString& secretId
    ) {
        try {
            return Base64StrictDecode(secretValue);
        } catch (...) {
            ythrow yexception() << "Error in base64 decode secret '"
                << secretAlias << "::" << secretId
                << "' (exception message hidden because it may contain secret value)";
        }
    }

} // namespace

TSecretMap GetSecretMap(
    const google::protobuf::RepeatedPtrField<NYP::NClient::NNodes::NProto::TPodSpec::TSecret>& secrets
) {
    TSecretMap secretMap;

    for (const auto& aliasToSecrets : secrets) {
        TMap<TString, TSecretInfo> keyToSecretInfo;

        if (!aliasToSecrets.values().empty()) {
            // New secrets delivery schema
            for (const auto& secret : aliasToSecrets.values()) {
                auto ptr = keyToSecretInfo.FindPtr(secret.key());
                Y_ENSURE(!ptr, "Two secret values in alias '" << aliasToSecrets.id() << "' have same key '" << secret.key() << "'");
                keyToSecretInfo.insert(
                    {
                        secret.key()
                        , {
                            secret.value()
                            , secret.encoding()
                        }
                    }
                );
            }
        } else {
            // Fallback to old secrets delivery schema
            for (const auto& [secretKey, secretValue] : aliasToSecrets.payload()) {
                keyToSecretInfo.insert(
                    {
                        secretKey
                        , {
                            secretValue
                            , "" /* Encoding_ */
                        }
                    }
                );
            }
        }

        auto ptr = secretMap.FindPtr(aliasToSecrets.id());
        Y_ENSURE(!ptr, "Two secret aliases have same id '" << aliasToSecrets.id() << "'");
        secretMap.insert({aliasToSecrets.id(), keyToSecretInfo});
    }

    return secretMap;
}

TString GetSecretValue(
    const API::SecretSelector& secret
    , const TSecretMap& secretMap
    , const bool autoDecodeBase64Secrets
) {
    auto keyToSecretInfo = secretMap.FindPtr(secret.alias());
    Y_ENSURE(keyToSecretInfo, "Secret Alias '" << secret.alias() << "' not found");

    auto secretInfoPtr = keyToSecretInfo->FindPtr(secret.id());
    Y_ENSURE(secretInfoPtr, "Secret '" << secret.alias() << "::" << secret.id() << "' not found");

    TString value = secretInfoPtr->Value_;
    if (((secretInfoPtr->Encoding_ == BASE64_ENCODING) && autoDecodeBase64Secrets) || secret.decode_base64()) {
        value = DecodeSecretValue(value, secret.alias(), secret.id());
    }

    return value;
}

TString GetMultiSecretFileContent(
    const API::TMultiSecretFileContent& secret
    , const TSecretMap& secretMap
    , const bool autoDecodeBase64Secrets
) {
    auto keyToSecretInfo = secretMap.FindPtr(secret.secret_alias());
    Y_ENSURE(keyToSecretInfo, "Secret Alias '" << secret.secret_alias() << "' not found");
    TMap<TString, TSecretInfo> secretMapToInfoCopy = *keyToSecretInfo;
    if (autoDecodeBase64Secrets) {
        for (auto& [key, secretInfo] : secretMapToInfoCopy) {
            if (secretInfo.Encoding_ == BASE64_ENCODING) {
                secretInfo.Value_ = DecodeSecretValue(secretInfo.Value_, secret.secret_alias(), key);
            }
        }
    }
    switch (secret.format()) {
        case API::EMultiSecretFileContentFormat_JSON:
            return NMapFormatter::MapToJsonFormat(secretMapToInfoCopy);
        case API::EMultiSecretFileContentFormat_JAVA:
            return NMapFormatter::MapToJavaFormat(secretMapToInfoCopy);
        case API::EMultiSecretFileContentFormat_YAML:
            return NMapFormatter::MapToYamlFormat(secretMapToInfoCopy);
        default:
            ythrow yexception() << "Unknown multi secret file format: '" << API::EMultiSecretFileContentFormat_Name(secret.format()) << "'";
    }
}

} // namespace NInfra::NPodAgent::NSecret
