#include "secret.h"

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NSecretTest {

Y_UNIT_TEST_SUITE(SecretTestSuite) {

Y_UNIT_TEST(TestGetSecretMapOldSchema) {
    google::protobuf::RepeatedPtrField<NYP::NClient::NNodes::NProto::TPodSpec::TSecret> secrets;
    NSecret::TSecretMap secretMap;

    UNIT_ASSERT_EQUAL(NSecret::GetSecretMap(secrets), secretMap);

    auto* secret = secrets.Add();
    secret->set_id("SecretAlias");
    auto* secretsPayload = secret->mutable_payload();
    (*secretsPayload)["SecretKey"] = "SomeValue";

    secretMap["SecretAlias"]["SecretKey"] = {
        "SomeValue"
        , ""
    };

    UNIT_ASSERT_EQUAL(NSecret::GetSecretMap(secrets), secretMap);

    {
        secret = secrets.Add();
        secret->set_id("SecretAlias");
        UNIT_ASSERT_EXCEPTION_CONTAINS(NSecret::GetSecretMap(secrets), yexception, "Two secret aliases have same id");
    }
}

Y_UNIT_TEST(TestGetSecretMapNewSchema) {
    google::protobuf::RepeatedPtrField<NYP::NClient::NNodes::NProto::TPodSpec::TSecret> secrets;
    NSecret::TSecretMap secretMap;

    UNIT_ASSERT_EQUAL(NSecret::GetSecretMap(secrets), secretMap);

    auto* secret = secrets.Add();
    secret->set_id("SecretAlias");

    auto* value = secret->mutable_values()->Add();
    value->set_key("SecretKey");
    value->set_value("SomeValue");
    value->set_encoding("SomeEncoding");

    // Payload is added to test that with two schemas the schema with values is selected
    auto* secretsPayload = secret->mutable_payload();
    (*secretsPayload)["SecretKey"] = "SomeValue";

    secretMap["SecretAlias"]["SecretKey"] = {
        "SomeValue"
        , "SomeEncoding"
    };

    UNIT_ASSERT_EQUAL(NSecret::GetSecretMap(secrets), secretMap);

    {
        auto* value = secret->mutable_values()->Add();

        value->set_key("SecretKey");
        value->set_value("OtherValue");
        value->set_encoding("OtherEncoding");
        UNIT_ASSERT_EXCEPTION_CONTAINS(NSecret::GetSecretMap(secrets), yexception, "have same key");

        value->set_key("OtherKey");
        UNIT_ASSERT_NO_EXCEPTION(NSecret::GetSecretMap(secrets));
    }

    {
        secret = secrets.Add();
        secret->set_id("SecretAlias");
        UNIT_ASSERT_EXCEPTION_CONTAINS(NSecret::GetSecretMap(secrets), yexception, "Two secret aliases have same id");
    }
}

Y_UNIT_TEST(GetSecret) {
    NSecret::TSecretMap secretMap;
    secretMap["alias"]["id"] = {
        "value"
        , ""
    };
    secretMap["alias"]["base64Wrong"] = {
        "wrongBase64"
        , "base64"
    };
    secretMap["alias"]["base64Flag"] = {
        "YmFzZTY0X2RlY29kZWRfdmFsdWU="
        , ""
    };
    secretMap["alias"]["base64Encoding"] = {
        "YmFzZTY0X2RlY29kZWRfdmFsdWU="
        , "base64"
    };
    secretMap["alias"]["base64FlagAndEncoding"] = {
        "YmFzZTY0X2RlY29kZWRfdmFsdWU="
        , "base64"
    };

    {
        API::SecretSelector wrongAlias;
        wrongAlias.set_alias("wrong_alias");
        wrongAlias.set_id("id");

        UNIT_ASSERT_EXCEPTION_CONTAINS(
            NSecret::GetSecretValue(wrongAlias, secretMap)
            , yexception
            , "Secret Alias 'wrong_alias' not found"
        );
    }

    {
        API::SecretSelector wrongId;
        wrongId.set_alias("alias");
        wrongId.set_id("wrong_id");

        UNIT_ASSERT_EXCEPTION_CONTAINS(
            NSecret::GetSecretValue(wrongId, secretMap)
            , yexception
            , "Secret 'alias::wrong_id' not found"
        );
    }

    {
        API::SecretSelector base64Wrong;
        base64Wrong.set_alias("alias");
        base64Wrong.set_id("base64Wrong");

        UNIT_ASSERT_EXCEPTION_CONTAINS(
            NSecret::GetSecretValue(base64Wrong, secretMap, /* autoDecodeBase64Secrets = */ true)
            , yexception
            , "Error in base64 decode secret 'alias::base64Wrong' (exception message hidden because it may contain secret value)"
        );

        // Old behaviour
        base64Wrong.set_decode_base64(true);
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            NSecret::GetSecretValue(base64Wrong, secretMap, /* autoDecodeBase64Secrets = */ false)
            , yexception
            , "Error in base64 decode secret 'alias::base64Wrong' (exception message hidden because it may contain secret value)"
        );
    }

    {
        API::SecretSelector correctSecret;
        correctSecret.set_alias("alias");
        correctSecret.set_id("id");

        UNIT_ASSERT_EQUAL_C(
            NSecret::GetSecretValue(correctSecret, secretMap)
            , "value"
            , NSecret::GetSecretValue(correctSecret, secretMap)
        );
    }

    {
        API::SecretSelector base64Flag;
        base64Flag.set_alias("alias");
        base64Flag.set_id("base64Flag");

        UNIT_ASSERT_EQUAL_C(
            NSecret::GetSecretValue(base64Flag, secretMap, /* autoDecodeBase64Secrets = */ true)
            , "YmFzZTY0X2RlY29kZWRfdmFsdWU="
            , NSecret::GetSecretValue(base64Flag, secretMap, /* autoDecodeBase64Secrets = */ true)
        );

        // Old behaviour
        base64Flag.set_decode_base64(true);
        UNIT_ASSERT_EQUAL_C(
            NSecret::GetSecretValue(base64Flag, secretMap, /* autoDecodeBase64Secrets = */ false)
            , "base64_decoded_value"
            , NSecret::GetSecretValue(base64Flag, secretMap, /* autoDecodeBase64Secrets = */ false)
        );
    }

    {
        API::SecretSelector base64Encoding;
        base64Encoding.set_alias("alias");
        base64Encoding.set_id("base64Encoding");

        UNIT_ASSERT_EQUAL_C(
            NSecret::GetSecretValue(base64Encoding, secretMap)
            , "YmFzZTY0X2RlY29kZWRfdmFsdWU="
            , NSecret::GetSecretValue(base64Encoding, secretMap)
        );
    }

    {
        API::SecretSelector base64FlagAndEncoding;
        base64FlagAndEncoding.set_alias("alias");
        base64FlagAndEncoding.set_id("base64FlagAndEncoding");

        UNIT_ASSERT_EQUAL_C(
            NSecret::GetSecretValue(base64FlagAndEncoding, secretMap, /* autoDecodeBase64Secrets = */ true)
            , "base64_decoded_value"
            , NSecret::GetSecretValue(base64FlagAndEncoding, secretMap, /* autoDecodeBase64Secrets = */ true)
        );

        // Old behaviour
        base64FlagAndEncoding.set_decode_base64(true);
        UNIT_ASSERT_EQUAL_C(
            NSecret::GetSecretValue(base64FlagAndEncoding, secretMap, /* autoDecodeBase64Secrets = */ false)
            , "base64_decoded_value"
            , NSecret::GetSecretValue(base64FlagAndEncoding, secretMap, /* autoDecodeBase64Secrets = */ false)
        );
    }
}

Y_UNIT_TEST(GetMultiSecretFileContent) {
    NSecret::TSecretMap secretMap;
    secretMap["alias"]["key1"] = {
        "value1"
        , ""
    };
    secretMap["alias"]["key2"] = {
        "value2"
        , ""
    };
    secretMap["alias"]["key3"] = {
        "\\value3!#\n"
        , ""
    };
    secretMap["alias"]["key4"] = {
        "YmFzZTY0X2RlY29kZWRfdmFsdWU="
        , "base64"
    };

    API::TMultiSecretFileContent secret;

    {
        secret.set_secret_alias("wrong_alias");
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            NSecret::GetMultiSecretFileContent(secret, secretMap)
            , yexception
            , "Secret Alias 'wrong_alias' not found"
        );
    }

    {
        const TString expected = TStringBuilder()
            << "{\n"
            << "    \"key1\":\"value1\",\n"
            << "    \"key2\":\"value2\",\n"
            << "    \"key3\":\"\\\\value3!#\\n\",\n"
            << "    \"key4\":\"YmFzZTY0X2RlY29kZWRfdmFsdWU=\"\n"
            << "}"
        ;

        secret.set_secret_alias("alias");
        secret.set_format(API::EMultiSecretFileContentFormat_JSON);
        UNIT_ASSERT_EQUAL_C(
            NSecret::GetMultiSecretFileContent(secret, secretMap)
            , expected
            , NSecret::GetMultiSecretFileContent(secret, secretMap)
        );
    }

    {
        const TString expected = TStringBuilder()
            << "key1 = value1\n"
            << "key2 = value2\n"
            << "key3 = \\value3\\!\\#\\\n\n"
            << "key4 = YmFzZTY0X2RlY29kZWRfdmFsdWU="
        ;

        secret.set_secret_alias("alias");
        secret.set_format(API::EMultiSecretFileContentFormat_JAVA);
        UNIT_ASSERT_EQUAL_C(
            NSecret::GetMultiSecretFileContent(secret, secretMap)
            , expected
            , NSecret::GetMultiSecretFileContent(secret, secretMap)
        );
    }

    {
        const TString expected = TStringBuilder()
            << "key1: value1\n"
            << "key2: value2\n"
            << "key3: \"\\\\value3!#\\n\"\n"
            << "key4: YmFzZTY0X2RlY29kZWRfdmFsdWU="
        ;

        secret.set_secret_alias("alias");
        secret.set_format(API::EMultiSecretFileContentFormat_YAML);
        UNIT_ASSERT_EQUAL_C(
            NSecret::GetMultiSecretFileContent(secret, secretMap)
            , expected
            , NSecret::GetMultiSecretFileContent(secret, secretMap)
        );
    }
}

Y_UNIT_TEST(GetMultiSecretFileContentWithDecodingSecrets) {
    NSecret::TSecretMap secretMap;
    secretMap["alias"]["key1"] = {
        "value1"
        , ""
    };
    secretMap["alias"]["key2"] = {
        "YmFzZTY0X2RlY29kZWRfdmFsdWU="
        , "base64"
    };

    API::TMultiSecretFileContent secret;

    {
        secret.set_secret_alias("wrong_alias");
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            NSecret::GetMultiSecretFileContent(secret, secretMap, /* autoDecodeBase64Secrets = */ true)
            , yexception
            , "Secret Alias 'wrong_alias' not found"
        );
    }

    {
        const TString expected = TStringBuilder()
            << "{\n"
            << "    \"key1\":\"value1\",\n"
            << "    \"key2\":\"base64_decoded_value\"\n"
            << "}"
        ;

        secret.set_secret_alias("alias");
        secret.set_format(API::EMultiSecretFileContentFormat_JSON);
        UNIT_ASSERT_EQUAL_C(
            NSecret::GetMultiSecretFileContent(secret, secretMap, /* autoDecodeBase64Secrets = */ true)
            , expected
            , NSecret::GetMultiSecretFileContent(secret, secretMap, /* autoDecodeBase64Secrets = */ true)
        );
    }

    {
        const TString expected = TStringBuilder()
            << "key1 = value1\n"
            << "key2 = base64_decoded_value"
        ;

        secret.set_secret_alias("alias");
        secret.set_format(API::EMultiSecretFileContentFormat_JAVA);
        UNIT_ASSERT_EQUAL_C(
            NSecret::GetMultiSecretFileContent(secret, secretMap, /* autoDecodeBase64Secrets = */ true)
            , expected
            , NSecret::GetMultiSecretFileContent(secret, secretMap, /* autoDecodeBase64Secrets = */ true)
        );
    }

    {
        const TString expected = TStringBuilder()
            << "key1: value1\n"
            << "key2: base64_decoded_value"
        ;

        secret.set_secret_alias("alias");
        secret.set_format(API::EMultiSecretFileContentFormat_YAML);
        UNIT_ASSERT_EQUAL_C(
            NSecret::GetMultiSecretFileContent(secret, secretMap, /* autoDecodeBase64Secrets = */ true)
            , expected
            , NSecret::GetMultiSecretFileContent(secret, secretMap, /* autoDecodeBase64Secrets = */ true)
        );
    }
}

}

} // namespace NInfra::NPodAgent::NSecretTest
