#include "static_resource_tree_generator.h"

#include "object_tree.h"
#include "support_functions.h"

#include <infra/pod_agent/libs/behaviour/loaders/behavior3_editor_json_reader.h>
#include <infra/pod_agent/libs/behaviour/loaders/behavior3_template_resolver.h>
#include <infra/pod_agent/libs/pod_agent/trees_generators/proto_hasher.h>
#include <infra/pod_agent/libs/util/string_utils.h>

#include <library/cpp/string_utils/base64/base64.h>

#include <util/folder/path.h>
#include <util/string/vector.h>
#include <util/string/split.h>
#include <util/string/type.h>

namespace NInfra::NPodAgent {

TStaticResourceTreeGenerator::TStaticResourcesToAdd TStaticResourceTreeGenerator::UpdateSpec(
    const API::TResourceGang& spec
    , API::EPodAgentTargetState podAgentTargetState
    , const NSecret::TSecretMap& secretMap
    , const double cpuToVcpuFactor
    , ui64 specTimestamp
    , ui32 revision
    , bool autoDecodeBase64Secrets
) {
    TStaticResourceTreeGenerator::TStaticResourcesToAdd staticResources;

    // Add static resources to internal state only in ACTIVE or SUSPENDED pod_agent target states
    // WARNING: We do not validate static resource specs in other cases (for example in REMOVED target state)
    if (
        podAgentTargetState == API::EPodAgentTargetState_ACTIVE
        || podAgentTargetState == API::EPodAgentTargetState_SUSPENDED
    ) {
        for (const auto& resource : spec.static_resources()) {
            const TString& resourceId = resource.id();

            Y_ENSURE(!resourceId.empty(), "One of resources has empty id");
            Y_ENSURE(!staticResources.StaticResources_.contains(resourceId)
                , "Pod agent spec resources' ids are not unique: "
                << "resource '" << resourceId << "' occurs twice"
            );

            try {
                staticResources.StaticResources_.insert(
                    {
                        resourceId
                        , GetStaticResourceToAdd(
                            resource
                            , secretMap
                            , spec.compute_resources()
                            , cpuToVcpuFactor
                            , specTimestamp
                            , revision
                            , autoDecodeBase64Secrets
                        )
                    }
                );
            } catch (yexception& e) {
                throw e << " at resource '" << resourceId << "'";
            }
        }
    }

    return staticResources;
}

TStaticResourceTreeGenerator::TCacheStaticResourcesToAdd TStaticResourceTreeGenerator::UpdateResourceCache(
    const API::TPodAgentResourceCacheSpec& spec
    , API::EPodAgentTargetState podAgentTargetState
    , const NSecret::TSecretMap& secretMap
    , const API::TComputeResources& computeResources
    , const double cpuToVcpuFactor
    , bool autoDecodeBase64Secrets
) {
    TStaticResourceTreeGenerator::TCacheStaticResourcesToAdd cacheStaticResources;

    // Add cache static resources to internal state only in ACTIVE or SUSPENDED pod_agent target states
    // WARNING: We do not validate cache static resource specs in other cases (for example in REMOVED target state)
    if (
        podAgentTargetState == API::EPodAgentTargetState_ACTIVE
        || podAgentTargetState == API::EPodAgentTargetState_SUSPENDED
    ) {
        for (const auto& cacheStaticResource : spec.static_resources()) {
            const API::TResource& resource = cacheStaticResource.resource();

            const TString& cacheStaticResourceId = resource.id();
            const ui32 cacheStaticResourceRevision = cacheStaticResource.revision();

            Y_ENSURE(!cacheStaticResourceId.empty(), "One of cache resources has empty id");
            Y_ENSURE(!cacheStaticResources.CacheStaticResources_.contains(TStatusRepositoryCommon::TCacheObject(cacheStaticResourceId, cacheStaticResourceRevision))
                , "Pod agent resource cache resources' ids and revisions are not unique: "
                << "cache resource '" << cacheStaticResourceId << "' with revision '" << cacheStaticResourceRevision << "' occurs twice"
            );

            try {
                TStaticResourceToAdd resourceToAdd = GetStaticResourceToAdd(
                    resource
                    , secretMap
                    , computeResources
                    , cpuToVcpuFactor
                    , 0 /* specTimestamp */
                    , cacheStaticResourceRevision
                    , autoDecodeBase64Secrets
                );

                if (StatusNTickerHolder_->HasCacheStaticResource(cacheStaticResourceId, cacheStaticResourceRevision)) {
                    Y_ENSURE(resourceToAdd.Target_.Meta_.DownloadHash_ == StatusNTickerHolder_->GetCacheStaticResourceHash(cacheStaticResourceId, cacheStaticResourceRevision)
                        , "cache resource '" << cacheStaticResourceId << "' with revision '" << ToString(cacheStaticResourceRevision) << "' hash was changed");
                }

                cacheStaticResources.CacheStaticResources_.insert(
                    {
                        TStatusRepositoryCommon::TCacheObject(cacheStaticResourceId, cacheStaticResourceRevision)
                        , resourceToAdd
                    }
                );
            } catch (yexception& e) {
                throw e << " at cache resource '" << cacheStaticResourceId << "' with revision '" << cacheStaticResourceRevision << "'";;
            }
        }
    }

    return cacheStaticResources;
}

void TStaticResourceTreeGenerator::AddAndRemoveCacheStaticResources(const TCacheStaticResourcesToAdd& cacheStaticResources) {
    // If StatusRepository contains two staticResources with the same download hash, staticResource will still be downloaded exactly once
    // but if at least at the moment the staticResource was removed from the StatusRepository and added there again, it can lead to a re-download
    // so you first need to add new ones, and only then remove the old ones

    // Add new cache staticResources
    for (const auto& it : cacheStaticResources.CacheStaticResources_) {
        const TString& cacheStaticResourceId = it.first.Id_;
        const ui32 cacheStaticResourceRevision = it.first.Revision_;
        const auto& staticResourceData = it.second;

        if (!StatusNTickerHolder_->HasCacheStaticResource(cacheStaticResourceId, cacheStaticResourceRevision)) {
            StatusNTickerHolder_->AddCacheStaticResource(staticResourceData.Target_);
        }
    }

    // remove unneeded cache staticResources
    for (const auto& cacheStaticResource : StatusNTickerHolder_->GetCacheStaticResourceIdsAndRevisions()) {
        if (!cacheStaticResources.CacheStaticResources_.contains(cacheStaticResource)) {
            StatusNTickerHolder_->RemoveCacheStaticResource(cacheStaticResource.Id_, cacheStaticResource.Revision_);
        }
    }
}

void TStaticResourceTreeGenerator::RemoveStaticResources(const TStaticResourceTreeGenerator::TStaticResourcesToAdd& staticResources) {
    for (const TString& staticResourceId : StatusNTickerHolder_->GetStaticResourceIds()) {
        if (!staticResources.StaticResources_.contains(staticResourceId)) {
            StatusNTickerHolder_->SetStaticResourceTargetRemove(staticResourceId);
        }
    }
}

void TStaticResourceTreeGenerator::AddStaticResources(const TStaticResourceTreeGenerator::TStaticResourcesToAdd& staticResources) {
    for (const auto& it : staticResources.StaticResources_) {
        const TString& staticResourceId = it.first;
        const auto& staticResourceData = it.second;

        if (!StatusNTickerHolder_->HasStaticResource(staticResourceId)) {
            StatusNTickerHolder_->AddStaticResourceWithTargetCheck(staticResourceData.Target_);
        }

        StatusNTickerHolder_->UpdateStaticResourceTarget(staticResourceData.Target_);
    }
}

TStaticResourceTreeGenerator::TStaticResourceToAdd TStaticResourceTreeGenerator::GetStaticResourceToAdd(
    const API::TResource& resource
    , const NSecret::TSecretMap& secretMap
    , const API::TComputeResources& computeResources
    , const double cpuToVcpuFactor
    , ui64 specTimestamp
    , ui32 revision
    , bool autoDecodeBase64Secrets
) const {

    const TString staticResourceDownloadHash = GetStaticResourceDownloadHash(resource, secretMap, autoDecodeBase64Secrets);
    const TString fullHash = GetStaticResourceHash(resource, secretMap, autoDecodeBase64Secrets);

    ui64 checkPeriodMs = (resource.verification().check_period_ms() == 0)
        ? NSupport::DEFAULT_STATIC_RESOURCE_CHECK_PERIOD_MS
        : resource.verification().check_period_ms()
    ;

    TBehavior3 resolvedTree = StaticResourceTreeTemplate_;
    TMap<TString, TString> cachedReplace;
    {
        TMap<TString, TString> replace;

        replace["RESOURCE_DOWNLOAD_QUEUE_LOCK"] = NSupport::RESOURCE_DOWNLOAD_QUEUE_LOCK;
        replace["RESOURCE_VERIFY_QUEUE_LOCK"] = NSupport::RESOURCE_VERIFY_QUEUE_LOCK;

        replace["RESOURCE_GANG_CONTAINER"] = PathHolder_->GetResourceGangMetaContainer();
        replace["RESOURCE_GANG_CONTAINER_LOCK"] = NSupport::RESOURCE_GANG_CONTAINER_LOCK;
        replace.merge(
            NSupport::FillComputeResourcesReplaceMap(
                computeResources
                , "RESOURCE_GANG_"
                , PathHolder_
                , cpuToVcpuFactor
            )
        );

        // TODO(DEPLOY-2825, DEPLOY-2988) use ContainerUser/ContainerGroup
        // https://st.yandex-team.ru/DEPLOY-2825#5f44f3c6558e052f3a2d261d
        replace["STATIC_RESOURCE_CONTAINER_USER"] = ""; // ContainerUser_;
        replace["STATIC_RESOURCE_CONTAINER_GROUP"] = ""; // ContainerGroup_;
        replace["STATIC_RESOURCE_DOWNLOAD_CONTAINER"] = PathHolder_->GetStaticResourceContainerWithNameFromHash(staticResourceDownloadHash, "download");
        replace["STATIC_RESOURCE_VERIFY_CONTAINER"] = PathHolder_->GetStaticResourceContainerWithNameFromHash(staticResourceDownloadHash, "verify");

        replace.merge(
            NSupport::FillVerifyResourceReplaceMap(
                "static_resource"
                , resource.verification().checksum()
                , resource.verification().disabled()
            )
        );

        TMap<TString, TString> envKeyToValue;
        replace["STATIC_RESOURCE_DOWNLOAD_CMD"] = GetDownloadCMD(resource, secretMap, envKeyToValue, autoDecodeBase64Secrets);
        replace["STATIC_RESOURCE_SECRET_ENVIRONMENT"] = UseEnvInDownloadContainersForSecretsInFiles_
            ? NSupport::EnvironmentMapToString(envKeyToValue)
            : "";
        replace["NEED_CHECK_DOWNLOAD_PROGRESS"] =
            (resource.download_method_case() == API::TResource::DownloadMethodCase::kUrl && NSupport::IsWgetOrSkyget(resource.url()))
            || (resource.download_method_case() == API::TResource::DownloadMethodCase::kSkyGet)
            ? "true"
            : "false"
        ;

        Y_ENSURE(PathHolder_->HasVirtualDisk(resource.virtual_disk_id_ref()), "Unknown virtual disk ref: '" << resource.virtual_disk_id_ref() << "'");
        const TString place = PathHolder_->GetPlaceFromVirtualDisk(resource.virtual_disk_id_ref());

        replace["STATIC_RESOURCE_DOWNLOAD_CWD"] = PathHolder_->GetStaticResourceDirectoryFromHash(staticResourceDownloadHash, place);
        replace["STATIC_RESOURCE_DOWNLOAD_DIRECTORY"] = PathHolder_->GetStaticResourceDownloadDirectoryFromHash(staticResourceDownloadHash, place);
        replace["STATIC_RESOURCE_FINAL_PATH"] = PathHolder_->GetFinalStaticResourcePathFromHash(staticResourceDownloadHash, place);

        // Set default aging_time for download container because we do not track its state after success death
        replace["STATIC_RESOURCE_DOWNLOAD_AGING_TIME"] = "";
        replace["STATIC_RESOURCE_VERIFY_AGING_TIME"] = ToString(NSupport::CONTAINER_AGING_TIME_SECONDS);
        
        const TString groupId = resource.has_group_id() ? ToString(resource.group_id().value()) : "";
        replace["STATIC_RESOURCE_GROUP_ID"] = groupId;

        const EFileAccessMode resourceAccessMode = FromProto(resource.access_permissions(), resource.id());
 
        replace["STATIC_RESOURCE_ACCESS_MODE"] = resourceAccessMode == EFileAccessMode::Mode_UNMODIFIED ? "" :  ToString(resourceAccessMode);

        replace["STATIC_RESOURCE_DOWNLOAD_HASH"] = staticResourceDownloadHash;

        cachedReplace = replace;
        TBehavior3TemplateResolver().Resolve(resolvedTree, replace);
    }

    return TStaticResourceToAdd{
        TUpdateHolder::TStaticResourceTarget(
            TStaticResourceMeta(
                resource.id()
                , specTimestamp
                , revision
                , staticResourceDownloadHash
                , checkPeriodMs
            )
            , new TObjectTree(
                Logger_
                , TStatusNTickerHolder::GetStaticResourceTreeId(staticResourceDownloadHash)
                , TBehavior3EditorJsonReader(resolvedTree)
                    .WithPorto(Porto_)
                    .WithPosixWorker(PosixWorker_)
                    .WithStaticResourceStatusRepository(StatusNTickerHolder_->GetStaticResourceStatusRepository())
                    .WithTemplateBTStorage(TemplateBTStorage_)
                    .WithPathHolder(PathHolder_)
                    .BuildRootNode()
                , staticResourceDownloadHash
                , StatusNTickerHolder_->GetUpdateHolder()->GetUpdateHolderTarget()
                , StatusNTickerHolder_->GetStaticResourceStatusRepository()
            )
        )
        , fullHash
        , resource.virtual_disk_id_ref()
        , cachedReplace
    };
}

TString TStaticResourceTreeGenerator::GetDownloadCMD(
    const API::TResource& resource
    , const NSecret::TSecretMap& secretMap
    , TMap<TString, TString>& envKeyToValue
    , bool autoDecodeBase64Secrets
) const {
    switch (resource.download_method_case()) {
        case API::TResource::DownloadMethodCase::kFiles:
            return GetDownloadCMDFromFiles(resource.files(), secretMap, envKeyToValue, autoDecodeBase64Secrets);
        case API::TResource::DownloadMethodCase::kUrl:
            return GetDownloadCMDFromURL(resource.url());
        case API::TResource::DownloadMethodCase::kSkyGet:
            return GetDownloadCMDFromSkyGet(resource.sky_get());
        case API::TResource::DownloadMethodCase::DOWNLOAD_METHOD_NOT_SET:
            ythrow yexception() << "Resource download method not set";
    }
}

TString TStaticResourceTreeGenerator::GetDownloadCMDFromFiles(
    const API::TFiles& files
    , const NSecret::TSecretMap& secretMap
    , TMap<TString, TString>& envKeyToValue
    , bool autoDecodeBase64Secrets
) const {
    ValidateFiles(files.files());

    TStringBuilder finalCMD;
    finalCMD << "cd downloaded; ";

    for (auto i = 0; i < files.files_size(); ++i) {
        auto file = files.files(i);
        const TString fileName = file.file_name();
        TString fileContent;

        if (file.has_secret_data()) {
            if (UseEnvInDownloadContainersForSecretsInFiles_) {
                auto envKeyToValueIt = AddEnvToMap(envKeyToValue, fileName, i, NSecret::GetSecretValue(file.secret_data(), secretMap, autoDecodeBase64Secrets));
                fileContent = "$" + envKeyToValueIt->first;
            } else {
                fileContent = Wrap(Base64Encode(NSecret::GetSecretValue(file.secret_data(), secretMap, autoDecodeBase64Secrets)), R"('\'')");
            }
        } else if (file.has_multi_secret_data()) {
            if (UseEnvInDownloadContainersForSecretsInFiles_) {
                auto envKeyToValueIt = AddEnvToMap(envKeyToValue, fileName, i, NSecret::GetMultiSecretFileContent(file.multi_secret_data(), secretMap, autoDecodeBase64Secrets));
                fileContent = "$" + envKeyToValueIt->first;
            } else {
                fileContent = Wrap(Base64Encode(NSecret::GetMultiSecretFileContent(file.multi_secret_data(), secretMap, autoDecodeBase64Secrets)), R"('\'')");
            }
        } else {
            fileContent = Wrap(Base64Encode(file.raw_data()), R"('\'')");
        }

        finalCMD << "echo -n "
            << fileContent
            << " | base64 -d > "
            << Wrap(NSupport::EnshieldString(fileName), R"('\'')") << "; ";
    }
    return NSupport::WrapDownloadCommand(finalCMD);
}

TString TStaticResourceTreeGenerator::GetDownloadCMDFromURL(const TString& url) const {
    Y_ENSURE(url, "Url not provided");

    if (url.StartsWith(NSupport::HTTP_OR_HTTPS_PREFIX)) {
        Y_ENSURE(NSupport::IsUrl(url), "String '" << url << "' doesn't fit the url pattern");
        return NSupport::WrapDownloadCommand(TStringBuilder() << "wget --no-check-certificate " << url << " -P downloaded");
    }
    if (url.StartsWith(NSupport::RBTORRENT_PREFIX)) {
        Y_ENSURE(NSupport::IsRbtorrent(url), "String '" << url << "' doesn't fit the rbtorrent pattern");
        return NSupport::WrapDownloadCommand(TStringBuilder() << "sky get -p -d downloaded " << url);
    }
    if (url.StartsWith(NSupport::LOCAL_PREFIX)) {
        TFsPath filepath(url.substr(NSupport::LOCAL_PREFIX.length()));
        return NSupport::WrapDownloadCommand(TStringBuilder() << "cp " << filepath.RealPath() << " downloaded");
    }
    if (url.StartsWith(NSupport::RAW_PREFIX)) {
        TString base64FileContent = Base64Encode(url.substr(NSupport::RAW_PREFIX.size()));
        return NSupport::WrapDownloadCommand(
            TStringBuilder() << R"(cd downloaded; echo -n '\'')"
                << base64FileContent
                << R"('\'' | base64 -d > raw_file)"
        );
    }

    ythrow yexception() << "Unable to parse URL '" << url << "'";
}

EFileAccessMode TStaticResourceTreeGenerator::FromProto(const API::EResourceAccessPermissions& permissions, const TString& resourceId) {
    switch(permissions) {
        case API::EResourceAccessPermissions::EResourceAccessPermissions_UNMODIFIED:
            return EFileAccessMode::Mode_UNMODIFIED;
        case API::EResourceAccessPermissions::EResourceAccessPermissions_660:
            return EFileAccessMode::Mode_660;
        case API::EResourceAccessPermissions::EResourceAccessPermissions_600:
            return EFileAccessMode::Mode_600;
        default:
            ythrow yexception() << "Unable to convert resource access permissions from proto, resource id: '" << resourceId << "'";
    }
}

TString TStaticResourceTreeGenerator::GetDownloadCMDFromSkyGet(const API::TSkyGetDownload& skyGetDownload) const {
    return NSupport::WrapDownloadCommand(NSupport::GetSkyGetDownloadCommand(skyGetDownload, " downloaded"));
}

void TStaticResourceTreeGenerator::ValidateFiles(const ::google::protobuf::RepeatedPtrField<API::TFile>& files) {
    Y_ENSURE(files.size() > 0, "Files not provided");
    TSet<TString> fileNames;
    size_t filesWithSecretCount = 0;
    for (auto file : files) {
        const TString fileName = file.file_name();
        Y_ENSURE(!fileName.empty(), "empty filename");
        Y_ENSURE(!fileNames.contains(fileName), "two or more files have same filename: " << Quote(fileName));
        Y_ENSURE(fileName.find_first_of('/') == TString::npos, "filename " << Quote(fileName) << " contains /");
        fileNames.insert(fileName);

        if (file.has_secret_data() || file.has_multi_secret_data()) {
            ++filesWithSecretCount;
        }
    }
    Y_ENSURE(filesWithSecretCount <= TStaticResourceTreeGenerator::FILES_WITH_SECRET_MAX_COUNT, "number of files with secrets more than " << TStaticResourceTreeGenerator::FILES_WITH_SECRET_MAX_COUNT);
}

TMap<TString, TString>::iterator TStaticResourceTreeGenerator::AddEnvToMap(
    TMap<TString, TString>& envKeyToValue
    , const TString& fileName
    , size_t fileIndex
    , const TString& value
) {
    const TString envKey = SECRET_ENV_PREFIX + ToString(fileIndex);

    Y_ENSURE(
        value.size() <= FILE_WITH_SECRET_MAX_SYMBOLS
        , "Size of secret for file " << Quote(fileName)
            << " is greater than " << FILE_WITH_SECRET_MAX_SYMBOLS << " symbols: "
            << value.size()
    );
    return envKeyToValue.insert({envKey, Base64Encode(value)}).first;
}


} // namespace NInfra::NPodAgent
