#pragma once

#include "secret.h"

#include <infra/pod_agent/libs/behaviour/template_bt_storage/template_bt_storage.h>
#include <infra/pod_agent/libs/path_util/path_holder.h>
#include <infra/pod_agent/libs/pod_agent/status_and_ticker_holder/status_and_ticker_holder.h>
#include <infra/pod_agent/libs/porto_client/async_client.h>
#include <infra/pod_agent/libs/posix_worker/posix_worker.h>

namespace NInfra::NPodAgent {

/**
    Creates behavior tree for each new static resource from provided StaticResourceTreeTemplate
    Removes unneeded trees
    Ticks trees via TMtpPeriodTicker
    Method Wait() waits for signal from Shutdown() and waits the TMtpPeriodTicker
    Method Stop() calls {Shutdown(); Wait();}
*/
class TStaticResourceTreeGenerator {
public:
    TStaticResourceTreeGenerator(
        TLogger& logger
        , TPathHolderPtr pathHolder
        , const TBehavior3& staticResourceTreeTemplate
        , TAsyncPortoClientPtr porto
        , TPosixWorkerPtr posixWorker
        , TStatusNTickerHolderPtr statusNTickerHolder
        , TTemplateBTStoragePtr templateBTStorage
        , const TString& containerUser
        , const TString& containerGroup
        , bool useEnvInDownloadContainersForSecretsInFiles
    )
        : Logger_(logger)
        , PathHolder_(pathHolder)
        , StaticResourceTreeTemplate_(staticResourceTreeTemplate)
        , Porto_(porto)
        , PosixWorker_(posixWorker)
        , StatusNTickerHolder_(statusNTickerHolder)
        , TemplateBTStorage_(templateBTStorage)
        , ContainerUser_(containerUser)
        , ContainerGroup_(containerGroup)
        , UseEnvInDownloadContainersForSecretsInFiles_(useEnvInDownloadContainersForSecretsInFiles)
    {}

    ~TStaticResourceTreeGenerator() {}

public:
    struct TStaticResourceToAdd {
        TUpdateHolder::TStaticResourceTarget Target_;
        TString FullHash_;
        TString VirtualDiskIdRef_;
        // need for tests
        TMap<TString, TString> TreeReplaceMap_;
    };

    struct TStaticResourcesToAdd {
        TMap<TString, TStaticResourceToAdd> StaticResources_;
    };

    struct TCacheStaticResourcesToAdd {
        TMap<TStatusRepositoryCommon::TCacheObject, TStaticResourceToAdd> CacheStaticResources_;
    };

public:
    TStaticResourceTreeGenerator::TStaticResourcesToAdd UpdateSpec(
        const API::TResourceGang& spec
        , API::EPodAgentTargetState podAgentTargetState
        , const NSecret::TSecretMap& secretMap
        , const double cpuToVcpuFactor
        , ui64 specTimestamp
        , ui32 revision
        , bool autoDecodeBase64Secrets = false
    );
    TStaticResourceTreeGenerator::TCacheStaticResourcesToAdd UpdateResourceCache(
        const API::TPodAgentResourceCacheSpec& spec
        , API::EPodAgentTargetState podAgentTargetState
        , const NSecret::TSecretMap& secretMap
        , const API::TComputeResources& computeResources
        , const double cpuToVcpuFactor
        , bool autoDecodeBase64Secrets = false
    );

    void AddAndRemoveCacheStaticResources(const TCacheStaticResourcesToAdd& toAdd);

    void RemoveStaticResources(const TStaticResourcesToAdd& toAdd);
    void AddStaticResources(const TStaticResourcesToAdd& toAdd);

    // public for tests
    static EFileAccessMode FromProto(const API::EResourceAccessPermissions& permissions, const TString& resourceId);

protected:
    TStaticResourceToAdd GetStaticResourceToAdd(
        const API::TResource& resource
        , const NSecret::TSecretMap& secretMap
        , const API::TComputeResources& computeResources
        , const double cpuToVcpuFactor
        , ui64 specTimestamp
        , ui32 revision
        , bool autoDecodeBase64Secrets
    ) const;

    TString GetDownloadCMD(
        const API::TResource& resource
        , const NSecret::TSecretMap& secretMap
        , TMap<TString, TString>& envKeyToValue
        , bool autoDecodeBase64Secrets
    ) const;
    TString GetDownloadCMDFromFiles(
        const API::TFiles& files
        , const NSecret::TSecretMap& secretMap
        , TMap<TString, TString>& envKeyToValue
        , bool autoDecodeBase64Secrets
    ) const;
    TString GetDownloadCMDFromURL(const TString& url) const;
    TString GetDownloadCMDFromSkyGet(const API::TSkyGetDownload& skyGetDownload) const;

private:
    static void ValidateFiles(const ::google::protobuf::RepeatedPtrField<API::TFile>& files);
    static TMap<TString, TString>::iterator AddEnvToMap(TMap<TString, TString>& envKeyToValue, const TString& fileName, size_t fileIndex, const TString& value);

protected:
    inline static const size_t FILES_WITH_SECRET_MAX_COUNT = 150;
    inline static const size_t FILE_WITH_SECRET_MAX_SYMBOLS = 8192; // 64Kb or 8192 symbols
    inline static const TString SECRET_ENV_PREFIX = "SECRET_IN_FILE_";

private:
    TLogger& Logger_;
    TPathHolderPtr PathHolder_;
    const TBehavior3 StaticResourceTreeTemplate_;

    TAsyncPortoClientPtr Porto_;
    TPosixWorkerPtr PosixWorker_;
    TStatusNTickerHolderPtr StatusNTickerHolder_;
    TTemplateBTStoragePtr TemplateBTStorage_;

    const TString ContainerUser_;
    const TString ContainerGroup_;
    const bool UseEnvInDownloadContainersForSecretsInFiles_;
};

} // namespace NInfra::NPodAgent
