#include "static_resource_tree_generator.h"
#include "test_canon.h"
#include "test_functions.h"

#include <infra/pod_agent/libs/porto_client/mock_client.h>
#include <infra/pod_agent/libs/util/string_utils.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/stream/file.h>
#include <util/system/fs.h>
#include <util/system/shellcommand.h>

namespace NInfra::NPodAgent::NTreeGeneratorTest {

class TMockStaticResourceTreeGenerator : public TStaticResourceTreeGenerator {
public:
    TMockStaticResourceTreeGenerator(
        TLogger& logger
        , TPathHolderPtr pathHolder
        , const TBehavior3& staticResourceTreeTemplate
        , TAsyncPortoClientPtr porto
        , TPosixWorkerPtr posixWorker
        , TStatusNTickerHolderPtr statusNTickerHolder
        , TTemplateBTStoragePtr templateBTStorage
        , const TString& containerUser
        , const TString& containerGroup
        , bool useEnvInDownloadContainersForSecretsInFiles
    )
        : TStaticResourceTreeGenerator(
            logger
            , pathHolder
            , staticResourceTreeTemplate
            , porto
            , posixWorker
            , statusNTickerHolder
            , templateBTStorage
            , containerUser
            , containerGroup
            , useEnvInDownloadContainersForSecretsInFiles
        )
    {}

    TString GetDownloadCMD(
        const API::TResource& resource
        , const NSecret::TSecretMap& secretMap
        , TMap<TString, TString>& envKeyToValue
        , bool autoDecodeBase64Secrets
    ) const {
        return TStaticResourceTreeGenerator::GetDownloadCMD(resource, secretMap, envKeyToValue, autoDecodeBase64Secrets);
    }

    TString GetDownloadCMD(
        const API::TResource& resource
        , const NSecret::TSecretMap& secretMap
        , TMap<TString, TString>& envKeyToValue
    ) const {
        return TStaticResourceTreeGenerator::GetDownloadCMD(resource, secretMap, envKeyToValue, false);
    }

    TString GetDownloadCMD(
        const API::TResource& resource
        , const NSecret::TSecretMap& secretMap
    ) const {
        TMap<TString, TString> mockMap;
        return TStaticResourceTreeGenerator::GetDownloadCMD(resource, secretMap, mockMap, false);
    }

    size_t GetFileWithSecretMaxCount() const {
        return FILES_WITH_SECRET_MAX_COUNT;
    }

    size_t GetFileWithSecretMaxSymbols() const {
        return FILE_WITH_SECRET_MAX_SYMBOLS;
    }

    TString GetSecretEnvPrefix() const {
        return SECRET_ENV_PREFIX;
    }
};

using TMockStaticResourceTreeGeneratorPtr = TSimpleSharedPtr<TMockStaticResourceTreeGenerator>;

TMockStaticResourceTreeGeneratorPtr MakeWorker(
    TPathHolderPtr pathHolder
    , TStatusNTickerHolderPtr statusNTickerHolder
    , bool useEnvInDownloadContainersForSecretsInFiles = false
) {
    TMap<TString, TString> properties;
    properties["object_id_or_hash"] = "<RSLV:STATIC_RESOURCE_DOWNLOAD_HASH>";
    properties["state"] = "EStaticResourceState_READY";
    properties["object_type"] = "static_resource";

    return new TMockStaticResourceTreeGenerator(
        logger
        , pathHolder
        , GetTestTemplateTree(
            properties
            , "FeedbackObjectState"
            , "FeedbackObjectState(static_resource, EStaticResourceState_READY)"
        )
        , new TAsyncPortoClient(new TMockPortoClient(), new TFakeThreadPool())
        , new TPosixWorker(new TFakeThreadPool())
        , statusNTickerHolder
        , nullptr
        , "ContainerUser"
        , "ContainerGroup"
        , useEnvInDownloadContainersForSecretsInFiles
    );

}

class ITestStaticResourceTreeGeneratorCanon: public ITestTreeGeneratorCanon {
public:
    ITestStaticResourceTreeGeneratorCanon() : ITestTreeGeneratorCanon()
    {}

protected:
    void CreateWorker() final {
        Worker_ = MakeWorker(PathHolder_, StatusNTickerHolder_);
    }

    TStaticResourceTreeGenerator::TStaticResourcesToAdd UpdateSpec(
        const API::TResourceGang& spec
        , API::EPodAgentTargetState podAgentTargetState
        , ui64 specTimestamp
        , ui32 revision
        , const NSecret::TSecretMap& secretMap
    ) {
        TStaticResourceTreeGenerator::TStaticResourcesToAdd newSpec = Worker_->UpdateSpec(
            spec
            , podAgentTargetState
            , secretMap
            , 1.0 /* CpuToVcpuFactor */
            , specTimestamp
            , revision
        );
        Worker_->RemoveStaticResources(newSpec);
        Worker_->AddStaticResources(newSpec);
        return newSpec;
    }

    void UpdateResourceCache(
        const API::TPodAgentResourceCacheSpec& spec
        , API::EPodAgentTargetState podAgentTargetState
        , const NSecret::TSecretMap& secretMap
    ) {
        auto newSpec = Worker_->UpdateResourceCache(
            spec
            , podAgentTargetState
            , secretMap
            , API::TComputeResources()
            , 1.0 /* CpuToVcpuFactor */
        );
        Worker_->AddAndRemoveCacheStaticResources(newSpec);
    }

protected:
    TMockStaticResourceTreeGeneratorPtr Worker_;
};

TString MakeSecretEnvName(const TString& prefix, size_t fileIndex) {
    return prefix + ToString(fileIndex);
}

void AddMultiSecretFile(API::TResource& resource, const TString alias, const TString fileName) {
    API::TFile* multiSecretDataFilePtr = resource.mutable_files()->add_files();
    multiSecretDataFilePtr->mutable_multi_secret_data()->set_secret_alias(alias);
    multiSecretDataFilePtr->set_file_name(fileName);
}

Y_UNIT_TEST_SUITE(StaticResourceGangWorkerSuite) {

Y_UNIT_TEST(TestResourceAccessPermissionsFromProto) {
    UNIT_ASSERT_EQUAL(EFileAccessMode::Mode_UNMODIFIED, TStaticResourceTreeGenerator::FromProto(API::EResourceAccessPermissions::EResourceAccessPermissions_UNMODIFIED, "resource_id"));
    UNIT_ASSERT_EQUAL(EFileAccessMode::Mode_660, TStaticResourceTreeGenerator::FromProto(API::EResourceAccessPermissions::EResourceAccessPermissions_660, "resource_id"));
    UNIT_ASSERT_EQUAL(EFileAccessMode::Mode_600, TStaticResourceTreeGenerator::FromProto(API::EResourceAccessPermissions::EResourceAccessPermissions_600, "resource_id"));
}

Y_UNIT_TEST(TestResourceWithGroupId) {
    class TTest : public ITestStaticResourceTreeGeneratorCanon {
    public:
        TTest() : ITestStaticResourceTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TResourceGang spec;
            API::TResource* resource = spec.add_static_resources();
            resource->set_id("my_resource");
            resource->set_url("raw:some_data");
            resource->set_virtual_disk_id_ref("virtual_disk");
            API::TVerification* verification = resource->mutable_verification();
            verification->set_checksum("MD5:3ea05169a95365cbf9b4ff4688cd8945");

            auto resourcesToAdd = UpdateSpec(spec, API::EPodAgentTargetState_ACTIVE, 0, 0, {});
            UNIT_ASSERT(resourcesToAdd.StaticResources_.at("my_resource").TreeReplaceMap_.at("STATIC_RESOURCE_GROUP_ID").empty());
           
            resource->mutable_group_id()->set_value(0);

            resourcesToAdd= UpdateSpec(spec, API::EPodAgentTargetState_ACTIVE, 0, 0, {});
            UNIT_ASSERT_EQUAL(resourcesToAdd.StaticResources_.at("my_resource").TreeReplaceMap_.at("STATIC_RESOURCE_GROUP_ID"), "0");
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestResourceWithAccessPermissions) {
    class TTest : public ITestStaticResourceTreeGeneratorCanon {
    public:
        TTest() : ITestStaticResourceTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TResourceGang spec;
            API::TResource* resource = spec.add_static_resources();
            resource->set_id("my_resource");
            resource->set_url("raw:some_data");
            resource->set_virtual_disk_id_ref("virtual_disk");
            API::TVerification* verification = resource->mutable_verification();
            verification->set_checksum("MD5:3ea05169a95365cbf9b4ff4688cd8945");

            auto resourcesToAdd = UpdateSpec(spec, API::EPodAgentTargetState_ACTIVE, 0, 0, {});
            UNIT_ASSERT_EQUAL(resourcesToAdd.StaticResources_.at("my_resource").TreeReplaceMap_.at("STATIC_RESOURCE_ACCESS_MODE"), "");

            resource->set_access_permissions(API::EResourceAccessPermissions::EResourceAccessPermissions_600);

            resourcesToAdd = UpdateSpec(spec, API::EPodAgentTargetState_ACTIVE, 0, 0, {});
            UNIT_ASSERT_EQUAL(resourcesToAdd.StaticResources_.at("my_resource").TreeReplaceMap_.at("STATIC_RESOURCE_ACCESS_MODE"), "600");           
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestSimpleTree) {
    class TTest : public ITestStaticResourceTreeGeneratorCanon {
    public:
        TTest() : ITestStaticResourceTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TResourceGang spec;
            API::TResource* resource = spec.add_static_resources();
            resource->set_id("my_resource");
            resource->set_url("raw:some_data");
            resource->set_virtual_disk_id_ref("virtual_disk");
            API::TVerification* verification = resource->mutable_verification();
            verification->set_checksum("MD5:3ea05169a95365cbf9b4ff4688cd8945");

            auto resourceCopy = spec.add_static_resources();
            resourceCopy->CopyFrom(*resource);

            UNIT_ASSERT_EXCEPTION_CONTAINS(UpdateSpec(spec, API::EPodAgentTargetState_ACTIVE, 0, 0, {}), yexception, "ids are not unique");
            *resourceCopy->mutable_id() += "_other";
            auto resourcesToAdd = UpdateSpec(spec, API::EPodAgentTargetState_ACTIVE, 0, 0, {});
            *resourceCopy->mutable_id() = "";
            UNIT_ASSERT_EXCEPTION(UpdateSpec(spec, API::EPodAgentTargetState_ACTIVE, 0, 0, {}), yexception);

            *resourceCopy->mutable_id() = "my_resource_other";
            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec, API::EPodAgentTargetState_ACTIVE, 0, 0, {}));
            resource->set_virtual_disk_id_ref("bad_virtual_disk");
            UNIT_ASSERT_EXCEPTION_CONTAINS(UpdateSpec(spec, API::EPodAgentTargetState_ACTIVE, 0, 0, {}), yexception, "Unknown virtual disk ref");

            Sleep(TDuration::MilliSeconds(500));
            UNIT_ASSERT_EQUAL(API::EStaticResourceState_READY, StaticResourceStatusRepository_->GetObjectStatus("my_resource").state());
            UNIT_ASSERT(resourcesToAdd.StaticResources_.at("my_resource").Target_.Tree_->GetUseLongTickPeriod());
        }

    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestSimpleCacheStaticResourceTree) {
    class TTest : public ITestStaticResourceTreeGeneratorCanon {
    public:
        TTest() : ITestStaticResourceTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TPodAgentResourceCacheSpec spec;
            API::TCacheResource* cacheResource = spec.add_static_resources();
            cacheResource->set_revision(1);
            API::TResource* resource = cacheResource->mutable_resource();
            resource->set_id("my_resource");
            resource->set_url("raw:some_data");
            API::TVerification* verification = resource->mutable_verification();
            verification->set_checksum("MD5:3ea05169a95365cbf9b4ff4688cd8945");

            auto resourceCopy = spec.add_static_resources();
            resourceCopy->CopyFrom(*cacheResource);

            UNIT_ASSERT_EXCEPTION_CONTAINS(UpdateResourceCache(spec, API::EPodAgentTargetState_ACTIVE, {}), yexception, "ids and revisions are not unique");
            *resourceCopy->mutable_resource()->mutable_id() += "_other";
            UNIT_ASSERT_NO_EXCEPTION(UpdateResourceCache(spec, API::EPodAgentTargetState_ACTIVE, {}));
            *resourceCopy->mutable_resource()->mutable_id() = "";
            UNIT_ASSERT_EXCEPTION(UpdateResourceCache(spec, API::EPodAgentTargetState_ACTIVE, {}), yexception);
            *resourceCopy->mutable_resource()->mutable_id() = "my_resource_other";
            UNIT_ASSERT_NO_EXCEPTION(UpdateResourceCache(spec, API::EPodAgentTargetState_ACTIVE, {}));
            *resourceCopy->mutable_resource()->mutable_url() += "_other";
            UNIT_ASSERT_EXCEPTION_CONTAINS(UpdateResourceCache(spec, API::EPodAgentTargetState_ACTIVE, {}), yexception, "hash was changed");

            Sleep(TDuration::MilliSeconds(500));
            UNIT_ASSERT_EQUAL(API::EStaticResourceState_READY, StaticResourceStatusRepository_->GetCacheObjectStatus("my_resource", 1).state());
        }

    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveStaticResourceTree) {
    class TTest : public ITestStaticResourceTreeGeneratorCanon {
    public:
        TTest() : ITestStaticResourceTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TResourceGang spec;
            API::TResource* resource = spec.add_static_resources();
            resource->set_id("my_resource");
            resource->set_url("raw:some_data");
            API::TVerification* verification = resource->mutable_verification();
            verification->set_checksum("MD5:3ea05169a95365cbf9b4ff4688cd8945");

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec, API::EPodAgentTargetState_ACTIVE, 0, 0, {}));
            UNIT_ASSERT(StaticResourceStatusRepository_->HasObject("my_resource"));
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->StaticResourceHasTarget("my_resource"));

            resource->set_id("other_resource");

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec, API::EPodAgentTargetState_ACTIVE, 1, 1, {}));
            UNIT_ASSERT(StaticResourceStatusRepository_->HasObject("my_resource"));
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->StaticResourceHasTarget("my_resource"));
            UNIT_ASSERT(UpdateHolder_->GetAndRemoveStaticResourceTarget("my_resource").TargetRemove_);
            UNIT_ASSERT(StaticResourceStatusRepository_->HasObject("other_resource"));
        }

    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveCacheStaticResourceTree) {
    class TTest : public ITestStaticResourceTreeGeneratorCanon {
    public:
        TTest() : ITestStaticResourceTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TPodAgentResourceCacheSpec spec;
            API::TCacheResource* cacheResource = spec.add_static_resources();
            cacheResource->set_revision(1);
            API::TResource* resource = cacheResource->mutable_resource();
            resource->set_id("my_resource");
            resource->set_url("raw:some_data");
            API::TVerification* verification = resource->mutable_verification();
            verification->set_checksum("MD5:3ea05169a95365cbf9b4ff4688cd8945");

            UNIT_ASSERT_NO_EXCEPTION(UpdateResourceCache(spec, API::EPodAgentTargetState_ACTIVE, {}));
            UNIT_ASSERT(StaticResourceStatusRepository_->HasCacheObject("my_resource", 1));

            resource->set_id("other_resource");

            UNIT_ASSERT_NO_EXCEPTION(UpdateResourceCache(spec, API::EPodAgentTargetState_ACTIVE, {}));
            UNIT_ASSERT(!StaticResourceStatusRepository_->HasCacheObject("my_resource", 1));
            UNIT_ASSERT(StaticResourceStatusRepository_->HasCacheObject("other_resource", 1));
        }

    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestChangeTarget) {
    class TTest : public ITestStaticResourceTreeGeneratorCanon {
    public:
        TTest() : ITestStaticResourceTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TResourceGang spec;
            API::TResource* resource = spec.add_static_resources();
            resource->set_id("my_resource");
            resource->set_url("raw:some_data");
            API::TVerification* verification = resource->mutable_verification();
            verification->set_checksum("MD5:3ea05169a95365cbf9b4ff4688cd8945");

            for (size_t i = 0; i < 3; ++i) {
                UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec, API::EPodAgentTargetState_ACTIVE, i + 1, i + 2, {}));
                UNIT_ASSERT(StaticResourceStatusRepository_->HasObject("my_resource"));
                UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->StaticResourceHasTarget("my_resource"));
                UNIT_ASSERT(StaticResourceStatusRepository_->GetObjectStatus("my_resource").spec_timestamp() == i + 1);
                UNIT_ASSERT(StaticResourceStatusRepository_->GetObjectStatus("my_resource").revision() == i + 2);
            }

            // change layer
            resource->set_url("raw:some_other_data");

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec, API::EPodAgentTargetState_ACTIVE, 4, 5, {}));

            UNIT_ASSERT(StaticResourceStatusRepository_->HasObject("my_resource"));
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->StaticResourceHasTarget("my_resource"));
            UNIT_ASSERT(StaticResourceStatusRepository_->GetObjectStatus("my_resource").spec_timestamp() == 3);
            UNIT_ASSERT(StaticResourceStatusRepository_->GetObjectStatus("my_resource").revision() == 4);
        }

    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestPodAgentTargetStateWithStaticResourceTree) {
    class TTest : public ITestStaticResourceTreeGeneratorCanon {
    public:
        TTest() : ITestStaticResourceTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TResourceGang spec;
            API::TResource* resource = spec.add_static_resources();
            resource->set_id("my_resource");
            resource->set_url("raw:some_data");
            API::TVerification* verification = resource->mutable_verification();
            verification->set_checksum("MD5:3ea05169a95365cbf9b4ff4688cd8945");

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec, API::EPodAgentTargetState_ACTIVE, 0, 0, {}));
            UNIT_ASSERT(StaticResourceStatusRepository_->HasObject("my_resource"));
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->StaticResourceHasTarget("my_resource"));

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec, API::EPodAgentTargetState_SUSPENDED, 1, 1, {}));
            UNIT_ASSERT(StaticResourceStatusRepository_->HasObject("my_resource"));
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->StaticResourceHasTarget("my_resource"));

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec, API::EPodAgentTargetState_REMOVED, 2, 2, {}));
            UNIT_ASSERT(StaticResourceStatusRepository_->HasObject("my_resource"));
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->StaticResourceHasTarget("my_resource"));
            UNIT_ASSERT(UpdateHolder_->GetAndRemoveStaticResourceTarget("my_resource").TargetRemove_);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestPodAgentTargetStateWithCacheStaticResourceTree) {
    class TTest : public ITestStaticResourceTreeGeneratorCanon {
    public:
        TTest() : ITestStaticResourceTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TPodAgentResourceCacheSpec spec;
            API::TCacheResource* cacheResource = spec.add_static_resources();
            cacheResource->set_revision(1);
            API::TResource* resource = cacheResource->mutable_resource();
            resource->set_id("my_resource");
            resource->set_url("raw:some_data");
            API::TVerification* verification = resource->mutable_verification();
            verification->set_checksum("MD5:3ea05169a95365cbf9b4ff4688cd8945");

            UNIT_ASSERT_NO_EXCEPTION(UpdateResourceCache(spec, API::EPodAgentTargetState_ACTIVE, {}));
            UNIT_ASSERT(StaticResourceStatusRepository_->HasCacheObject("my_resource", 1));

            UNIT_ASSERT_NO_EXCEPTION(UpdateResourceCache(spec, API::EPodAgentTargetState_SUSPENDED, {}));
            UNIT_ASSERT(StaticResourceStatusRepository_->HasCacheObject("my_resource", 1));

            UNIT_ASSERT_NO_EXCEPTION(UpdateResourceCache(spec, API::EPodAgentTargetState_REMOVED, {}));
            UNIT_ASSERT(!StaticResourceStatusRepository_->HasCacheObject("my_resource", 1));
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestGetDownloadCMDFromURL) {
    class TTest : public ITestStaticResourceTreeGeneratorCanon {
    public:
        TTest() : ITestStaticResourceTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            TFsPath("data.txt").Touch();
            const TVector<std::pair<TString, TString>> validItems = {
                {
                    "https://proxy.sandbox.yandex-team.ru/586812773"
                    , R"(bash -e -c 'rm -rf downloaded; rm -rf downloaded_result; mkdir downloaded; wget --no-check-certificate https://proxy.sandbox.yandex-team.ru/586812773 -P downloaded')"
                },
                {
                    "rbtorrent:73b5ea60d641225a71571d38dd8975a4fdf86825"
                    , TStringBuilder() << R"(bash -e -c 'rm -rf downloaded; rm -rf downloaded_result; mkdir downloaded; sky get -p -d downloaded rbtorrent:73b5ea60d641225a71571d38dd8975a4fdf86825')"
                },
                {
                    "local:data.txt"
                    , "bash -e -c 'rm -rf downloaded; rm -rf downloaded_result; mkdir downloaded; cp " + NFs::CurrentWorkingDirectory() + "/data.txt downloaded'"
                },
                {
                    "raw: You think you can get away, Rose? You stood your ground on that little speck called Earth, but you’re on our world now…"
                    , R"(bash -e -c 'rm -rf downloaded; rm -rf downloaded_result; mkdir downloaded; cd downloaded; echo -n '\''IFlvdSB0aGluayB5b3UgY2FuIGdldCBhd2F5LCBSb3NlPyBZb3Ugc3Rvb2QgeW91ciBncm91bmQgb24gdGhhdCBsaXR0bGUgc3BlY2sgY2FsbGVkIEVhcnRoLCBidXQgeW914oCZcmUgb24gb3VyIHdvcmxkIG5vd+KApg=='\'' | base64 -d > raw_file')"
                }
            };

            const TVector<TString> invalidUrls = {
                "htt\\ps://proxy.sandbox.yandex-team.ru/586812773"
                , "htts//proxy.sandbox.yandex-team.ru/586812773"
                , "https://proxy"
                , "http:///proxy.sandbox.yandex-team.ru/586812773"
                , "https:73b5ea60d641225a71571d38dd8975a4fdf86825"
                , "https://proxy.sandbox.yandex-team.ru/586812773 \"; mkdir directory; \""
                , "https:proxy.sandbox.yandex-team.ru/586812773"

                , "rtorrent:73b5ea60d641225a71571d38dd8975a4fdf86825"
                , "Rbtorrent:73b5ea60d641225a71571d38dd8975a4fdf86825"
                , "rbtorrent:73b5ea60d641225a71'571d38dd8975a4fdf86825"
                , "rbtorrent:73b5ea60d641225a7\"; pwd;\"1571d38dd8975a4fdf86825"
                , "rbtorrent:73b5Ёa60d641225a71571d38dd8975a4f"

                , "local:dir/layer.tar.gz"
                , "locl:data.txt"
                , "local:dir'layer.tar.gz"
                , "local:file;rm -rf /*"

                , "ruw: some_data"
                , "Raw: some_data"
                , "raW: some_data"
                , "raw : some_data"
                , ":raw: some_data"
                , "r\aw: some_data"
                , ""
            };

            for (auto& item : validItems) {
                API::TResource resource;
                resource.set_url(item.first);
                auto command = Worker_->GetDownloadCMD(resource, {});
                UNIT_ASSERT_EQUAL_C(command, item.second, "Commands are different, expected '" << item.second << "', got '" << command << "'");
            }

            for (auto& invalidUrl : invalidUrls) {
                API::TResource resource;
                resource.set_url(invalidUrl);
                UNIT_ASSERT_EXCEPTION_C(Worker_->GetDownloadCMD(resource, {}), yexception, invalidUrl);
            }
        }

    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestGetDownloadCMDFromFilesWithOldProtocolForSecretFiles) {
    class TTest : public ITestStaticResourceTreeGeneratorCanon {
    public:
        TTest() : ITestStaticResourceTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TResource resource;

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                Worker_->GetDownloadCMD(resource, {})
                , yexception
                , "Resource download method not set"
            );

            API::TFile* rawDataFilePtr = resource.mutable_files()->add_files();
            rawDataFilePtr->set_raw_data("raw data");
            rawDataFilePtr->set_file_name("file1");
            API::TFile* secretDataFilePtr = resource.mutable_files()->add_files();
            secretDataFilePtr->mutable_secret_data()->set_alias("alias");
            secretDataFilePtr->mutable_secret_data()->set_id("id");
            secretDataFilePtr->set_file_name("file2");
            API::TFile* multiSecretDataFilePtr = resource.mutable_files()->add_files();
            multiSecretDataFilePtr->mutable_multi_secret_data()->set_secret_alias("alias");
            multiSecretDataFilePtr->set_file_name("file3");

            NSecret::TSecretMap secretMap;
            secretMap["alias"]["id"] = {
                "secret data"
                , ""
            };
            secretMap["alias"]["id2"] = {
                "secret data2"
                , ""
            };

            auto command = Worker_->GetDownloadCMD(resource, secretMap);
            UNIT_ASSERT_EQUAL_C(
                command
                , R"(bash -e -c 'rm -rf downloaded; rm -rf downloaded_result; mkdir downloaded; cd downloaded; echo -n '\''cmF3IGRhdGE='\'' | base64 -d > '\''file1'\''; echo -n '\''c2VjcmV0IGRhdGE='\'' | base64 -d > '\''file2'\''; echo -n '\''ewogICAgImlkIjoic2VjcmV0IGRhdGEiLAogICAgImlkMiI6InNlY3JldCBkYXRhMiIKfQ=='\'' | base64 -d > '\''file3'\''; ')"
                , command
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestGetDownloadCMDFromFilesWithNewProtocolForSecretFiles) {
    class TTest : public ITestStaticResourceTreeGeneratorCanon {
    public:
        TTest() : ITestStaticResourceTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            Worker_ = MakeWorker(PathHolder_, StatusNTickerHolder_, true);

            API::TResource resource;
            NSecret::TSecretMap secretMap;

            secretMap["alias"]["id"] = {"c2VjcmV0IGRhdGE=", NSecret::BASE64_ENCODING};
            secretMap["alias"]["id2"] = {"secret data2", ""};

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                Worker_->GetDownloadCMD(resource, {})
                , yexception
                , "Resource download method not set"
            );

            TString firstFileName = "file1";
            TString secondFileName = "file2";
            API::TFile* rawDataFilePtr = resource.mutable_files()->add_files();
            rawDataFilePtr->set_raw_data("raw data");
            rawDataFilePtr->set_file_name(firstFileName);
            API::TFile* secretDataFilePtr = resource.mutable_files()->add_files();
            secretDataFilePtr->mutable_secret_data()->set_alias("alias");
            secretDataFilePtr->mutable_secret_data()->set_id("id");
            secretDataFilePtr->set_file_name(secondFileName);
            AddMultiSecretFile(resource, "alias", "file3");

            TString correctCommand = R"(bash -e -c 'rm -rf downloaded; rm -rf downloaded_result; mkdir downloaded; cd downloaded; echo -n '\''cmF3IGRhdGE='\'' | base64 -d > '\''file1'\''; echo -n $SECRET_IN_FILE_1 | base64 -d > '\''file2'\''; echo -n $SECRET_IN_FILE_2 | base64 -d > '\''file3'\''; ')";

            {
                TMap<TString, TString> envKeyToValue;
                TString singleSecretEncodedValueWithDisabledDecodingSecrets = "YzJWamNtVjBJR1JoZEdFPQ==";
                TString multiSecretEncodedValueWithDisabledDecodingSecrets = "ewogICAgImlkIjoiYzJWamNtVjBJR1JoZEdFPSIsCiAgICAiaWQyIjoic2VjcmV0IGRhdGEyIgp9";
                TMap<TString, TString> correctEnvKeyToValue = {
                    {"SECRET_IN_FILE_1", singleSecretEncodedValueWithDisabledDecodingSecrets}
                    , {"SECRET_IN_FILE_2", multiSecretEncodedValueWithDisabledDecodingSecrets}
                };

                auto command = Worker_->GetDownloadCMD(resource, secretMap, envKeyToValue);
                UNIT_ASSERT_EQUAL_C(command, correctCommand, command);
                UNIT_ASSERT_EQUAL(envKeyToValue, correctEnvKeyToValue);
            }

            TMap<TString, TString> envKeyToValue;
            TString singleSecretEncodedValue = "c2VjcmV0IGRhdGE=";
            TString multiSecretEncodedValue = "ewogICAgImlkIjoic2VjcmV0IGRhdGEiLAogICAgImlkMiI6InNlY3JldCBkYXRhMiIKfQ==";
            TMap<TString, TString> correctEnvKeyToValue = {
                {"SECRET_IN_FILE_1", singleSecretEncodedValue}
                , {"SECRET_IN_FILE_2", multiSecretEncodedValue}
            };
            TString command = Worker_->GetDownloadCMD(resource, secretMap, envKeyToValue, /* autoDecodeBase64Secrets = */ true);
            UNIT_ASSERT_EQUAL_C(command, correctCommand, command);
            UNIT_ASSERT_EQUAL(envKeyToValue, correctEnvKeyToValue);

            rawDataFilePtr->set_file_name("");
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                Worker_->GetDownloadCMD(resource, secretMap, envKeyToValue)
                , yexception
                , "empty filename"
            );

            rawDataFilePtr->set_file_name("dir/file");
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                Worker_->GetDownloadCMD(resource, secretMap, envKeyToValue)
                , yexception
                , "filename 'dir/file' contains /"
            );

            rawDataFilePtr->set_file_name(firstFileName);
            secretDataFilePtr->set_file_name(firstFileName);
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                Worker_->GetDownloadCMD(resource, secretMap, envKeyToValue)
                , yexception
                , "two or more files have same filename: " + Quote(firstFileName)
            );

            secretDataFilePtr->set_file_name(secondFileName);

            {
                // Test big secret
                TString sourceSecretValue = secretMap["alias"]["id"].Value_;
                TStringBuilder bigSecret;
                for (int i = 0; i < 741; i++) {
                    bigSecret << sourceSecretValue;
                }
                secretMap["alias"]["id"].Value_ = bigSecret;
                UNIT_ASSERT_NO_EXCEPTION(Worker_->GetDownloadCMD(resource, secretMap, envKeyToValue, /* autoDecodeBase64Secrets = */ true));

                bigSecret << sourceSecretValue;
                secretMap["alias"]["id"].Value_ = bigSecret;

                UNIT_ASSERT_EXCEPTION_CONTAINS(
                    Worker_->GetDownloadCMD(resource, secretMap, envKeyToValue, /* autoDecodeBase64Secrets = */ true)
                    , yexception
                    , "Size of secret for file 'file3' is greater than "
                        + ToString(Worker_->GetFileWithSecretMaxSymbols())
                        + " symbols: 8203"
                );

                secretMap["alias"]["id"].Value_ = sourceSecretValue;
            }

            {
                // Test big count of secrets

                // remove quote
                correctCommand.pop_back();

                for (size_t i = 0; i < Worker_->GetFileWithSecretMaxCount() - 2; ++i) {
                    TString newFileName = "file3" + ToString(i);
                    size_t newFileIndex = resource.files().files_size();
                    TString envName = MakeSecretEnvName(Worker_->GetSecretEnvPrefix(), newFileIndex);
                    AddMultiSecretFile(resource, "alias", newFileName);

                    correctCommand += "echo -n $" + envName + R"( | base64 -d > '\'')" + newFileName + R"('\''; )";
                    correctEnvKeyToValue[envName] = multiSecretEncodedValue;
                }
                correctCommand.push_back('\'');

                command = Worker_->GetDownloadCMD(resource, secretMap, envKeyToValue, /* autoDecodeBase64Secrets = */ true);

                UNIT_ASSERT_EQUAL_C(command, correctCommand, command);
                UNIT_ASSERT_EQUAL(envKeyToValue, correctEnvKeyToValue);

                AddMultiSecretFile(resource, "alias", "extra_file");

                UNIT_ASSERT_EXCEPTION_CONTAINS(
                    Worker_->GetDownloadCMD(resource, secretMap, envKeyToValue, /* autoDecodeBase64Secrets = */ true)
                    , yexception
                    , "number of files with secrets more than " + ToString(Worker_->GetFileWithSecretMaxCount())
                );
            }
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestGetDownloadCMDFromSkyGet) {
    class TTest : public ITestStaticResourceTreeGeneratorCanon {
    public:
        TTest() : ITestStaticResourceTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TResource resource;

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                Worker_->GetDownloadCMD(resource, {})
                , yexception
                , "Resource download method not set"
            );

            resource.mutable_sky_get()->set_resid("rbtorrent:1bc29b36f623ba82aaf6724fd3b16718");

            auto command = Worker_->GetDownloadCMD(resource, {});
            UNIT_ASSERT_EQUAL_C(
                command
                , R"(bash -e -c 'rm -rf downloaded; rm -rf downloaded_result; mkdir downloaded; sky get -p -d  downloaded rbtorrent:1bc29b36f623ba82aaf6724fd3b16718')"
                , command
            );
        }
    };

    TTest test;
    test.DoTest();
}

}

} // NInfra::NPodAgent::NTreeGeneratorTest
