#include "support_functions.h"

#include <infra/pod_agent/libs/behaviour/loaders/behavior3_editor_json_reader.h>
#include <infra/pod_agent/libs/pod_agent/trees_generators/types/trees_generators_types.h>

#include <library/cpp/digest/md5/md5.h>
#include <library/cpp/uri/uri.h>

#include <util/string/cast.h>
#include <util/string/join.h>
#include <util/string/type.h>
#include <util/string/vector.h>

namespace NInfra::NPodAgent::NSupport {

namespace {

const ui64 DEFAULT_MIN_RESTART_PERIOD_MS = 30000;
const ui64 DEFAULT_MAX_RESTART_PERIOD_MS = UINT64_MAX;
const ui64 DEFAULT_MAX_EXECUTION_TIME_MS = 1800000;

const ui64 DEFAULT_INITIAL_DELAY_MS = 5000;

const double MIN_POSSIBLE_CPU_TO_VCPU_FACTOR = 1e-2;

const TString MD5_PREFIX = "MD5";
const TString EMPTY_PREFIX = "EMPTY";
const TString SHA256_PREFIX = "SHA256";

namespace NSystemEnvironment {

const TString TVMTOOL_LOCAL_AUTHTOKEN = "TVMTOOL_LOCAL_AUTHTOKEN";
const TString DEPLOY_POD_ID = "DEPLOY_POD_ID";
const TString DEPLOY_POD_PERSISTENT_FQDN = "DEPLOY_POD_PERSISTENT_FQDN";
const TString DEPLOY_POD_TRANSIENT_FQDN = "DEPLOY_POD_TRANSIENT_FQDN";
const TString DEPLOY_NODE_CLUSTER = "DEPLOY_NODE_CLUSTER";
const TString DEPLOY_NODE_DC = "DEPLOY_NODE_DC";
const TString DEPLOY_NODE_FQDN = "DEPLOY_NODE_FQDN";

const TString DEPLOY_POD_IP_ADDRESS_PREFIX = "DEPLOY_POD_IP_";
const TString DEPLOY_POD_IP_ADDRESS_SUFFIX = "_ADDRESS";

const TString DEPLOY_CONTAINER_ID = "DEPLOY_CONTAINER_ID";

const THashSet<TString> ALL_CONST_NAMES = {
    TVMTOOL_LOCAL_AUTHTOKEN
    , DEPLOY_POD_ID
    , DEPLOY_POD_PERSISTENT_FQDN
    , DEPLOY_NODE_CLUSTER
    , DEPLOY_NODE_DC
    , DEPLOY_NODE_FQDN
    , DEPLOY_CONTAINER_ID
};

void ValidateNotSystemEnvironmentVariable(
    const TString& name
) {
    Y_ENSURE(
        !NSystemEnvironment::ALL_CONST_NAMES.contains(name)
        , name << " is not allowed at environment variable name"
    );
    Y_ENSURE(
        !(name.StartsWith(NSystemEnvironment::DEPLOY_POD_IP_ADDRESS_PREFIX) && name.EndsWith(NSystemEnvironment::DEPLOY_POD_IP_ADDRESS_SUFFIX))
        , NSystemEnvironment::DEPLOY_POD_IP_ADDRESS_PREFIX << "*" << NSystemEnvironment::DEPLOY_POD_IP_ADDRESS_SUFFIX
            << " is not allowed at environment variable name, but variable '" << name << "' found"
    );
}

}

TString EscapeCharactersForEnvironment(const TString& str) {
    TString ret = "";
    for (auto symb : str) {
        if (symb == ';' || symb == '\\') {
            ret.push_back('\\');
        }
        ret.push_back(symb);
    }

    return ret;
};

bool IsSHA256(const TString& checksum) {
    return checksum.size() == 64
        && checksum.find_first_not_of("0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ") == TString::npos
    ;
}

TString GetFullPath(const TString& rootfsPath, const TString& path) {
    if (path == "") {
        return "";
    } else if (rootfsPath.EndsWith("/")) {
        return rootfsPath + path;
    } else {
        return rootfsPath + "/" + path;
    }
}

TMap<TString, TString> FillContainerReplaceMapImpl(
    const API::TUtilityContainer& container
    , const TString& name
    , const TString& portoContainerName
    , const TString& rootfsPath
    , const TString& podAgentBinaryFilePath
    , const google::protobuf::RepeatedPtrField<API::TEnvVar>& env
    , const NSecret::TSecretMap& secretMap
    , const TPathHolderPtr pathHolder
    , const double cpuToVcpuFactor
    , bool needTimeLimit
    , bool needValidation
    , bool needDefaultInitialDelay
    , bool useEnvSecret = false
    , bool autoDecodeBase64Secrets = false
) {
    TMap<TString, TString> result;
    const TString prefix = to_upper(name) + '_';
    if (needValidation) {
        Y_ENSURE(!container.command_line().empty(), name << " container command_line is empty");
        Y_ENSURE(!IsSpace(container.command_line().substr(0, 1)), name << " container command_line '" << container.command_line() << "' starts with space character");
        Y_ENSURE(!IsSpace(container.command_line().substr(container.command_line().size() - 1, 1)), name << " container command_line '" << container.command_line() << "' ends with space character");
    }

    result[prefix + "CONTAINER"] = portoContainerName;
    NSupport::TEnvironmentList envList = GetEnvironmentList(
        env
        , secretMap
        , GetContainerSpecificSystemEnvironment(
            portoContainerName
        )
        , useEnvSecret
        , autoDecodeBase64Secrets
    );
    result[prefix + "ENVIRONMENT"] = envList.PublicEnv_;
    result[prefix + "SECRET_ENVIRONMENT"] = envList.SecretEnv_;

    result[prefix + "STDOUT_LOG_PATH"] = container.stdout_file();
    result[prefix + "STDERR_LOG_PATH"] = container.stderr_file();
    result[prefix + "STDOUT_LOG_FILE_FULL_PATH_TO_CREATE"] = GetFullPath(rootfsPath, container.stdout_file());
    result[prefix + "STDERR_LOG_FILE_FULL_PATH_TO_CREATE"] = GetFullPath(rootfsPath, container.stderr_file());
    if (container.stdout_and_stderr_limit() != 0) {
        result[prefix + "STDOUT_AND_STDERR_FILE_SIZE_LIMIT"] = ToString(container.stdout_and_stderr_limit());
    } else {
        result[prefix + "STDOUT_AND_STDERR_FILE_SIZE_LIMIT"] = "";
    }
    result[prefix + "CWD"] = container.cwd();
    result[prefix + "CORE_COMMAND"] = container.core_command();

    {
        const TString user = (container.user().empty() ? "root" : container.user());
        const TString group = (container.group().empty() ? "root" : container.group());

        // We need root privileges to call setuid/setgid/initgroups in exec_wrapper
        result[prefix + "USER"] = "root";
        result[prefix + "GROUP"] = "root";

        if (user == "root" && group == "root") {
            // User and group are root
            // Just simple exec
            result[prefix + "CMD"] = container.command_line();
        } else {
            // User or group is not a root
            // Use exec_wrapper to change them inside box environment
            result[prefix + "CMD"] = TStringBuilder()
                << podAgentBinaryFilePath << " "
                << "exec_wrapper" << " "
                << user << " "
                << group << " "
                << container.command_line()
            ;
        }
    }

    result[prefix + "AGING_TIME"] = ToString(CONTAINER_AGING_TIME_SECONDS);

    result.merge(
        FillComputeResourcesReplaceMap(
            container.compute_resources()
            , prefix
            , pathHolder
            , cpuToVcpuFactor
        )
    );

    if (needTimeLimit) {
        result.merge(FillTimeLimitReplaceMap(container.time_limit(), prefix, needDefaultInitialDelay));
    }

    return result;
}

TString ContainerULimitToString(API::EContainerULimitType ulimit) {
    switch (ulimit) {
        case API::EContainerULimit_AS:
            return "as";
        case API::EContainerULimit_CORE:
            return "core";
        case API::EContainerULimit_DATA:
            return "data";
        case API::EContainerULimit_MEMLOCK:
            return "memlock";
        case API::EContainerULimit_STACK:
            return "stack";
        case API::EContainerULimit_FSIZE:
            return "fsize";
        case API::EContainerULimit_NOFILE:
            return "nofile";
        case API::EContainerULimit_CPU:
            return "cpu";
        case API::EContainerULimit_LOCKS:
            return "locks";
        case API::EContainerULimit_MSGQUEUE:
            return "msgqueue";
        case API::EContainerULimit_NICE:
            return "nice";
        case API::EContainerULimit_NPROC:
            return "nproc";
        case API::EContainerULimit_RSS:
            return "rss";
        case API::EContainerULimit_RTPRIO:
            return "rtprio";
        case API::EContainerULimit_RTTIME:
            return "rttime";
        case API::EContainerULimit_SIGPENDING:
            return "sigpending";
        // We don't want to use default
        // So we need this to handle all cases
        case API::EContainerULimitType_INT_MIN_SENTINEL_DO_NOT_USE_:
        case API::EContainerULimitType_INT_MAX_SENTINEL_DO_NOT_USE_:
            ythrow yexception() << "Unexpected Ulimit name '" << API::EContainerULimitType_Name(ulimit) << "' for switch";
    }
}

TString CpuPolicyToString(API::ECpuPolicy cpuPolicy) {
    switch (cpuPolicy) {
        case API::ECpuPolicy_UNKNOWN:
            return "";
        case API::ECpuPolicy_NORMAL:
            return "normal";
        case API::ECpuPolicy_HIGH:
            return "high";
        case API::ECpuPolicy_BATCH:
            return "batch";
        case API::ECpuPolicy_IDLE:
            return "idle";
        case API::ECpuPolicy_ISO:
            return "iso";
        // We don't want to use default
        // So we need this to handle all cases
        case API::ECpuPolicy_INT_MIN_SENTINEL_DO_NOT_USE_:
        case API::ECpuPolicy_INT_MAX_SENTINEL_DO_NOT_USE_:
            ythrow yexception() << "Unexpected cpu policy '" << API::ECpuPolicy_Name(cpuPolicy) << "' for switch";
    }
}

TString IoPolicyToString(API::EIoPolicy ioPolicy) {
    switch (ioPolicy) {
        case API::EIoPolicy_UNKNOWN:
            return "";
        case API::EIoPolicy_NONE:
            return "none";
        case API::EIoPolicy_NORMAL:
            return "normal";
        case API::EIoPolicy_HIGH:
            return "high";
        case API::EIoPolicy_BATCH:
            return "batch";
        case API::EIoPolicy_IDLE:
            return "idle";
        // We don't want to use default
        // So we need this to handle all cases
        case API::EIoPolicy_INT_MIN_SENTINEL_DO_NOT_USE_:
        case API::EIoPolicy_INT_MAX_SENTINEL_DO_NOT_USE_:
            ythrow yexception() << "Unexpected io policy '" << API::EIoPolicy_Name(ioPolicy) << "' for switch";
    }
}

TString SkyGetDeduplicateModeToString(API::ESkyGetDeduplicateMode skyGetDeduplicateMode) {
    switch (skyGetDeduplicateMode) {
        case API::ESkyGetDeduplicateMode_NO:
            return "No";
        case API::ESkyGetDeduplicateMode_HARDLINK:
            return "Hardlink";
        case API::ESkyGetDeduplicateMode_SYMLINK:
            return "Symlink";
        // We don't want to use default
        // So we need this to handle all cases
        case API::ESkyGetDeduplicateMode_INT_MIN_SENTINEL_DO_NOT_USE_:
        case API::ESkyGetDeduplicateMode_INT_MAX_SENTINEL_DO_NOT_USE_:
            ythrow yexception() << "Unexpected sky get deduplicate mode '" << API::ESkyGetDeduplicateMode_Name(skyGetDeduplicateMode) << "' for switch";
    }
}

TString DefaultIoLimitTargetToString(
    const API::EDefaultIoLimitTarget defaultIoLimitTarget
) {
    switch (defaultIoLimitTarget) {
        case API::EDefaultIoLimitTarget_NONE:
            ythrow yexception() << "Io limit default target is none";
        case API::EDefaultIoLimitTarget_PLACE:
            return "/place";
        case API::EDefaultIoLimitTarget_SSD:
            return "/ssd";
        // We don't want to use default
        // So we need this to handle all cases
        case API::EDefaultIoLimitTarget_INT_MIN_SENTINEL_DO_NOT_USE_:
        case API::EDefaultIoLimitTarget_INT_MAX_SENTINEL_DO_NOT_USE_:
            ythrow yexception() << "Unexpected default io limit target '" << API::EDefaultIoLimitTarget_Name(defaultIoLimitTarget) << "' for switch";
    }
}

TString IoLimitsToString(
    const google::protobuf::RepeatedPtrField<API::TIoLimit>& ioLimits
    , const TPathHolderPtr pathHolder
) {
    TStringBuilder result;

    for (const auto& ioLimit : ioLimits) {
        TString target = "";
        switch (ioLimit.target_case()) {
            case API::TIoLimit::TargetCase::kRaw:
                target = ioLimit.raw();
                break;
            case API::TIoLimit::TargetCase::kDefaultTarget:
                target = DefaultIoLimitTargetToString(ioLimit.default_target());
                break;
            case API::TIoLimit::TargetCase::kVirtualDiskIdRef:
                Y_ENSURE(!ioLimit.virtual_disk_id_ref().empty(), "Empty virtual disk ref is forbidden at io limit target");
                Y_ENSURE(pathHolder->HasVirtualDisk(ioLimit.virtual_disk_id_ref()), "Unknown virtual disk ref at io limit target: '" << ioLimit.virtual_disk_id_ref() << "'");
                target = pathHolder->GetDom0PlaceFromVirtualDisk(ioLimit.virtual_disk_id_ref());
                break;
            case API::TIoLimit::TargetCase::TARGET_NOT_SET:
                ythrow yexception() << "Io limit target not set";
        };

        TString type = "";
        switch (ioLimit.type()) {
            case API::EIoLimitType_READ:
                type = "r";
                break;
            case API::EIoLimitType_WRITE:
                type = "w";
                break;
            // We don't want to use default
            // So we need this to handle all cases
            case API::EIoLimitType_INT_MIN_SENTINEL_DO_NOT_USE_:
            case API::EIoLimitType_INT_MAX_SENTINEL_DO_NOT_USE_:
                ythrow yexception() << "Unexpected io type '" << API::EIoLimitType_Name(ioLimit.type()) << "' for switch";
        }

        const TString& value = ToString(ioLimit.value());

        result << target << " " << type << ": " << value << ";";
    }

    if (!result.empty()) {
        result.pop_back(); // remove last ";"
    }

    return result;
}

} // namespace

TString CgroupFsMountModeToString(API::ECgroupFsMountMode cgroupFsMountMode) {
    switch (cgroupFsMountMode) {
        case API::ECgroupFsMountMode_NONE:
            return "";
        case API::ECgroupFsMountMode_RO:
            return "ro";
        case API::ECgroupFsMountMode_RW:
            return "rw";
        // We don't want to use default
        // So we need this to handle all cases
        case API::ECgroupFsMountMode_INT_MIN_SENTINEL_DO_NOT_USE_:
        case API::ECgroupFsMountMode_INT_MAX_SENTINEL_DO_NOT_USE_:
            ythrow yexception() << "Unexpected cgroupfs '" << API::ECgroupFsMountMode_Name(cgroupFsMountMode) << "' for switch";
    }
}

TString WrapDownloadCommand(const TString& command) {
    return TStringBuilder() << "bash -e -c 'rm -rf downloaded; rm -rf downloaded_result; mkdir downloaded; " << command << "'";
}

TString GetVerificationCommand(const TString& checksum) {
    if (checksum.StartsWith(MD5_PREFIX)) {
        return "bash -e -c 'export LC_COLLATE=C;find downloaded -type f -print0 | sort -z | xargs -0 cat | md5sum -b | cut -d\" \" -f 1 | awk \"{printf \\$0}\"'";
    }

    if (checksum.StartsWith(SHA256_PREFIX)) {
        return "bash -e -c 'export LC_COLLATE=C;find downloaded -type f -print0 | sort -z | xargs -0 cat | sha256sum -b | cut -d\" \" -f 1 | awk \"{printf \\$0}\"'";
    }

    ythrow yexception() << "unable to define suitable command from '" << checksum << "'";
}

TString EnshieldString(const TString& str) {
    TString shieldedStr;
    for (auto symbol : str) {
        if (symbol == '\'') {
            shieldedStr.append("'\\'\\\\\\'\\''");
        } else {
            shieldedStr.append(symbol);
        }
    }
    return shieldedStr;
}

TString GetVerificationChecksumValue(const TString& checksum) {
    if (checksum.StartsWith(MD5_PREFIX)) {
        TVector<TString> components;
        StringSplitter(checksum).Split(':').SkipEmpty().Collect(&components);

        if (components.size() != 2
            || components[0] != MD5_PREFIX
            || !MD5::IsMD5(components[1]))
        {
            ythrow yexception() << "'" << checksum << "' is not a valid MD5 hash";
        }

        return components[1];
    }

    if (checksum.StartsWith(SHA256_PREFIX)) {
        TVector<TString> components;
        StringSplitter(checksum).Split(':').SkipEmpty().Collect(&components);

        if (components.size() != 2
            || components[0] != SHA256_PREFIX
            || !NSupport::IsSHA256(components[1]))
        {
            ythrow yexception() << "'" << checksum << "' is not a valid SHA256 hash";
        }

        return components[1];
    }

    ythrow yexception() << "unable to parse checksum from '" << checksum << "'";
}

TString UnixSignalTypeToString(API::EUnixSignalType unixSignalType) {
    switch (unixSignalType) {
        case API::EUnixSignalType_DEFAULT:
            // Use SIGTERM by default
            return ToString(SIGTERM);
        case API::EUnixSignalType_SIGHUP:
            return ToString(SIGHUP);
        case API::EUnixSignalType_SIGINT:
            return ToString(SIGINT);
        case API::EUnixSignalType_SIGALRM:
            return ToString(SIGALRM);
        case API::EUnixSignalType_SIGKILL:
            return ToString(SIGKILL);
        case API::EUnixSignalType_SIGTERM:
            return ToString(SIGTERM);
        case API::EUnixSignalType_SIGUSR1:
            return ToString(SIGUSR1);
        case API::EUnixSignalType_SIGUSR2:
            return ToString(SIGUSR2);
        case API::EUnixSignalType_SIGURG:
            return ToString(SIGURG);
        case API::EUnixSignalType_SIGQUIT:
            return ToString(SIGQUIT);
        // We don't want to use default
        // So we need this to handle all cases
        case API::EUnixSignalType_INT_MIN_SENTINEL_DO_NOT_USE_:
        case API::EUnixSignalType_INT_MAX_SENTINEL_DO_NOT_USE_:
            ythrow yexception() << "Unexpected unix signal type '" << API::EUnixSignalType_Name(unixSignalType) << "' for switch";
    }
}

TMap<TString, TString> FillTimeLimitReplaceMap(
    const API::TTimeLimit& timeLimit
    , const TString& prefix
    , bool needDefaultInitialDelay
) {
    TMap<TString, TString> result;
    ui64 minRestartPeriodMs = timeLimit.min_restart_period_ms()
        ? timeLimit.min_restart_period_ms()
        : DEFAULT_MIN_RESTART_PERIOD_MS;
    ui64 maxRestartPeriodMs = timeLimit.max_restart_period_ms()
        ? timeLimit.max_restart_period_ms()
        : DEFAULT_MAX_RESTART_PERIOD_MS;
    ui64 maxExecutionTimeMs = timeLimit.max_execution_time_ms()
        ? timeLimit.max_execution_time_ms()
        : DEFAULT_MAX_EXECUTION_TIME_MS;
    Y_ENSURE(
        maxRestartPeriodMs >= minRestartPeriodMs
        , "min restart period " << minRestartPeriodMs
        << " is greater than max restart period " << maxRestartPeriodMs
    );

    ui64 initialDelayMs = needDefaultInitialDelay
        ? (timeLimit.initial_delay_ms() ? timeLimit.initial_delay_ms() : DEFAULT_INITIAL_DELAY_MS)
        : timeLimit.initial_delay_ms();

    result[prefix + "MIN_RESTART_PERIOD"] = ToString(minRestartPeriodMs);
    result[prefix + "MAX_RESTART_PERIOD"] = ToString(maxRestartPeriodMs);
    result[prefix + "RESTART_PERIOD_BACKOFF"] = ToString(timeLimit.restart_period_back_off());
    result[prefix + "RESTART_PERIOD_SCALE"] = ToString(timeLimit.restart_period_scale_ms());
    result[prefix + "INITIAL_DELAY"] = ToString(initialDelayMs);
    result[prefix + "MAX_EXECUTION_TIME"] = ToString(maxExecutionTimeMs);

    return result;
}

TMap<TString, TString> FillComputeResourcesReplaceMap(
    const API::TComputeResources& computeResources
    , const TString& prefix
    , const TPathHolderPtr pathHolder
    , const double cpuToVcpuFactor
) {
    Y_ENSURE(
        cpuToVcpuFactor > MIN_POSSIBLE_CPU_TO_VCPU_FACTOR
        , "CpuToVcpuFactor too small: '" << cpuToVcpuFactor
        << "' must be at least '" << MIN_POSSIBLE_CPU_TO_VCPU_FACTOR << "'"
    );

    TMap<TString, TString> result;

    // guaranties and limits are set to empty instead of 0 in order to avoid the creation of cgroup
    if (computeResources.vcpu_guarantee() == 0) {
        result[prefix + "CPU_GUARANTEE"] = "";
    } else {
        result[prefix + "CPU_GUARANTEE"] = ToString((computeResources.vcpu_guarantee() / 1000.0) / cpuToVcpuFactor) + 'c';
    }

    if (computeResources.vcpu_limit() == 0) {
        result[prefix + "CPU_LIMIT"] = "";
    } else {
        result[prefix + "CPU_LIMIT"] = ToString((computeResources.vcpu_limit() / 1000.0) / cpuToVcpuFactor) + 'c';
    }

    result[prefix + "CPU_POLICY"] = CpuPolicyToString(computeResources.cpu_policy());

    // CPU weight limits in porto is 0.01..100
    // We assume that any value < 0.01 is default porto value
    if (computeResources.cpu_weight() < 0.01) {
        result[prefix + "CPU_WEIGHT"] = "";
    } else {
        result[prefix + "CPU_WEIGHT"] = ToString(computeResources.cpu_weight());
    }

    if (computeResources.memory_guarantee() == 0) {
        result[prefix + "MEMORY_GUARANTEE"] = "";
    } else {
        result[prefix + "MEMORY_GUARANTEE"] = ToString(computeResources.memory_guarantee());
    }

    if (computeResources.memory_limit() == 0) {
        result[prefix + "MEMORY_LIMIT"] = "";
    } else {
        result[prefix + "MEMORY_LIMIT"] = ToString(computeResources.memory_limit());
    }

    if (computeResources.anonymous_memory_limit() == 0) {
        result[prefix + "ANON_LIMIT"] = "";
    } else {
        result[prefix + "ANON_LIMIT"] = ToString(computeResources.anonymous_memory_limit());
    }

    if (!computeResources.recharge_on_pgfault()) {
        result[prefix + "RECHARGE_ON_PGFAULT"] = "";
    } else {
        result[prefix + "RECHARGE_ON_PGFAULT"] = "true";
    }

    if (computeResources.thread_limit() == 0) {
        result[prefix + "THREAD_LIMIT"] = "";
    } else {
        result[prefix + "THREAD_LIMIT"] = ToString(computeResources.thread_limit());
    }


    result[prefix + "IO_LIMIT"] = IoLimitsToString(computeResources.io_limit(), pathHolder);
    result[prefix + "IO_OPS_LIMIT"] = IoLimitsToString(computeResources.io_ops_limit(), pathHolder);

    result[prefix + "IO_POLICY"] = IoPolicyToString(computeResources.io_policy());

    // IO weight limits in porto is 0.01..100
    // We assume that any value < 0.01 is default porto value
    if (computeResources.io_weight() < 0.01) {
        result[prefix + "IO_WEIGHT"] = "";
    } else {
        result[prefix + "IO_WEIGHT"] = ToString(computeResources.io_weight());
    }


    return result;
}

TMap<TString, TString> FillContainerReplaceMap(
    const API::TUtilityContainer& container
    , const TString& name
    , const TString& portoContainerName
    , const TString& rootfsPath
    , const TString& podAgentBinaryFilePath
    , const google::protobuf::RepeatedPtrField<API::TEnvVar>& env
    , const NSecret::TSecretMap& secretMap
    , const TPathHolderPtr pathHolder
    , const double cpuToVcpuFactor
    , bool needTimeLimit
    , bool needValidation
    , bool useEnvSecret
    , bool autoDecodeBase64Secrets
) {
    return FillContainerReplaceMapImpl(
        container
        , name
        , portoContainerName
        , rootfsPath
        , podAgentBinaryFilePath
        , env
        , secretMap
        , pathHolder
        , cpuToVcpuFactor
        , needTimeLimit
        , needValidation
        , false /* needDefaultInitialDelay */
        , useEnvSecret
        , autoDecodeBase64Secrets
    );
}

TMap<TString, TString> FillReadinessOrLivenessContainerReplaceMap(
    const API::TUtilityContainer& container
    , const TString& name
    , const TString& portoContainerName
    , const TString& rootfsPath
    , const TString& podAgentBinaryFilePath
    , const google::protobuf::RepeatedPtrField<API::TEnvVar>& env
    , const NSecret::TSecretMap& secretMap
    , const TPathHolderPtr pathHolder
    , const double cpuToVcpuFactor
    , bool needTimeLimit
    , bool needValidation
    , bool useEnvSecret
) {
    return FillContainerReplaceMapImpl(
        container
        , name
        , portoContainerName
        , rootfsPath
        , podAgentBinaryFilePath
        , env
        , secretMap
        , pathHolder
        , cpuToVcpuFactor
        , needTimeLimit
        , needValidation
        , true /* needDefaultInitialDelay */
        , useEnvSecret
    );
}

TMap<TString, TString> FillInitContainersReplaceMap(
    const google::protobuf::RepeatedPtrField<API::TUtilityContainer>& containers
    , const TVector<TString>& portoContainerNames
    , const TString& rootfsPath
    , const TString& podAgentBinaryFilePath
    , const google::protobuf::RepeatedPtrField<API::TEnvVar>& env
    , const NSecret::TSecretMap& secretMap
    , const TPathHolderPtr pathHolder
    , const double cpuToVcpuFactor
    , bool useEnvSecret
) {
    Y_ENSURE((ssize_t)portoContainerNames.size() == containers.size()
        , "Different size of portoContainerNames (" << portoContainerNames.size() << ")"
            << " and containers (" << containers.size() << ")"
    );

    TVector<TString> initNumList;
    TMap<TString, TVector<TString>> initContainersReplace;
    for (i32 i = 0; i < containers.size(); ++i) {
        TMap<TString, TString> current = FillContainerReplaceMap(
            containers[i]
            , "init"
            , portoContainerNames[i]
            , rootfsPath
            , podAgentBinaryFilePath
            , env
            , secretMap
            , pathHolder
            , cpuToVcpuFactor
            , true /* needTimeLimit */
            , true /* needValidation */
            , useEnvSecret
        );
        for (auto& it : current) {
            initContainersReplace[it.first].push_back(it.second);
        }
        initNumList.push_back(ToString(i));
    }

    TMap<TString, TString> result;
    const TMap<TString, TString> fakeContainer = FillContainerReplaceMap(
        {}
        , "init"
        , "fake_init_container_name"
        , rootfsPath
        , podAgentBinaryFilePath
        , env
        , secretMap
        , pathHolder
        , cpuToVcpuFactor
        , true
        , false
    );
    for (auto& it : fakeContainer) { // iterate over fakeContainer keys, cause initContainersReplace may be empty
        result[it.first + "_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator(initContainersReplace[it.first]);
    }
    result["INIT_NUM_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator(initNumList);

    TVector<TString> previousContainerNames;
    if (!portoContainerNames.empty()) {
        previousContainerNames.push_back("");
        previousContainerNames.insert(previousContainerNames.end(), portoContainerNames.begin(), portoContainerNames.end() - 1);
    }
    result["INIT_CONTAINER_PREVIOUS_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator(previousContainerNames);

    return result;
}

TMap<TString, TString> FillVerifyResourceReplaceMap(
    const TString& name
    , const TString& checksum
    , const bool disabled
) {
    TMap<TString, TString> result;
    const TString checksumKey = to_upper(name) + "_VERIFICATION_CHECKSUM";
    const TString cmdKey = to_upper(name) + "_VERIFICATION_CMD";
    const TString typeKey = to_upper(name) + "_VERIFICATION_TYPE";
    const TString EMPTY_CHECKSUM = "EMPTY:";

    const bool isEmptyPrefix = checksum.StartsWith(EMPTY_PREFIX);
    if (isEmptyPrefix) {
        Y_ENSURE(checksum == EMPTY_CHECKSUM, "'" << checksum << "' has schema 'EMPTY', but not equal to 'EMPTY:'");
    }
    if (disabled || isEmptyPrefix) {
        result[checksumKey] = "";
        result[cmdKey] = "";
        result[typeKey] = ToString(NTreesGeneratorsTypes::EResourceVerificationType::EMPTY);
    } else {
        result[checksumKey] = GetVerificationChecksumValue(checksum);
        result[cmdKey] = GetVerificationCommand(checksum);
        result[typeKey] = ToString(NTreesGeneratorsTypes::EResourceVerificationType::CONTAINER);
    }
    return result;
}

TMap<TString, TString> GetCommonSystemEnvironment(
    const TString& tvmToolLocalToken
    , const TString& podId
    , const API::TNodeMeta& nodeMeta
    , const API::TGpuManagerMeta& gpuManagerMeta
    , const NYP::NClient::NApi::NProto::TPodStatus::TDns& dns
    , const google::protobuf::RepeatedPtrField<NYP::NClient::NApi::NProto::TPodStatus::TIP6AddressAllocation>& ip6AddressAllocations
) {
    TMap<TString, TString> result;

    result[NSystemEnvironment::TVMTOOL_LOCAL_AUTHTOKEN] = tvmToolLocalToken;

    result[NSystemEnvironment::DEPLOY_POD_ID] = podId;

    result[NSystemEnvironment::DEPLOY_NODE_CLUSTER] = nodeMeta.cluster();
    result[NSystemEnvironment::DEPLOY_NODE_DC] = nodeMeta.dc();
    result[NSystemEnvironment::DEPLOY_NODE_FQDN] = nodeMeta.fqdn();

    result[NSystemEnvironment::DEPLOY_POD_PERSISTENT_FQDN] = dns.persistent_fqdn();
    result[NSystemEnvironment::DEPLOY_POD_TRANSIENT_FQDN] = dns.transient_fqdn();

    for (ssize_t i = 0; i < ip6AddressAllocations.size(); ++i) {
        result[TStringBuilder() << NSystemEnvironment::DEPLOY_POD_IP_ADDRESS_PREFIX << i << NSystemEnvironment::DEPLOY_POD_IP_ADDRESS_SUFFIX] = ip6AddressAllocations[i].address();
    }

    for (const auto& [name, value] : gpuManagerMeta.extra_env()) {
        NSystemEnvironment::ValidateNotSystemEnvironmentVariable(name);
        result[name] = value;
    }

    return result;
}

TMap<TString, TString> GetContainerSpecificSystemEnvironment(
    const TString& containerName
) {
    TMap<TString, TString> result;

    result[NSystemEnvironment::DEPLOY_CONTAINER_ID] = containerName;

    return result;
}

void ValidateNoCollisionWithSystemEnvironment(
    const google::protobuf::RepeatedPtrField<API::TEnvVar>& env
) {
    for (const auto& envVar : env) {
        NSystemEnvironment::ValidateNotSystemEnvironmentVariable(envVar.name());
    }
}

TString EnvironmentMapToString(const TMap<TString, TString>& envMap) {
    TStringBuilder result;

    for (const auto& [name, value] : envMap) {
        Y_ENSURE(!IsSpace(name.substr(0, 1)), "First symbol of environment variable name can't be whitespace: '" << name << "=<...>'");
        Y_ENSURE(
            value.empty() || !IsSpace(value.substr(value.size() - 1, 1))
            , "Last symbol of environment variable value can't be whitespace: '" << name << "=<...>'"
        );

        result << EscapeCharactersForEnvironment(name + "=" + value) << ";";
    }

    if (!result.empty()) {
        result.pop_back(); // remove last ";"
    }

    return result;
}

TEnvironmentList GetEnvironmentList(
    const google::protobuf::RepeatedPtrField<API::TEnvVar>& env
    , const NSecret::TSecretMap& secretMap
    , const TMap<TString, TString>& systemEnv
    , bool useEnvSecret
    , bool autoDecodeBase64Secrets
) {
    ValidateNoCollisionWithSystemEnvironment(env);
    TMap<TString, TString> publicEnvMap = systemEnv;
    TMap<TString, TString> secretEnvMap;

    bool hasSecret = false;
    for (const auto& envVar : env) {
        Y_ENSURE(!publicEnvMap.contains(envVar.name()), "Two or more variables have the same name: '" << envVar.name() << "'");
        Y_ENSURE(!secretEnvMap.contains(envVar.name()), "Two or more variables have the same name: '" << envVar.name() << "'");

        Y_ENSURE(envVar.name().size() > 0, "Environment variable can't have empty name");
        Y_ENSURE(envVar.name().find("=") == TStringBuf::npos, "'=' is not allowed at environment variable name: '" << envVar.name() << "'");

        if (envVar.value().has_secret_env()) {
            hasSecret = true;
            if (useEnvSecret) {
                secretEnvMap[envVar.name()] = NSecret::GetSecretValue(envVar.value().secret_env(), secretMap, autoDecodeBase64Secrets);
            } else {
                publicEnvMap[envVar.name()] = NSecret::GetSecretValue(envVar.value().secret_env(), secretMap, autoDecodeBase64Secrets);
            }
        } else if (envVar.value().has_literal_env()) {
            publicEnvMap[envVar.name()] = envVar.value().literal_env().value();
        }
    }

    TString secretEnvList = EnvironmentMapToString(secretEnvMap);
    if (!useEnvSecret && hasSecret) {
        Y_ENSURE(secretEnvList.empty(), "Secret and public envs must be in one list when we don't use env_secret property");
    }
    return {EnvironmentMapToString(publicEnvMap), secretEnvList};
}

TString GetULimitList(
    const google::protobuf::RepeatedPtrField<API::TUlimitSoft>& ulimitSoft
) {
    TStringBuilder ulimitList;
    TSet<API::EContainerULimitType> ulimitSet;

    for (const auto& ulimit : ulimitSoft) {
        Y_ENSURE(!ulimitSet.contains(ulimit.name()), "Two or more ulimits have the same type: '" << ContainerULimitToString(ulimit.name()) << "'");

        const TString ulimitName = ContainerULimitToString(ulimit.name());
        const TString value = ToString(ulimit.value());

        ulimitSet.insert(ulimit.name());
        ulimitList << ulimitName << ": " << value << " " << value << "; ";
    }

    return ulimitList;
}

TString GetSkyGetDownloadCommand(
    const API::TSkyGetDownload& skyGetDownload
    , const TString& downloadDir
) {
    Y_ENSURE(skyGetDownload.resid(), "Sky get resid not provided");
    Y_ENSURE(NSupport::IsRbtorrent(skyGetDownload.resid()), "String '" << skyGetDownload.resid() << "' doesn't fit the rbtorrent pattern");

    TStringBuilder command;
    command << "sky get -p";

    if (!downloadDir.empty()) {
        command << " -d " << downloadDir;
    }

    if (skyGetDownload.deduplicate() != API::ESkyGetDeduplicateMode_NO) {
        command << " -D " << SkyGetDeduplicateModeToString(skyGetDownload.deduplicate());
    }

    command << " " << skyGetDownload.resid();

    return command;
}

bool IsUrl(TStringBuf req) {
    if (req.find('.') == TStringBuf::npos)
        return false;

    ::NUri::TUri reguri;
    if (::NUri::TUri::ParsedOK != reguri.ParseAbsOrHttpUri(req, ::NUri::TUri::FeatureSchemeFlexible | ::NUri::TUri::FeatureToLower | ::NUri::TUri::FeatureAllowHostIDN | ::NUri::TUri::FeatureCheckHost))
        return false;

    return true;
}

bool IsRbtorrent(const TString& url) {
    TVector<TString> components;
    StringSplitter(url).Split(':').SkipEmpty().Collect(&components);
    return !(components.size() != 2
        || components[0] != RBTORRENT_PREFIX
        || components[1].find_first_not_of("0123456789abcdefghijklmnopqrstuvwxyz") != components[1].npos);
}

bool IsWgetOrSkyget(TStringBuf req) {
    return (req.StartsWith(HTTP_OR_HTTPS_PREFIX) && IsUrl(req)) || (req.StartsWith(RBTORRENT_PREFIX) && IsRbtorrent(ToString(req)));
}

TString NormalizeMountPoint(const TString& mountPoint) {
    TString normalizedMountPoint = mountPoint;

    Y_ENSURE(mountPoint != "/", "Mount point can't be '/'");
    Y_ENSURE(mountPoint != "", "Mount point can't be empty");

    if (normalizedMountPoint.StartsWith('/')) {
        normalizedMountPoint.erase(normalizedMountPoint.begin());
    }
    if (normalizedMountPoint.EndsWith('/')) {
        normalizedMountPoint.pop_back();
    }

    return normalizedMountPoint;
}

void ValidateMountPoints(const TVector<TString>& mountPoints) {
    TVector<TString> normalizedMountPoints;

    for (TString mountPoint : mountPoints) {
        Y_ENSURE(mountPoint.StartsWith('/'), "Mount point should start with slash: '" << mountPoint << "'");
        Y_ENSURE(!mountPoint.EndsWith('/'), "Mount point should not end with slash: '" << mountPoint << "'");

        normalizedMountPoints.push_back(mountPoint + "/");
    }

    Sort(normalizedMountPoints.begin(), normalizedMountPoints.end());
    for (size_t i = 1; i < normalizedMountPoints.size(); ++i) {
        Y_ENSURE(!normalizedMountPoints[i].StartsWith(normalizedMountPoints[i - 1]),
            "Mount point '" << normalizedMountPoints[i] << "' is a nested mount point of '" << normalizedMountPoints[i - 1] << "'");
    }
}

} // namespace NInfra::NPodAgent::NSupport
