#include "support_functions.h"

#include <library/cpp/testing/unittest/registar.h>

#include <util/system/fs.h>
#include <util/system/shellcommand.h>

namespace NInfra::NPodAgent::NSupportFunctionsTest {

namespace {

TPathHolderPtr GetPathHolder() {
    return TPathHolderPtr(
        new TPathHolder(
            "/root"
            , {
                {"", ""}
                , {"virtual_disk", "///place"}
            }
            , {
                {"", "/resource_storage"}
                , {"///place", "/place_resource_storage"}
            }
            , ""
            , ""
            , ""
            , ""
            , ""
            , ""
        )
    );
}

API::TIoLimit GetIoLimit(
    const TString& target
    , API::EIoLimitType type
    , ui64 value
) {
    API::TIoLimit ioLimit;

    ioLimit.set_raw(target);
    ioLimit.set_type(type);
    ioLimit.set_value(value);

    return ioLimit;
}

} // namespace

Y_UNIT_TEST_SUITE(SupportFunctionsTestSuite) {

Y_UNIT_TEST(TestEnshieldString) {
    TString alphabet = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
    TString numbers = "1234567890";
    TString apostrophe = "'";

    UNIT_ASSERT_EQUAL_C(NSupport::EnshieldString(alphabet), alphabet, NSupport::EnshieldString(alphabet));
    UNIT_ASSERT_EQUAL_C(NSupport::EnshieldString(numbers), numbers, NSupport::EnshieldString(numbers));
    UNIT_ASSERT_EQUAL_C(NSupport::EnshieldString(apostrophe), "'\\'\\\\\\'\\''", NSupport::EnshieldString(apostrophe));
}

Y_UNIT_TEST(TestGetVerificationCommand) {
    const TVector<std::pair<TString, TString>> valid = {
        {"MD5:checksum", "d41d8cd98f00b204e9800998ecf8427e"}
        , {"SHA256:checksum", "e3b0c44298fc1c149afbf4c8996fb92427ae41e4649b934ca495991b7852b855"}
    };
    const TString invalid = "INVALID";

    NFs::RemoveRecursive("downloaded");
    NFs::MakeDirectory("downloaded");

    for (auto& item : valid) {
        TString command = NSupport::GetVerificationCommand(item.first);

        TShellCommand shellCommand(command);
        shellCommand.Run();
        shellCommand.Wait();
        UNIT_ASSERT_EQUAL_C(TShellCommand::ECommandStatus::SHELL_FINISHED, shellCommand.GetStatus(), shellCommand.GetError());

        TString output = shellCommand.GetOutput();
        UNIT_ASSERT_EQUAL_C(item.second, output, output);
    }
    UNIT_ASSERT_EXCEPTION_CONTAINS(NSupport::GetVerificationCommand(invalid), yexception, invalid);
}

Y_UNIT_TEST(TestGetVerificationChecksumValue) {
    const TString MD5Checksum = "MD5:3ea05169a95365cbf9b4ff4688cd8945";
    const TString SHA256Checksum = "SHA256:ad8d6286db150ffe6ebd2575cce11cdaa10503d0824d11579c385101472b85a0";

    const TVector<TString> invalidChecksums = {
        "MD5:3ea05169a95365cbf9b4ff4688cd89450"
        , "MD5:3ea05169a95365cbf9b4ff4688"
        , "MD5:3ea05169a95365c\"; pwd; \"bf9b4ff4688cd8945"
        , "MD5:Ёea05169a95365cbf9b4ff4688cd8945"
        , "MD5:c57f6ac1b71f45a07dbd91a59fa47c23abcd87c2:631225"
        , "MD5:3ea05169a95365cb\n9b4ff4688cd8945"
        , "MD5:3ea05169А95365cbf9b4ff4688cd8945"
        , "MD5:3ea05169a 5365cbf9b4ff4688cd8945"

        , "EMPTY:hash"

        , "SHa256:ad8d6286db150ffe6ebd2575cce11cdaa10503d0824d11579c385101472b85a0"
        , "SHA256:ad8d6286db150ffe6ebd2575cce1 cdaa10503d0824d11579c385101472b85a0"
        , "SHA256:ad8d6286db150ffe6ebd2575cce11cdaa10503d0824d11579c3"
        , "SHA256:ad8d6286db150ffe6ebd25asdfasdf75cce11cdaa10503d0824d11579c385101472b85a0"
        , "SHA256:ad8d6286db150ffe6ebd2575cce1\ncdaa10503d0824d11579c385101472b85a0"
        , "SHA256:ad8d6286db150ffe6ebd2575cce1'cdaa10503d0824d11579c385101472b85a0"
        , "SHA256:ad8d6286db150ffe6ebd2575cce1; \"pwd; \"503d0824d11579c385101472b85a0"
        , "SHA256:ad8d6286db150ffe6ebd2:75cce11cdaa10503d0824d11579c385101472b85a0"
    };

    UNIT_ASSERT_EQUAL(NSupport::GetVerificationChecksumValue(MD5Checksum), TStringBuf(MD5Checksum, 4, TString::npos));
    UNIT_ASSERT_EQUAL(NSupport::GetVerificationChecksumValue(SHA256Checksum), TStringBuf(SHA256Checksum, 7, TString::npos));

    for (auto& invalidChecksum : invalidChecksums) {
        UNIT_ASSERT_EXCEPTION_CONTAINS(NSupport::GetVerificationChecksumValue(invalidChecksum), yexception, invalidChecksum);
    }
}

Y_UNIT_TEST(TestFillTimeLimitReplaceMap) {
    API::TTimeLimit timeLimit;
    timeLimit.set_initial_delay_ms(1);
    timeLimit.set_restart_period_scale_ms(2);
    timeLimit.set_restart_period_back_off(3);
    timeLimit.set_min_restart_period_ms(4);
    timeLimit.set_max_restart_period_ms(5);
    timeLimit.set_max_execution_time_ms(6);

    TMap<TString, TString> result = NSupport::FillTimeLimitReplaceMap(timeLimit, "PREFIX_", false);

    UNIT_ASSERT_EQUAL_C(result["PREFIX_INITIAL_DELAY"], "1", result["PREFIX_INITIAL_DELAY"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_MAX_EXECUTION_TIME"], "6", result["PREFIX_MAX_EXECUTION_TIME"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_MAX_RESTART_PERIOD"], "5", result["PREFIX_MAX_RESTART_PERIOD"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_MIN_RESTART_PERIOD"], "4", result["PREFIX_MIN_RESTART_PERIOD"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_RESTART_PERIOD_BACKOFF"], "3", result["PREFIX_RESTART_PERIOD_BACKOFF"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_RESTART_PERIOD_SCALE"], "2", result["PREFIX_RESTART_PERIOD_SCALE"]);
}

Y_UNIT_TEST(TestFillTimeLimitReplaceMapDefault) {
    TMap<TString, TString> result = NSupport::FillTimeLimitReplaceMap({}, "PREFIX_", false);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_INITIAL_DELAY"], "0", result["PREFIX_INITIAL_DELAY"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_MAX_EXECUTION_TIME"], "1800000", result["PREFIX_MAX_EXECUTION_TIME"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_MAX_RESTART_PERIOD"], "18446744073709551615", result["PREFIX_MAX_RESTART_PERIOD"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_MIN_RESTART_PERIOD"], "30000", result["PREFIX_MIN_RESTART_PERIOD"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_RESTART_PERIOD_BACKOFF"], "0", result["PREFIX_RESTART_PERIOD_BACKOFF"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_RESTART_PERIOD_SCALE"], "0", result["PREFIX_RESTART_PERIOD_SCALE"]);

    result = NSupport::FillTimeLimitReplaceMap({}, "PREFIX_", true);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_INITIAL_DELAY"], "5000", result["PREFIX_INITIAL_DELAY"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_MAX_EXECUTION_TIME"], "1800000", result["PREFIX_MAX_EXECUTION_TIME"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_MAX_RESTART_PERIOD"], "18446744073709551615", result["PREFIX_MAX_RESTART_PERIOD"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_MIN_RESTART_PERIOD"], "30000", result["PREFIX_MIN_RESTART_PERIOD"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_RESTART_PERIOD_BACKOFF"], "0", result["PREFIX_RESTART_PERIOD_BACKOFF"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_RESTART_PERIOD_SCALE"], "0", result["PREFIX_RESTART_PERIOD_SCALE"]);
}

Y_UNIT_TEST(TestFillComputeResourcesReplaceMap) {
    API::TComputeResources computeResources;
    computeResources.set_vcpu_guarantee(7);
    computeResources.set_vcpu_limit(8);
    computeResources.set_cpu_policy(API::ECpuPolicy_HIGH);
    computeResources.set_cpu_weight(9);
    computeResources.set_memory_guarantee(10);
    computeResources.set_memory_limit(11);
    computeResources.set_anonymous_memory_limit(12);
    computeResources.set_recharge_on_pgfault(true);
    computeResources.set_thread_limit(13);
    computeResources.add_io_limit()->CopyFrom(GetIoLimit("/tmp1", API::EIoLimitType_READ, 14));
    computeResources.add_io_limit()->CopyFrom(GetIoLimit("/tmp2", API::EIoLimitType_WRITE, 15));
    computeResources.add_io_ops_limit()->CopyFrom(GetIoLimit("/tmp3", API::EIoLimitType_READ, 16));
    computeResources.add_io_ops_limit()->CopyFrom(GetIoLimit("/tmp4", API::EIoLimitType_WRITE, 17));
    computeResources.set_io_policy(API::EIoPolicy_BATCH);
    computeResources.set_io_weight(18);

    TMap<TString, TString> result = NSupport::FillComputeResourcesReplaceMap(
        computeResources
        , "PREFIX_"
        , GetPathHolder()
        , 1.0
    );

    UNIT_ASSERT_EQUAL_C(result["PREFIX_CPU_GUARANTEE"], "0.007c", result["PREFIX_CPU_GUARANTEE"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_CPU_LIMIT"], "0.008c", result["PREFIX_CPU_LIMIT"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_CPU_POLICY"], "high", result["PREFIX_CPU_POLICY"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_CPU_WEIGHT"], "9", result["PREFIX_CPU_WEIGHT"]);

    UNIT_ASSERT_EQUAL_C(result["PREFIX_MEMORY_GUARANTEE"], "10", result["PREFIX_MEMORY_GUARANTEE"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_MEMORY_LIMIT"], "11", result["PREFIX_MEMORY_LIMIT"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_ANON_LIMIT"], "12", result["PREFIX_ANON_LIMIT"]);

    UNIT_ASSERT_EQUAL_C(result["PREFIX_RECHARGE_ON_PGFAULT"], "true", result["PREFIX_RECHARGE_ON_PGFAULT"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_THREAD_LIMIT"], "13", result["PREFIX_THREAD_LIMIT"]);

    UNIT_ASSERT_EQUAL_C(result["PREFIX_IO_LIMIT"], "/tmp1 r: 14;/tmp2 w: 15", result["PREFIX_IO_LIMIT"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_IO_OPS_LIMIT"], "/tmp3 r: 16;/tmp4 w: 17", result["PREFIX_IO_OPS_LIMIT"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_IO_POLICY"], "batch", result["PREFIX_IO_POLICY"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_IO_WEIGHT"], "18", result["PREFIX_IO_WEIGHT"]);
}

Y_UNIT_TEST(TestFillComputeResourcesReplaceMapWithDefaults) {
    API::TComputeResources computeResources;
    computeResources.set_vcpu_guarantee(0);
    computeResources.set_vcpu_limit(0);
    computeResources.set_memory_guarantee(0);
    computeResources.set_memory_limit(0);
    computeResources.set_anonymous_memory_limit(0);
    computeResources.set_recharge_on_pgfault(false);
    computeResources.set_thread_limit(0);

    TMap<TString, TString> result = NSupport::FillComputeResourcesReplaceMap(
        computeResources
        , "PREFIX_"
        , GetPathHolder()
        , 1.0
    );

    UNIT_ASSERT_EQUAL_C(result["PREFIX_CPU_GUARANTEE"], "", result["PREFIX_CPU_GUARANTEE"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_CPU_LIMIT"], "", result["PREFIX_CPU_LIMIT"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_CPU_POLICY"], "", result["PREFIX_CPU_POLICY"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_CPU_WEIGHT"], "", result["PREFIX_CPU_WEIGHT"]);

    UNIT_ASSERT_EQUAL_C(result["PREFIX_MEMORY_GUARANTEE"], "", result["PREFIX_MEMORY_GUARANTEE"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_MEMORY_LIMIT"], "", result["PREFIX_MEMORY_LIMIT"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_ANON_LIMIT"], "", result["PREFIX_ANON_LIMIT"]);

    UNIT_ASSERT_EQUAL_C(result["PREFIX_RECHARGE_ON_PGFAULT"], "", result["PREFIX_RECHARGE_ON_PGFAULT"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_THREAD_LIMIT"], "", result["PREFIX_THREAD_LIMIT"]);

    UNIT_ASSERT_EQUAL_C(result["PREFIX_IO_LIMIT"], "", result["PREFIX_IO_LIMIT"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_IO_OPS_LIMIT"], "", result["PREFIX_IO_OPS_LIMIT"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_IO_POLICY"], "", result["PREFIX_IO_POLICY"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_IO_WEIGHT"], "", result["PREFIX_IO_WEIGHT"]);
}

Y_UNIT_TEST(TestFillComputeResourcesReplaceMapWithCpuToVCpuFactor) {
    API::TComputeResources computeResources;
    computeResources.set_vcpu_guarantee(1000);
    computeResources.set_vcpu_limit(2000);

    TMap<TString, TString> result = NSupport::FillComputeResourcesReplaceMap(
        computeResources
        , "PREFIX_"
        , GetPathHolder()
        , 2.0
    );

    UNIT_ASSERT_EQUAL_C(result["PREFIX_CPU_GUARANTEE"], "0.5c", result["PREFIX_CPU_GUARANTEE"]);
    UNIT_ASSERT_EQUAL_C(result["PREFIX_CPU_LIMIT"], "1c", result["PREFIX_CPU_LIMIT"]);
}

Y_UNIT_TEST(TestFillComputeResourcesReplaceMapWithCpuToVCpuFactorValidation) {
    UNIT_ASSERT_EXCEPTION_CONTAINS(
        NSupport::FillComputeResourcesReplaceMap(
            {} /* computeResource */
            , "PREFIX_"
            , GetPathHolder()
            , 0.001
        )
        , yexception
        , "too small"
    );
}

Y_UNIT_TEST(TestFillComputeResourcesReplaceMapWithIoLimit) {
    API::TComputeResources computeResources;
    API::TIoLimit* ioLimit = computeResources.add_io_limit();

    API::TIoLimit ioLimitTemplate;
    ioLimitTemplate.set_type(API::EIoLimitType_READ);
    ioLimitTemplate.set_value(5);

    TPathHolderPtr pathHolder = GetPathHolder();

    {
        ioLimit->CopyFrom(ioLimitTemplate);
        ioLimit->set_raw("/tmp");
        TMap<TString, TString> result = NSupport::FillComputeResourcesReplaceMap(
            computeResources
            , "PREFIX_"
            , pathHolder
            , 1.0
        );
        UNIT_ASSERT_EQUAL_C(result["PREFIX_IO_LIMIT"], "/tmp r: 5", result["PREFIX_IO_LIMIT"]);
    }

    {
        ioLimit->CopyFrom(ioLimitTemplate);
        ioLimit->set_default_target(API::EDefaultIoLimitTarget_PLACE);
        TMap<TString, TString> result = NSupport::FillComputeResourcesReplaceMap(
            computeResources
            , "PREFIX_"
            , pathHolder
            , 1.0
        );
        UNIT_ASSERT_EQUAL_C(result["PREFIX_IO_LIMIT"], "/place r: 5", result["PREFIX_IO_LIMIT"]);
    }

    {
        ioLimit->CopyFrom(ioLimitTemplate);
        ioLimit->set_default_target(API::EDefaultIoLimitTarget_SSD);
        TMap<TString, TString> result = NSupport::FillComputeResourcesReplaceMap(
            computeResources
            , "PREFIX_"
            , pathHolder
            , 1.0
        );
        UNIT_ASSERT_EQUAL_C(result["PREFIX_IO_LIMIT"], "/ssd r: 5", result["PREFIX_IO_LIMIT"]);
    }

    {
        ioLimit->CopyFrom(ioLimitTemplate);
        ioLimit->set_virtual_disk_id_ref("virtual_disk");
        TMap<TString, TString> result = NSupport::FillComputeResourcesReplaceMap(
            computeResources
            , "PREFIX_"
            , pathHolder
            , 1.0
        );
        UNIT_ASSERT_EQUAL_C(result["PREFIX_IO_LIMIT"], "/root/place r: 5", result["PREFIX_IO_LIMIT"]);
    }

    {
        ioLimit->CopyFrom(ioLimitTemplate);
        ioLimit->set_default_target(API::EDefaultIoLimitTarget_NONE);
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            NSupport::FillComputeResourcesReplaceMap(
                computeResources
                , "PREFIX_"
                , pathHolder
                , 1.0
            )
            , yexception
            , "Io limit default target is none"
        );
    }

    {
        ioLimit->CopyFrom(ioLimitTemplate);
        ioLimit->set_virtual_disk_id_ref("");
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            NSupport::FillComputeResourcesReplaceMap(
                computeResources
                , "PREFIX_"
                , pathHolder
                , 1.0
            )
            , yexception
            , "Empty virtual disk ref is forbidden at io limit target"
        );
    }

    {
        ioLimit->CopyFrom(ioLimitTemplate);
        ioLimit->set_virtual_disk_id_ref("bad_virtual_disk");
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            NSupport::FillComputeResourcesReplaceMap(
                computeResources
                , "PREFIX_"
                , pathHolder
                , 1.0
            )
            , yexception
            , "Unknown virtual disk ref at io limit target: 'bad_virtual_disk'"
        );
    }

    {
        ioLimit->CopyFrom(ioLimitTemplate);
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            NSupport::FillComputeResourcesReplaceMap(
                computeResources
                , "PREFIX_"
                , pathHolder
                , 1.0
            )
            , yexception
            , "Io limit target not set"
        );
    }
}

Y_UNIT_TEST(TestFillContainerReplaceMapDefault) {
    API::TUtilityContainer container;

    TMap<TString, TString> result = NSupport::FillContainerReplaceMap(
        container
        , "start"
        , "" /* portoContainerName */
        , "" /* rootfsPath */
        , "" /* podAgentBinaryFilePath */
        , {} /* env */
        , {} /* secretMap */
        , GetPathHolder()
        , 1.0
        , false /* needTimeLimit */
        , false /* needValidation */
    );

    UNIT_ASSERT_EQUAL_C(result["START_GROUP"], "root", result["START_GROUP"]);
    UNIT_ASSERT_EQUAL_C(result["START_USER"], "root", result["START_USER"]);
}

Y_UNIT_TEST(TestFillContainerReplaceMapDoNotWrapRootCommand) {
    API::TUtilityContainer container;
    container.set_command_line("echo start");
    container.set_user("root");
    container.set_group("root");

    TMap<TString, TString> result = NSupport::FillContainerReplaceMap(
        container
        , "start"
        , "" /* portoContainerName */
        , "" /* rootfsPath */
        , "" /* podAgentBinaryFilePath */
        , {} /* env */
        , {} /* secretMap */
        , GetPathHolder()
        , 1.0
        , false /* needTimeLimit */
        , false /* needValidation */
    );

    UNIT_ASSERT_EQUAL_C(result["START_CMD"], "echo start", result["START_CMD"]);
}

Y_UNIT_TEST(TestFillContainerReplaceMap) {
    API::TUtilityContainer container;
    container.mutable_time_limit()->set_initial_delay_ms(1);
    container.mutable_time_limit()->set_restart_period_scale_ms(2);
    container.mutable_time_limit()->set_restart_period_back_off(3);
    container.mutable_time_limit()->set_min_restart_period_ms(4);
    container.mutable_time_limit()->set_max_restart_period_ms(5);
    container.mutable_time_limit()->set_max_execution_time_ms(6);
    container.mutable_compute_resources()->set_vcpu_guarantee(7);
    container.mutable_compute_resources()->set_vcpu_limit(8);
    container.mutable_compute_resources()->set_cpu_policy(API::ECpuPolicy_HIGH);
    container.mutable_compute_resources()->set_cpu_weight(9);
    container.mutable_compute_resources()->set_memory_guarantee(10);
    container.mutable_compute_resources()->set_memory_limit(11);
    container.mutable_compute_resources()->set_anonymous_memory_limit(12);
    container.mutable_compute_resources()->set_recharge_on_pgfault(true);
    container.mutable_compute_resources()->add_io_limit()->CopyFrom(GetIoLimit("/tmp1", API::EIoLimitType_READ, 13));
    container.mutable_compute_resources()->add_io_limit()->CopyFrom(GetIoLimit("/tmp2", API::EIoLimitType_WRITE, 14));
    container.mutable_compute_resources()->add_io_ops_limit()->CopyFrom(GetIoLimit("/tmp3", API::EIoLimitType_READ, 15));
    container.mutable_compute_resources()->add_io_ops_limit()->CopyFrom(GetIoLimit("/tmp4", API::EIoLimitType_WRITE, 16));
    container.mutable_compute_resources()->set_io_policy(API::EIoPolicy_BATCH);
    container.mutable_compute_resources()->set_io_weight(17);
    container.set_command_line("echo start");
    container.set_cwd("dir");
    container.set_stdout_file("stdout_file");
    container.set_stderr_file("stderr_file");
    container.set_stdout_and_stderr_limit(18);
    container.set_core_command("echo core");
    container.set_user("user");
    container.set_group("group");

    google::protobuf::RepeatedPtrField<API::TEnvVar> env;
    NSecret::TSecretMap secretMap;
    {
        secretMap["SecretAlias"]["SecretId"] = {
            "VGVzdFNlY3JldFZhbHVl"
            , "base64"
        };
        API::TEnvVar* secretEnvVar = env.Add();
        secretEnvVar->set_name("TestSecret");
        secretEnvVar->mutable_value()->mutable_secret_env()->set_alias("SecretAlias");
        secretEnvVar->mutable_value()->mutable_secret_env()->set_id("SecretId");

        API::TEnvVar* envVar = env.Add();
        envVar->set_name("TestLiteral");
        envVar->mutable_value()->mutable_literal_env()->set_value("TestLiteralValue");
    }

    for (auto useEnvSecret : {false, true}) {
        for (auto autoDecodeBase64Secrets : {false, true}) {
            TMap<TString, TString> result = NSupport::FillContainerReplaceMap(
                container
                , "start"
                , "start_container"
                , "path/"
                , "/pod_agent"
                , env
                , secretMap
                , GetPathHolder()
                , 1.0
                , true /* needTimeLimit */
                , true /* needValidation */
                , useEnvSecret
                , autoDecodeBase64Secrets
            );

            UNIT_ASSERT_EQUAL_C(result["START_CMD"], "/pod_agent exec_wrapper user group echo start", result["START_CMD"]);
            UNIT_ASSERT_EQUAL_C(result["START_CWD"], "dir", result["START_CMD"]);
            UNIT_ASSERT_EQUAL_C(result["START_CORE_COMMAND"], "echo core", result["START_CORE_COMMAND"]);
            UNIT_ASSERT_EQUAL_C(result["START_CONTAINER"], "start_container", result["START_CONTAINER"]);
            if (useEnvSecret) {
                UNIT_ASSERT_EQUAL_C(result["START_ENVIRONMENT"],"DEPLOY_CONTAINER_ID=start_container;TestLiteral=TestLiteralValue", result["START_ENVIRONMENT"]);
                TString secretEnvs = autoDecodeBase64Secrets
                    ? "TestSecret=TestSecretValue"
                    : "TestSecret=VGVzdFNlY3JldFZhbHVl";
                UNIT_ASSERT_EQUAL_C(result["START_SECRET_ENVIRONMENT"], secretEnvs, result["START_SECRET_ENVIRONMENT"]);
            } else {
                TString envs = autoDecodeBase64Secrets
                    ? "DEPLOY_CONTAINER_ID=start_container;TestLiteral=TestLiteralValue;TestSecret=TestSecretValue"
                    : "DEPLOY_CONTAINER_ID=start_container;TestLiteral=TestLiteralValue;TestSecret=VGVzdFNlY3JldFZhbHVl";
                UNIT_ASSERT_EQUAL_C(result["START_ENVIRONMENT"], envs, result["START_ENVIRONMENT"]);
                UNIT_ASSERT_EQUAL_C(result["START_SECRET_ENVIRONMENT"], "", result["START_SECRET_ENVIRONMENT"]);
            }

            UNIT_ASSERT_EQUAL_C(result["START_INITIAL_DELAY"], "1", result["START_INITIAL_DELAY"]);
            UNIT_ASSERT_EQUAL_C(result["START_MAX_EXECUTION_TIME"], "6", result["START_MAX_EXECUTION_TIME"]);
            UNIT_ASSERT_EQUAL_C(result["START_MAX_RESTART_PERIOD"], "5", result["START_MAX_RESTART_PERIOD"]);
            UNIT_ASSERT_EQUAL_C(result["START_MIN_RESTART_PERIOD"], "4", result["START_MIN_RESTART_PERIOD"]);
            UNIT_ASSERT_EQUAL_C(result["START_RESTART_PERIOD_BACKOFF"], "3", result["START_RESTART_PERIOD_BACKOFF"]);
            UNIT_ASSERT_EQUAL_C(result["START_RESTART_PERIOD_SCALE"], "2", result["START_RESTART_PERIOD_SCALE"]);

            UNIT_ASSERT_EQUAL_C(result["START_GROUP"], "root", result["START_GROUP"]);
            UNIT_ASSERT_EQUAL_C(result["START_USER"], "root", result["START_USER"]);

            UNIT_ASSERT_EQUAL_C(result["START_CPU_GUARANTEE"], "0.007c", result["START_CPU_GUARANTEE"]);
            UNIT_ASSERT_EQUAL_C(result["START_CPU_LIMIT"], "0.008c", result["START_CPU_LIMIT"]);
            UNIT_ASSERT_EQUAL_C(result["START_CPU_POLICY"], "high", result["START_CPU_POLICY"]);
            UNIT_ASSERT_EQUAL_C(result["START_CPU_WEIGHT"], "9", result["START_CPU_WEIGHT"]);

            UNIT_ASSERT_EQUAL_C(result["START_MEMORY_GUARANTEE"], "10", result["START_MEMORY_GUARANTEE"]);
            UNIT_ASSERT_EQUAL_C(result["START_MEMORY_LIMIT"], "11", result["START_MEMORY_LIMIT"]);
            UNIT_ASSERT_EQUAL_C(result["START_ANON_LIMIT"], "12", result["START_ANON_LIMIT"]);
            UNIT_ASSERT_EQUAL_C(result["START_RECHARGE_ON_PGFAULT"], "true", result["START_RECHARGE_ON_PGFAULT"]);

            UNIT_ASSERT_EQUAL_C(result["START_IO_LIMIT"], "/tmp1 r: 13;/tmp2 w: 14", result["START_IO_LIMIT"]);
            UNIT_ASSERT_EQUAL_C(result["START_IO_OPS_LIMIT"], "/tmp3 r: 15;/tmp4 w: 16", result["START_IO_OPS_LIMIT"]);
            UNIT_ASSERT_EQUAL_C(result["START_IO_POLICY"], "batch", result["START_IO_POLICY"]);
            UNIT_ASSERT_EQUAL_C(result["START_IO_WEIGHT"], "17", result["START_IO_WEIGHT"]);

            UNIT_ASSERT_EQUAL_C(result["START_STDOUT_LOG_PATH"], "stdout_file", result["START_STDOUT_LOG_PATH"]);
            UNIT_ASSERT_EQUAL_C(result["START_STDERR_LOG_PATH"], "stderr_file", result["START_STDERR_LOG_PATH"]);
            UNIT_ASSERT_EQUAL_C(result["START_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE"], "path/stdout_file", result["START_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE"]);
            UNIT_ASSERT_EQUAL_C(result["START_STDERR_LOG_FILE_FULL_PATH_TO_CREATE"], "path/stderr_file", result["START_STDERR_LOG_FILE_FULL_PATH_TO_CREATE"]);
            UNIT_ASSERT_EQUAL_C(result["START_STDOUT_AND_STDERR_FILE_SIZE_LIMIT"], "18", result["START_STDOUT_AND_STDERR_FILE_SIZE_LIMIT"]);

            UNIT_ASSERT_EQUAL_C(result["START_AGING_TIME"], "1073741824", result["START_AGING_TIME"]);
    }
    }
}

Y_UNIT_TEST(TestFillContainerReplaceMapCmdValidation) {
    API::TUtilityContainer container;
    TPathHolderPtr pathHolder = GetPathHolder();

    UNIT_ASSERT_EXCEPTION_CONTAINS(
        NSupport::FillContainerReplaceMap(
            container
            , "start"
            , "start_container"
            , "path/"
            , "/pod_agent"
            , {} /* env */
            , {} /* secretMap */
            , pathHolder
            , 1.0
            , true /* needTimeLimit */
            , true /* needValidation */
        )
        , yexception
        , "container command_line is empty"
    );

    container.set_command_line(" echo start");
    UNIT_ASSERT_EXCEPTION_CONTAINS(
        NSupport::FillContainerReplaceMap(
            container
            , "start"
            , "start_container"
            , "path/"
            , "/pod_agent"
            , {} /* env */
            , {} /* secretMap */
            , pathHolder
            , 1.0
            , true /* needTimeLimit */
            , true /* needValidation */
        )
        , yexception
        , "starts with space character"
    );

    container.set_command_line("echo start ");
    UNIT_ASSERT_EXCEPTION_CONTAINS(
        NSupport::FillContainerReplaceMap(
            container
            , "start"
            , "start_container"
            , "path/"
            , "/pod_agent"
            , {} /* env */
            , {} /* secretMap */
            , pathHolder
            , 1.0
            , true /* needTimeLimit */
            , true /* needValidation */
        )
        , yexception
        , "ends with space character"
    );
}

Y_UNIT_TEST(TestFillReadinessOrLivenessContainerReplaceMapDefault) {
    API::TUtilityContainer container;

    TMap<TString, TString> result = NSupport::FillReadinessOrLivenessContainerReplaceMap(
        container
        , "readiness"
        , "" /* portoContainerName */
        , "" /* rootfsPath */
        , "" /* podAgentBinaryFilePath */
        , {} /* env */
        , {} /* secretMap */
        , GetPathHolder()
        , 1.0
        , true /* needTimeLimit */
        , false /* needValidation */
    );

    UNIT_ASSERT_EQUAL_C(result["READINESS_GROUP"], "root", result["READINESS_GROUP"]);
    UNIT_ASSERT_EQUAL_C(result["READINESS_USER"], "root", result["READINESS_USER"]);

    UNIT_ASSERT_EQUAL_C(result["READINESS_INITIAL_DELAY"], "5000", result["READINESS_INITIAL_DELAY"]);
}

Y_UNIT_TEST(TestFillReadinessOrLivenessContainerReplaceMap) {
    API::TUtilityContainer container;
    container.mutable_time_limit()->set_initial_delay_ms(1);
    container.mutable_time_limit()->set_restart_period_scale_ms(2);
    container.mutable_time_limit()->set_restart_period_back_off(3);
    container.mutable_time_limit()->set_min_restart_period_ms(4);
    container.mutable_time_limit()->set_max_restart_period_ms(5);
    container.mutable_time_limit()->set_max_execution_time_ms(6);
    container.mutable_compute_resources()->set_vcpu_guarantee(7);
    container.mutable_compute_resources()->set_vcpu_limit(8);
    container.mutable_compute_resources()->set_cpu_policy(API::ECpuPolicy_HIGH);
    container.mutable_compute_resources()->set_cpu_weight(9);
    container.mutable_compute_resources()->set_memory_guarantee(10);
    container.mutable_compute_resources()->set_memory_limit(11);
    container.mutable_compute_resources()->set_anonymous_memory_limit(12);
    container.mutable_compute_resources()->set_recharge_on_pgfault(true);
    container.mutable_compute_resources()->add_io_limit()->CopyFrom(GetIoLimit("/tmp1", API::EIoLimitType_READ, 13));
    container.mutable_compute_resources()->add_io_limit()->CopyFrom(GetIoLimit("/tmp2", API::EIoLimitType_WRITE, 14));
    container.mutable_compute_resources()->add_io_ops_limit()->CopyFrom(GetIoLimit("/tmp3", API::EIoLimitType_READ, 15));
    container.mutable_compute_resources()->add_io_ops_limit()->CopyFrom(GetIoLimit("/tmp4", API::EIoLimitType_WRITE, 16));
    container.mutable_compute_resources()->set_io_policy(API::EIoPolicy_BATCH);
    container.mutable_compute_resources()->set_io_weight(17);
    container.set_command_line("echo start");
    container.set_cwd("dir");
    container.set_stdout_file("stdout_file");
    container.set_stderr_file("stderr_file");
    container.set_stdout_and_stderr_limit(18);
    container.set_core_command("echo core");
    container.set_user("user");
    container.set_group("group");

    TPathHolderPtr pathHolder = GetPathHolder();

    TMap<TString, TString> result1 = NSupport::FillContainerReplaceMap(
        container
        , "readiness"
        , "readiness_container"
        , "path/"
        , "/pod_agent"
        , {} /* env */
        , {} /* secretMap */
        , pathHolder
        , 1.0
        , true /* needTimeLimit */
        , true /* needValidation */
    );

    TMap<TString, TString> result2 = NSupport::FillReadinessOrLivenessContainerReplaceMap(
        container
        , "readiness"
        , "readiness_container"
        , "path/"
        , "/pod_agent"
        , {} /* env */
        , {} /* secretMap */
        , pathHolder
        , 1.0
        , true /* needTimeLimit */
        , true /* needValidation */
    );

    UNIT_ASSERT_EQUAL(result1, result2);
}

Y_UNIT_TEST(TestFillContainerReplaceMapNoNeedTimeLimit) {
    API::TUtilityContainer container;
    container.set_command_line("echo start");

    UNIT_ASSERT_NO_EXCEPTION(
        NSupport::FillContainerReplaceMap(
            container
            , "start"
            , "" /* portoContainerName */
            , "" /* rootfsPath */
            , "" /* podAgentBinaryFilePath */
            , {} /* env */
            , {} /* secretMap */
            , GetPathHolder()
            , 1.0
            , false /* needTimeLimit */
            , true /* needValidation */
        )
    );
}

Y_UNIT_TEST(TestFillContainerReplaceMapNoValidation) {
    UNIT_ASSERT_NO_EXCEPTION(
        NSupport::FillContainerReplaceMap(
            {}
            , "start"
            , "" /* portoContainerName */
            , "" /* rootfsPath */
            , "" /* podAgentBinaryFilePath */
            , {} /* env */
            , {} /* secretMap */
            , GetPathHolder()
            , 1.0
            , true /* needTimeLimit */
            , false /* needValidation */
        )
    );
}

Y_UNIT_TEST(TestFillInitContainersReplaceMap) {
    const i32 SIZE = 10;
    const TVector<API::ECpuPolicy> cpuPolicies = {
        API::ECpuPolicy_UNKNOWN
        , API::ECpuPolicy_NORMAL
        , API::ECpuPolicy_HIGH
        , API::ECpuPolicy_BATCH
        , API::ECpuPolicy_IDLE
        , API::ECpuPolicy_ISO
    };
    const TVector<API::EIoPolicy> ioPolicies = {
        API::EIoPolicy_UNKNOWN
        , API::EIoPolicy_NONE
        , API::EIoPolicy_NORMAL
        , API::EIoPolicy_HIGH
        , API::EIoPolicy_BATCH
        , API::EIoPolicy_IDLE
    };
    API::TBox box;
    TVector<TString> containerNames;
    for (i32 i = 0; i < SIZE; ++i) {
        API::TUtilityContainer* container = box.add_init();
        container->mutable_time_limit()->set_initial_delay_ms(1 + i);
        container->mutable_time_limit()->set_restart_period_scale_ms(2 + i);
        container->mutable_time_limit()->set_restart_period_back_off(3 + i);
        container->mutable_time_limit()->set_min_restart_period_ms(4 + i);
        container->mutable_time_limit()->set_max_restart_period_ms(5 + i);
        container->mutable_time_limit()->set_max_execution_time_ms(6 + i);
        container->mutable_compute_resources()->set_vcpu_guarantee(7 + i);
        container->mutable_compute_resources()->set_vcpu_limit(8 + i);
        container->mutable_compute_resources()->set_cpu_policy(cpuPolicies[i % cpuPolicies.size()]);
        container->mutable_compute_resources()->set_cpu_weight(9 + i);
        container->mutable_compute_resources()->set_memory_guarantee(10 + i);
        container->mutable_compute_resources()->set_memory_limit(11 + i);
        container->mutable_compute_resources()->set_anonymous_memory_limit(12 + i);
        container->mutable_compute_resources()->set_recharge_on_pgfault(i % 2);
        container->mutable_compute_resources()->add_io_limit()->CopyFrom(GetIoLimit("/tmp" + ToString(13 + i), API::EIoLimitType_READ, 14 + i));
        container->mutable_compute_resources()->add_io_limit()->CopyFrom(GetIoLimit("/tmp" + ToString(15 + i), API::EIoLimitType_WRITE, 16 + i));
        container->mutable_compute_resources()->add_io_ops_limit()->CopyFrom(GetIoLimit("/tmp" + ToString(17 + i), API::EIoLimitType_READ, 18 + i));
        container->mutable_compute_resources()->add_io_ops_limit()->CopyFrom(GetIoLimit("/tmp" + ToString(19 + i), API::EIoLimitType_WRITE, 20 + i));
        container->mutable_compute_resources()->set_io_policy((ioPolicies[i % ioPolicies.size()]));
        container->mutable_compute_resources()->set_io_weight(21 + i);
        container->set_command_line("echo init " + ToString(i));
        container->set_cwd("dir/" + ToString(i));
        container->set_stdout_file("stdout_file" + ToString(i));
        container->set_stderr_file("stderr_file" + ToString(i));
        container->set_stdout_and_stderr_limit(22 + i);
        container->set_core_command("echo core " + ToString(i));
        container->set_user("user" + ToString(i));
        container->set_group("group" + ToString(i));
        containerNames.push_back("container" + ToString(i));
    }

    TMap<TString, TString> result = NSupport::FillInitContainersReplaceMap(
        box.init()
        , containerNames
        , "path/"
        , "/pod_agent"
        , {} /* env */
        , {} /* secretMap */
        , GetPathHolder()
        , 1.0
    );

    UNIT_ASSERT_EQUAL_C(
        result["INIT_CMD_LIST"]
        , TStringBuilder()
            << "/pod_agent exec_wrapper user0 group0 echo init 0;/pod_agent exec_wrapper user1 group1 echo init 1;/pod_agent exec_wrapper user2 group2 echo init 2;"
            << "/pod_agent exec_wrapper user3 group3 echo init 3;/pod_agent exec_wrapper user4 group4 echo init 4;/pod_agent exec_wrapper user5 group5 echo init 5;"
            << "/pod_agent exec_wrapper user6 group6 echo init 6;/pod_agent exec_wrapper user7 group7 echo init 7;/pod_agent exec_wrapper user8 group8 echo init 8;"
            << "/pod_agent exec_wrapper user9 group9 echo init 9;"
        , result["INIT_CMD_LIST"]
    );
    UNIT_ASSERT_EQUAL_C(
        result["INIT_CWD_LIST"]
        , "dir/0;dir/1;dir/2;dir/3;dir/4;dir/5;dir/6;dir/7;dir/8;dir/9;"
        , result["INIT_CWD_LIST"]
    );
    UNIT_ASSERT_EQUAL_C(
        result["INIT_CORE_COMMAND_LIST"]
        , "echo core 0;echo core 1;echo core 2;echo core 3;echo core 4;echo core 5;echo core 6;echo core 7;echo core 8;echo core 9;"
        , result["INIT_CORE_COMMAND_LIST"]
    );
    UNIT_ASSERT_EQUAL_C(
        result["INIT_ENVIRONMENT_LIST"]
        , TStringBuilder()
            << "DEPLOY_CONTAINER_ID=container0;DEPLOY_CONTAINER_ID=container1;DEPLOY_CONTAINER_ID=container2;DEPLOY_CONTAINER_ID=container3;"
            << "DEPLOY_CONTAINER_ID=container4;DEPLOY_CONTAINER_ID=container5;DEPLOY_CONTAINER_ID=container6;DEPLOY_CONTAINER_ID=container7;"
            << "DEPLOY_CONTAINER_ID=container8;DEPLOY_CONTAINER_ID=container9;"
        , result["INIT_ENVIRONMENT_LIST"]
    );

    UNIT_ASSERT_EQUAL_C(
        result["INIT_INITIAL_DELAY_LIST"]
        , "1;2;3;4;5;6;7;8;9;10;"
        , result["INIT_INITIAL_DELAY_LIST"]
    );
    UNIT_ASSERT_EQUAL_C(
        result["INIT_MAX_EXECUTION_TIME_LIST"]
        , "6;7;8;9;10;11;12;13;14;15;"
        , result["INIT_MAX_EXECUTION_TIME_LIST"]
    );
    UNIT_ASSERT_EQUAL_C(
        result["INIT_MAX_RESTART_PERIOD_LIST"]
        , "5;6;7;8;9;10;11;12;13;14;"
        , result["INIT_MAX_RESTART_PERIOD_LIST"]
    );
    UNIT_ASSERT_EQUAL_C(
        result["INIT_MIN_RESTART_PERIOD_LIST"]
        , "4;5;6;7;8;9;10;11;12;13;"
        , result["INIT_MIN_RESTART_PERIOD_LIST"]
    );
    UNIT_ASSERT_EQUAL_C(
        result["INIT_RESTART_PERIOD_BACKOFF_LIST"]
        , "3;4;5;6;7;8;9;10;11;12;"
        , result["INIT_RESTART_PERIOD_BACKOFF_LIST"]
    );
    UNIT_ASSERT_EQUAL_C(
        result["INIT_RESTART_PERIOD_SCALE_LIST"]
        , "2;3;4;5;6;7;8;9;10;11;"
        , result["INIT_RESTART_PERIOD_SCALE_LIST"]
    );

    UNIT_ASSERT_EQUAL_C(
        result["INIT_GROUP_LIST"]
        , "root;root;root;root;root;root;root;root;root;root;"
        , result["INIT_GROUP_LIST"]
    );
    UNIT_ASSERT_EQUAL_C(
        result["INIT_USER_LIST"]
        , "root;root;root;root;root;root;root;root;root;root;"
        , result["INIT_USER_LIST"]
    );

    UNIT_ASSERT_EQUAL_C(
        result["INIT_CPU_GUARANTEE_LIST"]
        , "0.007c;0.008c;0.009c;0.01c;0.011c;0.012c;0.013c;0.014c;0.015c;0.016c;"
        , result["INIT_CPU_GUARANTEE_LIST"]
    );
    UNIT_ASSERT_EQUAL_C(
        result["INIT_CPU_LIMIT_LIST"]
        , "0.008c;0.009c;0.01c;0.011c;0.012c;0.013c;0.014c;0.015c;0.016c;0.017c;"
        , result["INIT_CPU_LIMIT_LIST"]
    );
    UNIT_ASSERT_EQUAL_C(
        result["INIT_CPU_POLICY_LIST"]
        , ";normal;high;batch;idle;iso;;normal;high;batch;"
        , result["INIT_CPU_POLICY_LIST"]
    );
    UNIT_ASSERT_EQUAL_C(
        result["INIT_CPU_WEIGHT_LIST"]
        , "9;10;11;12;13;14;15;16;17;18;"
        , result["INIT_CPU_WEIGHT_LIST"]
    );

    UNIT_ASSERT_EQUAL_C(
        result["INIT_ANON_LIMIT_LIST"]
        , "12;13;14;15;16;17;18;19;20;21;"
        , result["INIT_ANON_LIMIT_LIST"]
    );
    UNIT_ASSERT_EQUAL_C(
        result["INIT_MEMORY_GUARANTEE_LIST"]
        , "10;11;12;13;14;15;16;17;18;19;"
        , result["INIT_MEMORY_GUARANTEE_LIST"]
    );
    UNIT_ASSERT_EQUAL_C(
        result["INIT_MEMORY_LIMIT_LIST"]
        , "11;12;13;14;15;16;17;18;19;20;"
        , result["INIT_MEMORY_LIMIT_LIST"]
    );
    UNIT_ASSERT_EQUAL_C(
        result["INIT_RECHARGE_ON_PGFAULT_LIST"]
        , ";true;;true;;true;;true;;true;"
        , result["INIT_RECHARGE_ON_PGFAULT_LIST"]
    );

    UNIT_ASSERT_EQUAL_C(
        result["INIT_IO_LIMIT_LIST"]
        , TStringBuilder()
            << "/tmp13 r: 14\\;/tmp15 w: 16;/tmp14 r: 15\\;/tmp16 w: 17;/tmp15 r: 16\\;/tmp17 w: 18;"
            << "/tmp16 r: 17\\;/tmp18 w: 19;/tmp17 r: 18\\;/tmp19 w: 20;/tmp18 r: 19\\;/tmp20 w: 21;"
            << "/tmp19 r: 20\\;/tmp21 w: 22;/tmp20 r: 21\\;/tmp22 w: 23;/tmp21 r: 22\\;/tmp23 w: 24;"
            << "/tmp22 r: 23\\;/tmp24 w: 25;"
        , result["INIT_IO_LIMIT_LIST"]
    );
    UNIT_ASSERT_EQUAL_C(
        result["INIT_IO_OPS_LIMIT_LIST"]
        , TStringBuilder()
            << "/tmp17 r: 18\\;/tmp19 w: 20;/tmp18 r: 19\\;/tmp20 w: 21;/tmp19 r: 20\\;/tmp21 w: 22;"
            << "/tmp20 r: 21\\;/tmp22 w: 23;/tmp21 r: 22\\;/tmp23 w: 24;/tmp22 r: 23\\;/tmp24 w: 25;"
            << "/tmp23 r: 24\\;/tmp25 w: 26;/tmp24 r: 25\\;/tmp26 w: 27;/tmp25 r: 26\\;/tmp27 w: 28;"
            << "/tmp26 r: 27\\;/tmp28 w: 29;"
        , result["INIT_IO_OPS_LIMIT_LIST"]
    );
    UNIT_ASSERT_EQUAL_C(
        result["INIT_IO_POLICY_LIST"]
        , ";none;normal;high;batch;idle;;none;normal;high;"
        , result["INIT_IO_POLICY_LIST"]
    );
    UNIT_ASSERT_EQUAL_C(
        result["INIT_IO_WEIGHT_LIST"]
        , "21;22;23;24;25;26;27;28;29;30;"
        , result["INIT_IO_WEIGHT_LIST"]
    );

    UNIT_ASSERT_EQUAL_C(
        result["INIT_STDOUT_LOG_PATH_LIST"]
        , "stdout_file0;stdout_file1;stdout_file2;stdout_file3;stdout_file4;stdout_file5;stdout_file6;stdout_file7;stdout_file8;stdout_file9;"
        , result["INIT_STDOUT_LOG_PATH_LIST"]
    );
    UNIT_ASSERT_EQUAL_C(
        result["INIT_STDERR_LOG_PATH_LIST"]
        , "stderr_file0;stderr_file1;stderr_file2;stderr_file3;stderr_file4;stderr_file5;stderr_file6;stderr_file7;stderr_file8;stderr_file9;"
        , result["INIT_STDERR_LOG_PATH_LIST"]
    );
    UNIT_ASSERT_EQUAL_C(result["INIT_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE_LIST"]
        , "path/stdout_file0;path/stdout_file1;path/stdout_file2;path/stdout_file3;path/stdout_file4;path/stdout_file5;path/stdout_file6;path/stdout_file7;path/stdout_file8;path/stdout_file9;"
        , result["INIT_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE_LIST"]
    );
    UNIT_ASSERT_EQUAL_C(
        result["INIT_STDERR_LOG_FILE_FULL_PATH_TO_CREATE_LIST"]
        , "path/stderr_file0;path/stderr_file1;path/stderr_file2;path/stderr_file3;path/stderr_file4;path/stderr_file5;path/stderr_file6;path/stderr_file7;path/stderr_file8;path/stderr_file9;"
        , result["INIT_STDERR_LOG_FILE_FULL_PATH_TO_CREATE_LIST"]
    );
    UNIT_ASSERT_EQUAL_C(
        result["INIT_STDOUT_AND_STDERR_FILE_SIZE_LIMIT_LIST"]
        , "22;23;24;25;26;27;28;29;30;31;"
        , result["INIT_STDOUT_AND_STDERR_FILE_SIZE_LIMIT_LIST"]
    );

    UNIT_ASSERT_EQUAL_C(
        result["INIT_AGING_TIME_LIST"]
        , "1073741824;1073741824;1073741824;1073741824;1073741824;1073741824;1073741824;1073741824;1073741824;1073741824;"
        , result["INIT_AGING_TIME_LIST"]
    );
}

Y_UNIT_TEST(TestFillVerifyResourceReplaceMap) {
    TString checksum = "MD5:99754106633f94d350db34d548d6091a";
    {
        TMap<TString, TString> result = NSupport::FillVerifyResourceReplaceMap(
            "name"
            , checksum
            , /* disabled = */ false
        );
        UNIT_ASSERT_EQUAL_C(
           result["NAME_VERIFICATION_CMD"]
           , NSupport::GetVerificationCommand(checksum)
           , result["NAME_VERIFICATION_CMD"]
        );
        UNIT_ASSERT_EQUAL_C(
           result["NAME_VERIFICATION_TYPE"]
           , "container"
           , result["NAME_VERIFICATION_TYPE"]
        );
        UNIT_ASSERT_EQUAL_C(
           result["NAME_VERIFICATION_CHECKSUM"]
           , NSupport::GetVerificationChecksumValue(checksum)
           , result["NAME_VERIFICATION_CHECKSUM"]
        );
    }

    {
        auto checksums = TVector<TString>{"EMPTY:", "EMPTY:", checksum};
        auto disabled = TVector<bool>{false, true, true};
        for (size_t i = 0; i < checksums.size(); ++i) {
            TMap<TString, TString> result = NSupport::FillVerifyResourceReplaceMap(
                "another_name"
                , checksums[i]
                , disabled[i]
            );
            UNIT_ASSERT_EQUAL_C(
                result["ANOTHER_NAME_VERIFICATION_CMD"]
                , ""
                , result["ANOTHER_NAME_VERIFICATION_CMD"]
            );
            UNIT_ASSERT_EQUAL_C(
                result["ANOTHER_NAME_VERIFICATION_TYPE"]
                , "empty"
                , result["ANOTHER_NAME_VERIFICATION_TYPE"]
            );
            UNIT_ASSERT_EQUAL_C(
                result["ANOTHER_NAME_VERIFICATION_CHECKSUM"]
                , ""
                , result["ANOTHER_NAME_VERIFICATION_CHECKSUM"]
            );
        }
    }

    {
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            NSupport::FillVerifyResourceReplaceMap(
                "another_name"
                , "EMPTY"
            )
            , yexception
            , "has schema 'EMPTY', but not equal to 'EMPTY:'"
        );
    }
}

Y_UNIT_TEST(TestGetEnvironmentList) {
    google::protobuf::RepeatedPtrField<API::TEnvVar> env;
    NSecret::TSecretMap secretMap;

    API::TEnvVar* secretEnvVar = env.Add();
    secretEnvVar->set_name("TestSecret");
    secretEnvVar->mutable_value()->mutable_secret_env()->set_alias("SecretAlias");
    secretEnvVar->mutable_value()->mutable_secret_env()->set_id("SecretId");

    API::TEnvVar* envVar = env.Add();
    envVar->set_name("TestEnv");
    envVar->mutable_value()->mutable_literal_env()->set_value("TestEnv;\\Value");

    {
        UNIT_ASSERT_EXCEPTION_CONTAINS(NSupport::GetEnvironmentList(env, secretMap, {}), yexception, "Secret Alias 'SecretAlias'");
    }

    {
        secretMap["SecretAlias"]["OtherSecretId"] = {
            "Some;V\\alue"
            , ""
        };
        UNIT_ASSERT_EXCEPTION_CONTAINS(NSupport::GetEnvironmentList(env, secretMap, {}), yexception, "Secret 'SecretAlias::SecretId'");
    }

    {
        NSupport::TEnvironmentList envListWithEnvSecretExpected = {
            .PublicEnv_ = "TestEnv=TestEnv\\;\\\\Value"
            , .SecretEnv_ = "TestSecret=U29tZTtWXGFsdWU="
        };

        NSupport::TEnvironmentList envListWhenDoesNotUseEnvSecretAndDecodingExpected = {
            .PublicEnv_ = envListWithEnvSecretExpected.PublicEnv_ + ";" + envListWithEnvSecretExpected.SecretEnv_
            , .SecretEnv_ = ""
        };
        NSupport::TEnvironmentList envListWithEnvSecretAndDecodingExpected = {
            .PublicEnv_ = "TestEnv=TestEnv\\;\\\\Value"
            , .SecretEnv_ = "TestSecret=Some\\;V\\\\alue"
        };
        secretMap["SecretAlias"]["SecretId"] = {
            "U29tZTtWXGFsdWU="
            , "base64"
        };
        NSupport::TEnvironmentList envListResultWithoutEnvSecretAndDecoding = NSupport::GetEnvironmentList(env, secretMap, {});
        UNIT_ASSERT_EQUAL_C(envListResultWithoutEnvSecretAndDecoding.PublicEnv_, envListWhenDoesNotUseEnvSecretAndDecodingExpected.PublicEnv_, envListResultWithoutEnvSecretAndDecoding.PublicEnv_);
        UNIT_ASSERT_EQUAL_C(envListResultWithoutEnvSecretAndDecoding.SecretEnv_, envListWhenDoesNotUseEnvSecretAndDecodingExpected.SecretEnv_, envListResultWithoutEnvSecretAndDecoding.SecretEnv_);

        NSupport::TEnvironmentList envListWithSecretEnvResult = NSupport::GetEnvironmentList(env, secretMap, {}, /* useEnvSecret = */ true);
        UNIT_ASSERT_EQUAL_C(envListWithSecretEnvResult.PublicEnv_, envListWithEnvSecretExpected.PublicEnv_, envListWithSecretEnvResult.PublicEnv_);
        UNIT_ASSERT_EQUAL_C(envListWithSecretEnvResult.SecretEnv_, envListWithEnvSecretExpected.SecretEnv_, envListWithSecretEnvResult.SecretEnv_);

        NSupport::TEnvironmentList envListWithSecretEnvAndDecodingResult = NSupport::GetEnvironmentList(env, secretMap, {}, /* useEnvSecret = */ true, /* autoDecodeBase64Secrets = */ true);
        UNIT_ASSERT_EQUAL_C(envListWithSecretEnvAndDecodingResult.PublicEnv_, envListWithEnvSecretAndDecodingExpected.PublicEnv_, envListWithSecretEnvAndDecodingResult.PublicEnv_);
        UNIT_ASSERT_EQUAL_C(envListWithSecretEnvAndDecodingResult.SecretEnv_, envListWithEnvSecretAndDecodingExpected.SecretEnv_, envListWithSecretEnvAndDecodingResult.SecretEnv_);
    }

    {
        envVar->set_name("bad=name");
        UNIT_ASSERT_EXCEPTION_CONTAINS(NSupport::GetEnvironmentList(env, secretMap, {}), yexception, "'=' is not allowed");
    }

    {
        envVar->set_name("");
        UNIT_ASSERT_EXCEPTION_CONTAINS(NSupport::GetEnvironmentList(env, secretMap, {}), yexception, "Environment variable can't have empty name");
    }

    {
        envVar->set_name("   name");
        UNIT_ASSERT_EXCEPTION_CONTAINS(NSupport::GetEnvironmentList(env, secretMap, {}), yexception, "First symbol of environment variable name can't be whitespace");
    }

    {
        envVar->set_name("name");
        envVar->mutable_value()->mutable_literal_env()->set_value("value   ");
        UNIT_ASSERT_EXCEPTION_CONTAINS(NSupport::GetEnvironmentList(env, secretMap, {}), yexception, "Last symbol of environment variable value can't be whitespace: 'name=<...>'");
    }

    {
        envVar->mutable_value()->mutable_literal_env()->set_value("value");
        secretMap["SecretAlias"]["SecretId"] = {
            "value   "
            , ""
        };
        UNIT_ASSERT_EXCEPTION_CONTAINS(NSupport::GetEnvironmentList(env, secretMap, {}), yexception, "Last symbol of environment variable value can't be whitespace: 'TestSecret=<...>'");
    }
}

Y_UNIT_TEST(TestGetEnvironmentListWithSystemEnvironment) {
    const TString tvmToolLocalToken = "token";
    const TString podId = "podId";

    API::TNodeMeta nodeMeta;
    nodeMeta.set_dc("node_dc");
    nodeMeta.set_cluster("node_cluster");
    nodeMeta.set_fqdn("node_fqdn");

    API::TGpuManagerMeta gpuManagerMeta;
    gpuManagerMeta.mutable_extra_env()->insert({"GPU_MANAGER_EXTRA_ENV_NAME", "gpu_manager_extra_env_value"});

    NYP::NClient::NApi::NProto::TPodStatus::TDns dns;
    dns.set_persistent_fqdn("pod_persistent_fqdn");
    dns.set_transient_fqdn("pod_transient_fqdn");

    google::protobuf::RepeatedPtrField<NYP::NClient::NApi::NProto::TPodStatus::TIP6AddressAllocation> ip6AddressAllocations;
    ip6AddressAllocations.Add()->set_address("ip0");
    ip6AddressAllocations.Add()->set_address("ip1");

    google::protobuf::RepeatedPtrField<API::TEnvVar> env;
    API::TEnvVar* envVar = env.Add();
    envVar->set_name("TestEnv");
    envVar->mutable_value()->mutable_literal_env()->set_value("Value");

    const NSupport::TEnvironmentList envList = NSupport::GetEnvironmentList(
        env
        , {}
        , NSupport::GetCommonSystemEnvironment(
            tvmToolLocalToken
            , podId
            , nodeMeta
            , gpuManagerMeta
            , dns
            , ip6AddressAllocations
        )
    );

    TString expectedPublicEnvList = TStringBuilder()
        << "DEPLOY_NODE_CLUSTER=node_cluster"
        << ";DEPLOY_NODE_DC=node_dc"
        << ";DEPLOY_NODE_FQDN=node_fqdn"
        << ";DEPLOY_POD_ID=podId"
        << ";DEPLOY_POD_IP_0_ADDRESS=ip0"
        << ";DEPLOY_POD_IP_1_ADDRESS=ip1"
        << ";DEPLOY_POD_PERSISTENT_FQDN=pod_persistent_fqdn"
        << ";DEPLOY_POD_TRANSIENT_FQDN=pod_transient_fqdn"
        << ";GPU_MANAGER_EXTRA_ENV_NAME=gpu_manager_extra_env_value"
        << ";TVMTOOL_LOCAL_AUTHTOKEN=token"
        << ";TestEnv=Value"
    ;
    TString expectedSecretEnvList = "";

    UNIT_ASSERT_EQUAL_C(expectedPublicEnvList, envList.PublicEnv_, envList.PublicEnv_);
    UNIT_ASSERT_EQUAL_C(expectedSecretEnvList, envList.SecretEnv_, envList.SecretEnv_);

    {
        API::TGpuManagerMeta badGpuManagerMeta;
        badGpuManagerMeta.mutable_extra_env()->insert({"DEPLOY_NODE_CLUSTER", "value"});
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            NSupport::GetCommonSystemEnvironment(
                tvmToolLocalToken
                , podId
                , nodeMeta
                , badGpuManagerMeta
                , dns
                , ip6AddressAllocations
            )
            , yexception
            , "DEPLOY_NODE_CLUSTER is not allowed at environment variable name"
        );
    }
}

Y_UNIT_TEST(TestGetContainerSpecificSystemEnvironment) {
    const NSupport::TEnvironmentList envList = NSupport::GetEnvironmentList(
        {} /* env */
        , {} /* secretMap */
        , NSupport::GetContainerSpecificSystemEnvironment("containerName")
    );

    TString expectedPublicEnvList = TStringBuilder() << "DEPLOY_CONTAINER_ID=containerName";
    TString expectedSecretEnvList = "";

    UNIT_ASSERT_EQUAL_C(expectedPublicEnvList, envList.PublicEnv_, envList.PublicEnv_);
    UNIT_ASSERT_EQUAL_C(expectedSecretEnvList, envList.SecretEnv_, envList.SecretEnv_);
}

Y_UNIT_TEST(TestValidateNoCollisionWithSystemEnvironment) {
    google::protobuf::RepeatedPtrField<API::TEnvVar> env;
    API::TEnvVar* envVar = env.Add();
    envVar->set_name("TestEnv");
    envVar->mutable_value()->mutable_literal_env()->set_value("Value");

    UNIT_ASSERT_NO_EXCEPTION(NSupport::ValidateNoCollisionWithSystemEnvironment(env));

    {
        envVar->set_name("DEPLOY_NODE_CLUSTER");
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            NSupport::ValidateNoCollisionWithSystemEnvironment(env)
            , yexception
            , "DEPLOY_NODE_CLUSTER is not allowed at environment variable name"
        );
    }

    {
        envVar->set_name("DEPLOY_POD_IP_something_OTHER_SUFFIX");
        UNIT_ASSERT_NO_EXCEPTION(NSupport::ValidateNoCollisionWithSystemEnvironment(env));
    }

    {
        envVar->set_name("OTHER_PREFIX_DEPLOY_POD_IP_something_ADDRESS");
        UNIT_ASSERT_NO_EXCEPTION(NSupport::ValidateNoCollisionWithSystemEnvironment(env));
    }

    {
        envVar->set_name("DEPLOY_POD_IP_something_ADDRESS");
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            NSupport::ValidateNoCollisionWithSystemEnvironment(env)
            , yexception
            , "DEPLOY_POD_IP_*_ADDRESS is not allowed at environment variable name"
        );
    }

    UNIT_ASSERT_EXCEPTION_CONTAINS(
        NSupport::GetEnvironmentList(env, {}, {})
        , yexception
        , "DEPLOY_POD_IP_*_ADDRESS is not allowed at environment variable name"
    );
}

Y_UNIT_TEST(TestGetULimitList) {
    google::protobuf::RepeatedPtrField<API::TUlimitSoft> ulimitSoft;

    API::TUlimitSoft* ulimit1 = ulimitSoft.Add();
    ulimit1->set_name(API::EContainerULimit_CORE);
    ulimit1->set_value(1);

    API::TUlimitSoft* ulimit2 = ulimitSoft.Add();
    ulimit2->set_name(API::EContainerULimit_CPU);
    ulimit2->set_value(2);

    const TString ulimitList = NSupport::GetULimitList(ulimitSoft);

    TString expectedULimitList = TStringBuilder()
        << "core: 1 1; "
        << "cpu: 2 2; "
    ;

    UNIT_ASSERT_EQUAL_C(
        expectedULimitList
        , ulimitList
        , ulimitList
    );

    {
        ulimit2->set_name(API::EContainerULimit_CORE);
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            NSupport::GetULimitList(ulimitSoft)
            , yexception
            , "Two or more ulimits have the same type: 'core'"
        );
    }
}

Y_UNIT_TEST(TestGetSkyGetDownloadCommand) {
    API::TSkyGetDownload skyGetDownload;

    {
        skyGetDownload.set_resid("");
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            NSupport::GetSkyGetDownloadCommand(skyGetDownload, "")
            , yexception
            , "Sky get resid not provided"
        );
    }

    {
        skyGetDownload.set_resid("rbtorrentt:ebc3ccbbe8ba8eac209a565547433f85");
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            NSupport::GetSkyGetDownloadCommand(skyGetDownload, "")
            , yexception
            , "String 'rbtorrentt:ebc3ccbbe8ba8eac209a565547433f85' doesn't fit the rbtorrent pattern"
        );
    }

    {
        skyGetDownload.set_resid("rbtorrent:ebc3ccbbe8ba8eac209a565547433f85");
        UNIT_ASSERT_NO_EXCEPTION(
            NSupport::GetSkyGetDownloadCommand(skyGetDownload, "")
        );
    }

    {
        UNIT_ASSERT_EQUAL(
            NSupport::GetSkyGetDownloadCommand(skyGetDownload, "")
            , "sky get -p rbtorrent:ebc3ccbbe8ba8eac209a565547433f85"
        );
        UNIT_ASSERT_EQUAL(
            NSupport::GetSkyGetDownloadCommand(skyGetDownload, "downloadDir")
            , "sky get -p -d downloadDir rbtorrent:ebc3ccbbe8ba8eac209a565547433f85"
        );
    }

    {
        skyGetDownload.set_deduplicate(API::ESkyGetDeduplicateMode_HARDLINK);
        UNIT_ASSERT_EQUAL(
            NSupport::GetSkyGetDownloadCommand(skyGetDownload, "")
            , "sky get -p -D Hardlink rbtorrent:ebc3ccbbe8ba8eac209a565547433f85"
        );
        UNIT_ASSERT_EQUAL(
            NSupport::GetSkyGetDownloadCommand(skyGetDownload, "downloadDir")
            , "sky get -p -d downloadDir -D Hardlink rbtorrent:ebc3ccbbe8ba8eac209a565547433f85"
        );
    }

    {
        skyGetDownload.set_deduplicate(API::ESkyGetDeduplicateMode_SYMLINK);
        UNIT_ASSERT_EQUAL(
            NSupport::GetSkyGetDownloadCommand(skyGetDownload, "")
            , "sky get -p -D Symlink rbtorrent:ebc3ccbbe8ba8eac209a565547433f85"
        );
        UNIT_ASSERT_EQUAL(
            NSupport::GetSkyGetDownloadCommand(skyGetDownload, "downloadDir")
            , "sky get -p -d downloadDir -D Symlink rbtorrent:ebc3ccbbe8ba8eac209a565547433f85"
        );
    }
}

Y_UNIT_TEST(TestNormalizeMountPoint) {
    TVector<TString> emptyOrSlah = {"", "/"};

    for (const TString& firstSymb : emptyOrSlah) {
        for (const TString& lastSymb : emptyOrSlah) {
            UNIT_ASSERT_EQUAL_C(
                NSupport::NormalizeMountPoint(firstSymb + "mount_point" + lastSymb)
                , "mount_point"
                , NSupport::NormalizeMountPoint(firstSymb + "mount_point" + lastSymb)
            );
        }
    }

    UNIT_ASSERT_EXCEPTION_CONTAINS(NSupport::NormalizeMountPoint("/"), yexception, "Mount point can't be '/'");
    UNIT_ASSERT_EXCEPTION_CONTAINS(NSupport::NormalizeMountPoint(""), yexception, "Mount point can't be empty");
}

Y_UNIT_TEST(TestValidateMountPoints) {
    TVector<TString> mountPointsCorrect = {
        "/mount_point_1"
        , "/mount_point_11"
        , "/mount_point_2"
        , "/mount_point_3/dir1"
        , "/mount_point_3/dir2"
        , "/mount_point_3/dir3/dir4"
    };

    TVector<TString> mountPointsNoFirstSlash = {
        "/mount_point_1"
        , "mount_point_2"
    };

    TVector<TString> mountPointsLastSlash = {
        "/mount_point_1"
        , "/mount_point_2/"
    };

    TVector<TString> mountPointsNested = {
        "/mount_point_1"
        , "/mount_point_11"
        , "/mount_point_1/dir1/dir2"
    };

    TVector<TString> mountPointsSame = {
        "/mount_point_1"
        , "/mount_point_1"
    };

    UNIT_ASSERT_NO_EXCEPTION(NSupport::ValidateMountPoints(mountPointsCorrect));

    UNIT_ASSERT_EXCEPTION_CONTAINS(
        NSupport::ValidateMountPoints(mountPointsNoFirstSlash)
        , yexception
        , "Mount point should start with slash: 'mount_point_2'"
    );
    UNIT_ASSERT_EXCEPTION_CONTAINS(
        NSupport::ValidateMountPoints(mountPointsLastSlash)
        , yexception
        , "Mount point should not end with slash: '/mount_point_2/'"
    );
    UNIT_ASSERT_EXCEPTION_CONTAINS(
        NSupport::ValidateMountPoints(mountPointsNested)
        , yexception
        , "Mount point '/mount_point_1/dir1/dir2/' is a nested mount point of '/mount_point_1/'"
    );
    UNIT_ASSERT_EXCEPTION_CONTAINS(
        NSupport::ValidateMountPoints(mountPointsSame)
        , yexception
        , "Mount point '/mount_point_1/' is a nested mount point of '/mount_point_1/'"
    );
}

}

} // namespace NInfra::NPodAgent::NSupportFunctionsTest
