#include "trees_generator.h"
#include "support_functions.h"

#include <infra/pod_agent/libs/common/common.h>
#include <infra/pod_agent/libs/pod_agent/trees_generators/proto_hasher.h>
#include <infra/pod_agent/libs/util/string_utils.h>

#include <google/protobuf/util/message_differencer.h>

#include <util/generic/set.h>

namespace NInfra::NPodAgent {

namespace {
    const ui64 MAX_DIFF_SIZE = 1024;

    // returns true if equal
    // ignores some fields
    // generates diff with secrets hidden
    bool RequestsDiff(const API::TPodAgentRequest& constOldReq, const API::TPodAgentRequest& constNewReq, TString& diff) {
        API::TPodAgentRequest oldReq = constOldReq;
        API::TPodAgentRequest newReq = constNewReq;

        // ignore immutable_meta
        // (allow immutable_meta changes without changing spec_timestamp)
        oldReq.clear_immutable_meta();
        newReq.clear_immutable_meta();

        // DEPLOY-2008
        // In some valid cases it is possible that total capacity will change without changing spec_timestamp
        // However it is forbidden to change cpu_to_vcpu_factor without changing spec_timestamp
        oldReq.mutable_node_entry()->mutable_cpu()->clear_total_capacity();
        newReq.mutable_node_entry()->mutable_cpu()->clear_total_capacity();

        // DEPLOY-3243
        // Keys resource can change without changing spec_timestamp
        oldReq.mutable_node_entry()->clear_keys_resource_id();
        newReq.mutable_node_entry()->clear_keys_resource_id();

        {
            // TODO(ISS-6740) remove after release
            // ignore changes caused by iss-agent release

            if (!oldReq.ip6_address_allocations_size()) {
                newReq.clear_ip6_address_allocations();
            }
            if (!oldReq.has_dns()) {
                newReq.clear_dns();
            }
        }

        {
            // TODO(DEPLOY-2227) remove after release
            // ignore changes caused by iss-agent release

            if (!oldReq.disk_volume_allocations_size()) {
                newReq.clear_disk_volume_allocations();
            }
        }


        {
            // Correct order of secrets
            std::sort(
                oldReq.mutable_secrets()->begin()
                , oldReq.mutable_secrets()->end()
                , [](const NYP::NClient::NNodes::NProto::TPodSpec::TSecret& l, const NYP::NClient::NNodes::NProto::TPodSpec::TSecret& r) {
                    return l.id() < r.id();
                }
            );
            std::sort(
                newReq.mutable_secrets()->begin()
                , newReq.mutable_secrets()->end()
                , [](const NYP::NClient::NNodes::NProto::TPodSpec::TSecret& l, const NYP::NClient::NNodes::NProto::TPodSpec::TSecret& r) {
                    return l.id() < r.id();
                }
            );
        }

        for (i32 i = 0; i < Max(oldReq.secrets().size(), newReq.secrets().size()); ++i) {
            // Now the secrets come from both schemas
            // So we have to cut out their values from both old schema and new schema

            {
                // New secrets delivery schema
                google::protobuf::RepeatedPtrField<NYP::NClient::NNodes::NProto::TPodSpec::TSecret::TValue> emptyValues;
                auto& oldValues = (i < oldReq.secrets().size())
                    ? *(oldReq.mutable_secrets(i)->mutable_values())
                    : emptyValues
                ;
                auto& newValues = (i < newReq.secrets().size())
                    ? *(newReq.mutable_secrets(i)->mutable_values())
                    : emptyValues
                ;

                for (i32 j = 0; j < Max(oldValues.size(), newValues.size()); ++j) {
                    if (j < oldValues.size() && j < newValues.size() && oldValues[j].value() == newValues[j].value()) {
                        // Out of harm's way
                        oldValues[j].set_value("");
                        newValues[j].set_value("");
                    } else {
                        if (j < oldValues.size()) {
                            oldValues[j].set_value("<old>");
                        }
                        if (j < newValues.size()) {
                            newValues[j].set_value("<new>");
                        }
                    }
                }
            }

            {
                // Old secrets delivery schema
                google::protobuf::Map<TString, TString> emptyMap;
                auto& oldPayload = (i < oldReq.secrets().size())
                    ? *(oldReq.mutable_secrets(i)->mutable_payload())
                    : emptyMap
                ;
                auto& newPayload = (i < newReq.secrets().size())
                    ? *(newReq.mutable_secrets(i)->mutable_payload())
                    : emptyMap
                ;

                THashSet<TString> keys;
                for (auto& it : oldPayload) {
                    keys.insert(it.first);
                }
                for (auto& it : newPayload) {
                    keys.insert(it.first);
                }

                for (auto& key : keys) {
                    auto oldIt = oldPayload.find(key);
                    auto newIt = newPayload.find(key);
                    if (oldIt != oldPayload.end() && newIt != newPayload.end() && oldIt->second == newIt->second) {
                        // Out of harm's way
                        oldIt->second = "";
                        newIt->second = "";
                    } else {
                        if (oldIt != oldPayload.end()) {
                            oldIt->second = "<old>";
                        }
                        if (newIt != newPayload.end()) {
                            newIt->second = "<new>";
                        }
                    }
                }
            }
        }

        google::protobuf::util::MessageDifferencer messageDifferencer;
        messageDifferencer.ReportDifferencesToString(&diff);
        return messageDifferencer.Compare(oldReq, newReq);
    }
} // namespace

TTreesGenerator::TTreesGenerator(
    TLogger& logger
    , const TCacheConfig& cacheConfig
    , const TLogsTransmitterConfig& logsTransmitterConfig
    , const TResourcesConfig& resourcesConfig
    , const TString& hostname
    , const TString& ytPath
    , const TString& baseSearchPath
    , const TString& publicVolumePath
    , const TString& publicVolumeMountPath
    , const TString& podAgentBinaryFileName
    , const TString& networkDevice
    , bool isPossibleToTransmitSystemLogs
    , TAsyncIpClientPtr asyncIpClient
    , TAsyncPortoClientPtr asyncPorto
    , TNetworkClientPtr networkClient
    , TPathHolderPtr pathHolder
    , TPosixWorkerPtr posixWorker
    , TStatusNTickerHolderPtr statusNTickerHolder
    , TTemplateBTStoragePtr templateBTStorage
    , const bool isBoxAgentMode
)
    : StatusNTickerHolder_(statusNTickerHolder)
    , IsPossibleToTransmitSystemLogs_(isPossibleToTransmitSystemLogs)
    , IsBoxAgentMode_(isBoxAgentMode)
{
    const TString podAgentBinaryFilePathInBox = TStringBuilder()
        << "/" << NSupport::NormalizeMountPoint(publicVolumeMountPath)
        << "/" << podAgentBinaryFileName
    ;

    StaticResourceTreeGenerator_ = MakeHolder<TStaticResourceTreeGenerator>(
        logger
        , pathHolder
        , templateBTStorage->Get("StaticResourceTree")
        , asyncPorto
        , posixWorker
        , statusNTickerHolder
        , templateBTStorage
        , resourcesConfig.GetContainerUser()
        , resourcesConfig.GetContainerGroup()
        , resourcesConfig.GetUseNewProtocolForSecretsInFiles()
    );
    LayerTreeGenerator_ = MakeHolder<TLayerTreeGenerator>(
        logger
        , pathHolder
        , templateBTStorage->Get("LayerTree")
        , asyncPorto
        , posixWorker
        , statusNTickerHolder
        , templateBTStorage
        , resourcesConfig.GetContainerUser()
        , resourcesConfig.GetContainerGroup()
    );
    VolumeTreeGenerator_ = MakeHolder<TVolumeTreeGenerator>(
        logger
        , pathHolder
        , templateBTStorage->Get("VolumeTree")
        , asyncPorto
        , posixWorker
        , statusNTickerHolder
        , templateBTStorage
    );
    BoxTreeGenerator_ = MakeHolder<TBoxTreeGenerator>(
        logger
        , pathHolder
        , templateBTStorage->Get("BoxTree")
        , asyncIpClient
        , asyncPorto
        , posixWorker
        , statusNTickerHolder
        , templateBTStorage
        , cacheConfig.GetVolumePath() + "/" + cacheConfig.GetBoxesCacheFileName()
        , hostname
        , ytPath
        , baseSearchPath
        , publicVolumePath
        , publicVolumeMountPath
        , podAgentBinaryFilePathInBox
        , networkDevice
    );
    WorkloadTreeGenerator_ = MakeHolder<TWorkloadTreeGenerator>(
        logger
        , pathHolder
        , templateBTStorage->Get("WorkloadTree")
        , asyncPorto
        , posixWorker
        , networkClient
        , statusNTickerHolder
        , templateBTStorage
        , podAgentBinaryFilePathInBox
        , logsTransmitterConfig.GetMaxLogsFileSize()
        , IsBoxAgentMode_
    );

    TMultiUnistat::Instance().DrillFloatHole(
        TMultiUnistat::ESignalNamespace::INFRA
        , COUNTER_NON_INCREMENTED_SPEC_TIMESTAMP_ERRORS
        , "deee"
        , NUnistat::TPriority(TMultiUnistat::ESignalPriority::INFRA_INFO)
        , NUnistat::TStartValue(0)
        , EAggregationType::Sum
    );
}

API::TPodAgentStatus TTreesGenerator::UpdatePodAgentRequest(const API::TPodAgentRequest& podAgentRequest, TLogFramePtr logFrame) {
    logFrame->LogEvent(NLogEvent::TUpdatePodAgentRequestStart(podAgentRequest.spec_timestamp()));

    {
        TString diff;
        bool equal = RequestsDiff(Request_, podAgentRequest, diff);
        if (equal) {
            logFrame->LogEvent(NLogEvent::TUpdatePodAgentRequestSkip(TStringBuilder() << "Already updated to spec_timestamp " << podAgentRequest.spec_timestamp()));
            return StatusNTickerHolder_->GetStatusRepositoryTotalStatus(false); // nothing to update
        }

        // Log diff before any exception thrown
        logFrame->LogEvent(
            NLogEvent::TUpdatePodAgentRequestDiff(
                diff
                , Request_.spec_timestamp()
                , podAgentRequest.spec_timestamp()
            )
        );

        if (Request_.spec_timestamp() >= podAgentRequest.spec_timestamp()) {
            if (!TMultiUnistat::Instance().PushSignalUnsafe(TMultiUnistat::ESignalNamespace::INFRA, COUNTER_NON_INCREMENTED_SPEC_TIMESTAMP_ERRORS, 1)) {
                logFrame->LogEvent(ELogPriority::TLOG_ERR, NLogEvent::TUpdatePodAgentRequestError(
                    TStringBuilder() << "PushSingalUnsafe(" << COUNTER_NON_INCREMENTED_SPEC_TIMESTAMP_ERRORS << ", " << ToString(1) << ") returned false"
                ));
            }
            ythrow yexception() << "Request changed but spec timestamp wasn't incremented."
                << " Current spec_timestamp: " << Request_.spec_timestamp()
                << ", request spec_timestamp: " << podAgentRequest.spec_timestamp() << "."
                << " Diff: " << diff.substr(0, MAX_DIFF_SIZE);
        }
    }

    NSecret::TSecretMap secretMap = NSecret::GetSecretMap(podAgentRequest.secrets());

    const API::TPodAgentSpec& podAgentSpec = podAgentRequest.spec();

    bool useEnvSecret = GetBooleanDeployLabel(podAgentRequest, "use_env_secret");

    // Interpret UNKNOWN as ACTIVE
    const API::EPodAgentTargetState podAgentTargetState = (podAgentSpec.target_state() == API::EPodAgentTargetState_UNKNOWN)
        ? API::EPodAgentTargetState_ACTIVE
        : podAgentSpec.target_state()
    ;

    Y_ENSURE(
        podAgentTargetState == API::EPodAgentTargetState_ACTIVE
        || podAgentTargetState == API::EPodAgentTargetState_SUSPENDED
        || podAgentTargetState == API::EPodAgentTargetState_REMOVED

        , "Unknown pod agent target state " << Quote(API::EPodAgentTargetState_Name(podAgentTargetState));
    );

    const API::EAutoDecodeBase64Secrets autoDecodeBase64SecretsValue = podAgentSpec.auto_decode_base64_secrets();
    Y_ENSURE(
        autoDecodeBase64SecretsValue == API::EAutoDecodeBase64Secrets_NONE
        || autoDecodeBase64SecretsValue == API::EAutoDecodeBase64Secrets_DISABLED
        || autoDecodeBase64SecretsValue == API::EAutoDecodeBase64Secrets_ENABLED
        , "Unknown auto decoding base64 secrets state " << Quote(API::EAutoDecodeBase64Secrets_Name(autoDecodeBase64SecretsValue))
    );
    bool autoDecodeBase64Secrets = autoDecodeBase64SecretsValue == API::EAutoDecodeBase64Secrets_ENABLED;

    TStaticResourceTreeGenerator::TStaticResourcesToAdd staticResourcesToAdd;
    TLayerTreeGenerator::TLayersToAdd layersToAdd;
    TVolumeTreeGenerator::TVolumesToAdd volumesToAdd;
    TBoxTreeGenerator::TBoxesToAdd boxesToAdd;

    if (!IsBoxAgentMode_) {
        // NOTE: order is important
        staticResourcesToAdd = StaticResourceTreeGenerator_->UpdateSpec(
            podAgentSpec.resources()
            , podAgentTargetState
            , secretMap
            , podAgentRequest.node_entry().cpu().cpu_to_vcpu_factor()
            , podAgentRequest.spec_timestamp()
            , podAgentSpec.revision()
            , autoDecodeBase64Secrets
        );
        layersToAdd = LayerTreeGenerator_->UpdateSpec(
            podAgentSpec.resources()
            , podAgentTargetState
            , podAgentRequest.node_entry().cpu().cpu_to_vcpu_factor()
            , podAgentRequest.spec_timestamp()
            , podAgentSpec.revision()
        );
        volumesToAdd = VolumeTreeGenerator_->UpdateSpec(
            podAgentSpec.volumes()
            , podAgentTargetState
            , layersToAdd
            , staticResourcesToAdd
            , podAgentRequest.spec_timestamp()
            , podAgentSpec.revision()
        );
        boxesToAdd = BoxTreeGenerator_->UpdateSpec(
            podAgentSpec.boxes()
            , podAgentTargetState
            , layersToAdd
            , staticResourcesToAdd
            , volumesToAdd
            , podAgentRequest.immutable_meta().pod_id()
            , podAgentRequest.immutable_meta().node_meta()
            , podAgentRequest.immutable_meta().gpu_manager_meta()
            , podAgentRequest.dns()
            , podAgentRequest.ip6_subnet_allocations()
            , podAgentRequest.ip6_address_allocations()
            , secretMap
            , podAgentRequest.pod_dynamic_attributes().annotations()
            , podAgentRequest.node_entry().cpu().cpu_to_vcpu_factor()
            , podAgentRequest.spec_timestamp()
            , podAgentSpec.revision()
            , logFrame
            , useEnvSecret
            , autoDecodeBase64Secrets
        );
    }

    auto workloadsToAdd = WorkloadTreeGenerator_->UpdateSpec(
        podAgentSpec.workloads()
        , podAgentTargetState
        , podAgentSpec.mutable_workloads()
        , secretMap
        , boxesToAdd
        , podAgentRequest.node_entry().cpu().cpu_to_vcpu_factor()
        , podAgentRequest.spec_timestamp()
        , podAgentSpec.revision()
        , useEnvSecret
        , autoDecodeBase64Secrets
    );

    // Update spec
    // NOTE: order is important
    WorkloadTreeGenerator_->RemoveWorkloads(workloadsToAdd);
    BoxTreeGenerator_->RemoveBoxes(boxesToAdd);
    VolumeTreeGenerator_->RemoveVolumes(volumesToAdd);
    LayerTreeGenerator_->RemoveLayers(layersToAdd);
    StaticResourceTreeGenerator_->RemoveStaticResources(staticResourcesToAdd);
    StaticResourceTreeGenerator_->AddStaticResources(staticResourcesToAdd);
    LayerTreeGenerator_->AddLayers(layersToAdd);
    VolumeTreeGenerator_->AddVolumes(volumesToAdd);
    BoxTreeGenerator_->AddBoxes(boxesToAdd, logFrame);
    WorkloadTreeGenerator_->AddWorkloads(workloadsToAdd);

    if (!IsBoxAgentMode_) {
        auto cacheLayersToAdd = LayerTreeGenerator_->UpdateResourceCache(
            podAgentRequest.resource_cache_spec()
            , podAgentTargetState
            , podAgentSpec.resources().compute_resources()
            , podAgentRequest.node_entry().cpu().cpu_to_vcpu_factor()
        );
        auto cacheStaticResourcesToAdd = StaticResourceTreeGenerator_->UpdateResourceCache(
            podAgentRequest.resource_cache_spec()
            , podAgentTargetState
            , secretMap
            , podAgentSpec.resources().compute_resources()
            , podAgentRequest.node_entry().cpu().cpu_to_vcpu_factor()
            , autoDecodeBase64Secrets
        );
        // Update resource cache
        LayerTreeGenerator_->AddAndRemoveCacheLayers(cacheLayersToAdd);
        StaticResourceTreeGenerator_->AddAndRemoveCacheStaticResources(cacheStaticResourcesToAdd);
    }

    // Update ResourceGang
    StatusNTickerHolder_->UpdateActiveDownloadContainersLimit(
        podAgentSpec.resources().active_download_containers_limit() == 0
            ? NSupport::DEFAULT_ACTIVE_DOWNLOAD_CONTAINERS_LIMIT
            : podAgentSpec.resources().active_download_containers_limit()
    );
    // Update ResourceGang
    StatusNTickerHolder_->UpdateActiveVerifyContainersLimit(
        podAgentSpec.resources().active_verify_containers_limit() == 0
            ? NSupport::DEFAULT_ACTIVE_VERIFY_CONTAINERS_LIMIT
            : podAgentSpec.resources().active_verify_containers_limit()
    );

    // Update system logs transmit
    bool transmitSystemLogs = GetTransmitSystemLogs(podAgentSpec, IsPossibleToTransmitSystemLogs_);
    StatusNTickerHolder_->EnableTransmitSystemLogs(transmitSystemLogs);

    // NOTE: StatusRepository Revision must update after trees update
    StatusNTickerHolder_->SetStatusRepositorySpecTimestamp(podAgentRequest.spec_timestamp());
    StatusNTickerHolder_->SetStatusRepositoryRevision(podAgentSpec.revision());
    StatusNTickerHolder_->SetStatusRepositorySpecId(podAgentSpec.id());
    StatusNTickerHolder_->SetStatusRepositoryTargetState(podAgentTargetState);

    // NOTE: We must update state request only after validation
    Request_ = podAgentRequest;

    logFrame->LogEvent(NLogEvent::TUpdatePodAgentRequestEnd(podAgentRequest.spec_timestamp()));
    return StatusNTickerHolder_->GetStatusRepositoryTotalStatus(false);
}

API::TPodAgentStatus TTreesGenerator::GetStatus(bool conditionsOnly) {
    return StatusNTickerHolder_->GetStatusRepositoryTotalStatus(conditionsOnly);
}

bool TTreesGenerator::GetTransmitSystemLogs(const API::TPodAgentSpec& podAgentSpec, bool isPossibleToTransmitSystemLogs) {
    if (!isPossibleToTransmitSystemLogs) {
        return false;
    }

    if (!podAgentSpec.has_transmit_system_logs_policy()) {
        return false; 
    }

    API::TTransmitSystemLogsPolicy policy = podAgentSpec.transmit_system_logs_policy();

    API::ETransmitSystemLogs transmitSystemLogs = policy.transmit_system_logs();

    Y_ENSURE(
        transmitSystemLogs == API::ETransmitSystemLogsPolicy_NONE
        || transmitSystemLogs == API::ETransmitSystemLogsPolicy_DISABLED
        || transmitSystemLogs == API::ETransmitSystemLogsPolicy_ENABLED

        , "Unknown transmit system logs value '" << API::ETransmitSystemLogs_Name(transmitSystemLogs) << "'"
    );

    return transmitSystemLogs == API::ETransmitSystemLogsPolicy_ENABLED;
}

} // namespace NInfra::NPodAgent
