#include "trees_generator.h"

#include "test_canon.h"
#include "test_functions.h"

#include <infra/pod_agent/libs/ip_client/mock_client.h>
#include <infra/pod_agent/libs/network_client/mock_client.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/system/fs.h>

namespace NInfra::NPodAgent::NTreeGeneratorTest {

class TTemplateBTStorageMock: public ITemplateBTStorage {
public:
    TBehavior3 Get(const TString& /*key*/) const override {
        TMap<TString, TString> properties;
        properties["message"] = "hello";
        return GetTestTemplateTree(properties, "Succeeder");
    }

    bool Has(const TString& key) const override {
        return key != "Succeeder";
    }
};

class ITestTreesGeneratorCanon: public ITestTreeGeneratorCanon {
public:
    ITestTreesGeneratorCanon() : ITestTreeGeneratorCanon()
    {}

protected:
    void CreateWorker() final {
        // TreeGenerator assumes that this directory always exist
        const TString publicVolumePath = NFs::CurrentWorkingDirectory() + "/public_volume";
        NFs::MakeDirectory(publicVolumePath);

        Worker_ = new TTreesGenerator(
            logger
            , {} /* cacheConfig */
            , {} /* logsTransmitterConfig */
            , {} /* resourcesConfig */
            , "hostname"
            , "/yt"
            , "/basesearch"
            , publicVolumePath
            , "/pod_agent_public" /* publicVolumeMountPath */
            , "pod_agent" /* podAgentBinaryFileName */
            , "" /* networkDevice */
            , false /* is possible to transmit system logs */
            , new TAsyncIpClient(new TMockIpClient(), new TFakeThreadPool())
            , new TAsyncPortoClient(new TMockPortoClient(), new TFakeThreadPool())
            , new TMockNetworkClient()
            , PathHolder_
            , new TPosixWorker(new TFakeThreadPool())
            , StatusNTickerHolder_
            , new TTemplateBTStorageMock()
        );
    }

protected:
    TSimpleSharedPtr<TTreesGenerator> Worker_;
};

API::TPodAgentRequest MakePodAgentRequest(const API::TPodAgentSpec& spec, ui64 specTimestamp) {
    API::TPodAgentRequest podAgentRequest;
    podAgentRequest.set_spec_timestamp(specTimestamp);
    podAgentRequest.mutable_spec()->CopyFrom(spec);

    return podAgentRequest;
}

API::TPodAgentSpec CreateSpec() {
    API::TPodAgentSpec spec;

    spec.set_revision(1);

    API::TResourceGang* gang = spec.mutable_resources();
    API::TLayer* layer = gang->add_layers();
    layer->set_id("my_layer");
    layer->set_url("raw:some_data");
    layer->set_checksum("MD5:3ea05169a95365cbf9b4ff4688cd8945");
    layer = gang->add_layers();
    layer->set_id("my_layer_1");
    layer->set_url("raw:some_data_1");
    layer->set_checksum("MD5:3ea05169a95365cbf9b4ff4688cd8945");

    API::TResource* staticResource = gang->add_static_resources();
    staticResource->set_id("my_static_resource");
    staticResource->set_url("raw:some_data");
    API::TVerification* staticResourceVerification = staticResource->mutable_verification();
    staticResourceVerification->set_checksum("MD5:3ea05169a95365cbf9b4ff4688cd8945");
    staticResource = gang->add_static_resources();
    staticResource->set_id("my_static_resource_1");
    staticResource->set_url("raw:some_data_1");
    staticResourceVerification = staticResource->mutable_verification();
    staticResourceVerification->set_checksum("MD5:3ea05169a95365cbf9b4ff4688cd8945");

    API::TVolume* volume = spec.add_volumes();
    volume->set_id("my_volume");
    API::TGenericVolume* generic = volume->mutable_generic();
    generic->add_layer_refs("my_layer");
    volume = spec.add_volumes();
    volume->set_id("my_volume_1");
    generic = volume->mutable_generic();
    generic->add_layer_refs("my_layer_1");

    API::TBox* box = spec.add_boxes();
    box->set_id("my_box");
    API::TRootfsVolume* rootfs = box->mutable_rootfs();
    rootfs->add_layer_refs("my_layer");
    box = spec.add_boxes();
    box->set_id("my_box_1");
    rootfs = box->mutable_rootfs();
    rootfs->add_layer_refs("my_layer_1");

    API::TWorkload* workload = spec.add_workloads();
    workload->set_id("my_workload");
    workload->set_box_ref("my_box");

    workload->mutable_start()->set_command_line("echo start");
    workload->mutable_readiness_check()->mutable_container()->set_command_line("echo readiness");
    workload->mutable_liveness_check()->mutable_container()->set_command_line("echo liveness");
    workload->mutable_stop_policy()->mutable_container()->set_command_line("echo stop");

    workload->mutable_readiness_check()->mutable_container()->mutable_time_limit()->set_min_restart_period_ms(60000);
    workload->mutable_readiness_check()->mutable_container()->mutable_time_limit()->set_max_restart_period_ms(60000);
    workload->mutable_readiness_check()->mutable_container()->mutable_time_limit()->set_max_execution_time_ms(60000);
    workload->mutable_liveness_check()->mutable_container()->mutable_time_limit()->set_min_restart_period_ms(60000);
    workload->mutable_liveness_check()->mutable_container()->mutable_time_limit()->set_max_restart_period_ms(60000);
    workload->mutable_liveness_check()->mutable_container()->mutable_time_limit()->set_max_execution_time_ms(60000);
    workload->mutable_stop_policy()->mutable_container()->mutable_time_limit()->set_min_restart_period_ms(60000);
    workload->mutable_stop_policy()->mutable_container()->mutable_time_limit()->set_max_restart_period_ms(60000);
    workload->mutable_stop_policy()->mutable_container()->mutable_time_limit()->set_max_execution_time_ms(60000);
    workload->mutable_stop_policy()->set_max_tries(1000);

    workload = spec.add_workloads();
    workload->set_id("my_workload_1");
    workload->set_box_ref("my_box_1");

    workload->mutable_start()->set_command_line("echo start");
    workload->mutable_readiness_check()->mutable_container()->set_command_line("echo readiness");
    workload->mutable_liveness_check()->mutable_container()->set_command_line("echo liveness");
    workload->mutable_stop_policy()->mutable_container()->set_command_line("echo stop");

    workload->mutable_readiness_check()->mutable_container()->mutable_time_limit()->set_min_restart_period_ms(60000);
    workload->mutable_readiness_check()->mutable_container()->mutable_time_limit()->set_max_restart_period_ms(60000);
    workload->mutable_readiness_check()->mutable_container()->mutable_time_limit()->set_max_execution_time_ms(60000);
    workload->mutable_liveness_check()->mutable_container()->mutable_time_limit()->set_min_restart_period_ms(60000);
    workload->mutable_liveness_check()->mutable_container()->mutable_time_limit()->set_max_restart_period_ms(60000);
    workload->mutable_liveness_check()->mutable_container()->mutable_time_limit()->set_max_execution_time_ms(60000);
    workload->mutable_stop_policy()->mutable_container()->mutable_time_limit()->set_min_restart_period_ms(60000);
    workload->mutable_stop_policy()->mutable_container()->mutable_time_limit()->set_max_restart_period_ms(60000);
    workload->mutable_stop_policy()->mutable_container()->mutable_time_limit()->set_max_execution_time_ms(60000);
    workload->mutable_stop_policy()->set_max_tries(1000);

    API::TMutableWorkload* mutableWorkload = spec.add_mutable_workloads();
    mutableWorkload->set_workload_ref("my_workload");
    mutableWorkload->set_target_state(API::EWorkloadTarget_ACTIVE);

    mutableWorkload = spec.add_mutable_workloads();
    mutableWorkload->set_workload_ref("my_workload_1");
    mutableWorkload->set_target_state(API::EWorkloadTarget_ACTIVE);

    return spec;
}

API::TPodAgentRequest CreatePodAgentRequest() {
    API::TPodAgentRequest podAgentRequest = MakePodAgentRequest(CreateSpec(), 1);

    API::TPodAgentResourceCacheSpec* resourceCacheSpec = podAgentRequest.mutable_resource_cache_spec();

    API::TCacheLayer* cacheLayer1 = resourceCacheSpec->add_layers();
    cacheLayer1->set_revision(1);
    cacheLayer1->mutable_layer()->CopyFrom(podAgentRequest.spec().resources().layers(0));

    API::TCacheLayer* cacheLayer2 = resourceCacheSpec->add_layers();
    cacheLayer2->set_revision(1);
    cacheLayer2->mutable_layer()->CopyFrom(podAgentRequest.spec().resources().layers(1));

    API::TCacheResource* cacheStaticResource1 = resourceCacheSpec->add_static_resources();
    cacheStaticResource1->set_revision(1);
    cacheStaticResource1->mutable_resource()->CopyFrom(podAgentRequest.spec().resources().static_resources(0));

    API::TCacheResource* cacheStaticResource2 = resourceCacheSpec->add_static_resources();
    cacheStaticResource2->set_revision(1);
    cacheStaticResource2->mutable_resource()->CopyFrom(podAgentRequest.spec().resources().static_resources(1));

    return podAgentRequest;
}

Y_UNIT_TEST_SUITE(TreesGenerator) {

Y_UNIT_TEST(GetTransmitSystemLogsTest) {
    API::TPodAgentSpec spec;
    UNIT_ASSERT(!TTreesGenerator::GetTransmitSystemLogs(spec, true));

    spec.mutable_transmit_system_logs_policy()->set_transmit_system_logs(API::ETransmitSystemLogs::ETransmitSystemLogsPolicy_NONE);
    UNIT_ASSERT(!TTreesGenerator::GetTransmitSystemLogs(spec, true));

    spec.mutable_transmit_system_logs_policy()->set_transmit_system_logs(API::ETransmitSystemLogs::ETransmitSystemLogsPolicy_DISABLED);
    UNIT_ASSERT(!TTreesGenerator::GetTransmitSystemLogs(spec, true ));

    spec.mutable_transmit_system_logs_policy()->set_transmit_system_logs(API::ETransmitSystemLogs::ETransmitSystemLogsPolicy_ENABLED);
    UNIT_ASSERT(TTreesGenerator::GetTransmitSystemLogs(spec, true));

    UNIT_ASSERT(!TTreesGenerator::GetTransmitSystemLogs(spec, false));
}

Y_UNIT_TEST(ValidateCacheLayerResourceWithTicker) {
    class TTest : public ITestTreesGeneratorCanon {
    public:
        TTest() : ITestTreesGeneratorCanon()
        {}

    protected:
        void Test() override {
            auto request = CreatePodAgentRequest();

            UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdatePodAgentRequest(request, logger.SpawnFrame()));
            *request.mutable_resource_cache_spec()->mutable_layers()->Mutable(0)->mutable_layer()->mutable_url() += "_other";
            request.set_spec_timestamp(2);
            UNIT_ASSERT_EXCEPTION_CONTAINS(Worker_->UpdatePodAgentRequest(request, logger.SpawnFrame()), yexception, "hash was changed");
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(ValidateLayerResourceWithTicker) {
    class TTest : public ITestTreesGeneratorCanon {
    public:
        TTest() : ITestTreesGeneratorCanon()
        {}

    protected:
        void Test() override {
            auto spec = CreateSpec();

            UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdatePodAgentRequest(MakePodAgentRequest(spec, 1), logger.SpawnFrame()));
            *spec.mutable_resources()->mutable_layers()->Mutable(0)->mutable_url() += "_other";
            UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdatePodAgentRequest(MakePodAgentRequest(spec, 2), logger.SpawnFrame()));
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(ValidateCacheStaticResourceResourceWithTicker) {
    class TTest : public ITestTreesGeneratorCanon {
    public:
        TTest() : ITestTreesGeneratorCanon()
        {}

    protected:
        void Test() override {
            auto request = CreatePodAgentRequest();

            UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdatePodAgentRequest(request, logger.SpawnFrame()));
            *request.mutable_resource_cache_spec()->mutable_static_resources()->Mutable(0)->mutable_resource()->mutable_url() += "_other";
            request.set_spec_timestamp(2);
            UNIT_ASSERT_EXCEPTION_CONTAINS(Worker_->UpdatePodAgentRequest(request, logger.SpawnFrame()), yexception, "hash was changed");
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(ValidateStaticResourceResourceWithTicker) {
    class TTest : public ITestTreesGeneratorCanon {
    public:
        TTest() : ITestTreesGeneratorCanon()
        {}

    protected:
        void Test() override {
            auto spec = CreateSpec();

            UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdatePodAgentRequest(MakePodAgentRequest(spec, 1), logger.SpawnFrame()));
            *spec.mutable_resources()->mutable_static_resources()->Mutable(0)->mutable_url() += "_other";
            UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdatePodAgentRequest(MakePodAgentRequest(spec, 2), logger.SpawnFrame()));
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(ValidateVolumeWithTicker) {
    class TTest : public ITestTreesGeneratorCanon {
    public:
        TTest() : ITestTreesGeneratorCanon()
        {}

    protected:
        void Test() override {
            auto spec = CreateSpec();

            UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdatePodAgentRequest(MakePodAgentRequest(spec, 1), logger.SpawnFrame()));
            spec.mutable_volumes(0)->mutable_generic()->set_quota_bytes(1);
            UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdatePodAgentRequest(MakePodAgentRequest(spec, 2), logger.SpawnFrame()));
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(ValidateBoxWithTicker) {
    class TTest : public ITestTreesGeneratorCanon {
    public:
        TTest() : ITestTreesGeneratorCanon()
        {}

    protected:
        void Test() override {
            auto spec = CreateSpec();

            UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdatePodAgentRequest(MakePodAgentRequest(spec, 1), logger.SpawnFrame()));
            spec.mutable_boxes(0)->mutable_rootfs()->set_quota_bytes(spec.boxes(0).rootfs().quota_bytes() + 1);
            UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdatePodAgentRequest(MakePodAgentRequest(spec, 2), logger.SpawnFrame()));
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(ValidateWorkloadWithTicker) {
    class TTest : public ITestTreesGeneratorCanon {
    public:
        TTest() : ITestTreesGeneratorCanon()
        {}

    protected:
        void Test() override {
            auto spec = CreateSpec();

            UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdatePodAgentRequest(MakePodAgentRequest(spec, 1), logger.SpawnFrame()));
            *spec.mutable_workloads(0)->mutable_start()->mutable_command_line() += "_other";
            UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdatePodAgentRequest(MakePodAgentRequest(spec, 2), logger.SpawnFrame()));
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(ValidateSpecTimestamp) {
    class TTest : public ITestTreesGeneratorCanon {
    public:
        TTest() : ITestTreesGeneratorCanon()
        {}

    protected:
        void Test() override {
            auto spec = CreateSpec();

            spec.set_revision(2);
            UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdatePodAgentRequest(MakePodAgentRequest(spec, 1), logger.SpawnFrame()));

            spec.set_revision(0);
            UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdatePodAgentRequest(MakePodAgentRequest(spec, 2), logger.SpawnFrame()));

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                Worker_->UpdatePodAgentRequest(MakePodAgentRequest(spec, 1), logger.SpawnFrame())
                , yexception
                , "Request changed but spec timestamp wasn't incremented. Current spec_timestamp: 2, request spec_timestamp: 1. Diff: modified: spec_timestamp: 2 -> 1"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(ValidateSpecTimestampWithOldSchemaSecretsDiff) {
    class TTest : public ITestTreesGeneratorCanon {
    public:
        TTest() : ITestTreesGeneratorCanon()
        {}

    protected:
        void Test() override {
            auto podAgentRequest = MakePodAgentRequest(CreateSpec(), 1);
            auto* secrets = podAgentRequest.add_secrets();
            secrets->set_id("secret_alias");
            auto* secretsPayload = secrets->mutable_payload();
            (*secretsPayload)["old_key_1"] = "old_value_1";
            (*secretsPayload)["old_key_2"] = "old_value_2";
            (*secretsPayload)["old_key_3"] = "old_value_3";

            UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdatePodAgentRequest(podAgentRequest, logger.SpawnFrame()));

            secretsPayload = podAgentRequest.mutable_secrets(0)->mutable_payload();
            (*secretsPayload)["old_key_1"] = "changed_value";
            (*secretsPayload)["new_key"] = "new_value";
            (*secretsPayload).erase("old_key_3");

            secrets = podAgentRequest.add_secrets();
            secrets->set_id("secret_new_alias");
            secretsPayload = secrets->mutable_payload();
            (*secretsPayload)["new_key"] = "new_value";

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                Worker_->UpdatePodAgentRequest(podAgentRequest, logger.SpawnFrame()),
                yexception,
R"(Request changed but spec timestamp wasn't incremented. Current spec_timestamp: 1, request spec_timestamp: 1. Diff: modified: secrets[0].payload[old_key_1]: "<old>" -> "<new>"
added: secrets[0].payload[new_key]: "<new>"
deleted: secrets[0].payload[old_key_3]: "<old>"
added: secrets[1]: { id: "secret_new_alias" payload { key: "new_key" value: "<new>" } })"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(ValidateSpecTimestampWithNewSchemaSecretsDiff) {
    class TTest : public ITestTreesGeneratorCanon {
    public:
        TTest() : ITestTreesGeneratorCanon()
        {}

    protected:
        void Test() override {
            auto podAgentRequest = MakePodAgentRequest(CreateSpec(), 1);
            auto* secrets = podAgentRequest.add_secrets();
            secrets->set_id("secret_alias");

            {
                auto* value = secrets->add_values();
                value->set_key("old_key");
                value->set_value("old_value");
            }

            {
                auto* value = secrets->add_values();
                value->set_key("old_key_2");
                value->set_value("old_value_2");
            }

            {
                auto* value = secrets->add_values();
                value->set_key("old_key_3");
                value->set_value("old_value_3");
            }

            UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdatePodAgentRequest(podAgentRequest, logger.SpawnFrame()));

            secrets->mutable_values(0)->set_value("changed_value");
            secrets->mutable_values()->RemoveLast(); // Remove old_key_3

            secrets = podAgentRequest.add_secrets();
            secrets->set_id("secret_new_alias");

            {
                auto* value = secrets->add_values();
                value->set_key("new_key");
                value->set_value("new_value");
                value->set_encoding("base64");
            }

            UNIT_ASSERT_EXCEPTION_CONTAINS(
                Worker_->UpdatePodAgentRequest(podAgentRequest, logger.SpawnFrame())
                , yexception
                , "Request changed but spec timestamp wasn't incremented."
                " Current spec_timestamp: 1, request spec_timestamp: 1."
                " Diff: modified: secrets[0].values[0].value: \"<old>\" -> \"<new>\"\n"
                "deleted: secrets[0].values[2]: { key: \"old_key_3\" value: \"<old>\" }\n"
                "added: secrets[1]: { id: \"secret_new_alias\" values { key: \"new_key\" value: \"<new>\" encoding: \"base64\" } }"
            );
        }
    };

    TTest test;
    test.DoTest();
}


        Y_UNIT_TEST(IgnoreDifferentOrderOfSecrets) {
            class TTest : public ITestTreesGeneratorCanon {
            public:
                TTest() : ITestTreesGeneratorCanon()
                {}

            protected:
                void Test() override {
                    auto podAgentRequest = MakePodAgentRequest(CreateSpec(), 1);

                    {
                        auto* secret = podAgentRequest.add_secrets();
                        secret->set_id("secret_alias");
                        auto* value = secret->add_values();
                        value->set_key("key");
                        value->set_value("value");
                    }
                    {
                        auto* secret = podAgentRequest.add_secrets();
                        secret->set_id("secret_alias_2");
                        auto* value = secret->add_values();
                        value->set_key("key_2");
                        value->set_value("value_2");
                    }

                    UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdatePodAgentRequest(podAgentRequest, logger.SpawnFrame()));
                    auto podAgentRequest2 = MakePodAgentRequest(CreateSpec(), 1);

                    {
                        auto* secret = podAgentRequest2.add_secrets();
                        secret->set_id("secret_alias_2");
                        auto* value = secret->add_values();
                        value->set_key("key_2");
                        value->set_value("value_2");
                    }
                    {
                        auto* secret = podAgentRequest2.add_secrets();
                        secret->set_id("secret_alias");
                        auto* value = secret->add_values();
                        value->set_key("key");
                        value->set_value("value");
                    }

                    UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdatePodAgentRequest(podAgentRequest2, logger.SpawnFrame()));
                }
            };

            TTest test;
            test.DoTest();
        }



        Y_UNIT_TEST(IgnoreImmutableMetaAtSpecTimestampValidation) {
    class TTest : public ITestTreesGeneratorCanon {
    public:
        TTest() : ITestTreesGeneratorCanon()
        {}

    protected:
        void Test() override {
            auto podAgentRequest = MakePodAgentRequest(CreateSpec(), 1);

            UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdatePodAgentRequest(podAgentRequest, logger.SpawnFrame()));

            podAgentRequest.mutable_immutable_meta()->mutable_node_meta()->set_fqdn("fqdn");
            UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdatePodAgentRequest(podAgentRequest, logger.SpawnFrame()));
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(IgnoreCpuTotalCapacityAtSpecTimestampValidation) {
    class TTest : public ITestTreesGeneratorCanon {
    public:
        TTest() : ITestTreesGeneratorCanon()
        {}

    protected:
        void Test() override {
            auto podAgentRequest = MakePodAgentRequest(CreateSpec(), 1);

            UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdatePodAgentRequest(podAgentRequest, logger.SpawnFrame()));

            podAgentRequest.mutable_node_entry()->mutable_cpu()->set_total_capacity(94);
            UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdatePodAgentRequest(podAgentRequest, logger.SpawnFrame()));

            podAgentRequest.mutable_node_entry()->mutable_cpu()->set_cpu_to_vcpu_factor(0.5);
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                Worker_->UpdatePodAgentRequest(podAgentRequest, logger.SpawnFrame())
                , yexception
                , "Diff: added: node_entry.cpu.cpu_to_vcpu_factor: 0.5"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(IgnoreNodeEntryKeysResourceIdAtSpecTimestampValidation) {
    class TTest : public ITestTreesGeneratorCanon {
    public:
        TTest() : ITestTreesGeneratorCanon()
        {}

    protected:
        void Test() override {
            auto podAgentRequest = MakePodAgentRequest(CreateSpec(), 1);

            UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdatePodAgentRequest(podAgentRequest, logger.SpawnFrame()));

            podAgentRequest.mutable_node_entry()->set_keys_resource_id("other_keys_resource_id");
            UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdatePodAgentRequest(podAgentRequest, logger.SpawnFrame()));
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(WorkloadBadEnvironmentVariableName) {
    class TTest : public ITestTreesGeneratorCanon {
    public:
        TTest() : ITestTreesGeneratorCanon()
        {}

    protected:
        void Test() override {
            auto spec = CreateSpec();

            *spec.mutable_workloads(0)->add_env()->mutable_name() = "BadNameWith=Trash";
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                Worker_->UpdatePodAgentRequest(MakePodAgentRequest(spec, 1), logger.SpawnFrame())
                , yexception
                , "is not allowed at environment variable name"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(BoxBadVolumeReference) {
    class TTest : public ITestTreesGeneratorCanon {
    public:
        TTest() : ITestTreesGeneratorCanon()
        {}

    protected:
        void Test() override {
            auto spec = CreateSpec();

            *spec.mutable_boxes(0)->add_volumes()->mutable_volume_ref() = "bad_ref";
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                Worker_->UpdatePodAgentRequest(MakePodAgentRequest(spec, 1), logger.SpawnFrame())
                , yexception
                , "box with id my_box refers to nonexistent volume with id bad_ref"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(BoxUniqueVolumeMountPoints) {
    class TTest : public ITestTreesGeneratorCanon {
    public:
        TTest() : ITestTreesGeneratorCanon()
        {}

    protected:
        void Test() override {
            auto spec = CreateSpec();

            auto volume = spec.mutable_boxes(0)->add_volumes();
            *volume->mutable_volume_ref() = "my_volume";
            *volume->mutable_mount_point() = "folder";
            volume = spec.mutable_boxes(0)->add_volumes();
            *volume->mutable_volume_ref() = "my_volume_1";
            *volume->mutable_mount_point() = "folder";
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                Worker_->UpdatePodAgentRequest(MakePodAgentRequest(spec, 1), logger.SpawnFrame())
                , yexception
                , "Mount point '/rootfs_my_box/folder/' is a nested mount point of '/rootfs_my_box/folder/' at box 'my_box'"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(BoxUniqueRbindVolumeMountPoints) {
    class TTest : public ITestTreesGeneratorCanon {
    public:
        TTest() : ITestTreesGeneratorCanon()
        {}

    protected:
        void Test() override {
            auto spec = CreateSpec();

            auto volume = spec.mutable_boxes(0)->add_volumes();
            *volume->mutable_volume_ref() = "my_volume";
            *volume->mutable_mount_point() = "folder";
            volume = spec.mutable_boxes(0)->add_volumes();
            *volume->mutable_rbind_volume_ref() = "my_rbind_volume_ref";
            *volume->mutable_mount_point() = "folder";
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                Worker_->UpdatePodAgentRequest(MakePodAgentRequest(spec, 1), logger.SpawnFrame())
                , yexception
                , "Mount point '/rootfs_my_box/folder/' is a nested mount point of '/rootfs_my_box/folder/' at box 'my_box'"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(BoxUniqueStaticResourceMountPoints) {
    class TTest : public ITestTreesGeneratorCanon {
    public:
        TTest() : ITestTreesGeneratorCanon()
        {}

    protected:
        void Test() override {
            auto spec = CreateSpec();

            auto resource = spec.mutable_boxes(0)->add_static_resources();
            *resource->mutable_resource_ref() = "my_static_resource";
            *resource->mutable_mount_point() = "folder";
            resource = spec.mutable_boxes(0)->add_static_resources();
            *resource->mutable_resource_ref() = "my_static_resource_1";
            *resource->mutable_mount_point() = "folder";
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                Worker_->UpdatePodAgentRequest(MakePodAgentRequest(spec, 1), logger.SpawnFrame())
                , yexception
                , "Mount point '/rootfs_my_box/folder/' is a nested mount point of '/rootfs_my_box/folder/' at box 'my_box'"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(BoxUniqueStaticResourceAndVolumeMountPoints) {
    class TTest : public ITestTreesGeneratorCanon {
    public:
        TTest() : ITestTreesGeneratorCanon()
        {}

    protected:
        void Test() override {
            auto spec = CreateSpec();
            auto volume = spec.mutable_boxes(0)->add_volumes();
            *volume->mutable_volume_ref() = "my_volume";
            *volume->mutable_mount_point() = "/folder/dir";
            auto resource = spec.mutable_boxes(0)->add_static_resources();
            *resource->mutable_resource_ref() = "my_static_resource";
            *resource->mutable_mount_point() = "folder";
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                Worker_->UpdatePodAgentRequest(MakePodAgentRequest(spec, 1), logger.SpawnFrame())
                , yexception
                , "Mount point '/rootfs_my_box/folder/dir/' is a nested mount point of '/rootfs_my_box/folder/' at box 'my_box'"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(BoxUniqueLayerRefs) {
    class TTest : public ITestTreesGeneratorCanon {
    public:
        TTest() : ITestTreesGeneratorCanon()
        {}

    protected:
        void Test() override {
            auto spec = CreateSpec();

            auto rootfs = spec.mutable_boxes(0)->mutable_rootfs();
            *rootfs->add_layer_refs() = "my_layer";
            *rootfs->add_layer_refs() = "my_layer";
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                Worker_->UpdatePodAgentRequest(MakePodAgentRequest(spec, 1), logger.SpawnFrame())
                , yexception
                , "Two equal layers: 'my_layer' at box 'my_box'"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(BoxBadLayerReference) {
    class TTest : public ITestTreesGeneratorCanon {
    public:
        TTest() : ITestTreesGeneratorCanon()
        {}

    protected:
        void Test() override {
            auto spec = CreateSpec();

            *spec.mutable_boxes(0)->mutable_rootfs()->mutable_layer_refs(0) = "bad_ref";
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                Worker_->UpdatePodAgentRequest(MakePodAgentRequest(spec, 1), logger.SpawnFrame())
                , yexception
                , "box my_box rootfs volume refers to nonexistent layer with id bad_ref"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(VolumeBadLayerReference) {
    class TTest : public ITestTreesGeneratorCanon {
    public:
        TTest() : ITestTreesGeneratorCanon()
        {}

    protected:
        void Test() override {
            auto spec = CreateSpec();

            *spec.mutable_volumes(0)->mutable_generic()->mutable_layer_refs(0) = "bad_ref";
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                Worker_->UpdatePodAgentRequest(MakePodAgentRequest(spec, 1), logger.SpawnFrame())
                , yexception
                , "volume my_volume refers to nonexistent layer with id bad_ref"
            );
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(BadSpecTwoOrMoreTimes) {
    class TTest : public ITestTreesGeneratorCanon {
    public:
        TTest() : ITestTreesGeneratorCanon()
        {}

    protected:
        void Test() override {
            auto spec = CreateSpec();

            UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdatePodAgentRequest(MakePodAgentRequest(spec, 1), logger.SpawnFrame()));
            *spec.mutable_volumes(0)->mutable_generic()->mutable_layer_refs(0) = "bad_ref";  // make bad spec
            for (size_t i = 0; i < 10; ++i) {
                UNIT_ASSERT_EXCEPTION_CONTAINS(
                    Worker_->UpdatePodAgentRequest(MakePodAgentRequest(spec, 2), logger.SpawnFrame())
                    , yexception
                    , "volume my_volume refers to nonexistent layer with id bad_ref"
                );
            }
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(UpdateActiveDownloadContainersLimit) {
    class TTest : public ITestTreesGeneratorCanon {
    public:
        TTest() : ITestTreesGeneratorCanon()
        {}

    protected:
        void Test() override {
            const size_t cntIter = 10;
            auto podAgentRequest = MakePodAgentRequest(CreateSpec(), 1);

            for (size_t i = 0; i < cntIter; ++i) {
                podAgentRequest.set_spec_timestamp(i + 1);
                podAgentRequest.mutable_spec()->mutable_resources()->set_active_download_containers_limit(i);
                const ui32 targetValue = (i == 0) ? NSupport::DEFAULT_ACTIVE_DOWNLOAD_CONTAINERS_LIMIT : i;

                UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdatePodAgentRequest(podAgentRequest, logger.SpawnFrame()));
                UNIT_ASSERT_EQUAL_C(
                    StaticResourceStatusRepository_->GetActiveDownloadContainersLimit()
                    , targetValue
                    , StaticResourceStatusRepository_->GetActiveDownloadContainersLimit()
                );
                UNIT_ASSERT_EQUAL_C(
                    LayerStatusRepository_->GetActiveDownloadContainersLimit()
                    , targetValue
                    , LayerStatusRepository_->GetActiveDownloadContainersLimit()
                );
            }

            for (size_t i = cntIter; i < 2 * cntIter; ++i) {
                podAgentRequest.set_spec_timestamp(0); // make podAgentRequest bad
                podAgentRequest.mutable_spec()->mutable_resources()->set_active_download_containers_limit(i);

                UNIT_ASSERT_EXCEPTION(Worker_->UpdatePodAgentRequest(podAgentRequest, logger.SpawnFrame()), yexception);

                const ui32 targetValue = cntIter - 1;
                UNIT_ASSERT_EQUAL_C(
                    StaticResourceStatusRepository_->GetActiveDownloadContainersLimit()
                    , targetValue
                    , StaticResourceStatusRepository_->GetActiveDownloadContainersLimit()
                );
                UNIT_ASSERT_EQUAL_C(
                    LayerStatusRepository_->GetActiveDownloadContainersLimit()
                    , targetValue
                    , LayerStatusRepository_->GetActiveDownloadContainersLimit()
                );
            }
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(UpdateActiveVerifyContainersLimit) {
    class TTest : public ITestTreesGeneratorCanon {
    public:
        TTest() : ITestTreesGeneratorCanon()
        {}

    protected:
        void Test() override {
            const size_t cntIter = 10;
            auto podAgentRequest = MakePodAgentRequest(CreateSpec(), 1);

            for (size_t i = 0; i < cntIter; ++i) {
                podAgentRequest.set_spec_timestamp(i + 1);
                podAgentRequest.mutable_spec()->mutable_resources()->set_active_verify_containers_limit(i);
                const ui32 targetValue = (i == 0) ? NSupport::DEFAULT_ACTIVE_VERIFY_CONTAINERS_LIMIT : i;

                UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdatePodAgentRequest(podAgentRequest, logger.SpawnFrame()));
                UNIT_ASSERT_EQUAL_C(
                    StaticResourceStatusRepository_->GetActiveVerifyContainersLimit()
                    , targetValue
                    , StaticResourceStatusRepository_->GetActiveVerifyContainersLimit()
                );
                UNIT_ASSERT_EQUAL_C(
                    LayerStatusRepository_->GetActiveVerifyContainersLimit()
                    , targetValue
                    , LayerStatusRepository_->GetActiveVerifyContainersLimit()
                );
            }

            for (size_t i = cntIter; i < 2 * cntIter; ++i) {
                podAgentRequest.set_spec_timestamp(0); // make podAgentRequest bad
                podAgentRequest.mutable_spec()->mutable_resources()->set_active_verify_containers_limit(i);

                UNIT_ASSERT_EXCEPTION(Worker_->UpdatePodAgentRequest(podAgentRequest, logger.SpawnFrame()), yexception);

                const ui32 targetValue = cntIter - 1;
                UNIT_ASSERT_EQUAL_C(
                    StaticResourceStatusRepository_->GetActiveVerifyContainersLimit()
                    , targetValue
                    , StaticResourceStatusRepository_->GetActiveVerifyContainersLimit()
                );
                UNIT_ASSERT_EQUAL_C(
                    LayerStatusRepository_->GetActiveVerifyContainersLimit()
                    , targetValue
                    , LayerStatusRepository_->GetActiveVerifyContainersLimit()
                );
            }
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestPodAgentTargetState) {
    class TTest : public ITestTreesGeneratorCanon {
    public:
        TTest() : ITestTreesGeneratorCanon()
        {}

    protected:
        void Test() override {
            auto podAgentRequest = CreatePodAgentRequest();

            podAgentRequest.mutable_spec()->set_target_state(API::EPodAgentTargetState_INT_MIN_SENTINEL_DO_NOT_USE_);
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                Worker_->UpdatePodAgentRequest(podAgentRequest, logger.SpawnFrame())
                , yexception
                , "Unknown pod agent target state"
            );

            // Make totally bad request
            *podAgentRequest.mutable_spec()->mutable_resources()->mutable_layers(0)->mutable_checksum() = "bad_checksum";
            *podAgentRequest.mutable_spec()->mutable_resources()->mutable_static_resources(0)->mutable_verification()->mutable_checksum() = "bad_checksum";
            *podAgentRequest.mutable_resource_cache_spec()->mutable_layers(0)->mutable_layer()->mutable_checksum() = "bad_checksum";
            *podAgentRequest.mutable_resource_cache_spec()->mutable_static_resources(0)->mutable_resource()->mutable_verification()->mutable_checksum() = "bad_checksum";
            *podAgentRequest.mutable_spec()->mutable_volumes(0)->mutable_generic()->mutable_layer_refs(0) = "bad_ref";
            *podAgentRequest.mutable_spec()->mutable_boxes(0)->mutable_rootfs()->mutable_layer_refs(0) = "bad_ref";
            *podAgentRequest.mutable_spec()->mutable_workloads(0)->add_env()->mutable_name() = "BadNameWith=Trash";

            podAgentRequest.mutable_spec()->set_target_state(API::EPodAgentTargetState_ACTIVE);
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                Worker_->UpdatePodAgentRequest(podAgentRequest, logger.SpawnFrame())
                , yexception
                , "unable to parse checksum"
            );

            podAgentRequest.mutable_spec()->set_target_state(API::EPodAgentTargetState_SUSPENDED);
            UNIT_ASSERT_EXCEPTION_CONTAINS(
                Worker_->UpdatePodAgentRequest(podAgentRequest, logger.SpawnFrame())
                , yexception
                , "unable to parse checksum"
            );

            podAgentRequest.mutable_spec()->set_target_state(API::EPodAgentTargetState_REMOVED);
            UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdatePodAgentRequest(podAgentRequest, logger.SpawnFrame()));
        }
    };

    TTest test;
    test.DoTest();
}


Y_UNIT_TEST(TestPodDynamicAttributes) {
    class TTest : public ITestTreesGeneratorCanon {
    public:
        TTest() : ITestTreesGeneratorCanon()
        {}

    protected:
        void Test() override {
            auto podAgentRequest = CreatePodAgentRequest();

            auto* attr = podAgentRequest.mutable_pod_dynamic_attributes()->mutable_labels()->add_attributes();
            attr->set_key("use_env_secret");
            attr->set_value("true");

            UNIT_ASSERT_NO_EXCEPTION(Worker_->UpdatePodAgentRequest(podAgentRequest, logger.SpawnFrame()));
        }
    };

    TTest test;
    test.DoTest();
}

}

} // NInfra::NPodAgent::NTreeGeneratorTest
