#include "volume_tree_generator.h"

#include "layer_tree_generator.h"
#include "object_tree.h"
#include "support_functions.h"

#include <infra/pod_agent/libs/behaviour/loaders/behavior3_editor_json_reader.h>
#include <infra/pod_agent/libs/behaviour/loaders/behavior3_template_resolver.h>
#include <infra/pod_agent/libs/pod_agent/trees_generators/proto_hasher.h>
#include <infra/pod_agent/libs/util/string_utils.h>

namespace NInfra::NPodAgent {

TVolumeTreeGenerator::TVolumesToAdd TVolumeTreeGenerator::UpdateSpec(
    const google::protobuf::RepeatedPtrField<API::TVolume>& volumes
    , API::EPodAgentTargetState podAgentTargetState
    , const TLayerTreeGenerator::TLayersToAdd& layers
    , const TStaticResourceTreeGenerator::TStaticResourcesToAdd& staticResources
    , ui64 specTimestamp
    , ui32 revision
) {
    TVolumeTreeGenerator::TVolumesToAdd volumesToAdd;

    // Add volumes to internal state only in ACTIVE or SUSPENDED pod_agent target states
    // WARNING: We do not validate volume specs in other cases (for example in REMOVED target state)
    if (
        podAgentTargetState == API::EPodAgentTargetState_ACTIVE
        || podAgentTargetState == API::EPodAgentTargetState_SUSPENDED
    ) {
        for (const auto& volume : volumes) {
            const TString& volumeId = volume.id();

            Y_ENSURE(!volumeId.empty(), "One of volumes has empty id");
            Y_ENSURE(!volumesToAdd.Volumes_.contains(volumeId)
                , "Pod agent spec volumes' ids are not unique: "
                << "volume '" << volumeId << "' occurs twice"
            );

            try {
                volumesToAdd.Volumes_.insert(
                    {
                        volumeId
                        , GetVolumeToAdd(
                            volume
                            , layers
                            , staticResources
                            , specTimestamp
                            , revision
                        )
                    }
                );
            } catch (yexception& e) {
                throw e << " at volume '" << volumeId << "'";
            }
        }
    }

    return volumesToAdd;
}

TVolumeTreeGenerator::TVolumeToAdd TVolumeTreeGenerator::GetVolumeToAdd(
    const API::TVolume& volume
    , const TLayerTreeGenerator::TLayersToAdd& layers
    , const TStaticResourceTreeGenerator::TStaticResourcesToAdd& staticResources
    , ui64 specTimestamp
    , ui32 revision
) const {
    const TString volumeId = volume.id();
    const TString treeHash = GetVolumeHash(volume, layers, staticResources);

    TVector<TString> layerRefs;
    TVector<TString> staticResourceRefs;
    TBehavior3 resolvedTree = VolumeTreeTemplate_;
    {
        TMap<TString, TString> replace;
        replace["VOLUME_ID"] = volumeId;
        replace["VOLUME_PATH"] = PathHolder_->GetVolumePath(volumeId);
        replace["VOLUME_STORAGE"] = PathHolder_->GetVolumePersistentStorage(volumeId);
        replace["IS_VOLUME_PERSISTENT"] = volume.persistence_type() == API::EVolumePersistenceType_PERSISTENT
            ? "true"
            : "";

        Y_ENSURE(PathHolder_->HasVirtualDisk(volume.virtual_disk_id_ref()), "Unknown virtual disk ref: '" << volume.virtual_disk_id_ref() << "'");
        const TString place = PathHolder_->GetPlaceFromVirtualDisk(volume.virtual_disk_id_ref());
        replace["VOLUME_PLACE"] = place;

        TVector<TString> layerNames;
        TVector<TString> layerDownloadHashes;
        for (const TString& layerId : volume.generic().layer_refs()) {
            Y_ENSURE(layers.Layers_.contains(layerId), "Unknown layer: '" << layerId << "'");
            Y_ENSURE(
                Find(layerRefs, layerId) == layerRefs.end()
                , "Two equal layers: '" << layerId << "'"
            );
            Y_ENSURE(layers.Layers_.at(layerId).VirtualDiskIdRef_ == volume.virtual_disk_id_ref()
                , "Volume '" << volumeId
                << "' with virtual disk ref '" << volume.virtual_disk_id_ref()
                << "' refers to layer '" << layerId
                << "' with another virtual disk ref '" << layers.Layers_.at(layerId).VirtualDiskIdRef_ << "'"
            );

            TString layerName = PathHolder_->GetLayerNameFromHash(layers.Layers_.at(layerId).Target_.Meta_.DownloadHash_);
            layerNames.push_back(layerName);
            layerRefs.push_back(layerId);
            layerDownloadHashes.push_back(layers.Layers_.at(layerId).Target_.Meta_.DownloadHash_);
        }

        replace["LAYER_NAME_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator(layerNames);
        replace["LAYER_DOWNLOAD_HASH_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator(layerDownloadHashes);

        TVector<TString> allMountPoints;
        TVector<TString> staticResourceOriginalPaths;
        TVector<TString> staticResourceMountPoints;
        for(const auto& staticResource: volume.static_resources()) {
            const TString& staticResourceId = staticResource.resource_ref();
            const auto& resource = staticResources.StaticResources_.at(staticResourceId);
            Y_ENSURE(staticResources.StaticResources_.contains(staticResourceId), "Unknown static resource: " << Quote(staticResourceId));
            Y_ENSURE(resource.VirtualDiskIdRef_ == volume.virtual_disk_id_ref()
                , "Volume " << Quote(volumeId)
                << " with virtual disk ref " << Quote(volume.virtual_disk_id_ref())
                << " refers to static resource " << Quote(staticResourceId)
                << " with another virtual disk ref " << Quote(resource.VirtualDiskIdRef_)
            );

            staticResourceRefs.push_back(staticResourceId);
            staticResourceOriginalPaths.push_back(PathHolder_->GetFinalStaticResourcePathFromHash(resource.Target_.Meta_.DownloadHash_, place));

            TString staticResourceMountPoint = NSupport::NormalizeMountPoint(staticResource.volume_relative_mount_point());
            allMountPoints.push_back(PathHolder_->GetVolumePath(volumeId) + "/" + staticResourceMountPoint);
            staticResourceMountPoints.push_back(staticResourceMountPoint);
        }

        NSupport::ValidateMountPoints(allMountPoints);

        // TODO dkochetov add check for static resource is checkable
        replace["STATIC_RESOURCE_PATHS"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator(staticResourceMountPoints);
        replace["STATIC_RESOURCE_ORIGINAL_PATHS"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator(staticResourceOriginalPaths);

        replace["TREE_HASH"] = treeHash;

        TBehavior3TemplateResolver().Resolve(resolvedTree, replace);
    }

    return TVolumeToAdd{
        TUpdateHolder::TVolumeTarget(
            TVolumeMeta(
                volumeId
                , specTimestamp
                , revision
                , layerRefs
                , staticResourceRefs
            )
            , new TObjectTree(
                Logger_
                , TStatusNTickerHolder::GetVolumeTreeId(volumeId)
                , TBehavior3EditorJsonReader(resolvedTree)
                    .WithPorto(Porto_)
                    .WithPosixWorker(PosixWorker_)
                    .WithLayerStatusRepository(StatusNTickerHolder_->GetLayerStatusRepository())
                    .WithVolumeStatusRepository(StatusNTickerHolder_->GetVolumeStatusRepository())
                    .WithTemplateBTStorage(TemplateBTStorage_)
                    .BuildRootNode()
                , volumeId
                , StatusNTickerHolder_->GetUpdateHolder()->GetUpdateHolderTarget()
                , StatusNTickerHolder_->GetVolumeStatusRepository()
            )
            , treeHash
            , volume.persistence_type() == API::EVolumePersistenceType_PERSISTENT
        )
    };
}

void TVolumeTreeGenerator::RemoveVolumes(const TVolumesToAdd& volumes) {
    for (const TString& volumeId : StatusNTickerHolder_->GetVolumeIds()) {
        if (!volumes.Volumes_.contains(volumeId)) {
            StatusNTickerHolder_->SetVolumeTargetRemove(volumeId);
        }
    }
}

void TVolumeTreeGenerator::AddVolumes(const TVolumesToAdd& volumes) {
    for (const auto& it : volumes.Volumes_) {
        const TString& volumeId = it.first;
        const auto& volumeData = it.second;

        if (!StatusNTickerHolder_->HasVolume(volumeId)) {
            StatusNTickerHolder_->AddVolumeWithTargetCheck(volumeData.Target_);
        }

        StatusNTickerHolder_->UpdateVolumeTarget(volumeData.Target_);
    }
}

} // namespace NInfra::NPodAgent
