#include "volume_tree_generator.h"

#include "test_canon.h"
#include "test_functions.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/update_holder/test_lib/test_functions.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/system/thread.h>

namespace NInfra::NPodAgent::NTreeGeneratorTest {

class ITestVolumeTreeGeneratorCanon: public ITestTreeGeneratorCanon {
public:
    ITestVolumeTreeGeneratorCanon() : ITestTreeGeneratorCanon()
    {}

protected:
    void CreateWorker() final {
        TMap<TString, TString> properties;
        properties["object_id_or_hash"] = "<RSLV:VOLUME_ID>";
        properties["state"] = "EVolumeState_READY";
        properties["object_type"] = "volume";

        Worker_.Reset(new TVolumeTreeGenerator(
            logger
            , PathHolder_
            , GetTestTemplateTree(
                properties
                , "FeedbackObjectState"
                , "FeedbackObjectState(EVolumeState_READY)"
            )
            , new TAsyncPortoClient(new TMockPortoClient(), new TFakeThreadPool())
            , new TPosixWorker(new TFakeThreadPool())
            , StatusNTickerHolder_
            , nullptr
        ));
    }

    TVolumeTreeGenerator::TVolumesToAdd UpdateSpec(
        const ::google::protobuf::RepeatedPtrField<API::TVolume>& volumes
        , API::EPodAgentTargetState podAgentTargetState
        , const TLayerTreeGenerator::TLayersToAdd& layers
        , const TStaticResourceTreeGenerator::TStaticResourcesToAdd& staticResources
        , ui64 specTimestamp
        , ui32 revision
    ) {
        TVolumeTreeGenerator::TVolumesToAdd newSpec = Worker_->UpdateSpec(
            volumes
            , podAgentTargetState
            , layers
            , staticResources
            , specTimestamp
            , revision
        );
        Worker_->RemoveVolumes(newSpec);
        Worker_->AddVolumes(newSpec);
        return newSpec;
    }

protected:
    THolder<TVolumeTreeGenerator> Worker_;
};

Y_UNIT_TEST_SUITE(VolumeWorkerSuite) {

Y_UNIT_TEST(TestSimpleTree) {
    class TTest : public ITestVolumeTreeGeneratorCanon {
    public:
        TTest() : ITestVolumeTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TPodAgentSpec spec;
            API::TVolume* volume = spec.add_volumes();
            volume->set_id("my_volume");
            volume->set_virtual_disk_id_ref("virtual_disk");

            API::TGenericVolume* generic = volume->mutable_generic();
            generic->add_layer_refs("my_layer");

            API::TResourceGang* gang = spec.mutable_resources();
            API::TLayer* layer = gang->add_layers();
            layer->set_id("my_layer");

            API::TResource* staticResource = gang->add_static_resources();
            staticResource->set_id("my_static_resource");

            API::TVolumeMountedStaticResource* mountedStaticResource = volume->add_static_resources();
            mountedStaticResource->set_resource_ref("my_static_resource");
            mountedStaticResource->set_volume_relative_mount_point("/mount_point");

            auto volumeCopy = spec.add_volumes();
            volumeCopy->CopyFrom(*volume);

            volumeCopy->set_persistence_type(API::EVolumePersistenceType::EVolumePersistenceType_NON_PERSISTENT);

            LayerStatusRepository_->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple("my_layer", "my_layer_download_hash"));
            StaticResourceStatusRepository_->AddObject(NObjectMetaTestLib::CreateStaticResourceMetaSimple("my_static_resource", "my_static_resource_download_hash"));
            TLayerTreeGenerator::TLayersToAdd layers;
            layers.Layers_.insert(
                {
                    "my_layer"
                    , {
                        NObjectTargetTestLib::CreateLayerTargetSimple("my_layer", "my_layer_download_hash")
                        , "layer_tree_hash"
                        , "virtual_disk"
                    }
                }
            );

            TStaticResourceTreeGenerator::TStaticResourcesToAdd staticResources;
            staticResources.StaticResources_.insert(
                {
                    "my_static_resource"
                    , {
                          NObjectTargetTestLib::CreateStaticResourceTargetSimple("my_static_resource", "my_static_resource_download_hash")
                          , "static_resource_tree_hash"
                          , "virtual_disk"
                          , {}
                    }
                }
            );

            UNIT_ASSERT_EXCEPTION(UpdateSpec(spec.volumes(), API::EPodAgentTargetState_ACTIVE, layers, staticResources, 0, 0), yexception);

            *volumeCopy->mutable_id() += "_other";
            auto volumesToAdd = UpdateSpec(spec.volumes(), API::EPodAgentTargetState_ACTIVE, layers, staticResources, 0, 0);

            *volumeCopy->mutable_id() = "";
            UNIT_ASSERT_EXCEPTION(UpdateSpec(spec.volumes(), API::EPodAgentTargetState_ACTIVE, layers, staticResources, 0, 0), yexception);

            *volumeCopy->mutable_id() = *volume->mutable_id() + "_other";
            volumesToAdd = UpdateSpec(spec.volumes(), API::EPodAgentTargetState_ACTIVE, layers, staticResources, 0, 0);

            {
                layers.Layers_.at("my_layer").VirtualDiskIdRef_ = "";
                UNIT_ASSERT_EXCEPTION_CONTAINS(UpdateSpec(spec.volumes(), API::EPodAgentTargetState_ACTIVE, layers, staticResources, 0, 0), yexception, "another virtual disk ref");

                layers.Layers_.at("my_layer").VirtualDiskIdRef_ = "bad_virtual_disk";
                volume->set_virtual_disk_id_ref("bad_virtual_disk");
                UNIT_ASSERT_EXCEPTION_CONTAINS(UpdateSpec(spec.volumes(), API::EPodAgentTargetState_ACTIVE, layers, staticResources, 0, 0), yexception, "Unknown virtual disk ref");

                volume->set_virtual_disk_id_ref("virtual_disk");
                layers.Layers_.at("my_layer").VirtualDiskIdRef_ = "virtual_disk";
            }

            {
                staticResources.StaticResources_.at("my_static_resource").VirtualDiskIdRef_ = "";
                UNIT_ASSERT_EXCEPTION_CONTAINS(UpdateSpec(spec.volumes(), API::EPodAgentTargetState_ACTIVE, layers, staticResources, 0, 0), yexception, "another virtual disk ref");

                staticResources.StaticResources_.at("my_static_resource").VirtualDiskIdRef_ = "bad_virtual_disk";
                volume->set_virtual_disk_id_ref("bad_virtual_disk");
                UNIT_ASSERT_EXCEPTION_CONTAINS(UpdateSpec(spec.volumes(), API::EPodAgentTargetState_ACTIVE, layers, staticResources, 0, 0), yexception, "Unknown virtual disk ref");

                volume->set_virtual_disk_id_ref("virtual_disk");
                staticResources.StaticResources_.at("my_static_resource").VirtualDiskIdRef_ = "virtual_disk";
            }

            Sleep(TDuration::MilliSeconds(500));
            UNIT_ASSERT_EQUAL(API::EVolumeState_READY, VolumeStatusRepository_->GetObjectStatus("my_volume").state());
            UNIT_ASSERT(volumesToAdd.Volumes_.at("my_volume").Target_.Tree_->GetUseLongTickPeriod());
            UNIT_ASSERT(volumesToAdd.Volumes_.at("my_volume").Target_.IsPersistent_);
            UNIT_ASSERT(!volumesToAdd.Volumes_.at("my_volume_other").Target_.IsPersistent_);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveTree) {
    class TTest : public ITestVolumeTreeGeneratorCanon {
    public:
        TTest() : ITestVolumeTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TPodAgentSpec spec;
            API::TVolume* volume = spec.add_volumes();
            volume->set_id("my_volume");
            API::TGenericVolume* generic = volume->mutable_generic();
            generic->add_layer_refs("my_layer");
            API::TResourceGang* gang = spec.mutable_resources();
            API::TLayer* layer = gang->add_layers();
            layer->set_id("my_layer");
            API::TResource* staticResource = gang->add_static_resources();
            staticResource->set_id("my_static_resource");
            API::TVolumeMountedStaticResource* mountedStaticResource = volume->add_static_resources();
            mountedStaticResource->set_resource_ref("my_static_resource");
            mountedStaticResource->set_volume_relative_mount_point("/mount_point");

            LayerStatusRepository_->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple("my_layer", "my_layer_download_hash"));
            TLayerTreeGenerator::TLayersToAdd layers;
            layers.Layers_.insert(
                {
                    "my_layer"
                    , {
                        NObjectTargetTestLib::CreateLayerTargetSimple("my_layer", "my_layer_download_hash")
                        , "layer_tree_hash"
                        , "" /* VirtualDiskIdRef_ */
                    }
                }
            );

            StaticResourceStatusRepository_->AddObject(NObjectMetaTestLib::CreateStaticResourceMetaSimple("my_static_resource", "my_static_resource_download_hash"));
            TStaticResourceTreeGenerator::TStaticResourcesToAdd staticResources;
            staticResources.StaticResources_.insert(
                {
                    "my_static_resource"
                    , {
                       NObjectTargetTestLib::CreateStaticResourceTargetSimple("my_static_resource", "my_static_resource_download_hash")
                       , "static_resource_tree_hash"
                       , "" /* VirtualDiskIdRef_ */
                       , {}
                    }
                }
            );

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.volumes(), API::EPodAgentTargetState_ACTIVE, layers, staticResources, 0, 0));
            UNIT_ASSERT(VolumeStatusRepository_->HasObject("my_volume"));
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->VolumeHasTarget("my_volume"));

            volume->set_id("other_volume");

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.volumes(), API::EPodAgentTargetState_ACTIVE, layers, staticResources, 0, 0));
            UNIT_ASSERT(VolumeStatusRepository_->HasObject("my_volume"));
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->VolumeHasTarget("my_volume"));
            UNIT_ASSERT(UpdateHolder_->GetAndRemoveVolumeTarget("my_volume").TargetRemove_);
            UNIT_ASSERT(VolumeStatusRepository_->HasObject("other_volume"));
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestChangeTarget) {
    class TTest : public ITestVolumeTreeGeneratorCanon {
    public:
        TTest() : ITestVolumeTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TPodAgentSpec spec;
            API::TVolume* volume = spec.add_volumes();
            volume->set_id("my_volume");
            API::TGenericVolume* generic = volume->mutable_generic();
            generic->add_layer_refs("my_layer");
            API::TResourceGang* gang = spec.mutable_resources();
            API::TLayer* layer = gang->add_layers();
            layer->set_id("my_layer");
            API::TResource* staticResource = gang->add_static_resources();
            staticResource->set_id("my_static_resource");
            API::TVolumeMountedStaticResource* mountedStaticResource = volume->add_static_resources();
            mountedStaticResource->set_resource_ref("my_static_resource");
            mountedStaticResource->set_volume_relative_mount_point("/mount_point");

            LayerStatusRepository_->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple("my_layer", "my_layer_download_hash"));
            TLayerTreeGenerator::TLayersToAdd layers;
            layers.Layers_.insert(
                {
                    "my_layer"
                    , {
                        NObjectTargetTestLib::CreateLayerTargetSimple("my_layer", "my_layer_download_hash")
                        , "layer_tree_hash"
                        , "" /* VirtualDiskIdRef_ */
                    }
                }
            );

            StaticResourceStatusRepository_->AddObject(NObjectMetaTestLib::CreateStaticResourceMetaSimple("my_static_resource", "my_static_resource_download_hash"));
            TStaticResourceTreeGenerator::TStaticResourcesToAdd staticResources;
            staticResources.StaticResources_.insert(
                {
                     "my_static_resource"
                    , {
                        NObjectTargetTestLib::CreateStaticResourceTargetSimple("my_static_resource", "my_static_resource_download_hash")
                        , "static_resource_tree_hash"
                        , "" /* VirtualDiskIdRef_ */
                        , {}
                    }
                }
            );

            for (size_t i = 0; i < 3; ++i) {
                UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.volumes(), API::EPodAgentTargetState_ACTIVE, layers, staticResources, i + 1, i + 2));
                UNIT_ASSERT(VolumeStatusRepository_->HasObject("my_volume"));
                UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->VolumeHasTarget("my_volume"));
                UNIT_ASSERT(VolumeStatusRepository_->GetObjectStatus("my_volume").spec_timestamp() == i + 1);
                UNIT_ASSERT(VolumeStatusRepository_->GetObjectStatus("my_volume").revision() == i + 2);
            }

            // change volume
            spec.mutable_volumes(0)->mutable_generic()->set_quota_bytes(1);

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.volumes(), API::EPodAgentTargetState_ACTIVE, layers, staticResources, 4, 5));

            UNIT_ASSERT(VolumeStatusRepository_->HasObject("my_volume"));
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->VolumeHasTarget("my_volume"));
            UNIT_ASSERT(VolumeStatusRepository_->GetObjectStatus("my_volume").spec_timestamp() == 3);
            UNIT_ASSERT(VolumeStatusRepository_->GetObjectStatus("my_volume").revision() == 4);
        }

    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestAddWithTargetCheck) {
    class TTest : public ITestVolumeTreeGeneratorCanon {
    public:
        TTest() : ITestVolumeTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TPodAgentSpec spec;
            API::TVolume* volume = spec.add_volumes();
            volume->set_id("my_volume");
            API::TGenericVolume* generic = volume->mutable_generic();
            generic->add_layer_refs("my_layer");
            API::TResourceGang* gang = spec.mutable_resources();
            API::TLayer* layer = gang->add_layers();
            layer->set_id("my_layer");
            API::TResource* staticResource = gang->add_static_resources();
            staticResource->set_id("my_static_resource");
            API::TVolumeMountedStaticResource* mountedStaticResource = volume->add_static_resources();
            mountedStaticResource->set_resource_ref("my_static_resource");
            mountedStaticResource->set_volume_relative_mount_point("/mount_point");

            Holder_->SetRevision(1);
            StatusNTickerHolder_->AddLayer(NObjectTargetTestLib::CreateLayerTargetSimple("my_layer", "my_layer_download_hash", 1));
            StatusNTickerHolder_->UpdateLayerTarget(NObjectTargetTestLib::CreateLayerTargetSimple("my_layer", "my_layer_download_hash_other", 3));

            TLayerTreeGenerator::TLayersToAdd layers;
            layers.Layers_.insert(
                {
                    "my_layer"
                    , {
                        NObjectTargetTestLib::CreateLayerTargetSimple("my_layer", "my_layer_download_hash_other", 3)
                        , "layer_tree_hash"
                        , "" /* VirtualDiskIdRef_ */
                    }
                }
            );

            StaticResourceStatusRepository_->AddObject(NObjectMetaTestLib::CreateStaticResourceMetaSimple("my_static_resource", "my_static_resource_download_hash"));
            TStaticResourceTreeGenerator::TStaticResourcesToAdd staticResources;
            staticResources.StaticResources_.insert(
                {
                    "my_static_resource"
                    , {
                        NObjectTargetTestLib::CreateStaticResourceTargetSimple("my_static_resource", "my_static_resource_download_hash")
                        , "static_resource_tree_hash"
                        , "" /* VirtualDiskIdRef_ */
                        , {}
                    }
                }
            );

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.volumes(), API::EPodAgentTargetState_ACTIVE, layers, staticResources, 3, 4));
            UNIT_ASSERT(VolumeStatusRepository_->HasObject("my_volume"));
            UNIT_ASSERT(VolumeStatusRepository_->GetObjectStatus("my_volume").spec_timestamp() == 0);
            UNIT_ASSERT(VolumeStatusRepository_->GetObjectStatus("my_volume").revision() == 0);
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->VolumeHasTarget("my_volume"));
            TUpdateHolder::TVolumeTarget volumeTarget = UpdateHolder_->GetAndRemoveVolumeTarget("my_volume");
            UNIT_ASSERT(volumeTarget.Meta_.SpecTimestamp_ == 3);
            UNIT_ASSERT(volumeTarget.Meta_.Revision_ == 4);
        }
    };

    TTest test;
    test.DoTest();
}

Y_UNIT_TEST(TestPodAgentTargetStateWithVolumeTree) {
    class TTest : public ITestVolumeTreeGeneratorCanon {
    public:
        TTest() : ITestVolumeTreeGeneratorCanon()
        {}

    protected:
        void Test() override {
            API::TPodAgentSpec spec;
            API::TVolume* volume = spec.add_volumes();
            volume->set_id("my_volume");
            API::TGenericVolume* generic = volume->mutable_generic();
            generic->add_layer_refs("my_layer");
            API::TResourceGang* gang = spec.mutable_resources();
            API::TLayer* layer = gang->add_layers();
            layer->set_id("my_layer");
            API::TResource* staticResource = gang->add_static_resources();
            staticResource->set_id("my_static_resource");
            API::TVolumeMountedStaticResource* mountedStaticResource = volume->add_static_resources();
            mountedStaticResource->set_resource_ref("my_static_resource");
            mountedStaticResource->set_volume_relative_mount_point("/mount_point");

            LayerStatusRepository_->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple("my_layer", "my_layer_download_hash"));

            TLayerTreeGenerator::TLayersToAdd layers;
            layers.Layers_.insert(
                {
                    "my_layer"
                    , {
                        NObjectTargetTestLib::CreateLayerTargetSimple("my_layer", "my_layer_download_hash")
                        , "layer_tree_hash"
                        , "" /* VirtualDiskIdRef_ */
                    }
                }
            );

            StaticResourceStatusRepository_->AddObject(NObjectMetaTestLib::CreateStaticResourceMetaSimple("my_static_resource", "my_static_resource_download_hash"));
            TStaticResourceTreeGenerator::TStaticResourcesToAdd staticResources;
            staticResources.StaticResources_.insert(
                {
                    "my_static_resource"
                    , {
                        NObjectTargetTestLib::CreateStaticResourceTargetSimple("my_static_resource", "my_static_resource_download_hash")
                        , "static_resource_tree_hash"
                        , "" /* VirtualDiskIdRef_ */
                        , {}
                    }
                }
            );

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.volumes(), API::EPodAgentTargetState_ACTIVE, layers, staticResources, 0, 0));
            UNIT_ASSERT(VolumeStatusRepository_->HasObject("my_volume"));
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->VolumeHasTarget("my_volume"));

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.volumes(), API::EPodAgentTargetState_SUSPENDED, layers, staticResources, 1, 1));
            UNIT_ASSERT(VolumeStatusRepository_->HasObject("my_volume"));
            UNIT_ASSERT(!UpdateHolder_->GetUpdateHolderTarget()->VolumeHasTarget("my_volume"));

            UNIT_ASSERT_NO_EXCEPTION(UpdateSpec(spec.volumes(), API::EPodAgentTargetState_REMOVED, layers, staticResources, 2, 2));
            UNIT_ASSERT(VolumeStatusRepository_->HasObject("my_volume"));
            UNIT_ASSERT(UpdateHolder_->GetUpdateHolderTarget()->VolumeHasTarget("my_volume"));
            UNIT_ASSERT(UpdateHolder_->GetAndRemoveVolumeTarget("my_volume").TargetRemove_);
        }
    };

    TTest test;
    test.DoTest();
}

}

} // namespace NInfra::NPodAgent::NTreeGeneratorTest
