#include "workload_tree_generator.h"

#include "object_tree.h"
#include "support_functions.h"

#include <infra/pod_agent/libs/behaviour/loaders/behavior3_editor_json_reader.h>
#include <infra/pod_agent/libs/behaviour/loaders/behavior3_template_resolver.h>
#include <infra/pod_agent/libs/pod_agent/trees_generators/proto_hasher.h>
#include <infra/pod_agent/libs/util/string_utils.h>

namespace NInfra::NPodAgent {

TWorkloadMeta::THookInfo TWorkloadTreeGenerator::GetWorkloadHookInfo(
    const TString& containerName
    , bool isContainer
    , bool isHttp
    , bool isTcp
    , bool isUnixSignal
    , bool isStartContainerAliveCheck
) const {
    Y_ENSURE(
        (ui32)isContainer + (ui32)isHttp + (ui32)isTcp + (ui32)isUnixSignal + (ui32)isStartContainerAliveCheck <= 1
        , "at least two of four hooks are set to true. "
            << "isContainer is set to " << Quote(isContainer)
            << ", isHttp is set to " << Quote(isHttp)
            << ", isTcp is set to " << Quote(isTcp)
            << ", isUnixSignal is set to " << Quote(isUnixSignal)
            << ", isStartContainerAliveCheck is set to " << Quote(isStartContainerAliveCheck)
    );

    if (isContainer) {
        return TWorkloadMeta::TContainerInfo(containerName);
    } else if (isHttp) {
        return TWorkloadMeta::THttpGetInfo();
    } else if (isTcp) {
        return TWorkloadMeta::TTcpCheckInfo();
    } else if (isUnixSignal) {
        return TWorkloadMeta::TUnixSignalInfo();
    } else if (isStartContainerAliveCheck) {
        return TWorkloadMeta::TStartContainerAliveCheckInfo();
    } else {
        return TWorkloadMeta::TEmptyInfo();
    }
}

TMap<TString, TString> TWorkloadTreeGenerator::FillHookTypeReplaceMap(
    const TString& name
    , bool isContainer
    , bool isHttp
    , bool isTcp
    , bool isUnixSignal
    , bool isStartContainerAliveCheck
) const {
    TMap<TString, TString> result;

    Y_ENSURE(
        (ui32)isContainer + (ui32)isHttp + (ui32)isTcp + (ui32)isUnixSignal + (ui32)isStartContainerAliveCheck <= 1
        , "at least two of four hooks are set to true. "
            << "isContainer is set to " << Quote(isContainer)
            << ", isHttp is set to " << Quote(isHttp)
            << ", isTcp is set to " << Quote(isTcp)
            << ", isUnixSignal is set to " << Quote(isUnixSignal)
            << ", isStartContainerAliveCheck is set to " << Quote(isStartContainerAliveCheck)
    );

    const TString prefix = TStringBuilder() << to_upper(name) << "_HOOK_TYPE";

    if (isContainer) {
        result[prefix] = ToString(NStatusRepositoryTypes::EHookBackend::CONTAINER);
    } else if (isHttp) {
        result[prefix] = ToString(NStatusRepositoryTypes::EHookBackend::HTTP);
    } else if (isTcp) {
        result[prefix] = ToString(NStatusRepositoryTypes::EHookBackend::TCP);
    } else if (isUnixSignal) {
        result[prefix] = ToString(NStatusRepositoryTypes::EHookBackend::UNIX_SIGNAL);
    } else {
        result[prefix] = ToString(NStatusRepositoryTypes::EHookBackend::NO_HOOK);
    }

    return result;
}

TMap<TString, TString> TWorkloadTreeGenerator::FillHookTriesThresholdsReplaceMap(
    const TString& name
    , ui32 failureThreshold
    , ui32 successThreshold
) const {
    TMap<TString, TString> result;

    ui32 fixedFailureThreshold = failureThreshold ? failureThreshold : DEFAULT_HOOK_TRIES_THRESHOLD;
    ui32 fixedSuccessThreshold = successThreshold ? successThreshold : DEFAULT_HOOK_TRIES_THRESHOLD;

    const TString prefix = TStringBuilder() << to_upper(name) << '_';
    result[prefix + "FAILURE_THRESHOLD"] = ToString(fixedFailureThreshold);
    result[prefix + "SUCCESS_THRESHOLD"] = ToString(fixedSuccessThreshold);

    return result;
}

TMap<TString, TString> TWorkloadTreeGenerator::FillHttpGetReplaceMap(
    const API::THttpGet& httpGet
    , const TString& name
    , bool needValidation
    , bool needDefaultInitialDelay
) const {
    TMap<TString, TString> result;
    const TString prefix = TStringBuilder() << "HTTP_" << to_upper(name) << '_';
    if (needValidation) {
        Y_ENSURE(httpGet.port() > 0, name << " port is zero");
    }
    result[prefix + "PATH"] = (httpGet.path().StartsWith("/") ? httpGet.path() : "/" + httpGet.path());
    result[prefix + "PORT"] = (httpGet.port() ? ToString(httpGet.port()) : "");

    result[prefix + "RESPONSE"] = httpGet.expected_answer();
    result[prefix + "ANY_RESPONSE"] = ToString(httpGet.any());

    result.merge(NSupport::FillTimeLimitReplaceMap(httpGet.time_limit(), prefix, needDefaultInitialDelay));
    return result;
}

TMap<TString, TString> TWorkloadTreeGenerator::FillTcpCheckReplaceMap(
    const API::TTcpCheck& tcpCheck
    , const TString& name
    , bool needValidation
    , bool needDefaultInitialDelay
) const {
    TMap<TString, TString> result;
    const TString prefix = TStringBuilder() << "TCP_" << to_upper(name) << '_';
    if (needValidation) {
        Y_ENSURE(tcpCheck.port() > 0, name << " port is zero");
    }
    result[prefix + "PORT"] = (tcpCheck.port() ? ToString(tcpCheck.port()) : "");
    result.merge(NSupport::FillTimeLimitReplaceMap(tcpCheck.time_limit(), prefix, needDefaultInitialDelay));
    return result;
}

TMap<TString, TString> TWorkloadTreeGenerator::FillUnixSignalReplaceMap(
    const API::TUnixSignal& unixSignal
    , const TString& name
) const {
    TMap<TString, TString> result;
    const TString prefix = TStringBuilder() << "UNIX_SIGNAL_" << to_upper(name) << '_';

    result[prefix + "SIGNAL"] = NSupport::UnixSignalTypeToString(unixSignal.signal());

    result.merge(NSupport::FillTimeLimitReplaceMap(unixSignal.time_limit(), prefix, true));
    return result;
}

TWorkloadTreeGenerator::TWorkloadHookData TWorkloadTreeGenerator::GetReadinessCheckData(
    const API::TReadinessCheck& readinessCheck
    , const TString& workloadId
    , const TString& boxId
    , const NSecret::TSecretMap& secretMap
    , const google::protobuf::RepeatedPtrField<API::TEnvVar>& env
    , const double cpuToVcpuFactor
    , bool useEnvSecret
) const {
    TMap<TString, TString> treeReplaceMap;

    ui32 failureThreshold = readinessCheck.failure_threshold();
    ui32 successThreshold = readinessCheck.success_threshold();
    if (readinessCheck.has_start_container_alive_check()) {
        Y_ENSURE((failureThreshold == 0 || failureThreshold == 1)
            && (successThreshold == 0 || successThreshold == 1)
            , "Success and failure thresholds must not be greater than 1 for readiness check of start container aliveness");
    }
    treeReplaceMap.merge(
        FillHookTypeReplaceMap(
            "readiness"
            , readinessCheck.has_container()
            , readinessCheck.has_http_get()
            , readinessCheck.has_tcp_check()
            , false /* isUnixSignal */
            , readinessCheck.has_start_container_alive_check()
        )
    );

    treeReplaceMap.merge(
        FillHookTriesThresholdsReplaceMap(
            "readiness"
            , failureThreshold
            , successThreshold
        )
    );

    treeReplaceMap.merge(
        NSupport::FillReadinessOrLivenessContainerReplaceMap(
            readinessCheck.container()
            , "readiness"
            , PathHolder_->GetWorkloadContainerWithName(boxId, workloadId, "readiness")
            , PathHolder_->GetBoxRootfsPath(boxId)
            , PodAgentBinaryFilePathInBox_
            , env
            , secretMap
            , PathHolder_
            , cpuToVcpuFactor
            , true /* needTimeLimit */
            , readinessCheck.has_container() /* needValidation */
            , useEnvSecret
        )
    );

    treeReplaceMap.merge(
        FillHttpGetReplaceMap(
            readinessCheck.http_get()
            , "readiness"
            , readinessCheck.has_http_get()
            , true /* needDefaultInitialDelay */
        )
    );

    treeReplaceMap.merge(
        FillTcpCheckReplaceMap(
            readinessCheck.tcp_check()
            , "readiness"
            , readinessCheck.has_tcp_check()
            , true /* needDefaultInitialDelay */
        )
    );

    return TWorkloadHookData{
        GetWorkloadHookInfo(
            PathHolder_->GetWorkloadContainerWithName(boxId, workloadId, "readiness")
            , readinessCheck.has_container()
            , readinessCheck.has_http_get()
            , readinessCheck.has_tcp_check()
            , false /* isUnixSignal */
            , readinessCheck.has_start_container_alive_check()
        )
        , treeReplaceMap
    };
}

TWorkloadTreeGenerator::TWorkloadHookData TWorkloadTreeGenerator::GetLivenessCheckData(
    const API::TLivenessCheck& livenessCheck
    , const TString& workloadId
    , const TString& boxId
    , const NSecret::TSecretMap& secretMap
    , const google::protobuf::RepeatedPtrField<API::TEnvVar>& env
    , const double cpuToVcpuFactor
    , bool useEnvSecret
) const {
    TMap<TString, TString> treeReplaceMap;

    treeReplaceMap.merge(
        FillHookTypeReplaceMap(
            "liveness"
            , livenessCheck.has_container()
            , livenessCheck.has_http_get()
            , livenessCheck.has_tcp_check()
            , false /* isUnixSignal */
            , false /* isStartContainerAliveCheck */
        )
    );

    treeReplaceMap.merge(
        FillHookTriesThresholdsReplaceMap(
            "liveness"
            , livenessCheck.failure_threshold()
            , 1 // successThreshold is always 1 for liveness
        )
    );

    treeReplaceMap.merge(
        NSupport::FillReadinessOrLivenessContainerReplaceMap(
            livenessCheck.container()
            , "liveness"
            , PathHolder_->GetWorkloadContainerWithName(boxId, workloadId, "liveness")
            , PathHolder_->GetBoxRootfsPath(boxId)
            , PodAgentBinaryFilePathInBox_
            , env
            , secretMap
            , PathHolder_
            , cpuToVcpuFactor
            , true /* needTimeLimit */
            , livenessCheck.has_container() /* needValidation */
            , useEnvSecret
        ))
    ;

    treeReplaceMap.merge(
        FillHttpGetReplaceMap(
            livenessCheck.http_get()
            , "liveness"
            , livenessCheck.has_http_get()
            , true /* needDefaultInitialDelay */
        )
    );

    treeReplaceMap.merge(
        FillTcpCheckReplaceMap(
            livenessCheck.tcp_check()
            , "liveness"
            , livenessCheck.has_tcp_check()
            , true /* needDefaultInitialDelay */
        )
    );

    return TWorkloadHookData{
        GetWorkloadHookInfo(
            PathHolder_->GetWorkloadContainerWithName(boxId, workloadId, "liveness")
            , livenessCheck.has_container()
            , livenessCheck.has_http_get()
            , livenessCheck.has_tcp_check()
            , false /* isUnixSignal */
            , false /* isStartContainerAliveCheck */
        )
        , treeReplaceMap
    };
}

TWorkloadTreeGenerator::TWorkloadHookData TWorkloadTreeGenerator::GetStopPolicyData(
    const API::TStopPolicy& stopPolicy
    , const TString& workloadId
    , const TString& boxId
    , const NSecret::TSecretMap& secretMap
    , const google::protobuf::RepeatedPtrField<API::TEnvVar>& env
    , const double cpuToVcpuFactor
    , bool needValidation
    , bool useEnvSecret
) const {
    TMap<TString, TString> treeReplaceMap;

    treeReplaceMap.merge(
        NSupport::FillContainerReplaceMap(
            stopPolicy.container()
            , "stop"
            , PathHolder_->GetWorkloadContainerWithName(boxId, workloadId, "stop")
            , PathHolder_->GetBoxRootfsPath(boxId)
            , PodAgentBinaryFilePathInBox_
            , env
            , secretMap
            , PathHolder_
            , cpuToVcpuFactor
            , true /* needTimeLimit */
            , stopPolicy.has_container() /* needValidation */
            , useEnvSecret
        )
    );

    treeReplaceMap.merge(
        FillHttpGetReplaceMap(
            stopPolicy.http_get()
            , "stop"
            , stopPolicy.has_http_get()
            , false /* needDefaultInitialDelay */
        )
    );

    // For empty unix_signal we use SIGTERM
    // and default time_limit replace map with initial delay
    treeReplaceMap.merge(
        FillUnixSignalReplaceMap(
            stopPolicy.unix_signal()
            , "stop"
        )
    );

    if (needValidation) {
        Y_ENSURE(stopPolicy.max_tries() > 0, "stop_policy has 0 max_tries");
        treeReplaceMap["STOP_MAX_TRIES"] = ToString(stopPolicy.max_tries());
    } else {
        // If stop policy is not set, we give one attempt to default stop policy
        treeReplaceMap["STOP_MAX_TRIES"] = "1";
    }

    if (stopPolicy.backend_case() == API::TStopPolicy::BackendCase::BACKEND_NOT_SET) {
        treeReplaceMap.merge(
            FillHookTypeReplaceMap(
                "stop"
                , false /* isContainer */
                , false /* isHttp */
                , false /* isTcp */
                , true  /* isUnixSignal */
                , false /* isStartContainerAliveCheck */
            )
        );

        return TWorkloadHookData{
            GetWorkloadHookInfo(
                PathHolder_->GetWorkloadContainerWithName(boxId, workloadId, "stop")
                , false /* isContainer */
                , false /* isHttp */
                , false /* isTcp */
                , true  /* isUnixSignal */
                , false
            )
            , treeReplaceMap
        };
    } else {
        treeReplaceMap.merge(
            FillHookTypeReplaceMap(
                "stop"
                , stopPolicy.has_container()
                , stopPolicy.has_http_get()
                , false /* isTcp */
                , stopPolicy.has_unix_signal()
                , false /* isStartContainerAliveCheck */
            )
        );

        return TWorkloadHookData{
            GetWorkloadHookInfo(
                PathHolder_->GetWorkloadContainerWithName(boxId, workloadId, "stop")
                , stopPolicy.has_container()
                , stopPolicy.has_http_get()
                , false /* isTcp */
                , stopPolicy.has_unix_signal()
                , false /* isStartContainerAliveCheck */
            )
            , treeReplaceMap
        };
    }
}

TWorkloadTreeGenerator::TWorkloadHookData TWorkloadTreeGenerator::GetDestroyPolicyData(
    const API::TDestroyPolicy& destroyPolicy
    , const TString& workloadId
    , const TString& boxId
    , const NSecret::TSecretMap& secretMap
    , const google::protobuf::RepeatedPtrField<API::TEnvVar>& env
    , const double cpuToVcpuFactor
    , bool needValidation
    , bool useEnvSecret
) const {
    TMap<TString, TString> treeReplaceMap;

    treeReplaceMap.merge(
        FillHookTypeReplaceMap(
            "destroy"
            , destroyPolicy.has_container()
            , destroyPolicy.has_http_get()
            , false /* isTcp */
            , false /* isUnixSignal */
            , false /* isStartContainerAliveCheck */
        )
    );

    treeReplaceMap.merge(
        NSupport::FillContainerReplaceMap(
            destroyPolicy.container()
            , "destroy"
            , PathHolder_->GetWorkloadContainerWithName(boxId, workloadId, "destroy")
            , PathHolder_->GetBoxRootfsPath(boxId)
            , PodAgentBinaryFilePathInBox_
            , env
            , secretMap
            , PathHolder_
            , cpuToVcpuFactor
            , true /* needTimeLimit */
            , destroyPolicy.has_container() /* needValidation */
            , useEnvSecret
        )
    );

    treeReplaceMap.merge(
        FillHttpGetReplaceMap(
            destroyPolicy.http_get()
            , "destroy"
            , destroyPolicy.has_http_get()
            , false /* needDefaultInitialDelay */
        )
    );

    if (needValidation) {
        Y_ENSURE(destroyPolicy.max_tries() > 0, "destroy_policy has 0 max_tries");
        treeReplaceMap["DESTROY_MAX_TRIES"] = ToString(destroyPolicy.max_tries());
    } else {
        treeReplaceMap["DESTROY_MAX_TRIES"] = "";
    }

    return TWorkloadHookData{
        GetWorkloadHookInfo(
            PathHolder_->GetWorkloadContainerWithName(boxId, workloadId, "destroy")
            , destroyPolicy.has_container()
            , destroyPolicy.has_http_get()
            , false /* isTcp */
            , false /* isUnixSignal */
            , false /* isStartContainerAliveCheck */
        )
        , treeReplaceMap
    };
}

TWorkloadTreeGenerator::TWorkloadsToAdd TWorkloadTreeGenerator::UpdateSpec(
    const google::protobuf::RepeatedPtrField<API::TWorkload>& workloads
    , API::EPodAgentTargetState podAgentTargetState
    , const google::protobuf::RepeatedPtrField<API::TMutableWorkload>& mutableWorkloads
    , const NSecret::TSecretMap& secretMap
    , const TBoxTreeGenerator::TBoxesToAdd& boxes
    , const double cpuToVcpuFactor
    , ui64 specTimestamp
    , ui32 revision
    , bool useEnvSecret
    , bool autoDecodeBase64Secrets
) {
    TWorkloadTreeGenerator::TWorkloadsToAdd workloadsToAdd;

    // Add workloads to internal state only in ACTIVE or SUSPENDED pod_agent target states
    // WARNING: We do not validate workload specs in other cases (for example in REMOVED target state)
    if (
        podAgentTargetState == API::EPodAgentTargetState_ACTIVE
        || podAgentTargetState == API::EPodAgentTargetState_SUSPENDED
    ) {
        // Store mutalbe workloads WorkloadRef->MutableWorkload
        TMap<TString, const API::TMutableWorkload&> mutableWorkloadMap;
        for (const auto& mutableWorkload : mutableWorkloads) {
            mutableWorkloadMap.insert({mutableWorkload.workload_ref(), mutableWorkload});
        }

        Y_ENSURE(
            mutableWorkloads.size() == workloads.size()
            , "Different size of mutableWorkloads (" << mutableWorkloads.size() << ")"
            << " and workloads (" << workloads.size() << ")"
        );

        for (const auto& workload : workloads) {
            const TString& workloadId = workload.id();

            Y_ENSURE(!workloadId.empty(), "One of workloads has empty id");
            Y_ENSURE(!workloadsToAdd.Workloads_.contains(workloadId)
                , "Pod agent spec workloads' ids are not unique: "
                << "workload " << Quote(workloadId) << " occurs twice"
            );

            try {
                workloadsToAdd.Workloads_.insert(
                    {
                        workloadId
                        , GetWorkloadToAdd(
                            workload
                            , podAgentTargetState
                            , mutableWorkloadMap
                            , boxes
                            , secretMap
                            , cpuToVcpuFactor
                            , specTimestamp
                            , revision
                            , useEnvSecret
                            , autoDecodeBase64Secrets
                        )
                    }
                );
            } catch (yexception& e) {
                throw e << " at workload " << Quote(workloadId);
            }
        }
    }

    return workloadsToAdd;
}

TWorkloadTreeGenerator::TWorkloadToAdd TWorkloadTreeGenerator::GetWorkloadToAdd(
    const API::TWorkload& workload
    , API::EPodAgentTargetState podAgentTargetState
    , const TMap<TString, const API::TMutableWorkload&>& mutableWorkloadMap
    , const TBoxTreeGenerator::TBoxesToAdd& boxes
    , const NSecret::TSecretMap& secretMap
    , const double cpuToVcpuFactor
    , ui64 specTimestamp
    , ui32 revision
    , bool useEnvSecret
    , bool autoDecodeBase64Secrets
) const {
    const TString& workloadId = workload.id();
    TString boxId;
    TString boxHash;
    if (!IsBoxAgentMode_) {
        boxId = workload.box_ref();
        auto boxPtr = boxes.Boxes_.FindPtr(boxId);
        Y_ENSURE(boxPtr, "Unknown box: " << Quote(boxId));
        boxHash = boxPtr->Target_.Hash_;
    }

    const TString treeHash = GetWorkloadHash(workload, secretMap, boxHash, useEnvSecret);

    auto mutableWorkloadPtr = mutableWorkloadMap.FindPtr(workloadId);
    Y_ENSURE(mutableWorkloadPtr, "Mutable workload not found");
    // Shutdown workload in SUSPENDED target state
    // otherwise use workload target state from mutable workload spec
    const API::EWorkloadTargetState workloadTargetState = (podAgentTargetState == API::EPodAgentTargetState_SUSPENDED)
        ? API::EWorkloadTarget_REMOVED
        : mutableWorkloadPtr->target_state()
    ;

    TVector<TString> initContainers;

    TWorkloadHookData readinessCheckData = GetReadinessCheckData(
        workload.readiness_check()
        , workloadId
        , boxId
        , secretMap
        , workload.env()
        , cpuToVcpuFactor
        , useEnvSecret
    );
    TWorkloadHookData livenessCheckData = GetLivenessCheckData(
        workload.liveness_check()
        , workloadId
        , boxId
        , secretMap
        , workload.env()
        , cpuToVcpuFactor
        , useEnvSecret
    );
    TWorkloadHookData stopPolicyData = GetStopPolicyData(
        workload.stop_policy()
        , workloadId
        , boxId
        , secretMap
        , workload.env()
        , cpuToVcpuFactor
        , workload.has_stop_policy()
        , useEnvSecret
    );
    TWorkloadHookData destroyPolicyData = GetDestroyPolicyData(
        workload.destroy_policy()
        , workloadId
        , boxId
        , secretMap
        , workload.env()
        , cpuToVcpuFactor
        , workload.has_destroy_policy()
        , useEnvSecret
    );

    TBehavior3 resolvedTree = WorkloadTreeTemplate_;
    TMap<TString, TString> cachedReplace;
    const TString rootfsPath = IsBoxAgentMode_ ? "/" : PathHolder_->GetBoxRootfsPath(boxId);
    {
        TMap<TString, TString> replace;
        replace["WORKLOAD_ID"] = workloadId;
        replace["LOOKUP_CONTAINER_MASK"] = PathHolder_->GetWorkloadContainerLookupMask(boxId, workloadId);

        replace["WORKLOAD_ULIMIT"] = NSupport::GetULimitList(workload.ulimit_soft());
        replace["WORKLOAD_CAPABILITIES_AMBIENT"] = NSupport::CONTAINER_CAPABILITIES_AMBIENT;

        replace["POD_AGENT_MODE"] = IsBoxAgentMode_ ? "box_mode" : "full_mode";

        replace.merge(
            NSupport::FillContainerReplaceMap(
                workload.start()
                , "start"
                , PathHolder_->GetWorkloadContainerWithName(boxId, workloadId, "start")
                , rootfsPath
                , PodAgentBinaryFilePathInBox_
                , workload.env()
                , secretMap
                , PathHolder_
                , cpuToVcpuFactor
                , false /* needTimeLimit */
                , true /* needValidation */
                , useEnvSecret
                , autoDecodeBase64Secrets
            )
        );

        replace.merge(readinessCheckData.TreeReplaceMap_);
        replace.merge(livenessCheckData.TreeReplaceMap_);
        replace.merge(stopPolicyData.TreeReplaceMap_);
        replace.merge(destroyPolicyData.TreeReplaceMap_);

        // TODO(DEPLOY-420) remove this
        if (workload.transmit_logs()) {
            // assuming that stdout_file/stderr_file coming from stage_ctl are equal to those hardcoded at TPathHolder. Migration code for DEPLOY-1350
            TString stdoutPathInBox = workload.start().stdout_file() ? workload.start().stdout_file() : TStringBuilder() << "/" << PathHolder_->GetWorkloadLogsFileName(workloadId, "stdout");
            TString stderrPathInBox = workload.start().stderr_file() ? workload.start().stderr_file() : TStringBuilder() << "/" << PathHolder_->GetWorkloadLogsFileName(workloadId, "stderr");

            if (IsBoxAgentMode_) {
                replace["START_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE"] = stdoutPathInBox;
                replace["START_STDERR_LOG_FILE_FULL_PATH_TO_CREATE"] = stderrPathInBox;
            } else {
                replace["START_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE"] = TStringBuilder() << PathHolder_->GetBoxRootfsPath(boxId) << stdoutPathInBox;
                replace["START_STDERR_LOG_FILE_FULL_PATH_TO_CREATE"] = TStringBuilder() << PathHolder_->GetBoxRootfsPath(boxId) << stderrPathInBox;
            }

            replace["START_STDOUT_LOG_PATH"] = stdoutPathInBox;
            replace["START_STDERR_LOG_PATH"] = stderrPathInBox;

            replace["START_STDOUT_AND_STDERR_FILE_SIZE_LIMIT"] = ToString(MaxLogsFileSize_ * PORTO_LOG_FILE_SIZE_LIMIT_MULTIPLIER);
        }

        {
            replace["BOX_TREE_HASH"] = boxHash;
            replace["BOX_CONTAINER"] = PathHolder_->GetBoxContainer(boxId);
        }

        {
            for (size_t i = 0; i < workload.initSize(); ++i) {
                initContainers.push_back(PathHolder_->GetWorkloadInitContainer(boxId, workloadId, i));
            }

            replace.merge(
                NSupport::FillInitContainersReplaceMap(
                    workload.init()
                    , initContainers
                    , rootfsPath
                    , PodAgentBinaryFilePathInBox_
                    , workload.env()
                    , secretMap
                    , PathHolder_
                    , cpuToVcpuFactor
                    , useEnvSecret
                )
            );
        }

        replace["TREE_HASH"] = treeHash;
        cachedReplace = replace;
        TBehavior3TemplateResolver().Resolve(resolvedTree, replace);
    }

    TVector<TWorkloadMeta::TContainerInfo> initContainersInfo;
    initContainersInfo.reserve(initContainers.size());
    Transform(initContainers.begin(), initContainers.end(), std::back_inserter(initContainersInfo), [](const TString &containerName) {
        return TWorkloadMeta::TContainerInfo(containerName);
    });

    return TWorkloadToAdd{
        TUpdateHolder::TWorkloadTarget(
            TWorkloadMeta(
                workloadId
                , specTimestamp
                , revision
                , boxId

                , TWorkloadMeta::TContainerInfo(PathHolder_->GetWorkloadContainerWithName(boxId, workloadId, "start"))
                , initContainersInfo

                , readinessCheckData.Info_
                , livenessCheckData.Info_
                , stopPolicyData.Info_
                , destroyPolicyData.Info_

            )
            , new TObjectTree(
                Logger_
                , TStatusNTickerHolder::GetWorkloadTreeId(workloadId)
                , TBehavior3EditorJsonReader(resolvedTree)
                    .WithPorto(Porto_)
                    .WithPosixWorker(PosixWorker_)
                    .WithNetworkClient(NetworkClient_)
                    .WithUpdateHolderTarget(StatusNTickerHolder_->GetUpdateHolder()->GetUpdateHolderTarget())
                    .WithBoxStatusRepository(StatusNTickerHolder_->GetBoxStatusRepository())
                    .WithWorkloadStatusRepository(StatusNTickerHolder_->GetWorkloadStatusRepository())
                    .WithTemplateBTStorage(TemplateBTStorage_)
                    .WithWorkloadInternalStatusRepository(StatusNTickerHolder_->GetWorkloadStatusRepositoryInternal())
                    .WithPathHolder(PathHolder_)
                    .WithSystemLogsSender(StatusNTickerHolder_->GetWorkloadSystemLogsSender())
                    .BuildRootNode()
                , workloadId
                , StatusNTickerHolder_->GetUpdateHolder()->GetUpdateHolderTarget()
                , StatusNTickerHolder_->GetWorkloadStatusRepository()
            )
            , treeHash
        )
        , workloadTargetState
        , cachedReplace
    };
}

void TWorkloadTreeGenerator::RemoveWorkloads(const TWorkloadsToAdd& workloads) {
    for (const TString& workloadId : StatusNTickerHolder_->GetWorkloadIds()) {
        if (!workloads.Workloads_.contains(workloadId)) {
            StatusNTickerHolder_->SetWorkloadTargetRemove(workloadId);
        }
    }
}

void TWorkloadTreeGenerator::AddWorkloads(const TWorkloadsToAdd& workloads) {
    for (const auto& it : workloads.Workloads_) {
        const TString& workloadId = it.first;
        const auto& workloadData = it.second;

        if (!StatusNTickerHolder_->HasWorkload(workloadId)) {
            StatusNTickerHolder_->AddWorkloadWithTargetCheck(workloadData.Target_);
        }

        StatusNTickerHolder_->UpdateWorkloadTargetState(workloadId, workloadData.TargetState_);
        StatusNTickerHolder_->UpdateWorkloadTarget(workloadData.Target_);
    }
}

} // namespace NInfra::NPodAgent
